import generateBarcode from "./generateBarcode.js"
import path from 'path'
import Product from '../models/product.js'
import currencyModel from "../models/currencyModel.js";


const addNewProduct = async (req, res) => {
  try {
    const { productName, type, gram, karat, bellNumber, wage, auns } = req.body;

    const barcode = generateBarcode();
    const isSold = false;
    const date = new Date();

    if(!gram || !karat || !wage || !auns){
      return res.json({success:false , message:"لطفا موارد مهم را برسانید"})
    }

    let imageFile = req.file ? req.file.path : null;

    const today = new Date().toISOString().split("T")[0];

    const rate = await currencyModel.findOne({ date: today });

    if (!rate) {
      return res.json({
        success: false,
        message: "نرخ امروز دالر ثبت نشده"
      });
    }

    const purchase = (Number(auns / 12.15 / 24) * (Number(karat) + 0.12) + Number(wage)) * Number(rate.usdToAfn) * Number(gram);

    const productSchema = {
      productName,
      type,
      gram,
      karat,
      purchasePriceToAfn: purchase,
      bellNumber,
      isSold,
      barcode,
      image: imageFile,
      wage,
      auns
    };

    const newProduct = new Product(productSchema);
    await newProduct.save();

    res.json({
      success: true,
      message: "دیتا موفقانه ثبت شد",
      newProduct
    });

  } catch (error) {
    console.log(error);
    res.json({
      success: false,
      message: error.message
    });
  }
};


const existProduct = async (req, res) => {
  try {

    const products = await Product.find({ isSold: false }).sort({ createdAt: -1 })

    if (!products) {
      return ({ success: false, message: "هیچ دیتایی وجود ندارد" })
    } else {
      res.json({ success: true, products })
    }


  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message })
  }
}

const searchByBarcode = async (req, res) => {
  try {
    const { code } = req.params

    const productByBarcode = await Product.findOne({ barcode: code, isSold: false });

    if (!productByBarcode) {
      return res.json({
        success: false,
        message: "جنس یافت نشد",
      })
    }

    return res.json({
      success: true,
      productByBarcode,
    })

  } catch (error) {
    console.log(error)
    return res.json({
      success: false,
      message: error.message,
    })
  }
}

const isSold = async (req, res) => {

  const { id } = req.params;
  const { isSold } = req.body;

  try {

    const productIsSold = await Product.findById(id)

    if (!productIsSold) {
      return res.json({ success: false, message: "جنس پیدا نشد" })
    }

    if (productIsSold.isSold === isSold) {
      return res.json({ message: `محصول قبلاً ${isSold ? "فروخته شده" : "موجود"} است` })
    }

    productIsSold.isSold = isSold
    await productIsSold.save()

    res.json({ success: true, message: `وضعیت موفقانه تغیر خورده است`, productIsSold })
  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message })
  }
}

const updateProduct = async (req, res) => {
  try {
    const { productName, type, weight, karat, purchasePrice, currency } = req.body
    const productId = req.params.id
    console.log("req", req.body)

    const newUpdatedProduct = await Product.findByIdAndUpdate(productId, { productName, type, weight, karat, purchasePrice, currency }, { new: true })

    console.log("product:", newUpdatedProduct)

    res.json({ success: true, newUpdatedProduct })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message })
  }
}

const deleteProduct = async (req, res) => {
  try {

    const id = req.params.productId

    await Product.findByIdAndDelete(id)

    res.json({ success: true, message: "محصول حذف شد" })


  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message })
  }
}

const todayRegistredProduct = async (req, res) => {

  try {
    const today = new Date().toISOString().split("T")[0];

    const startOfDay = new Date(`${today}T00:00:00.000Z`);
    const endOfDay   = new Date(`${today}T23:59:59.999Z`);


    const dailyProduct = await Product.find({})
    .sort({ createdAt: -1 })
    .limit(20);

    if (dailyProduct.length === 0) {
      return res.json({
        success: false,
        message: "امروز هیچ جنسی ثبت نشده"
      });
    }

    res.json({
      success: true,
      count: dailyProduct.length,
      dailyProduct
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};


const searchByDate = async (req, res) => {
  try {

    const { fromDate, toDate } = req.query

    if (!fromDate || !toDate) {
      return res.json({ success: false, message: "تاریخ شروع و ختم را وارد کنید" })
    }

    const startDate = new Date(fromDate);
    startDate.setHours(0, 0, 0, 0);

    const endDate = new Date(toDate);
    endDate.setHours(23, 59, 59, 999);

    const productByDate = await Product.find({
      isSold: false,
      createdAt: {
        $gte: startDate,
        $lte: endDate
      }
    })

    if (productByDate.length === 0) {
      return res.json({ success: false, message: "در این تاریخ دیتا وجود ندارد" })
    }

    res.json({ success: true, productByDate })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const getTotalOfWarehouse = async (req, res) => {
  try {

    const totalProduct = await Product.aggregate([
      {
        $match: {
          isSold: false
        }
      },
      {
        $group: {
          _id: null,
          totalGold: { $sum: { $ifNull: ["$gram", 0] } },
          totalAmountToAfghani: { $sum: { $ifNull: ["$purchasePriceToAfn", 0] } },
          count: { $sum: 1 }
        }
      },
      {
        $project: {
          _id: 0,
          totalGold: 1,
          totalAmountToAfghani: 1,
          count: 1
        }
      },
    ])

    res.json({ success: true, totalProduct })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const addFromFragment = async(req , res)=>{
  try {
    const {productName , type , gram , karat , purchase } = req.body

    if(productName ==="" || type ==="" || !gram || !karat || !purchase ){
      return res.json({success: false , message:"لطفا موارد مهم را وارد کنید"})
    }
    const barcode = generateBarcode()
    let imageFile = req.file ? req.file.path : null;

    const isFragment = true;

    const isSold = false
    
    const productSchema={
      productName,
      type,
      gram,
      karat,
      purchasePriceToAfn:purchase,
      isSold,
      image:imageFile,
      barcode,
      isFragment
    }

    const newProduct = new Product(productSchema)
    await newProduct.save()

    res.json({success:true , message:"جنس ثبت شد"})

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const getDailyFragment = async(req , res)=>{

  const startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    const endDate = new Date();
    endDate.setHours(23, 59, 59, 999);

  try {
    const daily = await Product.find({
      createdAt: {
        $gte: startDate,
        $lte: endDate
      },
      isFragment:true
    }).sort({createdAt : -1})

    if(!daily){
      return res.json({success:false , message:"دیتا وجود ندارد"})
    }

    res.json({success:true , daily})

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}


export {
  addNewProduct,
  existProduct,
  searchByBarcode,
  isSold,
  updateProduct,
  deleteProduct,
  todayRegistredProduct,
  searchByDate,
  getTotalOfWarehouse,
  addFromFragment,
  getDailyFragment
}