import currencyModel from "../models/currencyModel.js";
import productMasterModel from "../models/productMasterModel.js";
import purchaseModel from "../models/purchaseModel.js";
import supplierModel from "../models/supplierModel.js";


const createPurchase = async (req, res) => {
  try {
    const { supplierId, items, date, bellNumber, currency, paidAmount } = req.body;
    console.log(req.body)


    const supplier = await supplierModel.findById(supplierId);
    if (!supplier) {
      return res.json({ success: false, message: "Supplier پیدا نشد" });
    }


    const preparedItems = items.map(item => ({
      productMasterId: item.productMasterId,
      name: item.name,
      type: item.type,
      gram: item.gram,
      karat: item.karat,
      price: item.price,
      quantity: item.quantity,
      registeredQty: 0,
      remainingQty: item.quantity,
      isCompleted: false
    }));


    const totalAmount = preparedItems.reduce(
      (sum, i) => sum + i.price * i.quantity,
      0
    );

    const purchase = await purchaseModel.create({
      supplierId,
      supplierName: supplier.name,
      items: preparedItems,
      totalAmount,
      date,
      bellNumber,
      currency,
      paidAmount: Number(paidAmount) || 0
    });

    res.json({ success: true, message: "دیتا موفقانه ثبت شد", purchase });
  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message })
  }
};



const getItemsWithSupplierName = async (req, res) => {
  try {
    const { search } = req.query;

    if (!search) {
      return res.json({ success: false, message: "نام جنس را وارد کنید" })
    }

    const items = await purchaseModel.aggregate([

      { $unwind: "$items" },

      {
        $match: {
          "items.name": { $regex: search, $options: "i" }
        }
      },

      {
        $project: {
          _id: 0,
          supplierName: 1,
          bellNumber: 1,
          currency: 1,
          item: "$items"
        }
      }
    ]);

    res.json({ success: true, items })

  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const updatePurchaseItemQuantity = async (req, res) => {
  try {
    const { bellNumber, productMasterId } = req.body;

    if (!bellNumber || !productMasterId) {
      return res.json({
        success: false,
        message: "bellNumber و productMasterId الزامی است"
      });
    }


    const purchase = await purchaseModel.findOne({ bellNumber });

    if (!purchase) {
      return res.status(404).json({
        success: false,
        message: "فاکتور پیدا نشد"
      });
    }


    const item = purchase.items.find(i =>
      i._id.toString() === productMasterId
    );

    if (!item) {
      return res.json({
        success: false,
        message: "محصول در فاکتور پیدا نشد"
      });
    }


    if (item.quantity > item.registeredQty) {
      item.registeredQty += 1
      item.remainingQty -= 1
    }


    if (item.quantity === item.registeredQty) {
      item.isCompleted = true;
    }


    await purchase.save();

    res.json({
      success: true,
      message: "آیتم با موفقیت آپدیت شد",
      updatedItem: item
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};

const getSupplierTransaction = async (req, res) => {
  try {

    const purchase = await purchaseModel.aggregate([
      {
        $unwind: "$items"
      },
      {
        $group: {
          _id: "$_id",
          supplierId: { $first: "$supplierId" },
          supplierName: { $first: "$supplierName" },
          bellNumber: { $first: "$bellNumber" },
          totalAmount: { $first: "$totalAmount" },
          paidAmount: { $first: "$paidAmount" },
          currency: { $first: "$currency" },
          date: { $first: "$createdAt" },

          items: {
            $push: {
              name: "$items.name",
              type: "$items.type",
              gram: "$items.gram",
              karat: "$items.karat",
              quantity: "$items.quantity",
              registeredQty: "$items.registeredQty",
              remainingQty: "$items.remainingQty",
              price: "$items.price",
              isCompleted: "$items.isCompleted",
              purchaseId: "$items._id"
            }
          },

          count: { $sum: 1 }
        }
      },
      {
        $sort:{date:-1}
      }
    ]);


    if (purchase.length === 0) {
      return res.json({ success: false, message: "دیتا وجود ندارد" })
    }
    res.json({ success: true, purchase })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const TotalPurchase = async (req, res) => {

  let totalGram = 0;
  let totalAmount = 0;
  let totalQuantity = 0;
  let paidAmount = 0;

  try {
    const { dateFrom, dateTo } = req.query;

    const today = new Date().toISOString().split("T")[0];
    const rate = await currencyModel.findOne({ date: today });

    let filter = {};

    if (dateFrom && dateTo) {
      const startOfDay = new Date(dateFrom);
      startOfDay.setHours(0, 0, 0, 0);

      const endOfDay = new Date(dateTo);
      endOfDay.setHours(23, 59, 59, 999);

      filter.createdAt = {
        $gte: startOfDay,
        $lte: endOfDay
      };
    }

    const transactions = await purchaseModel.find(filter);

    for (const trx of transactions) {

      const isDollarPurchase = trx.currency === "دالر";

      if (isDollarPurchase) {
        if (!rate) {
          return res.json({
            success: false,
            message: "نرخ امروز دالر ثبت نشده"
          });
        }

        totalAmount += trx.totalAmount * rate.usdToAfn;
        paidAmount += trx.paidAmount * rate.usdToAfn;

      } else {
        totalAmount += trx.totalAmount;
        paidAmount += trx.paidAmount;
      }

      trx.items.forEach(item => {
        totalGram += item.gram * item.quantity
        totalQuantity += item.quantity;
      });
    }

    res.json({
      success: true,
      data: {
        totalAmount,
        paidAmount,
        totalGram,
        totalQuantity
      }
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};





export { createPurchase, getItemsWithSupplierName, updatePurchaseItemQuantity, getSupplierTransaction , TotalPurchase }