import supplierModel from "../models/supplierModel.js";
import supplierProductModel from "../models/supplierProductModel.js";
import mongoose from 'mongoose'

const addSupplierProduct = async (req, res) => {
    try {
        const { products, supplierId, bellNumber , detail } = req.body;

        if (!products || products.length === 0 || !supplierId) {
            return res.json({
                success: false,
                message: "لطفاً همه فیلدهای مهم را پر نمایید"
            });
        }

        const supplier = await supplierModel.findById(supplierId);
        if (!supplier) {
            return res.json({
                success: false,
                message: "تمویل‌کننده یافت نشد"
            });
        }


        const formattedProducts = products.map(item => {
            const weight = Number(item.weight);

            if (!weight || weight <= 0) {
                throw new Error("وزن محصول نامعتبر است");
            }

            return {
                ...item,
                weight,
                registeredWeight: 0,
                remainWeight: weight
            };
        });

        const supplierProduct = {
            products: formattedProducts,
            supplierId,
            supplierName: supplier.name,
            bellNumber,
            detail
        };

        const newSupplierProduct = new supplierProductModel(supplierProduct);
        await newSupplierProduct.save();

        res.json({
            success: true,
            message: "محصول تمویل‌کننده با موفقیت ثبت شد",
            data: newSupplierProduct
        });

    } catch (error) {
        console.error(error);
        res.json({ success: false, message: error.message });
    }
};


const getSupplierProduct = async (req, res) => {
    try {
        const { supplierId } = req.query;

        let matchStage = {};
        if (supplierId) {
            matchStage.supplierId = new mongoose.Types.ObjectId(supplierId);
        }

        const supplierProducts = await supplierProductModel.aggregate([
            {
                $match: matchStage
            },
            {
                $unwind: "$products"
            },
            {
                $group: {
                    _id: "$_id", // 👈 گروپ بر اساس آیدی خود سند
                    supplierId: { $first: "$supplierId" },
                    supplierName: { $first: "$supplierName" },
                    bellNumber: { $first: "$bellNumber" },
                    currency: { $first: "$currency" },
                    detail : {$first : "$detail"},
                    createdAt: { $first: "$createdAt" },
                    products: {
                        $push: {
                            productId: "$products._id",
                            name: "$products.name",
                            type: "$products.type",
                            karat: "$products.karat",
                            weight: "$products.weight",
                            registeredWeight: "$products.registeredWeight",
                            remainWeight: "$products.remainWeight",
                            pasa: "$products.pasa",
                            pasaReceipt: "$products.pasaReceipt",
                            pasaRemaining: "$products.pasaRemaining",
                            wagePerGram: "$products.wagePerGram",
                            totalWage: "$products.totalWage",
                            wageReceipt: "$products.wageReceipt",
                            wageRemaining: "$products.wageRemaining"
                        }
                    }
                }
            },
            {
                $sort: { createdAt: -1 }
            }
        ]);

        res.json({
            success: true,
            data: supplierProducts
        });

    } catch (error) {
        console.error(error);
        res.json({
            success: false,
            message: error.message
        });
    }
};

const updatePay = async (req, res) => {
    try {
        const { transactionId, productId } = req.query
        if (!transactionId || !productId) {
            return res.json({ success: false, message: "موارد مهم وجود ندارد" })
        }
        const transaction = await supplierProductModel.findById(transaction)
    } catch (error) {

    }
}

const searchToSave = async (req, res) => {
    try {
        const { search } = req.query;

        if (!search) {
            return res.json({ success: false, message: "نام جنس را وارد کنید" })
        }

        const product = await supplierProductModel.aggregate([

            { $unwind: "$products" },

            {
                $match: {
                    "products.name": { $regex: search, $options: "i" }
                }
            },

            {
                $project: {
                    _id: 1,
                    supplierName: 1,
                    bellNumber: 1,
                    currency: 1,
                    products: "$products"
                }
            }
        ]);

        if (product.length === 0) {
            return res.json({ success: false, message: "جنس یافت نشد" })
        }

        res.json({ success: true, product })

    } catch (error) {
        res.status(500).json({ message: error.message });
    }
};

const updateWeight = async (req, res) => {
    try {
        let { bellNumber, productId, gram } = req.body;

        gram = Number(gram);

        if (!bellNumber || !productId || !gram || gram <= 0) {
            return res.json({
                success: false,
                message: "bellNumber، productId و gram معتبر الزامی است"
            });
        }

        const purchase = await supplierProductModel.findOne({ bellNumber });

        if (!purchase) {
            return res.status(404).json({
                success: false,
                message: "فاکتور پیدا نشد"
            });
        }

        const item = purchase.products.find(
            i => i._id.toString() === productId
        );

        if (!item) {
            return res.json({
                success: false,
                message: "محصول در فاکتور پیدا نشد"
            });
        }

        // اگر قبلاً کامل شده باشد
        if (item.isCompleted) {
            return res.json({
                success: false,
                message: "این محصول قبلاً به طور کامل ثبت شده است"
            });
        }

        const remaining = item.weight - item.registeredWeight;

        if (gram > remaining) {
            return res.json({
                success: false,
                message: "مقدار وارد شده بیشتر از وزن باقی‌مانده است"
            });
        }

        item.registeredWeight += gram;
        item.remainWeight = item.weight - item.registeredWeight;

        // بررسی اگر وزن کامل شده باشد
        if (item.registeredWeight >= item.weight) {
            item.registeredWeight = item.weight;
            item.remainWeight = 0;
            item.isCompleted = true;
        }

        await purchase.save();

        if (item.isCompleted) {
            return res.json({
                success: true,
                message: "جنس مکمل شد",
                updatedItem: item
            });
        }

        res.json({
            success: true,
            message: "وزن با موفقیت ثبت شد",
            updatedItem: item
        });

    } catch (error) {
        console.log(error);
        res.json({ success: false, message: error.message });
    }
};

const updateLoan = async (req, res) => {
    try {
        const { transactionId, productId } = req.query;
        const { pasa, wage } = req.body;

        if (!transactionId || !productId) {
            return res.json({ success: false, message: "آیدی شناسایی نشده است" });
        }

        const transaction = await supplierProductModel.findById(transactionId);
        if (!transaction) {
            return res.json({ success: false, message: "ترانزکشن یافت نشد" });
        }

        const product = transaction.products.find(
            (i) => i._id.toString() === productId
        );

        if (!product) {
            return res.json({ success: false, message: "محصول یافت نشد" });
        }

        // --- بروزرسانی پاسه ---
        if (pasa) {
            const newPasaReceipt = product.pasaReceipt + Number(pasa);
            if (newPasaReceipt > product.pasa) {
                return res.json({ success: false, message: "مقدار پاسه بیش از حد است" });
            }
            product.pasaReceipt = newPasaReceipt;
            product.pasaRemaining = product.pasa - product.pasaReceipt;
        }

        // --- بروزرسانی مزد ---
        if (wage) {
            const newWageReceipt = product.wageReceipt + Number(wage);
            if (newWageReceipt > product.totalWage) {
                return res.json({ success: false, message: "مقدار مزد بیش از حد است" });
            }
            product.wageReceipt = newWageReceipt;
            product.wageRemaining = product.totalWage - product.wageReceipt;
        }

        // --- علامت تکمیل ---
        if (product.pasaRemaining === 0 && product.wageRemaining === 0) {
            product.isCompleted = true;
        }

        // ذخیره تغییرات در ترانزکشن اصلی
        await transaction.save();

        res.json({ success: true, message: "موفقانه آپدیت شد", product });
    } catch (error) {
        console.log(error);
        res.json({ success: false, message: error.message });
    }
};

const getTotal = async (req, res) => {
    try {
        const total = await supplierProductModel.aggregate([
            {
                $unwind: "$products"
            },
            {
                $group: {
                    _id: null,
                    remainPasa: { $sum: "$products.pasaRemaining" },
                    remainWage: { $sum: "$products.wageRemaining" }
                }
            }
        ]);

        if (!total.length) {
            return res.json({
                success: false,
                message: "هیچ دیتا وجود ندارد"
            });
        }

        res.json({
            success: true,
            total: total[0] // 👈 مستقیم آبجکت
        });

    } catch (error) {
        console.log(error);
        res.json({ success: false, message: error.message });
    }
};



const getDailyTotal = async (req, res) => {
    try {
        const startOfDay = new Date();
        startOfDay.setHours(0, 0, 0, 0);

        const endOfDay = new Date();
        endOfDay.setHours(23, 59, 59, 999);

        const result = await supplierProductModel.aggregate([
            {
                $match: {
                    createdAt: { $gte: startOfDay, $lte: endOfDay }
                }
            },
            {
                $unwind: "$products"
            },
            {
                $group: {
                    _id: null,
                    totalWeight: { $sum: { $toDouble: "$products.weight" } },
                    totalPasa: { $sum: { $toDouble: "$products.pasa" } },
                    pasaReceipt: { $sum: "$products.pasaReceipt" },
                    remainPasa: { $sum: { $toDouble: "$products.pasaRemaining" } },
                    totalWage: { $sum: { $toDouble: "$products.totalWage" } },
                    wageReceipt: { $sum: { $toDouble: "$products.wageReceipt" } },
                    remainWage: { $sum: { $toDouble: "$products.wageRemaining" } }
                }
            },
            {
                $project: {
                    _id: 0,
                    totalWeight: 1,
                    totalPasa: 1,
                    pasaReceipt: 1,
                    remainPasa: 1,
                    totalWage: 1,
                    wageReceipt: 1,
                    remainWage: 1
                }
            }
        ]);

        if (result.length === 0) {
            return res.json({
                success: false,
                message: "امروز خرید جنس از تمویل کننده صورت نگرفته است"
            });
        }

        res.json({
            success: true,
            total: result[0] // 👈 آبجکت نهایی
        });

    } catch (error) {
        console.log(error);
        res.json({ success: false, message: error.message });
    }
};


const remain = async (req, res) => {
    try {
        const product = await supplierProductModel.aggregate([
            { $unwind: "$products" },

            {
                $match: {
                    "products.remainWeight": { $gt: 0 }
                }
            },

            {
                $project: {
                    _id: 1,
                    supplierName: 1,
                    bellNumber: 1,
                    currency: 1,
                    products: 1
                }
            }
        ]);

        if (product.length === 0) {
            return res.json({
                success: false,
                message: "جنس یافت نشد"
            });
        }

        res.json({
            success: true,
            product
        });

    } catch (error) {
        res.status(500).json({ message: error.message });
    }
};


const getBySupplier = async (req, res) => {
  try {
    const { supplierId } = req.query;

    if (!supplierId) {
      return res.json({
        success: false,
        message: "supplierId ارسال نشده"
      });
    }

    const total = await supplierProductModel.aggregate([
      {
        $match: {
          supplierId: new mongoose.Types.ObjectId(supplierId)
        }
      },
      {
        $unwind: "$products"
      },
      {
        $group: {
          _id: null,
          totalWeight: { $sum: "$products.weight" },
          totalPasa: { $sum: "$products.pasa" },
          pasaReceipt: { $sum: "$products.pasaReceipt" },
          pasaRemaining: { $sum: "$products.pasaRemaining" },
          totalWage: { $sum: "$products.totalWage" },
          wageReceipt: { $sum: "$products.wageReceipt" },
          wageRemaining: { $sum: "$products.wageRemaining" }
        }
      },
      {
        $project: {
          _id: 0,
          karat: "$_id",
          totalWeight: 1,
          totalPasa: 1,
          pasaReceipt: 1,
          pasaRemaining: 1,
          totalWage: 1,
          wageReceipt: 1,
          wageRemaining: 1
        }
      }
    ]);

    if (total.length === 0) {
      return res.json({
        success: false,
        message: "دیتا وجود ندارد"
      });
    }

    res.json({ success: true, total });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};


const addOldTransaction = async(req , res)=>{
    try {
        const { name , remainPasa , remainWage , detail} = req.body
        const {supplierId} = req.query

        if(!supplierId || !name ){
            return res.json({success:false , message:"لطفا موارد مهم را خانه پری نمایید"})
        }

        const supplier = await supplierModel.findById(supplierId)
        if(!supplier){
            return res.json({success:false , message:"تمویل کننده یافت نشد"})
        }

        const products = []

        products.push({
            name,
            weight:0,
            pasa:remainPasa,
            pasaRemaining:remainPasa,
            totalWage:remainWage,
            wageRemaining:remainWage
        })

        const schema = {
            products,
            supplierId,
            supplierName:supplier.name,
            detail
        } 
        const newSchema = new supplierProductModel(schema)
        await newSchema.save()

        res.json({success:true , message:"موفقانه ثبت شد"})
        
    } catch (error) {
        console.log(error);
    res.json({ success: false, message: error.message });
    }
}


export { addSupplierProduct, getSupplierProduct, searchToSave, updateWeight, updateLoan, getTotal, getDailyTotal, remain , getBySupplier , addOldTransaction };
