import tradeModel from "../models/tradeModel.js"
import traderModel from "../models/tradersModel.js"
import mongoose from "mongoose"

const search = async (req, res) => {
    try {
        const { search } = req.query

        if (!search) {
            return res.json({ success: false, message: "جستجو کنید" })
        }

        const query = {
            name: { $regex: search, $options: "i" }
        }

        const data = await traderModel.find(query).limit(10)

        if (data.length === 0) {
            return res.json({ success: false, message: "دیتا یافت نشد" })
        }

        res.json({ success: true, data })

    } catch (error) {
        console.log(error)
        res.json({ success: false, message: error.message })
    }
}

const addTrade = async (req, res) => {
    try {

        const { traderId, amount, type, detail, currency } = req.body

        if (!traderId || !amount || !type || !currency) {
            return res.json({ success: false, message: "لطفا موارد مهم را خانه پری نمایید" })
        }

        const { name } = await traderModel.findById(traderId)

        if (!name) {
            return res.json({ success: false, message: "اسم شخص پیدا نشد" })
        }

        const trade = {
            traderId,
            traderName: name,
            amount,
            type,
            detail,
            currency
        }

        const newTrade = new tradeModel(trade)
        await newTrade.save()

        res.json({ success: true, message: "موفقانه ذخیره شد" })

    } catch (error) {
        console.log(error)
        res.json({ success: false, message: error.message })
    }
}

const getPersonTrade = async (req, res) => {
    try {
        const { traderId } = req.query
        if (!traderId) {
            return res.json({ success: false, message: "Trader ID لازم است" })
        }

        const personTrade = await tradeModel.aggregate([
            {
                $match: {
                    traderId: new mongoose.Types.ObjectId(traderId) // 🔹 حتما new
                }
            },
            {
                $facet: {
                    summary: [
                        {
                            $group: {
                                _id: "$currency",
                                totalCredit: {
                                    $sum: {
                                        $cond: [
                                            { $eq: ["$type", "طلب"] },
                                            { $toDecimal: "$amount" },
                                            0
                                        ]
                                    }
                                },
                                totalDebit: {
                                    $sum: {
                                        $cond: [
                                            { $eq: ["$type", "پرداخت"] },
                                            { $toDecimal: "$amount" },
                                            0
                                        ]
                                    }
                                }
                            }
                        },
                        {
                            $project: {
                                _id: 0,
                                currency: "$_id",
                                totalCredit: 1,
                                totalDebit: 1,
                                balance: { $subtract: ["$totalCredit", "$totalDebit"] }
                            }
                        }
                    ],
                    transactions: [
                        { $sort: { createdAt: -1 } },
                        {
                            $project: {
                                _id: 1,
                                amount: 1,
                                type: 1,
                                currency: 1,
                                detail: 1,
                                createdAt: 1
                            }
                        }
                    ]
                }
            }
        ])

        if (!personTrade.length ||
            (personTrade[0].summary.length === 0 && personTrade[0].transactions.length === 0)) {
            return res.json({ success: false, message: "دیتا وجود ندارد" })
        }

        res.json({ success: true, personTrade })
    } catch (error) {
        console.log(error)
        res.json({ success: false, message: error.message })
    }
}


const getTotal = async (req, res) => {
  try {
    const today = new Date().toISOString().split("T")[0];

    const startOfDay = new Date(`${today}T00:00:00.000Z`);
    const endOfDay   = new Date(`${today}T23:59:59.999Z`);

    const total = await tradeModel.aggregate([
      {
        $match: {
          createdAt: {
            $gte: startOfDay,
            $lte: endOfDay
          }
        }
      },
      {
        $group: {
          _id: {
            type: "$type",
            currency: "$currency"
          },
          totalAmount: { $sum: "$amount" }
        }
      },
      {
        $project: {
          _id: 0,
          type: "$_id.type",
          currency: "$_id.currency",
          totalAmount: 1
        }
      }
    ]);

    if (total.length === 0) {
      return res.json({
        success: false,
        message: "امروز هیچ معامله‌ای ثبت نشده"
      });
    }

    res.json({
      success: true,
      total
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};

const getDaily = async(req , res)=>{
    try {
        const startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    const endDate = new Date();
    endDate.setHours(23, 59, 59, 999);

    const daily = await tradeModel.find({createdAt:{
        $gte:startDate,
        $lte:endDate
    }})

    if(daily.length===0){
        return res.json({success:false , message:"دیتا برای امروز وجود ندارد"})
    }

    res.json({success:true , daily})
    } catch (error) {
        console.log(error);
    res.json({ success: false, message: error.message });
    }
}


const getAllTotal = async (req, res) => {
  try {

    const total = await tradeModel.aggregate([
      
      {
        $group: {
          _id: {
            type: "$type",
            currency: "$currency"
          },
          totalAmount: { $sum: "$amount" }
        }
      },
      {
        $project: {
          _id: 0,
          type: "$_id.type",
          currency: "$_id.currency",
          totalAmount: 1
        }
      }
    ]);

    if (total.length === 0) {
      return res.json({
        success: false,
        message: "امروز هیچ معامله‌ای ثبت نشده"
      });
    }

    res.json({
      success: true,
      total
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};

export { search, addTrade, getPersonTrade , getTotal , getDaily , getAllTotal }
