import customerModel from "../models/customerModel.js";
import transactionModel from "../models/transactionModel.js";
import Product from "../models/product.js";
import mongoose from "mongoose";
import currencyModel from "../models/currencyModel.js";

const createTransaction = async (req, res) => {

  try {
    const { customerId, product, receipt, bellNumber, note } = req.body;


    if (!customerId) {
      return res.json({ success: false, message: "مشتری وجود ندارد" });
    }

    const customerData = await customerModel.findById(customerId);
    if (!customerData) {
      return res.json({ success: false, message: "مشتری یافت نشد" });
    }

    const { customerName, phone } = customerData;


    if (!product || !Array.isArray(product) || product.length === 0) {
      return res.json({ success: false, message: "سبد خرید خالی است" });
    }


    if (!receipt) {
      return res.json({ success: false, message: "رسید موجود نیست" });
    }


    const today = new Date().toISOString().split("T")[0];
    const rate = await currencyModel.findOne({ date: today });

    if (!rate) {
      return res.json({
        success: false,
        message: "نرخ ارز برای امروز موجود نیست"
      });
    }


    for (const item of product) {
      const prod = await Product.findById(item.productId);

      if (!prod) {
        throw new Error(`محصول با آی‌دی ${item.productId} پیدا نشد`);
      }

      if (prod.isSold) {
        throw new Error(`محصول ${prod.productName} قبلاً فروخته شده`);
      }

      prod.isSold = true;
      await prod.save();
    }
    let isDollarSale = false

    for (const item of product) {

      if (item.salePrice.currency === "دالر") {
        isDollarSale = true
        item.salePrice.price *= rate.usdToAfn;
        item.salePrice.currency = "افغانی";
      }
    }

    if (isDollarSale) {
      receipt.totalAmount *= rate.usdToAfn
      receipt.paidAmount *= rate.usdToAfn
      receipt.remainingAmount *= rate.usdToAfn
    }

    // ساخت ترانسکشن
    const newTransaction = new transactionModel({
      customerId,
      customerName,
      customerPhone: phone,
      product,
      receipt,
      bellNumber,
      note
    });

    const savedTransaction = await newTransaction.save();

    console.log("SAVED:", savedTransaction);

    res.json({
      success: true,
      message: "ترانسکشن موفقانه اجرا شد",
      data: savedTransaction
    });

  } catch (error) {
    console.error("ERROR:", error);
    res.json({
      success: false,
      message: error.message
    });
  }
};


const getLoanTransaction = async (req, res) => {
  try {
    const loans = await transactionModel.aggregate([

      {
        $match: {
          "receipt.remainingAmount": { $gt: 0 }
        }
      },


      {
        $group: {
          _id: "$customerId",
          customerId: { $first: "$customerId" },
          transactionId: { $first: "$_id" },
          customerName: { $first: "$customerName" },
          customerPhone: { $first: "$customerPhone" },
          totalAmount: { $sum: "$receipt.totalAmount" },
          totalLoan: { $sum: "$receipt.remainingAmount" },
          totalPaid: { $sum: "$receipt.paidAmount" },
          totalDiscount: { $sum: "$receipt.discount" },
          transactionsCount: { $sum: 1 }
        }
      },

      {
        $sort: { totalLoan: -1 }
      }
    ]);

    if (loans.length === 0) {
      return res.json({
        success: false,
        message: "هیچ مشتری قرضدار نیست"
      });
    }

    res.json({
      success: true,
      loans
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false });
  }
};

const getSaleReport = async (req, res) => {

  try {
    const sold = await transactionModel.aggregate([

      { $unwind: "$product" },

      {
        $group: {
          _id: "product.barcode",
          totalGram: { $sum: "$product.gram" },
          totalAMount: { $sum: "$product.price" }
        }
      },

      {
        $project: {
          _id: 0,
          totalGram: 1,
          totalAMount: 1
        }
      }
    ]);

    if (sold.length === 0) {
      return res.json({
        success: false,
        message: "هیچ فروشی صورت نگرفته"
      });
    }

    res.json({
      success: true,
      sold
    });

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
};

const getAllSaleReport = async (req, res) => {
  try {

    const { dateFrom, dateTo } = req.query;

    let filter = {};

    if (dateFrom && dateTo) {
      const startOfDay = new Date(dateFrom);
      startOfDay.setHours(0, 0, 0, 0);

      const endOfDay = new Date(dateTo);
      endOfDay.setHours(23, 59, 59, 999);

      filter.createdAt = {
        $gte: startOfDay,
        $lte: endOfDay
      };
    }

    const transactions = await transactionModel.find(filter);

    if (transactions.length === 0) {
      return res.json({
        success: false,
        message: "هیچ فروشی صورت نگرفته"
      });
    }

    const today = new Date().toISOString().split("T")[0];
    const rate = await currencyModel.findOne({ date: today });

    let allProducts = [];
    let totalGram = 0;
    let totalAmount = 0;
    let remainAmount = 0;
    let totalDiscount = 0;

    for (const trx of transactions) {

      const isDollarSale = trx.product.some(
        p => p.salePrice?.currency === "دالر"
      );

      if (isDollarSale) {
        if (!rate) {
          return res.json({
            success: false,
            message: "نرخ امروز دالر ثبت نشده"
          });
        }
        totalAmount += trx.receipt.totalAmount * rate.usdToAfn;
        remainAmount += trx.receipt.remainingAmount * rate.usdToAfn
        totalDiscount += trx.receipt.discount * rate.usdToAfn
      } else {
        totalAmount += trx.receipt.totalAmount;
        remainAmount += trx.receipt.remainingAmount
        totalDiscount += trx.receipt.discount
      }


      trx.product.forEach(p => {
        allProducts.push({
          name: p.productName,
          barcode: p.barcode,
          gram: p.gram,
          price: p.salePrice
        });

        totalGram += p.gram;
      });
    }

    res.json({
      success: true,
      totalProducts: allProducts.length,
      totalGram,
      totalAmount,
      remainAmount,
      totalDiscount,
      products: allProducts
    });

  } catch (error) {
    console.log(error);
    res.json({success: false, message: error.message});
  }
};

const getSoldProduct = async (req, res) => {
  try {
    const soldProduct = await transactionModel.aggregate([
      {
        $unwind: '$product'
      },
      {
        $project: {
          _id: 0,
          transactionId: "$_id",
          customerName: 1,
          customerPhone: 1,
          productName: "$product.productName",
          productId: "$product.productId",
          salePrice: "$product.salePrice",
          gram: "$product.gram",
          karat: "$product.karat",
          barcode: "$product.barcode",
          image: "$product.image",
          createdAt: 1
        }
      },
      {
        $sort: { createdAt: -1 }
      }

    ])

    if (soldProduct.length === 0) {
      return res.json({ success: false, message: "هیچ فروشی صورت نگرفته" })
    }

    res.json({ success: true, soldProduct })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const dailyReport = async (req, res) => {

  const startOfDay = new Date();
  startOfDay.setHours(0, 0, 0, 0);

  const endOfDay = new Date();
  endOfDay.setHours(23, 59, 59, 999);

  try {

    const daily = await transactionModel.aggregate([
      {
        $match: {
          createdAt: {
            $gte: startOfDay,
            $lte: endOfDay
          }

        }
      },
      {
        $unwind: "$product"
      },
      {
        $group: {
          _id: "$_id",
          customerName: { $first: "$customerName" },
          customerPhone: { $first: "$customerPhone" },
          discount: { $first: "$receipt.discount" },
          totalAmount: { $first: "$receipt.totalAmount" },
          paidAmount: { $first: "$receipt.paidAmount" },
          remainingAmount: { $first: "$receipt.remainingAmount" },
          bellNumber: { $first: "$bellNumber" },
          date: { $first: "$createdAt" },

          product: {
            $push: {
              productId: "$product.productId",
              name: "$product.productName",
              purchase: "$product.purchasePriceToAfn",
              sold: "$product.salePrice.price",
              barcode: "$product.barcode",
              gram: "$product.gram",
              karat: "$product.karat"
            }
          }

        }
      },
      {
        $sort: {
          date: -1
        }
      }
    ])

    if (daily.length === 0) {
      return res.json({ success: false, message: "دیتا وجود ندارد" })
    }
    res.json({ success: true, daily })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}

const dailySumReport = async (req, res) => {

  const startOfDay = new Date();
  startOfDay.setHours(0, 0, 0, 0);

  const endOfDay = new Date();
  endOfDay.setHours(23, 59, 59, 999);

  try {
    const transactions = await transactionModel.find({
      createdAt: {
        $gte: startOfDay,
        $lte: endOfDay
      }
    });

    if (transactions.length === 0) {
      return res.json({
        success: false,
        message: "هیچ فروشی صورت نگرفته"
      });
    }

    const today = new Date().toISOString().split("T")[0];
    const rate = await currencyModel.findOne({ date: today });

    let allProducts = [];
    let totalPurchase = 0;
    let totalGram = 0;
    let totalAmount = 0;
    let remainAmount = 0;
    let totalDiscount = 0;

    for (const trx of transactions) {

      const isDollarSale = trx.product.some(
        p => p.salePrice?.currency === "دالر"
      );

      if (isDollarSale) {
        if (!rate) {
          return res.json({
            success: false,
            message: "نرخ امروز دالر ثبت نشده"
          });
        }
        totalAmount += trx.receipt.totalAmount * rate.usdToAfn;
        remainAmount += trx.receipt.remainingAmount * rate.usdToAfn
        totalDiscount += trx.receipt.discount * rate.usdToAfn
      } else {
        totalAmount += trx.receipt.totalAmount;
        remainAmount += trx.receipt.remainingAmount
        totalDiscount += trx.receipt.discount
      }


      trx.product.forEach(p => {
        allProducts.push({
          name: p.productName,
          barcode: p.barcode,
          gram: p.gram,
          price: p.salePrice
        });

        totalGram += p.gram;
        totalPurchase += p.purchasePriceToAfn
      });
    }

    res.json({
      success: true,
      totalProducts: allProducts.length,
      totalGram,
      totalPurchase,
      totalAmount,
      remainAmount,
      totalDiscount,
      products: allProducts
    });

  } catch (error) {
    console.log(error);
    res.json({
      success: false,
      message: error.message
    });
  }
}


const deleteProductFromTransaction = async (req, res) => {
  try {
    const { transactionId, productId } = req.query


    const transaction = await transactionModel.findById(transactionId);
    if (!transaction) {
      return res.json({
        success: false,
        message: "ترانسکشن پیدا نشد"
      });
    }



    const productIndex = transaction.product.findIndex(
      p => p.productId.toString() === productId
    );

    if (productIndex === -1) {
      return res.json({
        success: false,
        message: "محصول در ترانسکشن وجود ندارد"
      });
    }

    const removedProduct = transaction.product[productIndex];

    await Product.findByIdAndUpdate(removedProduct.productId, {
      isSold: false
    });


    const productAmount = removedProduct.salePrice.price;


    transaction.product.splice(productIndex, 1);


    transaction.set('receipt.totalAmount', Number(transaction.receipt.totalAmount) - Number(productAmount));
    transaction.set('receipt.paidAmount', Number(transaction.receipt.paid) - Number(productAmount));
    transaction.set('receipt.totalQuantity', Number(transaction.receipt.totalQuantity) - 1);
    transaction.set('receipt.remainingAmount', Number(transaction.receipt.totalAmount) - Number(transaction.receipt.paidAmount));

    if (transaction.receipt.remainingAmount < 0) {
      transaction.set('receipt.remainingAmount', 0);
    }


    if (transaction.receipt.remainingAmount < 0) {
      transaction.receipt.remainingAmount = 0;
    }

    // 6️⃣ اگر محصولی نمانده → حذف کامل ترانسکشن
    if (transaction.product.length === 0) {
      await transaction.deleteOne();
      return res.json({
        success: true,
        message: "تمام محصولات حذف شد، ترانسکشن پاک گردید"
      });
    }

    // 7️⃣ ذخیره تغییرات
    await transaction.save();

    res.json({
      success: true,
      message: "محصول با موفقیت برگشت داده شد",
      transaction
    });

  } catch (error) {
    console.error(error);
    res.status(500).json({
      success: false,
      message: "خطای سرور"
    });
  }
};

const getLoanToPay = async (req, res) => {

  try {

    const { customerId } = req.query

    if (!mongoose.Types.ObjectId.isValid(customerId)) {
      return res.json({
        success: false,
        message: "transactionId نامعتبر است"
      });
    }

    const loan = await transactionModel.aggregate([
      {
        $match: {
          customerId: new mongoose.Types.ObjectId(customerId)
        }
      },
      {
        $unwind: "$product"
      },
      {
        $group: {
          _id: "$_id",
          customerName: { $first: "$customerName" },
          customerPhone: { $first: "$customerPhone" },
          discount: { $first: "$receipt.discount" },
          totalAmount: { $first: "$receipt.totalAmount" },
          paidAmount: { $first: "$receipt.paidAmount" },
          remainingAmount: { $first: "$receipt.remainingAmount" },
          bellNumber: { $first: "$bellNumber" },
          date: { $first: "$createdAt" },

          product: {
            $push: {
              productId: "$product.productId",
              name: "$product.productName",
              purchase: "$product.purchasePriceToAfn",
              sold: "$product.salePrice.price",
              barcode: "$product.barcode",
              gram: "$product.gram",
              karat: "$product.karat"
            }
          }

        }
      },
      {
        $sort: {
          date: -1
        }
      }
    ])

    if (loan.length === 0) {
      return res.json({ success: false, message: "دیتا وجود ندارد" })
    }
    res.json({ success: true, loan })

  } catch (error) {
    console.log(error);
    res.json({ success: false, message: error.message });
  }
}



const payLoan = async (req, res) => {
  try {
    const { amount, currency } = req.body;
    const { transactionId } = req.query;

    if (!transactionId) {
      return res.json({ success: false, message: "transactionId موجود نیست" });
    }

    const paid = Number(amount);
    if (!paid || paid <= 0) {
      return res.json({ success: false, message: "مبلغ پرداخت نادرست است" });
    }

    // دریافت نرخ ارز
    const today = new Date().toISOString().split("T")[0];
    const rate = await currencyModel.findOne({ date: today });
    if (!rate) {
      return res.json({ success: false, message: "نرخ ارز امروز موجود نیست" });
    }

    const paidAmount = currency === "دالر" ? paid * rate.usdToAfn : paid;

    // ابتدا تراکنش را پیدا می‌کنیم
    const transaction = await transactionModel.findById(transactionId);
    if (!transaction) {
      return res.json({ success: false, message: "ترانسکشن یافت نشد" });
    }

    const remaining = transaction.receipt.remainingAmount;

    if (remaining <= 0) {
      return res.json({ success: false, message: "قبلاً تسویه شده است" });
    }

    // فقط همانی که باقی‌مانده است کم شود
    const paidToApply = Math.min(paidAmount, remaining);

    // آپدیت تراکنش
    const updatedTransaction = await transactionModel.findByIdAndUpdate(
      transactionId,
      {
        $inc: {
          "receipt.paidAmount": paidToApply,
          "receipt.remainingAmount": -paidToApply
        }
      },
      { new: true }
    );

    res.json({
      success: true,
      message: "پرداخت موفقانه ثبت شد",
      transaction: updatedTransaction
    });

  } catch (error) {
    console.error(error);
    res.json({ success: false, message: error.message });
  }
};


export { createTransaction, getLoanTransaction, getAllSaleReport, getSoldProduct, dailyReport, dailySumReport, deleteProductFromTransaction, getLoanToPay, payLoan };
