// This file is part of the bwip-js project available at:
//
//    http://metafloor.github.io/bwip-js
//
// Copyright (c) 2011-2025 Mark Warren
//
// This file contains code automatically generated from:
// Barcode Writer in Pure PostScript - Version 2025-10-24
// Copyright (c) 2004-2024 Terry Burton
//
// The MIT License
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
"use strict";

(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        define([], factory);
    } else if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else {
        root.bwipjs = factory();
    }
}(typeof self !== 'undefined' ? self : this, function () {
// exports.js
const BWIPJS_VERSION = '4.8.0 (2025-11-11)';

// Context insensitive canvas element test.
function IsCanvas(elt) {
    return elt && /HTMLCanvasElement|OffscreenCanvas/.test(Object.getPrototypeOf(elt).constructor.name);
}
// bwipjs.toCanvas(canvas, options)
// bwipjs.toCanvas(options, canvas)
//
// Uses the built-in canvas drawing.
//
// `canvas` can be an HTMLCanvasElement|OffscreenCanvas or
// an ID string or unique selector string.
// `options` are a bwip-js/BWIPP options object.
//
// This function is synchronous and throws on error.
//
// Returns the canvas element.
function ToCanvas(cvs, opts) {
    if (typeof opts == 'string' || IsCanvas(opts)) {
        let tmp = cvs;
        cvs = opts;
        opts = tmp;
    }
    return _ToAny(bwipp_lookup(opts.bcid), opts, cvs);
}
// Entry point for the symbol-specific exports
//
// Polymorphic internal interface
// _ToAny(encoder, string, opts) : HTMLCanvasElement
// _ToAny(encoder, HTMLCanvasElement, opts) : HTMLCanvasElement
// _ToAny(encoder, OffscreenCanvas, opts) : OffscreenCanvas
// _ToAny(encoder, opts, string) : HTMLCanvasElement
// _ToAny(encoder, opts, OffscreenCanvas) : OffscreenCanvas
// _ToAny(encoder, opts, drawing) : any
//
// 'string` can be either an `id` or query selector returning a single canvas element.
function _ToAny(encoder, opts, drawing) {
    if (typeof opts == 'string') {
        var canvas = document.getElementById(opts) || document.querySelector(opts);
        if (!IsCanvas(canvas)) {
            throw new Error('bwipjs: `' + opts + '`: not a canvas');
        }
        opts = drawing;
        drawing = DrawingCanvas(canvas);
    } else if (IsCanvas(opts)) {
        var canvas = opts;
        opts = drawing;
        drawing = DrawingCanvas(canvas);
    } else if (typeof drawing == 'string') {
        var canvas = document.getElementById(drawing) || document.querySelector(drawing);
        if (!IsCanvas(canvas)) {
            throw new Error('bwipjs: `' + drawing + '`: not a canvas');
        }
        drawing = DrawingCanvas(canvas);
    } else if (IsCanvas(drawing)) {
        drawing = DrawingCanvas(drawing);
    } else if (!drawing || typeof drawing != 'object' || !drawing.init) {
        throw new Error('bwipjs: not a canvas or drawing object');
    }
    return _Render(encoder, opts, drawing);
}

// bwipjs.toSVG(options)
//
// Uses the built-in svg drawing interface.
//
// `options` are a bwip-js/BWIPP options object.
//
// This function is synchronous and throws on error.
//
// Returns a string containing a fully qualified SVG definition,
// including the natural width and height of the image, in pixels:
//
//  <svg viewBox="0 0 242 200" xmlns="http://www.w3.org/2000/svg">
//   ...
//  </svg>
//
// Available on all platforms.
function ToSVG(opts) {
    return _Render(bwipp_lookup(opts.bcid), opts, DrawingSVG());
}

function FixupOptions(opts) {
    var scale   = opts.scale || 2;
    var scaleX  = +opts.scaleX || scale;
    var scaleY  = +opts.scaleY || scaleX;

    // Fix up padding.
    opts.paddingleft = padding(opts.paddingleft, opts.paddingwidth, opts.padding, scaleX);
    opts.paddingright = padding(opts.paddingright, opts.paddingwidth, opts.padding, scaleX);
    opts.paddingtop = padding(opts.paddingtop, opts.paddingheight, opts.padding, scaleY);
    opts.paddingbottom = padding(opts.paddingbottom, opts.paddingheight, opts.padding, scaleY);

    // We override BWIPP's background color functionality.  If in CMYK, convert to RRGGBB so
    // the drawing interface is consistent.  Likewise, if in CSS-style #rgb or #rrggbb.
    if (opts.backgroundcolor) {
        var bgc = ''+opts.backgroundcolor;
        if (/^[0-9a-fA-F]{8}$/.test(bgc)) {
            var c = parseInt(bgc.substr(0,2), 16) / 255;
            var m = parseInt(bgc.substr(2,2), 16) / 255;
            var y = parseInt(bgc.substr(4,2), 16) / 255;
            var k = parseInt(bgc.substr(6,2), 16) / 255;
            var r = Math.floor((1-c) * (1-k) * 255).toString(16);
            var g = Math.floor((1-m) * (1-k) * 255).toString(16);
            var b = Math.floor((1-y) * (1-k) * 255).toString(16);
            opts.backgroundcolor = (r.length == 1 ? '0' : '') + r +
                                   (g.length == 1 ? '0' : '') + g +
                                   (b.length == 1 ? '0' : '') + b;
        } else {
            if (bgc[0] == '#') {
                bgc = bgc.substr(1);
            }
            if (/^[0-9a-fA-F]{6}$/.test(bgc)) {
                opts.backgroundcolor = bgc;
            } else if (/^[0-9a-fA-F]{3}$/.test(bgc)) {
                opts.backgroundcolor = bgc[0] + bgc[0] + bgc[1] + bgc[1] + bgc[2] + bgc[2];
            } else {
                throw new Error('bwip-js: invalid backgroundcolor: ' + opts.backgroundcolor);
            }
        }
    }

    return opts;

    // a is the most specific padding value, e.g. paddingleft
    // b is the next most specific value, e.g. paddingwidth
    // c is the general padding value.
    // s is the scale, either scalex or scaley
    function padding(a, b, c, s) {
        if (a != null) {
            a = a >>> 0;
            return a*s >>> 0;
        }
        if (b != null) {
            b = b >>> 0;
            return b*s >>> 0;
        }
        c = c >>> 0;
        return (c*s >>> 0) || 0;
    }
}

var BWIPJS_OPTIONS = {
    bcid:1,
    text:1,
    scale:1,
    scaleX:1,
    scaleY:1,
    rotate:1,
    padding:1,
    paddingwidth:1,
    paddingheight:1,
    paddingtop:1,
    paddingleft:1,
    paddingright:1,
    paddingbottom:1,
    backgroundcolor:1,
};

// bwipjs.render(options, drawing)
//
// Renders a barcode using the provided drawing object.
//
// This function is synchronous and throws on error.
//
// Browser and nodejs usage.
function Render(options, drawing) {
    return _Render(bwipp_lookup(options.bcid), options, drawing);
}

// Called by the public exports
function _Render(encoder, options, drawing) {
    var text = options.text;
    if (!text) {
        throw new ReferenceError('bwip-js: bar code text not specified.');
    }

    // setopts() is optional on the drawing object.
    FixupOptions(options);
    drawing.setopts && drawing.setopts(options);

    // Set the bwip-js defaults
    var scale   = options.scale || 2;
    var scaleX  = +options.scaleX || scale;
    var scaleY  = +options.scaleY || scaleX;
    var rotate  = options.rotate || 'N';

    // Create a barcode writer object.  This is the interface between
    // the low-level BWIPP code, the bwip-js graphics context, and the
    // drawing interface.
    var bw = new BWIPJS(drawing);

    // Set the BWIPP options
    var bwippopts = {};
    for (var id in options) {
        if (!BWIPJS_OPTIONS[id]) {
            bwippopts[id] = options[id];
        }
    }

    // Fix a disconnect in the BWIPP rendering logic
    if (bwippopts.alttext) {
        bwippopts.includetext = true;
    }
    // We use mm rather than inches for height - except pharmacode2 height
    // which is already in mm.
    if (+bwippopts.height && encoder != bwipp_pharmacode2) {
        bwippopts.height = bwippopts.height / 25.4 || 0.5;
    }
    // Likewise, width
    if (+bwippopts.width) {
        bwippopts.width = bwippopts.width / 25.4 || 0;
    }

    // Scale the image
    bw.scale(scaleX, scaleY);

    // Call into the BWIPP cross-compiled code and render the image.
    bwipp_encode(bw, encoder, text, bwippopts);

    // Returns whatever drawing.end() returns, or `false` if nothing rendered.
    return bw.render();
}

// bwipjs.raw(options)
// bwipjs.raw(bcid, text, opts-string)
//
// Invokes the low level BWIPP code and returns the raw encoding data.
//
// This function is synchronous and throws on error.
//
// Browser and nodejs usage.
function ToRaw(bcid, text, options) {
    if (arguments.length == 1) {
        options = bcid;
        bcid = options.bcid;
        text = options.text;
    }

    // The drawing interface is just needed for the pre-init() calls.
    // Don't need to fixup the drawing specific options.
    var drawing = DrawingBuiltin();
    drawing.setopts(options);

    var bw = new BWIPJS(drawing);
    var stack = bwipp_encode(bw, bwipp_lookup(bcid), text, options, true);

    // bwip-js uses Maps to emulate PostScript dictionary objects; but Maps
    // are not a typical/expected return value.  Convert to plain-old-objects.
    var ids = { pixs:1, pixx:1, pixy:1, sbs:1, bbs:1, bhs:1, width:1, height:1 };
    for (var i = 0; i < stack.length; i++) {
        var elt = stack[i];
        if (elt instanceof Map) {
            var obj = {};
            // Could they make Maps any harder to iterate over???
            for (var keys = elt.keys(), size = elt.size, k = 0; k < size; k++) {
                var id = keys.next().value;
                if (ids[id]) {
                    var val = elt.get(id);
                    if (val instanceof Array) {
                        // The postscript arrays have extra named properties
                        // to emulate array views.  Return cleaned up arrays.
                        obj[id] = val.b.slice(val.o, val.o + val.length);
                    } else {
                        obj[id] = val;
                    }
                }
            }
            stack[i] = obj;
        } else {
            // This should never exec...
            stack.splice(i--, 1);
        }
    }
    return stack;
}
// bwip-js // Barcode Writer in Pure JavaScript
// https://github.com/metafloor/bwip-js
//
// This code was automatically generated from:
// Barcode Writer in Pure PostScript - Version 2025-10-24
//
// Copyright (c) 2011-2025 Mark Warren
// Copyright (c) 2004-2024 Terry Burton
//
// Licensed MIT. See the LICENSE file in the bwip-js root directory.
// bwip-js/barcode-hdr.js
//
// This code is injected above the cross-compiled barcode.js.

// The BWIPJS object (graphics interface)
var $$ = null;
var $j = 0; // stack pointer
var $k = []; // operand stack
var $_ = {}; // base of the dictionary stack

// Aliases from Math ops
const $abs = Math.abs;
const $ceil = Math.ceil;
const $floor = Math.floor;
const $log = Math.log;
const $pow = Math.pow
const $round = Math.round;
const $sqrt = Math.sqrt;

// Code instrumenting
const $metrics = {};

// Array ctor
//  $a()    : Build a new array up to the Infinity-marker on the stack.
//  $a(arr) : Convert native array to a "view" of the array.
//  $a(len) : Create a new array of length `len`
function $a(a) {
    if (!arguments.length) {
        for (var i = $j - 1; i >= 0 && $k[i] !== Infinity; i--);
        if (i < 0) {
            throw new Error('array-marker-not-found');
        }
        a = $k.splice(i + 1, $j - 1 - i);
        $j = i;
    } else if (!(a instanceof Array)) {
        a = new Array(+arguments[0]);
        for (var i = 0, l = a.length; i < l; i++) {
            a[i] = null;
        }
    }
    a.b = a; // base array
    a.o = 0; // offset into base
    return a;
}

// dict ctor
//  $d() : look for the Infinity marker on the stack
function $d() {
    // Build the dictionary in the order the keys/values were pushed so enumeration
    // occurs in the correct sequence.
    for (var mark = $j - 1; mark >= 0 && $k[mark] !== Infinity; mark -= 2) {
        if ($k[mark - 1] === Infinity) {
            throw new Error('dict-malformed-stack');
        }
    }
    if (mark < 0) {
        throw new Error('dict-marker-not-found');
    }
    var d = new Map;
    for (var i = mark + 1; i < $j; i += 2) {
        // Unlike javascript, postscript dict keys differentiate between
        // numbers and the string representation of a number.
        var k = $k[i]; // "key" into the dict entry
        var t = typeof k;
        if (t == 'number' || t == 'string') {
            d.set(k, $k[i + 1]);
        } else if (k instanceof Uint8Array) {
            d.set($z(k), $k[i + 1]);
        } else {
            throw new Error('dict-not-a-valid-key(' + k + ')');
        }
    }
    $j = mark;
    return d;
}

// string ctor
//  s(number)   : create zero-filled string of number-length
//  s(string)   : make a copy of the string
//  s(uint8[])  : make a copy of the string
//
// Returns a Uint8Array-string.
function $s(v) {
    var t = typeof v;
    if (t === 'number') {
        return new Uint8Array(v);
    }
    if (t !== 'string') {
        v = '' + v;
    }
    var s = new Uint8Array(v.length);
    for (var i = 0, l = v.length; i < l; i++) {
        s[i] = v.charCodeAt(i);
    }
    return s;
}

// ... n c roll
function $r(n, c) {
    if ($j < n) {
        throw new Error('roll: --stack-underflow--');
    }
    if (!c) {
        return;
    }
    if (c < 0) {
        var t = $k.splice($j - n, -c);
    } else {
        var t = $k.splice($j - n, n - c);
    }
    $k.splice.apply($k, [$j - t.length, 0].concat(t));
}

// Primarily designed to convert uint8-string to string, but will call the
// the toString() method on any value.
function $z(s) {
    if (s instanceof Uint8Array) {
        return String.fromCharCode.apply(null, s);
    }
    return '' + s;
}

// Copies source to dest and returns a view of just the copied characters
function $strcpy(dst, src) {
    if (typeof dst === 'string') {
        dst = $s(dst);
    }
    if (src instanceof Uint8Array) {
        for (var i = 0, l = src.length; i < l; i++) {
            dst[i] = src[i];
        }
    } else {
        for (var i = 0, l = src.length; i < l; i++) {
            dst[i] = src.charCodeAt(i);
        }
    }
    return src.length < dst.length ? dst.subarray(0, src.length) : dst;
}

// Copies source to dest and should (but doesn't) return a view of just the copied elements
function $arrcpy(dst, src) {
    for (var i = 0, l = src.length; i < l; i++) {
        dst[i] = src[i];
    }
    dst.length = src.length;
    return dst;
}

// cvs operator - convert a value to its string representation
//  s : string to store into
//  v : any value
function $cvs(s, v) {
    var t = typeof v;
    if (t == 'number' || t == 'boolean' || v === null) {
        v = '' + v;
    } else if (t !== 'string') {
        v = '--nostringval--';
    }
    for (var i = 0, l = v.length; i < l; i++) {
        s[i] = v.charCodeAt(i);
    }
    $k[$j++] = i < s.length ? s.subarray(0, i) : s;
}
// cvi operator - converts a numeric string value to integer/real.
function $cvi(s) {
    if (s instanceof Uint8Array) {
        // nul-chars on the end of a string are ignored by postscript but cause javascript
        // to return a zero result.
        return $floor(String.fromCharCode.apply(null, s).replace(/\0+$/, ''));
    }
    return $floor('' + s);
}

// cvrs operator - convert a number to a radix string
//  s : string to store into
//  n : number
//  r : radix
function $cvrs(s, n, r) {
    return $strcpy(s, (~~n).toString(r).toUpperCase());
}

// get operator
//  s : source
//  k : key
function $get(s, k) {
    if (s instanceof Uint8Array) {
        return s[k];
    }
    if (typeof s === 'string') {
        return s.charCodeAt(k);
    }
    if (s instanceof Array) {
        return s.b[s.o + k];
    }
    // Map or Object - need a string key
    if (k instanceof Uint8Array) {
        k = $z(k);
    }
    if (s instanceof Map) {
        return s.get(k);
    }
    return s[k];
}

// put operator
//  d : dest
//  k : key
//  v : value
function $put(d, k, v) {
    if (d instanceof Uint8Array) {
        d[k] = v;
    } else if (d instanceof Array) {
        d.b[d.o + k] = v;
    } else if (typeof d == 'object') {
        if (k instanceof Uint8Array) {
            d.set($z(k), v);
        } else {
            d.set(k, v);
        }
    } else {
        throw new Error('put-not-writable-' + (typeof d));
    }
}

// getinterval operator
//  s : src
//  o : offset
//  l : length
function $geti(s, o, l) {
    if (s instanceof Uint8Array) {
        return s.subarray(o, o + l);
    }
    if (s instanceof Array) {
        var a = new Array(l);
        a.b = s.b; // base array
        a.o = s.o + o; // offset into base
        return a;
    }
    // Must be a string
    return s.substr(o, l);
}

// putinterval operator
//  d : dst
//  o : offset
//  s : src
function $puti(d, o, s) {
    if (d instanceof Uint8Array) {
        if (typeof s == 'string') {
            for (var i = 0, l = s.length; i < l; i++) {
                d[o + i] = s.charCodeAt(i);
            }
        } else {
            // When both d and s are the same, we want to copy
            // backwards, which works for the general case as well.
            for (var i = s.length - 1; i >= 0; i--) {
                d[o + i] = s[i];
            }
        }
    } else if (d instanceof Array) {
        // Operate on the base arrays
        var darr = d.b;
        var doff = o + d.o;
        var sarr = s.b;
        var soff = s.o;

        for (var i = 0, l = s.length; i < l; i++) {
            darr[doff + i] = sarr[soff + i];
        }
    } else {
        throw new Error('putinterval-not-writable-' + (typeof d));
    }
}

// type operator
function $type(v) {
    // null can be mis-typed - get it out of the way
    if (v == null) {
        return 'nulltype';
    }
    var t = typeof v;
    if (t == 'number') {
        return v % 1 ? 'realtype' : 'integertype';
    }
    if (t == 'boolean') {
        return 'booleantype';
    }
    if (t == 'string' || v instanceof Uint8Array) {
        return 'stringtype';
    }
    if (t == 'function') {
        return 'operatortype';
    }
    if (v instanceof Array) {
        return 'arraytype';
    }
    return 'dicttype';
    // filetype
    // fonttype
    // gstatetype
    // marktype (v === Infinity)
    // nametype
    // savetype
}

// anchorsearch operator
//      string seek anchorsearch suffix seek true %if-found
//                               string false     %if-not-found
function $anchorsearch(str, seek) {
    if (!(str instanceof Uint8Array)) {
        str = $s(str);
    }
    var i = 0,
        ls = str.length,
        lk = seek.length;

    // Optimize for single characters.
    if (lk == 1) {
        var cd = seek instanceof Uint8Array ? seek[0] : seek.charCodeAt(0);
        i = str[0] == cd ? 1 : ls;
    } else if (seek.length <= ls) {
        // Slow path,
        if (!(seek instanceof Uint8Array)) {
            seek = $s(seek);
        }
        for (; i < lk && str[i] == seek[i]; i++);
    }
    if (i == lk) {
        $k[$j++] = str.subarray(lk);
        $k[$j++] = str.subarray(0, lk);
        $k[$j++] = true;
    } else {
        $k[$j++] = str;
        $k[$j++] = false;
    }
}

// search operator
//      string seek search suffix match prefix true %if-found
//                         string false             %if-not-found
function $search(str, seek) {
    if (!(str instanceof Uint8Array)) {
        str = $s(str);
    }
    var ls = str.length;

    // Virtually all uses of search in BWIPP are for single-characters.
    // Optimize for that case.
    if (seek.length == 1) {
        var lk = 1;
        var cd = seek instanceof Uint8Array ? seek[0] : seek.charCodeAt(0);
        for (var i = 0; i < ls && str[i] != cd; i++);
    } else {
        // Slow path,
        if (!(seek instanceof Uint8Array)) {
            seek = $s(seek);
        }
        var lk = seek.length;
        var cd = seek[0];
        for (var i = 0; i < ls && str[i] != cd; i++);
        while (i < ls) {
            for (var j = 1; j < lk && str[i + j] === seek[j]; j++);
            if (j === lk) {
                break;
            }
            for (i++; i < ls && str[i] != cd; i++);
        }
    }
    if (i < ls) {
        $k[$j++] = str.subarray(i + lk);
        $k[$j++] = str.subarray(i, i + lk);
        $k[$j++] = str.subarray(0, i);
        $k[$j++] = true;
    } else {
        $k[$j++] = str;
        $k[$j++] = false;
    }
}

// The callback is omitted when forall is being used just to push onto the
// stack.  The callback normally returns undefined.  A return of true means break.
function $forall(o, cb) {
    if (o instanceof Uint8Array) {
        for (var i = 0, l = o.length; i < l; i++) {
            $k[$j++] = o[i];
            if (cb && cb()) break;
        }
    } else if (o instanceof Array) {
        // The array may be a view.
        for (var a = o.b, i = o.o, l = o.o + o.length; i < l; i++) {
            $k[$j++] = a[i];
            if (cb && cb()) break;
        }
    } else if (typeof o === 'string') {
        for (var i = 0, l = o.length; i < l; i++) {
            $k[$j++] = o.charCodeAt(i);
            if (cb && cb()) break;
        }
    } else if (o instanceof Map) {
        for (var keys = o.keys(), i = 0, l = o.size; i < l; i++) {
            var id = keys.next().value;
            $k[$j++] = id;
            $k[$j++] = o.get(id);
            if (cb && cb()) break;
        }
    } else {
        for (var id in o) {
            $k[$j++] = id;
            $k[$j++] = o[id];
            if (cb && cb()) break;
        }
    }
}

function $cleartomark() {
    while ($j > 0 && $k[--$j] !== Infinity);
}

function $counttomark() {
    for (var i = $j - 1; i >= 0 && $k[i] !== Infinity; i--);
    return $j - i - 1;
}

function $aload(a) {
    for (var i = 0, l = a.length, b = a.b, o = a.o; i < l; i++) {
        $k[$j++] = b[o + i];
    }
    // This push has been optimized out.  See $.aload() in psc.js.
    //$k[$j++] = a;
}

function $astore(a) {
    for (var i = 0, l = a.length, b = a.b, o = a.o + l - 1; i < l; i++) {
        b[o - i] = $k[--$j];
    }
    $k[$j++] = a;
}

function $eq(a, b) {
    if (typeof a === 'string' && typeof b === 'string') {
        return a == b;
    }
    if (a instanceof Uint8Array && b instanceof Uint8Array) {
        if (a.length != b.length) {
            return false;
        }
        for (var i = 0, l = a.length; i < l; i++) {
            if (a[i] != b[i]) {
                return false;
            }
        }
        return true;
    }
    if (a instanceof Uint8Array && typeof b === 'string' ||
        b instanceof Uint8Array && typeof a === 'string') {
        if (a instanceof Uint8Array) {
            a = $z(a);
        } else {
            b = $z(b);
        }
        return a == b;
    }
    return a == b;
}

function $ne(a, b) {
    return !$eq(a, b);
}

function $lt(a, b) {
    if (a instanceof Uint8Array) {
        a = $z(a);
    }
    if (b instanceof Uint8Array) {
        b = $z(b);
    }
    return a < b;
}

function $le(a, b) {
    if (a instanceof Uint8Array) {
        a = $z(a);
    }
    if (b instanceof Uint8Array) {
        b = $z(b);
    }
    return a <= b;
}

function $gt(a, b) {
    if (a instanceof Uint8Array) {
        a = $z(a);
    }
    if (b instanceof Uint8Array) {
        b = $z(b);
    }
    return a > b;
}

function $ge(a, b) {
    if (a instanceof Uint8Array) {
        a = $z(a);
    }
    if (b instanceof Uint8Array) {
        b = $z(b);
    }
    return a >= b;
}

function $an(a, b) { // and
    return (typeof a === 'boolean') ? a && b : a & b;
}

function $or(a, b) { // or
    return (typeof a === 'boolean') ? a || b : a | b;
}

function $xo(a, b) { // xor
    return (typeof a === 'boolean') ? a != b : a ^ b;
}

function $nt(a) {
    return typeof a == 'boolean' ? !a : ~a;
}
// emulate single-precision floating-point.  This is not Math.fround().
// More like ffloor()...
var $f = (function(fa) {
    return (v) => {
        //return Number.isInteger(v) ? v : (fa[0] = v, fa[0]);
        return (v | 0) == v ? v : (fa[0] = v, fa[0]);
    };
})(new Float32Array(1));

// This is a replacement for the BWIPP raiseerror function.
function bwipp_raiseerror() {
    var info = $k[--$j];
    var name = $k[--$j];
    if (typeof info == 'string' || info instanceof Uint8Array) {
        throw new Error($z(name) + ": " + $z(info));
    } else {
        $k[$j++] = info; // see mktests debugEqual
        // Match ghostscript output
        throw $z(name) + '\nAdditional information: ' + tostring(info);
    }

    function tostring(v) {
        if (v instanceof Array) {
            let s = '';
            for (let i = v.o, l = v.o + v.length; i < l; i++) {
                s += ' ' + tostring(v.b[i]);
            }
            return '[' + s.substr(1) + ']';
        } else if (v instanceof Uint8Array) {
            return String.fromCharCode.apply(String, v);
        } else if (v instanceof Map) {
            let s = '';
            for (const [key, val] of v) {
                s += ' ' + tostring(key) + ' ' + tostring(val);
            }
            return '<<' + s.substr(1) + '>>';
        } else if (v && typeof v == 'object') {
            let s = '';
            for (let id in v) {
                s += ' ' + tostring(id) + ' ' + tostring(v[id]);
            }
            return '<<' + s.substr(1) + '>>';
        } else {
            return '' + v;
        }
    }
}

// This is a replacement for the BWIPP processoptions function.
// We cannot use the BWIPP version due to two reasons:
// - legacy code allows strings to be numbers and numbers to be strings
// - in javascript, there is no way to tell the difference between a real
//   number that is an integer, and an actual integer.
//
// options currentdict processoptions exec -> options
function bwipp_processoptions() {
    var dict = $k[--$j];
    var opts = $k[$j - 1];
    if (opts instanceof Uint8Array) {
        opts = $z(opts);
    }
    if (typeof opts == 'string') {
        let vals = opts.trim().split(/ +/g)
        $k[$j - 1] = opts = new Map();
        for (let i = 0; i < vals.length; i++) {
            let pair = vals[i].split('=');
            if (pair.length == 1) {
                opts.set(pair[0], true);
            } else {
                opts.set(pair[0], pair[1]);
            }
        }
    }
    for (var id in dict) {
        var val;
        if (!opts.has(id)) {
            continue;
        }
        val = opts.get(id);
        var def = dict[id];
        var typ = typeof def;

        // null is a placeholder for realtype
        if (def == null || typ == 'number') {
            // This allows for numeric strings
            if (!isFinite(+val)) {
                throw new Error('bwipp.invalidOptionType: ' + id +
                    ': not a realtype: ' + val);
            }
            if (typeof val == 'string') {
                val = +val;
                opts.set(id, val);
            }
        } else if (typ == 'boolean') {
            if (val !== true && val !== false) {
                // In keeping with the ethos of javascript, allow a more relaxed
                // interpretation of boolean.
                if (val == null || (val | 0) === val) {
                    val = !!val;
                } else if (val == 'true') {
                    val = true;
                } else if (val == 'false') {
                    val = false;
                } else {
                    throw new Error('bwipp.invalidOptionType: ' + id +
                        ': not a booleantype: ' + val);
                }
                opts.set(id, val);
            }
        } else if (typ == 'string' || def instanceof Uint8Array) {
            // This allows numbers to be strings
            if (typeof val == 'number') {
                val = '' + val;
                opts.set(id, val);
            } else if ((id === 'extratext' || id === 'alttext') && typeof val === 'string') {
                // BWIPP 2025-06-13 introduced (alt|extra)textsubspace which 
                // allows replacing a marker character with space.  This
                // requires the text to be a uint8array otherwise we get
                //      Error: put-not-writable-string
                val = $s(val);
                opts.set(id, val);
            } else if (typeof val != 'string' && !(val instanceof Uint8Array)) {
                throw new Error('bwipp.invalidOptionType: ' + id +
                    ': not a stringtype: ' + val);
            }
        }
        // Set the option into the dictionary
        dict[id] = val;
    }
}

function bwipp_parseinput() {
    var $__ = $_; //#200
    $_ = Object.assign({}, $_); //#200
    $_.fncvals = $k[--$j]; //#204
    $_.barcode = $k[--$j]; //#205
    var _2 = 'parse'; //#207
    $_[_2] = $get($_.fncvals, _2); //#207
    delete $_.fncvals[_2]; //#207
    var _6 = 'parsefnc'; //#208
    $_[_6] = $get($_.fncvals, _6); //#208
    delete $_.fncvals[_6]; //#208
    var _A = 'parseonly'; //#209
    var _C = $get($_.fncvals, _A) !== undefined; //#209
    $_[_A] = _C; //#209
    delete $_.fncvals[_A]; //#209
    var _E = 'eci'; //#210
    var _G = $get($_.fncvals, _E) !== undefined; //#210
    $_[_E] = _G; //#210
    delete $_.fncvals[_E]; //#210
    $k[$j++] = Infinity; //#214
    var _I = $a(['NUL', 'SOH', 'STX', 'ETX', 'EOT', 'ENQ', 'ACK', 'BEL', 'BS', 'TAB', 'LF', 'VT', 'FF', 'CR', "", "", 'DLE', 'DC1', 'DC2', 'DC3', 'DC4', 'NAK', 'SYN', 'ETB', 'CAN', 'EM', 'SUB', 'ESC', 'FS', 'GS', 'RS', 'US']); //#219
    $k[$j++] = 0; //#223
    for (var _J = 0, _K = _I.length; _J < _K; _J++) { //#223
        var _M = $k[--$j]; //#222
        $k[$j++] = $get(_I, _J); //#222
        $k[$j++] = _M; //#222
        $k[$j++] = _M + 1; //#222
    } //#222
    $j--; //#223
    $_.ctrl = $d(); //#224
    $_.msg = $a($_.barcode.length); //#227
    $_.j = 0; //#228
    $k[$j++] = $_.barcode; //#351
    for (;;) { //#351
        $search($k[--$j], "^"); //#230
        var _T = $k[--$j]; //#230
        var _U = $k[--$j]; //#230
        $k[$j++] = _T; //#233
        $k[$j++] = _U.length; //#233
        $k[$j++] = $_.msg; //#233
        $k[$j++] = $_.j; //#233
        var _X = Infinity; //#233
        var _Y = _U; //#233
        $k[$j++] = _X; //#233
        $forall(_Y); //#233
        var _Z = $a(); //#233
        var _a = $k[--$j]; //#233
        $puti($k[--$j], _a, _Z); //#233
        $_.j = $f($k[--$j] + $_.j); //#234
        if ($k[--$j]) { //#349
            $j--; //#236
            for (var _f = 0; _f < 1; _f++) { //#347
                if (!$_.parse && !$_.parsefnc) { //#243
                    $put($_.msg, $_.j, 94); //#240
                    $_.j = $_.j + 1; //#241
                    break; //#242
                } //#242
                $put($_.msg, $_.j, 94); //#246
                $_.j = $_.j + 1; //#247
                if ($_.parse) { //#290
                    var _p = $k[$j - 1]; //#251
                    if (_p.length >= 3) { //#262
                        var _q = $k[$j - 1]; //#252
                        var _r = $geti(_q, 0, 3); //#252
                        var _s = $_.ctrl; //#252
                        var _t = $get(_s, _r) !== undefined; //#253
                        $k[$j++] = _s; //#261
                        $k[$j++] = _r; //#261
                        if (_t) { //#260
                            $_.j = $_.j - 1; //#254
                            var _v = $k[--$j]; //#255
                            $put($_.msg, $_.j, $get($k[--$j], _v)); //#255
                            $_.j = $_.j + 1; //#256
                            var _11 = $k[--$j]; //#257
                            $k[$j++] = $geti(_11, 3, _11.length - 3); //#258
                            break; //#258
                        } else { //#260
                            $j -= 2; //#260
                        } //#260
                    } //#260
                    var _13 = $k[$j - 1]; //#263
                    if (_13.length >= 2) { //#274
                        var _14 = $k[$j - 1]; //#264
                        var _15 = $geti(_14, 0, 2); //#264
                        var _16 = $_.ctrl; //#264
                        var _17 = $get(_16, _15) !== undefined; //#265
                        $k[$j++] = _16; //#273
                        $k[$j++] = _15; //#273
                        if (_17) { //#272
                            $_.j = $_.j - 1; //#266
                            var _19 = $k[--$j]; //#267
                            $put($_.msg, $_.j, $get($k[--$j], _19)); //#267
                            $_.j = $_.j + 1; //#268
                            var _1F = $k[--$j]; //#269
                            $k[$j++] = $geti(_1F, 2, _1F.length - 2); //#270
                            break; //#270
                        } else { //#272
                            $j -= 2; //#272
                        } //#272
                    } //#272
                    var _1H = $k[$j - 1]; //#275
                    if (_1H.length >= 3) { //#289
                        var _1I = $k[$j - 1]; //#276
                        var _1J = $geti(_1I, 0, 3); //#276
                        $k[$j++] = true; //#278
                        for (var _1K = 0, _1L = _1J.length; _1K < _1L; _1K++) { //#278
                            var _1M = $get(_1J, _1K); //#278
                            if ((_1M < 48) || (_1M > 57)) { //#277
                                $k[$j - 1] = false; //#277
                            } //#277
                        } //#277
                        if ($k[--$j]) { //#288
                            var _1O = $k[$j - 1]; //#280
                            var _1Q = $cvi($geti(_1O, 0, 3)); //#280
                            $k[$j++] = _1Q; //#283
                            if (_1Q > 255) { //#283
                                $j -= 2; //#281
                                $k[$j++] = 'bwipp.invalidOrdinal#282'; //#282
                                $k[$j++] = "Ordinal must be 000 to 255"; //#282
                                bwipp_raiseerror(); //#282
                            } //#282
                            $_.j = $_.j - 1; //#284
                            $put($_.msg, $_.j, $k[--$j]); //#285
                            $_.j = $_.j + 1; //#286
                            var _1W = $k[--$j]; //#287
                            $k[$j++] = $geti(_1W, 3, _1W.length - 3); //#287
                        } //#287
                    } //#287
                } //#287
                if ($_.parseonly || !$_.parsefnc || $get($_.msg, $_.j - 1) != 94) { //#295
                    break; //#295
                } //#295
                $_.j = $_.j - 1; //#298
                var _1e = $k[$j - 1]; //#299
                if (_1e.length < 1) { //#302
                    $j--; //#300
                    $k[$j++] = 'bwipp.truncatedCaret#301'; //#301
                    $k[$j++] = "Caret character truncated"; //#301
                    bwipp_raiseerror(); //#301
                } //#301
                var _1f = $k[$j - 1]; //#303
                if ($get(_1f, 0) == 94) { //#308
                    $put($_.msg, $_.j, 94); //#304
                    $_.j = $_.j + 1; //#305
                    var _1k = $k[--$j]; //#306
                    $k[$j++] = $geti(_1k, 1, _1k.length - 1); //#307
                    break; //#307
                } //#307
                var _1m = $k[$j - 1]; //#309
                if (_1m.length < 3) { //#312
                    $j--; //#310
                    $k[$j++] = 'bwipp.truncatedFNC#311'; //#311
                    $k[$j++] = "Function character truncated"; //#311
                    bwipp_raiseerror(); //#311
                } //#311
                var _1n = $k[$j - 1]; //#313
                if ($eq($geti(_1n, 0, 3), "ECI") && $_.eci) { //#330
                    var _1q = $k[$j - 1]; //#314
                    if (_1q.length < 9) { //#317
                        $j--; //#315
                        $k[$j++] = 'bwipp.truncatedECI#316'; //#316
                        $k[$j++] = "ECI truncated"; //#316
                        bwipp_raiseerror(); //#316
                    } //#316
                    var _1r = $k[$j - 1]; //#318
                    var _1s = $geti(_1r, 3, 6); //#318
                    $k[$j++] = _1s; //#324
                    for (var _1t = 0, _1u = _1s.length; _1t < _1u; _1t++) { //#324
                        var _1v = $get(_1s, _1t); //#324
                        if ((_1v < 48) || (_1v > 57)) { //#323
                            $j -= 2; //#321
                            $k[$j++] = 'bwipp.invalidECI#322'; //#322
                            $k[$j++] = "ECI must be 000000 to 999999"; //#322
                            bwipp_raiseerror(); //#322
                        } //#322
                    } //#322
                    var _1w = $k[--$j]; //#325
                    $k[$j++] = 0; //#325
                    $forall(_1w, function() { //#325
                        var _1x = $k[--$j]; //#325
                        var _1y = $k[--$j]; //#325
                        $k[$j++] = ($f(_1y - (_1x - 48))) * 10; //#325
                    }); //#325
                    $put($_.msg, $_.j, (~~($k[--$j] / 10)) - 1000000); //#326
                    $_.j = $_.j + 1; //#327
                    var _23 = $k[--$j]; //#328
                    $k[$j++] = $geti(_23, 9, _23.length - 9); //#329
                    break; //#329
                } //#329
                var _25 = $k[$j - 1]; //#331
                if (_25.length < 4) { //#334
                    $j--; //#332
                    $k[$j++] = 'bwipp.truncatedFNC#333'; //#333
                    $k[$j++] = "Function character truncated"; //#333
                    bwipp_raiseerror(); //#333
                } //#333
                var _26 = $k[$j - 1]; //#335
                var _27 = $geti(_26, 0, 4); //#335
                var _29 = $get($_.fncvals, _27) !== undefined; //#335
                $k[$j++] = _27; //#340
                if (!_29) { //#340
                    var _2A = $k[--$j]; //#336
                    var _2B = $s(_2A.length + 28); //#336
                    $puti(_2B, 28, _2A); //#336
                    $puti(_2B, 0, "Unknown function character: "); //#337
                    $k[$j - 1] = _2B; //#338
                    var _2D = $k[--$j]; //#339
                    $k[$j++] = 'bwipp.unknownFNC#339'; //#339
                    $k[$j++] = _2D; //#339
                    bwipp_raiseerror(); //#339
                } //#339
                $put($_.msg, $_.j, $get($_.fncvals, $k[--$j])); //#342
                $_.j = $_.j + 1; //#343
                var _2K = $k[--$j]; //#344
                $k[$j++] = $geti(_2K, 4, _2K.length - 4); //#345
                break; //#345
            } //#345
        } else { //#349
            break; //#349
        } //#349
    } //#349
    if (!$_.parseonly) { //#357
        $k[$j++] = $geti($_.msg, 0, $_.j); //#354
    } else { //#357
        $k[$j++] = $s($_.j); //#357
        for (var _2U = 0, _2T = $_.j - 1; _2U <= _2T; _2U += 1) { //#357
            var _2V = $k[$j - 1]; //#357
            $put(_2V, _2U, $get($_.msg, _2U)); //#357
        } //#357
    } //#357
    $_ = $__; //#362
} //bwipp_parseinput
function bwipp_gs1process() {
    var $__ = $_; //#387
    $_ = Object.assign({}, $_); //#387
    var _1 = new Map([
        ["cset", 'N'],
        ["min", 18],
        ["max", 18],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos2'])]
    ]); //#397
    var _2 = $a([_1]); //#397
    var _3 = $a([]); //#399
    var _4 = new Map([
        ["parts", _2],
        ["dlpkey", _3]
    ]); //#399
    var _5 = $a(['lintcsum', 'lintgcppos2']); //#406
    var _6 = new Map([
        ["cset", 'N'],
        ["min", 14],
        ["max", 14],
        ["opt", false],
        ["linters", _5]
    ]); //#406
    var _7 = $a([_6]); //#406
    var _8 = $a(["255", "37"]); //#408
    var _9 = $a(["22", "10", "21"]); //#409
    var _A = $a(["235"]); //#409
    var _B = $a([_9, _A]); //#409
    var _C = new Map([
        ["parts", _7],
        ["ex", _8],
        ["dlpkey", _B]
    ]); //#409
    var _D = $a(['lintcsum', 'lintgcppos2']); //#416
    var _E = new Map([
        ["cset", 'N'],
        ["min", 14],
        ["max", 14],
        ["opt", false],
        ["linters", _D]
    ]); //#416
    var _F = $a([_E]); //#416
    var _G = $a(["01", "03"]); //#418
    var _H = $a(["37"]); //#419
    var _I = $a([_H]); //#419
    var _J = $a([_I]); //#419
    var _K = new Map([
        ["parts", _F],
        ["ex", _G],
        ["req", _J]
    ]); //#419
    var _L = $a(['lintcsum', 'lintgcppos2']); //#426
    var _M = new Map([
        ["cset", 'N'],
        ["min", 14],
        ["max", 14],
        ["opt", false],
        ["linters", _L]
    ]); //#426
    var _N = $a([_M]); //#426
    var _O = $a(["01", "02", "37"]); //#428
    var _P = new Map([
        ["parts", _N],
        ["ex", _O],
        ["dlattr", false]
    ]); //#429
    var _R = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#436
    var _a = new Map([
        ["parts", $a([_R])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["03"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#438
    var _c = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmd0'])]
    ]); //#445
    var _l = new Map([
        ["parts", $a([_c])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["03"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#447
    var _n = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmd0'])]
    ]); //#454
    var _r = $a([$a([$a(["8020"])])]); //#456
    var _s = new Map([
        ["parts", $a([_n])],
        ["req", _r]
    ]); //#456
    var _u = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmd0'])]
    ]); //#463
    var _13 = new Map([
        ["parts", $a([_u])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["03"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#465
    var _15 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmd0'])]
    ]); //#474
    var _1F = new Map([
        ["parts", $a([_15])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["03"]), $a(["255"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#476
    var _1H = new Map([
        ["cset", 'N'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a([])]
    ]); //#483
    var _1P = new Map([
        ["parts", $a([_1H])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#485
    var _1R = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#492
    var _1Z = new Map([
        ["parts", $a([_1R])],
        ["ex", $a(["235"])],
        ["req", $a([$a([$a(["01"]), $a(["03"]), $a(["8006"])])])],
        ["dlattr", false]
    ]); //#496
    var _1b = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#503
    var _1g = new Map([
        ["parts", $a([_1b])],
        ["req", $a([$a([$a(["01"])])])],
        ["dlattr", false]
    ]); //#506
    var _1i = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 28],
        ["opt", false],
        ["linters", $a([])]
    ]); //#513
    var _1n = new Map([
        ["parts", $a([_1i])],
        ["req", $a([$a([$a(["01"])])])],
        ["dlattr", false]
    ]); //#516
    var _1p = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#523
    var _1x = new Map([
        ["parts", $a([_1p])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#525
    var _1z = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#533
    var _27 = new Map([
        ["parts", $a([_1z])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#535
    var _29 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#542
    var _2E = new Map([
        ["parts", $a([_29])],
        ["req", $a([$a([$a(["01"])])])]
    ]); //#544
    var _2G = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#551
    var _2O = new Map([
        ["parts", $a([_2G])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])]), $a([$a(["21"])])])]
    ]); //#553
    var _2Q = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#560
    var _2W = new Map([
        ["parts", $a([_2Q])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])])])]
    ]); //#562
    var _2Y = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#569
    var _2a = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 17],
        ["opt", true],
        ["linters", $a([])]
    ]); //#570
    var _2d = new Map([
        ["parts", $a([_2Y, _2a])],
        ["dlpkey", $a([])]
    ]); //#572
    var _2f = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#579
    var _2k = new Map([
        ["parts", $a([_2f])],
        ["req", $a([$a([$a(["414"])])])],
        ["dlattr", false]
    ]); //#582
    var _2m = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#589
    var _2o = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 12],
        ["opt", true],
        ["linters", $a([])]
    ]); //#590
    var _2s = new Map([
        ["parts", $a([_2m, _2o])],
        ["ex", $a(["01", "02", "415", "8006", "8020", "8026"])],
        ["dlpkey", $a([])]
    ]); //#593
    var _2u = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 8],
        ["opt", false],
        ["linters", $a([])]
    ]); //#600
    var _30 = new Map([
        ["parts", $a([_2u])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#602
    var _32 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#609
    var _39 = new Map([
        ["parts", $a([_32])],
        ["ex", $a(["310n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#612
    var _3B = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#624
    var _3I = new Map([
        ["parts", $a([_3B])],
        ["ex", $a(["311n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#627
    var _3K = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#639
    var _3R = new Map([
        ["parts", $a([_3K])],
        ["ex", $a(["312n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#642
    var _3T = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#654
    var _3a = new Map([
        ["parts", $a([_3T])],
        ["ex", $a(["313n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#657
    var _3c = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#669
    var _3j = new Map([
        ["parts", $a([_3c])],
        ["ex", $a(["314n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#672
    var _3l = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#684
    var _3s = new Map([
        ["parts", $a([_3l])],
        ["ex", $a(["315n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#687
    var _3u = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#699
    var _41 = new Map([
        ["parts", $a([_3u])],
        ["ex", $a(["316n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#702
    var _43 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#714
    var _4A = new Map([
        ["parts", $a([_43])],
        ["ex", $a(["320n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#717
    var _4C = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#729
    var _4J = new Map([
        ["parts", $a([_4C])],
        ["ex", $a(["321n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#732
    var _4L = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#744
    var _4S = new Map([
        ["parts", $a([_4L])],
        ["ex", $a(["322n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#747
    var _4U = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#759
    var _4b = new Map([
        ["parts", $a([_4U])],
        ["ex", $a(["323n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#762
    var _4d = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#774
    var _4k = new Map([
        ["parts", $a([_4d])],
        ["ex", $a(["324n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#777
    var _4m = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#789
    var _4t = new Map([
        ["parts", $a([_4m])],
        ["ex", $a(["325n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#792
    var _4v = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#804
    var _52 = new Map([
        ["parts", $a([_4v])],
        ["ex", $a(["326n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#807
    var _54 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#819
    var _5B = new Map([
        ["parts", $a([_54])],
        ["ex", $a(["327n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#822
    var _5D = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#834
    var _5K = new Map([
        ["parts", $a([_5D])],
        ["ex", $a(["328n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#837
    var _5M = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#849
    var _5T = new Map([
        ["parts", $a([_5M])],
        ["ex", $a(["329n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#852
    var _5V = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#864
    var _5c = new Map([
        ["parts", $a([_5V])],
        ["ex", $a(["330n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#867
    var _5e = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#879
    var _5l = new Map([
        ["parts", $a([_5e])],
        ["ex", $a(["331n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#882
    var _5n = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#894
    var _5u = new Map([
        ["parts", $a([_5n])],
        ["ex", $a(["332n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#897
    var _5w = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#909
    var _63 = new Map([
        ["parts", $a([_5w])],
        ["ex", $a(["333n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#912
    var _65 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#924
    var _6C = new Map([
        ["parts", $a([_65])],
        ["ex", $a(["334n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#927
    var _6E = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#939
    var _6L = new Map([
        ["parts", $a([_6E])],
        ["ex", $a(["335n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#942
    var _6N = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#954
    var _6U = new Map([
        ["parts", $a([_6N])],
        ["ex", $a(["336n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#957
    var _6W = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#969
    var _6c = new Map([
        ["parts", $a([_6W])],
        ["ex", $a(["337n"])],
        ["req", $a([$a([$a(["01"])])])]
    ]); //#972
    var _6e = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#984
    var _6l = new Map([
        ["parts", $a([_6e])],
        ["ex", $a(["340n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#987
    var _6n = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#999
    var _6u = new Map([
        ["parts", $a([_6n])],
        ["ex", $a(["341n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1002
    var _6w = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1014
    var _73 = new Map([
        ["parts", $a([_6w])],
        ["ex", $a(["342n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1017
    var _75 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1029
    var _7C = new Map([
        ["parts", $a([_75])],
        ["ex", $a(["343n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1032
    var _7E = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1044
    var _7L = new Map([
        ["parts", $a([_7E])],
        ["ex", $a(["344n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1047
    var _7N = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1059
    var _7U = new Map([
        ["parts", $a([_7N])],
        ["ex", $a(["345n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1062
    var _7W = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1074
    var _7d = new Map([
        ["parts", $a([_7W])],
        ["ex", $a(["346n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1077
    var _7f = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1089
    var _7m = new Map([
        ["parts", $a([_7f])],
        ["ex", $a(["347n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1092
    var _7o = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1104
    var _7v = new Map([
        ["parts", $a([_7o])],
        ["ex", $a(["348n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1107
    var _7x = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1119
    var _84 = new Map([
        ["parts", $a([_7x])],
        ["ex", $a(["349n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1122
    var _86 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1134
    var _8D = new Map([
        ["parts", $a([_86])],
        ["ex", $a(["350n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1137
    var _8F = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1149
    var _8M = new Map([
        ["parts", $a([_8F])],
        ["ex", $a(["351n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1152
    var _8O = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1164
    var _8V = new Map([
        ["parts", $a([_8O])],
        ["ex", $a(["352n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1167
    var _8X = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1179
    var _8e = new Map([
        ["parts", $a([_8X])],
        ["ex", $a(["353n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1182
    var _8g = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1194
    var _8n = new Map([
        ["parts", $a([_8g])],
        ["ex", $a(["354n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1197
    var _8p = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1209
    var _8w = new Map([
        ["parts", $a([_8p])],
        ["ex", $a(["355n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1212
    var _8y = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1224
    var _95 = new Map([
        ["parts", $a([_8y])],
        ["ex", $a(["356n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1227
    var _97 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1239
    var _9E = new Map([
        ["parts", $a([_97])],
        ["ex", $a(["357n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1242
    var _9G = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1254
    var _9N = new Map([
        ["parts", $a([_9G])],
        ["ex", $a(["360n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1257
    var _9P = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1269
    var _9W = new Map([
        ["parts", $a([_9P])],
        ["ex", $a(["361n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1272
    var _9Y = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1284
    var _9f = new Map([
        ["parts", $a([_9Y])],
        ["ex", $a(["362n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1287
    var _9h = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1299
    var _9o = new Map([
        ["parts", $a([_9h])],
        ["ex", $a(["363n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1302
    var _9q = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1314
    var _9x = new Map([
        ["parts", $a([_9q])],
        ["ex", $a(["364n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1317
    var _9z = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1329
    var _A6 = new Map([
        ["parts", $a([_9z])],
        ["ex", $a(["365n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1332
    var _A8 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1344
    var _AF = new Map([
        ["parts", $a([_A8])],
        ["ex", $a(["366n"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1347
    var _AH = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1359
    var _AO = new Map([
        ["parts", $a([_AH])],
        ["ex", $a(["367n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1362
    var _AQ = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1374
    var _AX = new Map([
        ["parts", $a([_AQ])],
        ["ex", $a(["368n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1377
    var _AZ = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1389
    var _Ag = new Map([
        ["parts", $a([_AZ])],
        ["ex", $a(["369n"])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#1392
    var _Ai = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 8],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1404
    var _Aq = new Map([
        ["parts", $a([_Ai])],
        ["req", $a([$a([$a(["00"])]), $a([$a(["02"]), $a(["8026"])])])]
    ]); //#1406
    var _As = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 15],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1413
    var _Az = new Map([
        ["parts", $a([_As])],
        ["ex", $a(["390n", "391n", "394n", "8111"])],
        ["req", $a([$a([$a(["255"]), $a(["8020"])])])]
    ]); //#1416
    var _B1 = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso4217'])]
    ]); //#1432
    var _B3 = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 15],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1433
    var _B9 = new Map([
        ["parts", $a([_B1, _B3])],
        ["ex", $a(["391n"])],
        ["req", $a([$a([$a(["8020"])])])]
    ]); //#1436
    var _BB = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 15],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1452
    var _BN = new Map([
        ["parts", $a([_BB])],
        ["ex", $a(["392n", "393n"])],
        ["req", $a([$a([$a(["01"])]), $a([$a(["30"]), $a(["31nn"]), $a(["32nn"]), $a(["35nn"]), $a(["36nn"])])])]
    ]); //#1455
    var _BP = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso4217'])]
    ]); //#1471
    var _BR = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 15],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1472
    var _Bb = new Map([
        ["parts", $a([_BP, _BR])],
        ["ex", $a(["393n"])],
        ["req", $a([$a([$a(["30"]), $a(["31nn"]), $a(["32nn"]), $a(["35nn"]), $a(["36nn"])])])]
    ]); //#1475
    var _Bd = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1491
    var _Bj = new Map([
        ["parts", $a([_Bd])],
        ["ex", $a(["394n", "8111"])],
        ["req", $a([$a([$a(["255"])])])]
    ]); //#1494
    var _Bl = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1504
    var _Bv = new Map([
        ["parts", $a([_Bl])],
        ["ex", $a(["392n", "393n", "395n", "8005"])],
        ["req", $a([$a([$a(["30"]), $a(["31nn"]), $a(["32nn"]), $a(["35nn"]), $a(["36nn"])])])]
    ]); //#1507
    var _Bx = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1519
    var _Bz = new Map([
        ["parts", $a([_Bx])]
    ]); //#1519
    var _C1 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a(['lintgcppos1'])]
    ]); //#1527
    var _C4 = new Map([
        ["parts", $a([_C1])],
        ["dlpkey", $a([])]
    ]); //#1529
    var _C6 = new Map([
        ["cset", 'N'],
        ["min", 17],
        ["max", 17],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1536
    var _C9 = new Map([
        ["parts", $a([_C6])],
        ["dlpkey", $a([])]
    ]); //#1538
    var _CB = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1545
    var _CG = new Map([
        ["parts", $a([_CB])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1547
    var _CI = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1554
    var _CK = new Map([
        ["parts", $a([_CI])]
    ]); //#1554
    var _CM = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1565
    var _CR = new Map([
        ["parts", $a([_CM])],
        ["dlpkey", $a([$a(["254"]), $a(["7040"])])]
    ]); //#1567
    var _CT = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1574
    var _Ca = new Map([
        ["parts", $a([_CT])],
        ["req", $a([$a([$a(["8020"])])])],
        ["dlpkey", $a([$a(["8020"])])]
    ]); //#1577
    var _Cc = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1584
    var _Ce = new Map([
        ["parts", $a([_Cc])]
    ]); //#1584
    var _Cg = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#1592
    var _Ck = new Map([
        ["parts", $a([_Cg])],
        ["dlpkey", $a([$a(["7040"])])]
    ]); //#1594
    var _Cm = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1601
    var _Cp = new Map([
        ["parts", $a([_Cm])],
        ["ex", $a(["421"])]
    ]); //#1603
    var _Cr = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1610
    var _Ct = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 9],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1611
    var _Cw = new Map([
        ["parts", $a([_Cr, _Ct])],
        ["ex", $a(["4307"])]
    ]); //#1613
    var _Cy = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1620
    var _D7 = new Map([
        ["parts", $a([_Cy])],
        ["ex", $a(["426"])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#1623
    var _D9 = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1630
    var _DB = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1631
    var _DD = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1632
    var _DF = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1633
    var _DH = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1634
    var _DO = new Map([
        ["parts", $a([_D9, _DB, _DD, _DF, _DH])],
        ["ex", $a(["426"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1637
    var _DQ = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1644
    var _DX = new Map([
        ["parts", $a([_DQ])],
        ["ex", $a(["426"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1647
    var _DZ = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1654
    var _Db = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1655
    var _Dd = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1656
    var _Df = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1657
    var _Dh = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", true],
        ["linters", $a(['lintiso3166'])]
    ]); //#1658
    var _Do = new Map([
        ["parts", $a([_DZ, _Db, _Dd, _Df, _Dh])],
        ["ex", $a(["426"])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1661
    var _Dq = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166'])]
    ]); //#1668
    var _Dw = new Map([
        ["parts", $a([_Dq])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1670
    var _Dy = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 3],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1677
    var _E6 = new Map([
        ["parts", $a([_Dy])],
        ["req", $a([$a([$a(["01"]), $a(["02"])]), $a([$a(["422"])])])]
    ]); //#1679
    var _E8 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 35],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1686
    var _ED = new Map([
        ["parts", $a([_E8])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1688
    var _EF = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1696
    var _EK = new Map([
        ["parts", $a([_EF])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1698
    var _EM = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1705
    var _ER = new Map([
        ["parts", $a([_EM])],
        ["req", $a([$a([$a(["4302"])])])]
    ]); //#1707
    var _ET = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1714
    var _EY = new Map([
        ["parts", $a([_ET])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1716
    var _Ea = new Map([
        ["cset", 'X'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a(['lintiso3166alpha2'])]
    ]); //#1725
    var _Ef = new Map([
        ["parts", $a([_Ea])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1727
    var _Eh = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1734
    var _Em = new Map([
        ["parts", $a([_Eh])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1736
    var _Eo = new Map([
        ["cset", 'N'],
        ["min", 10],
        ["max", 10],
        ["opt", false],
        ["linters", $a(['lintlatitude'])]
    ]); //#1743
    var _Eq = new Map([
        ["cset", 'N'],
        ["min", 10],
        ["max", 10],
        ["opt", false],
        ["linters", $a(['lintlongitude'])]
    ]); //#1744
    var _Ev = new Map([
        ["parts", $a([_Eo, _Eq])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1746
    var _Ex = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 35],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1753
    var _F2 = new Map([
        ["parts", $a([_Ex])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1755
    var _F4 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1763
    var _F9 = new Map([
        ["parts", $a([_F4])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1765
    var _FB = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1772
    var _FG = new Map([
        ["parts", $a([_FB])],
        ["req", $a([$a([$a(["4312"])])])]
    ]); //#1774
    var _FI = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1781
    var _FN = new Map([
        ["parts", $a([_FI])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1783
    var _FP = new Map([
        ["cset", 'X'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a(['lintiso3166alpha2'])]
    ]); //#1792
    var _FU = new Map([
        ["parts", $a([_FP])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1794
    var _FW = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1801
    var _Fb = new Map([
        ["parts", $a([_FW])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1803
    var _Fd = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1810
    var _Fi = new Map([
        ["parts", $a([_Fd])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1812
    var _Fk = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 35],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#1819
    var _Fp = new Map([
        ["parts", $a([_Fk])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1821
    var _Fr = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a(['lintyesno'])]
    ]); //#1828
    var _Fw = new Map([
        ["parts", $a([_Fr])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1830
    var _Fy = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmd0'])]
    ]); //#1839
    var _G0 = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['linthhmi'])]
    ]); //#1840
    var _G5 = new Map([
        ["parts", $a([_Fy, _G0])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1842
    var _G7 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1850
    var _GC = new Map([
        ["parts", $a([_G7])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1852
    var _GE = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1859
    var _GG = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", true],
        ["linters", $a(['linthyphen'])]
    ]); //#1860
    var _GM = new Map([
        ["parts", $a([_GE, _GG])],
        ["ex", $a(["4331"])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1863
    var _GO = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1870
    var _GQ = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", true],
        ["linters", $a(['linthyphen'])]
    ]); //#1871
    var _GW = new Map([
        ["parts", $a([_GO, _GQ])],
        ["ex", $a(["4330"])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1874
    var _GY = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1881
    var _Ga = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", true],
        ["linters", $a(['linthyphen'])]
    ]); //#1882
    var _Gg = new Map([
        ["parts", $a([_GY, _Ga])],
        ["ex", $a(["4333"])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1885
    var _Gi = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1892
    var _Gk = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", true],
        ["linters", $a(['linthyphen'])]
    ]); //#1893
    var _Gq = new Map([
        ["parts", $a([_Gi, _Gk])],
        ["ex", $a(["4332"])],
        ["req", $a([$a([$a(["00"])])])]
    ]); //#1896
    var _Gs = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1903
    var _H0 = new Map([
        ["parts", $a([_Gs])],
        ["req", $a([$a([$a(["01"]), $a(["02"]), $a(["8006"]), $a(["8026"])])])]
    ]); //#1905
    var _H2 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1912
    var _H8 = new Map([
        ["parts", $a([_H2])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1914
    var _HA = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1921
    var _HC = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['linthhmi'])]
    ]); //#1922
    var _HI = new Map([
        ["parts", $a([_HA, _HC])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1924
    var _HK = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 4],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1931
    var _HR = new Map([
        ["parts", $a([_HK])],
        ["req", $a([$a([$a(["01"])]), $a([$a(["10"])])])]
    ]); //#1933
    var _HT = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 12],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1940
    var _HZ = new Map([
        ["parts", $a([_HT])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1942
    var _Hb = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1949
    var _Hh = new Map([
        ["parts", $a([_Hb])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1951
    var _Hj = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1958
    var _Hl = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", true],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1959
    var _Hr = new Map([
        ["parts", $a([_Hj, _Hl])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1961
    var _Ht = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 3],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1968
    var _Hz = new Map([
        ["parts", $a([_Ht])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1970
    var _I1 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 10],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1977
    var _I7 = new Map([
        ["parts", $a([_I1])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1979
    var _I9 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 2],
        ["opt", false],
        ["linters", $a([])]
    ]); //#1986
    var _IF = new Map([
        ["parts", $a([_I9])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1988
    var _IH = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#1995
    var _IJ = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", true],
        ["linters", $a(['linthhmi'])]
    ]); //#1996
    var _IP = new Map([
        ["parts", $a([_IH, _IJ])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#1998
    var _IR = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2005
    var _IZ = new Map([
        ["parts", $a([_IR])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])]), $a([$a(["416"])])])]
    ]); //#2007
    var _Ib = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2014
    var _Ih = new Map([
        ["parts", $a([_Ib])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])])])]
    ]); //#2016
    var _Ij = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2023
    var _Io = new Map([
        ["parts", $a([_Ij])],
        ["req", $a([$a([$a(["7021"])])])]
    ]); //#2025
    var _Iq = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a(['lintgcppos1'])]
    ]); //#2032
    var _Is = new Map([
        ["parts", $a([_Iq])]
    ]); //#2032
    var _Iu = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintiso3166999'])]
    ]); //#2040
    var _Iw = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 27],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2041
    var _J2 = new Map([
        ["parts", $a([_Iu, _Iw])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#2043
    var _J4 = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2059
    var _J6 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2060
    var _J8 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2061
    var _JA = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a(['lintimporteridx'])]
    ]); //#2062
    var _JC = new Map([
        ["parts", $a([_J4, _J6, _J8, _JA])],
        ["dlattr", false]
    ]); //#2064
    var _JE = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['lintpackagetype'])]
    ]); //#2071
    var _JJ = new Map([
        ["parts", $a([_JE])],
        ["req", $a([$a([$a(["00"])])])],
        ["dlattr", false]
    ]); //#2074
    var _JL = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2081
    var _JQ = new Map([
        ["parts", $a([_JL])],
        ["req", $a([$a([$a(["01"])])])]
    ]); //#2083
    var _JS = new Map([
        ["cset", 'X'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2096
    var _JU = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 28],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2097
    var _Ja = new Map([
        ["parts", $a([_JS, _JU])],
        ["req", $a([$a([$a(["01"]), $a(["8004"])])])]
    ]); //#2099
    var _Jc = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2115
    var _Ji = new Map([
        ["parts", $a([_Jc])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])])])]
    ]); //#2117
    var _Jk = new Map([
        ["cset", 'N'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a(['lintmediatype'])]
    ]); //#2124
    var _Jq = new Map([
        ["parts", $a([_Jk])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])]
    ]); //#2126
    var _Js = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 25],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2133
    var _Jy = new Map([
        ["parts", $a([_Js])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])]
    ]); //#2135
    var _K0 = new Map([
        ["cset", 'N'],
        ["min", 8],
        ["max", 8],
        ["opt", false],
        ["linters", $a(['lintyyyymmdd'])]
    ]); //#2142
    var _K6 = new Map([
        ["parts", $a([_K0])],
        ["ex", $a(["7251"])],
        ["req", $a([$a([$a(["8018"])])])]
    ]); //#2145
    var _K8 = new Map([
        ["cset", 'N'],
        ["min", 8],
        ["max", 8],
        ["opt", false],
        ["linters", $a(['lintyyyymmdd'])]
    ]); //#2152
    var _KA = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['linthhmi'])]
    ]); //#2153
    var _KG = new Map([
        ["parts", $a([_K8, _KA])],
        ["ex", $a(["7250"])],
        ["req", $a([$a([$a(["8018"])])])]
    ]); //#2156
    var _KI = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a(['lintiso5218'])]
    ]); //#2163
    var _KN = new Map([
        ["parts", $a([_KI])],
        ["req", $a([$a([$a(["8018"])])])]
    ]); //#2165
    var _KP = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 40],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#2172
    var _KW = new Map([
        ["parts", $a([_KP])],
        ["ex", $a(["7256", "7259"])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])]
    ]); //#2175
    var _KY = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 10],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2183
    var _Kf = new Map([
        ["parts", $a([_KY])],
        ["ex", $a(["7256", "7259"])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])]
    ]); //#2186
    var _Kh = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 90],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#2193
    var _Kn = new Map([
        ["parts", $a([_Kh])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])]
    ]); //#2195
    var _Kp = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#2202
    var _Ku = new Map([
        ["parts", $a([_Kp])],
        ["req", $a([$a([$a(["8018"])])])]
    ]); //#2204
    var _Kw = new Map([
        ["cset", 'X'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintposinseqslash'])]
    ]); //#2211
    var _L1 = new Map([
        ["parts", $a([_Kw])],
        ["req", $a([$a([$a(["8018", "7259"])])])]
    ]); //#2213
    var _L3 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 40],
        ["opt", false],
        ["linters", $a(['lintpcenc'])]
    ]); //#2220
    var _L9 = new Map([
        ["parts", $a([_L3])],
        ["ex", $a(["7256"])],
        ["req", $a([$a([$a(["8018"])])])]
    ]); //#2223
    var _LB = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['lintnonzero'])]
    ]); //#2230
    var _LD = new Map([
        ["cset", 'N'],
        ["min", 5],
        ["max", 5],
        ["opt", false],
        ["linters", $a(['lintnonzero'])]
    ]); //#2231
    var _LF = new Map([
        ["cset", 'N'],
        ["min", 3],
        ["max", 3],
        ["opt", false],
        ["linters", $a(['lintnonzero'])]
    ]); //#2232
    var _LH = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a(['lintwinding'])]
    ]); //#2233
    var _LJ = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2234
    var _LO = new Map([
        ["parts", $a([_LB, _LD, _LF, _LH, _LJ])],
        ["req", $a([$a([$a(["01"])])])]
    ]); //#2236
    var _LQ = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2243
    var _LS = new Map([
        ["parts", $a([_LQ])]
    ]); //#2243
    var _LU = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 1],
        ["opt", false],
        ["linters", $a(['lintzero'])]
    ]); //#2251
    var _LW = new Map([
        ["cset", 'N'],
        ["min", 13],
        ["max", 13],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#2252
    var _LY = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 16],
        ["opt", true],
        ["linters", $a([])]
    ]); //#2253
    var _Lb = new Map([
        ["parts", $a([_LU, _LW, _LY])],
        ["dlpkey", $a([])]
    ]); //#2255
    var _Ld = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a(['lintgcppos1'])]
    ]); //#2262
    var _Lh = new Map([
        ["parts", $a([_Ld])],
        ["dlpkey", $a([$a(["7040"])])]
    ]); //#2264
    var _Lj = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2271
    var _Lp = new Map([
        ["parts", $a([_Lj])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#2273
    var _Lr = new Map([
        ["cset", 'N'],
        ["min", 14],
        ["max", 14],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos2'])]
    ]); //#2280
    var _Lt = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['lintpieceoftotal'])]
    ]); //#2281
    var _Ly = new Map([
        ["parts", $a([_Lr, _Lt])],
        ["ex", $a(["01", "37"])],
        ["dlpkey", $a([$a(["22", "10", "21"])])]
    ]); //#2284
    var _M0 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 34],
        ["opt", false],
        ["linters", $a(['lintiban'])]
    ]); //#2291
    var _M5 = new Map([
        ["parts", $a([_M0])],
        ["req", $a([$a([$a(["415"])])])]
    ]); //#2293
    var _M7 = new Map([
        ["cset", 'N'],
        ["min", 6],
        ["max", 6],
        ["opt", false],
        ["linters", $a(['lintyymmdd'])]
    ]); //#2300
    var _M9 = new Map([
        ["cset", 'N'],
        ["min", 2],
        ["max", 2],
        ["opt", false],
        ["linters", $a(['linthh'])]
    ]); //#2301
    var _MB = new Map([
        ["cset", 'N'],
        ["min", 2],
        ["max", 2],
        ["opt", true],
        ["linters", $a(['lintmi'])]
    ]); //#2302
    var _MD = new Map([
        ["cset", 'N'],
        ["min", 2],
        ["max", 2],
        ["opt", true],
        ["linters", $a(['lintss'])]
    ]); //#2303
    var _MJ = new Map([
        ["parts", $a([_M7, _M9, _MB, _MD])],
        ["req", $a([$a([$a(["01"]), $a(["02"])])])]
    ]); //#2305
    var _ML = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 50],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2312
    var _MR = new Map([
        ["parts", $a([_ML])],
        ["req", $a([$a([$a(["00"]), $a(["01"])])])]
    ]); //#2314
    var _MT = new Map([
        ["cset", 'Y'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a(['lintgcppos1'])]
    ]); //#2321
    var _MX = new Map([
        ["parts", $a([_MT])],
        ["dlpkey", $a([$a(["8011"])])]
    ]); //#2323
    var _MZ = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 12],
        ["opt", false],
        ["linters", $a(['lintnozeroprefix'])]
    ]); //#2330
    var _Me = new Map([
        ["parts", $a([_MZ])],
        ["req", $a([$a([$a(["8010"])])])],
        ["dlattr", false]
    ]); //#2333
    var _Mg = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 20],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2340
    var _Mm = new Map([
        ["parts", $a([_Mg])],
        ["req", $a([$a([$a(["01"]), $a(["8006"])])])]
    ]); //#2342
    var _Mo = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 25],
        ["opt", false],
        ["linters", $a(['lintcsumalpha', 'lintgcppos1'])]
    ]); //#2349
    var _Mr = new Map([
        ["parts", $a([_Mo])],
        ["dlpkey", $a([])]
    ]); //#2351
    var _Mt = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 25],
        ["opt", false],
        ["linters", $a(['lintcsumalpha', 'lintgcppos1', 'linthasnondigit'])]
    ]); //#2358
    var _My = new Map([
        ["parts", $a([_Mt])],
        ["req", $a([$a([$a(["01"])])])],
        ["dlattr", false]
    ]); //#2361
    var _N0 = new Map([
        ["cset", 'N'],
        ["min", 18],
        ["max", 18],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#2368
    var _N5 = new Map([
        ["parts", $a([_N0])],
        ["ex", $a(["8018"])],
        ["dlpkey", $a([$a(["8019"])])]
    ]); //#2371
    var _N7 = new Map([
        ["cset", 'N'],
        ["min", 18],
        ["max", 18],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos1'])]
    ]); //#2378
    var _NC = new Map([
        ["parts", $a([_N7])],
        ["ex", $a(["8017"])],
        ["dlpkey", $a([$a(["8019"])])]
    ]); //#2381
    var _NE = new Map([
        ["cset", 'N'],
        ["min", 1],
        ["max", 10],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2388
    var _NK = new Map([
        ["parts", $a([_NE])],
        ["req", $a([$a([$a(["8017"]), $a(["8018"])])])],
        ["dlattr", false]
    ]); //#2391
    var _NM = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 25],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2398
    var _NR = new Map([
        ["parts", $a([_NM])],
        ["req", $a([$a([$a(["415"])])])],
        ["dlattr", false]
    ]); //#2401
    var _NT = new Map([
        ["cset", 'N'],
        ["min", 14],
        ["max", 14],
        ["opt", false],
        ["linters", $a(['lintcsum', 'lintgcppos2'])]
    ]); //#2408
    var _NV = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a(['lintpieceoftotal'])]
    ]); //#2409
    var _Nb = new Map([
        ["parts", $a([_NT, _NV])],
        ["ex", $a(["02", "8006"])],
        ["req", $a([$a([$a(["37"])])])]
    ]); //#2412
    var _Nd = new Map([
        ["cset", 'Z'],
        ["min", 1],
        ["max", 90],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2419
    var _Nr = new Map([
        ["parts", $a([_Nd])],
        ["req", $a([$a([$a(["00"]), $a(["01", "21"]), $a(["253"]), $a(["255"]), $a(["8003"]), $a(["8004"]), $a(["8006", "21"]), $a(["8010", "8011"]), $a(["8017"]), $a(["8018"])])])]
    ]); //#2421
    var _Nt = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintcouponcode'])]
    ]); //#2428
    var _Nv = new Map([
        ["parts", $a([_Nt])]
    ]); //#2428
    var _Nx = new Map([
        ["cset", 'N'],
        ["min", 4],
        ["max", 4],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2436
    var _O2 = new Map([
        ["parts", $a([_Nx])],
        ["req", $a([$a([$a(["255"])])])]
    ]); //#2438
    var _O4 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a(['lintcouponposoffer'])]
    ]); //#2445
    var _O6 = new Map([
        ["parts", $a([_O4])]
    ]); //#2445
    var _O8 = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 70],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2453
    var _OD = new Map([
        ["parts", $a([_O8])],
        ["req", $a([$a([$a(["01"])])])],
        ["dlattr", false]
    ]); //#2456
    var _OF = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 30],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2463
    var _OH = new Map([
        ["parts", $a([_OF])]
    ]); //#2463
    var _OJ = new Map([
        ["cset", 'X'],
        ["min", 1],
        ["max", 90],
        ["opt", false],
        ["linters", $a([])]
    ]); //#2471
    var _OL = new Map([
        ["parts", $a([_OJ])]
    ]); //#2471
    var _OM = new Map([
        ["00", _4],
        ["01", _C],
        ["02", _K],
        ["03", _P],
        ["10", _a],
        ["11", _l],
        ["12", _s],
        ["13", _13],
        ["15", _13],
        ["16", _13],
        ["17", _1F],
        ["20", _1P],
        ["21", _1Z],
        ["22", _1g],
        ["235", _1n],
        ["240", _1x],
        ["241", _1x],
        ["242", _27],
        ["243", _2E],
        ["250", _2O],
        ["251", _2W],
        ["253", _2d],
        ["254", _2k],
        ["255", _2s],
        ["30", _30],
        ["3100", _39],
        ["3101", _39],
        ["3102", _39],
        ["3103", _39],
        ["3104", _39],
        ["3105", _39],
        ["3110", _3I],
        ["3111", _3I],
        ["3112", _3I],
        ["3113", _3I],
        ["3114", _3I],
        ["3115", _3I],
        ["3120", _3R],
        ["3121", _3R],
        ["3122", _3R],
        ["3123", _3R],
        ["3124", _3R],
        ["3125", _3R],
        ["3130", _3a],
        ["3131", _3a],
        ["3132", _3a],
        ["3133", _3a],
        ["3134", _3a],
        ["3135", _3a],
        ["3140", _3j],
        ["3141", _3j],
        ["3142", _3j],
        ["3143", _3j],
        ["3144", _3j],
        ["3145", _3j],
        ["3150", _3s],
        ["3151", _3s],
        ["3152", _3s],
        ["3153", _3s],
        ["3154", _3s],
        ["3155", _3s],
        ["3160", _41],
        ["3161", _41],
        ["3162", _41],
        ["3163", _41],
        ["3164", _41],
        ["3165", _41],
        ["3200", _4A],
        ["3201", _4A],
        ["3202", _4A],
        ["3203", _4A],
        ["3204", _4A],
        ["3205", _4A],
        ["3210", _4J],
        ["3211", _4J],
        ["3212", _4J],
        ["3213", _4J],
        ["3214", _4J],
        ["3215", _4J],
        ["3220", _4S],
        ["3221", _4S],
        ["3222", _4S],
        ["3223", _4S],
        ["3224", _4S],
        ["3225", _4S],
        ["3230", _4b],
        ["3231", _4b],
        ["3232", _4b],
        ["3233", _4b],
        ["3234", _4b],
        ["3235", _4b],
        ["3240", _4k],
        ["3241", _4k],
        ["3242", _4k],
        ["3243", _4k],
        ["3244", _4k],
        ["3245", _4k],
        ["3250", _4t],
        ["3251", _4t],
        ["3252", _4t],
        ["3253", _4t],
        ["3254", _4t],
        ["3255", _4t],
        ["3260", _52],
        ["3261", _52],
        ["3262", _52],
        ["3263", _52],
        ["3264", _52],
        ["3265", _52],
        ["3270", _5B],
        ["3271", _5B],
        ["3272", _5B],
        ["3273", _5B],
        ["3274", _5B],
        ["3275", _5B],
        ["3280", _5K],
        ["3281", _5K],
        ["3282", _5K],
        ["3283", _5K],
        ["3284", _5K],
        ["3285", _5K],
        ["3290", _5T],
        ["3291", _5T],
        ["3292", _5T],
        ["3293", _5T],
        ["3294", _5T],
        ["3295", _5T],
        ["3300", _5c],
        ["3301", _5c],
        ["3302", _5c],
        ["3303", _5c],
        ["3304", _5c],
        ["3305", _5c],
        ["3310", _5l],
        ["3311", _5l],
        ["3312", _5l],
        ["3313", _5l],
        ["3314", _5l],
        ["3315", _5l],
        ["3320", _5u],
        ["3321", _5u],
        ["3322", _5u],
        ["3323", _5u],
        ["3324", _5u],
        ["3325", _5u],
        ["3330", _63],
        ["3331", _63],
        ["3332", _63],
        ["3333", _63],
        ["3334", _63],
        ["3335", _63],
        ["3340", _6C],
        ["3341", _6C],
        ["3342", _6C],
        ["3343", _6C],
        ["3344", _6C],
        ["3345", _6C],
        ["3350", _6L],
        ["3351", _6L],
        ["3352", _6L],
        ["3353", _6L],
        ["3354", _6L],
        ["3355", _6L],
        ["3360", _6U],
        ["3361", _6U],
        ["3362", _6U],
        ["3363", _6U],
        ["3364", _6U],
        ["3365", _6U],
        ["3370", _6c],
        ["3371", _6c],
        ["3372", _6c],
        ["3373", _6c],
        ["3374", _6c],
        ["3375", _6c],
        ["3400", _6l],
        ["3401", _6l],
        ["3402", _6l],
        ["3403", _6l],
        ["3404", _6l],
        ["3405", _6l],
        ["3410", _6u],
        ["3411", _6u],
        ["3412", _6u],
        ["3413", _6u],
        ["3414", _6u],
        ["3415", _6u],
        ["3420", _73],
        ["3421", _73],
        ["3422", _73],
        ["3423", _73],
        ["3424", _73],
        ["3425", _73],
        ["3430", _7C],
        ["3431", _7C],
        ["3432", _7C],
        ["3433", _7C],
        ["3434", _7C],
        ["3435", _7C],
        ["3440", _7L],
        ["3441", _7L],
        ["3442", _7L],
        ["3443", _7L],
        ["3444", _7L],
        ["3445", _7L],
        ["3450", _7U],
        ["3451", _7U],
        ["3452", _7U],
        ["3453", _7U],
        ["3454", _7U],
        ["3455", _7U],
        ["3460", _7d],
        ["3461", _7d],
        ["3462", _7d],
        ["3463", _7d],
        ["3464", _7d],
        ["3465", _7d],
        ["3470", _7m],
        ["3471", _7m],
        ["3472", _7m],
        ["3473", _7m],
        ["3474", _7m],
        ["3475", _7m],
        ["3480", _7v],
        ["3481", _7v],
        ["3482", _7v],
        ["3483", _7v],
        ["3484", _7v],
        ["3485", _7v],
        ["3490", _84],
        ["3491", _84],
        ["3492", _84],
        ["3493", _84],
        ["3494", _84],
        ["3495", _84],
        ["3500", _8D],
        ["3501", _8D],
        ["3502", _8D],
        ["3503", _8D],
        ["3504", _8D],
        ["3505", _8D],
        ["3510", _8M],
        ["3511", _8M],
        ["3512", _8M],
        ["3513", _8M],
        ["3514", _8M],
        ["3515", _8M],
        ["3520", _8V],
        ["3521", _8V],
        ["3522", _8V],
        ["3523", _8V],
        ["3524", _8V],
        ["3525", _8V],
        ["3530", _8e],
        ["3531", _8e],
        ["3532", _8e],
        ["3533", _8e],
        ["3534", _8e],
        ["3535", _8e],
        ["3540", _8n],
        ["3541", _8n],
        ["3542", _8n],
        ["3543", _8n],
        ["3544", _8n],
        ["3545", _8n],
        ["3550", _8w],
        ["3551", _8w],
        ["3552", _8w],
        ["3553", _8w],
        ["3554", _8w],
        ["3555", _8w],
        ["3560", _95],
        ["3561", _95],
        ["3562", _95],
        ["3563", _95],
        ["3564", _95],
        ["3565", _95],
        ["3570", _9E],
        ["3571", _9E],
        ["3572", _9E],
        ["3573", _9E],
        ["3574", _9E],
        ["3575", _9E],
        ["3600", _9N],
        ["3601", _9N],
        ["3602", _9N],
        ["3603", _9N],
        ["3604", _9N],
        ["3605", _9N],
        ["3610", _9W],
        ["3611", _9W],
        ["3612", _9W],
        ["3613", _9W],
        ["3614", _9W],
        ["3615", _9W],
        ["3620", _9f],
        ["3621", _9f],
        ["3622", _9f],
        ["3623", _9f],
        ["3624", _9f],
        ["3625", _9f],
        ["3630", _9o],
        ["3631", _9o],
        ["3632", _9o],
        ["3633", _9o],
        ["3634", _9o],
        ["3635", _9o],
        ["3640", _9x],
        ["3641", _9x],
        ["3642", _9x],
        ["3643", _9x],
        ["3644", _9x],
        ["3645", _9x],
        ["3650", _A6],
        ["3651", _A6],
        ["3652", _A6],
        ["3653", _A6],
        ["3654", _A6],
        ["3655", _A6],
        ["3660", _AF],
        ["3661", _AF],
        ["3662", _AF],
        ["3663", _AF],
        ["3664", _AF],
        ["3665", _AF],
        ["3670", _AO],
        ["3671", _AO],
        ["3672", _AO],
        ["3673", _AO],
        ["3674", _AO],
        ["3675", _AO],
        ["3680", _AX],
        ["3681", _AX],
        ["3682", _AX],
        ["3683", _AX],
        ["3684", _AX],
        ["3685", _AX],
        ["3690", _Ag],
        ["3691", _Ag],
        ["3692", _Ag],
        ["3693", _Ag],
        ["3694", _Ag],
        ["3695", _Ag],
        ["37", _Aq],
        ["3900", _Az],
        ["3901", _Az],
        ["3902", _Az],
        ["3903", _Az],
        ["3904", _Az],
        ["3905", _Az],
        ["3906", _Az],
        ["3907", _Az],
        ["3908", _Az],
        ["3909", _Az],
        ["3910", _B9],
        ["3911", _B9],
        ["3912", _B9],
        ["3913", _B9],
        ["3914", _B9],
        ["3915", _B9],
        ["3916", _B9],
        ["3917", _B9],
        ["3918", _B9],
        ["3919", _B9],
        ["3920", _BN],
        ["3921", _BN],
        ["3922", _BN],
        ["3923", _BN],
        ["3924", _BN],
        ["3925", _BN],
        ["3926", _BN],
        ["3927", _BN],
        ["3928", _BN],
        ["3929", _BN],
        ["3930", _Bb],
        ["3931", _Bb],
        ["3932", _Bb],
        ["3933", _Bb],
        ["3934", _Bb],
        ["3935", _Bb],
        ["3936", _Bb],
        ["3937", _Bb],
        ["3938", _Bb],
        ["3939", _Bb],
        ["3940", _Bj],
        ["3941", _Bj],
        ["3942", _Bj],
        ["3943", _Bj],
        ["3950", _Bv],
        ["3951", _Bv],
        ["3952", _Bv],
        ["3953", _Bv],
        ["3954", _Bv],
        ["3955", _Bv],
        ["400", _Bz],
        ["401", _C4],
        ["402", _C9],
        ["403", _CG],
        ["410", _CK],
        ["411", _CK],
        ["412", _CK],
        ["413", _CK],
        ["414", _CR],
        ["415", _Ca],
        ["416", _Ce],
        ["417", _Ck],
        ["420", _Cp],
        ["421", _Cw],
        ["422", _D7],
        ["423", _DO],
        ["424", _DX],
        ["425", _Do],
        ["426", _Dw],
        ["427", _E6],
        ["4300", _ED],
        ["4301", _ED],
        ["4302", _EK],
        ["4303", _ER],
        ["4304", _EY],
        ["4305", _EY],
        ["4306", _EY],
        ["4307", _Ef],
        ["4308", _Em],
        ["4309", _Ev],
        ["4310", _F2],
        ["4311", _F2],
        ["4312", _F9],
        ["4313", _FG],
        ["4314", _FN],
        ["4315", _FN],
        ["4316", _FN],
        ["4317", _FU],
        ["4318", _Fb],
        ["4319", _Fi],
        ["4320", _Fp],
        ["4321", _Fw],
        ["4322", _Fw],
        ["4323", _Fw],
        ["4324", _G5],
        ["4325", _G5],
        ["4326", _GC],
        ["4330", _GM],
        ["4331", _GW],
        ["4332", _Gg],
        ["4333", _Gq],
        ["7001", _H0],
        ["7002", _H8],
        ["7003", _HI],
        ["7004", _HR],
        ["7005", _HZ],
        ["7006", _Hh],
        ["7007", _Hr],
        ["7008", _Hz],
        ["7009", _I7],
        ["7010", _IF],
        ["7011", _IP],
        ["7020", _IZ],
        ["7021", _Ih],
        ["7022", _Io],
        ["7023", _Is],
        ["7030", _J2],
        ["7031", _J2],
        ["7032", _J2],
        ["7033", _J2],
        ["7034", _J2],
        ["7035", _J2],
        ["7036", _J2],
        ["7037", _J2],
        ["7038", _J2],
        ["7039", _J2],
        ["7040", _JC],
        ["7041", _JJ],
        ["710", _JQ],
        ["711", _JQ],
        ["712", _JQ],
        ["713", _JQ],
        ["714", _JQ],
        ["715", _JQ],
        ["716", _JQ],
        ["7230", _Ja],
        ["7231", _Ja],
        ["7232", _Ja],
        ["7233", _Ja],
        ["7234", _Ja],
        ["7235", _Ja],
        ["7236", _Ja],
        ["7237", _Ja],
        ["7238", _Ja],
        ["7239", _Ja],
        ["7240", _Ji],
        ["7241", _Jq],
        ["7242", _Jy],
        ["7250", _K6],
        ["7251", _KG],
        ["7252", _KN],
        ["7253", _KW],
        ["7254", _KW],
        ["7255", _Kf],
        ["7256", _Kn],
        ["7257", _Ku],
        ["7258", _L1],
        ["7259", _L9],
        ["8001", _LO],
        ["8002", _LS],
        ["8003", _Lb],
        ["8004", _Lh],
        ["8005", _Lp],
        ["8006", _Ly],
        ["8007", _M5],
        ["8008", _MJ],
        ["8009", _MR],
        ["8010", _MX],
        ["8011", _Me],
        ["8012", _Mm],
        ["8013", _Mr],
        ["8014", _My],
        ["8017", _N5],
        ["8018", _NC],
        ["8019", _NK],
        ["8020", _NR],
        ["8026", _Nb],
        ["8030", _Nr],
        ["8110", _Nv],
        ["8111", _O2],
        ["8112", _O6],
        ["8200", _OD],
        ["90", _OH],
        ["91", _OL],
        ["92", _OL],
        ["93", _OL],
        ["94", _OL],
        ["95", _OL],
        ["96", _OL],
        ["97", _OL],
        ["98", _OL],
        ["99", _OL]
    ]); //#2483
    $_.gs1syntax = _OM; //#2485
    if ($eq($k[--$j], 'ai')) { //#2717
        $anchorsearch($k[--$j], "\("); //#2493
        if (!$k[--$j]) { //#2495
            $j--; //#2494
            $k[$j++] = 'bwipp.GS1aiMissingOpenParen#2494'; //#2494
            $k[$j++] = "AIs must start with '\('"; //#2494
            bwipp_raiseerror(); //#2494
        } //#2494
        $k[$j - 1] = Infinity; //#2497
        var _OR = $k[--$j]; //#2497
        var _OS = $k[--$j]; //#2497
        $k[$j++] = _OR; //#2504
        $k[$j++] = _OS; //#2504
        for (;;) { //#2504
            var _OT = $k[$j - 1]; //#2498
            if (_OT.length == 0) { //#2498
                $j--; //#2498
                break; //#2498
            } //#2498
            $search($k[--$j], "\)"); //#2499
            if (!$k[--$j]) { //#2501
                $cleartomark(); //#2500
                $k[$j++] = 'bwipp.GS1aiMissingCloseParen#2500'; //#2500
                $k[$j++] = "AIs must end with '\)'"; //#2500
                bwipp_raiseerror(); //#2500
            } //#2500
            var _OW = $k[--$j]; //#2502
            $k[$j - 1] = _OW; //#2502
            var _OY = $k[--$j]; //#2502
            var _OZ = $k[--$j]; //#2502
            $k[$j++] = _OY; //#2503
            $search(_OZ, "\("); //#2503
            if ($k[--$j]) { //#2503
                var _Ob = $k[--$j]; //#2503
                $k[$j - 1] = _Ob; //#2503
                var _Od = $k[--$j]; //#2503
                var _Oe = $k[--$j]; //#2503
                $k[$j++] = _Od; //#2503
                $k[$j++] = _Oe; //#2503
            } else { //#2503
                $k[$j++] = ""; //#2503
            } //#2503
        } //#2503
        var _Of = $counttomark(); //#2505
        $k[$j++] = _Of; //#2505
        if (_Of > 0) { //#2505
            var _Og = $k[--$j]; //#2505
            for (var _Oi = _Og, _Oh = (~~(_Og / 2)) + 1; _Oi >= _Oh; _Oi -= 1) { //#2505
                $r(_Oi, -1); //#2505
            } //#2505
        } else { //#2505
            $j--; //#2505
        } //#2505
        $astore($a(~~($counttomark() / 2))); //#2506
        $_.ais = $k[--$j]; //#2506
        $astore($a($counttomark())); //#2507
        $k[$j++] = Infinity; //#2507
        var _Oo = $k[--$j]; //#2507
        var _Op = $k[--$j]; //#2507
        $k[$j++] = _Oo; //#2509
        $forall(_Op, function() { //#2509
            var _Or = new Map([
                ["parse", $_.parse],
                ["parseonly", true],
                ["parsefnc", false]
            ]); //#2508
            $k[$j++] = _Or; //#2508
            bwipp_parseinput(); //#2508
        }); //#2508
        $_.vals = $a(); //#2509
        $j--; //#2510
    } else { //#2717
        $anchorsearch($k[--$j], "http://"); //#2515
        if (!$k[--$j]) { //#2520
            $anchorsearch($k[--$j], "HTTP://"); //#2516
            if (!$k[--$j]) { //#2520
                $anchorsearch($k[--$j], "https://"); //#2517
                if (!$k[--$j]) { //#2520
                    $anchorsearch($k[--$j], "HTTPS://"); //#2518
                    if (!$k[--$j]) { //#2520
                        $j--; //#2519
                        $k[$j++] = 'bwipp.GS1dlURIbadScheme#2519'; //#2519
                        $k[$j++] = "Scheme must be http:// HTTP:// https:// or HTTPS://"; //#2519
                        bwipp_raiseerror(); //#2519
                    } //#2519
                } //#2519
            } //#2519
        } //#2519
        $j--; //#2521
        $search($k[--$j], "/"); //#2524
        if (!$k[--$j]) { //#2526
            $j--; //#2525
            $k[$j++] = 'bwipp.GS1dlMissingDomainOrPathInfo#2525'; //#2525
            $k[$j++] = "URI must contain a domain and path info"; //#2525
            bwipp_raiseerror(); //#2525
        } //#2525
        $j -= 2; //#2527
        $search($k[--$j], "#"); //#2530
        if ($k[--$j]) { //#2532
            var _P5 = $k[--$j]; //#2531
            $k[$j - 1] = _P5; //#2531
            var _P7 = $k[--$j]; //#2531
            $k[$j - 1] = _P7; //#2531
        } //#2531
        $search($k[--$j], "?"); //#2535
        if (!$k[--$j]) { //#2537
            var _PB = $k[--$j]; //#2536
            $k[$j++] = ""; //#2536
            $k[$j++] = null; //#2536
            $k[$j++] = _PB; //#2536
        } //#2536
        $_.pp = $k[--$j]; //#2538
        $j--; //#2539
        $_.qp = $k[--$j]; //#2540
        $k[$j++] = Infinity; //#2543
        $k[$j++] = 0; //#2547
        $k[$j++] = 0; //#2547
        $k[$j++] = $_.pp; //#2547
        for (;;) { //#2547
            $search($k[--$j], "/"); //#2545
            if (!$k[--$j]) { //#2545
                $j--; //#2545
                break; //#2545
            } //#2545
            var _PH = $k[--$j]; //#2546
            $k[$j - 1] = _PH.length + 1; //#2546
            var _PJ = $k[--$j]; //#2546
            var _PK = $k[--$j]; //#2546
            var _PL = $k[$j - 1]; //#2546
            $k[$j++] = $f(_PJ + _PL); //#2546
            $k[$j++] = _PK; //#2546
        } //#2546
        $astore($a($counttomark() - 1)); //#2548
        $_.pipos = $k[--$j]; //#2548
        $j -= 2; //#2548
        if ($_.pipos.length <= 1) { //#2551
            $k[$j++] = 'bwipp.GS1dlNoAIinfo#2550'; //#2550
            $k[$j++] = "The path was too short to contain AI info"; //#2550
            bwipp_raiseerror(); //#2550
        } //#2550
        $k[$j++] = false; //#2567
        for (var _PR = $_.pipos.length - 2; _PR >= 0; _PR -= 2) { //#2567
            var _PT = $get($_.pipos, _PR); //#2557
            var _PX = $geti($_.pp, _PT, $f($get($_.pipos, _PR + 1) - _PT) - 1); //#2558
            var _PZ = $get($_.gs1syntax, _PX) !== undefined; //#2559
            $k[$j++] = _PR; //#2565
            $k[$j++] = _PX; //#2565
            if (_PZ) { //#2564
                var _Pd = $get($get($_.gs1syntax, $k[--$j]), 'dlpkey') !== undefined; //#2560
                if (_Pd) { //#2562
                    var _Pe = $k[--$j]; //#2561
                    $k[$j - 1] = _Pe; //#2561
                    $k[$j++] = true; //#2561
                    break; //#2561
                } //#2561
            } else { //#2564
                $j--; //#2564
            } //#2564
            $j--; //#2566
        } //#2566
        if (!$k[--$j]) { //#2570
            $k[$j++] = 'bwipp.GS1dlNoAIinfo#2569'; //#2569
            $k[$j++] = "The path does not contain a valid primary key"; //#2569
            bwipp_raiseerror(); //#2569
        } //#2569
        var _Pj = $get($_.pipos, $k[--$j]); //#2571
        $_.pp = $geti($_.pp, _Pj, $f($_.pp.length - _Pj)); //#2571
        $_.uriunescape = function() {
            $_.qq = $k[--$j]; //#2574
            $_.in = $k[--$j]; //#2575
            $_.out = $s($_.in.length); //#2576
            $k[$j++] = 0; //#2600
            $k[$j++] = 0; //#2600
            for (;;) { //#2600
                var _Ps = $k[$j - 2]; //#2578
                if (_Ps >= $_.in.length) { //#2578
                    break; //#2578
                } //#2578
                var _Pw = $k[$j - 2]; //#2579
                var _Px = $get($_.in, _Pw); //#2579
                $k[$j++] = _Px; //#2599
                if ((_Px == 43) && $_.qq) { //#2598
                    $j--; //#2581
                    var _Q0 = $k[--$j]; //#2581
                    $put($_.out, _Q0, 32); //#2581
                    var _Q1 = $k[--$j]; //#2582
                    $k[$j++] = _Q1 + 1; //#2582
                    $k[$j++] = _Q0 + 1; //#2582
                } else { //#2598
                    var _Q2 = $k[$j - 1]; //#2584
                    var _Q4 = $k[$j - 3]; //#2584
                    if ((_Q2 == 37) && (_Q4 < ($_.in.length - 2))) { //#2598
                        $j--; //#2585
                        var _Q8 = $k[$j - 2]; //#2586
                        var _Q9 = $geti($_.in, _Q8 + 1, 2); //#2586
                        $k[$j++] = 0; //#2593
                        for (var _QA = 0, _QB = _Q9.length; _QA < _QB; _QA++) { //#2593
                            var _QC = $get(_Q9, _QA); //#2593
                            $k[$j++] = _QC; //#2591
                            if ((_QC >= 48) && (_QC <= 57)) { //#2590
                                var _QD = $k[--$j]; //#2587
                                $k[$j++] = _QD - 48; //#2587
                            } else { //#2590
                                var _QE = $k[$j - 1]; //#2588
                                if ((_QE >= 65) && (_QE <= 70)) { //#2590
                                    var _QF = $k[--$j]; //#2588
                                    $k[$j++] = _QF - 55; //#2588
                                } else { //#2590
                                    var _QG = $k[$j - 1]; //#2589
                                    if ((_QG >= 97) && (_QG <= 102)) { //#2590
                                        var _QH = $k[--$j]; //#2589
                                        $k[$j++] = _QH - 87; //#2589
                                    } else { //#2590
                                        $j -= 2; //#2590
                                        $k[$j++] = 'bwipp.GS1dlBadHexCharacter#2590'; //#2590
                                        $k[$j++] = "Invalid hex character"; //#2590
                                        bwipp_raiseerror(); //#2590
                                    } //#2590
                                } //#2590
                            } //#2590
                            var _QI = $k[--$j]; //#2592
                            var _QJ = $k[--$j]; //#2592
                            $k[$j++] = $f(_QI + (_QJ * 16)); //#2592
                        } //#2592
                        var _QL = $k[--$j]; //#2594
                        var _QM = $k[--$j]; //#2594
                        $put($_.out, _QM, _QL); //#2594
                        var _QN = $k[--$j]; //#2595
                        $k[$j++] = _QN + 3; //#2595
                        $k[$j++] = _QM + 1; //#2595
                    } else { //#2598
                        var _QP = $k[--$j]; //#2597
                        var _QQ = $k[--$j]; //#2597
                        $put($_.out, _QQ, _QP); //#2597
                        var _QR = $k[--$j]; //#2598
                        $k[$j++] = _QR + 1; //#2598
                        $k[$j++] = _QQ + 1; //#2598
                    } //#2598
                } //#2598
            } //#2598
            $_.out = $geti($_.out, 0, $k[--$j]); //#2601
            $k[$j - 1] = $_.out; //#2603
        }; //#2603
        $_.isvaliddlpkeyseq = function() {
            $_.in = $k[--$j]; //#2607
            $k[$j++] = false; //#2621
            $forall($get($get($_.gs1syntax, $get($_.in, 0)), 'dlpkey'), function() { //#2621
                $_.seq = $k[--$j]; //#2610
                $_.i = 1; //#2611
                $_.j = 0; //#2611
                for (;;) { //#2619
                    if ($_.i >= $_.in.length) { //#2613
                        break; //#2613
                    } //#2613
                    if ($_.j >= $_.seq.length) { //#2614
                        break; //#2614
                    } //#2614
                    if ($eq($get($_.in, $_.i), $get($_.seq, $_.j))) { //#2617
                        $_.i = $_.i + 1; //#2616
                    } //#2616
                    $_.j = $_.j + 1; //#2618
                } //#2618
                if ($_.i == $_.in.length) { //#2620
                    $k[$j - 1] = true; //#2620
                    return true; //#2620
                } //#2620
            }); //#2620
        }; //#2620
        $_.ais = $a(99); //#2624
        $_.vals = $a(99); //#2625
        $k[$j++] = 0; //#2639
        $k[$j++] = $_.pp; //#2639
        for (;;) { //#2639
            $search($k[--$j], "/"); //#2630
            if ($k[--$j]) { //#2636
                var _Qx = $k[--$j]; //#2631
                $j--; //#2631
                var _Qz = $k[--$j]; //#2631
                var _R0 = $k[$j - 1]; //#2631
                $put($_.ais, _R0, _Qx); //#2631
                $search(_Qz, "/"); //#2633
                if ($k[--$j]) { //#2633
                    var _R2 = $k[--$j]; //#2633
                    $k[$j - 1] = _R2; //#2633
                } else { //#2633
                    var _R4 = $k[--$j]; //#2633
                    $k[$j++] = ""; //#2633
                    $k[$j++] = _R4; //#2633
                } //#2633
                var _R6 = $k[--$j]; //#2634
                var _R8 = $k[$j - 2]; //#2634
                $k[$j++] = $_.vals; //#2634
                $k[$j++] = _R8; //#2634
                $k[$j++] = _R6; //#2634
                $k[$j++] = false; //#2634
                $_.uriunescape(); //#2634
                var _R9 = $k[--$j]; //#2634
                var _RA = $k[--$j]; //#2634
                $put($k[--$j], _RA, _R9); //#2634
            } else { //#2636
                $j--; //#2636
                break; //#2636
            } //#2636
            var _RC = $k[--$j]; //#2638
            var _RD = $k[--$j]; //#2638
            $k[$j++] = _RD + 1; //#2638
            $k[$j++] = _RC; //#2638
        } //#2638
        var _RE = $k[--$j]; //#2640
        $_.plen = _RE; //#2640
        $k[$j++] = _RE; //#2647
        if (_RE > 1) { //#2647
            $k[$j++] = $geti($_.ais, 0, $_.plen); //#2644
            $_.isvaliddlpkeyseq(); //#2644
            if (!$k[--$j]) { //#2646
                $j--; //#2645
                $k[$j++] = 'bwipp.GS1dlBadPathInfo#2645'; //#2645
                $k[$j++] = "The AIs in the path are not a valid key-qualifier sequence for the key"; //#2645
                bwipp_raiseerror(); //#2645
            } //#2645
        } //#2645
        $k[$j++] = $_.qp; //#2665
        for (;;) { //#2665
            var _RK = $k[$j - 1]; //#2651
            if (_RK.length == 0) { //#2651
                $j--; //#2651
                break; //#2651
            } //#2651
            $search($k[--$j], "&"); //#2652
            if ($k[--$j]) { //#2652
                var _RN = $k[--$j]; //#2652
                $k[$j - 1] = _RN; //#2652
            } else { //#2652
                var _RP = $k[--$j]; //#2652
                $k[$j++] = ""; //#2652
                $k[$j++] = _RP; //#2652
            } //#2652
            $search($k[--$j], "="); //#2653
            if ($k[--$j]) { //#2663
                var _RS = $k[$j - 1]; //#2654
                $k[$j++] = true; //#2654
                $forall(_RS, function() { //#2654
                    var _RT = $k[--$j]; //#2654
                    var _RU = $k[--$j]; //#2654
                    $k[$j++] = _RU && ((_RT >= 48) && (_RT <= 57)); //#2654
                }); //#2654
                if ($k[--$j]) { //#2660
                    var _RX = $k[--$j]; //#2655
                    $j--; //#2655
                    var _RZ = $k[--$j]; //#2655
                    var _Rb = $k[$j - 2]; //#2655
                    $put($_.ais, _Rb, _RX); //#2655
                    $k[$j++] = $_.vals; //#2657
                    $k[$j++] = _Rb; //#2657
                    $k[$j++] = _RZ; //#2657
                    $k[$j++] = true; //#2657
                    $_.uriunescape(); //#2657
                    var _Rd = $k[--$j]; //#2657
                    var _Re = $k[--$j]; //#2657
                    $put($k[--$j], _Re, _Rd); //#2657
                    var _Rg = $k[--$j]; //#2658
                    var _Rh = $k[--$j]; //#2658
                    $k[$j++] = _Rh + 1; //#2658
                    $k[$j++] = _Rg; //#2658
                } else { //#2660
                    $j -= 3; //#2660
                } //#2660
            } else { //#2663
                $j--; //#2663
            } //#2663
        } //#2663
        var _Rk = $k[$j - 1]; //#2668
        var _Rm = $geti($_.ais, $_.plen, $f(_Rk - $_.plen)); //#2668
        for (var _Rn = 0, _Ro = _Rm.length; _Rn < _Ro; _Rn++) { //#2682
            var _Rp = $get(_Rm, _Rn); //#2682
            var _Rr = $get($_.gs1syntax, _Rp) !== undefined; //#2669
            $k[$j++] = _Rp; //#2673
            if (_Rr) { //#2672
                var _Rt = $k[$j - 1]; //#2670
                var _Ru = $get($_.gs1syntax, _Rt); //#2670
                var _Rv = $get(_Ru, 'dlattr') !== undefined; //#2670
                $k[$j++] = _Ru; //#2670
                $k[$j++] = 'dlattr'; //#2670
                if (_Rv) { //#2670
                    var _Rw = $k[--$j]; //#2670
                    var _Ry = $get($k[--$j], _Rw); //#2670
                    $k[$j++] = _Ry; //#2670
                } else { //#2670
                    $j -= 2; //#2670
                    $k[$j++] = true; //#2670
                } //#2670
            } else { //#2672
                $k[$j++] = $_.dontlint; //#2672
            } //#2672
            if (!$k[--$j]) { //#2680
                var _S1 = $k[$j - 1]; //#2675
                var _S2 = _S1.length; //#2675
                var _S3 = $s(_S2 + 46); //#2675
                $puti(_S3, 0, "AI \("); //#2676
                $puti(_S3, 4, _S1); //#2677
                $puti(_S3, _S2 + 4, "\) is not a valid GS1 DL URI data attribute"); //#2678
                $k[$j++] = _S3; //#2679
                var _S4 = $k[--$j]; //#2679
                $k[$j - 1] = _S4; //#2679
                var _S6 = $k[--$j]; //#2679
                $k[$j++] = 'bwipp.GS1dlInvalidDataAttribute#2679'; //#2679
                $k[$j++] = _S6; //#2679
                bwipp_raiseerror(); //#2679
            } //#2679
            $j--; //#2681
        } //#2681
        var _S9 = $k[$j - 1]; //#2685
        var _SB = $geti($_.ais, $_.plen, $f(_S9 - $_.plen)); //#2685
        for (var _SC = 0, _SD = _SB.length; _SC < _SD; _SC++) { //#2700
            $k[$j++] = $get(_SB, _SC); //#2698
            for (var _SG = $_.plen; _SG >= 1; _SG -= 1) { //#2698
                var _SH = $k[$j - 1]; //#2687
                $k[$j++] = _SG; //#2689
                $k[$j++] = _SH; //#2689
                $k[$j++] = Infinity; //#2687
                $aload($geti($_.ais, 0, $_.plen)); //#2688
                $r($counttomark() + 3, -2); //#2689
                var _SM = $k[--$j]; //#2689
                var _SN = $k[--$j]; //#2689
                $k[$j++] = _SM; //#2689
                $r(_SN, 1); //#2689
                var _SO = $a(); //#2689
                $k[$j++] = _SO; //#2690
                $_.isvaliddlpkeyseq(); //#2690
                if ($k[--$j]) { //#2697
                    var _SQ = $k[--$j]; //#2691
                    $k[$j - 1] = _SQ; //#2691
                    var _SS = $k[$j - 1]; //#2692
                    var _ST = _SS.length; //#2692
                    var _SU = $s(_ST + 50); //#2692
                    $puti(_SU, 0, "AI \("); //#2693
                    $puti(_SU, 4, _SS); //#2694
                    $puti(_SU, _ST + 4, "\) from query params should be in the path info"); //#2695
                    $k[$j++] = _SU; //#2696
                    var _SV = $k[--$j]; //#2696
                    $k[$j - 1] = _SV; //#2696
                    var _SX = $k[--$j]; //#2696
                    $k[$j++] = 'bwipp.GS1dlAttributeMustBeQualifier#2696'; //#2696
                    $k[$j++] = _SX; //#2696
                    bwipp_raiseerror(); //#2696
                } //#2696
            } //#2696
            $j--; //#2699
        } //#2699
        var _SZ = $k[--$j]; //#2702
        $_.ais = $geti($_.ais, 0, _SZ); //#2702
        $_.vals = $geti($_.vals, 0, _SZ); //#2703
        for (var _Sf = 0, _Se = $_.ais.length - 1; _Sf <= _Se; _Sf += 1) { //#2718
            $k[$j++] = _Sf; //#2716
            if ($eq($get($_.ais, _Sf), "01")) { //#2716
                var _Si = $k[$j - 1]; //#2709
                var _Sk = $get($_.vals, _Si); //#2709
                var _Sl = _Sk.length; //#2710
                $k[$j++] = _Sk; //#2715
                if ((_Sl == 8) || ((_Sl == 12) || (_Sl == 13))) { //#2714
                    var _Sn = $strcpy($s(14), "00000000000000"); //#2711
                    $k[$j++] = _Sn; //#2711
                    $k[$j++] = _Sn; //#2711
                    var _So = $k[$j - 3]; //#2711
                    $puti($k[$j - 1], 14 - _So.length, _So); //#2711
                    var _Sr = $k[$j - 2]; //#2712
                    $j -= 3; //#2711
                    var _Ss = $k[$j - 1]; //#2712
                    $put($_.vals, _Ss, _Sr); //#2712
                } else { //#2714
                    $j--; //#2714
                } //#2714
            } //#2714
            $j--; //#2717
        } //#2717
    } //#2717
    $k[$j++] = Infinity; //#2724
    var _St = $a(["00", "01", "02", "03", "04", "11", "12", "13", "14", "15", "16", "17", "18", "19", "20", "31", "32", "33", "34", "35", "36", "41"]); //#2729
    for (var _Su = 0, _Sv = _St.length; _Su < _Sv; _Su++) { //#2730
        var _Sw = $get(_St, _Su); //#2730
        $k[$j++] = _Sw; //#2730
        $k[$j++] = _Sw; //#2730
    } //#2730
    $_.aifixed = $d(); //#2730
    $k[$j++] = Infinity; //#2733
    var _Sz = $_.ais; //#2734
    for (var _T0 = 0, _T1 = _Sz.length; _T0 < _T1; _T0++) { //#2741
        var _T2 = $get(_Sz, _T0); //#2741
        $k[$j++] = true; //#2740
        $k[$j++] = _T2; //#2740
        if (_T2.length >= 2) { //#2739
            var _T6 = $get($_.aifixed, $geti($k[--$j], 0, 2)) !== undefined; //#2737
            if (_T6) { //#2737
                $k[$j - 1] = false; //#2737
            } //#2737
        } else { //#2739
            $j--; //#2739
        } //#2739
    } //#2739
    $_.fncs = $a(); //#2739
    $k[$j++] = Infinity; //#2747
    $k[$j++] = 0; //#2749
    for (var _T8 = 0, _T9 = "!\"%&'\(\)*+,-./0123456789:;<=>?ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz".length; _T8 < _T9; _T8++) { //#2749
        var _TB = $k[--$j]; //#2749
        $k[$j++] = $get("!\"%&'\(\)*+,-./0123456789:;<=>?ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz", _T8); //#2749
        $k[$j++] = _TB; //#2749
        $k[$j++] = _TB + 1; //#2749
    } //#2749
    $j--; //#2749
    $_.cset82 = $d(); //#2750
    $k[$j++] = Infinity; //#2752
    $k[$j++] = 0; //#2754
    for (var _TD = 0, _TE = "#-/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ".length; _TD < _TE; _TD++) { //#2754
        var _TG = $k[--$j]; //#2754
        $k[$j++] = $get("#-/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", _TD); //#2754
        $k[$j++] = _TG; //#2754
        $k[$j++] = _TG + 1; //#2754
    } //#2754
    $j--; //#2754
    $_.cset39 = $d(); //#2755
    $k[$j++] = Infinity; //#2757
    $k[$j++] = 0; //#2759
    for (var _TI = 0, _TJ = "23456789ABCDEFGHJKLMNPQRSTUVWXYZ".length; _TI < _TJ; _TI++) { //#2759
        var _TL = $k[--$j]; //#2759
        $k[$j++] = $get("23456789ABCDEFGHJKLMNPQRSTUVWXYZ", _TI); //#2759
        $k[$j++] = _TL; //#2759
        $k[$j++] = _TL + 1; //#2759
    } //#2759
    $j--; //#2759
    $_.cset32 = $d(); //#2760
    $k[$j++] = Infinity; //#2762
    $k[$j++] = 0; //#2764
    for (var _TN = 0, _TO = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_".length; _TN < _TO; _TN++) { //#2764
        var _TQ = $k[--$j]; //#2764
        $k[$j++] = $get("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_", _TN); //#2764
        $k[$j++] = _TQ; //#2764
        $k[$j++] = _TQ + 1; //#2764
    } //#2764
    $j--; //#2764
    $_.cset64 = $d(); //#2765
    $_.lintnumeric = function() {
        var _TT = $k[--$j]; //#2769
        $k[$j++] = true; //#2769
        $forall(_TT, function() { //#2769
            var _TU = $k[--$j]; //#2769
            if ((_TU < 48) || (_TU > 57)) { //#2769
                $k[$j - 1] = false; //#2769
                return true; //#2769
            } //#2769
        }); //#2769
        if (!$k[--$j]) { //#2770
            $j--; //#2770
            $k[$j++] = 'bwipp.GS1notNumeric#2770'; //#2770
            $k[$j++] = "Not numeric"; //#2770
            $k[$j++] = false; //#2770
            return true; //#2770
        } //#2770
    }; //#2770
    $_.lintcset82 = function() {
        var _TW = $k[--$j]; //#2774
        $k[$j++] = true; //#2774
        $forall(_TW, function() { //#2774
            var _TZ = $get($_.cset82, $k[--$j]) !== undefined; //#2774
            if (!_TZ) { //#2774
                $k[$j - 1] = false; //#2774
                return true; //#2774
            } //#2774
        }); //#2774
        if (!$k[--$j]) { //#2775
            $j--; //#2775
            $k[$j++] = 'bwipp.GS1badCSET82character#2775'; //#2775
            $k[$j++] = "Invalid CSET 82 character"; //#2775
            $k[$j++] = false; //#2775
            return true; //#2775
        } //#2775
    }; //#2775
    $_.lintcset39 = function() {
        var _Tb = $k[--$j]; //#2779
        $k[$j++] = true; //#2779
        $forall(_Tb, function() { //#2779
            var _Te = $get($_.cset39, $k[--$j]) !== undefined; //#2779
            if (!_Te) { //#2779
                $k[$j - 1] = false; //#2779
                return true; //#2779
            } //#2779
        }); //#2779
        if (!$k[--$j]) { //#2780
            $j--; //#2780
            $k[$j++] = 'bwipp.GS1badCSET39character#2780'; //#2780
            $k[$j++] = "Invalid CSET 39 character"; //#2780
            $k[$j++] = false; //#2780
            return true; //#2780
        } //#2780
    }; //#2780
    $_.lintcset64 = function() {
        var _Tg = $k[$j - 1]; //#2785
        $search(_Tg, "="); //#2785
        if ($k[--$j]) { //#2791
            var _Ti = $k[--$j]; //#2786
            $k[$j - 1] = _Ti; //#2786
            var _Tk = $k[--$j]; //#2786
            $k[$j++] = _Tk.length % 3; //#2786
            $k[$j++] = _Tk; //#2786
            var _Tl = $k[$j - 2]; //#2787
            var _Tm = $k[$j - 3]; //#2787
            $k[$j - 3] = $k[$j - 1]; //#2786
            $j -= 2; //#2786
            $k[$j++] = _Tm; //#2787
            $k[$j++] = _Tl; //#2787
            $k[$j++] = false; //#2787
            if ((_Tl == 1) && $eq(_Tm, "=")) { //#2787
                $k[$j - 1] = true; //#2787
            } //#2787
            var _To = $k[$j - 2]; //#2788
            var _Tp = $k[$j - 3]; //#2788
            if ((_To == 2) && (_Tp.length == 0)) { //#2788
                $k[$j - 1] = true; //#2788
            } //#2788
            if (!$k[--$j]) { //#2789
                $j -= 5; //#2789
                $k[$j++] = 'bwipp.GS1badCSET64padding#2789'; //#2789
                $k[$j++] = "Invalid CSET 64 padding"; //#2789
                $k[$j++] = false; //#2789
                return true; //#2789
            } //#2789
            $j -= 2; //#2790
            var _Tr = $k[--$j]; //#2790
            var _Ts = $k[--$j]; //#2790
            $k[$j++] = _Tr; //#2790
            $k[$j++] = _Ts; //#2790
        } //#2790
        $j--; //#2792
        var _Tt = $k[--$j]; //#2794
        $k[$j++] = true; //#2794
        $forall(_Tt, function() { //#2794
            var _Tw = $get($_.cset64, $k[--$j]) !== undefined; //#2794
            if (!_Tw) { //#2794
                $k[$j - 1] = false; //#2794
                return true; //#2794
            } //#2794
        }); //#2794
        if (!$k[--$j]) { //#2795
            $j--; //#2795
            $k[$j++] = 'bwipp.GS1badCSET64character#2795'; //#2795
            $k[$j++] = "Invalid CSET 64 character"; //#2795
            $k[$j++] = false; //#2795
            return true; //#2795
        } //#2795
    }; //#2795
    $_.lintgcppos1 = function() {
        var _Ty = $k[$j - 1]; //#2799
        if (_Ty.length < 2) { //#2799
            $j -= 2; //#2799
            $k[$j++] = 'bwipp.GS1gcpTooShort#2799'; //#2799
            $k[$j++] = "Value is too short to contain a GS1 Company Prefix"; //#2799
            $k[$j++] = false; //#2799
            return true; //#2799
        } //#2799
        var _Tz = $k[--$j]; //#2800
        var _U0 = $get(_Tz, 0); //#2800
        var _U1 = $get(_Tz, 1); //#2801
        if (((_U0 < 48) || (_U0 > 57)) || ((_U1 < 48) || (_U1 > 57))) { //#2803
            $j--; //#2802
            $k[$j++] = 'bwipp.GS1badGCP#2802'; //#2802
            $k[$j++] = "Non-numeric GS1 Company Prefix"; //#2802
            $k[$j++] = false; //#2802
            return true; //#2802
        } //#2802
    }; //#2802
    $_.lintgcppos2 = function() {
        var _U2 = $k[$j - 1]; //#2807
        if (_U2.length < 1) { //#2807
            $j -= 2; //#2807
            $k[$j++] = 'bwipp.GS1valueTooShortForOffsetGCP#2807'; //#2807
            $k[$j++] = "Value is too short to contain an offset GS1 Company Prefix"; //#2807
            $k[$j++] = false; //#2807
            return true; //#2807
        } //#2807
        var _U3 = $k[--$j]; //#2808
        $k[$j++] = $geti(_U3, 1, _U3.length - 1); //#2808
        $_.lintgcppos1(); //#2808
    }; //#2808
    $_.lintimporteridx = function() {
        $search("-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz", $k[--$j]); //#2813
        if (!$k[--$j]) { //#2813
            $j -= 2; //#2813
            $k[$j++] = 'bwipp.GS1badImporterIndex#2813'; //#2813
            $k[$j++] = "Invalid importer index"; //#2813
            $k[$j++] = false; //#2813
            return true; //#2813
        } //#2813
        $j -= 3; //#2814
    }; //#2814
    $_.lintcsum = function() {
        $k[$j++] = Infinity; //#2818
        var _U7 = $k[--$j]; //#2818
        var _U8 = $k[--$j]; //#2818
        var _U9 = ((_U8.length % 2) == 0) ? 3 : 1; //#2819
        $k[$j++] = _U7; //#2820
        $k[$j++] = _U9; //#2820
        $forall(_U8, function() { //#2820
            var _UA = $k[--$j]; //#2820
            var _UB = $k[--$j]; //#2820
            $k[$j++] = (_UA - 48) * _UB; //#2820
            $k[$j++] = 4 - _UB; //#2820
        }); //#2820
        $j--; //#2820
        var _UC = $counttomark() + 1; //#2821
        $k[$j++] = 0; //#2821
        for (var _UD = 0, _UE = _UC - 1; _UD < _UE; _UD++) { //#2821
            var _UF = $k[--$j]; //#2821
            var _UG = $k[--$j]; //#2821
            $k[$j++] = $f(_UG + _UF); //#2821
        } //#2821
        var _UH = $k[--$j]; //#2821
        $k[$j - 1] = _UH; //#2821
        if (($k[--$j] % 10) != 0) { //#2822
            $j--; //#2822
            $k[$j++] = 'bwipp.GS1badChecksum#2822'; //#2822
            $k[$j++] = "Bad checksum"; //#2822
            $k[$j++] = false; //#2822
            return true; //#2822
        } //#2822
    }; //#2822
    $_.lintcsumalpha = function() {
        var _UK = $k[$j - 1]; //#2826
        if (_UK.length < 2) { //#2826
            $j -= 2; //#2826
            $k[$j++] = 'bwipp.GS1alphaTooShort#2826'; //#2826
            $k[$j++] = "Alphanumeric string is too short to check"; //#2826
            $k[$j++] = false; //#2826
            return true; //#2826
        } //#2826
        var _UL = $k[$j - 1]; //#2827
        $k[$j++] = _UL.length - 2; //#2829
        var _UN = Infinity; //#2829
        var _UO = $geti(_UL, 0, _UL.length - 2); //#2829
        $k[$j++] = _UN; //#2831
        $forall(_UO, function() { //#2831
            var _UP = $k[$j - 1]; //#2830
            var _UR = $get($_.cset82, _UP) !== undefined; //#2830
            if (_UR) { //#2830
                var _UU = $get($_.cset82, $k[--$j]); //#2830
                $k[$j++] = _UU; //#2830
            } else { //#2830
                $k[$j++] = -1; //#2830
                return true; //#2830
            } //#2830
        }); //#2830
        var _UV = $k[$j - 1]; //#2832
        if (_UV == -1) { //#2832
            $cleartomark(); //#2832
            $j -= 3; //#2832
            $k[$j++] = 'bwipp.GS1UnknownCSET82Character#2832'; //#2832
            $k[$j++] = "Unknown CSET 82 character"; //#2832
            $k[$j++] = false; //#2832
            return true; //#2832
        } //#2832
        $astore($a($counttomark())); //#2833
        var _UY = $k[--$j]; //#2833
        $k[$j - 1] = _UY; //#2833
        var _Ua = $k[$j - 2]; //#2835
        var _Uc = $geti($k[$j - 3], _Ua, 2); //#2835
        $k[$j - 3] = $k[$j - 1]; //#2834
        $j -= 2; //#2834
        var _Ud = Infinity; //#2835
        var _Ue = _Uc; //#2835
        $k[$j++] = _Ud; //#2837
        $forall(_Ue, function() { //#2837
            var _Uf = $k[$j - 1]; //#2836
            var _Uh = $get($_.cset32, _Uf) !== undefined; //#2836
            if (_Uh) { //#2836
                var _Uk = $get($_.cset32, $k[--$j]); //#2836
                $k[$j++] = _Uk; //#2836
            } else { //#2836
                $k[$j++] = -1; //#2836
                return true; //#2836
            } //#2836
        }); //#2836
        var _Ul = $k[$j - 1]; //#2838
        if (_Ul == -1) { //#2838
            $cleartomark(); //#2838
            $j -= 2; //#2838
            $k[$j++] = 'bwipp.GS1UnknownCSET32Character#2838'; //#2838
            $k[$j++] = "Unknown CSET 32 character"; //#2838
            $k[$j++] = false; //#2838
            return true; //#2838
        } //#2838
        $astore($a($counttomark())); //#2839
        var _Uo = $k[--$j]; //#2839
        $k[$j - 1] = _Uo; //#2839
        var _Uq = $k[--$j]; //#2840
        var _Ut = $k[--$j]; //#2840
        var _Uu = $a([2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83]); //#2841
        var _Uv = _Ut.length; //#2843
        $k[$j++] = $f(($get(_Uq, 0) << 5) + $get(_Uq, 1)); //#2843
        $k[$j++] = _Ut; //#2843
        $k[$j++] = _Uu; //#2843
        $k[$j++] = _Uv; //#2843
        if (_Uv > _Uu.length) { //#2843
            $j -= 5; //#2843
            $k[$j++] = 'bwipp.GS1alphaTooLong#2843'; //#2843
            $k[$j++] = "Alphanumeric string is too long to check"; //#2843
            $k[$j++] = false; //#2843
            return true; //#2843
        } //#2843
        var _Uw = $k[--$j]; //#2844
        var _Uy = $geti($k[--$j], 0, _Uw); //#2844
        for (var _Uz = 0, _V0 = _Uy.length; _Uz < _V0; _Uz++) { //#2844
            var _V2 = $k[--$j]; //#2844
            $k[$j++] = $get(_Uy, _Uz); //#2844
            $k[$j++] = _V2; //#2844
        } //#2844
        var _V3 = $k[--$j]; //#2845
        $k[$j++] = 0; //#2845
        $forall(_V3, function() { //#2845
            var _V4 = $k[$j - 3]; //#2845
            var _V5 = $k[$j - 1]; //#2845
            var _V6 = $k[$j - 2]; //#2845
            $j -= 3; //#2845
            $k[$j++] = $f(_V6 + (_V4 * _V5)); //#2845
        }); //#2845
        var _V7 = $k[--$j]; //#2845
        if ($k[--$j] != (_V7 % 1021)) { //#2846
            $j--; //#2846
            $k[$j++] = 'bwipp.GS1badAlphaCheckCharacters#2846'; //#2846
            $k[$j++] = "Bad alphanumeric check characters"; //#2846
            $k[$j++] = false; //#2846
            return true; //#2846
        } //#2846
    }; //#2846
    $k[$j++] = Infinity; //#2850
    var _V9 = $a(['004', '008', '010', '012', '016', '020', '024', '028', '031', '032', '036', '040', '044', '048', '050', '051', '052', '056', '060', '064', '068', '070', '072', '074', '076', '084', '086', '090', '092', '096', '100', '104', '108', '112', '116', '120', '124', '132', '136', '140', '144', '148', '152', '156', '158', '162', '166', '170', '174', '175', '178', '180', '184', '188', '191', '192', '196', '203', '204', '208', '212', '214', '218', '222', '226', '231', '232', '233', '234', '238', '239', '242', '246', '248', '250', '254', '258', '260', '262', '266', '268', '270', '275', '276', '288', '292', '296', '300', '304', '308', '312', '316', '320', '324', '328', '332', '334', '336', '340', '344', '348', '352', '356', '360', '364', '368', '372', '376', '380', '384', '388', '392', '398', '400', '404', '408', '410', '414', '417', '418', '422', '426', '428', '430', '434', '438', '440', '442', '446', '450', '454', '458', '462', '466', '470', '474', '478', '480', '484', '492', '496', '498', '499', '500', '504', '508', '512', '516', '520', '524', '528', '531', '533', '534', '535', '540', '548', '554', '558', '562', '566', '570', '574', '578', '580', '581', '583', '584', '585', '586', '591', '598', '600', '604', '608', '612', '616', '620', '624', '626', '630', '634', '638', '642', '643', '646', '652', '654', '659', '660', '662', '663', '666', '670', '674', '678', '682', '686', '688', '690', '694', '702', '703', '704', '705', '706', '710', '716', '724', '728', '729', '732', '740', '744', '748', '752', '756', '760', '762', '764', '768', '772', '776', '780', '784', '788', '792', '795', '796', '798', '800', '804', '807', '818', '826', '831', '832', '833', '834', '840', '850', '854', '858', '860', '862', '876', '882', '887', '894']); //#2868
    for (var _VA = 0, _VB = _V9.length; _VA < _VB; _VA++) { //#2869
        var _VC = $get(_V9, _VA); //#2869
        $k[$j++] = _VC; //#2869
        $k[$j++] = _VC; //#2869
    } //#2869
    $_.iso3166 = $d(); //#2869
    $_.lintiso3166 = function() {
        var _VH = $get($_.iso3166, $k[--$j]) !== undefined; //#2873
        if (!_VH) { //#2873
            $j--; //#2873
            $k[$j++] = 'bwipp.GS1UnknownCountry#2873'; //#2873
            $k[$j++] = "Unknown country code"; //#2873
            $k[$j++] = false; //#2873
            return true; //#2873
        } //#2873
    }; //#2873
    $_.lintiso3166999 = function() {
        var _VI = $k[$j - 1]; //#2877
        if ($ne(_VI, '999')) { //#2880
            var _VL = $get($_.iso3166, $k[--$j]) !== undefined; //#2878
            if (!_VL) { //#2878
                $j--; //#2878
                $k[$j++] = 'bwipp.GS1UnknownCountryOr999#2878'; //#2878
                $k[$j++] = "Unknown country code or not 999"; //#2878
                $k[$j++] = false; //#2878
                return true; //#2878
            } //#2878
        } else { //#2880
            $j--; //#2880
        } //#2880
    }; //#2880
    $k[$j++] = Infinity; //#2885
    var _VM = $a(['AD', 'AE', 'AF', 'AG', 'AI', 'AL', 'AM', 'AO', 'AQ', 'AR', 'AS', 'AT', 'AU', 'AW', 'AX', 'AZ', 'BA', 'BB', 'BD', 'BE', 'BF', 'BG', 'BH', 'BI', 'BJ', 'BL', 'BM', 'BN', 'BO', 'BQ', 'BR', 'BS', 'BT', 'BV', 'BW', 'BY', 'BZ', 'CA', 'CC', 'CD', 'CF', 'CG', 'CH', 'CI', 'CK', 'CL', 'CM', 'CN', 'CO', 'CR', 'CU', 'CV', 'CW', 'CX', 'CY', 'CZ', 'DE', 'DJ', 'DK', 'DM', 'DO', 'DZ', 'EC', 'EE', 'EG', 'EH', 'ER', 'ES', 'ET', 'FI', 'FJ', 'FK', 'FM', 'FO', 'FR', 'GA', 'GB', 'GD', 'GE', 'GF', 'GG', 'GH', 'GI', 'GL', 'GM', 'GN', 'GP', 'GQ', 'GR', 'GS', 'GT', 'GU', 'GW', 'GY', 'HK', 'HM', 'HN', 'HR', 'HT', 'HU', 'ID', 'IE', 'IL', 'IM', 'IN', 'IO', 'IQ', 'IR', 'IS', 'IT', 'JE', 'JM', 'JO', 'JP', 'KE', 'KG', 'KH', 'KI', 'KM', 'KN', 'KP', 'KR', 'KW', 'KY', 'KZ', 'LA', 'LB', 'LC', 'LI', 'LK', 'LR', 'LS', 'LT', 'LU', 'LV', 'LY', 'MA', 'MC', 'MD', 'ME', 'MF', 'MG', 'MH', 'MK', 'ML', 'MM', 'MN', 'MO', 'MP', 'MQ', 'MR', 'MS', 'MT', 'MU', 'MV', 'MW', 'MX', 'MY', 'MZ', 'NA', 'NC', 'NE', 'NF', 'NG', 'NI', 'NL', 'NO', 'NP', 'NR', 'NU', 'NZ', 'OM', 'PA', 'PE', 'PF', 'PG', 'PH', 'PK', 'PL', 'PM', 'PN', 'PR', 'PS', 'PT', 'PW', 'PY', 'QA', 'RE', 'RO', 'RS', 'RU', 'RW', 'SA', 'SB', 'SC', 'SD', 'SE', 'SG', 'SH', 'SI', 'SJ', 'SK', 'SL', 'SM', 'SN', 'SO', 'SR', 'SS', 'ST', 'SV', 'SX', 'SY', 'SZ', 'TC', 'TD', 'TF', 'TG', 'TH', 'TJ', 'TK', 'TL', 'TM', 'TN', 'TO', 'TR', 'TT', 'TV', 'TW', 'TZ', 'UA', 'UG', 'UM', 'US', 'UY', 'UZ', 'VA', 'VC', 'VE', 'VG', 'VI', 'VN', 'VU', 'WF', 'WS', 'YE', 'YT', 'ZA', 'ZM', 'ZW']); //#2909
    for (var _VN = 0, _VO = _VM.length; _VN < _VO; _VN++) { //#2910
        var _VP = $get(_VM, _VN); //#2910
        $k[$j++] = _VP; //#2910
        $k[$j++] = _VP; //#2910
    } //#2910
    $_.iso3166alpha2 = $d(); //#2910
    $_.lintiso3166alpha2 = function() {
        var _VU = $get($_.iso3166alpha2, $k[--$j]) !== undefined; //#2914
        if (!_VU) { //#2914
            $j--; //#2914
            $k[$j++] = 'bwipp.GS1UnknownCountryAlpha#2914'; //#2914
            $k[$j++] = "Unknown country alpha code"; //#2914
            $k[$j++] = false; //#2914
            return true; //#2914
        } //#2914
    }; //#2914
    $k[$j++] = Infinity; //#2918
    var _VV = $a(['008', '012', '032', '036', '044', '048', '050', '051', '052', '060', '064', '068', '072', '084', '090', '096', '104', '108', '116', '124', '132', '136', '144', '152', '156', '170', '174', '188', '191', '192', '203', '208', '214', '222', '230', '232', '238', '242', '262', '270', '292', '320', '324', '328', '332', '340', '344', '348', '352', '356', '360', '364', '368', '376', '388', '392', '396', '398', '400', '404', '408', '410', '414', '417', '418', '422', '426', '430', '434', '446', '454', '458', '462', '480', '484', '496', '498', '504', '512', '516', '524', '532', '533', '548', '554', '558', '566', '578', '586', '590', '598', '600', '604', '608', '634', '643', '646', '654', '682', '690', '694', '702', '704', '706', '710', '728', '748', '752', '756', '760', '764', '776', '780', '784', '788', '800', '807', '818', '826', '834', '840', '858', '860', '882', '886', '901', '925', '927', '928', '929', '930', '931', '932', '933', '934', '936', '938', '940', '941', '943', '944', '946', '947', '948', '949', '950', '951', '952', '953', '955', '956', '957', '958', '959', '960', '961', '962', '963', '964', '965', '967', '968', '969', '970', '971', '972', '973', '975', '976', '977', '978', '979', '980', '981', '984', '985', '986', '990', '994', '997', '999']); //#2940
    for (var _VW = 0, _VX = _VV.length; _VW < _VX; _VW++) { //#2941
        var _VY = $get(_VV, _VW); //#2941
        $k[$j++] = _VY; //#2941
        $k[$j++] = _VY; //#2941
    } //#2941
    $_.iso4217 = $d(); //#2941
    $_.lintiso4217 = function() {
        var _Vd = $get($_.iso4217, $k[--$j]) !== undefined; //#2945
        if (!_Vd) { //#2945
            $j--; //#2945
            $k[$j++] = 'bwipp.GS1UnknownCurrency#2945'; //#2945
            $k[$j++] = "Unknown currency code"; //#2945
            $k[$j++] = false; //#2945
            return true; //#2945
        } //#2945
    }; //#2945
    $_.lintiso5218 = function() {
        var _Ve = $k[--$j]; //#2949
        if ($ne(_Ve, "0") && ($ne(_Ve, "1") && ($ne(_Ve, "2") && $ne(_Ve, "9")))) { //#2951
            $j--; //#2950
            $k[$j++] = 'bwipp.GS1biologicalSexCode#2950'; //#2950
            $k[$j++] = "Invalid biological sex code"; //#2950
            $k[$j++] = false; //#2950
            return true; //#2950
        } //#2950
    }; //#2950
    $_.lintiban = function() {
        var _Vf = $k[$j - 1]; //#2955
        if (_Vf.length <= 4) { //#2955
            $j -= 2; //#2955
            $k[$j++] = 'bwipp.GS1tooShort#2955'; //#2955
            $k[$j++] = "IBAN too short"; //#2955
            $k[$j++] = false; //#2955
            return true; //#2955
        } //#2955
        var _Vg = $k[$j - 1]; //#2956
        $k[$j++] = true; //#2961
        $forall(_Vg, function() { //#2961
            var _Vh = $s(1); //#2957
            $k[$j++] = _Vh; //#2957
            $k[$j++] = _Vh; //#2957
            $k[$j++] = 0; //#2957
            $r(4, -1); //#2957
            var _Vi = $k[--$j]; //#2957
            var _Vj = $k[--$j]; //#2957
            $put($k[--$j], _Vj, _Vi); //#2957
            $search("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", $k[--$j]); //#2959
            if (!$k[--$j]) { //#2959
                $j -= 2; //#2959
                $k[$j++] = false; //#2959
                return true; //#2959
            } //#2959
            $j -= 3; //#2960
        }); //#2960
        if (!$k[--$j]) { //#2962
            $j--; //#2962
            $k[$j++] = 'bwipp.GS1badIBANcharacter#2962'; //#2962
            $k[$j++] = "Invalid IBAN character"; //#2962
            $k[$j++] = false; //#2962
            return true; //#2962
        } //#2962
        var _Vo = $k[$j - 1]; //#2963
        var _Vp = Infinity; //#2963
        var _Vq = _Vo; //#2963
        $k[$j++] = _Vp; //#2963
        var _Vr = Infinity; //#2963
        var _Vs = _Vq; //#2963
        $k[$j++] = _Vr; //#2963
        $forall(_Vs); //#2963
        $r($counttomark(), -4); //#2963
        $astore($a($counttomark())); //#2963
        var _Vw = $k[--$j]; //#2963
        $k[$j - 1] = _Vw; //#2963
        $forall($k[--$j], function() { //#2965
            var _W0 = $k[--$j] - 48; //#2964
            $k[$j++] = _W0; //#2964
            if (_W0 > 9) { //#2964
                var _W2 = $k[--$j] - 7; //#2964
                $k[$j++] = ~~(_W2 / 10); //#2964
                $k[$j++] = _W2 % 10; //#2964
            } //#2964
        }); //#2964
        $astore($a($counttomark())); //#2965
        var _W5 = $k[--$j]; //#2965
        $k[$j - 1] = _W5; //#2965
        var _W7 = $k[--$j]; //#2966
        $k[$j++] = 0; //#2966
        $forall(_W7, function() { //#2966
            var _W8 = $k[--$j]; //#2966
            var _W9 = $k[--$j]; //#2966
            $k[$j++] = ($f(_W8 + (_W9 * 10))) % 97; //#2966
        }); //#2966
        if ($k[--$j] != 1) { //#2967
            $j -= 2; //#2967
            $k[$j++] = 'bwipp.GS1badIBANchecksum#2967'; //#2967
            $k[$j++] = "IBAN checksum incorrect"; //#2967
            $k[$j++] = false; //#2967
            return true; //#2967
        } //#2967
        var _WC = $geti($k[--$j], 0, 2); //#2968
        $k[$j++] = _WC; //#2968
        $_.lintiso3166alpha2(); //#2968
    }; //#2968
    $_.lintzero = function() {
        if ($ne($k[--$j], "0")) { //#2972
            $j--; //#2972
            $k[$j++] = 'bwipp.GS1zeroRequired#2972'; //#2972
            $k[$j++] = "Zero is required"; //#2972
            $k[$j++] = false; //#2972
            return true; //#2972
        } //#2972
    }; //#2972
    $_.lintnonzero = function() {
        var _WE = $k[--$j]; //#2976
        $k[$j++] = false; //#2976
        $forall(_WE, function() { //#2976
            if ($k[--$j] != 48) { //#2976
                $k[$j - 1] = true; //#2976
            } //#2976
        }); //#2976
        if (!$k[--$j]) { //#2977
            $j--; //#2977
            $k[$j++] = 'bwipp.GS1zeroNotPermitted#2977'; //#2977
            $k[$j++] = "Zero not permitted"; //#2977
            $k[$j++] = false; //#2977
            return true; //#2977
        } //#2977
    }; //#2977
    $_.lintnozeroprefix = function() {
        var _WH = $k[--$j]; //#2981
        if ((_WH.length > 1) && ($get(_WH, 0) == 48)) { //#2983
            $j--; //#2982
            $k[$j++] = 'bwipp.GS1badZeroPrefix#2982'; //#2982
            $k[$j++] = "Zero prefix is not permitted"; //#2982
            $k[$j++] = false; //#2982
            return true; //#2982
        } //#2982
    }; //#2982
    $_.linthasnondigit = function() {
        var _WJ = $k[--$j]; //#2987
        $k[$j++] = false; //#2987
        $forall(_WJ, function() { //#2987
            var _WK = $k[--$j]; //#2987
            if ((_WK < 48) || (_WK > 57)) { //#2987
                $k[$j - 1] = true; //#2987
                return true; //#2987
            } //#2987
        }); //#2987
        if (!$k[--$j]) { //#2988
            $k[$j++] = 'bwipp.GS1requiresNonDigit#2988'; //#2988
            $k[$j++] = "A non-digit character is required"; //#2988
            $k[$j++] = false; //#2988
            return true; //#2988
        } //#2988
    }; //#2988
    $_.linthyphen = function() {
        if ($ne($k[--$j], "-")) { //#2992
            $j--; //#2992
            $k[$j++] = 'bwipp.GS1hyphenRequired#2992'; //#2992
            $k[$j++] = "A hyphen is required"; //#2992
            $k[$j++] = false; //#2992
            return true; //#2992
        } //#2992
    }; //#2992
    $_.lintyyyymmd0 = function() {
        var _WN = $k[$j - 1]; //#2996
        var _WP = $cvi($geti(_WN, 4, 2)); //#2996
        if ((_WP < 1) || (_WP > 12)) { //#2996
            $j--; //#2996
            $k[$j++] = 'bwipp.GS1badMonth#2996'; //#2996
            $k[$j++] = "Invalid month"; //#2996
            $k[$j++] = false; //#2996
            return true; //#2996
        } //#2996
        var _WQ = $k[$j - 1]; //#2997
        var _WS = $cvi($geti(_WQ, 0, 4)); //#2997
        $k[$j++] = ((_WS % 400) == 0) || (((_WS % 4) == 0) && ((_WS % 100) != 0)); //#2998
        $k[$j++] = Infinity; //#2998
        $k[$j++] = 31; //#2998
        $r(3, -1); //#2998
        var _WU = $k[--$j] ? 29 : 28; //#2998
        $k[$j++] = _WU; //#2998
        $k[$j++] = 31; //#2998
        $k[$j++] = 30; //#2998
        $k[$j++] = 31; //#2998
        $k[$j++] = 30; //#2998
        $k[$j++] = 31; //#2998
        $k[$j++] = 31; //#2998
        $k[$j++] = 30; //#2998
        $k[$j++] = 31; //#2998
        $k[$j++] = 30; //#2998
        $k[$j++] = 31; //#2998
        var _WV = $a(); //#2998
        var _WW = $k[--$j]; //#2999
        if ($get(_WV, $cvi($geti(_WW, 4, 2)) - 1) < $cvi($geti(_WW, 6, 2))) { //#3000
            $j--; //#3000
            $k[$j++] = 'bwipp.GS1badDay#3000'; //#3000
            $k[$j++] = "Invalid day of month"; //#3000
            $k[$j++] = false; //#3000
            return true; //#3000
        } //#3000
    }; //#3000
    $_.lintyyyymmdd = function() {
        var _Wa = $k[$j - 1]; //#3004
        if (_Wa.length != 8) { //#3004
            $j--; //#3004
            $k[$j++] = 'bwipp.GS1badDateLength#3004'; //#3004
            $k[$j++] = "Invalid length for yyyymmdd date"; //#3004
            $k[$j++] = false; //#3004
            return true; //#3004
        } //#3004
        var _Wb = $k[$j - 1]; //#3005
        if ($cvi($geti(_Wb, 6, 2)) < 1) { //#3005
            $j--; //#3005
            $k[$j++] = 'bwipp.GS1badDay#3005'; //#3005
            $k[$j++] = "Invalid day of month"; //#3005
            $k[$j++] = false; //#3005
            return true; //#3005
        } //#3005
        $_.lintyyyymmd0(); //#3006
    }; //#3006
    $_.lintyymmd0 = function() {
        var _Wd = $k[$j - 1]; //#3010
        var _Wf = $cvi($geti(_Wd, 0, 2)); //#3010
        var _Wg = _Wf - 21; //#3011
        $k[$j++] = _Wf; //#3011
        $k[$j++] = _Wg; //#3011
        if (_Wg >= 51) { //#3011
            $k[$j - 1] = "19"; //#3011
        } else { //#3011
            if ($k[--$j] <= -50) { //#3011
                $k[$j++] = "21"; //#3011
            } else { //#3011
                $k[$j++] = "20"; //#3011
            } //#3011
        } //#3011
        var _Wi = $k[--$j]; //#3011
        $k[$j - 1] = _Wi; //#3011
        var _Wk = $s(8); //#3012
        $k[$j++] = _Wk; //#3012
        $k[$j++] = _Wk; //#3012
        $k[$j++] = 0; //#3012
        $r(4, -1); //#3012
        var _Wl = $k[--$j]; //#3012
        var _Wm = $k[--$j]; //#3012
        $puti($k[--$j], _Wm, _Wl); //#3012
        var _Wo = $k[$j - 1]; //#3012
        $k[$j++] = _Wo; //#3012
        $k[$j++] = 2; //#3012
        $r(4, -1); //#3012
        var _Wp = $k[--$j]; //#3012
        var _Wq = $k[--$j]; //#3012
        $puti($k[--$j], _Wq, _Wp); //#3012
        $_.lintyyyymmd0(); //#3013
    }; //#3013
    $_.lintyymmdd = function() {
        var _Ws = $k[$j - 1]; //#3017
        if (_Ws.length != 6) { //#3017
            $j--; //#3017
            $k[$j++] = 'bwipp.GS1badDateLength#3017'; //#3017
            $k[$j++] = "Invalid length for yymmdd date"; //#3017
            $k[$j++] = false; //#3017
            return true; //#3017
        } //#3017
        var _Wt = $k[$j - 1]; //#3018
        if ($cvi($geti(_Wt, 4, 2)) < 1) { //#3018
            $j--; //#3018
            $k[$j++] = 'bwipp.GS1badDay#3018'; //#3018
            $k[$j++] = "Invalid day of month"; //#3018
            $k[$j++] = false; //#3018
            return true; //#3018
        } //#3018
        $_.lintyymmd0(); //#3019
    }; //#3019
    $_.linthh = function() {
        if ($cvi($geti($k[--$j], 0, 2)) > 23) { //#3023
            $j -= 2; //#3023
            $k[$j++] = 'bwipp.GS1badHour#3023'; //#3023
            $k[$j++] = "Invalid hour of day"; //#3023
            $k[$j++] = false; //#3023
            return true; //#3023
        } //#3023
    }; //#3023
    $_.lintmi = function() {
        if ($cvi($geti($k[--$j], 0, 2)) > 59) { //#3027
            $j--; //#3027
            $k[$j++] = 'bwipp.GS1badMinute#3027'; //#3027
            $k[$j++] = "Invalid minute in the hour"; //#3027
            $k[$j++] = false; //#3027
            return true; //#3027
        } //#3027
    }; //#3027
    $_.lintss = function() {
        if ($cvi($geti($k[--$j], 0, 2)) > 59) { //#3031
            $j--; //#3031
            $k[$j++] = 'bwipp.GS1badSecond#3031'; //#3031
            $k[$j++] = "Invalid second in the minute"; //#3031
            $k[$j++] = false; //#3031
            return true; //#3031
        } //#3031
    }; //#3031
    $_.linthhmi = function() {
        var _X1 = $k[$j - 1]; //#3035
        if ($cvi($geti(_X1, 0, 2)) > 23) { //#3035
            $j -= 2; //#3035
            $k[$j++] = 'bwipp.GS1badHour#3035'; //#3035
            $k[$j++] = "Invalid hour of day"; //#3035
            $k[$j++] = false; //#3035
            return true; //#3035
        } //#3035
        if ($cvi($geti($k[--$j], 2, 2)) > 59) { //#3036
            $j--; //#3036
            $k[$j++] = 'bwipp.GS1badMinute#3036'; //#3036
            $k[$j++] = "Invalid minute in the hour"; //#3036
            $k[$j++] = false; //#3036
            return true; //#3036
        } //#3036
    }; //#3036
    $_.lintmmoptss = function() {
        var _X5 = $k[$j - 1]; //#3040
        var _X6 = _X5.length; //#3040
        if ((_X6 != 2) && (_X6 != 4)) { //#3042
            $j--; //#3041
            $k[$j++] = 'bwipp.GS1badTimeLength#3041'; //#3041
            $k[$j++] = "Invalid length for optional minutes and seconds"; //#3041
            $k[$j++] = false; //#3041
            return true; //#3041
        } //#3041
        var _X7 = $k[$j - 1]; //#3043
        if ($cvi($geti(_X7, 0, 2)) > 59) { //#3043
            $j -= 2; //#3043
            $k[$j++] = 'bwipp.GS1badMinute#3043'; //#3043
            $k[$j++] = "Invalid minute in the hour"; //#3043
            $k[$j++] = false; //#3043
            return true; //#3043
        } //#3043
        var _X9 = $k[$j - 1]; //#3044
        if (_X9.length >= 4) { //#3046
            var _XA = $k[$j - 1]; //#3045
            if ($cvi($geti(_XA, 2, 2)) > 59) { //#3045
                $j -= 2; //#3045
                $k[$j++] = 'bwipp.GS1badSecond#3045'; //#3045
                $k[$j++] = "Invalid second in the minute"; //#3045
                $k[$j++] = false; //#3045
                return true; //#3045
            } //#3045
        } //#3045
        $j--; //#3047
    }; //#3047
    $_.lintyesno = function() {
        var _XC = $k[--$j]; //#3051
        if ($ne(_XC, "0") && $ne(_XC, "1")) { //#3053
            $j--; //#3052
            $k[$j++] = 'bwipp.GS1badBoolean#3052'; //#3052
            $k[$j++] = "Neither 0 nor 1 for yes or no"; //#3052
            $k[$j++] = false; //#3052
            return true; //#3052
        } //#3052
    }; //#3052
    $_.lintwinding = function() {
        var _XD = $k[--$j]; //#3057
        if ($ne(_XD, "0") && ($ne(_XD, "1") && $ne(_XD, "9"))) { //#3059
            $j--; //#3058
            $k[$j++] = 'bwipp.GS1badWinding#3058'; //#3058
            $k[$j++] = "Invalid winding direction"; //#3058
            $k[$j++] = false; //#3058
            return true; //#3058
        } //#3058
    }; //#3058
    $_.lintpieceoftotal = function() {
        var _XE = $k[$j - 1]; //#3063
        if ((_XE.length % 2) != 0) { //#3063
            $j -= 2; //#3063
            $k[$j++] = 'bwipp.GS1badPieceTotalLength#3063'; //#3063
            $k[$j++] = "Invalid piece/total length"; //#3063
            $k[$j++] = false; //#3063
            return true; //#3063
        } //#3063
        var _XF = $k[$j - 1]; //#3064
        var _XH = $cvi($geti(_XF, 0, ~~(_XF.length / 2))); //#3065
        $k[$j++] = _XH; //#3065
        if (_XH == 0) { //#3065
            $j -= 3; //#3065
            $k[$j++] = 'bwipp.GS1badPieceNumber#3065'; //#3065
            $k[$j++] = "Invalid piece number"; //#3065
            $k[$j++] = false; //#3065
            return true; //#3065
        } //#3065
        var _XI = $k[--$j]; //#3066
        var _XJ = $k[--$j]; //#3066
        var _XK = ~~(_XJ.length / 2); //#3066
        var _XM = $cvi($geti(_XJ, _XK, _XK)); //#3067
        $k[$j++] = _XI; //#3067
        $k[$j++] = _XM; //#3067
        if (_XM == 0) { //#3067
            $j -= 3; //#3067
            $k[$j++] = 'bwipp.GS1badPieceTotal#3067'; //#3067
            $k[$j++] = "Invalid total number"; //#3067
            $k[$j++] = false; //#3067
            return true; //#3067
        } //#3067
        var _XN = $k[--$j]; //#3068
        if ($gt($k[--$j], _XN)) { //#3068
            $j--; //#3068
            $k[$j++] = 'bwipp.GS1pieceExceedsTotal#3068'; //#3068
            $k[$j++] = "Piece number exceeds total"; //#3068
            $k[$j++] = false; //#3068
            return true; //#3068
        } //#3068
    }; //#3068
    $_.lintposinseqslash = function() {
        $search($k[--$j], "/"); //#3072
        if (!$k[--$j]) { //#3074
            $j--; //#3073
            $k[$j++] = 'bwipp.invalidPosInSeqFormat#3073'; //#3073
            $k[$j++] = "Invalid <pos>/<end> format"; //#3073
            $k[$j++] = false; //#3073
            return true; //#3073
        } //#3073
        var _XR = $k[--$j]; //#3075
        $k[$j - 1] = _XR; //#3075
        var _XT = $k[$j - 1]; //#3076
        if (_XT.length == 0) { //#3078
            $j -= 2; //#3077
            $k[$j++] = 'bwipp.invalidPosInSeqFormat#3077'; //#3077
            $k[$j++] = "Invalid <pos>/<end> format"; //#3077
            $k[$j++] = false; //#3077
            return true; //#3077
        } //#3077
        var _XV = $k[$j - 2]; //#3079
        if (_XV.length == 0) { //#3081
            $j -= 2; //#3080
            $k[$j++] = 'bwipp.invalidPosInSeqFormat#3080'; //#3080
            $k[$j++] = "Invalid <pos>/<end> format"; //#3080
            $k[$j++] = false; //#3080
            return true; //#3080
        } //#3080
        var _XW = $k[$j - 1]; //#3082
        $k[$j++] = true; //#3084
        $forall(_XW, function() { //#3084
            var _XX = $k[--$j]; //#3083
            if ((_XX < 48) || (_XX > 57)) { //#3083
                $k[$j - 1] = false; //#3083
                return true; //#3083
            } //#3083
        }); //#3083
        if (!$k[--$j]) { //#3085
            $j -= 2; //#3085
            $k[$j++] = 'bwipp.positionNotNumeric#3085'; //#3085
            $k[$j++] = "Position is not numeric"; //#3085
            $k[$j++] = false; //#3085
            return true; //#3085
        } //#3085
        var _Xa = $k[$j - 2]; //#3086
        $k[$j++] = true; //#3088
        $forall(_Xa, function() { //#3088
            var _Xb = $k[--$j]; //#3087
            if ((_Xb < 48) || (_Xb > 57)) { //#3087
                $k[$j - 1] = false; //#3087
                return true; //#3087
            } //#3087
        }); //#3087
        if (!$k[--$j]) { //#3089
            $j -= 2; //#3089
            $k[$j++] = 'bwipp.endNotNumeric#3089'; //#3089
            $k[$j++] = "End is not numeric"; //#3089
            $k[$j++] = false; //#3089
            return true; //#3089
        } //#3089
        var _Xd = $k[$j - 1]; //#3090
        if ($get(_Xd, 0) == 48) { //#3090
            $j -= 2; //#3090
            $k[$j++] = 'bwipp.positionZeroPrefix#3090'; //#3090
            $k[$j++] = "Position cannot have zero prefix"; //#3090
            $k[$j++] = false; //#3090
            return true; //#3090
        } //#3090
        var _Xg = $k[$j - 2]; //#3091
        if ($get(_Xg, 0) == 48) { //#3091
            $j -= 2; //#3091
            $k[$j++] = 'bwipp.endZeroPrefix#3091'; //#3091
            $k[$j++] = "End cannot have zero prefix"; //#3091
            $k[$j++] = false; //#3091
            return true; //#3091
        } //#3091
        var _Xi = $k[--$j]; //#3092
        if ($cvi($k[--$j]) < $cvi(_Xi)) { //#3093
            $k[$j++] = 'bwipp.positionExceedsEnd#3093'; //#3093
            $k[$j++] = "Position exceeds end"; //#3093
            $k[$j++] = false; //#3093
            return true; //#3093
        } //#3093
    }; //#3093
    $_.lintpcenc = function() {
        for (;;) { //#3107
            $search($k[--$j], "%"); //#3098
            if (!$k[--$j]) { //#3098
                $j--; //#3098
                break; //#3098
            } //#3098
            $j -= 2; //#3099
            var _Xm = $k[$j - 1]; //#3099
            if (_Xm.length < 2) { //#3099
                $j -= 2; //#3099
                $k[$j++] = 'bwipp.GS1badPercentEscape#3099'; //#3099
                $k[$j++] = "Invalid % escape"; //#3099
                $k[$j++] = false; //#3099
                break; //#3099
            } //#3099
            var _Xn = $k[$j - 1]; //#3100
            var _Xo = $geti(_Xn, 0, 2); //#3100
            $k[$j++] = true; //#3105
            for (var _Xp = 0, _Xq = _Xo.length; _Xp < _Xq; _Xp++) { //#3105
                var _Xs = $s(1); //#3101
                $put(_Xs, 0, $get(_Xo, _Xp)); //#3101
                $search("0123456789ABCDEFabcdef", _Xs); //#3103
                if (!$k[--$j]) { //#3103
                    $j -= 2; //#3103
                    $k[$j++] = false; //#3103
                    break; //#3103
                } //#3103
                $j -= 3; //#3104
            } //#3104
            if (!$k[--$j]) { //#3106
                $j -= 2; //#3106
                $k[$j++] = 'bwipp.GS1badPercentChars#3106'; //#3106
                $k[$j++] = "Invalid characters for percent encoding"; //#3106
                $k[$j++] = false; //#3106
                break; //#3106
            } //#3106
        } //#3106
    }; //#3106
    $_.lintcouponcode = function() {
        var _Xv = $k[$j - 1]; //#3111
        $k[$j++] = true; //#3113
        $forall(_Xv, function() { //#3113
            var _Xw = $k[--$j]; //#3112
            if ((_Xw < 48) || (_Xw > 57)) { //#3112
                $k[$j - 1] = false; //#3112
                return true; //#3112
            } //#3112
        }); //#3112
        if (!$k[--$j]) { //#3114
            $j -= 2; //#3114
            $k[$j++] = 'bwipp.GS1couponNotNumeric#3114'; //#3114
            $k[$j++] = "Coupon not numeric"; //#3114
            $k[$j++] = false; //#3114
            return true; //#3114
        } //#3114
        var _Xy = $k[$j - 1]; //#3117
        if (_Xy.length < 1) { //#3119
            $j -= 2; //#3118
            $k[$j++] = 'bwipp.GS1couponTooShortGCPVLI#3118'; //#3118
            $k[$j++] = "Coupon too short: Missing GCP VLI"; //#3118
            $k[$j++] = false; //#3118
            return true; //#3118
        } //#3118
        var _Xz = $k[$j - 1]; //#3120
        var _Y1 = $cvi($geti(_Xz, 0, 1)); //#3120
        $k[$j++] = _Y1; //#3122
        if (_Y1 > 6) { //#3122
            $j -= 2; //#3121
            $k[$j++] = 'bwipp.GS1couponBadGCPVLI#3121'; //#3121
            $k[$j++] = "Coupon GCP length indicator must be 0-6"; //#3121
            $k[$j++] = false; //#3121
            return true; //#3121
        } //#3121
        var _Y2 = $k[--$j]; //#3123
        var _Y3 = $k[$j - 1]; //#3124
        $k[$j++] = (_Y2 + 6) + 1; //#3126
        if (((_Y2 + 6) + 1) > _Y3.length) { //#3126
            $j -= 3; //#3125
            $k[$j++] = 'bwipp.GS1couponTooShortGCP#3125'; //#3125
            $k[$j++] = "Coupon too short: GCP truncated"; //#3125
            $k[$j++] = false; //#3125
            return true; //#3125
        } //#3125
        var _Y4 = $k[--$j]; //#3127
        var _Y5 = $k[--$j]; //#3127
        var _Y6 = $geti(_Y5, _Y4, $f(_Y5.length - _Y4)); //#3127
        $k[$j++] = _Y6; //#3132
        if (_Y6.length < 6) { //#3132
            $j -= 2; //#3131
            $k[$j++] = 'bwipp.GS1couponTooShortOfferCode#3131'; //#3131
            $k[$j++] = "Coupon too short: Offer Code truncated"; //#3131
            $k[$j++] = false; //#3131
            return true; //#3131
        } //#3131
        var _Y7 = $k[--$j]; //#3133
        var _Y8 = $geti(_Y7, 6, _Y7.length - 6); //#3133
        $k[$j++] = _Y8; //#3138
        if (_Y8.length < 1) { //#3138
            $j -= 2; //#3137
            $k[$j++] = 'bwipp.GS1couponTooShortSaveValueVLI#3137'; //#3137
            $k[$j++] = "Coupon too short: Missing Save Value VLI"; //#3137
            $k[$j++] = false; //#3137
            return true; //#3137
        } //#3137
        var _Y9 = $k[$j - 1]; //#3139
        var _YB = $cvi($geti(_Y9, 0, 1)); //#3139
        $k[$j++] = _YB; //#3141
        if ((_YB < 1) || (_YB > 5)) { //#3141
            $j -= 2; //#3140
            $k[$j++] = 'bwipp.GS1couponBadSaveValueVLI#3140'; //#3140
            $k[$j++] = "Coupon Save Value length indicator must be 1-5"; //#3140
            $k[$j++] = false; //#3140
            return true; //#3140
        } //#3140
        var _YC = $k[--$j]; //#3142
        var _YD = $k[$j - 1]; //#3143
        $k[$j++] = _YC + 1; //#3145
        if ((_YC + 1) > _YD.length) { //#3145
            $j -= 3; //#3144
            $k[$j++] = 'bwipp.GS1couponTooShortSaveValue#3144'; //#3144
            $k[$j++] = "Coupon too short: Save Value truncated"; //#3144
            $k[$j++] = false; //#3144
            return true; //#3144
        } //#3144
        var _YE = $k[--$j]; //#3146
        var _YF = $k[--$j]; //#3146
        var _YG = $geti(_YF, _YE, $f(_YF.length - _YE)); //#3146
        $k[$j++] = _YG; //#3151
        if (_YG.length < 1) { //#3151
            $j -= 2; //#3150
            $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirementVLI#3150'; //#3150
            $k[$j++] = "Coupon too short: Missing 1st Purchase Requirement VLI"; //#3150
            $k[$j++] = false; //#3150
            return true; //#3150
        } //#3150
        var _YH = $k[$j - 1]; //#3152
        var _YJ = $cvi($geti(_YH, 0, 1)); //#3152
        $k[$j++] = _YJ; //#3154
        if ((_YJ < 1) || (_YJ > 5)) { //#3154
            $j -= 2; //#3153
            $k[$j++] = 'bwipp.GS1couponBad1stPurchaseRequirementVLI#3153'; //#3153
            $k[$j++] = "Coupon 1st Purchase Requirement length indicator must be 1-5"; //#3153
            $k[$j++] = false; //#3153
            return true; //#3153
        } //#3153
        var _YK = $k[--$j]; //#3155
        var _YL = $k[$j - 1]; //#3156
        $k[$j++] = _YK + 1; //#3158
        if ((_YK + 1) > _YL.length) { //#3158
            $j -= 3; //#3157
            $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirement#3157'; //#3157
            $k[$j++] = "Coupon too short: 1st Purchase Requirement truncated"; //#3157
            $k[$j++] = false; //#3157
            return true; //#3157
        } //#3157
        var _YM = $k[--$j]; //#3159
        var _YN = $k[--$j]; //#3159
        var _YO = $geti(_YN, _YM, $f(_YN.length - _YM)); //#3159
        $k[$j++] = _YO; //#3164
        if (_YO.length < 1) { //#3164
            $j -= 2; //#3163
            $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseRequirementCode#3163'; //#3163
            $k[$j++] = "Coupon too short: Missing 1st Purchase Requirement Code"; //#3163
            $k[$j++] = false; //#3163
            return true; //#3163
        } //#3163
        var _YP = $k[$j - 1]; //#3165
        var _YR = $cvi($geti(_YP, 0, 1)); //#3165
        if ((_YR > 4) && (_YR != 9)) { //#3167
            $j -= 2; //#3166
            $k[$j++] = 'bwipp.GS1couponBad1stPurchaseRequirementCode#3166'; //#3166
            $k[$j++] = "Coupon 1st Purchase Requirement Code must be 0-4 or 9"; //#3166
            $k[$j++] = false; //#3166
            return true; //#3166
        } //#3166
        var _YS = $k[--$j]; //#3168
        var _YT = $geti(_YS, 1, _YS.length - 1); //#3168
        $k[$j++] = _YT; //#3173
        if (_YT.length < 3) { //#3173
            $j -= 2; //#3172
            $k[$j++] = 'bwipp.GS1couponTooShort1stPurchaseFamilyCode#3172'; //#3172
            $k[$j++] = "Coupon too short: 1st Purchase Family Code truncated"; //#3172
            $k[$j++] = false; //#3172
            return true; //#3172
        } //#3172
        var _YU = $k[--$j]; //#3174
        var _YV = $geti(_YU, 3, _YU.length - 3); //#3174
        $k[$j++] = _YV; //#3230
        if (_YV.length >= 1) { //#3230
            var _YW = $k[$j - 1]; //#3177
            if ($cvi($geti(_YW, 0, 1)) == 1) { //#3230
                var _YY = $k[--$j]; //#3178
                var _YZ = $geti(_YY, 1, _YY.length - 1); //#3178
                $k[$j++] = _YZ; //#3183
                if (_YZ.length < 1) { //#3183
                    $j -= 2; //#3182
                    $k[$j++] = 'bwipp.GS1couponTooShortAdditionalPurchaseRulesCode#3182'; //#3182
                    $k[$j++] = "Coupon too short: Missing Additional Purchase Rules Code"; //#3182
                    $k[$j++] = false; //#3182
                    return true; //#3182
                } //#3182
                var _Ya = $k[$j - 1]; //#3184
                if ($cvi($geti(_Ya, 0, 1)) > 3) { //#3186
                    $j -= 2; //#3185
                    $k[$j++] = 'bwipp.GS1couponBadAdditionalPurchaseRulesCode#3185'; //#3185
                    $k[$j++] = "Coupon Additional Purchase Rules Code must be 0-3"; //#3185
                    $k[$j++] = false; //#3185
                    return true; //#3185
                } //#3185
                var _Yc = $k[--$j]; //#3187
                var _Yd = $geti(_Yc, 1, _Yc.length - 1); //#3187
                $k[$j++] = _Yd; //#3192
                if (_Yd.length < 1) { //#3192
                    $j -= 2; //#3191
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirementVLI#3191'; //#3191
                    $k[$j++] = "Coupon too short: Missing 2nd Purchase Requirement VLI"; //#3191
                    $k[$j++] = false; //#3191
                    return true; //#3191
                } //#3191
                var _Ye = $k[$j - 1]; //#3193
                var _Yg = $cvi($geti(_Ye, 0, 1)); //#3193
                $k[$j++] = _Yg; //#3195
                if ((_Yg < 1) || (_Yg > 5)) { //#3195
                    $j -= 2; //#3194
                    $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseRequirementVLI#3194'; //#3194
                    $k[$j++] = "Coupon 2nd Purchase Requirement length indicator must be 1-5"; //#3194
                    $k[$j++] = false; //#3194
                    return true; //#3194
                } //#3194
                var _Yh = $k[--$j]; //#3196
                var _Yi = $k[$j - 1]; //#3197
                $k[$j++] = _Yh + 1; //#3199
                if ((_Yh + 1) > _Yi.length) { //#3199
                    $j -= 3; //#3198
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirement#3198'; //#3198
                    $k[$j++] = "Coupon too short: 2nd Purchase Requirement truncated"; //#3198
                    $k[$j++] = false; //#3198
                    return true; //#3198
                } //#3198
                var _Yj = $k[--$j]; //#3200
                var _Yk = $k[--$j]; //#3200
                var _Yl = $geti(_Yk, _Yj, $f(_Yk.length - _Yj)); //#3200
                $k[$j++] = _Yl; //#3205
                if (_Yl.length < 1) { //#3205
                    $j -= 2; //#3204
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseRequirementCode#3204'; //#3204
                    $k[$j++] = "Coupon too short: Missing 2nd Purchase Requirement Code"; //#3204
                    $k[$j++] = false; //#3204
                    return true; //#3204
                } //#3204
                var _Ym = $k[$j - 1]; //#3206
                var _Yo = $cvi($geti(_Ym, 0, 1)); //#3206
                if ((_Yo > 4) && (_Yo != 9)) { //#3208
                    $j -= 2; //#3207
                    $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseRequirementCode#3207'; //#3207
                    $k[$j++] = "Coupon 2nd Purchase Requirement Code must be 0-4 or 9"; //#3207
                    $k[$j++] = false; //#3207
                    return true; //#3207
                } //#3207
                var _Yp = $k[--$j]; //#3209
                var _Yq = $geti(_Yp, 1, _Yp.length - 1); //#3209
                $k[$j++] = _Yq; //#3214
                if (_Yq.length < 3) { //#3214
                    $j -= 2; //#3213
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseFamilyCode#3213'; //#3213
                    $k[$j++] = "Coupon too short: 2nd Purchase Family Code truncated"; //#3213
                    $k[$j++] = false; //#3213
                    return true; //#3213
                } //#3213
                var _Yr = $k[--$j]; //#3215
                var _Ys = $geti(_Yr, 3, _Yr.length - 3); //#3215
                $k[$j++] = _Ys; //#3220
                if (_Ys.length < 1) { //#3220
                    $j -= 2; //#3219
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseGCPVLI#3219'; //#3219
                    $k[$j++] = "Coupon too short: Missing 2nd Purchase GCP VLI"; //#3219
                    $k[$j++] = false; //#3219
                    return true; //#3219
                } //#3219
                var _Yt = $k[$j - 1]; //#3221
                var _Yv = $cvi($geti(_Yt, 0, 1)); //#3221
                $k[$j++] = _Yv; //#3223
                if ((_Yv > 6) && (_Yv != 9)) { //#3223
                    $j -= 2; //#3222
                    $k[$j++] = 'bwipp.GS1couponBad2ndPurchaseGCPVLI#3222'; //#3222
                    $k[$j++] = "Coupon 2nd Purchase GCP length indicator must be 0-6 or 9"; //#3222
                    $k[$j++] = false; //#3222
                    return true; //#3222
                } //#3222
                var _Yw = $k[$j - 1]; //#3224
                if (_Yw != 9) { //#3224
                    var _Yx = $k[--$j]; //#3224
                    $k[$j++] = _Yx + 6; //#3224
                } else { //#3224
                    $k[$j - 1] = 0; //#3224
                } //#3224
                var _Yy = $k[--$j]; //#3224
                var _Yz = $k[$j - 1]; //#3225
                $k[$j++] = _Yy + 1; //#3227
                if ((_Yy + 1) > _Yz.length) { //#3227
                    $j -= 3; //#3226
                    $k[$j++] = 'bwipp.GS1couponTooShort2ndPurchaseGCP#3226'; //#3226
                    $k[$j++] = "Coupon too short: 2nd Purchase GCP truncated"; //#3226
                    $k[$j++] = false; //#3226
                    return true; //#3226
                } //#3226
                var _Z0 = $k[--$j]; //#3228
                var _Z1 = $k[--$j]; //#3228
                $k[$j++] = $geti(_Z1, _Z0, $f(_Z1.length - _Z0)); //#3228
            } //#3228
        } //#3228
        var _Z3 = $k[$j - 1]; //#3233
        if (_Z3.length >= 1) { //#3277
            var _Z4 = $k[$j - 1]; //#3233
            if ($cvi($geti(_Z4, 0, 1)) == 2) { //#3277
                var _Z6 = $k[--$j]; //#3234
                var _Z7 = $geti(_Z6, 1, _Z6.length - 1); //#3234
                $k[$j++] = _Z7; //#3239
                if (_Z7.length < 1) { //#3239
                    $j -= 2; //#3238
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirementVLI#3238'; //#3238
                    $k[$j++] = "Coupon too short: Missing 3rd Purchase Requirement VLI"; //#3238
                    $k[$j++] = false; //#3238
                    return true; //#3238
                } //#3238
                var _Z8 = $k[$j - 1]; //#3240
                var _ZA = $cvi($geti(_Z8, 0, 1)); //#3240
                $k[$j++] = _ZA; //#3242
                if ((_ZA < 1) || (_ZA > 5)) { //#3242
                    $j -= 2; //#3241
                    $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseRequirementVLI#3241'; //#3241
                    $k[$j++] = "Coupon 3rd Purchase Requirement length indicator must be 1-5"; //#3241
                    $k[$j++] = false; //#3241
                    return true; //#3241
                } //#3241
                var _ZB = $k[--$j]; //#3243
                var _ZC = $k[$j - 1]; //#3244
                $k[$j++] = _ZB + 1; //#3246
                if ((_ZB + 1) > _ZC.length) { //#3246
                    $j -= 3; //#3245
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirement#3245'; //#3245
                    $k[$j++] = "Coupon too short: 3rd Purchase Requirement truncated"; //#3245
                    $k[$j++] = false; //#3245
                    return true; //#3245
                } //#3245
                var _ZD = $k[--$j]; //#3247
                var _ZE = $k[--$j]; //#3247
                var _ZF = $geti(_ZE, _ZD, $f(_ZE.length - _ZD)); //#3247
                $k[$j++] = _ZF; //#3252
                if (_ZF.length < 1) { //#3252
                    $j -= 2; //#3251
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseRequirementCode#3251'; //#3251
                    $k[$j++] = "Coupon too short: Missing 3rd Purchase Requirement Code"; //#3251
                    $k[$j++] = false; //#3251
                    return true; //#3251
                } //#3251
                var _ZG = $k[$j - 1]; //#3253
                var _ZI = $cvi($geti(_ZG, 0, 1)); //#3253
                if ((_ZI > 4) && (_ZI != 9)) { //#3255
                    $j -= 2; //#3254
                    $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseRequirementCode#3254'; //#3254
                    $k[$j++] = "Coupon 3rd Purchase Requirement Code must be 0-4 or 9"; //#3254
                    $k[$j++] = false; //#3254
                    return true; //#3254
                } //#3254
                var _ZJ = $k[--$j]; //#3256
                var _ZK = $geti(_ZJ, 1, _ZJ.length - 1); //#3256
                $k[$j++] = _ZK; //#3261
                if (_ZK.length < 3) { //#3261
                    $j -= 2; //#3260
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseFamilyCode#3260'; //#3260
                    $k[$j++] = "Coupon too short: 3rd Purchase Family Code truncated"; //#3260
                    $k[$j++] = false; //#3260
                    return true; //#3260
                } //#3260
                var _ZL = $k[--$j]; //#3262
                var _ZM = $geti(_ZL, 3, _ZL.length - 3); //#3262
                $k[$j++] = _ZM; //#3267
                if (_ZM.length < 1) { //#3267
                    $j -= 2; //#3266
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseGCPVLI#3266'; //#3266
                    $k[$j++] = "Coupon too short: Missing 3rd Purchase GCP VLI"; //#3266
                    $k[$j++] = false; //#3266
                    return true; //#3266
                } //#3266
                var _ZN = $k[$j - 1]; //#3268
                var _ZP = $cvi($geti(_ZN, 0, 1)); //#3268
                $k[$j++] = _ZP; //#3270
                if ((_ZP > 6) && (_ZP != 9)) { //#3270
                    $j -= 2; //#3269
                    $k[$j++] = 'bwipp.GS1couponBad3rdPurchaseGCPVLI#3269'; //#3269
                    $k[$j++] = "Coupon 3rd Purchase GCP length indicator must be 0-6 or 9"; //#3269
                    $k[$j++] = false; //#3269
                    return true; //#3269
                } //#3269
                var _ZQ = $k[$j - 1]; //#3271
                if (_ZQ != 9) { //#3271
                    var _ZR = $k[--$j]; //#3271
                    $k[$j++] = _ZR + 6; //#3271
                } else { //#3271
                    $k[$j - 1] = 0; //#3271
                } //#3271
                var _ZS = $k[--$j]; //#3271
                var _ZT = $k[$j - 1]; //#3272
                $k[$j++] = _ZS + 1; //#3274
                if ((_ZS + 1) > _ZT.length) { //#3274
                    $j -= 3; //#3273
                    $k[$j++] = 'bwipp.GS1couponTooShort3rdPurchaseGCP#3273'; //#3273
                    $k[$j++] = "Coupon too short: 3rd Purchase GCP truncated"; //#3273
                    $k[$j++] = false; //#3273
                    return true; //#3273
                } //#3273
                var _ZU = $k[--$j]; //#3275
                var _ZV = $k[--$j]; //#3275
                $k[$j++] = $geti(_ZV, _ZU, $f(_ZV.length - _ZU)); //#3275
            } //#3275
        } //#3275
        $_.couponexpire = -1; //#3280
        var _ZX = $k[$j - 1]; //#3281
        if (_ZX.length >= 1) { //#3299
            var _ZY = $k[$j - 1]; //#3281
            if ($cvi($geti(_ZY, 0, 1)) == 3) { //#3299
                var _Za = $k[--$j]; //#3282
                var _Zb = $geti(_Za, 1, _Za.length - 1); //#3282
                $k[$j++] = _Zb; //#3287
                if (_Zb.length < 6) { //#3287
                    $j -= 2; //#3286
                    $k[$j++] = 'bwipp.GS1couponTooShortExpirationDate#3286'; //#3286
                    $k[$j++] = "Coupon too short: Expiration date"; //#3286
                    $k[$j++] = false; //#3286
                    return true; //#3286
                } //#3286
                var _Zc = $k[$j - 1]; //#3288
                var _Ze = $cvi($geti(_Zc, 2, 2)); //#3288
                if ((_Ze < 1) || (_Ze > 12)) { //#3288
                    $j -= 2; //#3288
                    $k[$j++] = 'bwipp.GS1couponExpirationDateBadMonth#3288'; //#3288
                    $k[$j++] = "Invalid month in expiration date"; //#3288
                    $k[$j++] = false; //#3288
                    return true; //#3288
                } //#3288
                var _Zf = $k[$j - 1]; //#3289
                var _Zh = $cvi($geti(_Zf, 0, 2)); //#3289
                var _Zi = _Zh - 21; //#3290
                $k[$j++] = _Zh; //#3290
                $k[$j++] = _Zi; //#3290
                if (_Zi >= 51) { //#3290
                    $j--; //#3290
                    var _Zj = $k[--$j]; //#3290
                    $k[$j++] = _Zj + 1900; //#3290
                } else { //#3290
                    if ($k[--$j] <= -50) { //#3290
                        var _Zl = $k[--$j]; //#3290
                        $k[$j++] = _Zl + 2100; //#3290
                    } else { //#3290
                        var _Zm = $k[--$j]; //#3290
                        $k[$j++] = _Zm + 2000; //#3290
                    } //#3290
                } //#3290
                var _Zn = $k[--$j]; //#3291
                $k[$j++] = ((_Zn % 400) == 0) || (((_Zn % 4) == 0) && ((_Zn % 100) != 0)); //#3292
                $k[$j++] = Infinity; //#3292
                $k[$j++] = 31; //#3292
                $r(3, -1); //#3292
                var _Zp = $k[--$j] ? 29 : 28; //#3292
                $k[$j++] = _Zp; //#3292
                $k[$j++] = 31; //#3292
                $k[$j++] = 30; //#3292
                $k[$j++] = 31; //#3292
                $k[$j++] = 30; //#3292
                $k[$j++] = 31; //#3292
                $k[$j++] = 31; //#3292
                $k[$j++] = 30; //#3292
                $k[$j++] = 31; //#3292
                $k[$j++] = 30; //#3292
                $k[$j++] = 31; //#3292
                var _Zq = $a(); //#3292
                var _Zr = $k[$j - 1]; //#3293
                var _Zv = $cvi($geti(_Zr, 4, 2)); //#3294
                if (($get(_Zq, $cvi($geti(_Zr, 2, 2)) - 1) < _Zv) || (_Zv < 1)) { //#3295
                    $j -= 2; //#3295
                    $k[$j++] = 'bwipp.GS1couponExpirationDateBadDay#3295'; //#3295
                    $k[$j++] = "Invalid day of month in expiration date"; //#3295
                    $k[$j++] = false; //#3295
                    return true; //#3295
                } //#3295
                var _Zw = $k[--$j]; //#3296
                $_.couponexpire = $cvi($geti(_Zw, 0, 6)); //#3296
                $k[$j++] = $geti(_Zw, 6, _Zw.length - 6); //#3297
            } //#3297
        } //#3297
        var _Zz = $k[$j - 1]; //#3302
        if (_Zz.length >= 1) { //#3323
            var _a0 = $k[$j - 1]; //#3302
            if ($cvi($geti(_a0, 0, 1)) == 4) { //#3323
                var _a2 = $k[--$j]; //#3303
                var _a3 = $geti(_a2, 1, _a2.length - 1); //#3303
                $k[$j++] = _a3; //#3308
                if (_a3.length < 6) { //#3308
                    $j -= 2; //#3307
                    $k[$j++] = 'bwipp.GS1couponTooShortStartDate#3307'; //#3307
                    $k[$j++] = "Coupon too short: Start date"; //#3307
                    $k[$j++] = false; //#3307
                    return true; //#3307
                } //#3307
                var _a4 = $k[$j - 1]; //#3309
                var _a6 = $cvi($geti(_a4, 2, 2)); //#3309
                if ((_a6 < 1) || (_a6 > 12)) { //#3309
                    $j -= 2; //#3309
                    $k[$j++] = 'bwipp.GS1couponStartDateBadMonth#3309'; //#3309
                    $k[$j++] = "Invalid month in start date"; //#3309
                    $k[$j++] = false; //#3309
                    return true; //#3309
                } //#3309
                var _a7 = $k[$j - 1]; //#3310
                var _a9 = $cvi($geti(_a7, 0, 2)); //#3310
                var _aA = _a9 - 21; //#3311
                $k[$j++] = _a9; //#3311
                $k[$j++] = _aA; //#3311
                if (_aA >= 51) { //#3311
                    $j--; //#3311
                    var _aB = $k[--$j]; //#3311
                    $k[$j++] = _aB + 1900; //#3311
                } else { //#3311
                    if ($k[--$j] <= -50) { //#3311
                        var _aD = $k[--$j]; //#3311
                        $k[$j++] = _aD + 2100; //#3311
                    } else { //#3311
                        var _aE = $k[--$j]; //#3311
                        $k[$j++] = _aE + 2000; //#3311
                    } //#3311
                } //#3311
                var _aF = $k[--$j]; //#3312
                $k[$j++] = ((_aF % 400) == 0) || (((_aF % 4) == 0) && ((_aF % 100) != 0)); //#3313
                $k[$j++] = Infinity; //#3313
                $k[$j++] = 31; //#3313
                $r(3, -1); //#3313
                var _aH = $k[--$j] ? 29 : 28; //#3313
                $k[$j++] = _aH; //#3313
                $k[$j++] = 31; //#3313
                $k[$j++] = 30; //#3313
                $k[$j++] = 31; //#3313
                $k[$j++] = 30; //#3313
                $k[$j++] = 31; //#3313
                $k[$j++] = 31; //#3313
                $k[$j++] = 30; //#3313
                $k[$j++] = 31; //#3313
                $k[$j++] = 30; //#3313
                $k[$j++] = 31; //#3313
                var _aI = $a(); //#3313
                var _aJ = $k[$j - 1]; //#3314
                var _aN = $cvi($geti(_aJ, 4, 2)); //#3315
                if (($get(_aI, $cvi($geti(_aJ, 2, 2)) - 1) < _aN) || (_aN < 1)) { //#3316
                    $j -= 2; //#3316
                    $k[$j++] = 'bwipp.GS1couponStartDateBadDay#3316'; //#3316
                    $k[$j++] = "Invalid day of month in start date"; //#3316
                    $k[$j++] = false; //#3316
                    return true; //#3316
                } //#3316
                var _aO = $k[--$j]; //#3317
                $_.couponstart = $cvi($geti(_aO, 0, 6)); //#3317
                $k[$j++] = _aO; //#3320
                if (($_.couponexpire != -1) && ($_.couponexpire < $_.couponstart)) { //#3320
                    $j -= 2; //#3319
                    $k[$j++] = 'bwipp.GS1couponExpireDateBeforeStartDate#3319'; //#3319
                    $k[$j++] = "Coupon expires before it starts"; //#3319
                    $k[$j++] = false; //#3319
                    return true; //#3319
                } //#3319
                var _aT = $k[--$j]; //#3321
                $k[$j++] = $geti(_aT, 6, _aT.length - 6); //#3321
            } //#3321
        } //#3321
        var _aV = $k[$j - 1]; //#3326
        if (_aV.length >= 1) { //#3339
            var _aW = $k[$j - 1]; //#3326
            if ($cvi($geti(_aW, 0, 1)) == 5) { //#3339
                var _aY = $k[--$j]; //#3327
                var _aZ = $geti(_aY, 1, _aY.length - 1); //#3327
                $k[$j++] = _aZ; //#3332
                if (_aZ.length < 1) { //#3332
                    $j -= 2; //#3331
                    $k[$j++] = 'bwipp.GS1couponTooShortSerialNumberVLI#3331'; //#3331
                    $k[$j++] = "Coupon too short: Missing Serial Number VLI"; //#3331
                    $k[$j++] = false; //#3331
                    return true; //#3331
                } //#3331
                var _aa = $k[$j - 1]; //#3333
                var _ab = $geti(_aa, 0, 1); //#3333
                $k[$j++] = ($cvi(_ab) + 6) + 1; //#3336
                if ((($cvi(_ab) + 6) + 1) > _aa.length) { //#3336
                    $j -= 3; //#3335
                    $k[$j++] = 'bwipp.GS1couponTooShortSerialNumber#3335'; //#3335
                    $k[$j++] = "Coupon too short: Serial Number truncated"; //#3335
                    $k[$j++] = false; //#3335
                    return true; //#3335
                } //#3335
                var _ac = $k[--$j]; //#3337
                var _ad = $k[--$j]; //#3337
                $k[$j++] = $geti(_ad, _ac, $f(_ad.length - _ac)); //#3337
            } //#3337
        } //#3337
        var _af = $k[$j - 1]; //#3342
        if (_af.length >= 1) { //#3358
            var _ag = $k[$j - 1]; //#3342
            if ($cvi($geti(_ag, 0, 1)) == 6) { //#3358
                var _ai = $k[--$j]; //#3343
                var _aj = $geti(_ai, 1, _ai.length - 1); //#3343
                $k[$j++] = _aj; //#3348
                if (_aj.length < 1) { //#3348
                    $j -= 2; //#3347
                    $k[$j++] = 'bwipp.GS1couponTooShortRetailerGCPGLNVLI#3347'; //#3347
                    $k[$j++] = "Coupon too short: Missing Retailer GCP/GLN VLI"; //#3347
                    $k[$j++] = false; //#3347
                    return true; //#3347
                } //#3347
                var _ak = $k[$j - 1]; //#3349
                var _am = $cvi($geti(_ak, 0, 1)); //#3349
                $k[$j++] = _am; //#3351
                if ((_am < 1) || (_am > 7)) { //#3351
                    $j -= 2; //#3350
                    $k[$j++] = 'bwipp.GS1couponBadRetailerGCPGLNVLI#3350'; //#3350
                    $k[$j++] = "Coupon Retailer GCP/GLN length indicator must be 1-7"; //#3350
                    $k[$j++] = false; //#3350
                    return true; //#3350
                } //#3350
                var _an = $k[--$j]; //#3352
                var _ao = $k[$j - 1]; //#3353
                $k[$j++] = (_an + 6) + 1; //#3355
                if (((_an + 6) + 1) > _ao.length) { //#3355
                    $j -= 3; //#3354
                    $k[$j++] = 'bwipp.GS1couponTooShortRetailerGCPGLN#3354'; //#3354
                    $k[$j++] = "Coupon too short: Retailer GCP/GLN truncated"; //#3354
                    $k[$j++] = false; //#3354
                    return true; //#3354
                } //#3354
                var _ap = $k[--$j]; //#3356
                var _aq = $k[--$j]; //#3356
                $k[$j++] = $geti(_aq, _ap, $f(_aq.length - _ap)); //#3356
            } //#3356
        } //#3356
        var _as = $k[$j - 1]; //#3361
        if (_as.length >= 1) { //#3397
            var _at = $k[$j - 1]; //#3361
            if ($cvi($geti(_at, 0, 1)) == 9) { //#3397
                var _av = $k[--$j]; //#3362
                var _aw = $geti(_av, 1, _av.length - 1); //#3362
                $k[$j++] = _aw; //#3367
                if (_aw.length < 1) { //#3367
                    $j -= 2; //#3366
                    $k[$j++] = 'bwipp.GS1couponTooShortSaveValueCode#3366'; //#3366
                    $k[$j++] = "Coupon too short: Missing Save Value Code"; //#3366
                    $k[$j++] = false; //#3366
                    return true; //#3366
                } //#3366
                var _ax = $k[$j - 1]; //#3368
                var _az = $cvi($geti(_ax, 0, 1)); //#3368
                if ((_az > 6) || ((_az == 3) || (_az == 4))) { //#3370
                    $j -= 2; //#3369
                    $k[$j++] = 'bwipp.GS1couponBadSaveValueCode#3369'; //#3369
                    $k[$j++] = "Coupon Save Value Code must be 0,1,2,5 or 6"; //#3369
                    $k[$j++] = false; //#3369
                    return true; //#3369
                } //#3369
                var _b0 = $k[--$j]; //#3371
                var _b1 = $geti(_b0, 1, _b0.length - 1); //#3371
                $k[$j++] = _b1; //#3376
                if (_b1.length < 1) { //#3376
                    $j -= 2; //#3375
                    $k[$j++] = 'bwipp.GS1couponTooShortSaveValueAppliesToItem#3375'; //#3375
                    $k[$j++] = "Coupon too short: Missing Save Value Applies to Item"; //#3375
                    $k[$j++] = false; //#3375
                    return true; //#3375
                } //#3375
                var _b2 = $k[$j - 1]; //#3377
                if ($cvi($geti(_b2, 0, 1)) > 2) { //#3379
                    $j -= 2; //#3378
                    $k[$j++] = 'bwipp.GS1couponBadSaveValueAppliesToItem#3378'; //#3378
                    $k[$j++] = "Coupon Save Value Applies to Item must be 0-2"; //#3378
                    $k[$j++] = false; //#3378
                    return true; //#3378
                } //#3378
                var _b4 = $k[--$j]; //#3380
                var _b5 = $geti(_b4, 1, _b4.length - 1); //#3380
                $k[$j++] = _b5; //#3385
                if (_b5.length < 1) { //#3385
                    $j -= 2; //#3384
                    $k[$j++] = 'bwipp.GS1couponTooShortStoreCouponFlag#3384'; //#3384
                    $k[$j++] = "Coupon too short: Missing Store Coupon Flag"; //#3384
                    $k[$j++] = false; //#3384
                    return true; //#3384
                } //#3384
                var _b6 = $k[--$j]; //#3386
                var _b7 = $geti(_b6, 1, _b6.length - 1); //#3386
                $k[$j++] = _b7; //#3391
                if (_b7.length < 1) { //#3391
                    $j -= 2; //#3390
                    $k[$j++] = 'bwipp.GS1couponTooShortDontMultiplyFlag#3390'; //#3390
                    $k[$j++] = "Coupon too short: Missing Don't Multiply Flag"; //#3390
                    $k[$j++] = false; //#3390
                    return true; //#3390
                } //#3390
                var _b8 = $k[$j - 1]; //#3392
                if ($cvi($geti(_b8, 0, 1)) > 1) { //#3394
                    $j -= 2; //#3393
                    $k[$j++] = 'bwipp.GS1couponBadDontMultiplyFlag#3393'; //#3393
                    $k[$j++] = "Don't Multiply Flag must be 0 or 1"; //#3393
                    $k[$j++] = false; //#3393
                    return true; //#3393
                } //#3393
                var _bA = $k[--$j]; //#3395
                $k[$j++] = $geti(_bA, 1, _bA.length - 1); //#3395
            } //#3395
        } //#3395
        var _bC = $k[$j - 1]; //#3399
        if (_bC.length != 0) { //#3401
            $j -= 2; //#3400
            $k[$j++] = 'bwipp.GS1couponUnrecognisedOptionalField#3400'; //#3400
            $k[$j++] = "Coupon fields must be 1,2,3,4,5,6 or 9, increasing order"; //#3400
            $k[$j++] = false; //#3400
            return true; //#3400
        } //#3400
        $j--; //#3402
    }; //#3402
    $_.lintcouponposoffer = function() {
        var _bD = $k[$j - 1]; //#3406
        $k[$j++] = true; //#3408
        $forall(_bD, function() { //#3408
            var _bE = $k[--$j]; //#3407
            if ((_bE < 48) || (_bE > 57)) { //#3407
                $k[$j - 1] = false; //#3407
                return true; //#3407
            } //#3407
        }); //#3407
        if (!$k[--$j]) { //#3409
            $j -= 2; //#3409
            $k[$j++] = 'bwipp.GS1couponNotNumeric#3409'; //#3409
            $k[$j++] = "Coupon not numeric"; //#3409
            $k[$j++] = false; //#3409
            return true; //#3409
        } //#3409
        var _bG = $k[$j - 1]; //#3411
        if (_bG.length < 1) { //#3413
            $j -= 2; //#3412
            $k[$j++] = 'bwipp.GS1couponTooShortFormatCode#3412'; //#3412
            $k[$j++] = "Coupon too short: Missing Format Code"; //#3412
            $k[$j++] = false; //#3412
            return true; //#3412
        } //#3412
        var _bH = $k[$j - 1]; //#3414
        var _bI = $geti(_bH, 0, 1); //#3414
        if ($ne(_bI, "0") && $ne(_bI, "1")) { //#3416
            $j -= 2; //#3415
            $k[$j++] = 'bwipp.GS1couponBadFormatCode#3415'; //#3415
            $k[$j++] = "Coupon format must be 0 or 1"; //#3415
            $k[$j++] = false; //#3415
            return true; //#3415
        } //#3415
        var _bJ = $k[--$j]; //#3417
        var _bK = $geti(_bJ, 1, _bJ.length - 1); //#3417
        $k[$j++] = _bK; //#3421
        if (_bK.length < 1) { //#3421
            $j -= 2; //#3420
            $k[$j++] = 'bwipp.GS1couponTooShortFunderVLI#3420'; //#3420
            $k[$j++] = "Coupon too short: Missing Funder VLI"; //#3420
            $k[$j++] = false; //#3420
            return true; //#3420
        } //#3420
        var _bL = $k[$j - 1]; //#3422
        var _bN = $cvi($geti(_bL, 0, 1)); //#3422
        $k[$j++] = _bN; //#3424
        if (_bN > 6) { //#3424
            $j -= 3; //#3423
            $k[$j++] = 'bwipp.GS1couponBadFunderVLI#3423'; //#3423
            $k[$j++] = "Coupon Funder length indicator must be 0-6"; //#3423
            $k[$j++] = false; //#3423
            return true; //#3423
        } //#3423
        var _bO = $k[--$j]; //#3425
        var _bP = $k[$j - 1]; //#3426
        $k[$j++] = (_bO + 6) + 1; //#3428
        if (((_bO + 6) + 1) > _bP.length) { //#3428
            $j -= 3; //#3427
            $k[$j++] = 'bwipp.GS1couponTooShortFunder#3427'; //#3427
            $k[$j++] = "Coupon too short: Truncated Funder ID"; //#3427
            $k[$j++] = false; //#3427
            return true; //#3427
        } //#3427
        var _bQ = $k[--$j]; //#3429
        var _bR = $k[--$j]; //#3429
        var _bS = $geti(_bR, _bQ, $f(_bR.length - _bQ)); //#3429
        $k[$j++] = _bS; //#3433
        if (_bS.length < 6) { //#3433
            $j -= 2; //#3432
            $k[$j++] = 'bwipp.GS1couponTooShortOfferCode#3432'; //#3432
            $k[$j++] = "Coupon too short: Truncated Offer Code"; //#3432
            $k[$j++] = false; //#3432
            return true; //#3432
        } //#3432
        var _bT = $k[--$j]; //#3434
        var _bU = $geti(_bT, 6, _bT.length - 6); //#3434
        $k[$j++] = _bU; //#3438
        if (_bU.length < 1) { //#3438
            $j -= 2; //#3437
            $k[$j++] = 'bwipp.GS1couponTooShortSnVLI#3437'; //#3437
            $k[$j++] = "Coupon too short: Missing SN VLI"; //#3437
            $k[$j++] = false; //#3437
            return true; //#3437
        } //#3437
        var _bV = $k[$j - 1]; //#3439
        var _bW = $geti(_bV, 0, 1); //#3439
        $k[$j++] = ($cvi(_bW) + 6) + 1; //#3443
        if ((($cvi(_bW) + 6) + 1) > _bV.length) { //#3443
            $j -= 3; //#3442
            $k[$j++] = 'bwipp.GS1couponTooShortSn#3442'; //#3442
            $k[$j++] = "Coupon too short: Truncated SN"; //#3442
            $k[$j++] = false; //#3442
            return true; //#3442
        } //#3442
        var _bX = $k[--$j]; //#3444
        var _bY = $k[--$j]; //#3444
        var _bZ = $geti(_bY, _bX, $f(_bY.length - _bX)); //#3444
        $k[$j++] = _bZ; //#3447
        if (_bZ.length != 0) { //#3447
            $j -= 2; //#3446
            $k[$j++] = 'bwipp.GS1couponTooLong#3446'; //#3446
            $k[$j++] = "Coupon too long"; //#3446
            $k[$j++] = false; //#3446
            return true; //#3446
        } //#3446
        $j--; //#3448
    }; //#3448
    $_.lintlatitude = function() {
        var _ba = $k[$j - 1]; //#3452
        if (_ba.length != 10) { //#3454
            $j--; //#3453
            $k[$j++] = 'bwipp.GS1badLatitudeLength#3453'; //#3453
            $k[$j++] = "Invalid length for a latitude"; //#3453
            $k[$j++] = false; //#3453
            return true; //#3453
        } //#3453
        if ($cvi($k[--$j]) > $cvi("1800000000")) { //#3457
            $j--; //#3457
            $k[$j++] = 'bwipp.GS1badLatitude#3457'; //#3457
            $k[$j++] = "Invalid value for latitude"; //#3457
            $k[$j++] = false; //#3457
            return true; //#3457
        } //#3457
    }; //#3457
    $_.lintlongitude = function() {
        var _bc = $k[$j - 1]; //#3461
        if (_bc.length != 10) { //#3463
            $j--; //#3462
            $k[$j++] = 'bwipp.GS1badLongitudeLength#3462'; //#3462
            $k[$j++] = "Invalid length for a longitude"; //#3462
            $k[$j++] = false; //#3462
            return true; //#3462
        } //#3462
        if ($cvi($k[--$j]) > $cvi("3600000000")) { //#3467
            $j--; //#3467
            $k[$j++] = 'bwipp.GS1badLongitude#3467'; //#3467
            $k[$j++] = "Invalid value for longitude"; //#3467
            $k[$j++] = false; //#3467
            return true; //#3467
        } //#3467
    }; //#3467
    $k[$j++] = Infinity; //#3471
    var _be = $a(['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '80', '81', '82', '83', '84', '85', '86', '87', '88', '89', '90', '91', '92', '93', '94', '95', '96', '97', '98', '99']); //#3474
    for (var _bf = 0, _bg = _be.length; _bf < _bg; _bf++) { //#3475
        var _bh = $get(_be, _bf); //#3475
        $k[$j++] = _bh; //#3475
        $k[$j++] = _bh; //#3475
    } //#3475
    $_.aidcmediatype = $d(); //#3475
    $_.lintmediatype = function() {
        var _bm = $get($_.aidcmediatype, $k[--$j]) !== undefined; //#3479
        if (!_bm) { //#3479
            $j--; //#3479
            $k[$j++] = 'bwipp.GS1UnknownMediaType#3479'; //#3479
            $k[$j++] = "Unknown AIDC media type"; //#3479
            $k[$j++] = false; //#3479
            return true; //#3479
        } //#3479
    }; //#3479
    $k[$j++] = Infinity; //#3483
    var _bn = $a(['1A', '1B', '1D', '1F', '1G', '1W', '200', '201', '202', '203', '204', '205', '206', '210', '211', '212', '2C', '3A', '3H', '43', '44', '4A', '4B', '4C', '4D', '4F', '4G', '4H', '5H', '5L', '5M', '6H', '6P', '7A', '7B', '8', '8A', '8B', '8C', '9', 'AA', 'AB', 'AC', 'AD', 'AF', 'AG', 'AH', 'AI', 'AJ', 'AL', 'AM', 'AP', 'APE/AT', 'AV', 'B4', 'BB', 'BC', 'BD', 'BE', 'BF', 'BG', 'BGE', 'BH', 'BI', 'BJ', 'BK', 'BL', 'BM', 'BME', 'BN', 'BO', 'BP', 'BQ', 'BR', 'BRI', 'BS', 'BT', 'BU', 'BV', 'BW', 'BX', 'BY', 'BZ', 'CA', 'CB', 'CBL', 'CC', 'CCE', 'CD', 'CE', 'CF', 'CG', 'CH', 'CI', 'CJ', 'CK', 'CL', 'CM', 'CN', 'CO', 'CP', 'CQ', 'CR', 'CS', 'CT', 'CU', 'CV', 'CW', 'CX', 'CY', 'CZ', 'DA', 'DB', 'DC', 'DG', 'DH', 'DI', 'DJ', 'DK', 'DL', 'DM', 'DN', 'DP', 'DPE', 'DR', 'DS', 'DT', 'DU', 'DV', 'DW', 'DX', 'DY', 'E1', 'E2', 'E3', 'EC', 'ED', 'EE', 'EF', 'EG', 'EH', 'EI', 'EN', 'FB', 'FC', 'FD', 'FE', 'FI', 'FL', 'FO', 'FOB', 'FP', 'FPE', 'FR', 'FT', 'FW', 'FX', 'GB', 'GI', 'GL', 'GR', 'GU', 'GY', 'GZ', 'HA', 'HB', 'HC', 'HG', 'HN', 'HR', 'IA', 'IB', 'IC', 'ID', 'IE', 'IF', 'IG', 'IH', 'IK', 'IL', 'IN', 'IZ', 'JB', 'JC', 'JG', 'JR', 'JT', 'JY', 'KG', 'KI', 'LAB', 'LE', 'LG', 'LT', 'LU', 'LV', 'LZ', 'MA', 'MB', 'MC', 'ME', 'MPE', 'MR', 'MS', 'MT', 'MW', 'MX', 'NA', 'NE', 'NF', 'NG', 'NS', 'NT', 'NU', 'NV', 'OA', 'OB', 'OC', 'OD', 'OE', 'OF', 'OK', 'OPE', 'OT', 'OU', 'P2', 'PA', 'PAE', 'PB', 'PC', 'PD', 'PE', 'PF', 'PG', 'PH', 'PI', 'PJ', 'PK', 'PL', 'PLP', 'PN', 'PO', 'POP', 'PP', 'PPE', 'PR', 'PT', 'PU', 'PUE', 'PV', 'PX', 'PY', 'PZ', 'QA', 'QB', 'QC', 'QD', 'QF', 'QG', 'QH', 'QJ', 'QK', 'QL', 'QM', 'QN', 'QP', 'QQ', 'QR', 'QS', 'RB1', 'RB2', 'RB3', 'RCB', 'RD', 'RG', 'RJ', 'RK', 'RL', 'RO', 'RT', 'RZ', 'S1', 'SA', 'SB', 'SC', 'SD', 'SE', 'SEC', 'SH', 'SI', 'SK', 'SL', 'SM', 'SO', 'SP', 'SS', 'ST', 'STL', 'SU', 'SV', 'SW', 'SX', 'SY', 'SZ', 'T1', 'TB', 'TC', 'TD', 'TE', 'TEV', 'TG', 'THE', 'TI', 'TK', 'TL', 'TN', 'TO', 'TR', 'TRE', 'TS', 'TT', 'TTE', 'TU', 'TV', 'TW', 'TWE', 'TY', 'TZ', 'UC', 'UN', 'UUE', 'VA', 'VG', 'VI', 'VK', 'VL', 'VN', 'VO', 'VP', 'VQ', 'VR', 'VS', 'VY', 'WA', 'WB', 'WC', 'WD', 'WF', 'WG', 'WH', 'WJ', 'WK', 'WL', 'WM', 'WN', 'WP', 'WQ', 'WR', 'WRP', 'WS', 'WT', 'WU', 'WV', 'WW', 'WX', 'WY', 'WZ', 'X11', 'X12', 'X15', 'X16', 'X17', 'X18', 'X19', 'X20', 'X3', 'XA', 'XB', 'XC', 'XD', 'XF', 'XG', 'XH', 'XJ', 'XK', 'YA', 'YB', 'YC', 'YD', 'YF', 'YG', 'YH', 'YJ', 'YK', 'YL', 'YM', 'YN', 'YP', 'YQ', 'YR', 'YS', 'YT', 'YV', 'YW', 'YX', 'YY', 'YZ', 'ZA', 'ZB', 'ZC', 'ZD', 'ZF', 'ZG', 'ZH', 'ZJ', 'ZK', 'ZL', 'ZM', 'ZN', 'ZP', 'ZQ', 'ZR', 'ZS', 'ZT', 'ZU', 'ZV', 'ZW', 'ZX', 'ZY', 'ZZ']); //#3518
    for (var _bo = 0, _bp = _bn.length; _bo < _bp; _bo++) { //#3519
        var _bq = $get(_bn, _bo); //#3519
        $k[$j++] = _bq; //#3519
        $k[$j++] = _bq; //#3519
    } //#3519
    $_.packagetype = $d(); //#3519
    $_.lintpackagetype = function() {
        var _bv = $get($_.packagetype, $k[--$j]) !== undefined; //#3523
        if (!_bv) { //#3523
            $j--; //#3523
            $k[$j++] = 'bwipp.GS1unknownPackageType#3523'; //#3523
            $k[$j++] = "Unknown package type"; //#3523
            $k[$j++] = false; //#3523
            return true; //#3523
        } //#3523
    }; //#3523
    if (!$_.dontlint) { //#3671
        $k[$j++] = true; //#3567
        for (var _bz = 0, _by = $_.vals.length - 1; _bz <= _by; _bz += 1) { //#3567
            $_.ai = $get($_.ais, _bz); //#3531
            $_.val = $get($_.vals, _bz); //#3532
            var _c6 = $get($_.gs1syntax, $_.ai) !== undefined; //#3533
            if (_c6) { //#3565
                var _cA = $get($get($_.gs1syntax, $_.ai), 'parts'); //#3534
                $k[$j++] = _cA; //#3536
                $k[$j++] = 0; //#3536
                $forall(_cA, function() { //#3536
                    var _cB = $k[$j - 1]; //#3536
                    if ($get(_cB, 'opt')) { //#3536
                        $k[$j - 1] = 0; //#3536
                    } else { //#3536
                        var _cE = $get($k[--$j], 'min'); //#3536
                        $k[$j++] = _cE; //#3536
                    } //#3536
                    var _cF = $k[--$j]; //#3536
                    var _cG = $k[--$j]; //#3536
                    $k[$j++] = $f(_cG + _cF); //#3536
                }); //#3536
                if ($k[--$j] > $_.val.length) { //#3538
                    $j--; //#3537
                    $k[$j++] = 'bwipp.GS1valueTooShort#3537'; //#3537
                    $k[$j++] = "Too short"; //#3537
                    $k[$j++] = false; //#3537
                    break; //#3537
                } //#3537
                var _cJ = $k[$j - 1]; //#3539
                $k[$j++] = 0; //#3539
                $forall(_cJ, function() { //#3539
                    var _cL = $get($k[--$j], 'max'); //#3539
                    var _cM = $k[--$j]; //#3539
                    $k[$j++] = $f(_cM + _cL); //#3539
                }); //#3539
                if ($k[--$j] < $_.val.length) { //#3541
                    $j--; //#3540
                    $k[$j++] = 'bwipp.GS1valueTooLong#3540'; //#3540
                    $k[$j++] = "Too long"; //#3540
                    $k[$j++] = false; //#3540
                    break; //#3540
                } //#3540
                $forall($k[--$j], function() { //#3559
                    $_.props = $k[--$j]; //#3543
                    var _cS = $get($_.props, 'max'); //#3544
                    var _cT = $_.val; //#3544
                    var _cU = _cT.length; //#3544
                    if (_cS > _cT.length) { //#3544
                        var _ = _cU; //#3544
                        _cU = _cS; //#3544
                        _cS = _; //#3544
                    } //#3544
                    $_.eval = $geti($_.val, 0, _cS); //#3545
                    var _cZ = $_.eval.length; //#3546
                    $_.val = $geti($_.val, _cZ, $_.val.length - _cZ); //#3546
                    if ($_.eval.length == 0) { //#3557
                        if (!$get($_.props, 'opt')) { //#3550
                            $j--; //#3549
                            $k[$j++] = 'bwipp.GS1valueTooShort#3549'; //#3549
                            $k[$j++] = "Too short"; //#3549
                            $k[$j++] = false; //#3549
                            return true; //#3549
                        } //#3549
                    } else { //#3557
                        if ($_.eval.length < $get($_.props, 'min')) { //#3554
                            $j--; //#3553
                            $k[$j++] = 'bwipp.GS1valueTooShort#3553'; //#3553
                            $k[$j++] = "Too short"; //#3553
                            $k[$j++] = false; //#3553
                            return true; //#3553
                        } //#3553
                        var _ci = new Map([
                            ["N", 'lintnumeric'],
                            ["X", 'lintcset82'],
                            ["Y", 'lintcset39'],
                            ["Z", 'lintcset64']
                        ]); //#3555
                        $k[$j++] = $_.eval; //#3556
                        if ($_[$get(_ci, $get($_.props, 'cset'))]() === true) {
                            return true;
                        } //#3556
                        $forall($get($_.props, 'linters'), function() { //#3557
                            var _cs = $_[$k[--$j]]; //#3557
                            $k[$j++] = $_.eval; //#3557
                            if (_cs() === true) {
                                return true;
                            } //#3557
                        }); //#3557
                    } //#3557
                }); //#3557
                var _ct = $k[$j - 1]; //#3560
                if (!_ct) { //#3560
                    break; //#3560
                } //#3560
                if ($_.val.length != 0) { //#3563
                    $j--; //#3562
                    $k[$j++] = 'bwipp.GS1valueTooLong#3562'; //#3562
                    $k[$j++] = "Too long"; //#3562
                    $k[$j++] = false; //#3562
                    break; //#3562
                } //#3562
            } else { //#3565
                $j--; //#3565
                $k[$j++] = 'bwipp.GS1unknownAI#3565'; //#3565
                $k[$j++] = "Unrecognised AI"; //#3565
                $k[$j++] = false; //#3565
                break; //#3565
            } //#3565
        } //#3565
        if (!$k[--$j]) { //#3575
            var _cw = $k[--$j]; //#3569
            var _cy = $s((_cw.length + $_.ai.length) + 5); //#3569
            $puti(_cy, 0, "AI "); //#3570
            $puti(_cy, 3, $_.ai); //#3571
            $puti(_cy, 3 + $_.ai.length, ": "); //#3572
            $puti(_cy, 5 + $_.ai.length, _cw); //#3573
            $k[$j++] = _cy; //#3574
            bwipp_raiseerror(); //#3574
        } //#3574
        $_.aiexists = function() {
            $_.this = $k[--$j]; //#3579
            $_.patt = $k[--$j]; //#3580
            for (var _d4 = 0; _d4 < 1; _d4++) { //#3601
                var _d7 = $get($_.aivals, $_.patt) !== undefined; //#3582
                if (_d7) { //#3582
                    $k[$j++] = true; //#3582
                    break; //#3582
                } //#3582
                if ($_.patt.length == 4) { //#3599
                    if ($eq($geti($_.patt, 3, 1), "n")) { //#3597
                        var _dD = $eq($geti($_.patt, 2, 1), "n") ? 2 : 3; //#3585
                        $_.pfxlen = _dD; //#3585
                        var _dE = $_.ais; //#3587
                        $k[$j++] = false; //#3595
                        for (var _dF = 0, _dG = _dE.length; _dF < _dG; _dF++) { //#3595
                            var _dH = $get(_dE, _dF); //#3595
                            $k[$j++] = _dH; //#3594
                            if ($ne(_dH, $_.this) && (_dH.length == 4)) { //#3593
                                if ($eq($geti($k[--$j], 0, $_.pfxlen), $geti($_.patt, 0, $_.pfxlen))) { //#3591
                                    $k[$j - 1] = true; //#3590
                                } //#3590
                            } else { //#3593
                                $j--; //#3593
                            } //#3593
                        } //#3593
                        break; //#3596
                    } //#3596
                    $k[$j++] = false; //#3598
                    break; //#3598
                } //#3598
                $k[$j++] = false; //#3600
                break; //#3600
            } //#3600
        }; //#3600
        var _dP = $_.vals; //#3605
        $_.aivals = new Map; //#3605
        for (var _dS = 0, _dR = $_.vals.length - 1; _dS <= _dR; _dS += 1) { //#3620
            $_.ai = $get($_.ais, _dS); //#3607
            $_.val = $get($_.vals, _dS); //#3608
            var _dZ = $get($_.aivals, $_.ai) !== undefined; //#3609
            if (_dZ) { //#3618
                if ($ne($get($_.aivals, $_.ai), $_.val)) { //#3616
                    var _df = $s($_.ai.length + 40); //#3611
                    $puti(_df, 0, "Repeated AIs \("); //#3612
                    $puti(_df, 14, $_.ai); //#3613
                    $puti(_df, 14 + $_.ai.length, "\) must have the same value"); //#3614
                    $k[$j++] = 'bwipp.GS1repeatedDifferingAIs#3615'; //#3615
                    $k[$j++] = _df; //#3615
                    bwipp_raiseerror(); //#3615
                } //#3615
            } else { //#3618
                $put($_.aivals, $_.ai, $_.val); //#3618
            } //#3618
        } //#3618
        for (var _dn = 0, _dm = $_.vals.length - 1; _dn <= _dm; _dn += 1) { //#3669
            $_.ai = $get($_.ais, _dn); //#3624
            var _ds = $get($_.gs1syntax, $_.ai); //#3625
            var _dt = $get(_ds, 'ex') !== undefined; //#3625
            $k[$j++] = _ds; //#3640
            $k[$j++] = 'ex'; //#3640
            if (_dt) { //#3639
                var _du = $k[--$j]; //#3626
                $forall($get($k[--$j], _du), function() { //#3637
                    $_.patt = $k[--$j]; //#3627
                    $k[$j++] = $_.patt; //#3628
                    $k[$j++] = $_.ai; //#3628
                    $_.aiexists(); //#3628
                    if ($k[--$j]) { //#3636
                        var _e3 = $s(($_.ai.length + $_.patt.length) + 36); //#3629
                        $puti(_e3, 0, "AIs \("); //#3630
                        $puti(_e3, 5, $_.ai); //#3631
                        $puti(_e3, 5 + $_.ai.length, "\) and \("); //#3632
                        $puti(_e3, 12 + $_.ai.length, $_.patt); //#3633
                        $puti(_e3, (12 + $_.ai.length) + $_.patt.length, "\) are mutually exclusive"); //#3634
                        $k[$j++] = 'bwipp.GS1exclusiveAIs#3635'; //#3635
                        $k[$j++] = _e3; //#3635
                        bwipp_raiseerror(); //#3635
                    } //#3635
                }); //#3635
            } else { //#3639
                $j -= 2; //#3639
            } //#3639
            if ($_.lintreqs) { //#3668
                var _eD = $get($_.gs1syntax, $_.ai); //#3642
                var _eE = $get(_eD, 'req') !== undefined; //#3642
                $k[$j++] = _eD; //#3667
                $k[$j++] = 'req'; //#3667
                if (_eE) { //#3666
                    var _eF = $k[--$j]; //#3643
                    $forall($get($k[--$j], _eF), function() { //#3664
                        var _eI = $k[--$j]; //#3644
                        $k[$j++] = false; //#3649
                        $forall(_eI, function() { //#3649
                            var _eJ = $k[--$j]; //#3645
                            $k[$j++] = true; //#3647
                            $forall(_eJ, function() { //#3647
                                $k[$j++] = $_.ai; //#3646
                                $_.aiexists(); //#3646
                                if (!$k[--$j]) { //#3646
                                    $k[$j - 1] = false; //#3646
                                } //#3646
                            }); //#3646
                            if ($k[--$j]) { //#3648
                                $k[$j - 1] = true; //#3648
                                return true; //#3648
                            } //#3648
                        }); //#3648
                        if (!$k[--$j]) { //#3663
                            $k[$j++] = Infinity; //#3651
                            $forall($get($get($_.gs1syntax, $_.ai), 'req'), function() { //#3653
                                $forall($k[--$j], function() { //#3653
                                    $forall($k[--$j], function() { //#3653
                                        $k[$j++] = "+"; //#3653
                                    }); //#3653
                                    $k[$j - 1] = " OR "; //#3653
                                }); //#3653
                                $k[$j - 1] = " WITH "; //#3653
                            }); //#3653
                            $j--; //#3653
                            var _eU = $a(); //#3653
                            $k[$j++] = _eU; //#3654
                            $k[$j++] = 0; //#3654
                            for (var _eV = 0, _eW = _eU.length; _eV < _eW; _eV++) { //#3654
                                var _eY = $k[--$j]; //#3654
                                $k[$j++] = $f(_eY + $get(_eU, _eV).length); //#3654
                            } //#3654
                            var _ea = $_.ai.length + 49; //#3655
                            $k[$j++] = _ea; //#3655
                            $k[$j++] = _ea; //#3655
                            var _eb = $k[$j - 3]; //#3655
                            var _ed = $s($f($k[$j - 1] + _eb)); //#3655
                            var _ee = $k[$j - 2]; //#3655
                            $j -= 3; //#3655
                            $k[$j++] = _ed; //#3655
                            $k[$j++] = _ee; //#3655
                            $r(3, -1); //#3655
                            $forall($k[--$j], function() { //#3658
                                var _eg = $k[--$j]; //#3656
                                var _eh = $k[--$j]; //#3656
                                var _ei = $k[$j - 1]; //#3657
                                $puti(_ei, _eh, _eg); //#3657
                                $k[$j++] = $f(_eg.length + _eh); //#3657
                            }); //#3657
                            $j--; //#3658
                            var _ej = $k[--$j]; //#3659
                            $puti(_ej, 0, "One of more requisite AIs for AI \("); //#3659
                            $puti(_ej, 34, $_.ai); //#3660
                            $puti(_ej, 34 + $_.ai.length, "\) are missing: "); //#3661
                            $k[$j++] = 'bwipp.GS1missingAIs#3662'; //#3662
                            $k[$j++] = _ej; //#3662
                            bwipp_raiseerror(); //#3662
                        } //#3662
                    }); //#3662
                } else { //#3666
                    $j -= 2; //#3666
                } //#3666
            } //#3666
        } //#3666
    } //#3666
    $k[$j++] = $_.ais; //#3675
    $k[$j++] = $_.vals; //#3675
    $k[$j++] = $_.fncs; //#3675
    $_ = $__; //#3677
} //bwipp_gs1process
function bwipp_ean5() {
    var $__ = $_; //#4975
    $_ = Object.assign({}, $_); //#4975
    $_.dontdraw = false; //#4978
    $_.includetext = false; //#4979
    $_.textfont = "OCR-B"; //#4980
    $_.textsize = 12; //#4981
    $_.textxoffset = 0; //#4982
    $_.textyoffset = null; //#4983
    $_.height = 0.7; //#4984
    $k[$j++] = $_; //#4986
    bwipp_processoptions(); //#4986
    $_.options = $k[--$j]; //#4986
    $_.barcode = $k[--$j]; //#4987
    if ($_.textyoffset == null) { //#4991
        $_.textyoffset = ($_.height * 72) + 1; //#4990
    } //#4990
    if ($_.barcode.length != 5) { //#4998
        $k[$j++] = 'bwipp.ean5badLength#4997'; //#4997
        $k[$j++] = "EAN-5 add-on must be 5 digits"; //#4997
        bwipp_raiseerror(); //#4997
    } //#4997
    $forall($_.barcode, function() { //#5003
        var _6 = $k[--$j]; //#5000
        if ((_6 < 48) || (_6 > 57)) { //#5002
            $k[$j++] = 'bwipp.ean5badCharacter#5001'; //#5001
            $k[$j++] = "EAN-5 add-on must contain only digits"; //#5001
            bwipp_raiseerror(); //#5001
        } //#5001
    }); //#5001
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "112", "11"]); //#5011
    $_.barchars = "0123456789"; //#5014
    $_.mirrormaps = $a(["11000", "10100", "10010", "10001", "01100", "00110", "00011", "01010", "01001", "00101"]); //#5020
    $_.checksum = 0; //#5023
    for (var _A = 0; _A <= 4; _A += 1) { //#5032
        $_.i = _A; //#5025
        $_.barchar = $get($_.barcode, $_.i) - 48; //#5026
        if (($_.i % 2) == 0) { //#5030
            $_.checksum = $f(($_.barchar * 3) + $_.checksum); //#5028
        } else { //#5030
            $_.checksum = $f(($_.barchar * 9) + $_.checksum); //#5030
        } //#5030
    } //#5030
    $_.checksum = $_.checksum % 10; //#5033
    $_.mirrormap = $get($_.mirrormaps, $_.checksum); //#5034
    $_.sbs = $s(31); //#5036
    $_.txt = $a(5); //#5037
    for (var _P = 0; _P <= 4; _P += 1) { //#5067
        $_.i = _P; //#5040
        if ($_.i == 0) { //#5046
            $puti($_.sbs, 0, $get($_.encs, 10)); //#5044
        } else { //#5046
            $puti($_.sbs, (($_.i - 1) * 6) + 7, $get($_.encs, 11)); //#5046
        } //#5046
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5050
        $j--; //#5051
        $_.indx = $k[--$j].length; //#5052
        $j -= 2; //#5053
        $_.enc = $get($_.encs, $_.indx); //#5054
        if ($get($_.mirrormap, $_.i) == 49) { //#5064
            $_.enclen = $_.enc.length; //#5056
            $_.revenc = $s($_.enclen); //#5057
            for (var _o = 0, _n = $_.enclen - 1; _o <= _n; _o += 1) { //#5062
                $_.j = _o; //#5059
                $_.char = $get($_.enc, $_.j); //#5060
                $put($_.revenc, ($_.enclen - $_.j) - 1, $_.char); //#5061
            } //#5061
            $_.enc = $_.revenc; //#5063
        } //#5063
        $puti($_.sbs, ($_.i * 6) + 3, $_.enc); //#5065
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 1) * 9) + 10) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5066
    } //#5066
    $k[$j++] = Infinity; //#5070
    $k[$j++] = Infinity; //#5072
    var _1B = $_.sbs; //#5072
    for (var _1C = 0, _1D = _1B.length; _1C < _1D; _1C++) { //#5072
        $k[$j++] = $get(_1B, _1C) - 48; //#5072
    } //#5072
    var _1F = $a(); //#5072
    $k[$j++] = Infinity; //#5073
    for (var _1G = 0; _1G < 16; _1G++) { //#5073
        $k[$j++] = $_.height; //#5073
    } //#5073
    var _1I = $a(); //#5073
    $k[$j++] = Infinity; //#5074
    for (var _1J = 0; _1J < 16; _1J++) { //#5074
        $k[$j++] = 0; //#5074
    } //#5074
    var _1K = $a(); //#5074
    $k[$j++] = 'ren'; //#5077
    $k[$j++] = 'renlinear'; //#5077
    $k[$j++] = 'sbs'; //#5077
    $k[$j++] = _1F; //#5077
    $k[$j++] = 'bhs'; //#5077
    $k[$j++] = _1I; //#5077
    $k[$j++] = 'bbs'; //#5077
    $k[$j++] = _1K; //#5077
    if ($_.includetext) { //#5077
        $k[$j++] = 'txt'; //#5076
        $k[$j++] = $_.txt; //#5076
    } //#5076
    $k[$j++] = 'opt'; //#5083
    $k[$j++] = $_.options; //#5083
    $k[$j++] = 'guardrightpos'; //#5083
    $k[$j++] = 5; //#5083
    $k[$j++] = 'guardrightypos'; //#5083
    $k[$j++] = $_.textyoffset + 4; //#5083
    $k[$j++] = 'borderleft'; //#5083
    $k[$j++] = 12; //#5083
    $k[$j++] = 'borderright'; //#5083
    $k[$j++] = 5; //#5083
    $k[$j++] = 'bordertop'; //#5083
    $k[$j++] = 10; //#5083
    var _1P = $d(); //#5083
    $k[$j++] = _1P; //#5086
    if (!$_.dontdraw) { //#5086
        bwipp_renlinear(); //#5086
    } //#5086
    $_ = $__; //#5090
} //bwipp_ean5
function bwipp_ean2() {
    var $__ = $_; //#5120
    $_ = Object.assign({}, $_); //#5120
    $_.dontdraw = false; //#5123
    $_.includetext = false; //#5124
    $_.textfont = "OCR-B"; //#5125
    $_.textsize = 12; //#5126
    $_.textxoffset = 0; //#5127
    $_.textyoffset = null; //#5128
    $_.height = 0.7; //#5129
    $k[$j++] = $_; //#5131
    bwipp_processoptions(); //#5131
    $_.options = $k[--$j]; //#5131
    $_.barcode = $k[--$j]; //#5132
    if ($_.textyoffset == null) { //#5136
        $_.textyoffset = ($_.height * 72) + 1; //#5135
    } //#5135
    if ($_.barcode.length != 2) { //#5143
        $k[$j++] = 'bwipp.ean2badLength#5142'; //#5142
        $k[$j++] = "EAN-2 add-on must be 2 digits"; //#5142
        bwipp_raiseerror(); //#5142
    } //#5142
    $forall($_.barcode, function() { //#5148
        var _6 = $k[--$j]; //#5145
        if ((_6 < 48) || (_6 > 57)) { //#5147
            $k[$j++] = 'bwipp.ean2badCharacter#5146'; //#5146
            $k[$j++] = "EAN-2 add-on must contain only digits"; //#5146
            bwipp_raiseerror(); //#5146
        } //#5146
    }); //#5146
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "112", "11"]); //#5156
    $_.barchars = "0123456789"; //#5159
    $_.mirrormaps = $a(["00", "01", "10", "11"]); //#5162
    $_.mirrormap = $get($_.mirrormaps, $cvi($geti($_.barcode, 0, 2)) % 4); //#5165
    $_.sbs = $s(13); //#5167
    $_.txt = $a(2); //#5168
    for (var _G = 0; _G <= 1; _G += 1) { //#5198
        $_.i = _G; //#5171
        if ($_.i == 0) { //#5177
            $puti($_.sbs, 0, $get($_.encs, 10)); //#5175
        } else { //#5177
            $puti($_.sbs, (($_.i - 1) * 6) + 7, $get($_.encs, 11)); //#5177
        } //#5177
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5181
        $j--; //#5182
        $_.indx = $k[--$j].length; //#5183
        $j -= 2; //#5184
        $_.enc = $get($_.encs, $_.indx); //#5185
        if ($get($_.mirrormap, $_.i) == 49) { //#5195
            $_.enclen = $_.enc.length; //#5187
            $_.revenc = $s($_.enclen); //#5188
            for (var _f = 0, _e = $_.enclen - 1; _f <= _e; _f += 1) { //#5193
                $_.j = _f; //#5190
                $_.char = $get($_.enc, $_.j); //#5191
                $put($_.revenc, ($_.enclen - $_.j) - 1, $_.char); //#5192
            } //#5192
            $_.enc = $_.revenc; //#5194
        } //#5194
        $puti($_.sbs, ($_.i * 6) + 3, $_.enc); //#5196
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 1) * 9) + 10) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5197
    } //#5197
    $k[$j++] = Infinity; //#5201
    $k[$j++] = Infinity; //#5203
    var _12 = $_.sbs; //#5203
    for (var _13 = 0, _14 = _12.length; _13 < _14; _13++) { //#5203
        $k[$j++] = $get(_12, _13) - 48; //#5203
    } //#5203
    var _16 = $a(); //#5203
    $k[$j++] = Infinity; //#5204
    for (var _17 = 0; _17 < 12; _17++) { //#5204
        $k[$j++] = $_.height; //#5204
    } //#5204
    var _19 = $a(); //#5204
    $k[$j++] = Infinity; //#5205
    for (var _1A = 0; _1A < 12; _1A++) { //#5205
        $k[$j++] = 0; //#5205
    } //#5205
    var _1B = $a(); //#5205
    $k[$j++] = 'ren'; //#5208
    $k[$j++] = 'renlinear'; //#5208
    $k[$j++] = 'sbs'; //#5208
    $k[$j++] = _16; //#5208
    $k[$j++] = 'bhs'; //#5208
    $k[$j++] = _19; //#5208
    $k[$j++] = 'bbs'; //#5208
    $k[$j++] = _1B; //#5208
    if ($_.includetext) { //#5208
        $k[$j++] = 'txt'; //#5207
        $k[$j++] = $_.txt; //#5207
    } //#5207
    $k[$j++] = 'opt'; //#5214
    $k[$j++] = $_.options; //#5214
    $k[$j++] = 'guardrightpos'; //#5214
    $k[$j++] = 5; //#5214
    $k[$j++] = 'guardrightypos'; //#5214
    $k[$j++] = $_.textyoffset + 4; //#5214
    $k[$j++] = 'borderleft'; //#5214
    $k[$j++] = 12; //#5214
    $k[$j++] = 'borderright'; //#5214
    $k[$j++] = 5; //#5214
    $k[$j++] = 'bordertop'; //#5214
    $k[$j++] = 10; //#5214
    var _1G = $d(); //#5214
    $k[$j++] = _1G; //#5217
    if (!$_.dontdraw) { //#5217
        bwipp_renlinear(); //#5217
    } //#5217
    $_ = $__; //#5221
} //bwipp_ean2
function bwipp_ean13() {
    var $__ = $_; //#5253
    $_ = Object.assign({}, $_); //#5253
    $_.dontdraw = false; //#5256
    $_.includetext = false; //#5257
    $_.textfont = "OCR-B"; //#5258
    $_.textsize = 12; //#5259
    $_.textxoffset = -11; //#5260
    $_.textyoffset = -4; //#5261
    $_.height = 1; //#5262
    $_.addongap = 12; //#5263
    $_.addontextfont = "unset"; //#5264
    $_.addontextsize = null; //#5265
    $_.addontextxoffset = null; //#5266
    $_.addontextyoffset = null; //#5267
    $k[$j++] = $_; //#5269
    bwipp_processoptions(); //#5269
    $_.options = $k[--$j]; //#5269
    $_.barcode = $k[--$j]; //#5270
    $search($_.barcode, " "); //#5275
    if ($k[--$j]) { //#5281
        $_.barcode = $k[--$j]; //#5276
        $j--; //#5277
        $_.addon = $k[--$j]; //#5278
    } else { //#5281
        $j--; //#5280
        $_.addon = ""; //#5281
    } //#5281
    if (($_.barcode.length != 12) && ($_.barcode.length != 13)) { //#5287
        $k[$j++] = 'bwipp.ean13badLength#5286'; //#5286
        $k[$j++] = "EAN-13 must be 12 or 13 digits"; //#5286
        bwipp_raiseerror(); //#5286
    } //#5286
    $forall($_.barcode, function() { //#5292
        var _9 = $k[--$j]; //#5289
        if ((_9 < 48) || (_9 > 57)) { //#5291
            $k[$j++] = 'bwipp.ean13badCharacter#5290'; //#5290
            $k[$j++] = "EAN-13 must contain only digits"; //#5290
            bwipp_raiseerror(); //#5290
        } //#5290
    }); //#5290
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#5295
        $k[$j++] = 'bwipp.ean13badAddOnLength#5294'; //#5294
        $k[$j++] = "Add-on for EAN-13 must be 2 or 5 digits"; //#5294
        bwipp_raiseerror(); //#5294
    } //#5294
    $_.pad = $s(13); //#5298
    $_.checksum = 0; //#5299
    for (var _E = 0; _E <= 11; _E += 1) { //#5308
        $_.i = _E; //#5301
        $_.barchar = $get($_.barcode, $_.i) - 48; //#5302
        if (($_.i % 2) == 0) { //#5306
            $_.checksum = $f($_.barchar + $_.checksum); //#5304
        } else { //#5306
            $_.checksum = $f(($_.barchar * 3) + $_.checksum); //#5306
        } //#5306
    } //#5306
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#5309
    if ($_.barcode.length == 13) { //#5314
        if ($get($_.barcode, 12) != ($_.checksum + 48)) { //#5313
            $k[$j++] = 'bwipp.ean13badCheckDigit#5312'; //#5312
            $k[$j++] = "Incorrect EAN-13 check digit provided"; //#5312
            bwipp_raiseerror(); //#5312
        } //#5312
    } //#5312
    $puti($_.pad, 0, $_.barcode); //#5315
    $put($_.pad, 12, $_.checksum + 48); //#5316
    $_.barcode = $_.pad; //#5317
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#5325
    $_.barchars = "0123456789"; //#5328
    $_.mirrormaps = $a(["000000", "001011", "001101", "001110", "010011", "011001", "011100", "010101", "010110", "011010"]); //#5334
    $_.sbs = $s(59); //#5337
    $_.txt = $a(13); //#5338
    $puti($_.sbs, 0, $get($_.encs, 10)); //#5341
    $_.mirrormap = $get($_.mirrormaps, $get($_.barcode, 0) - 48); //#5344
    $put($_.txt, 0, $a([$geti($_.barcode, 0, 1), $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5345
    for (var _r = 1; _r <= 6; _r += 1) { //#5368
        $_.i = _r; //#5350
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5351
        $j--; //#5352
        $_.indx = $k[--$j].length; //#5353
        $j -= 2; //#5354
        $_.enc = $get($_.encs, $_.indx); //#5355
        if ($get($_.mirrormap, $_.i - 1) == 49) { //#5365
            $_.enclen = $_.enc.length; //#5357
            $_.revenc = $s($_.enclen); //#5358
            for (var _18 = 0, _17 = $_.enclen - 1; _18 <= _17; _18 += 1) { //#5363
                $_.j = _18; //#5360
                $_.char = $get($_.enc, $_.j); //#5361
                $put($_.revenc, ($_.enclen - $_.j) - 1, $_.char); //#5362
            } //#5362
            $_.enc = $_.revenc; //#5364
        } //#5364
        $puti($_.sbs, (($_.i - 1) * 4) + 3, $_.enc); //#5366
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 1) * 7) + 15) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5367
    } //#5367
    $puti($_.sbs, ((7 - 1) * 4) + 3, $get($_.encs, 11)); //#5371
    for (var _1Y = 7; _1Y <= 12; _1Y += 1) { //#5384
        $_.i = _1Y; //#5376
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5377
        $j--; //#5378
        $_.indx = $k[--$j].length; //#5379
        $j -= 2; //#5380
        $_.enc = $get($_.encs, $_.indx); //#5381
        $puti($_.sbs, (($_.i - 1) * 4) + 8, $_.enc); //#5382
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 1) * 7) + 19) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5383
    } //#5383
    $puti($_.sbs, 56, $get($_.encs, 12)); //#5387
    $k[$j++] = Infinity; //#5389
    var _1y = $_.sbs; //#5389
    for (var _1z = 0, _20 = _1y.length; _1z < _20; _1z++) { //#5389
        $k[$j++] = $get(_1y, _1z) - 48; //#5389
    } //#5389
    $_.sbs = $a(); //#5389
    if ($_.includetext) { //#5396
        $k[$j++] = Infinity; //#5391
        $k[$j++] = $_.height; //#5391
        $k[$j++] = $_.height; //#5391
        for (var _26 = 0; _26 < 12; _26++) { //#5391
            $k[$j++] = $f($_.height - 0.075); //#5391
        } //#5391
        $k[$j++] = $_.height; //#5391
        $k[$j++] = $_.height; //#5391
        for (var _2A = 0; _2A < 12; _2A++) { //#5391
            $k[$j++] = $f($_.height - 0.075); //#5391
        } //#5391
        $k[$j++] = $_.height; //#5391
        $k[$j++] = $_.height; //#5391
        $_.bhs = $a(); //#5391
        $k[$j++] = Infinity; //#5392
        $k[$j++] = 0; //#5392
        $k[$j++] = 0; //#5392
        for (var _2F = 0; _2F < 12; _2F++) { //#5392
            $k[$j++] = 0.075; //#5392
        } //#5392
        $k[$j++] = 0; //#5392
        $k[$j++] = 0; //#5392
        for (var _2G = 0; _2G < 12; _2G++) { //#5392
            $k[$j++] = 0.075; //#5392
        } //#5392
        $k[$j++] = 0; //#5392
        $k[$j++] = 0; //#5392
        $_.bbs = $a(); //#5392
    } else { //#5396
        $k[$j++] = Infinity; //#5394
        for (var _2I = 0; _2I < 30; _2I++) { //#5394
            $k[$j++] = $_.height; //#5394
        } //#5394
        $_.bhs = $a(); //#5394
        $k[$j++] = Infinity; //#5395
        for (var _2L = 0; _2L < 30; _2L++) { //#5395
            $k[$j++] = 0; //#5395
        } //#5395
        $_.bbs = $a(); //#5395
        $_.txt = $a([]); //#5396
    } //#5396
    $_.guardrightypos = 0; //#5398
    if ($_.addon.length != 0) { //#5419
        $k[$j++] = Infinity; //#5402
        $k[$j++] = 'dontdraw'; //#5405
        $k[$j++] = true; //#5405
        $k[$j++] = 'includetext'; //#5405
        $k[$j++] = true; //#5405
        $k[$j++] = 'height'; //#5405
        $k[$j++] = $_.height; //#5405
        if ($_.includetext) { //#5405
            var _2R = $k[--$j]; //#5405
            $k[$j++] = $f(_2R - 0.15); //#5405
        } //#5405
        $k[$j++] = 'textxoffset'; //#5406
        if ($_.addontextxoffset != null) { //#5406
            $k[$j++] = $_.addontextxoffset; //#5406
        } else { //#5406
            $k[$j++] = 95 + $_.addongap; //#5406
        } //#5406
        if ($_.addontextyoffset != null) { //#5407
            $k[$j++] = 'textyoffset'; //#5407
            $k[$j++] = $_.addontextyoffset; //#5407
        } //#5407
        $k[$j++] = 'textsize'; //#5408
        if ($_.addontextsize != null) { //#5408
            $k[$j++] = $_.addontextsize; //#5408
        } else { //#5408
            $k[$j++] = $_.textsize; //#5408
        } //#5408
        var _2b = $ne($_.addontextfont, "unset") ? $_.addontextfont : $_.textfont; //#5409
        $k[$j++] = 'textfont'; //#5409
        $k[$j++] = _2b; //#5409
        $_.addopts = $d(); //#5410
        if ($_.addon.length == 2) { //#5411
            $k[$j++] = $_.addon; //#5411
            $k[$j++] = $_.addopts; //#5411
            bwipp_ean2(); //#5411
        } //#5411
        if ($_.addon.length == 5) { //#5412
            $k[$j++] = $_.addon; //#5412
            $k[$j++] = $_.addopts; //#5412
            bwipp_ean5(); //#5412
        } //#5412
        $_.addcode = $k[--$j]; //#5413
        $k[$j++] = Infinity; //#5414
        $aload($_.sbs); //#5414
        $k[$j++] = $_.addongap; //#5414
        $aload($get($_.addcode, "sbs")); //#5414
        $_.sbs = $a(); //#5414
        $k[$j++] = Infinity; //#5415
        $aload($_.bhs); //#5415
        $aload($get($_.addcode, "bhs")); //#5415
        $_.bhs = $a(); //#5415
        $k[$j++] = Infinity; //#5416
        $aload($_.bbs); //#5416
        $aload($get($_.addcode, "bbs")); //#5416
        $_.bbs = $a(); //#5416
        $k[$j++] = Infinity; //#5417
        $aload($_.txt); //#5417
        $aload($get($_.addcode, "txt")); //#5417
        $_.txt = $a(); //#5417
        $_.guardrightypos = ($_.height * 72) - 6; //#5418
    } //#5418
    $k[$j++] = Infinity; //#5422
    $k[$j++] = 'ren'; //#5429
    $k[$j++] = 'renlinear'; //#5429
    $k[$j++] = 'sbs'; //#5429
    $k[$j++] = $_.sbs; //#5429
    $k[$j++] = 'bhs'; //#5429
    $k[$j++] = $_.bhs; //#5429
    $k[$j++] = 'bbs'; //#5429
    $k[$j++] = $_.bbs; //#5429
    $k[$j++] = 'txt'; //#5429
    $k[$j++] = $_.txt; //#5429
    $k[$j++] = 'opt'; //#5429
    $k[$j++] = $_.options; //#5429
    $k[$j++] = 'guardrightpos'; //#5429
    if ($_.addon.length == 0) { //#5429
        $k[$j++] = 7; //#5429
    } else { //#5429
        $k[$j++] = 5; //#5429
    } //#5429
    $k[$j++] = 'guardrightypos'; //#5432
    $k[$j++] = $_.guardrightypos; //#5432
    $k[$j++] = 'borderleft'; //#5432
    $k[$j++] = 11; //#5432
    $k[$j++] = 'borderright'; //#5432
    if ($_.addon.length == 0) { //#5432
        $k[$j++] = 7; //#5432
    } else { //#5432
        $k[$j++] = 5; //#5432
    } //#5432
    $k[$j++] = 'bordertop'; //#5434
    $k[$j++] = 0; //#5434
    $k[$j++] = 'borderbottom'; //#5434
    $k[$j++] = 5; //#5434
    var _3A = $d(); //#5434
    $k[$j++] = _3A; //#5437
    if (!$_.dontdraw) { //#5437
        bwipp_renlinear(); //#5437
    } //#5437
    $_ = $__; //#5441
} //bwipp_ean13
function bwipp_ean8() {
    var $__ = $_; //#5473
    $_ = Object.assign({}, $_); //#5473
    $_.dontdraw = false; //#5476
    $_.includetext = false; //#5477
    $_.textfont = "OCR-B"; //#5478
    $_.textsize = 12; //#5479
    $_.textxoffset = 4; //#5480
    $_.textyoffset = -4; //#5481
    $_.height = 1; //#5482
    $_.permitaddon = false; //#5483
    $_.addongap = 12; //#5484
    $_.addontextfont = "unset"; //#5485
    $_.addontextsize = null; //#5486
    $_.addontextxoffset = null; //#5487
    $_.addontextyoffset = null; //#5488
    $k[$j++] = $_; //#5490
    bwipp_processoptions(); //#5490
    $_.options = $k[--$j]; //#5490
    $_.barcode = $k[--$j]; //#5491
    $search($_.barcode, " "); //#5496
    if ($k[--$j]) { //#5502
        $_.barcode = $k[--$j]; //#5497
        $j--; //#5498
        $_.addon = $k[--$j]; //#5499
    } else { //#5502
        $j--; //#5501
        $_.addon = ""; //#5502
    } //#5502
    if (($_.barcode.length != 7) && ($_.barcode.length != 8)) { //#5508
        $k[$j++] = 'bwipp.ean8badLength#5507'; //#5507
        $k[$j++] = "EAN-8 must be 7 or 8 digits"; //#5507
        bwipp_raiseerror(); //#5507
    } //#5507
    $forall($_.barcode, function() { //#5513
        var _9 = $k[--$j]; //#5510
        if ((_9 < 48) || (_9 > 57)) { //#5512
            $k[$j++] = 'bwipp.ean8badCharacter#5511'; //#5511
            $k[$j++] = "EAN-8 must contain only digits"; //#5511
            bwipp_raiseerror(); //#5511
        } //#5511
    }); //#5511
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#5516
        $k[$j++] = 'bwipp.ean8badAddOnLength#5515'; //#5515
        $k[$j++] = "Add-on for EAN-8 must be 2 or 5 digits"; //#5515
        bwipp_raiseerror(); //#5515
    } //#5515
    if ((!$_.permitaddon) && ($_.addon.length != 0)) { //#5519
        $k[$j++] = 'bwipp.ean8badAddOnLength#5518'; //#5518
        $k[$j++] = "Non-standard use of add-on for EAN-8 requires \"permitaddon\" option"; //#5518
        bwipp_raiseerror(); //#5518
    } //#5518
    $_.pad = $s(8); //#5522
    $_.checksum = 0; //#5523
    for (var _G = 0; _G <= 6; _G += 1) { //#5532
        $_.i = _G; //#5525
        $_.barchar = $get($_.barcode, $_.i) - 48; //#5526
        if (($_.i % 2) != 0) { //#5530
            $_.checksum = $f($_.barchar + $_.checksum); //#5528
        } else { //#5530
            $_.checksum = $f(($_.barchar * 3) + $_.checksum); //#5530
        } //#5530
    } //#5530
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#5533
    if ($_.barcode.length == 8) { //#5538
        if ($get($_.barcode, 7) != ($_.checksum + 48)) { //#5537
            $k[$j++] = 'bwipp.ean8badCheckDigit#5536'; //#5536
            $k[$j++] = "Incorrect EAN-8 check digit provided"; //#5536
            bwipp_raiseerror(); //#5536
        } //#5536
    } //#5536
    $puti($_.pad, 0, $_.barcode); //#5539
    $put($_.pad, 7, $_.checksum + 48); //#5540
    $_.barcode = $_.pad; //#5541
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#5549
    $_.barchars = "0123456789"; //#5552
    $_.sbs = $s(43); //#5555
    $_.txt = $a(8); //#5556
    $puti($_.sbs, 0, $get($_.encs, 10)); //#5559
    for (var _g = 0; _g <= 3; _g += 1) { //#5572
        $_.i = _g; //#5564
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5565
        $j--; //#5566
        $_.indx = $k[--$j].length; //#5567
        $j -= 2; //#5568
        $_.enc = $get($_.encs, $_.indx); //#5569
        $puti($_.sbs, ($_.i * 4) + 3, $_.enc); //#5570
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 7) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5571
    } //#5571
    $puti($_.sbs, (4 * 4) + 3, $get($_.encs, 11)); //#5575
    for (var _16 = 4; _16 <= 7; _16 += 1) { //#5588
        $_.i = _16; //#5580
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5581
        $j--; //#5582
        $_.indx = $k[--$j].length; //#5583
        $j -= 2; //#5584
        $_.enc = $get($_.encs, $_.indx); //#5585
        $puti($_.sbs, ($_.i * 4) + 8, $_.enc); //#5586
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 7) + $_.textxoffset) + 4, $_.textyoffset, $_.textfont, $_.textsize])); //#5587
    } //#5587
    $puti($_.sbs, 40, $get($_.encs, 12)); //#5591
    $k[$j++] = Infinity; //#5593
    var _1W = $_.sbs; //#5593
    for (var _1X = 0, _1Y = _1W.length; _1X < _1Y; _1X++) { //#5593
        $k[$j++] = $get(_1W, _1X) - 48; //#5593
    } //#5593
    $_.sbs = $a(); //#5593
    if ($_.includetext) { //#5600
        $k[$j++] = Infinity; //#5595
        $k[$j++] = $_.height; //#5595
        $k[$j++] = $_.height; //#5595
        for (var _1e = 0; _1e < 8; _1e++) { //#5595
            $k[$j++] = $f($_.height - 0.075); //#5595
        } //#5595
        $k[$j++] = $_.height; //#5595
        $k[$j++] = $_.height; //#5595
        for (var _1i = 0; _1i < 8; _1i++) { //#5595
            $k[$j++] = $f($_.height - 0.075); //#5595
        } //#5595
        $k[$j++] = $_.height; //#5595
        $k[$j++] = $_.height; //#5595
        $_.bhs = $a(); //#5595
        $k[$j++] = Infinity; //#5596
        $k[$j++] = 0; //#5596
        $k[$j++] = 0; //#5596
        for (var _1n = 0; _1n < 8; _1n++) { //#5596
            $k[$j++] = 0.075; //#5596
        } //#5596
        $k[$j++] = 0; //#5596
        $k[$j++] = 0; //#5596
        for (var _1o = 0; _1o < 8; _1o++) { //#5596
            $k[$j++] = 0.075; //#5596
        } //#5596
        $k[$j++] = 0; //#5596
        $k[$j++] = 0; //#5596
        $_.bbs = $a(); //#5596
    } else { //#5600
        $k[$j++] = Infinity; //#5598
        for (var _1q = 0; _1q < 22; _1q++) { //#5598
            $k[$j++] = $_.height; //#5598
        } //#5598
        $_.bhs = $a(); //#5598
        $k[$j++] = Infinity; //#5599
        for (var _1t = 0; _1t < 22; _1t++) { //#5599
            $k[$j++] = 0; //#5599
        } //#5599
        $_.bbs = $a(); //#5599
        $_.txt = $a([]); //#5600
    } //#5600
    $_.guardrightypos = 0; //#5602
    if ($_.addon.length != 0) { //#5623
        $k[$j++] = Infinity; //#5606
        $k[$j++] = 'dontdraw'; //#5609
        $k[$j++] = true; //#5609
        $k[$j++] = 'includetext'; //#5609
        $k[$j++] = true; //#5609
        $k[$j++] = 'height'; //#5609
        $k[$j++] = $_.height; //#5609
        if ($_.includetext) { //#5609
            var _1z = $k[--$j]; //#5609
            $k[$j++] = $f(_1z - 0.15); //#5609
        } //#5609
        $k[$j++] = 'textxoffset'; //#5610
        if ($_.addontextxoffset != null) { //#5610
            $k[$j++] = $_.addontextxoffset; //#5610
        } else { //#5610
            $k[$j++] = 67 + $_.addongap; //#5610
        } //#5610
        if ($_.addontextyoffset != null) { //#5611
            $k[$j++] = 'textyoffset'; //#5611
            $k[$j++] = $_.addontextyoffset; //#5611
        } //#5611
        $k[$j++] = 'textsize'; //#5612
        if ($_.addontextsize != null) { //#5612
            $k[$j++] = $_.addontextsize; //#5612
        } else { //#5612
            $k[$j++] = $_.textsize; //#5612
        } //#5612
        var _29 = $ne($_.addontextfont, "unset") ? $_.addontextfont : $_.textfont; //#5613
        $k[$j++] = 'textfont'; //#5613
        $k[$j++] = _29; //#5613
        $_.addopts = $d(); //#5614
        if ($_.addon.length == 2) { //#5615
            $k[$j++] = $_.addon; //#5615
            $k[$j++] = $_.addopts; //#5615
            bwipp_ean2(); //#5615
        } //#5615
        if ($_.addon.length == 5) { //#5616
            $k[$j++] = $_.addon; //#5616
            $k[$j++] = $_.addopts; //#5616
            bwipp_ean5(); //#5616
        } //#5616
        $_.addcode = $k[--$j]; //#5617
        $k[$j++] = Infinity; //#5618
        $aload($_.sbs); //#5618
        $k[$j++] = $_.addongap; //#5618
        $aload($get($_.addcode, "sbs")); //#5618
        $_.sbs = $a(); //#5618
        $k[$j++] = Infinity; //#5619
        $aload($_.bhs); //#5619
        $aload($get($_.addcode, "bhs")); //#5619
        $_.bhs = $a(); //#5619
        $k[$j++] = Infinity; //#5620
        $aload($_.bbs); //#5620
        $aload($get($_.addcode, "bbs")); //#5620
        $_.bbs = $a(); //#5620
        $k[$j++] = Infinity; //#5621
        $aload($_.txt); //#5621
        $aload($get($_.addcode, "txt")); //#5621
        $_.txt = $a(); //#5621
        $_.guardrightypos = ($_.height * 72) - 6; //#5622
    } //#5622
    $k[$j++] = Infinity; //#5626
    $k[$j++] = 'ren'; //#5634
    $k[$j++] = 'renlinear'; //#5634
    $k[$j++] = 'sbs'; //#5634
    $k[$j++] = $_.sbs; //#5634
    $k[$j++] = 'bhs'; //#5634
    $k[$j++] = $_.bhs; //#5634
    $k[$j++] = 'bbs'; //#5634
    $k[$j++] = $_.bbs; //#5634
    $k[$j++] = 'txt'; //#5634
    $k[$j++] = $_.txt; //#5634
    $k[$j++] = 'opt'; //#5634
    $k[$j++] = $_.options; //#5634
    $k[$j++] = 'guardleftpos'; //#5634
    $k[$j++] = 7; //#5634
    $k[$j++] = 'guardrightpos'; //#5634
    if ($_.addon.length == 0) { //#5634
        $k[$j++] = 7; //#5634
    } else { //#5634
        $k[$j++] = 5; //#5634
    } //#5634
    $k[$j++] = 'guardrightypos'; //#5637
    $k[$j++] = $_.guardrightypos; //#5637
    $k[$j++] = 'borderleft'; //#5637
    $k[$j++] = 7; //#5637
    $k[$j++] = 'borderright'; //#5637
    if ($_.addon.length == 0) { //#5637
        $k[$j++] = 7; //#5637
    } else { //#5637
        $k[$j++] = 5; //#5637
    } //#5637
    $k[$j++] = 'bordertop'; //#5639
    $k[$j++] = 0; //#5639
    $k[$j++] = 'borderbottom'; //#5639
    $k[$j++] = 5; //#5639
    var _2i = $d(); //#5639
    $k[$j++] = _2i; //#5642
    if (!$_.dontdraw) { //#5642
        bwipp_renlinear(); //#5642
    } //#5642
    $_ = $__; //#5646
} //bwipp_ean8
function bwipp_upca() {
    var $__ = $_; //#5678
    $_ = Object.assign({}, $_); //#5678
    $_.dontdraw = false; //#5681
    $_.includetext = false; //#5682
    $_.textfont = "OCR-B"; //#5683
    $_.textsize = 12; //#5684
    $_.textxoffset = -9; //#5685
    $_.textyoffset = -4; //#5686
    $_.height = 1; //#5687
    $_.addongap = 12; //#5688
    $_.addontextfont = "unset"; //#5689
    $_.addontextsize = null; //#5690
    $_.addontextxoffset = null; //#5691
    $_.addontextyoffset = null; //#5692
    $k[$j++] = $_; //#5694
    bwipp_processoptions(); //#5694
    $_.options = $k[--$j]; //#5694
    $_.barcode = $k[--$j]; //#5695
    $search($_.barcode, " "); //#5700
    if ($k[--$j]) { //#5706
        $_.barcode = $k[--$j]; //#5701
        $j--; //#5702
        $_.addon = $k[--$j]; //#5703
    } else { //#5706
        $j--; //#5705
        $_.addon = ""; //#5706
    } //#5706
    if (($_.barcode.length == 7) || ($_.barcode.length == 8)) { //#5752
        $forall($_.barcode, function() { //#5716
            var _9 = $k[--$j]; //#5713
            if ((_9 < 48) || (_9 > 57)) { //#5715
                $k[$j++] = 'bwipp.upcAupcEbadCharacter#5714'; //#5714
                $k[$j++] = "UPC-E must contain only digits"; //#5714
                bwipp_raiseerror(); //#5714
            } //#5714
        }); //#5714
        var _B = $get($_.barcode, 0); //#5717
        if ((_B != 48) && (_B != 49)) { //#5719
            $k[$j++] = 'bwipp.upcAupcEbadNumberSystem#5718'; //#5718
            $k[$j++] = "UPC-E must have number system 0 or 1"; //#5718
            bwipp_raiseerror(); //#5718
        } //#5718
        for (var _C = 0; _C < 1; _C++) { //#5747
            var _E = ($_.barcode.length == 8) ? 12 : 11; //#5721
            $_.upcacode = $s(_E); //#5721
            if (($get($_.barcode, 6) - 48) <= 2) { //#5728
                $puti($_.upcacode, 0, $geti($_.barcode, 0, 3)); //#5723
                $puti($_.upcacode, 3, $geti($_.barcode, 6, 1)); //#5724
                $puti($_.upcacode, 4, "0000"); //#5725
                $puti($_.upcacode, 8, $geti($_.barcode, 3, 3)); //#5726
                break; //#5727
            } //#5727
            if (($get($_.barcode, 6) - 48) == 3) { //#5734
                $puti($_.upcacode, 0, $geti($_.barcode, 0, 4)); //#5730
                $puti($_.upcacode, 4, "00000"); //#5731
                $puti($_.upcacode, 9, $geti($_.barcode, 4, 2)); //#5732
                break; //#5733
            } //#5733
            if (($get($_.barcode, 6) - 48) == 4) { //#5740
                $puti($_.upcacode, 0, $geti($_.barcode, 0, 5)); //#5736
                $puti($_.upcacode, 5, "00000"); //#5737
                $puti($_.upcacode, 10, $geti($_.barcode, 5, 1)); //#5738
                break; //#5739
            } //#5739
            if (($get($_.barcode, 6) - 48) >= 5) { //#5746
                $puti($_.upcacode, 0, $geti($_.barcode, 0, 6)); //#5742
                $puti($_.upcacode, 6, "0000"); //#5743
                $puti($_.upcacode, 10, $geti($_.barcode, 6, 1)); //#5744
                break; //#5745
            } //#5745
        } //#5745
        if ($_.barcode.length == 8) { //#5750
            $puti($_.upcacode, 11, $geti($_.barcode, 7, 1)); //#5749
        } //#5749
        $_.barcode = $_.upcacode; //#5751
    } //#5751
    if (($_.barcode.length != 11) && ($_.barcode.length != 12)) { //#5757
        $k[$j++] = 'bwipp.upcAbadLength#5756'; //#5756
        $k[$j++] = "UPC-A must be 11 or 12 digits"; //#5756
        bwipp_raiseerror(); //#5756
    } //#5756
    var _10 = $_.barcode; //#5758
    for (var _11 = 0, _12 = _10.length; _11 < _12; _11++) { //#5762
        var _13 = $get(_10, _11); //#5762
        if ((_13 < 48) || (_13 > 57)) { //#5761
            $k[$j++] = 'bwipp.upcAbadCharacter#5760'; //#5760
            $k[$j++] = "UPC-A must contain only digits"; //#5760
            bwipp_raiseerror(); //#5760
        } //#5760
    } //#5760
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#5765
        $k[$j++] = 'bwipp.upcAbadAddOnLength#5764'; //#5764
        $k[$j++] = "Add-on for UPC-A must be 2 or 5 digits"; //#5764
        bwipp_raiseerror(); //#5764
    } //#5764
    $_.pad = $s(12); //#5768
    $_.checksum = 0; //#5769
    for (var _18 = 0; _18 <= 10; _18 += 1) { //#5778
        $_.i = _18; //#5771
        $_.barchar = $get($_.barcode, $_.i) - 48; //#5772
        if (($_.i % 2) != 0) { //#5776
            $_.checksum = $_.checksum + $_.barchar; //#5774
        } else { //#5776
            $_.checksum = $_.checksum + ($_.barchar * 3); //#5776
        } //#5776
    } //#5776
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#5779
    if ($_.barcode.length == 12) { //#5784
        if ($get($_.barcode, 11) != ($_.checksum + 48)) { //#5783
            $k[$j++] = 'bwipp.upcAbadCheckDigit#5782'; //#5782
            $k[$j++] = "Incorrect UPC check digit provided"; //#5782
            bwipp_raiseerror(); //#5782
        } //#5782
    } //#5782
    $puti($_.pad, 0, $_.barcode); //#5785
    $put($_.pad, 11, $_.checksum + 48); //#5786
    $_.barcode = $_.pad; //#5787
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "11111", "111"]); //#5795
    $_.barchars = "0123456789"; //#5798
    $_.sbs = $s(59); //#5801
    $_.txt = $a(12); //#5802
    $puti($_.sbs, 0, $get($_.encs, 10)); //#5805
    for (var _1Y = 0; _1Y <= 5; _1Y += 1) { //#5822
        $_.i = _1Y; //#5810
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5811
        $j--; //#5812
        $_.indx = $k[--$j].length; //#5813
        $j -= 2; //#5814
        $_.enc = $get($_.encs, $_.indx); //#5815
        $puti($_.sbs, ($_.i * 4) + 3, $_.enc); //#5816
        if ($_.i == 0) { //#5820
            $put($_.txt, 0, $a([$geti($_.barcode, 0, 1), $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize - 2])); //#5818
        } else { //#5820
            $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 7) + 13) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#5820
        } //#5820
    } //#5820
    $puti($_.sbs, (6 * 4) + 3, $get($_.encs, 11)); //#5825
    for (var _27 = 6; _27 <= 11; _27 += 1) { //#5842
        $_.i = _27; //#5830
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#5831
        $j--; //#5832
        $_.indx = $k[--$j].length; //#5833
        $j -= 2; //#5834
        $_.enc = $get($_.encs, $_.indx); //#5835
        $puti($_.sbs, ($_.i * 4) + 8, $_.enc); //#5836
        if ($_.i == 11) { //#5840
            $put($_.txt, 11, $a([$geti($_.barcode, 11, 1), $_.textxoffset + 107, $_.textyoffset, $_.textfont, $_.textsize - 2])); //#5838
        } else { //#5840
            $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 7) + $_.textxoffset) + 17, $_.textyoffset, $_.textfont, $_.textsize])); //#5840
        } //#5840
    } //#5840
    $puti($_.sbs, 56, $get($_.encs, 12)); //#5845
    $k[$j++] = Infinity; //#5847
    var _2g = $_.sbs; //#5847
    for (var _2h = 0, _2i = _2g.length; _2h < _2i; _2h++) { //#5847
        $k[$j++] = $get(_2g, _2h) - 48; //#5847
    } //#5847
    $_.sbs = $a(); //#5847
    if ($_.includetext) { //#5854
        $k[$j++] = Infinity; //#5849
        for (var _2m = 0; _2m < 4; _2m++) { //#5849
            $k[$j++] = $_.height; //#5849
        } //#5849
        for (var _2o = 0; _2o < 10; _2o++) { //#5849
            $k[$j++] = $f($_.height - 0.075); //#5849
        } //#5849
        $k[$j++] = $_.height; //#5849
        $k[$j++] = $_.height; //#5849
        for (var _2s = 0; _2s < 10; _2s++) { //#5849
            $k[$j++] = $f($_.height - 0.075); //#5849
        } //#5849
        for (var _2u = 0; _2u < 4; _2u++) { //#5849
            $k[$j++] = $_.height; //#5849
        } //#5849
        $_.bhs = $a(); //#5849
        $k[$j++] = Infinity; //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        for (var _2x = 0; _2x < 10; _2x++) { //#5850
            $k[$j++] = 0.075; //#5850
        } //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        for (var _2y = 0; _2y < 10; _2y++) { //#5850
            $k[$j++] = 0.075; //#5850
        } //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        $k[$j++] = 0; //#5850
        $_.bbs = $a(); //#5850
    } else { //#5854
        $k[$j++] = Infinity; //#5852
        for (var _30 = 0; _30 < 30; _30++) { //#5852
            $k[$j++] = $_.height; //#5852
        } //#5852
        $_.bhs = $a(); //#5852
        $k[$j++] = Infinity; //#5853
        for (var _33 = 0; _33 < 30; _33++) { //#5853
            $k[$j++] = 0; //#5853
        } //#5853
        $_.bbs = $a(); //#5853
        $_.txt = $a([]); //#5854
    } //#5854
    $_.guardrightypos = 0; //#5856
    if ($_.addon.length != 0) { //#5877
        $k[$j++] = Infinity; //#5860
        $k[$j++] = 'dontdraw'; //#5863
        $k[$j++] = true; //#5863
        $k[$j++] = 'includetext'; //#5863
        $k[$j++] = true; //#5863
        $k[$j++] = 'height'; //#5863
        $k[$j++] = $_.height; //#5863
        if ($_.includetext) { //#5863
            var _39 = $k[--$j]; //#5863
            $k[$j++] = $f(_39 - 0.15); //#5863
        } //#5863
        $k[$j++] = 'textxoffset'; //#5864
        if ($_.addontextxoffset != null) { //#5864
            $k[$j++] = $_.addontextxoffset; //#5864
        } else { //#5864
            $k[$j++] = 95 + $_.addongap; //#5864
        } //#5864
        if ($_.addontextyoffset != null) { //#5865
            $k[$j++] = 'textyoffset'; //#5865
            $k[$j++] = $_.addontextyoffset; //#5865
        } //#5865
        $k[$j++] = 'textsize'; //#5866
        if ($_.addontextsize != null) { //#5866
            $k[$j++] = $_.addontextsize; //#5866
        } else { //#5866
            $k[$j++] = $_.textsize; //#5866
        } //#5866
        var _3J = $ne($_.addontextfont, "unset") ? $_.addontextfont : $_.textfont; //#5867
        $k[$j++] = 'textfont'; //#5867
        $k[$j++] = _3J; //#5867
        $_.addopts = $d(); //#5868
        if ($_.addon.length == 2) { //#5869
            $k[$j++] = $_.addon; //#5869
            $k[$j++] = $_.addopts; //#5869
            bwipp_ean2(); //#5869
        } //#5869
        if ($_.addon.length == 5) { //#5870
            $k[$j++] = $_.addon; //#5870
            $k[$j++] = $_.addopts; //#5870
            bwipp_ean5(); //#5870
        } //#5870
        $_.addcode = $k[--$j]; //#5871
        $k[$j++] = Infinity; //#5872
        $aload($_.sbs); //#5872
        $k[$j++] = $_.addongap; //#5872
        $aload($get($_.addcode, "sbs")); //#5872
        $_.sbs = $a(); //#5872
        $k[$j++] = Infinity; //#5873
        $aload($_.bhs); //#5873
        $aload($get($_.addcode, "bhs")); //#5873
        $_.bhs = $a(); //#5873
        $k[$j++] = Infinity; //#5874
        $aload($_.bbs); //#5874
        $aload($get($_.addcode, "bbs")); //#5874
        $_.bbs = $a(); //#5874
        $k[$j++] = Infinity; //#5875
        $aload($_.txt); //#5875
        $aload($get($_.addcode, "txt")); //#5875
        $_.txt = $a(); //#5875
        $_.guardrightypos = ($_.height * 72) - 6; //#5876
    } //#5876
    $k[$j++] = Infinity; //#5880
    $k[$j++] = 'ren'; //#5887
    $k[$j++] = 'renlinear'; //#5887
    $k[$j++] = 'sbs'; //#5887
    $k[$j++] = $_.sbs; //#5887
    $k[$j++] = 'bhs'; //#5887
    $k[$j++] = $_.bhs; //#5887
    $k[$j++] = 'bbs'; //#5887
    $k[$j++] = $_.bbs; //#5887
    $k[$j++] = 'txt'; //#5887
    $k[$j++] = $_.txt; //#5887
    $k[$j++] = 'opt'; //#5887
    $k[$j++] = $_.options; //#5887
    $k[$j++] = 'guardrightpos'; //#5887
    if ($_.addon.length == 0) { //#5887
        $k[$j++] = 9; //#5887
    } else { //#5887
        $k[$j++] = 5; //#5887
    } //#5887
    $k[$j++] = 'guardrightypos'; //#5890
    $k[$j++] = $_.guardrightypos; //#5890
    $k[$j++] = 'borderleft'; //#5890
    $k[$j++] = 9; //#5890
    $k[$j++] = 'borderright'; //#5890
    if ($_.addon.length == 0) { //#5890
        $k[$j++] = 9; //#5890
    } else { //#5890
        $k[$j++] = 5; //#5890
    } //#5890
    $k[$j++] = 'bordertop'; //#5892
    $k[$j++] = 0; //#5892
    $k[$j++] = 'borderbottom'; //#5892
    $k[$j++] = 5; //#5892
    var _3s = $d(); //#5892
    $k[$j++] = _3s; //#5895
    if (!$_.dontdraw) { //#5895
        bwipp_renlinear(); //#5895
    } //#5895
    $_ = $__; //#5899
} //bwipp_upca
function bwipp_upce() {
    var $__ = $_; //#5931
    $_ = Object.assign({}, $_); //#5931
    $_.dontdraw = false; //#5934
    $_.includetext = false; //#5935
    $_.textfont = "OCR-B"; //#5936
    $_.textsize = 12; //#5937
    $_.textxoffset = -9; //#5938
    $_.textyoffset = -4; //#5939
    $_.height = 1; //#5940
    $_.addongap = 12; //#5941
    $_.addontextfont = "unset"; //#5942
    $_.addontextsize = null; //#5943
    $_.addontextxoffset = null; //#5944
    $_.addontextyoffset = null; //#5945
    $k[$j++] = $_; //#5947
    bwipp_processoptions(); //#5947
    $_.options = $k[--$j]; //#5947
    $_.barcode = $k[--$j]; //#5948
    $search($_.barcode, " "); //#5953
    if ($k[--$j]) { //#5959
        $_.barcode = $k[--$j]; //#5954
        $j--; //#5955
        $_.addon = $k[--$j]; //#5956
    } else { //#5959
        $j--; //#5958
        $_.addon = ""; //#5959
    } //#5959
    if (($_.barcode.length == 11) || ($_.barcode.length == 12)) { //#6001
        $forall($_.barcode, function() { //#5969
            var _9 = $k[--$j]; //#5966
            if ((_9 < 48) || (_9 > 57)) { //#5968
                $k[$j++] = 'bwipp.upcEupcAbadCharacter#5967'; //#5967
                $k[$j++] = "UPC-A must contain only digits"; //#5967
                bwipp_raiseerror(); //#5967
            } //#5967
        }); //#5967
        for (var _A = 0; _A < 1; _A++) { //#5996
            var _C = ($_.barcode.length == 12) ? 8 : 7; //#5971
            $_.upcecode = $s(_C); //#5971
            if ((($get($_.barcode, 3) - 48) <= 2) && $eq($geti($_.barcode, 4, 4), "0000")) { //#5977
                $puti($_.upcecode, 0, $geti($_.barcode, 0, 3)); //#5973
                $puti($_.upcecode, 3, $geti($_.barcode, 8, 3)); //#5974
                $puti($_.upcecode, 6, $geti($_.barcode, 3, 1)); //#5975
                break; //#5976
            } //#5976
            if ($eq($geti($_.barcode, 4, 5), "00000")) { //#5983
                $puti($_.upcecode, 0, $geti($_.barcode, 0, 4)); //#5979
                $puti($_.upcecode, 4, $geti($_.barcode, 9, 2)); //#5980
                $puti($_.upcecode, 6, "3"); //#5981
                break; //#5982
            } //#5982
            if ($eq($geti($_.barcode, 5, 5), "00000")) { //#5989
                $puti($_.upcecode, 0, $geti($_.barcode, 0, 5)); //#5985
                $puti($_.upcecode, 5, $geti($_.barcode, 10, 1)); //#5986
                $puti($_.upcecode, 6, "4"); //#5987
                break; //#5988
            } //#5988
            if ((($get($_.barcode, 10) - 48) >= 5) && $eq($geti($_.barcode, 6, 4), "0000")) { //#5994
                $puti($_.upcecode, 0, $geti($_.barcode, 0, 6)); //#5991
                $puti($_.upcecode, 6, $geti($_.barcode, 10, 1)); //#5992
                break; //#5993
            } //#5993
            $k[$j++] = 'bwipp.upcEupcAnotCompressible#5995'; //#5995
            $k[$j++] = "UPC-A cannot be converted to a UPC-E"; //#5995
            bwipp_raiseerror(); //#5995
        } //#5995
        if ($_.barcode.length == 12) { //#5999
            $puti($_.upcecode, 7, $geti($_.barcode, 11, 1)); //#5998
        } //#5998
        $_.barcode = $_.upcecode; //#6000
    } //#6000
    if (($_.barcode.length != 7) && ($_.barcode.length != 8)) { //#6006
        $k[$j++] = 'bwipp.upcEbadLength#6005'; //#6005
        $k[$j++] = "UPC-E must be 7 or 8 digits"; //#6005
        bwipp_raiseerror(); //#6005
    } //#6005
    var _10 = $_.barcode; //#6007
    for (var _11 = 0, _12 = _10.length; _11 < _12; _11++) { //#6011
        var _13 = $get(_10, _11); //#6011
        if ((_13 < 48) || (_13 > 57)) { //#6010
            $k[$j++] = 'bwipp.upcEbadCharacter#6009'; //#6009
            $k[$j++] = "UPC-E must contain only digits"; //#6009
            bwipp_raiseerror(); //#6009
        } //#6009
    } //#6009
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#6014
        $k[$j++] = 'bwipp.upcEbadAddOnLength#6013'; //#6013
        $k[$j++] = "Add-on for UPC-E must be 2 or 5 digits"; //#6013
        bwipp_raiseerror(); //#6013
    } //#6013
    var _18 = $get($_.barcode, 0); //#6017
    if ((_18 != 48) && (_18 != 49)) { //#6019
        $k[$j++] = 'bwipp.upcEbadNumberSystem#6018'; //#6018
        $k[$j++] = "UPC-E must have number system 0 or 1"; //#6018
        bwipp_raiseerror(); //#6018
    } //#6018
    $_.encs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "1312", "1213", "3112", "111", "111111"]); //#6027
    $_.barchars = "0123456789"; //#6030
    $_.mirrormaps = $a(["000111", "001011", "001101", "001110", "010011", "011001", "011100", "010101", "010110", "011010"]); //#6035
    for (var _1C = 0; _1C < 1; _1C++) { //#6066
        $_.upcacode = $s(11); //#6040
        if (($get($_.barcode, 6) - 48) <= 2) { //#6047
            $puti($_.upcacode, 0, $geti($_.barcode, 0, 3)); //#6042
            $puti($_.upcacode, 3, $geti($_.barcode, 6, 1)); //#6043
            $puti($_.upcacode, 4, "0000"); //#6044
            $puti($_.upcacode, 8, $geti($_.barcode, 3, 3)); //#6045
            break; //#6046
        } //#6046
        if (($get($_.barcode, 6) - 48) == 3) { //#6053
            $puti($_.upcacode, 0, $geti($_.barcode, 0, 4)); //#6049
            $puti($_.upcacode, 4, "00000"); //#6050
            $puti($_.upcacode, 9, $geti($_.barcode, 4, 2)); //#6051
            break; //#6052
        } //#6052
        if (($get($_.barcode, 6) - 48) == 4) { //#6059
            $puti($_.upcacode, 0, $geti($_.barcode, 0, 5)); //#6055
            $puti($_.upcacode, 5, "00000"); //#6056
            $puti($_.upcacode, 10, $geti($_.barcode, 5, 1)); //#6057
            break; //#6058
        } //#6058
        if (($get($_.barcode, 6) - 48) >= 5) { //#6065
            $puti($_.upcacode, 0, $geti($_.barcode, 0, 6)); //#6061
            $puti($_.upcacode, 6, "0000"); //#6062
            $puti($_.upcacode, 10, $geti($_.barcode, 6, 1)); //#6063
            break; //#6064
        } //#6064
    } //#6064
    $_.checksum = 0; //#6067
    for (var _1r = 0; _1r <= 10; _1r += 1) { //#6076
        $_.i = _1r; //#6069
        $_.barchar = $get($_.upcacode, $_.i) - 48; //#6070
        if (($_.i % 2) != 0) { //#6074
            $_.checksum = $_.checksum + $_.barchar; //#6072
        } else { //#6074
            $_.checksum = $_.checksum + ($_.barchar * 3); //#6074
        } //#6074
    } //#6074
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#6077
    if ($_.barcode.length == 8) { //#6082
        if ($get($_.barcode, 7) != ($_.checksum + 48)) { //#6081
            $k[$j++] = 'bwipp.upcEbadCheckDigit#6080'; //#6080
            $k[$j++] = "Incorrect UPC check digit provided"; //#6080
            bwipp_raiseerror(); //#6080
        } //#6080
    } //#6080
    $_.pad = $s(8); //#6083
    $puti($_.pad, 0, $_.barcode); //#6084
    $put($_.pad, 7, $_.checksum + 48); //#6085
    $_.barcode = $_.pad; //#6086
    $_.txt = $a(8); //#6087
    $put($_.txt, 0, $a([$geti($_.barcode, 0, 1), $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize - 2])); //#6088
    $_.mirrormap = $get($_.mirrormaps, $_.checksum); //#6091
    if ($get($_.barcode, 0) == 48) { //#6105
        $_.invt = $s($_.mirrormap.length); //#6095
        for (var _2T = 0, _2S = $_.mirrormap.length - 1; _2T <= _2S; _2T += 1) { //#6103
            $_.i = _2T; //#6097
            if ($get($_.mirrormap, $_.i) == 48) { //#6101
                $put($_.invt, $_.i, 49); //#6099
            } else { //#6101
                $put($_.invt, $_.i, 48); //#6101
            } //#6101
        } //#6101
        $_.mirrormap = $_.invt; //#6104
    } //#6104
    $_.sbs = $s(33); //#6107
    $puti($_.sbs, 0, $get($_.encs, 10)); //#6110
    for (var _2g = 1; _2g <= 6; _2g += 1) { //#6132
        $_.i = _2g; //#6113
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#6115
        $j--; //#6116
        $_.indx = $k[--$j].length; //#6117
        $j -= 2; //#6118
        $_.enc = $get($_.encs, $_.indx); //#6119
        if ($get($_.mirrormap, $_.i - 1) == 49) { //#6129
            $_.enclen = $_.enc.length; //#6121
            $_.revenc = $s($_.enclen); //#6122
            for (var _2x = 0, _2w = $_.enclen - 1; _2x <= _2w; _2x += 1) { //#6127
                $_.j = _2x; //#6124
                $_.char = $get($_.enc, $_.j); //#6125
                $put($_.revenc, ($_.enclen - $_.j) - 1, $_.char); //#6126
            } //#6126
            $_.enc = $_.revenc; //#6128
        } //#6128
        $puti($_.sbs, (($_.i - 1) * 4) + 3, $_.enc); //#6130
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 1) * 7) + 13) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])); //#6131
    } //#6131
    $put($_.txt, 7, $a([$geti($_.barcode, 7, 1), ((6 * 7) + 21) + $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize - 2])); //#6134
    $puti($_.sbs, 27, $get($_.encs, 11)); //#6137
    $k[$j++] = Infinity; //#6139
    var _3V = $_.sbs; //#6139
    for (var _3W = 0, _3X = _3V.length; _3W < _3X; _3W++) { //#6139
        $k[$j++] = $get(_3V, _3W) - 48; //#6139
    } //#6139
    $_.sbs = $a(); //#6139
    if ($_.includetext) { //#6146
        $k[$j++] = Infinity; //#6141
        $k[$j++] = $_.height; //#6141
        $k[$j++] = $_.height; //#6141
        for (var _3d = 0; _3d < 12; _3d++) { //#6141
            $k[$j++] = $f($_.height - 0.075); //#6141
        } //#6141
        $k[$j++] = $_.height; //#6141
        $k[$j++] = $_.height; //#6141
        $k[$j++] = $_.height; //#6141
        $_.bhs = $a(); //#6141
        $k[$j++] = Infinity; //#6142
        $k[$j++] = 0; //#6142
        $k[$j++] = 0; //#6142
        for (var _3j = 0; _3j < 12; _3j++) { //#6142
            $k[$j++] = 0.075; //#6142
        } //#6142
        $k[$j++] = 0; //#6142
        $k[$j++] = 0; //#6142
        $k[$j++] = 0; //#6142
        $_.bbs = $a(); //#6142
    } else { //#6146
        $k[$j++] = Infinity; //#6144
        for (var _3l = 0; _3l < 17; _3l++) { //#6144
            $k[$j++] = $_.height; //#6144
        } //#6144
        $_.bhs = $a(); //#6144
        $k[$j++] = Infinity; //#6145
        for (var _3o = 0; _3o < 17; _3o++) { //#6145
            $k[$j++] = 0; //#6145
        } //#6145
        $_.bbs = $a(); //#6145
        $_.txt = $a([]); //#6146
    } //#6146
    $_.guardrightypos = 0; //#6148
    if ($_.addon.length != 0) { //#6169
        $k[$j++] = Infinity; //#6152
        $k[$j++] = 'dontdraw'; //#6155
        $k[$j++] = true; //#6155
        $k[$j++] = 'includetext'; //#6155
        $k[$j++] = true; //#6155
        $k[$j++] = 'height'; //#6155
        $k[$j++] = $_.height; //#6155
        if ($_.includetext) { //#6155
            var _3u = $k[--$j]; //#6155
            $k[$j++] = $f(_3u - 0.15); //#6155
        } //#6155
        $k[$j++] = 'textxoffset'; //#6156
        if ($_.addontextxoffset != null) { //#6156
            $k[$j++] = $_.addontextxoffset; //#6156
        } else { //#6156
            $k[$j++] = 51 + $_.addongap; //#6156
        } //#6156
        if ($_.addontextyoffset != null) { //#6157
            $k[$j++] = 'textyoffset'; //#6157
            $k[$j++] = $_.addontextyoffset; //#6157
        } //#6157
        $k[$j++] = 'textsize'; //#6158
        if ($_.addontextsize != null) { //#6158
            $k[$j++] = $_.addontextsize; //#6158
        } else { //#6158
            $k[$j++] = $_.textsize; //#6158
        } //#6158
        var _44 = $ne($_.addontextfont, "unset") ? $_.addontextfont : $_.textfont; //#6159
        $k[$j++] = 'textfont'; //#6159
        $k[$j++] = _44; //#6159
        $_.addopts = $d(); //#6160
        if ($_.addon.length == 2) { //#6161
            $k[$j++] = $_.addon; //#6161
            $k[$j++] = $_.addopts; //#6161
            bwipp_ean2(); //#6161
        } //#6161
        if ($_.addon.length == 5) { //#6162
            $k[$j++] = $_.addon; //#6162
            $k[$j++] = $_.addopts; //#6162
            bwipp_ean5(); //#6162
        } //#6162
        $_.addcode = $k[--$j]; //#6163
        $k[$j++] = Infinity; //#6164
        $aload($_.sbs); //#6164
        $k[$j++] = $_.addongap; //#6164
        $aload($get($_.addcode, "sbs")); //#6164
        $_.sbs = $a(); //#6164
        $k[$j++] = Infinity; //#6165
        $aload($_.bhs); //#6165
        $aload($get($_.addcode, "bhs")); //#6165
        $_.bhs = $a(); //#6165
        $k[$j++] = Infinity; //#6166
        $aload($_.bbs); //#6166
        $aload($get($_.addcode, "bbs")); //#6166
        $_.bbs = $a(); //#6166
        $k[$j++] = Infinity; //#6167
        $aload($_.txt); //#6167
        $aload($get($_.addcode, "txt")); //#6167
        $_.txt = $a(); //#6167
        $_.guardrightypos = ($_.height * 72) - 6; //#6168
    } //#6168
    $k[$j++] = Infinity; //#6172
    $k[$j++] = 'ren'; //#6179
    $k[$j++] = 'renlinear'; //#6179
    $k[$j++] = 'sbs'; //#6179
    $k[$j++] = $_.sbs; //#6179
    $k[$j++] = 'bhs'; //#6179
    $k[$j++] = $_.bhs; //#6179
    $k[$j++] = 'bbs'; //#6179
    $k[$j++] = $_.bbs; //#6179
    $k[$j++] = 'txt'; //#6179
    $k[$j++] = $_.txt; //#6179
    $k[$j++] = 'opt'; //#6179
    $k[$j++] = $_.options; //#6179
    $k[$j++] = 'guardrightpos'; //#6179
    if ($_.addon.length == 0) { //#6179
        $k[$j++] = 9; //#6179
    } else { //#6179
        $k[$j++] = 5; //#6179
    } //#6179
    $k[$j++] = 'guardrightypos'; //#6182
    $k[$j++] = $_.guardrightypos; //#6182
    $k[$j++] = 'borderleft'; //#6182
    $k[$j++] = 9; //#6182
    $k[$j++] = 'borderright'; //#6182
    if ($_.addon.length == 0) { //#6182
        $k[$j++] = 9; //#6182
    } else { //#6182
        $k[$j++] = 5; //#6182
    } //#6182
    $k[$j++] = 'bordertop'; //#6184
    $k[$j++] = 0; //#6184
    $k[$j++] = 'borderbottom'; //#6184
    $k[$j++] = 5; //#6184
    var _4d = $d(); //#6184
    $k[$j++] = _4d; //#6187
    if (!$_.dontdraw) { //#6187
        bwipp_renlinear(); //#6187
    } //#6187
    $_ = $__; //#6191
} //bwipp_upce
function bwipp_isbn() {
    var $__ = $_; //#6220
    $_ = Object.assign({}, $_); //#6220
    $_.dontdraw = false; //#6223
    $_.includetext = false; //#6224
    $_.isbntextfont = 'OCR-A'; //#6225
    $_.isbntextsize = 8; //#6226
    $_.isbntextxoffset = null; //#6227
    $_.isbntextyoffset = null; //#6228
    $_.height = 1; //#6229
    $_.addongap = 12; //#6230
    $_.legacy = false; //#6231
    $k[$j++] = $_; //#6233
    bwipp_processoptions(); //#6233
    $_.options = $k[--$j]; //#6233
    $_.barcode = $k[--$j]; //#6234
    $search($_.barcode, " "); //#6237
    if ($k[--$j]) { //#6243
        $_.barcode = $k[--$j]; //#6238
        $j--; //#6239
        $_.addon = $k[--$j]; //#6240
    } else { //#6243
        $j--; //#6242
        $_.addon = ""; //#6243
    } //#6243
    if ((($_.barcode.length != 15) && ($_.barcode.length != 17)) && (($_.barcode.length != 11) && ($_.barcode.length != 13))) { //#6250
        $k[$j++] = 'bwipp.isbnBadLength#6249'; //#6249
        $k[$j++] = "ISBN-13 must be 15 or 17 characters including dashes. ISBN-10 must be 11 or 13 characters including dashes"; //#6249
        bwipp_raiseerror(); //#6249
    } //#6249
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#6253
        $k[$j++] = 'bwipp.isbnBadAddOnLength#6252'; //#6252
        $k[$j++] = "Add-on for ISBN must be 2 or 5 digits"; //#6252
        bwipp_raiseerror(); //#6252
    } //#6252
    if ($_.barcode.length >= 15) { //#6317
        var _F = $geti($_.barcode, 0, 4); //#6255
        if ($ne(_F, "978-") && $ne(_F, "979-")) { //#6257
            $k[$j++] = 'bwipp.isbn13badPrefix#6256'; //#6256
            $k[$j++] = "ISBN-13 prefix must be 978- or 979-"; //#6256
            bwipp_raiseerror(); //#6256
        } //#6256
        $_.wasdash = false; //#6258
        $_.numdash = 0; //#6258
        $_.numdigit = 0; //#6258
        var _H = $geti($_.barcode, 5, 9); //#6259
        for (var _I = 0, _J = _H.length; _I < _J; _I++) { //#6272
            var _K = $get(_H, _I); //#6272
            $k[$j++] = _K; //#6267
            if (_K == 45) { //#6267
                if ($_.wasdash) { //#6264
                    $k[$j++] = 'bwipp.isbn13adjacentDashes#6263'; //#6263
                    $k[$j++] = "ISBN-13 does not permit adjacent dashes"; //#6263
                    bwipp_raiseerror(); //#6263
                } //#6263
                $_.wasdash = true; //#6265
                $_.numdash = $_.numdash + 1; //#6266
            } //#6266
            var _N = $k[--$j]; //#6268
            if ((_N >= 48) && (_N <= 57)) { //#6271
                $_.wasdash = false; //#6269
                $_.numdigit = $_.numdigit + 1; //#6270
            } //#6270
        } //#6270
        if (($_.numdash != 2) || ($_.numdigit != 7)) { //#6275
            $k[$j++] = 'bwipp.isbn13numDashesDigits#6274'; //#6274
            $k[$j++] = "Incorrect number of dashes and digits for ISBN-13"; //#6274
            bwipp_raiseerror(); //#6274
        } //#6274
        var _S = $get($_.barcode, 14); //#6276
        if ((_S < 48) || (_S > 57)) { //#6278
            $k[$j++] = 'bwipp.isbn13character15#6277'; //#6277
            $k[$j++] = "ISBN-13 character 15 must be a digit"; //#6277
            bwipp_raiseerror(); //#6277
        } //#6277
        if ($_.barcode.length == 17) { //#6286
            if ($ne($geti($_.barcode, 15, 1), "-")) { //#6282
                $k[$j++] = 'bwipp.isbn13character16#6281'; //#6281
                $k[$j++] = "ISBN-13 penultimate character must be a dash"; //#6281
                bwipp_raiseerror(); //#6281
            } //#6281
            var _X = $get($_.barcode, 16); //#6283
            if ((_X < 48) || (_X > 57)) { //#6285
                $k[$j++] = 'bwipp.isbn13character17#6284'; //#6284
                $k[$j++] = "ISBN-13 final character must be a digit"; //#6284
                bwipp_raiseerror(); //#6284
            } //#6284
        } //#6284
    } else { //#6317
        var _Z = $get($_.barcode, 0); //#6288
        if ((_Z < 48) || (_Z > 57)) { //#6290
            $k[$j++] = 'bwipp.isbn10FirstDigit#6289'; //#6289
            $k[$j++] = "ISBN-10 first character must be a digit"; //#6289
            bwipp_raiseerror(); //#6289
        } //#6289
        $_.wasdash = false; //#6291
        $_.numdash = 0; //#6291
        $_.numdigit = 0; //#6291
        var _b = $geti($_.barcode, 1, 9); //#6292
        for (var _c = 0, _d = _b.length; _c < _d; _c++) { //#6305
            var _e = $get(_b, _c); //#6305
            $k[$j++] = _e; //#6300
            if (_e == 45) { //#6300
                if ($_.wasdash) { //#6297
                    $k[$j++] = 'bwipp.isbn10adjacentDashes#6296'; //#6296
                    $k[$j++] = "ISBN-10 does not permit adjacent dashes"; //#6296
                    bwipp_raiseerror(); //#6296
                } //#6296
                $_.wasdash = true; //#6298
                $_.numdash = $_.numdash + 1; //#6299
            } //#6299
            var _h = $k[--$j]; //#6301
            if ((_h >= 48) && (_h <= 57)) { //#6304
                $_.wasdash = false; //#6302
                $_.numdigit = $_.numdigit + 1; //#6303
            } //#6303
        } //#6303
        if (($_.numdash != 2) || ($_.numdigit != 7)) { //#6308
            $k[$j++] = 'bwipp.isbn10numDashesDigits#6307'; //#6307
            $k[$j++] = "Incorrect number of dashes and digits for ISBN-10"; //#6307
            bwipp_raiseerror(); //#6307
        } //#6307
        var _m = $get($_.barcode, 10); //#6309
        if ((_m < 48) || (_m > 57)) { //#6311
            $k[$j++] = 'bwipp.isbn10character11#6310'; //#6310
            $k[$j++] = "ISBN-10 character 11 must be a digit"; //#6310
            bwipp_raiseerror(); //#6310
        } //#6310
        if ($_.barcode.length == 13) { //#6319
            if ($ne($geti($_.barcode, 11, 1), "-")) { //#6315
                $k[$j++] = 'bwipp.isbn10character12#6314'; //#6314
                $k[$j++] = "ISBN-10 penultimate character must be a dash"; //#6314
                bwipp_raiseerror(); //#6314
            } //#6314
            var _r = $get($_.barcode, 12); //#6316
            if (((_r < 48) || (_r > 57)) && (_r != 88)) { //#6318
                $k[$j++] = 'bwipp.isbn10character13#6317'; //#6317
                $k[$j++] = "ISBN-10 final character must be a digit or X"; //#6317
                bwipp_raiseerror(); //#6317
            } //#6317
        } //#6317
    } //#6317
    $_.isbntxt = $_.barcode; //#6322
    if ($_.isbntxt.length <= 13) { //#6348
        $_.isbn = $s(10); //#6326
        $_.checksum = 0; //#6327
        $_.i = 0; //#6328
        $_.n = 0; //#6328
        for (;;) { //#6340
            $_.isbnchar = $get($_.isbntxt, $_.i) - 48; //#6330
            if ($_.isbnchar != -3) { //#6337
                $put($_.isbn, $_.n, $_.isbnchar + 48); //#6332
                if ($_.n < 9) { //#6335
                    $_.checksum = $f($_.checksum + ((10 - $_.n) * $_.isbnchar)); //#6334
                } //#6334
                $_.n = $_.n + 1; //#6336
            } //#6336
            $_.i = $_.i + 1; //#6338
            if ($_.i == $_.isbntxt.length) { //#6339
                break; //#6339
            } //#6339
        } //#6339
        var _1B = (11 - ($_.checksum % 11)) % 11; //#6341
        $k[$j++] = 'checksum'; //#6341
        $k[$j++] = _1B; //#6341
        if (_1B == 10) { //#6341
            $k[$j - 1] = 40; //#6341
        } //#6341
        var _1C = $k[--$j]; //#6341
        $_[$k[--$j]] = _1C + 48; //#6341
        $_.isbn = $geti($_.isbn, 0, 9); //#6342
        if ($_.isbntxt.length == 13) { //#6347
            if ($get($_.isbntxt, 12) != $_.checksum) { //#6346
                $k[$j++] = 'bwipp.isbn10badCheckDigit#6345'; //#6345
                $k[$j++] = "Incorrect ISBN-10 check digit provided"; //#6345
                bwipp_raiseerror(); //#6345
            } //#6345
        } //#6345
    } //#6345
    if ((!$_.legacy) || ($_.isbntxt.length >= 15)) { //#6384
        if ($_.isbntxt.length <= 13) { //#6358
            $_.pad = $s(15); //#6354
            $puti($_.pad, 0, "978-"); //#6355
            $puti($_.pad, 4, $geti($_.isbntxt, 0, 11)); //#6356
            $_.isbntxt = $_.pad; //#6357
        } //#6357
        $_.isbn = $s(13); //#6361
        $_.checksum = 0; //#6362
        $_.i = 0; //#6363
        $_.n = 0; //#6363
        for (;;) { //#6375
            $_.isbnchar = $get($_.isbntxt, $_.i) - 48; //#6365
            if ($_.isbnchar != -3) { //#6372
                $put($_.isbn, $_.n, $_.isbnchar + 48); //#6367
                if ($_.n < 12) { //#6370
                    $k[$j++] = 'checksum'; //#6369
                    $k[$j++] = $_.isbnchar; //#6369
                    if (($_.n % 2) != 0) { //#6369
                        var _1e = $k[--$j]; //#6369
                        $k[$j++] = _1e * 3; //#6369
                    } //#6369
                    var _1g = $k[--$j]; //#6369
                    $_[$k[--$j]] = $f(_1g + $_.checksum); //#6369
                } //#6369
                $_.n = $_.n + 1; //#6371
            } //#6371
            $_.i = $_.i + 1; //#6373
            if ($_.i == $_.isbntxt.length) { //#6374
                break; //#6374
            } //#6374
        } //#6374
        $_.checksum = ((10 - ($_.checksum % 10)) % 10) + 48; //#6376
        $_.isbn = $geti($_.isbn, 0, 12); //#6377
        if ($_.isbntxt.length == 17) { //#6382
            if ($get($_.isbntxt, 16) != $_.checksum) { //#6381
                $k[$j++] = 'bwipp.isbn13badCheckDigit#6380'; //#6380
                $k[$j++] = "Incorrect ISBN-13 check digit provided"; //#6380
                bwipp_raiseerror(); //#6380
            } //#6380
        } //#6380
    } //#6380
    var _1u = ($_.isbn.length == 12) ? 22 : 18; //#6387
    $_.pad = $s(_1u); //#6387
    $puti($_.pad, 0, "ISBN "); //#6388
    $puti($_.pad, 5, $_.isbntxt); //#6389
    $put($_.pad, $_.pad.length - 2, 45); //#6390
    $put($_.pad, $_.pad.length - 1, $_.checksum); //#6391
    $_.isbntxt = $_.pad; //#6392
    $_.barcode = $s(12); //#6395
    if ($_.isbn.length == 9) { //#6400
        $puti($_.barcode, 0, "978"); //#6397
        $puti($_.barcode, 3, $_.isbn); //#6398
    } else { //#6400
        $puti($_.barcode, 0, $_.isbn); //#6400
    } //#6400
    if ($_.addon.length != 0) { //#6410
        var _2E = $s((12 + $_.addon.length) + 1); //#6405
        $puti(_2E, 0, $_.barcode); //#6406
        $puti(_2E, 12, " "); //#6407
        $puti(_2E, 13, $_.addon); //#6408
        $_.barcode = _2E; //#6409
    } //#6409
    $put($_.options, "dontdraw", true); //#6413
    $put($_.options, "addongap", $_.addongap); //#6414
    $k[$j++] = 'args'; //#6415
    $k[$j++] = $_.barcode; //#6415
    $k[$j++] = $_.options; //#6415
    bwipp_ean13(); //#6415
    var _2M = $k[--$j]; //#6415
    $_[$k[--$j]] = _2M; //#6415
    if ($_.includetext) { //#6434
        if ($_.isbntextxoffset == null) { //#6421
            $k[$j++] = 'isbntextxoffset'; //#6420
            if ($_.isbn.length == 9) { //#6420
                $k[$j++] = -1; //#6420
            } else { //#6420
                $k[$j++] = -12; //#6420
            } //#6420
            var _2R = $k[--$j]; //#6420
            $_[$k[--$j]] = _2R; //#6420
        } //#6420
        if ($_.isbntextyoffset == null) { //#6424
            $_.isbntextyoffset = ($_.height * 72) + 3; //#6423
        } //#6423
        var _2W = $get($_.args, "txt") !== undefined; //#6425
        if (_2W) { //#6432
            $_.txt = $get($_.args, "txt"); //#6426
            $_.newtxt = $a($_.txt.length + 1); //#6427
            $puti($_.newtxt, 0, $_.txt); //#6428
            $put($_.newtxt, $_.newtxt.length - 1, $a([$_.isbntxt, $_.isbntextxoffset, $_.isbntextyoffset, $_.isbntextfont, $_.isbntextsize])); //#6429
            $put($_.args, "txt", $_.newtxt); //#6430
        } else { //#6432
            $put($_.args, "txt", $a([$a([$_.isbntxt, $_.isbntextxoffset, $_.isbntextyoffset, $_.isbntextfont, $_.isbntextsize])])); //#6432
        } //#6432
    } //#6432
    $put($_.args, "opt", $_.options); //#6436
    $k[$j++] = $_.args; //#6439
    if (!$_.dontdraw) { //#6439
        bwipp_renlinear(); //#6439
    } //#6439
    $_ = $__; //#6441
} //bwipp_isbn
function bwipp_ismn() {
    var $__ = $_; //#6470
    $_ = Object.assign({}, $_); //#6470
    $_.dontdraw = false; //#6473
    $_.includetext = false; //#6474
    $_.ismntextfont = 'OCR-A'; //#6475
    $_.ismntextsize = 8; //#6476
    $_.ismntextxoffset = null; //#6477
    $_.ismntextyoffset = null; //#6478
    $_.height = 1; //#6479
    $_.addongap = 12; //#6480
    $_.legacy = false; //#6481
    $k[$j++] = $_; //#6483
    bwipp_processoptions(); //#6483
    $_.options = $k[--$j]; //#6483
    $_.barcode = $k[--$j]; //#6484
    $search($_.barcode, " "); //#6487
    if ($k[--$j]) { //#6493
        $_.barcode = $k[--$j]; //#6488
        $j--; //#6489
        $_.addon = $k[--$j]; //#6490
    } else { //#6493
        $j--; //#6492
        $_.addon = ""; //#6493
    } //#6493
    if ((($_.barcode.length != 15) && ($_.barcode.length != 17)) && (($_.barcode.length != 11) && ($_.barcode.length != 13))) { //#6500
        $k[$j++] = 'bwipp.ismnBadLength#6499'; //#6499
        $k[$j++] = "ISMN-13 must be 15 or 17 characters including dashes. ISMN-10 must be 11 or 13 characters including dashes"; //#6499
        bwipp_raiseerror(); //#6499
    } //#6499
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#6503
        $k[$j++] = 'bwipp.ismnBadAddOnLength#6502'; //#6502
        $k[$j++] = "Add-on for ISMN must be 2 or 5 digits"; //#6502
        bwipp_raiseerror(); //#6502
    } //#6502
    if ($_.barcode.length >= 15) { //#6570
        if ($ne($geti($_.barcode, 0, 4), "979-")) { //#6507
            $k[$j++] = 'bwipp.ismn13badPrefix#6506'; //#6506
            $k[$j++] = "ISMN-13 prefix must be 979-"; //#6506
            bwipp_raiseerror(); //#6506
        } //#6506
        $_.wasdash = false; //#6508
        $_.numdash = 0; //#6508
        $_.numdigit = 0; //#6508
        var _H = $geti($_.barcode, 5, 9); //#6509
        for (var _I = 0, _J = _H.length; _I < _J; _I++) { //#6522
            var _K = $get(_H, _I); //#6522
            $k[$j++] = _K; //#6517
            if (_K == 45) { //#6517
                if ($_.wasdash) { //#6514
                    $k[$j++] = 'bwipp.ismn13adjacentDashes#6513'; //#6513
                    $k[$j++] = "ISMN-13 does not permit adjacent dashes"; //#6513
                    bwipp_raiseerror(); //#6513
                } //#6513
                $_.wasdash = true; //#6515
                $_.numdash = $_.numdash + 1; //#6516
            } //#6516
            var _N = $k[--$j]; //#6518
            if ((_N >= 48) && (_N <= 57)) { //#6521
                $_.wasdash = false; //#6519
                $_.numdigit = $_.numdigit + 1; //#6520
            } //#6520
        } //#6520
        if (($_.numdash != 2) || ($_.numdigit != 7)) { //#6525
            $k[$j++] = 'bwipp.ismn13numDashesDigits#6524'; //#6524
            $k[$j++] = "Incorrect number of dashes and digits for ISMN-13"; //#6524
            bwipp_raiseerror(); //#6524
        } //#6524
        var _S = $get($_.barcode, 14); //#6526
        if ((_S < 48) || (_S > 57)) { //#6528
            $k[$j++] = 'bwipp.ismn13character15#6527'; //#6527
            $k[$j++] = "ISMN-13 character 15 must be a digit"; //#6527
            bwipp_raiseerror(); //#6527
        } //#6527
        if ($_.barcode.length == 17) { //#6536
            if ($ne($geti($_.barcode, 15, 1), "-")) { //#6532
                $k[$j++] = 'bwipp.ismn13character16#6531'; //#6531
                $k[$j++] = "ISMN-13 penultimate character must be a dash"; //#6531
                bwipp_raiseerror(); //#6531
            } //#6531
            var _X = $get($_.barcode, 16); //#6533
            if ((_X < 48) || (_X > 57)) { //#6535
                $k[$j++] = 'bwipp.ismn13character17#6534'; //#6534
                $k[$j++] = "ISMN-13 final character must be a digit"; //#6534
                bwipp_raiseerror(); //#6534
            } //#6534
        } //#6534
    } else { //#6570
        if ($ne($geti($_.barcode, 0, 2), "M-")) { //#6540
            $k[$j++] = 'bwipp.ismn10badPrefix#6539'; //#6539
            $k[$j++] = "ISMN-10 prefix must be M-"; //#6539
            bwipp_raiseerror(); //#6539
        } //#6539
        var _b = $get($_.barcode, 2); //#6541
        if ((_b < 48) || (_b > 57)) { //#6543
            $k[$j++] = 'bwipp.ismn10character3#6542'; //#6542
            $k[$j++] = "ISMN-10 character 3 must be a digit"; //#6542
            bwipp_raiseerror(); //#6542
        } //#6542
        $_.wasdash = false; //#6544
        $_.numdash = 0; //#6544
        $_.numdigit = 0; //#6544
        var _d = $geti($_.barcode, 3, 7); //#6545
        for (var _e = 0, _f = _d.length; _e < _f; _e++) { //#6558
            var _g = $get(_d, _e); //#6558
            $k[$j++] = _g; //#6553
            if (_g == 45) { //#6553
                if ($_.wasdash) { //#6550
                    $k[$j++] = 'bwipp.ismn10adjacentDashes#6549'; //#6549
                    $k[$j++] = "ISMN-10 does not permit adjacent dashes"; //#6549
                    bwipp_raiseerror(); //#6549
                } //#6549
                $_.wasdash = true; //#6551
                $_.numdash = $_.numdash + 1; //#6552
            } //#6552
            var _j = $k[--$j]; //#6554
            if ((_j >= 48) && (_j <= 57)) { //#6557
                $_.wasdash = false; //#6555
                $_.numdigit = $_.numdigit + 1; //#6556
            } //#6556
        } //#6556
        if (($_.numdash != 1) || ($_.numdigit != 6)) { //#6561
            $k[$j++] = 'bwipp.ismn10numDashesDigits#6560'; //#6560
            $k[$j++] = "Incorrect number of dashes and digits for ISMN-10"; //#6560
            bwipp_raiseerror(); //#6560
        } //#6560
        var _o = $get($_.barcode, 10); //#6562
        if ((_o < 48) || (_o > 57)) { //#6564
            $k[$j++] = 'bwipp.ismn10character11#6563'; //#6563
            $k[$j++] = "ISMN-10 character 11 must be a digit"; //#6563
            bwipp_raiseerror(); //#6563
        } //#6563
        if ($_.barcode.length == 13) { //#6572
            var _r = $geti($_.barcode, 11, 1); //#6566
            if ($ne(_r, "-")) { //#6568
                $k[$j++] = 'bwipp.ismn10character12#6567'; //#6567
                $k[$j++] = "ISMN-10 penultimate character must be a dash"; //#6567
                bwipp_raiseerror(); //#6567
            } //#6567
            var _t = $get($_.barcode, 12); //#6569
            if (((_t < 48) || (_t > 57)) && (_t != 88)) { //#6571
                $k[$j++] = 'bwipp.ismn10character13#6570'; //#6570
                $k[$j++] = "ISMN-10 final character must be a digit or X"; //#6570
                bwipp_raiseerror(); //#6570
            } //#6570
        } //#6570
    } //#6570
    $_.ismntxt = $_.barcode; //#6575
    $_.legacytxt = ""; //#6578
    if ($_.ismntxt.length <= 13) { //#6585
        $_.legacytxt = $_.ismntxt; //#6580
        $_.pad = $s($_.ismntxt.length + 4); //#6581
        $puti($_.pad, 0, "979-0-"); //#6582
        $puti($_.pad, 6, $geti($_.ismntxt, 2, $_.ismntxt.length - 2)); //#6583
        $_.ismntxt = $_.pad; //#6584
    } //#6584
    $_.ismn = $s(13); //#6588
    $_.checksum = 0; //#6589
    $_.i = 0; //#6590
    $_.n = 0; //#6590
    for (;;) { //#6606
        $_.ismnchar = $get($_.ismntxt, $_.i) - 48; //#6592
        if ($_.ismnchar != -3) { //#6603
            $put($_.ismn, $_.n, $_.ismnchar + 48); //#6594
            if ($_.n < 12) { //#6601
                if (($_.n % 2) == 0) { //#6599
                    $_.checksum = $_.ismnchar + $_.checksum; //#6597
                } else { //#6599
                    $_.checksum = ($_.ismnchar * 3) + $_.checksum; //#6599
                } //#6599
            } //#6599
            $_.n = $_.n + 1; //#6602
        } //#6602
        $_.i = $_.i + 1; //#6604
        if ($_.i == $_.ismntxt.length) { //#6605
            break; //#6605
        } //#6605
    } //#6605
    $_.checksum = ((10 - ($_.checksum % 10)) % 10) + 48; //#6607
    if (($_.barcode.length == 13) || ($_.barcode.length == 17)) { //#6614
        var _1Q = $_.barcode; //#6611
        if ($get(_1Q, _1Q.length - 1) != $_.checksum) { //#6613
            $k[$j++] = 'bwipp.ismnBadCheckDigit#6612'; //#6612
            $k[$j++] = "Incorrect ISMN check digit provided"; //#6612
            bwipp_raiseerror(); //#6612
        } //#6612
    } //#6612
    if ($_.legacy && ($_.legacytxt.length != 0)) { //#6621
        $_.ismntxt = $_.legacytxt; //#6618
        $_.pad = $s(18); //#6619
    } else { //#6621
        $_.pad = $s(22); //#6621
    } //#6621
    $puti($_.pad, 0, "ISMN "); //#6623
    $puti($_.pad, 5, $_.ismntxt); //#6624
    $put($_.pad, $_.pad.length - 2, 45); //#6625
    $put($_.pad, $_.pad.length - 1, $_.checksum); //#6626
    $_.ismntxt = $_.pad; //#6627
    $_.barcode = $geti($_.ismn, 0, 12); //#6630
    if ($_.addon.length != 0) { //#6639
        var _1l = $s((12 + $_.addon.length) + 1); //#6634
        $puti(_1l, 0, $_.barcode); //#6635
        $puti(_1l, 12, " "); //#6636
        $puti(_1l, 13, $_.addon); //#6637
        $_.barcode = _1l; //#6638
    } //#6638
    $put($_.options, "dontdraw", true); //#6642
    $put($_.options, "addongap", $_.addongap); //#6643
    $k[$j++] = 'args'; //#6644
    $k[$j++] = $_.barcode; //#6644
    $k[$j++] = $_.options; //#6644
    bwipp_ean13(); //#6644
    var _1t = $k[--$j]; //#6644
    $_[$k[--$j]] = _1t; //#6644
    if ($_.includetext) { //#6663
        if ($_.ismntextxoffset == null) { //#6650
            $k[$j++] = 'ismntextxoffset'; //#6649
            if ($_.ismntxt.length == 18) { //#6649
                $k[$j++] = -1; //#6649
            } else { //#6649
                $k[$j++] = -12; //#6649
            } //#6649
            var _1y = $k[--$j]; //#6649
            $_[$k[--$j]] = _1y; //#6649
        } //#6649
        if ($_.ismntextyoffset == null) { //#6653
            $_.ismntextyoffset = ($_.height * 72) + 3; //#6652
        } //#6652
        var _23 = $get($_.args, "txt") !== undefined; //#6654
        if (_23) { //#6661
            $_.txt = $get($_.args, "txt"); //#6655
            $_.newtxt = $a($_.txt.length + 1); //#6656
            $puti($_.newtxt, 0, $_.txt); //#6657
            $put($_.newtxt, $_.newtxt.length - 1, $a([$_.ismntxt, $_.ismntextxoffset, $_.ismntextyoffset, $_.ismntextfont, $_.ismntextsize])); //#6658
            $put($_.args, "txt", $_.newtxt); //#6659
        } else { //#6661
            $put($_.args, "txt", $a([$a([$_.ismntxt, $_.ismntextxoffset, $_.ismntextyoffset, $_.ismntextfont, $_.ismntextsize])])); //#6661
        } //#6661
    } //#6661
    $put($_.args, "opt", $_.options); //#6665
    $k[$j++] = $_.args; //#6668
    if (!$_.dontdraw) { //#6668
        bwipp_renlinear(); //#6668
    } //#6668
    $_ = $__; //#6670
} //bwipp_ismn
function bwipp_issn() {
    var $__ = $_; //#6699
    $_ = Object.assign({}, $_); //#6699
    $_.dontdraw = false; //#6702
    $_.includetext = false; //#6703
    $_.issntextfont = 'OCR-A'; //#6704
    $_.issntextsize = 8; //#6705
    $_.issntextxoffset = null; //#6706
    $_.issntextyoffset = null; //#6707
    $_.height = 1; //#6708
    $_.addongap = 12; //#6709
    $k[$j++] = $_; //#6711
    bwipp_processoptions(); //#6711
    $_.options = $k[--$j]; //#6711
    $_.issntxt = $k[--$j]; //#6712
    $search($_.issntxt, " "); //#6715
    if ($k[--$j]) { //#6721
        $_.issntxt = $k[--$j]; //#6716
        $j--; //#6717
        $_.seqvar = $k[--$j]; //#6718
    } else { //#6721
        $j--; //#6720
        $_.seqvar = "00"; //#6721
    } //#6721
    $search($_.seqvar, " "); //#6725
    if ($k[--$j]) { //#6731
        $_.seqvar = $k[--$j]; //#6726
        $j--; //#6727
        $_.addon = $k[--$j]; //#6728
    } else { //#6731
        $j--; //#6730
        $_.addon = ""; //#6731
    } //#6731
    if (($_.issntxt.length != 8) && ($_.issntxt.length != 9)) { //#6737
        $k[$j++] = 'bwipp.issnBadLength#6736'; //#6736
        $k[$j++] = "ISSN must be 8 or 9 characters including dash, in the format XXXX-XXXX"; //#6736
        bwipp_raiseerror(); //#6736
    } //#6736
    var _D = $geti($_.issntxt, 0, 4); //#6738
    for (var _E = 0, _F = _D.length; _E < _F; _E++) { //#6742
        var _G = $get(_D, _E); //#6742
        if ((_G < 48) || (_G > 57)) { //#6741
            $k[$j++] = 'bwipp.issnFirstThroughFourthNotNumeric#6740'; //#6740
            $k[$j++] = "ISSN first four characters must be numeral characters"; //#6740
            bwipp_raiseerror(); //#6740
        } //#6740
    } //#6740
    if ($ne($geti($_.issntxt, 4, 1), "-")) { //#6745
        $k[$j++] = 'bwipp.issnNeedsDash#6744'; //#6744
        $k[$j++] = "ISSN fifth character must be a dash"; //#6744
        bwipp_raiseerror(); //#6744
    } //#6744
    var _K = $geti($_.issntxt, 5, 3); //#6746
    for (var _L = 0, _M = _K.length; _L < _M; _L++) { //#6750
        var _N = $get(_K, _L); //#6750
        if ((_N < 48) || (_N > 57)) { //#6749
            $k[$j++] = 'bwipp.issnSixthThroughEighthNotNumeric#6748'; //#6748
            $k[$j++] = "ISSN sixth through eighth characters must be numerals"; //#6748
            bwipp_raiseerror(); //#6748
        } //#6748
    } //#6748
    if ($_.issntxt.length == 9) { //#6755
        var _Q = $get($_.issntxt, 8); //#6752
        if (((_Q < 48) || (_Q > 57)) && (_Q != 88)) { //#6754
            $k[$j++] = 'bwipp.issnNinthCharacterBadFormat#6753'; //#6753
            $k[$j++] = "ISSN ninth character must be a number or the character X"; //#6753
            bwipp_raiseerror(); //#6753
        } //#6753
    } //#6753
    if ($_.seqvar.length != 2) { //#6758
        $k[$j++] = 'bwipp.issnBadSequenceVariantLength#6757'; //#6757
        $k[$j++] = "Sequence variant for ISSN must be 2 digits"; //#6757
        bwipp_raiseerror(); //#6757
    } //#6757
    $forall($_.seqvar, function() { //#6763
        var _T = $k[--$j]; //#6760
        if ((_T < 48) || (_T > 57)) { //#6762
            $k[$j++] = 'bwipp.issnSequenceVariantBadCharacter#6761'; //#6761
            $k[$j++] = "Sequence variant for ISSN must contain only digits"; //#6761
            bwipp_raiseerror(); //#6761
        } //#6761
    }); //#6761
    if ((($_.addon.length != 0) && ($_.addon.length != 2)) && ($_.addon.length != 5)) { //#6766
        $k[$j++] = 'bwipp.issnBadAddOnLength#6765'; //#6765
        $k[$j++] = "Add-on for ISSN must be 2 or 5 digits"; //#6765
        bwipp_raiseerror(); //#6765
    } //#6765
    $_.issn = $s(8); //#6769
    $_.checksum = 0; //#6770
    $_.i = 0; //#6771
    $_.n = 0; //#6771
    for (;;) { //#6783
        $_.issnchar = $get($_.issntxt, $_.i) - 48; //#6773
        if ($_.issnchar != -3) { //#6780
            $put($_.issn, $_.n, $_.issnchar + 48); //#6775
            if ($_.n < 7) { //#6778
                $_.checksum = $f($_.checksum + ($_.issnchar * (8 - $_.n))); //#6777
            } //#6777
            $_.n = $_.n + 1; //#6779
        } //#6779
        $_.i = $_.i + 1; //#6781
        if ($_.i == $_.issntxt.length) { //#6782
            break; //#6782
        } //#6782
    } //#6782
    $_.checksum = (11 - ($_.checksum % 11)) % 11; //#6784
    var _p = $_.checksum + 48; //#6785
    $k[$j++] = 'checksum'; //#6785
    $k[$j++] = _p; //#6785
    if (_p == 58) { //#6785
        $k[$j - 1] = 88; //#6785
    } //#6785
    var _q = $k[--$j]; //#6785
    var _r = $k[--$j]; //#6785
    $_[_r] = _q; //#6785
    if ($_.issntxt.length == 9) { //#6790
        if ($get($_.issntxt, 8) != $_.checksum) { //#6789
            $k[$j++] = 'bwipp.issnBadCheckDigit#6788'; //#6788
            $k[$j++] = "Incorrect ISSN check digit provided"; //#6788
            bwipp_raiseerror(); //#6788
        } //#6788
    } //#6788
    $_.pad = $s(14); //#6793
    $puti($_.pad, 0, "ISSN "); //#6794
    $puti($_.pad, 5, $_.issntxt); //#6795
    $put($_.pad, 13, $_.checksum); //#6796
    $_.issntxt = $_.pad; //#6797
    $_.barcode = $geti($_.issn, 0, 7); //#6800
    $_.barcode = $s(12); //#6803
    $puti($_.barcode, 0, "977"); //#6804
    $puti($_.barcode, 3, $_.issn); //#6805
    $puti($_.barcode, 10, $_.seqvar); //#6806
    if ($_.addon.length != 0) { //#6815
        var _1D = $s((12 + $_.addon.length) + 1); //#6810
        $puti(_1D, 0, $_.barcode); //#6811
        $puti(_1D, 12, " "); //#6812
        $puti(_1D, 13, $_.addon); //#6813
        $_.barcode = _1D; //#6814
    } //#6814
    $put($_.options, "dontdraw", true); //#6818
    $put($_.options, "addongap", $_.addongap); //#6819
    $k[$j++] = 'args'; //#6820
    $k[$j++] = $_.barcode; //#6820
    $k[$j++] = $_.options; //#6820
    bwipp_ean13(); //#6820
    var _1L = $k[--$j]; //#6820
    $_[$k[--$j]] = _1L; //#6820
    if ($_.includetext) { //#6835
        if ($_.issntextxoffset == null) { //#6824
            $_.issntextxoffset = 10; //#6824
        } //#6824
        if ($_.issntextyoffset == null) { //#6825
            $_.issntextyoffset = ($_.height * 72) + 3; //#6825
        } //#6825
        var _1S = $get($_.args, "txt") !== undefined; //#6826
        if (_1S) { //#6833
            $_.txt = $get($_.args, "txt"); //#6827
            $_.newtxt = $a($_.txt.length + 1); //#6828
            $puti($_.newtxt, 0, $_.txt); //#6829
            $put($_.newtxt, $_.newtxt.length - 1, $a([$_.issntxt, $_.issntextxoffset, $_.issntextyoffset, $_.issntextfont, $_.issntextsize])); //#6830
            $put($_.args, "txt", $_.newtxt); //#6831
        } else { //#6833
            $put($_.args, "txt", $a([$a([$_.issntxt, $_.issntextxoffset, $_.issntextyoffset, $_.issntextfont, $_.issntextsize])])); //#6833
        } //#6833
    } //#6833
    $put($_.args, "opt", $_.options); //#6837
    $k[$j++] = $_.args; //#6840
    if (!$_.dontdraw) { //#6840
        bwipp_renlinear(); //#6840
    } //#6840
    $_ = $__; //#6842
} //bwipp_issn
function bwipp_mands() {
    var $__ = $_; //#6871
    $_ = Object.assign({}, $_); //#6871
    $_.dontdraw = false; //#6874
    $_.includetext = false; //#6875
    $k[$j++] = $_; //#6877
    bwipp_processoptions(); //#6877
    $_.options = $k[--$j]; //#6877
    $_.barcode = $k[--$j]; //#6878
    $_.barlen = $_.barcode.length; //#6880
    if (($_.barlen != 7) && ($_.barlen != 8)) { //#6885
        $k[$j++] = 'bwipp.MandSbadLength#6884'; //#6884
        $k[$j++] = "M&S barcode must be 7 or 8 characters"; //#6884
        bwipp_raiseerror(); //#6884
    } //#6884
    if ($_.barlen == 7) { //#6889
        var _7 = $strcpy($s(8), "00000000"); //#6888
        $puti(_7, 1, $_.barcode); //#6888
        $_.barcode = _7; //#6888
    } //#6888
    $put($_.options, "dontdraw", true); //#6892
    $k[$j++] = 'args'; //#6893
    $k[$j++] = $_.barcode; //#6893
    $k[$j++] = $_.options; //#6893
    bwipp_ean8(); //#6893
    var _C = $k[--$j]; //#6893
    $_[$k[--$j]] = _C; //#6893
    var _F = $get($_.args, "bbs"); //#6896
    $put(_F, 10, $get(_F, 2)); //#6897
    $put(_F, 11, $get(_F, 2)); //#6898
    var _J = $get($_.args, "bhs"); //#6899
    $put(_J, 10, $get(_J, 2)); //#6900
    $put(_J, 11, $get(_J, 2)); //#6901
    $_.txt = $get($_.args, "txt"); //#6904
    if ($_.barlen == 7) { //#6912
        for (var _P = 0; _P <= 6; _P += 1) { //#6910
            var _Q = $_.txt; //#6907
            $puti($get($get(_Q, _P), 0), 0, $get($get(_Q, _P + 1), 0)); //#6909
        } //#6909
        $puti($get($get($_.txt, 7), 0), 0, " "); //#6911
    } //#6911
    var _Y = $a(10); //#6915
    $puti(_Y, 0, $_.txt); //#6915
    $_.txt = _Y; //#6915
    var _b = $get($_.txt, 0); //#6916
    $put($_.txt, 8, $arrcpy($a(_b.length), _b)); //#6916
    var _g = $get($_.txt, 0); //#6917
    $put($_.txt, 9, $arrcpy($a(_g.length), _g)); //#6917
    $put($get($_.txt, 8), 0, "M"); //#6918
    $put($get($_.txt, 8), 1, -12); //#6918
    $put($get($_.txt, 9), 0, "S"); //#6919
    var _r = $get($_.txt, 9); //#6919
    $put(_r, 1, 69); //#6919
    $put($_.args, "txt", $_.txt); //#6921
    $put($_.args, "opt", $_.options); //#6922
    $k[$j++] = $_.args; //#6925
    if (!$_.dontdraw) { //#6925
        bwipp_renlinear(); //#6925
    } //#6925
    $_ = $__; //#6927
} //bwipp_mands
function bwipp_code128() {
    var $__ = $_; //#6958
    $_ = Object.assign({}, $_); //#6958
    $_.dontdraw = false; //#6961
    $_.includetext = false; //#6962
    $_.textfont = "OCR-B"; //#6963
    $_.textsize = 10; //#6964
    $_.textxoffset = 0; //#6965
    $_.textyoffset = -8; //#6966
    $_.height = 1; //#6967
    $_.raw = false; //#6968
    $_.newencoder = false; //#6969
    $_.parse = false; //#6970
    $_.parsefnc = false; //#6971
    $_.suppressc = false; //#6972
    $_.unlatchextbeforec = false; //#6973
    $k[$j++] = $_; //#6975
    bwipp_processoptions(); //#6975
    $_.options = $k[--$j]; //#6975
    $_.barcode = $k[--$j]; //#6976
    $_.sta = -1; //#6982
    $_.stb = -2; //#6982
    $_.stc = -3; //#6982
    $_.swa = -4; //#6983
    $_.swb = -5; //#6983
    $_.swc = -6; //#6983
    $_.fn1 = -7; //#6984
    $_.fn2 = -8; //#6984
    $_.fn3 = -9; //#6984
    $_.fn4 = -10; //#6985
    $_.sft = -11; //#6985
    $_.stp = -12; //#6985
    $_.lka = -13; //#6986
    $_.lkc = -14; //#6986
    var _A = new Map([
        ["parse", $_.parse],
        ["parsefnc", $_.parsefnc],
        ["FNC1", $_.fn1],
        ["FNC2", $_.fn2],
        ["FNC3", $_.fn3],
        ["LNKA", $_.lka],
        ["LNKC", $_.lkc]
    ]); //#6998
    $_.fncvals = _A; //#6999
    $k[$j++] = 'msg'; //#7000
    $k[$j++] = $_.barcode; //#7000
    $k[$j++] = $_.fncvals; //#7000
    bwipp_parseinput(); //#7000
    var _D = $k[--$j]; //#7000
    $_[$k[--$j]] = _D; //#7000
    $_.msglen = $_.msg.length; //#7001
    $_.encoding = "legacy"; //#7003
    if ($_.newencoder) { //#7004
        $_.encoding = "new"; //#7004
    } //#7004
    if ($_.raw) { //#7005
        $_.encoding = "raw"; //#7005
    } //#7005
    if ($eq($_.encoding, "raw")) { //#7023
        $_.cws = $a($_.barcode.length); //#7008
        $_.i = 0; //#7009
        $_.j = 0; //#7009
        for (;;) { //#7016
            if ($_.i == $_.barcode.length) { //#7011
                break; //#7011
            } //#7011
            $_.cw = $cvi($geti($_.barcode, $_.i + 1, 3)); //#7012
            $put($_.cws, $_.j, $_.cw); //#7013
            $_.i = $_.i + 4; //#7014
            $_.j = $_.j + 1; //#7015
        } //#7015
        $_.cws = $geti($_.cws, 0, $_.j); //#7017
        $_.text = ""; //#7018
    } else { //#7023
        $_.text = $s($_.msglen); //#7020
        for (var _c = 0, _b = $_.msglen - 1; _c <= _b; _c += 1) { //#7024
            $_.i = _c; //#7022
            var _h = $get($_.msg, $_.i); //#7023
            $k[$j++] = $_.text; //#7023
            $k[$j++] = $_.i; //#7023
            $k[$j++] = _h; //#7023
            if (_h < 0) { //#7023
                $k[$j - 1] = 32; //#7023
            } //#7023
            var _i = $k[--$j]; //#7023
            var _j = $k[--$j]; //#7023
            $put($k[--$j], _j, _i); //#7023
        } //#7023
    } //#7023
    $_.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$_.fn3, $_.fn3, "96"]), $a([$_.fn2, $_.fn2, "97"]), $a([$_.sft, $_.sft, "98"]), $a([$_.swc, $_.swc, "99"]), $a([$_.swb, $_.fn4, $_.swb]), $a([$_.fn4, $_.swa, $_.swa]), $a([$_.fn1, $_.fn1, $_.fn1]), $a([$_.sta, $_.sta, $_.sta]), $a([$_.stb, $_.stb, $_.stb]), $a([$_.stc, $_.stc, $_.stc]), $a([$_.stp, $_.stp, $_.stp])]); //#7067
    $_.charvals = $a([new Map, new Map, new Map]); //#7070
    for (var _31 = 0, _30 = $_.charmaps.length - 1; _31 <= _30; _31 += 1) { //#7079
        $_.i = _31; //#7072
        $_.encs = $get($_.charmaps, $_.i); //#7073
        for (var _35 = 0; _35 <= 2; _35 += 1) { //#7078
            $_.j = _35; //#7075
            var _38 = $get($_.encs, $_.j); //#7076
            $k[$j++] = _38; //#7076
            if ($eq($type(_38), 'stringtype')) { //#7076
                var _3B = $get($k[--$j], 0); //#7076
                $k[$j++] = _3B; //#7076
            } //#7076
            $put($get($_.charvals, $_.j), $k[--$j], $_.i); //#7077
        } //#7077
    } //#7077
    if ($eq($_.encoding, "legacy")) { //#7306
        $_.seta = $get($_.charvals, 0); //#7085
        $put($_.seta, $_.lka, $get($_.seta, $_.swb)); //#7085
        $put($_.seta, $_.lkc, $get($_.seta, $_.swc)); //#7085
        $_.setb = $get($_.charvals, 1); //#7086
        $put($_.setb, $_.lka, $get($_.setb, $_.swc)); //#7086
        $put($_.setb, $_.lkc, $get($_.setb, $_.swa)); //#7086
        $_.setc = $get($_.charvals, 2); //#7087
        $put($_.setc, $_.lka, $get($_.setc, $_.swa)); //#7087
        $put($_.setc, $_.lkc, $get($_.setc, $_.swb)); //#7087
        $k[$j++] = Infinity; //#7090
        for (var _3u = 0, _3v = $_.msglen; _3u < _3v; _3u++) { //#7090
            $k[$j++] = 0; //#7090
        } //#7090
        $k[$j++] = 0; //#7090
        $_.numSA = $a(); //#7090
        $k[$j++] = Infinity; //#7091
        for (var _3y = 0, _3z = $_.msglen; _3y < _3z; _3y++) { //#7091
            $k[$j++] = 0; //#7091
        } //#7091
        $k[$j++] = 0; //#7091
        $_.numEA = $a(); //#7091
        for (var _42 = $_.msglen - 1; _42 >= 0; _42 -= 1) { //#7101
            $_.i = _42; //#7093
            if ($get($_.msg, $_.i) >= 0) { //#7100
                if ($get($_.msg, $_.i) >= 128) { //#7098
                    $put($_.numEA, $_.i, $get($_.numEA, $_.i + 1) + 1); //#7096
                } else { //#7098
                    $put($_.numSA, $_.i, $get($_.numSA, $_.i + 1) + 1); //#7098
                } //#7098
            } //#7098
        } //#7098
        $_.ea = false; //#7104
        $_.msgtmp = $a([]); //#7104
        for (var _4M = 0, _4L = $_.msglen - 1; _4M <= _4L; _4M += 1) { //#7119
            $_.i = _4M; //#7106
            $_.c = $get($_.msg, $_.i); //#7107
            if ((!($_.ea != ($_.c < 128))) && ($_.c >= 0)) { //#7117
                if ($_.ea) { //#7109
                    $k[$j++] = $_.numSA; //#7109
                } else { //#7109
                    $k[$j++] = $_.numEA; //#7109
                } //#7109
                var _4Y = $get($k[--$j], $_.i); //#7109
                var _4b = ($f(_4Y + $_.i) == $_.msglen) ? 3 : 5; //#7110
                if (_4Y < _4b) { //#7115
                    $k[$j++] = Infinity; //#7112
                    $aload($_.msgtmp); //#7112
                    $k[$j++] = $_.fn4; //#7112
                    $_.msgtmp = $a(); //#7112
                } else { //#7115
                    $k[$j++] = Infinity; //#7114
                    $aload($_.msgtmp); //#7114
                    $k[$j++] = $_.fn4; //#7114
                    $k[$j++] = $_.fn4; //#7114
                    $_.msgtmp = $a(); //#7114
                    $_.ea = !$_.ea; //#7115
                } //#7115
            } //#7115
            $k[$j++] = Infinity; //#7118
            $aload($_.msgtmp); //#7118
            if ($_.c >= 0) { //#7118
                $k[$j++] = $_.c & 127; //#7118
            } else { //#7118
                $k[$j++] = $_.c; //#7118
            } //#7118
            $_.msgtmp = $a(); //#7118
        } //#7118
        $_.msg = $_.msgtmp; //#7120
        $_.msglen = $_.msg.length; //#7121
        $_.numsscr = function() {
            $_.s = 0; //#7125
            $_.p = $k[--$j]; //#7126
            for (;;) { //#7138
                if ($_.p >= $_.msglen) { //#7127
                    break; //#7127
                } //#7127
                var _4w = $get($_.msg, $_.p); //#7128
                var _4y = $get($_.setc, _4w) !== undefined; //#7129
                $k[$j++] = _4w; //#7129
                if (!_4y) { //#7129
                    $j--; //#7129
                    break; //#7129
                } //#7129
                var _4z = $k[$j - 1]; //#7130
                if (_4z <= -1) { //#7134
                    var _51 = $k[--$j]; //#7132
                    if ((_51 == $_.fn1) && (($_.s % 2) == 0)) { //#7132
                        $_.s = $_.s + 1; //#7132
                    } else { //#7132
                        break; //#7132
                    } //#7132
                } else { //#7134
                    $j--; //#7134
                } //#7134
                $_.s = $_.s + 1; //#7136
                $_.p = $_.p + 1; //#7137
            } //#7137
            $k[$j++] = $_.s; //#7139
        }; //#7139
        $_.enca = function() {
            $put($_.cws, $_.j, $get($_.seta, $k[--$j])); //#7144
            $_.j = $_.j + 1; //#7145
        }; //#7145
        $_.encb = function() {
            $put($_.cws, $_.j, $get($_.setb, $k[--$j])); //#7148
            $_.j = $_.j + 1; //#7149
        }; //#7149
        $_.encc = function() {
            var _5J = $k[$j - 1]; //#7152
            if ($ne($type(_5J), 'arraytype')) { //#7155
                var _5N = $get($_.setc, $k[--$j]); //#7153
                $k[$j++] = _5N; //#7153
            } else { //#7155
                $aload($k[--$j]); //#7155
                var _5P = $k[--$j]; //#7155
                var _5Q = $k[--$j]; //#7155
                $k[$j++] = $f((_5P - 48) + ((_5Q - 48) * 10)); //#7155
            } //#7155
            $put($_.cws, $_.j, $k[--$j]); //#7157
            $_.j = $_.j + 1; //#7158
        }; //#7158
        $_.anotb = function() {
            var _5V = $k[--$j]; //#7162
            var _5X = $get($_.seta, _5V) !== undefined; //#7162
            var _5Z = $get($_.setb, _5V) !== undefined; //#7162
            $k[$j++] = _5X && (!_5Z); //#7162
        }; //#7162
        $_.bnota = function() {
            var _5a = $k[--$j]; //#7163
            var _5c = $get($_.setb, _5a) !== undefined; //#7163
            var _5e = $get($_.seta, _5a) !== undefined; //#7163
            $k[$j++] = _5c && (!_5e); //#7163
        }; //#7163
        $k[$j++] = Infinity; //#7166
        for (var _5g = 0, _5h = $_.msg.length; _5g < _5h; _5g++) { //#7166
            $k[$j++] = 0; //#7166
        } //#7166
        $k[$j++] = 9999; //#7166
        $_.nextanotb = $a(); //#7166
        $k[$j++] = Infinity; //#7167
        for (var _5k = 0, _5l = $_.msg.length; _5k < _5l; _5k++) { //#7167
            $k[$j++] = 0; //#7167
        } //#7167
        $k[$j++] = 9999; //#7167
        $_.nextbnota = $a(); //#7167
        for (var _5o = $_.msg.length - 1; _5o >= 0; _5o -= 1) { //#7180
            $_.i = _5o; //#7169
            $k[$j++] = $get($_.msg, $_.i); //#7170
            $_.anotb(); //#7170
            if ($k[--$j]) { //#7173
                $put($_.nextanotb, $_.i, 0); //#7171
            } else { //#7173
                $put($_.nextanotb, $_.i, $get($_.nextanotb, $_.i + 1) + 1); //#7173
            } //#7173
            $k[$j++] = $get($_.msg, $_.i); //#7175
            $_.bnota(); //#7175
            if ($k[--$j]) { //#7178
                $put($_.nextbnota, $_.i, 0); //#7176
            } else { //#7178
                $put($_.nextbnota, $_.i, $get($_.nextbnota, $_.i + 1) + 1); //#7178
            } //#7178
        } //#7178
        $_.abeforeb = function() {
            var _6B = $k[--$j]; //#7183
            $k[$j++] = $lt($get($_.nextanotb, _6B), $get($_.nextbnota, _6B)); //#7183
        }; //#7183
        $_.bbeforea = function() {
            var _6G = $k[--$j]; //#7184
            $k[$j++] = $lt($get($_.nextbnota, _6G), $get($_.nextanotb, _6G)); //#7184
        }; //#7184
        $_.cws = $a(($_.barcode.length * 2) + 3); //#7186
        $_.j = 0; //#7189
        if ($_.msglen > 0) { //#7190
            $k[$j++] = 0; //#7190
            $_.numsscr(); //#7190
        } else { //#7190
            $k[$j++] = -1; //#7190
        } //#7190
        $_.nums = $k[--$j]; //#7190
        for (;;) { //#7215
            if ($_.msglen == 0) { //#7196
                $k[$j++] = $_.stb; //#7193
                $_.enca(); //#7193
                $_.cset = "setb"; //#7194
                break; //#7195
            } //#7195
            if (($_.msglen == 2) && ($_.nums == 2)) { //#7201
                $k[$j++] = $_.stc; //#7198
                $_.enca(); //#7198
                $_.cset = "setc"; //#7199
                break; //#7200
            } //#7200
            if ($_.nums >= 4) { //#7206
                $k[$j++] = $_.stc; //#7203
                $_.enca(); //#7203
                $_.cset = "setc"; //#7204
                break; //#7205
            } //#7205
            $k[$j++] = 0; //#7207
            $_.abeforeb(); //#7207
            if ($k[--$j]) { //#7211
                $k[$j++] = $_.sta; //#7208
                $_.enca(); //#7208
                $_.cset = "seta"; //#7209
                break; //#7210
            } //#7210
            $k[$j++] = $_.stb; //#7212
            $_.enca(); //#7212
            $_.cset = "setb"; //#7213
            break; //#7214
        } //#7214
        $_.i = 0; //#7218
        for (;;) { //#7303
            if ($_.i == $_.msglen) { //#7219
                break; //#7219
            } //#7219
            $k[$j++] = $_.i; //#7221
            $_.numsscr(); //#7221
            $_.nums = $k[--$j]; //#7221
            for (;;) { //#7301
                if ((($eq($_.cset, "seta") || $eq($_.cset, "setb")) && ($_.nums >= 4)) && ($get($_.msg, $_.i) != $_.fn1)) { //#7240
                    if (($_.nums % 2) == 0) { //#7237
                        $k[$j++] = $_.swc; //#7228
                        if ($eq($_.cset, "seta")) { //#7228
                            $_.enca(); //#7228
                        } else { //#7228
                            $_.encb(); //#7228
                        } //#7228
                        $_.cset = "setc"; //#7229
                        break; //#7230
                    } else { //#7237
                        $k[$j++] = $get($_.msg, $_.i); //#7232
                        if ($eq($_.cset, "seta")) { //#7232
                            $_.enca(); //#7232
                        } else { //#7232
                            $_.encb(); //#7232
                        } //#7232
                        $_.i = $_.i + 1; //#7233
                        $k[$j++] = $_.i; //#7234
                        $_.numsscr(); //#7234
                        if ($k[--$j] >= 4) { //#7238
                            $k[$j++] = $_.swc; //#7235
                            if ($eq($_.cset, "seta")) { //#7235
                                $_.enca(); //#7235
                            } else { //#7235
                                $_.encb(); //#7235
                            } //#7235
                            $_.cset = "setc"; //#7236
                            break; //#7237
                        } //#7237
                    } //#7237
                } //#7237
                $k[$j++] = $eq($_.cset, "setb"); //#7241
                $k[$j++] = $get($_.msg, $_.i); //#7241
                $_.anotb(); //#7241
                var _70 = $k[--$j]; //#7241
                if ($k[--$j] && _70) { //#7253
                    if ($_.i < ($_.msglen - 1)) { //#7249
                        $k[$j++] = $_.i + 1; //#7243
                        $_.bbeforea(); //#7243
                        if ($k[--$j]) { //#7248
                            $k[$j++] = $_.sft; //#7244
                            $_.encb(); //#7244
                            $k[$j++] = $get($_.msg, $_.i); //#7245
                            $_.enca(); //#7245
                            $_.i = $_.i + 1; //#7246
                            break; //#7247
                        } //#7247
                    } //#7247
                    $k[$j++] = $_.swa; //#7250
                    $_.encb(); //#7250
                    $_.cset = "seta"; //#7251
                    break; //#7252
                } //#7252
                $k[$j++] = $eq($_.cset, "seta"); //#7254
                $k[$j++] = $get($_.msg, $_.i); //#7254
                $_.bnota(); //#7254
                var _7G = $k[--$j]; //#7254
                if ($k[--$j] && _7G) { //#7266
                    if ($_.i < ($_.msglen - 1)) { //#7262
                        $k[$j++] = $_.i + 1; //#7256
                        $_.abeforeb(); //#7256
                        if ($k[--$j]) { //#7261
                            $k[$j++] = $_.sft; //#7257
                            $_.enca(); //#7257
                            $k[$j++] = $get($_.msg, $_.i); //#7258
                            $_.encb(); //#7258
                            $_.i = $_.i + 1; //#7259
                            break; //#7260
                        } //#7260
                    } //#7260
                    $k[$j++] = $_.swb; //#7263
                    $_.enca(); //#7263
                    $_.cset = "setb"; //#7264
                    break; //#7265
                } //#7265
                if (($eq($_.cset, "setc") && ($_.nums < 2)) && (($get($_.msg, $_.i) > -1) || ($get($_.msg, $_.i) == $_.fn4))) { //#7276
                    $k[$j++] = $_.i; //#7268
                    $_.abeforeb(); //#7268
                    if ($k[--$j]) { //#7272
                        $k[$j++] = $_.swa; //#7269
                        $_.encc(); //#7269
                        $_.cset = "seta"; //#7270
                        break; //#7271
                    } //#7271
                    $k[$j++] = $_.swb; //#7273
                    $_.encc(); //#7273
                    $_.cset = "setb"; //#7274
                    break; //#7275
                } //#7275
                if ($eq($_.cset, "seta")) { //#7283
                    $k[$j++] = $get($_.msg, $_.i); //#7280
                    $_.enca(); //#7280
                    $_.i = $_.i + 1; //#7281
                    break; //#7282
                } //#7282
                if ($eq($_.cset, "setb")) { //#7288
                    $k[$j++] = $get($_.msg, $_.i); //#7285
                    $_.encb(); //#7285
                    $_.i = $_.i + 1; //#7286
                    break; //#7287
                } //#7287
                if ($eq($_.cset, "setc")) { //#7298
                    if ($get($_.msg, $_.i) <= -1) { //#7295
                        $k[$j++] = $get($_.msg, $_.i); //#7291
                        $_.encc(); //#7291
                        $_.i = $_.i + 1; //#7292
                    } else { //#7295
                        $k[$j++] = $geti($_.msg, $_.i, 2); //#7294
                        $_.encc(); //#7294
                        $_.i = $_.i + 2; //#7295
                    } //#7295
                    break; //#7297
                } //#7297
                break; //#7300
            } //#7300
        } //#7300
        $_.cws = $geti($_.cws, 0, $_.j); //#7304
    } //#7304
    if ($eq($_.encoding, "new")) { //#7469
        $k[$j++] = Infinity; //#7311
        $k[$j++] = $_.fn3; //#7311
        $k[$j++] = 96; //#7311
        $k[$j++] = $_.fn2; //#7311
        $k[$j++] = 97; //#7311
        $k[$j++] = $_.fn1; //#7311
        $k[$j++] = 102; //#7311
        $k[$j++] = $_.stp; //#7311
        $k[$j++] = 106; //#7311
        $k[$j++] = $_.lka; //#7311
        $k[$j++] = 100; //#7311
        $k[$j++] = $_.lkc; //#7311
        $k[$j++] = 99; //#7311
        $_.seta = $d(); //#7311
        $k[$j++] = Infinity; //#7312
        $k[$j++] = $_.fn3; //#7312
        $k[$j++] = 96; //#7312
        $k[$j++] = $_.fn2; //#7312
        $k[$j++] = 97; //#7312
        $k[$j++] = $_.fn1; //#7312
        $k[$j++] = 102; //#7312
        $k[$j++] = $_.stp; //#7312
        $k[$j++] = 106; //#7312
        $k[$j++] = $_.lka; //#7312
        $k[$j++] = 99; //#7312
        $k[$j++] = $_.lkc; //#7312
        $k[$j++] = 101; //#7312
        $_.setb = $d(); //#7312
        $k[$j++] = Infinity; //#7313
        $k[$j++] = $_.fn1; //#7313
        $k[$j++] = 102; //#7313
        $k[$j++] = $_.stp; //#7313
        $k[$j++] = 106; //#7313
        $k[$j++] = $_.lka; //#7313
        $k[$j++] = 101; //#7313
        $k[$j++] = $_.lkc; //#7313
        $k[$j++] = 100; //#7313
        $_.setc = $d(); //#7313
        $_.latch_a0 = $a(["", "e", "e", "ee", "eee", "eee"]); //#7326
        $_.latch_b0 = $a(["d", "", "d", "ddd", "dd", "ddd"]); //#7327
        $_.latch_c0 = $a(["c", "c", "", "eec", "ddc", "dddc"]); //#7328
        $_.latch_a1 = $a(["ee", "eee", "eee", "", "e", "e"]); //#7329
        $_.latch_b1 = $a(["ddd", "dd", "ddd", "d", "", "d"]); //#7330
        $_.latch_c1 = $a(["eec", "ddc", "dddc", "c", "c", ""]); //#7331
        $k[$j++] = Infinity; //#7333
        var _8U = $_.latch_a0; //#7333
        for (var _8V = 0, _8W = _8U.length; _8V < _8W; _8V++) { //#7333
            $k[$j++] = $get(_8U, _8V).length; //#7333
        } //#7333
        $_.latch_length_a0 = $a(); //#7333
        $k[$j++] = Infinity; //#7334
        var _8Z = $_.latch_a1; //#7334
        for (var _8a = 0, _8b = _8Z.length; _8a < _8b; _8a++) { //#7334
            $k[$j++] = $get(_8Z, _8a).length; //#7334
        } //#7334
        $_.latch_length_a1 = $a(); //#7334
        $k[$j++] = Infinity; //#7335
        var _8e = $_.latch_b0; //#7335
        for (var _8f = 0, _8g = _8e.length; _8f < _8g; _8f++) { //#7335
            $k[$j++] = $get(_8e, _8f).length; //#7335
        } //#7335
        $_.latch_length_b0 = $a(); //#7335
        $k[$j++] = Infinity; //#7336
        var _8j = $_.latch_b1; //#7336
        for (var _8k = 0, _8l = _8j.length; _8k < _8l; _8k++) { //#7336
            $k[$j++] = $get(_8j, _8k).length; //#7336
        } //#7336
        $_.latch_length_b1 = $a(); //#7336
        $k[$j++] = Infinity; //#7337
        var _8o = $_.latch_c0; //#7337
        for (var _8p = 0, _8q = _8o.length; _8p < _8q; _8p++) { //#7337
            $k[$j++] = $get(_8o, _8p).length; //#7337
        } //#7337
        $_.latch_length_c0 = $a(); //#7337
        $k[$j++] = Infinity; //#7338
        var _8t = $_.latch_c1; //#7338
        for (var _8u = 0, _8v = _8t.length; _8u < _8v; _8u++) { //#7338
            $k[$j++] = $get(_8t, _8u).length; //#7338
        } //#7338
        $_.latch_length_c1 = $a(); //#7338
        $_.latch_sequence = $a([$_.latch_a0, $_.latch_b0, $_.latch_c0, $_.latch_a1, $_.latch_b1, $_.latch_c1]); //#7341
        var _95 = $a([function() {
            $_.enc_a0(); //#7342
        }, function() {
            $_.enc_b0(); //#7342
        }, function() {
            $_.enc_c(); //#7342
        }, function() {
            $_.enc_a1(); //#7342
        }, function() {
            $_.enc_b1(); //#7342
        }, function() {
            $_.enc_c(); //#7342
        }]); //#7342
        $_.encode = _95; //#7342
        $_.start_code = $a([103, 104, 105]); //#7343
        $_.state_priority = $a([1, 0, 2, 4, 3, 5]); //#7345
        $_.start_state = $a([0, 1, 2, 0, 1, 2]); //#7346
        $_.start_length = $a([1, 1, 1, 1, 1, 1]); //#7347
        $k[$j++] = Infinity; //#7350
        var _9A = $a([5, 4, 3, 2, 1, 0]); //#7350
        for (var _9B = 0, _9C = _9A.length; _9B < _9C; _9B++) { //#7350
            $k[$j++] = $get($_.state_priority, $get(_9A, _9B)); //#7350
        } //#7350
        $_.reverse_priority = $a(); //#7350
        $k[$j++] = Infinity; //#7353
        var _9H = $_.reverse_priority; //#7353
        for (var _9I = 0, _9J = _9H.length; _9I < _9J; _9I++) { //#7353
            var _9K = $get(_9H, _9I); //#7353
            $k[$j++] = _9K; //#7353
            $k[$j++] = $get($_.latch_length_a0, _9K); //#7353
            $k[$j++] = _9K; //#7353
        } //#7353
        $_.prioritized_latch_length_a0 = $a(); //#7353
        $k[$j++] = Infinity; //#7354
        var _9O = $_.reverse_priority; //#7354
        for (var _9P = 0, _9Q = _9O.length; _9P < _9Q; _9P++) { //#7354
            var _9R = $get(_9O, _9P); //#7354
            $k[$j++] = _9R; //#7354
            $k[$j++] = $get($_.latch_length_a1, _9R); //#7354
            $k[$j++] = _9R; //#7354
        } //#7354
        $_.prioritized_latch_length_a1 = $a(); //#7354
        $k[$j++] = Infinity; //#7355
        var _9V = $_.reverse_priority; //#7355
        for (var _9W = 0, _9X = _9V.length; _9W < _9X; _9W++) { //#7355
            var _9Y = $get(_9V, _9W); //#7355
            $k[$j++] = _9Y; //#7355
            $k[$j++] = $get($_.latch_length_b0, _9Y); //#7355
            $k[$j++] = _9Y; //#7355
        } //#7355
        $_.prioritized_latch_length_b0 = $a(); //#7355
        $k[$j++] = Infinity; //#7356
        var _9c = $_.reverse_priority; //#7356
        for (var _9d = 0, _9e = _9c.length; _9d < _9e; _9d++) { //#7356
            var _9f = $get(_9c, _9d); //#7356
            $k[$j++] = _9f; //#7356
            $k[$j++] = $get($_.latch_length_b1, _9f); //#7356
            $k[$j++] = _9f; //#7356
        } //#7356
        $_.prioritized_latch_length_b1 = $a(); //#7356
        $k[$j++] = Infinity; //#7357
        var _9j = $_.reverse_priority; //#7357
        for (var _9k = 0, _9l = _9j.length; _9k < _9l; _9k++) { //#7357
            var _9m = $get(_9j, _9k); //#7357
            $k[$j++] = _9m; //#7357
            $k[$j++] = $get($_.latch_length_c0, _9m); //#7357
            $k[$j++] = _9m; //#7357
        } //#7357
        $_.prioritized_latch_length_c0 = $a(); //#7357
        $k[$j++] = Infinity; //#7358
        var _9q = $_.reverse_priority; //#7358
        for (var _9r = 0, _9s = _9q.length; _9r < _9s; _9r++) { //#7358
            var _9t = $get(_9q, _9r); //#7358
            $k[$j++] = _9t; //#7358
            $k[$j++] = $get($_.latch_length_c1, _9t); //#7358
            $k[$j++] = _9t; //#7358
        } //#7358
        $_.prioritized_latch_length_c1 = $a(); //#7358
        $_.max_int = 2147483647; //#7360
        $_.can_a = function() {
            if ($_.c >= 0) { //#7363
                $k[$j++] = true; //#7363
            } else { //#7363
                var _A0 = $get($_.seta, $_.c) !== undefined; //#7363
                $k[$j++] = _A0; //#7363
            } //#7363
        }; //#7363
        $_.can_b = function() {
            if ($_.c >= 0) { //#7364
                $k[$j++] = true; //#7364
            } else { //#7364
                var _A4 = $get($_.setb, $_.c) !== undefined; //#7364
                $k[$j++] = _A4; //#7364
            } //#7364
        }; //#7364
        $_.can_c0 = function() {
            if ($_.num_digits >= 2) { //#7365
                $k[$j++] = true; //#7365
            } else { //#7365
                var _A8 = $get($_.setc, $_.c) !== undefined; //#7365
                $k[$j++] = _A8; //#7365
            } //#7365
        }; //#7365
        $_.can_c1 = function() {
            if ($_.num_digits >= 2) { //#7366
                $k[$j++] = true; //#7366
            } else { //#7366
                var _AC = $get($_.setc, $_.c) !== undefined; //#7366
                $k[$j++] = _AC; //#7366
            } //#7366
        }; //#7366
        if ($_.suppressc) { //#7369
            $_.can_c0 = function() {
                $k[$j++] = false; //#7369
            }; //#7369
        } //#7369
        if ($_.suppressc || $_.unlatchextbeforec) { //#7370
            $_.can_c1 = function() {
                $k[$j++] = false; //#7370
            }; //#7370
        } //#7370
        $_.out_a0 = function() {
            $k[$j++] = 1; //#7373
            if ($_.c >= 0) { //#7373
                if ($_.c >= 128) { //#7373
                    var _AI = $k[--$j]; //#7373
                    $k[$j++] = _AI + 1; //#7373
                } //#7373
                if (($_.c & 127) >= 96) { //#7373
                    var _AK = $k[--$j]; //#7373
                    $k[$j++] = _AK + 1; //#7373
                } //#7373
            } //#7373
        }; //#7373
        $_.out_a1 = function() {
            $k[$j++] = 1; //#7374
            if ($_.c >= 0) { //#7374
                if ($_.c < 128) { //#7374
                    var _AN = $k[--$j]; //#7374
                    $k[$j++] = _AN + 1; //#7374
                } //#7374
                if (($_.c & 127) >= 96) { //#7374
                    var _AP = $k[--$j]; //#7374
                    $k[$j++] = _AP + 1; //#7374
                } //#7374
            } //#7374
        }; //#7374
        $_.out_b0 = function() {
            $k[$j++] = 1; //#7375
            if ($_.c >= 0) { //#7375
                if ($_.c >= 128) { //#7375
                    var _AS = $k[--$j]; //#7375
                    $k[$j++] = _AS + 1; //#7375
                } //#7375
                if (($_.c & 127) < 32) { //#7375
                    var _AU = $k[--$j]; //#7375
                    $k[$j++] = _AU + 1; //#7375
                } //#7375
            } //#7375
        }; //#7375
        $_.out_b1 = function() {
            $k[$j++] = 1; //#7376
            if ($_.c >= 0) { //#7376
                if ($_.c < 128) { //#7376
                    var _AX = $k[--$j]; //#7376
                    $k[$j++] = _AX + 1; //#7376
                } //#7376
                if (($_.c & 127) < 32) { //#7376
                    var _AZ = $k[--$j]; //#7376
                    $k[$j++] = _AZ + 1; //#7376
                } //#7376
            } //#7376
        }; //#7376
        $_.map_ab = function() {
            var _Aa = $k[$j - 1]; //#7379
            if (_Aa < 32) { //#7379
                var _Ab = $k[--$j]; //#7379
                $k[$j++] = _Ab + 64; //#7379
            } else { //#7379
                var _Ac = $k[--$j]; //#7379
                $k[$j++] = _Ac - 32; //#7379
            } //#7379
        }; //#7379
        $_.enc_a0 = function() {
            $k[$j++] = Infinity; //#7380
            if ($_.c < 0) { //#7380
                $k[$j++] = $get($_.seta, $_.c); //#7380
            } else { //#7380
                if ($_.c >= 128) { //#7380
                    $k[$j++] = 101; //#7380
                } //#7380
                var _Aj = $_.c & 127; //#7380
                $k[$j++] = _Aj; //#7380
                if (_Aj >= 96) { //#7380
                    var _Ak = $k[--$j]; //#7380
                    $k[$j++] = 98; //#7380
                    $k[$j++] = _Ak; //#7380
                } //#7380
                $_.map_ab(); //#7380
            } //#7380
            var _Al = $a(); //#7380
            $k[$j++] = _Al; //#7380
        }; //#7380
        $_.enc_a1 = function() {
            $k[$j++] = Infinity; //#7381
            if ($_.c < 0) { //#7381
                $k[$j++] = $get($_.seta, $_.c); //#7381
            } else { //#7381
                if ($_.c < 128) { //#7381
                    $k[$j++] = 101; //#7381
                } //#7381
                var _As = $_.c & 127; //#7381
                $k[$j++] = _As; //#7381
                if (_As >= 96) { //#7381
                    var _At = $k[--$j]; //#7381
                    $k[$j++] = 98; //#7381
                    $k[$j++] = _At; //#7381
                } //#7381
                $_.map_ab(); //#7381
            } //#7381
            var _Au = $a(); //#7381
            $k[$j++] = _Au; //#7381
        }; //#7381
        $_.enc_b0 = function() {
            $k[$j++] = Infinity; //#7382
            if ($_.c < 0) { //#7382
                $k[$j++] = $get($_.setb, $_.c); //#7382
            } else { //#7382
                if ($_.c >= 128) { //#7382
                    $k[$j++] = 100; //#7382
                } //#7382
                var _B1 = $_.c & 127; //#7382
                $k[$j++] = _B1; //#7382
                if (_B1 < 32) { //#7382
                    var _B2 = $k[--$j]; //#7382
                    $k[$j++] = 98; //#7382
                    $k[$j++] = _B2; //#7382
                } //#7382
                $_.map_ab(); //#7382
            } //#7382
            var _B3 = $a(); //#7382
            $k[$j++] = _B3; //#7382
        }; //#7382
        $_.enc_b1 = function() {
            $k[$j++] = Infinity; //#7383
            if ($_.c < 0) { //#7383
                $k[$j++] = $get($_.setb, $_.c); //#7383
            } else { //#7383
                if ($_.c < 128) { //#7383
                    $k[$j++] = 100; //#7383
                } //#7383
                var _BA = $_.c & 127; //#7383
                $k[$j++] = _BA; //#7383
                if (_BA < 32) { //#7383
                    var _BB = $k[--$j]; //#7383
                    $k[$j++] = 98; //#7383
                    $k[$j++] = _BB; //#7383
                } //#7383
                $_.map_ab(); //#7383
            } //#7383
            var _BC = $a(); //#7383
            $k[$j++] = _BC; //#7383
        }; //#7383
        $_.enc_c = function() {
            $k[$j++] = Infinity; //#7384
            if ($_.c < 0) { //#7384
                $k[$j++] = $get($_.setc, $_.c); //#7384
            } else { //#7384
                $k[$j++] = $f((($get($_.msg, $_.n) - 48) * 10) + ($get($_.msg, $_.n + 1) - 48)); //#7384
            } //#7384
            var _BN = $a(); //#7384
            $k[$j++] = _BN; //#7384
        }; //#7384
        $_.get_best_prior_state = function() {
            var _BQ = $get($_.bln_0, $k[--$j]); //#7389
            $_.len = $f($k[--$j] + _BQ); //#7389
            $_.o = $k[--$j]; //#7389
            var _BV = $get($_.bln_0, $k[--$j]); //#7390
            var _BX = $f($k[--$j] + _BV); //#7390
            $k[$j++] = _BX; //#7390
            if (_BX < $_.len) { //#7390
                $_.len = $k[--$j]; //#7390
                $_.o = $k[--$j]; //#7390
            } else { //#7390
                $j -= 2; //#7390
            } //#7390
            var _Bd = $get($_.bln_0, $k[--$j]); //#7391
            var _Bf = $f($k[--$j] + _Bd); //#7391
            $k[$j++] = _Bf; //#7391
            if (_Bf < $_.len) { //#7391
                $_.len = $k[--$j]; //#7391
                $_.o = $k[--$j]; //#7391
            } else { //#7391
                $j -= 2; //#7391
            } //#7391
            var _Bl = $get($_.bln_0, $k[--$j]); //#7392
            var _Bn = $f($k[--$j] + _Bl); //#7392
            $k[$j++] = _Bn; //#7392
            if (_Bn < $_.len) { //#7392
                $_.len = $k[--$j]; //#7392
                $_.o = $k[--$j]; //#7392
            } else { //#7392
                $j -= 2; //#7392
            } //#7392
            var _Bt = $get($_.bln_0, $k[--$j]); //#7393
            var _Bv = $f($k[--$j] + _Bt); //#7393
            $k[$j++] = _Bv; //#7393
            if (_Bv < $_.len) { //#7393
                $_.len = $k[--$j]; //#7393
                $_.o = $k[--$j]; //#7393
            } else { //#7393
                $j -= 2; //#7393
            } //#7393
            var _C1 = $get($_.bln_0, $k[--$j]); //#7394
            if ($f($k[--$j] + _C1) < $_.len) { //#7394
                $_.o = $k[--$j]; //#7394
            } else { //#7394
                $j--; //#7394
            } //#7394
            $k[$j++] = $_.o; //#7395
        }; //#7395
        $_.bln_0 = $_.start_length; //#7401
        $_.bln_1 = $_.start_length; //#7401
        $k[$j++] = Infinity; //#7401
        for (var _C8 = 0; _C8 < 4; _C8++) { //#7401
            $k[$j++] = $a([0, 0, 0, 0, 0, 0]); //#7401
        } //#7401
        $_.bln = $a(); //#7401
        $_.bps_0 = $_.start_state; //#7402
        $_.bps_1 = $_.start_state; //#7402
        $k[$j++] = Infinity; //#7402
        for (var _CD = 0; _CD < 4; _CD++) { //#7402
            $k[$j++] = $a([0, 0, 0, 0, 0, 0]); //#7402
        } //#7402
        $_.bps = $a(); //#7402
        $k[$j++] = Infinity; //#7405
        for (var _CH = 0, _CI = $_.msg.length; _CH < _CI; _CH++) { //#7405
            $k[$j++] = $a([0, 0, 0, 0, 0, 0]); //#7405
        } //#7405
        $_.path = $a(); //#7405
        $_.make_tables = function() {
            $_.num_digits = 0; //#7408
            for (var _CN = 0, _CM = $_.msg.length - 1; _CN <= _CM; _CN += 1) { //#7439
                $_.n = _CN; //#7410
                $_.c = $get($_.msg, $_.n); //#7411
                $k[$j++] = 'num_digits'; //#7414
                if (($_.c >= 48) && ($_.c < 58)) { //#7414
                    $k[$j++] = $_.num_digits + 1; //#7414
                } else { //#7414
                    $k[$j++] = 0; //#7414
                } //#7414
                var _CU = $k[--$j]; //#7414
                $_[$k[--$j]] = _CU; //#7414
                $_.bln_2 = $_.bln_1; //#7417
                $_.bln_1 = $_.bln_0; //#7417
                $_.bln_0 = $get($_.bln, $_.n & 3); //#7417
                $_.bps_2 = $_.bps_1; //#7418
                $_.bps_1 = $_.bps_0; //#7418
                $_.bps_0 = $get($_.bps, $_.n & 3); //#7418
                $k[$j++] = 'bps_c'; //#7421
                if ($_.num_digits >= 2) { //#7421
                    $k[$j++] = $_.bps_2; //#7421
                } else { //#7421
                    $k[$j++] = $_.bps_1; //#7421
                } //#7421
                var _Cj = $k[--$j]; //#7421
                $_[$k[--$j]] = _Cj; //#7421
                $k[$j++] = 'bln_c'; //#7422
                if ($_.num_digits >= 2) { //#7422
                    $k[$j++] = $_.bln_2; //#7422
                } else { //#7422
                    $k[$j++] = $_.bln_1; //#7422
                } //#7422
                var _Co = $k[--$j]; //#7422
                $_[$k[--$j]] = _Co; //#7422
                $k[$j++] = $_.bln_0; //#7425
                $k[$j++] = 0; //#7425
                $_.can_a(); //#7425
                if ($k[--$j]) { //#7425
                    $_.p = $get($_.bps_1, 0); //#7425
                    $put($get($_.path, $_.n), 0, $_.p); //#7425
                    $k[$j++] = $f($get($_.bln_1, $_.p) + $get($_.latch_length_a0, $_.p)); //#7425
                    $_.out_a0(); //#7425
                    var _D4 = $k[--$j]; //#7425
                    var _D5 = $k[--$j]; //#7425
                    $k[$j++] = $f(_D5 + _D4); //#7425
                } else { //#7425
                    $k[$j++] = $_.max_int; //#7425
                } //#7425
                var _D7 = $k[--$j]; //#7425
                var _D8 = $k[--$j]; //#7425
                $put($k[--$j], _D8, _D7); //#7425
                $k[$j++] = $_.bln_0; //#7426
                $k[$j++] = 3; //#7426
                $_.can_a(); //#7426
                if ($k[--$j]) { //#7426
                    $_.p = $get($_.bps_1, 3); //#7426
                    $put($get($_.path, $_.n), 3, $_.p); //#7426
                    $k[$j++] = $f($get($_.bln_1, $_.p) + $get($_.latch_length_a1, $_.p)); //#7426
                    $_.out_a1(); //#7426
                    var _DO = $k[--$j]; //#7426
                    var _DP = $k[--$j]; //#7426
                    $k[$j++] = $f(_DP + _DO); //#7426
                } else { //#7426
                    $k[$j++] = $_.max_int; //#7426
                } //#7426
                var _DR = $k[--$j]; //#7426
                var _DS = $k[--$j]; //#7426
                $put($k[--$j], _DS, _DR); //#7426
                $k[$j++] = $_.bln_0; //#7427
                $k[$j++] = 1; //#7427
                $_.can_b(); //#7427
                if ($k[--$j]) { //#7427
                    $_.p = $get($_.bps_1, 1); //#7427
                    $put($get($_.path, $_.n), 1, $_.p); //#7427
                    $k[$j++] = $f($get($_.bln_1, $_.p) + $get($_.latch_length_b0, $_.p)); //#7427
                    $_.out_b0(); //#7427
                    var _Di = $k[--$j]; //#7427
                    var _Dj = $k[--$j]; //#7427
                    $k[$j++] = $f(_Dj + _Di); //#7427
                } else { //#7427
                    $k[$j++] = $_.max_int; //#7427
                } //#7427
                var _Dl = $k[--$j]; //#7427
                var _Dm = $k[--$j]; //#7427
                $put($k[--$j], _Dm, _Dl); //#7427
                $k[$j++] = $_.bln_0; //#7428
                $k[$j++] = 4; //#7428
                $_.can_b(); //#7428
                if ($k[--$j]) { //#7428
                    $_.p = $get($_.bps_1, 4); //#7428
                    $put($get($_.path, $_.n), 4, $_.p); //#7428
                    $k[$j++] = $f($get($_.bln_1, $_.p) + $get($_.latch_length_b1, $_.p)); //#7428
                    $_.out_b1(); //#7428
                    var _E2 = $k[--$j]; //#7428
                    var _E3 = $k[--$j]; //#7428
                    $k[$j++] = $f(_E3 + _E2); //#7428
                } else { //#7428
                    $k[$j++] = $_.max_int; //#7428
                } //#7428
                var _E5 = $k[--$j]; //#7428
                var _E6 = $k[--$j]; //#7428
                $put($k[--$j], _E6, _E5); //#7428
                $k[$j++] = $_.bln_0; //#7429
                $k[$j++] = 2; //#7429
                $_.can_c0(); //#7429
                if ($k[--$j]) { //#7429
                    $_.p = $get($_.bps_c, 2); //#7429
                    $put($get($_.path, $_.n), 2, $_.p); //#7429
                    $k[$j++] = $f($get($_.bln_c, $_.p) + $get($_.latch_length_c0, $_.p)) + 1; //#7429
                } else { //#7429
                    $k[$j++] = $_.max_int; //#7429
                } //#7429
                var _EN = $k[--$j]; //#7429
                var _EO = $k[--$j]; //#7429
                $put($k[--$j], _EO, _EN); //#7429
                $k[$j++] = $_.bln_0; //#7430
                $k[$j++] = 5; //#7430
                $_.can_c1(); //#7430
                if ($k[--$j]) { //#7430
                    $_.p = $get($_.bps_c, 5); //#7430
                    $put($get($_.path, $_.n), 5, $_.p); //#7430
                    $k[$j++] = $f($get($_.bln_c, $_.p) + $get($_.latch_length_c1, $_.p)) + 1; //#7430
                } else { //#7430
                    $k[$j++] = $_.max_int; //#7430
                } //#7430
                var _Ef = $k[--$j]; //#7430
                var _Eg = $k[--$j]; //#7430
                $put($k[--$j], _Eg, _Ef); //#7430
                $k[$j++] = $_.bps_0; //#7433
                $k[$j++] = 0; //#7433
                $aload($_.prioritized_latch_length_a0); //#7433
                $_.get_best_prior_state(); //#7433
                var _Ek = $k[--$j]; //#7433
                var _El = $k[--$j]; //#7433
                $put($k[--$j], _El, _Ek); //#7433
                $k[$j++] = $_.bps_0; //#7434
                $k[$j++] = 3; //#7434
                $aload($_.prioritized_latch_length_a1); //#7434
                $_.get_best_prior_state(); //#7434
                var _Ep = $k[--$j]; //#7434
                var _Eq = $k[--$j]; //#7434
                $put($k[--$j], _Eq, _Ep); //#7434
                $k[$j++] = $_.bps_0; //#7435
                $k[$j++] = 1; //#7435
                $aload($_.prioritized_latch_length_b0); //#7435
                $_.get_best_prior_state(); //#7435
                var _Eu = $k[--$j]; //#7435
                var _Ev = $k[--$j]; //#7435
                $put($k[--$j], _Ev, _Eu); //#7435
                $k[$j++] = $_.bps_0; //#7436
                $k[$j++] = 4; //#7436
                $aload($_.prioritized_latch_length_b1); //#7436
                $_.get_best_prior_state(); //#7436
                var _Ez = $k[--$j]; //#7436
                var _F0 = $k[--$j]; //#7436
                $put($k[--$j], _F0, _Ez); //#7436
                $k[$j++] = $_.bps_0; //#7437
                $k[$j++] = 2; //#7437
                $aload($_.prioritized_latch_length_c0); //#7437
                $_.get_best_prior_state(); //#7437
                var _F4 = $k[--$j]; //#7437
                var _F5 = $k[--$j]; //#7437
                $put($k[--$j], _F5, _F4); //#7437
                $k[$j++] = $_.bps_0; //#7438
                $k[$j++] = 5; //#7438
                $aload($_.prioritized_latch_length_c1); //#7438
                $_.get_best_prior_state(); //#7438
                var _F9 = $k[--$j]; //#7438
                var _FA = $k[--$j]; //#7438
                $put($k[--$j], _FA, _F9); //#7438
            } //#7438
        }; //#7438
        $_.backtrack = function() {
            $_.n = $_.msg.length; //#7443
            var _FD = $_.reverse_priority; //#7446
            for (var _FE = 0, _FF = _FD.length; _FE < _FF; _FE++) { //#7446
                var _FG = $get(_FD, _FE); //#7446
                var _FI = $get($_.bln_0, _FG); //#7446
                $k[$j++] = _FG; //#7446
                $k[$j++] = _FI; //#7446
                $k[$j++] = _FI; //#7446
            } //#7446
            $j--; //#7447
            $_.len = $k[--$j]; //#7447
            $_.state = $k[--$j]; //#7447
            for (var _FL = 0; _FL < 5; _FL++) { //#7448
                if ($lt($k[--$j], $_.len)) { //#7448
                    $_.len = $k[--$j]; //#7448
                    $_.state = $k[--$j]; //#7448
                } else { //#7448
                    $j -= 2; //#7448
                } //#7448
            } //#7448
            $k[$j++] = $a($_.len); //#7461
            for (;;) { //#7461
                if ($_.n <= 0) { //#7452
                    break; //#7452
                } //#7452
                $_.prior_state = $get($get($_.path, $_.n - 1), $_.state); //#7453
                $k[$j++] = Infinity; //#7454
                $forall($get($get($_.latch_sequence, $_.state), $_.prior_state)); //#7454
                $_.latch = $a(); //#7454
                var _Fk = ((($_.state == 2) || ($_.state == 5)) && ($get($_.msg, $_.n - 1) >= 48)) ? 2 : 1; //#7455
                $_.n = $_.n - _Fk; //#7455
                $_.c = $get($_.msg, $_.n); //#7456
                $k[$j++] = 'enc'; //#7457
                if ($get($_.encode, $_.state)() === true) {
                    break;
                } //#7457
                var _Fr = $k[--$j]; //#7457
                $_[$k[--$j]] = _Fr; //#7457
                $_.len = $f($_.len - $_.latch.length - $_.enc.length); //#7458
                var _Fw = $k[--$j]; //#7459
                var _Fx = $_.len; //#7459
                var _Fy = $_.latch; //#7459
                $puti(_Fw, _Fx, _Fy); //#7459
                $puti(_Fw, $f(_Fx + _Fy.length), $_.enc); //#7459
                $_.state = $_.prior_state; //#7460
                $k[$j++] = _Fw; //#7460
            } //#7460
            var _G1 = $k[$j - 1]; //#7462
            $put(_G1, 0, $get($_.start_code, $_.state)); //#7462
        }; //#7462
        $_.make_tables(); //#7465
        $_.backtrack(); //#7465
        $_.cws = $k[--$j]; //#7467
    } //#7467
    $_.j = $_.cws.length; //#7472
    var _G8 = $a($_.j + 2); //#7473
    $puti(_G8, 0, $_.cws); //#7473
    $_.cws = _G8; //#7473
    $_.csum = $get($_.cws, 0); //#7474
    for (var _GE = 1, _GD = $_.j - 1; _GE <= _GD; _GE += 1) { //#7478
        $_.i = _GE; //#7476
        $_.csum = $f($_.csum + ($get($_.cws, $_.i) * $_.i)); //#7477
    } //#7477
    $_.csum = $_.csum % 103; //#7479
    $put($_.cws, $_.j, $_.csum); //#7480
    $put($_.cws, $_.j + 1, 106); //#7481
    var _GR = $get($_.options, 'debugcws') !== undefined; //#7483
    if (_GR) { //#7483
        $k[$j++] = 'bwipp.debugcws#7483'; //#7483
        $k[$j++] = $_.cws; //#7483
        bwipp_raiseerror(); //#7483
    } //#7483
    $_.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "211214", "211232", "2331112"]); //#7504
    $_.sbs = $s(($_.cws.length * 6) + 1); //#7508
    for (var _GZ = 0, _GY = $_.cws.length - 1; _GZ <= _GY; _GZ += 1) { //#7512
        $_.i = _GZ; //#7510
        $puti($_.sbs, $_.i * 6, $get($_.encs, $get($_.cws, $_.i))); //#7511
    } //#7511
    $k[$j++] = Infinity; //#7515
    $k[$j++] = Infinity; //#7517
    var _Gh = $_.sbs; //#7517
    for (var _Gi = 0, _Gj = _Gh.length; _Gi < _Gj; _Gi++) { //#7517
        $k[$j++] = $get(_Gh, _Gi) - 48; //#7517
    } //#7517
    var _Gl = $a(); //#7517
    $k[$j++] = Infinity; //#7518
    for (var _Gn = 0, _Go = ~~(($_.sbs.length + 1) / 2); _Gn < _Go; _Gn++) { //#7518
        $k[$j++] = $_.height; //#7518
    } //#7518
    var _Gq = $a(); //#7518
    $k[$j++] = Infinity; //#7519
    for (var _Gs = 0, _Gt = ~~(($_.sbs.length + 1) / 2); _Gs < _Gt; _Gs++) { //#7519
        $k[$j++] = 0; //#7519
    } //#7519
    var _Gu = $a(); //#7519
    $k[$j++] = 'ren'; //#7522
    $k[$j++] = 'renlinear'; //#7522
    $k[$j++] = 'sbs'; //#7522
    $k[$j++] = _Gl; //#7522
    $k[$j++] = 'bhs'; //#7522
    $k[$j++] = _Gq; //#7522
    $k[$j++] = 'bbs'; //#7522
    $k[$j++] = _Gu; //#7522
    $k[$j++] = 'txt'; //#7522
    $k[$j++] = $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])]); //#7522
    $k[$j++] = 'textxalign'; //#7522
    $k[$j++] = "center"; //#7522
    $k[$j++] = 'opt'; //#7522
    $k[$j++] = $_.options; //#7522
    var _H3 = $d(); //#7522
    $k[$j++] = _H3; //#7525
    if (!$_.dontdraw) { //#7525
        bwipp_renlinear(); //#7525
    } //#7525
    $_ = $__; //#7529
} //bwipp_code128
function bwipp_gs1_128() {
    var $__ = $_; //#7560
    $_ = Object.assign({}, $_); //#7560
    $_.dontdraw = false; //#7563
    $_.includetext = false; //#7564
    $_.textfont = "OCR-B"; //#7565
    $_.textsize = 10; //#7566
    $_.textxoffset = 0; //#7567
    $_.textyoffset = -8; //#7568
    $_.height = 0.5; //#7569
    $_.linkagea = false; //#7570
    $_.linkagec = false; //#7571
    $_.parse = false; //#7572
    $_.dontlint = false; //#7573
    $_.lintreqs = false; //#7574
    $k[$j++] = $_; //#7576
    bwipp_processoptions(); //#7576
    $_.options = $k[--$j]; //#7576
    $_.barcode = $k[--$j]; //#7577
    if ($_.linkagea && $_.linkagec) { //#7581
        $k[$j++] = 'bwipp.gs1128badLinkage#7580'; //#7580
        $k[$j++] = "linkagea and linkagec cannot be used together"; //#7580
        bwipp_raiseerror(); //#7580
    } //#7580
    $_.text = $_.barcode; //#7583
    $k[$j++] = $_.barcode; //#7585
    $k[$j++] = 'ai'; //#7585
    bwipp_gs1process(); //#7585
    $_.fncs = $k[--$j]; //#7586
    $_.vals = $k[--$j]; //#7587
    $_.ais = $k[--$j]; //#7588
    $_.fnc1 = -1; //#7591
    $_.c128 = $a([$_.fnc1]); //#7592
    for (var _D = 0, _C = $_.ais.length - 1; _D <= _C; _D += 1) { //#7608
        $_.i = _D; //#7594
        $_.ai = $get($_.ais, $_.i); //#7595
        $_.val = $get($_.vals, $_.i); //#7596
        var _N = $a(($_.c128.length + $_.ai.length) + $_.val.length); //#7597
        $puti(_N, 0, $_.c128); //#7598
        $k[$j++] = _N; //#7599
        $k[$j++] = _N; //#7599
        $k[$j++] = $_.c128.length; //#7599
        var _R = Infinity; //#7599
        var _S = $_.ai; //#7599
        $k[$j++] = _R; //#7599
        $forall(_S); //#7599
        var _T = $a(); //#7599
        var _U = $k[--$j]; //#7599
        $puti($k[--$j], _U, _T); //#7599
        var _W = $k[$j - 1]; //#7600
        $k[$j++] = _W; //#7600
        $k[$j++] = $_.c128.length + $_.ai.length; //#7600
        var _a = Infinity; //#7600
        var _b = $_.val; //#7600
        $k[$j++] = _a; //#7600
        $forall(_b); //#7600
        var _c = $a(); //#7600
        var _d = $k[--$j]; //#7600
        $puti($k[--$j], _d, _c); //#7600
        $_.c128 = $k[--$j]; //#7601
        if (($_.i != ($_.ais.length - 1)) && $get($_.fncs, $_.i)) { //#7607
            var _m = $a($_.c128.length + 1); //#7603
            $puti(_m, 0, $_.c128); //#7604
            $put(_m, $_.c128.length, $_.fnc1); //#7605
            $_.c128 = _m; //#7606
        } //#7606
    } //#7606
    $_.barcode = $s(($_.c128.length + 1) * 5); //#7611
    $_.i = 0; //#7612
    $_.j = 0; //#7612
    for (;;) { //#7622
        if ($_.i == $_.c128.length) { //#7613
            break; //#7613
        } //#7613
        var _w = $get($_.c128, $_.i); //#7614
        $k[$j++] = _w; //#7619
        if (_w == $_.fnc1) { //#7618
            $j--; //#7615
            $puti($_.barcode, $_.j, "^FNC1"); //#7615
            $_.j = $_.j + 4; //#7616
        } else { //#7618
            $put($_.barcode, $_.j, $k[--$j]); //#7618
        } //#7618
        $_.i = $_.i + 1; //#7620
        $_.j = $_.j + 1; //#7621
    } //#7621
    if ($_.linkagea || $_.linkagec) { //#7626
        $k[$j++] = $_.barcode; //#7624
        $k[$j++] = $_.j; //#7624
        if ($_.linkagea) { //#7624
            $k[$j++] = "^LNKA"; //#7624
        } else { //#7624
            $k[$j++] = "^LNKC"; //#7624
        } //#7624
        var _1B = $k[--$j]; //#7624
        var _1C = $k[--$j]; //#7624
        $puti($k[--$j], _1C, _1B); //#7624
        $_.j = $_.j + 5; //#7625
    } //#7625
    $_.barcode = $geti($_.barcode, 0, $_.j); //#7627
    delete $_.options["parse"]; //#7630
    $put($_.options, "height", $_.height); //#7631
    $put($_.options, "dontdraw", true); //#7632
    $put($_.options, "parsefnc", true); //#7633
    $k[$j++] = 'args'; //#7634
    $k[$j++] = $_.barcode; //#7634
    $k[$j++] = $_.options; //#7634
    bwipp_code128(); //#7634
    var _1P = $k[--$j]; //#7634
    $_[$k[--$j]] = _1P; //#7634
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])])); //#7636
    $put($_.args, "textxalign", "center"); //#7637
    $put($_.args, "opt", $_.options); //#7638
    $k[$j++] = $_.args; //#7641
    if (!$_.dontdraw) { //#7641
        bwipp_renlinear(); //#7641
    } //#7641
    $_ = $__; //#7643
} //bwipp_gs1_128
function bwipp_ean14() {
    var $__ = $_; //#7672
    $_ = Object.assign({}, $_); //#7672
    $_.dontdraw = false; //#7675
    $_.includetext = false; //#7676
    $_.textfont = "OCR-B"; //#7677
    $_.textsize = 10; //#7678
    $_.textxoffset = 0; //#7679
    $_.textyoffset = -8; //#7680
    $_.height = 1; //#7681
    $k[$j++] = $_; //#7683
    bwipp_processoptions(); //#7683
    $_.options = $k[--$j]; //#7683
    $_.barcode = $k[--$j]; //#7684
    $_.text = $_.barcode; //#7687
    var _3 = $_.barcode; //#7688
    $k[$j++] = $s(_3.length); //#7689
    $k[$j++] = 0; //#7689
    $forall(_3, function() { //#7689
        var _5 = $k[$j - 1]; //#7689
        if (_5 != 32) { //#7689
            var _6 = $k[--$j]; //#7689
            var _7 = $k[--$j]; //#7689
            var _8 = $k[$j - 1]; //#7689
            $put(_8, _7, _6); //#7689
            $k[$j++] = _7 + 1; //#7689
        } else { //#7689
            $j--; //#7689
        } //#7689
    }); //#7689
    var _9 = $k[--$j]; //#7690
    $_.barcode = $geti($k[--$j], 0, _9); //#7690
    $_.hasspace = $_.text.length != $_.barcode.length; //#7691
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#7696
        $k[$j++] = 'bwipp.ean14badLength#7695'; //#7695
        $k[$j++] = "EAN-14 must be 13 or 14 digits"; //#7695
        bwipp_raiseerror(); //#7695
    } //#7695
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#7699
        $k[$j++] = 'bwipp.ean14badAI#7698'; //#7698
        $k[$j++] = "EAN-14 must begin with (01) application identifier"; //#7698
        bwipp_raiseerror(); //#7698
    } //#7698
    var _K = $geti($_.barcode, 4, $_.barcode.length - 4); //#7700
    for (var _L = 0, _M = _K.length; _L < _M; _L++) { //#7704
        var _N = $get(_K, _L); //#7704
        if ((_N < 48) || (_N > 57)) { //#7703
            $k[$j++] = 'bwipp.ean14badCharacter#7702'; //#7702
            $k[$j++] = "EAN-14 must contain only digits"; //#7702
            bwipp_raiseerror(); //#7702
        } //#7702
    } //#7702
    $_.checksum = 0; //#7707
    for (var _O = 0; _O <= 12; _O += 1) { //#7711
        $_.i = _O; //#7709
        $k[$j++] = 'checksum'; //#7710
        $k[$j++] = $_.checksum; //#7710
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#7710
        if (($_.i % 2) == 0) { //#7710
            var _U = $k[--$j]; //#7710
            $k[$j++] = _U * 3; //#7710
        } //#7710
        var _V = $k[--$j]; //#7710
        var _W = $k[--$j]; //#7710
        $_[$k[--$j]] = $f(_W + _V); //#7710
    } //#7710
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#7712
    if ($_.barcode.length == 18) { //#7726
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#7716
            $k[$j++] = 'bwipp.ean14badCheckDigit#7715'; //#7715
            $k[$j++] = "Incorrect EAN-14 check digit provided"; //#7715
            bwipp_raiseerror(); //#7715
        } //#7715
    } else { //#7726
        var _d = $s(18); //#7718
        $puti(_d, 0, $_.barcode); //#7719
        $put(_d, 17, $_.checksum + 48); //#7720
        $_.barcode = _d; //#7721
        var _i = $_.hasspace ? 2 : 1; //#7722
        var _j = $s($_.text.length + _i); //#7722
        $puti(_j, _j.length - 2, " "); //#7723
        $put(_j, _j.length - 1, $_.checksum + 48); //#7724
        $puti(_j, 0, $_.text); //#7725
        $_.text = _j; //#7726
    } //#7726
    $_.gtin = $geti($_.barcode, 4, 14); //#7728
    $_.barcode = $s(21); //#7731
    $puti($_.barcode, 0, "^FNC101"); //#7732
    var _r = $_.gtin; //#7733
    $puti($_.barcode, 7, _r); //#7733
    $put($_.options, "dontdraw", true); //#7736
    $put($_.options, "parsefnc", true); //#7737
    $k[$j++] = 'args'; //#7738
    $k[$j++] = $_.barcode; //#7738
    $k[$j++] = $_.options; //#7738
    bwipp_code128(); //#7738
    var _w = $k[--$j]; //#7738
    $_[$k[--$j]] = _w; //#7738
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])])); //#7740
    $put($_.args, "textxalign", "center"); //#7741
    $put($_.args, "opt", $_.options); //#7742
    $k[$j++] = $_.args; //#7745
    if (!$_.dontdraw) { //#7745
        bwipp_renlinear(); //#7745
    } //#7745
    $_ = $__; //#7747
} //bwipp_ean14
function bwipp_sscc18() {
    var $__ = $_; //#7776
    $_ = Object.assign({}, $_); //#7776
    $_.dontdraw = false; //#7779
    $_.includetext = false; //#7780
    $_.textfont = "OCR-B"; //#7781
    $_.textsize = 10; //#7782
    $_.textxoffset = 0; //#7783
    $_.textyoffset = -8; //#7784
    $_.height = 1; //#7785
    $k[$j++] = $_; //#7787
    bwipp_processoptions(); //#7787
    $_.options = $k[--$j]; //#7787
    $_.barcode = $k[--$j]; //#7788
    $_.text = $_.barcode; //#7791
    var _3 = $_.barcode; //#7792
    $k[$j++] = $s(_3.length); //#7793
    $k[$j++] = 0; //#7793
    $forall(_3, function() { //#7793
        var _5 = $k[$j - 1]; //#7793
        if (_5 != 32) { //#7793
            var _6 = $k[--$j]; //#7793
            var _7 = $k[--$j]; //#7793
            var _8 = $k[$j - 1]; //#7793
            $put(_8, _7, _6); //#7793
            $k[$j++] = _7 + 1; //#7793
        } else { //#7793
            $j--; //#7793
        } //#7793
    }); //#7793
    var _9 = $k[--$j]; //#7794
    $_.barcode = $geti($k[--$j], 0, _9); //#7794
    $_.hasspace = $_.text.length != $_.barcode.length; //#7795
    if (($_.barcode.length != 21) && ($_.barcode.length != 22)) { //#7800
        $k[$j++] = 'bwipp.sscc18badLength#7799'; //#7799
        $k[$j++] = "SSCC-18 must be 17 or 18 digits"; //#7799
        bwipp_raiseerror(); //#7799
    } //#7799
    if ($ne($geti($_.barcode, 0, 4), "\(00\)")) { //#7803
        $k[$j++] = 'bwipp.sscc18badAI#7802'; //#7802
        $k[$j++] = "SSCC-18 must begin with (00) application identifier"; //#7802
        bwipp_raiseerror(); //#7802
    } //#7802
    var _K = $geti($_.barcode, 4, $_.barcode.length - 4); //#7804
    for (var _L = 0, _M = _K.length; _L < _M; _L++) { //#7808
        var _N = $get(_K, _L); //#7808
        if ((_N < 48) || (_N > 57)) { //#7807
            $k[$j++] = 'bwipp.sscc18badCharacter#7806'; //#7806
            $k[$j++] = "SSCC-18 must contain only digits"; //#7806
            bwipp_raiseerror(); //#7806
        } //#7806
    } //#7806
    $_.checksum = 0; //#7811
    for (var _O = 0; _O <= 16; _O += 1) { //#7815
        $_.i = _O; //#7813
        $k[$j++] = 'checksum'; //#7814
        $k[$j++] = $_.checksum; //#7814
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#7814
        if (($_.i % 2) == 0) { //#7814
            var _U = $k[--$j]; //#7814
            $k[$j++] = _U * 3; //#7814
        } //#7814
        var _V = $k[--$j]; //#7814
        var _W = $k[--$j]; //#7814
        $_[$k[--$j]] = $f(_W + _V); //#7814
    } //#7814
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#7816
    if ($_.barcode.length == 22) { //#7830
        if ($get($_.barcode, 21) != ($_.checksum + 48)) { //#7820
            $k[$j++] = 'bwipp.sscc18badCheckDigit#7819'; //#7819
            $k[$j++] = "Incorrect SSCC-18 check digit provided"; //#7819
            bwipp_raiseerror(); //#7819
        } //#7819
    } else { //#7830
        var _d = $s(22); //#7822
        $puti(_d, 0, $_.barcode); //#7823
        $put(_d, 21, $_.checksum + 48); //#7824
        $_.barcode = _d; //#7825
        var _i = $_.hasspace ? 2 : 1; //#7826
        var _j = $s($_.text.length + _i); //#7826
        $puti(_j, _j.length - 2, " "); //#7827
        $put(_j, _j.length - 1, $_.checksum + 48); //#7828
        $puti(_j, 0, $_.text); //#7829
        $_.text = _j; //#7830
    } //#7830
    $_.sscc = $geti($_.barcode, 4, 18); //#7832
    $_.barcode = $s(25); //#7835
    $puti($_.barcode, 0, "^FNC100"); //#7836
    var _r = $_.sscc; //#7837
    $puti($_.barcode, 7, _r); //#7837
    $put($_.options, "dontdraw", true); //#7840
    $put($_.options, "parsefnc", true); //#7841
    $k[$j++] = 'args'; //#7842
    $k[$j++] = $_.barcode; //#7842
    $k[$j++] = $_.options; //#7842
    bwipp_code128(); //#7842
    var _w = $k[--$j]; //#7842
    $_[$k[--$j]] = _w; //#7842
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])])); //#7844
    $put($_.args, "textxalign", "center"); //#7845
    $put($_.args, "opt", $_.options); //#7846
    $k[$j++] = $_.args; //#7849
    if (!$_.dontdraw) { //#7849
        bwipp_renlinear(); //#7849
    } //#7849
    $_ = $__; //#7851
} //bwipp_sscc18
function bwipp_code39() {
    var $__ = $_; //#7881
    $_ = Object.assign({}, $_); //#7881
    $_.dontdraw = false; //#7884
    $_.includecheck = false; //#7885
    $_.validatecheck = false; //#7886
    $_.includetext = false; //#7887
    $_.includecheckintext = false; //#7888
    $_.hidestars = false; //#7889
    $_.textfont = "OCR-B"; //#7890
    $_.textsize = 10; //#7891
    $_.textyoffset = -8; //#7892
    $_.height = 1; //#7893
    $k[$j++] = $_; //#7895
    bwipp_processoptions(); //#7895
    $_.options = $k[--$j]; //#7895
    $_.barcode = $k[--$j]; //#7896
    if ($_.barcode.length == 0) { //#7900
        $k[$j++] = 'bwipp.code39emptyData#7899'; //#7899
        $k[$j++] = "The data must not be empty"; //#7899
        bwipp_raiseerror(); //#7899
    } //#7899
    $_.encs = $a(["1113313111", "3113111131", "1133111131", "3133111111", "1113311131", "3113311111", "1133311111", "1113113131", "3113113111", "1133113111", "3111131131", "1131131131", "3131131111", "1111331131", "3111331111", "1131331111", "1111133131", "3111133111", "1131133111", "1111333111", "3111111331", "1131111331", "3131111311", "1111311331", "3111311311", "1131311311", "1111113331", "3111113311", "1131113311", "1111313311", "3311111131", "1331111131", "3331111111", "1311311131", "3311311111", "1331311111", "1311113131", "3311113111", "1331113111", "1313131111", "1313111311", "1311131311", "1113131311", "1311313111"]); //#7916
    $_.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%"; //#7919
    $_.charvals = new Map; //#7920
    for (var _4 = 0; _4 <= 42; _4 += 1) { //#7921
        $put($_.charvals, $geti($_.barchars, _4, 1), _4); //#7921
    } //#7921
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#7929
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#7926
        if (!_F) { //#7928
            $k[$j++] = 'bwipp.code39badCharacter#7927'; //#7927
            $k[$j++] = "Code 39 must contain only digits, capital letters, spaces and the symbols -.$/+%"; //#7927
            bwipp_raiseerror(); //#7927
        } //#7927
    } //#7927
    $k[$j++] = 'barlen'; //#7931
    $k[$j++] = $_.barcode.length; //#7931
    if ($_.validatecheck) { //#7931
        var _I = $k[--$j]; //#7931
        $k[$j++] = _I - 1; //#7931
    } //#7931
    var _J = $k[--$j]; //#7931
    $_[$k[--$j]] = _J; //#7931
    $_.checksum = 0; //#7933
    for (var _N = 0, _M = $_.barlen - 1; _N <= _M; _N += 1) { //#7937
        $_.checksum = $f($get($_.charvals, $geti($_.barcode, _N, 1)) + $_.checksum); //#7936
    } //#7936
    $_.checksum = $_.checksum % 43; //#7938
    if ($_.validatecheck) { //#7945
        if ($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum)) { //#7942
            $k[$j++] = 'bwipp.code39badCheckDigit#7941'; //#7941
            $k[$j++] = "Incorrect Code 39 check digit provided"; //#7941
            bwipp_raiseerror(); //#7941
        } //#7941
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#7943
        $_.includecheck = true; //#7944
    } //#7944
    var _g = $_.includecheck ? 3 : 2; //#7947
    $_.sbs = $s($f($_.barlen + _g) * 10); //#7947
    var _k = $_.includecheck ? 3 : 2; //#7948
    $_.txt = $a($f($_.barlen + _k)); //#7948
    $puti($_.sbs, 0, $get($_.encs, 43)); //#7951
    if (!$_.hidestars) { //#7955
        $put($_.txt, 0, $a(["*", 0, $_.textyoffset, $_.textfont, $_.textsize])); //#7953
    } else { //#7955
        $put($_.txt, 0, $a(["", 0, $_.textyoffset, $_.textfont, $_.textsize])); //#7955
    } //#7955
    for (var _12 = 0, _11 = $_.barlen - 1; _12 <= _11; _12 += 1) { //#7963
        $_.i = _12; //#7959
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#7960
        $puti($_.sbs, ($_.i * 10) + 10, $get($_.encs, $_.indx)); //#7961
        $put($_.txt, $_.i + 1, $a([$geti($_.barcode, $_.i, 1), ($_.i + 1) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7962
    } //#7962
    if ($_.includecheck) { //#7984
        $puti($_.sbs, ($_.barlen * 10) + 10, $get($_.encs, $_.checksum)); //#7967
        if ($_.includecheckintext) { //#7971
            $put($_.txt, $_.barlen + 1, $a([$geti($_.barchars, $_.checksum, 1), ($_.barlen + 1) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7969
        } else { //#7971
            $put($_.txt, $_.barlen + 1, $a(["", ($_.barlen + 1) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7971
        } //#7971
        $puti($_.sbs, ($_.barlen * 10) + 20, $get($_.encs, 43)); //#7973
        if (!$_.hidestars) { //#7977
            $put($_.txt, $_.barlen + 2, $a(["*", ($_.barlen + 2) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7975
        } else { //#7977
            $put($_.txt, $_.barlen + 2, $a(["", ($_.barlen + 2) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7977
        } //#7977
    } else { //#7984
        $puti($_.sbs, ($_.barlen * 10) + 10, $get($_.encs, 43)); //#7980
        if (!$_.hidestars) { //#7984
            $put($_.txt, $_.barlen + 1, $a(["*", ($_.barlen + 1) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7982
        } else { //#7984
            $put($_.txt, $_.barlen + 1, $a(["", ($_.barlen + 1) * 16, $_.textyoffset, $_.textfont, $_.textsize])); //#7984
        } //#7984
    } //#7984
    $k[$j++] = Infinity; //#7989
    $k[$j++] = Infinity; //#7991
    var _2N = $_.sbs; //#7991
    for (var _2O = 0, _2P = _2N.length; _2O < _2P; _2O++) { //#7991
        $k[$j++] = $get(_2N, _2O) - 48; //#7991
    } //#7991
    var _2R = $a(); //#7991
    $k[$j++] = Infinity; //#7992
    for (var _2T = 0, _2U = ~~(($_.sbs.length + 1) / 2); _2T < _2U; _2T++) { //#7992
        $k[$j++] = $_.height; //#7992
    } //#7992
    var _2W = $a(); //#7992
    $k[$j++] = Infinity; //#7993
    for (var _2Y = 0, _2Z = ~~(($_.sbs.length + 1) / 2); _2Y < _2Z; _2Y++) { //#7993
        $k[$j++] = 0; //#7993
    } //#7993
    var _2a = $a(); //#7993
    $k[$j++] = 'ren'; //#7996
    $k[$j++] = 'renlinear'; //#7996
    $k[$j++] = 'sbs'; //#7996
    $k[$j++] = _2R; //#7996
    $k[$j++] = 'bhs'; //#7996
    $k[$j++] = _2W; //#7996
    $k[$j++] = 'bbs'; //#7996
    $k[$j++] = _2a; //#7996
    if ($_.includetext) { //#7996
        $k[$j++] = 'txt'; //#7995
        $k[$j++] = $_.txt; //#7995
    } //#7995
    $k[$j++] = 'opt'; //#7997
    $k[$j++] = $_.options; //#7997
    var _2e = $d(); //#7997
    $k[$j++] = _2e; //#8000
    if (!$_.dontdraw) { //#8000
        bwipp_renlinear(); //#8000
    } //#8000
    $_ = $__; //#8004
} //bwipp_code39
function bwipp_code39ext() {
    var $__ = $_; //#8036
    $_ = Object.assign({}, $_); //#8036
    $_.dontdraw = false; //#8039
    $_.includetext = false; //#8040
    $_.parse = false; //#8041
    $k[$j++] = $_; //#8043
    bwipp_processoptions(); //#8043
    $_.options = $k[--$j]; //#8043
    $_.barcode = $k[--$j]; //#8044
    var _3 = new Map([
        ["parse", $_.parse],
        ["parseonly", true],
        ["parsefnc", false]
    ]); //#8052
    $_.fncvals = _3; //#8053
    $k[$j++] = 'barcode'; //#8054
    $k[$j++] = $_.barcode; //#8054
    $k[$j++] = $_.fncvals; //#8054
    bwipp_parseinput(); //#8054
    var _6 = $k[--$j]; //#8054
    $_[$k[--$j]] = _6; //#8054
    $_.barlen = $_.barcode.length; //#8055
    delete $_.options["parse"]; //#8056
    $forall($_.barcode, function() { //#8063
        if ($k[--$j] >= 128) { //#8062
            $k[$j++] = 'bwipp.code39extBadCharacter#8061'; //#8061
            $k[$j++] = "Code 39 Extended must contain only ASCII characters"; //#8061
            bwipp_raiseerror(); //#8061
        } //#8061
    }); //#8061
    $_.extencs = $a(["%U", "$A", "$B", "$C", "$D", "$E", "$F", "$G", "$H", "$I", "$J", "$K", "$L", "$M", "$N", "$O", "$P", "$Q", "$R", "$S", "$T", "$U", "$V", "$W", "$X", "$Y", "$Z", "%A", "%B", "%C", "%D", "%E", " ", "/A", "/B", "/C", "/D", "/E", "/F", "/G", "/H", "/I", "/J", "/K", "/L", "-", ".", "/O", "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "/Z", "%F", "%G", "%H", "%I", "%J", "%V", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "%K", "%L", "%M", "%N", "%O", "%W", "+A", "+B", "+C", "+D", "+E", "+F", "+G", "+H", "+I", "+J", "+K", "+L", "+M", "+N", "+O", "+P", "+Q", "+R", "+S", "+T", "+U", "+V", "+W", "+X", "+Y", "+Z", "%P", "%Q", "%R", "%S", "%T"]); //#8076
    $_.newcode = $s($_.barlen * 2); //#8079
    $_.newtext = $s($_.barlen * 2); //#8080
    $_.j = 0; //#8081
    for (var _K = 0, _J = $_.barlen - 1; _K <= _J; _K += 1) { //#8089
        $_.i = _K; //#8083
        $_.extchar = $get($_.extencs, $get($_.barcode, $_.i)); //#8084
        $puti($_.newcode, $_.j, $_.extchar); //#8085
        var _X = $get($_.barcode, $_.i); //#8086
        $k[$j++] = $_.newtext; //#8086
        $k[$j++] = $_.j; //#8086
        $k[$j++] = _X; //#8086
        if ((_X < 32) || (_X == 127)) { //#8086
            $k[$j - 1] = 32; //#8086
        } //#8086
        var _Y = $k[--$j]; //#8086
        var _Z = $k[--$j]; //#8086
        $put($k[--$j], _Z, _Y); //#8086
        if ($_.extchar.length != 1) { //#8087
            $put($_.newtext, $_.j + 1, 32); //#8087
        } //#8087
        $_.j = $_.j + $_.extchar.length; //#8088
    } //#8088
    $_.newcode = $geti($_.newcode, 0, $_.j); //#8090
    $_.newtext = $geti($_.newtext, 0, $_.j); //#8091
    $put($_.options, "dontdraw", true); //#8094
    $k[$j++] = 'args'; //#8095
    $k[$j++] = $_.newcode; //#8095
    $k[$j++] = $_.options; //#8095
    bwipp_code39(); //#8095
    var _p = $k[--$j]; //#8095
    $_[$k[--$j]] = _p; //#8095
    if ($_.includetext) { //#8107
        $_.txt = $get($_.args, "txt"); //#8099
        for (var _w = 0, _v = $_.newtext.length - 1; _w <= _v; _w += 1) { //#8105
            $_.i = _w; //#8101
            $_.txtentry = $get($_.txt, $_.i + 1); //#8102
            $put($_.txtentry, 0, $geti($_.newtext, $_.i, 1)); //#8103
            $put($_.txt, $_.i + 1, $_.txtentry); //#8104
        } //#8104
        $put($_.args, "txt", $_.txt); //#8106
    } //#8106
    $put($_.args, "opt", $_.options); //#8108
    $k[$j++] = $_.args; //#8111
    if (!$_.dontdraw) { //#8111
        bwipp_renlinear(); //#8111
    } //#8111
    $_ = $__; //#8115
} //bwipp_code39ext
function bwipp_code32() {
    var $__ = $_; //#8144
    $_ = Object.assign({}, $_); //#8144
    $_.dontdraw = false; //#8147
    $_.includetext = false; //#8148
    $_.textfont = "OCR-B"; //#8149
    $_.textsize = 10; //#8150
    $_.textyoffset = -8; //#8151
    $_.textxoffset = 0; //#8152
    $_.height = 1; //#8153
    $k[$j++] = $_; //#8155
    bwipp_processoptions(); //#8155
    $_.options = $k[--$j]; //#8155
    $_.barcode = $k[--$j]; //#8156
    if (($_.barcode.length != 8) && ($_.barcode.length != 9)) { //#8161
        $k[$j++] = 'bwipp.code32badLength#8160'; //#8160
        $k[$j++] = "Italian Pharmacode must be 8 or 9 digits"; //#8160
        bwipp_raiseerror(); //#8160
    } //#8160
    $forall($_.barcode, function() { //#8166
        var _5 = $k[--$j]; //#8163
        if ((_5 < 48) || (_5 > 57)) { //#8165
            $k[$j++] = 'bwipp.code32badCharacter#8164'; //#8164
            $k[$j++] = "Italian Pharmacode must contain only digits"; //#8164
            bwipp_raiseerror(); //#8164
        } //#8164
    }); //#8164
    $_.checksum = 0; //#8169
    for (var _6 = 0; _6 <= 7; _6 += 1) { //#8174
        $_.i = _6; //#8171
        $k[$j++] = $get($_.barcode, $_.i) - 48; //#8172
        if (($_.i % 2) != 0) { //#8172
            var _B = $k[--$j]; //#8172
            $k[$j++] = _B * 2; //#8172
        } //#8172
        var _C = $k[$j - 1]; //#8172
        if (_C > 9) { //#8172
            var _D = $k[--$j]; //#8172
            $k[$j++] = _D - 9; //#8172
        } //#8172
        $_.checksum = $f($k[--$j] + $_.checksum); //#8173
    } //#8173
    $_.checksum = $_.checksum % 10; //#8175
    if ($_.barcode.length == 9) { //#8180
        if ($get($_.barcode, 8) != ($_.checksum + 48)) { //#8179
            $k[$j++] = 'bwipp.code32badCheckDigit#8178'; //#8178
            $k[$j++] = "Incorrect Italian Pharmacode check digit provided"; //#8178
            bwipp_raiseerror(); //#8178
        } //#8178
    } //#8178
    var _L = $s(10); //#8181
    $puti(_L, 0, $_.barcode); //#8182
    $put(_L, 8, $_.checksum + 48); //#8183
    $_.text = _L; //#8184
    $_.val = $cvrs($s(6), $cvi($_.text), 32); //#8187
    $_.barcode = $s(6); //#8188
    for (var _S = 0; _S <= 5; _S += 1) { //#8189
        $put($_.barcode, _S, 48); //#8189
    } //#8189
    $puti($_.barcode, 6 - $_.val.length, $_.val); //#8190
    for (var _X = 0; _X <= 5; _X += 1) { //#8197
        var _Z = $get($_.barcode, _X); //#8194
        $k[$j++] = _X; //#8195
        $k[$j++] = _Z; //#8195
        $k[$j++] = _Z; //#8195
        for (var _a = 0, _b = "AEIO".length; _a < _b; _a++) { //#8195
            if ($k[--$j] >= $get("AEIO", _a)) { //#8195
                var _e = $k[--$j]; //#8195
                $k[$j++] = _e + 1; //#8195
            } //#8195
            var _f = $k[$j - 1]; //#8195
            $k[$j++] = _f; //#8195
        } //#8195
        $k[$j - 1] = $_.barcode; //#8196
        var _h = $k[$j - 2]; //#8196
        var _i = $k[$j - 3]; //#8196
        $put($k[$j - 1], _i, _h); //#8196
        $j -= 3; //#8196
    } //#8196
    $puti($_.text, 1, $geti($_.text, 0, 9)); //#8200
    $put($_.text, 0, 65); //#8201
    $put($_.options, "dontdraw", true); //#8204
    $k[$j++] = 'args'; //#8205
    $k[$j++] = $_.barcode; //#8205
    $k[$j++] = $_.options; //#8205
    bwipp_code39(); //#8205
    var _r = $k[--$j]; //#8205
    $_[$k[--$j]] = _r; //#8205
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])])); //#8207
    $put($_.args, "textxalign", "center"); //#8208
    $put($_.args, "opt", $_.options); //#8209
    $k[$j++] = $_.args; //#8212
    if (!$_.dontdraw) { //#8212
        bwipp_renlinear(); //#8212
    } //#8212
    $_ = $__; //#8214
} //bwipp_code32
function bwipp_pzn() {
    var $__ = $_; //#8243
    $_ = Object.assign({}, $_); //#8243
    $_.dontdraw = false; //#8246
    $_.includetext = false; //#8247
    $_.textfont = "OCR-B"; //#8248
    $_.textsize = 10; //#8249
    $_.textyoffset = -8; //#8250
    $_.textxoffset = 0; //#8251
    $_.height = 1; //#8252
    $_.pzn8 = false; //#8253
    $k[$j++] = $_; //#8255
    bwipp_processoptions(); //#8255
    $_.options = $k[--$j]; //#8255
    $_.barcode = $k[--$j]; //#8256
    if ($_.pzn8) { //#8265
        if (($_.barcode.length != 7) && ($_.barcode.length != 8)) { //#8262
            $k[$j++] = 'bwipp.pzn8badLength#8261'; //#8261
            $k[$j++] = "PZN8 must be 7 or 8 digits"; //#8261
            bwipp_raiseerror(); //#8261
        } //#8261
    } else { //#8265
        if (($_.barcode.length != 6) && ($_.barcode.length != 7)) { //#8266
            $k[$j++] = 'bwipp.pzn7badLength#8265'; //#8265
            $k[$j++] = "PZN7 must be 6 or 7 digits"; //#8265
            bwipp_raiseerror(); //#8265
        } //#8265
    } //#8265
    $forall($_.barcode, function() { //#8272
        var _8 = $k[--$j]; //#8269
        if ((_8 < 48) || (_8 > 57)) { //#8271
            $k[$j++] = 'bwipp.pznBadCharacter#8270'; //#8270
            $k[$j++] = "PZN must contain only digits"; //#8270
            bwipp_raiseerror(); //#8270
        } //#8270
    }); //#8270
    var _A = $_.pzn8 ? 9 : 8; //#8275
    $_.msglen = _A; //#8275
    $_.checksum = 0; //#8276
    for (var _D = 0, _C = $_.msglen - 3; _D <= _C; _D += 1) { //#8281
        $_.i = _D; //#8278
        var _J = $_.pzn8 ? 1 : 2; //#8279
        $_.checksum = $f((($get($_.barcode, $_.i) - 48) * ($_.i + _J)) + $_.checksum); //#8280
    } //#8280
    $_.checksum = $_.checksum % 11; //#8282
    if ($_.checksum == 10) { //#8285
        $k[$j++] = 'bwipp.pznBadInputSequence#8284'; //#8284
        $k[$j++] = "Incorrect PZN input sequence provided"; //#8284
        bwipp_raiseerror(); //#8284
    } //#8284
    if ($_.barcode.length == ($_.msglen - 1)) { //#8290
        if ($get($_.barcode, $_.msglen - 2) != ($_.checksum + 48)) { //#8289
            $k[$j++] = 'bwipp.pznBadCheckDigit#8288'; //#8288
            $k[$j++] = "Incorrect PZN check digit provided"; //#8288
            bwipp_raiseerror(); //#8288
        } //#8288
    } //#8288
    var _U = $s($_.msglen); //#8291
    $put(_U, 0, 45); //#8292
    $puti(_U, 1, $_.barcode); //#8293
    $put(_U, $_.msglen - 1, $_.checksum + 48); //#8294
    $_.msg = _U; //#8295
    $put($_.options, "dontdraw", true); //#8298
    $k[$j++] = 'args'; //#8299
    $k[$j++] = $_.msg; //#8299
    $k[$j++] = $_.options; //#8299
    bwipp_code39(); //#8299
    var _b = $k[--$j]; //#8299
    $_[$k[--$j]] = _b; //#8299
    $_.text = $s($_.msglen + 5); //#8302
    $puti($_.text, 0, "PZN - "); //#8303
    $puti($_.text, 6, $geti($_.msg, 1, $_.msglen - 1)); //#8304
    var _r = $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])]); //#8306
    $put($_.args, "txt", _r); //#8306
    $put($_.args, "textxalign", "center"); //#8307
    $put($_.args, "opt", $_.options); //#8308
    $k[$j++] = $_.args; //#8311
    if (!$_.dontdraw) { //#8311
        bwipp_renlinear(); //#8311
    } //#8311
    $_ = $__; //#8313
} //bwipp_pzn
function bwipp_code93() {
    var $__ = $_; //#8344
    $_ = Object.assign({}, $_); //#8344
    $_.dontdraw = false; //#8347
    $_.includecheck = false; //#8348
    $_.includetext = false; //#8349
    $_.textfont = "OCR-B"; //#8350
    $_.textsize = 10; //#8351
    $_.textyoffset = -8; //#8352
    $_.height = 1; //#8353
    $_.parse = false; //#8354
    $_.parsefnc = false; //#8355
    $k[$j++] = $_; //#8357
    bwipp_processoptions(); //#8357
    $_.options = $k[--$j]; //#8357
    $_.barcode = $k[--$j]; //#8358
    $_.encs = $a(["131112", "111213", "111312", "111411", "121113", "121212", "121311", "111114", "131211", "141111", "211113", "211212", "211311", "221112", "221211", "231111", "112113", "112212", "112311", "122112", "132111", "111123", "111222", "111321", "121122", "131121", "212112", "212211", "211122", "211221", "221121", "222111", "112122", "112221", "122121", "123111", "121131", "311112", "311211", "321111", "112131", "113121", "211131", "121221", "312111", "311121", "122211", "111141", "1111411"]); //#8374
    $_.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%"; //#8377
    $_.charvals = new Map; //#8378
    for (var _3 = 0; _3 <= 42; _3 += 1) { //#8379
        $put($_.charvals, $get($_.barchars, _3), _3); //#8379
    } //#8379
    $_.sft1 = -1; //#8381
    $_.sft2 = -2; //#8381
    $_.sft3 = -3; //#8381
    $_.sft4 = -4; //#8381
    $put($_.charvals, $_.sft1, 43); //#8383
    $put($_.charvals, $_.sft2, 44); //#8384
    $put($_.charvals, $_.sft3, 45); //#8385
    $put($_.charvals, $_.sft4, 46); //#8386
    var _M = new Map([
        ["parse", $_.parse],
        ["parsefnc", $_.parsefnc],
        ["SFT$", $_.sft1],
        ["SFT%", $_.sft2],
        ["SFT/", $_.sft3],
        ["SFT+", $_.sft4]
    ]); //#8397
    $_.fncvals = _M; //#8398
    $k[$j++] = 'msg'; //#8399
    $k[$j++] = $_.barcode; //#8399
    $k[$j++] = $_.fncvals; //#8399
    bwipp_parseinput(); //#8399
    var _P = $k[--$j]; //#8399
    $_[$k[--$j]] = _P; //#8399
    $_.msglen = $_.msg.length; //#8400
    $forall($_.msg, function() { //#8407
        var _V = $get($_.charvals, $k[--$j]) !== undefined; //#8404
        if (!_V) { //#8406
            $k[$j++] = 'bwipp.code93BadCharacter#8405'; //#8405
            $k[$j++] = "Code 93 must only contain digits, capital letters, spaces, the symbols -.$/+ and shift non-data characters"; //#8405
            bwipp_raiseerror(); //#8405
        } //#8405
    }); //#8405
    if ($_.includecheck) { //#8412
        $_.sbs = $s(($_.msglen * 6) + 25); //#8410
    } else { //#8412
        $_.sbs = $s(($_.msglen * 6) + 13); //#8412
    } //#8412
    $_.txt = $a($_.msglen); //#8414
    $puti($_.sbs, 0, $get($_.encs, 47)); //#8417
    $_.checksum1 = 0; //#8419
    $_.checksum2 = 0; //#8419
    for (var _i = 0, _h = $_.msglen - 1; _i <= _h; _i += 1) { //#8428
        $_.i = _i; //#8421
        $_.indx = $get($_.charvals, $get($_.msg, $_.i)); //#8422
        $puti($_.sbs, ($_.i * 6) + 6, $get($_.encs, $_.indx)); //#8423
        var _v = $get($_.msg, $_.i); //#8424
        $k[$j++] = 'char'; //#8424
        $k[$j++] = _v; //#8424
        if (_v < 0) { //#8424
            $k[$j - 1] = " "; //#8424
        } else { //#8424
            var _w = $s(1); //#8424
            $k[$j++] = _w; //#8424
            $k[$j++] = _w; //#8424
            $k[$j++] = 0; //#8424
            $r(4, -1); //#8424
            var _x = $k[--$j]; //#8424
            var _y = $k[--$j]; //#8424
            $put($k[--$j], _y, _x); //#8424
        } //#8424
        var _10 = $k[--$j]; //#8424
        $_[$k[--$j]] = _10; //#8424
        $put($_.txt, $_.i, $a([$_.char, ($_.i * 9) + 9, $_.textyoffset, $_.textfont, $_.textsize])); //#8425
        $_.checksum1 = $f($_.checksum1 + ((((($_.msglen - $_.i) - 1) % 20) + 1) * $_.indx)); //#8426
        $_.checksum2 = $f($_.checksum2 + (((($_.msglen - $_.i) % 15) + 1) * $_.indx)); //#8427
    } //#8427
    if ($_.includecheck) { //#8440
        $_.checksum1 = $_.checksum1 % 47; //#8432
        $_.checksum2 = $f($_.checksum2 + $_.checksum1) % 47; //#8433
        $puti($_.sbs, ($_.msglen * 6) + 6, $get($_.encs, $_.checksum1)); //#8434
        $puti($_.sbs, ($_.msglen * 6) + 12, $get($_.encs, $_.checksum2)); //#8435
        $puti($_.sbs, ($_.msglen * 6) + 18, $get($_.encs, 48)); //#8437
    } else { //#8440
        $puti($_.sbs, ($_.msglen * 6) + 6, $get($_.encs, 48)); //#8440
    } //#8440
    $k[$j++] = Infinity; //#8444
    $k[$j++] = Infinity; //#8446
    var _1e = $_.sbs; //#8446
    for (var _1f = 0, _1g = _1e.length; _1f < _1g; _1f++) { //#8446
        $k[$j++] = $get(_1e, _1f) - 48; //#8446
    } //#8446
    var _1i = $a(); //#8446
    $k[$j++] = Infinity; //#8447
    for (var _1k = 0, _1l = ~~(($_.sbs.length + 1) / 2); _1k < _1l; _1k++) { //#8447
        $k[$j++] = $_.height; //#8447
    } //#8447
    var _1n = $a(); //#8447
    $k[$j++] = Infinity; //#8448
    for (var _1p = 0, _1q = ~~(($_.sbs.length + 1) / 2); _1p < _1q; _1p++) { //#8448
        $k[$j++] = 0; //#8448
    } //#8448
    var _1r = $a(); //#8448
    $k[$j++] = 'ren'; //#8451
    $k[$j++] = 'renlinear'; //#8451
    $k[$j++] = 'sbs'; //#8451
    $k[$j++] = _1i; //#8451
    $k[$j++] = 'bhs'; //#8451
    $k[$j++] = _1n; //#8451
    $k[$j++] = 'bbs'; //#8451
    $k[$j++] = _1r; //#8451
    if ($_.includetext) { //#8451
        $k[$j++] = 'txt'; //#8450
        $k[$j++] = $_.txt; //#8450
    } //#8450
    $k[$j++] = 'opt'; //#8452
    $k[$j++] = $_.options; //#8452
    var _1v = $d(); //#8452
    $k[$j++] = _1v; //#8455
    if (!$_.dontdraw) { //#8455
        bwipp_renlinear(); //#8455
    } //#8455
    $_ = $__; //#8459
} //bwipp_code93
function bwipp_code93ext() {
    var $__ = $_; //#8491
    $_ = Object.assign({}, $_); //#8491
    $_.dontdraw = false; //#8494
    $_.includetext = false; //#8495
    $_.parse = false; //#8496
    $k[$j++] = $_; //#8498
    bwipp_processoptions(); //#8498
    $_.options = $k[--$j]; //#8498
    $_.barcode = $k[--$j]; //#8499
    var _3 = new Map([
        ["parse", $_.parse],
        ["parseonly", true],
        ["parsefnc", false]
    ]); //#8507
    $_.fncvals = _3; //#8508
    $k[$j++] = 'barcode'; //#8509
    $k[$j++] = $_.barcode; //#8509
    $k[$j++] = $_.fncvals; //#8509
    bwipp_parseinput(); //#8509
    var _6 = $k[--$j]; //#8509
    $_[$k[--$j]] = _6; //#8509
    $_.barlen = $_.barcode.length; //#8510
    delete $_.options["parse"]; //#8511
    $_.extencs = $a(["^SFT%U", "^SFT$A", "^SFT$B", "^SFT$C", "^SFT$D", "^SFT$E", "^SFT$F", "^SFT$G", "^SFT$H", "^SFT$I", "^SFT$J", "^SFT$K", "^SFT$L", "^SFT$M", "^SFT$N", "^SFT$O", "^SFT$P", "^SFT$Q", "^SFT$R", "^SFT$S", "^SFT$T", "^SFT$U", "^SFT$V", "^SFT$W", "^SFT$X", "^SFT$Y", "^SFT$Z", "^SFT%A", "^SFT%B", "^SFT%C", "^SFT%D", "^SFT%E", " ", "^SFT/A", "^SFT/B", "^SFT/C", "$", "%", "^SFT/F", "^SFT/G", "^SFT/H", "^SFT/I", "^SFT/J", "+", "^SFT/L", "-", ".", "/", "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "^SFT/Z", "^SFT%F", "^SFT%G", "^SFT%H", "^SFT%I", "^SFT%J", "^SFT%V", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "^SFT%K", "^SFT%L", "^SFT%M", "^SFT%N", "^SFT%O", "^SFT%W", "^SFT+A", "^SFT+B", "^SFT+C", "^SFT+D", "^SFT+E", "^SFT+F", "^SFT+G", "^SFT+H", "^SFT+I", "^SFT+J", "^SFT+K", "^SFT+L", "^SFT+M", "^SFT+N", "^SFT+O", "^SFT+P", "^SFT+Q", "^SFT+R", "^SFT+S", "^SFT+T", "^SFT+U", "^SFT+V", "^SFT+W", "^SFT+X", "^SFT+Y", "^SFT+Z", "^SFT%P", "^SFT%Q", "^SFT%R", "^SFT%S", "^SFT%T"]); //#8532
    $_.newcode = $s($_.barlen * 6); //#8535
    $_.newtext = $s($_.barlen * 6); //#8536
    $_.j = 0; //#8537
    $_.k = 0; //#8537
    for (var _I = 0, _H = $_.barlen - 1; _I <= _H; _I += 1) { //#8551
        $_.i = _I; //#8539
        var _L = $get($_.barcode, $_.i); //#8540
        $k[$j++] = _L; //#8543
        if (_L > 127) { //#8543
            $j--; //#8542
            $k[$j++] = 'bwipp.code93extbadCharacter#8542'; //#8542
            $k[$j++] = "Code 93 Extended only support ASCII characters with values 0 to 127"; //#8542
            bwipp_raiseerror(); //#8542
        } //#8542
        $_.extchar = $get($_.extencs, $k[--$j]); //#8544
        $_.extlen = $_.extchar.length; //#8545
        $puti($_.newcode, $_.j, $_.extchar); //#8546
        var _X = $get($_.barcode, $_.i); //#8547
        $k[$j++] = $_.newtext; //#8547
        $k[$j++] = $_.k; //#8547
        $k[$j++] = _X; //#8547
        if ((_X < 32) || (_X == 127)) { //#8547
            $k[$j - 1] = 32; //#8547
        } //#8547
        var _Y = $k[--$j]; //#8547
        var _Z = $k[--$j]; //#8547
        $put($k[--$j], _Z, _Y); //#8547
        if ($_.extlen != 1) { //#8548
            $put($_.newtext, $_.k + 1, 32); //#8548
        } //#8548
        $_.j = $_.j + $_.extlen; //#8549
        var _i = ($_.extlen == 1) ? 1 : 2; //#8550
        $_.k = $_.k + _i; //#8550
    } //#8550
    $_.newcode = $geti($_.newcode, 0, $_.j); //#8552
    $_.newtext = $geti($_.newtext, 0, $_.k); //#8553
    $put($_.options, "dontdraw", true); //#8556
    $put($_.options, "parsefnc", true); //#8557
    $k[$j++] = 'args'; //#8558
    $k[$j++] = $_.newcode; //#8558
    $k[$j++] = $_.options; //#8558
    bwipp_code93(); //#8558
    var _t = $k[--$j]; //#8558
    $_[$k[--$j]] = _t; //#8558
    if ($_.includetext) { //#8570
        $_.txt = $get($_.args, "txt"); //#8562
        for (var _10 = 0, _z = $_.newtext.length - 1; _10 <= _z; _10 += 1) { //#8568
            $_.i = _10; //#8564
            $_.txtentry = $get($_.txt, $_.i); //#8565
            $put($_.txtentry, 0, $geti($_.newtext, $_.i, 1)); //#8566
            $put($_.txt, $_.i, $_.txtentry); //#8567
        } //#8567
        $put($_.args, "txt", $_.txt); //#8569
    } //#8569
    $put($_.args, "opt", $_.options); //#8571
    $k[$j++] = $_.args; //#8574
    if (!$_.dontdraw) { //#8574
        bwipp_renlinear(); //#8574
    } //#8574
    $_ = $__; //#8578
} //bwipp_code93ext
function bwipp_interleaved2of5() {
    var $__ = $_; //#8608
    $_ = Object.assign({}, $_); //#8608
    $_.dontdraw = false; //#8611
    $_.includecheck = false; //#8612
    $_.includetext = false; //#8613
    $_.includecheckintext = false; //#8614
    $_.textfont = "OCR-B"; //#8615
    $_.textsize = 10; //#8616
    $_.textyoffset = -8; //#8617
    $_.height = 1; //#8618
    $k[$j++] = $_; //#8620
    bwipp_processoptions(); //#8620
    $_.options = $k[--$j]; //#8620
    $_.barcode = $k[--$j]; //#8621
    $forall($_.barcode, function() { //#8630
        var _3 = $k[--$j]; //#8627
        if ((_3 < 48) || (_3 > 57)) { //#8629
            $k[$j++] = 'bwipp.interleaved2of5badCharacter#8628'; //#8628
            $k[$j++] = "Interleaved 2 of 5 must contain only digits"; //#8628
            bwipp_raiseerror(); //#8628
        } //#8628
    }); //#8628
    $_.barlen = $_.barcode.length; //#8632
    if (((($_.barlen % 2) == 0) && $_.includecheck) || ((($_.barlen % 2) != 0) && (!$_.includecheck))) { //#8643
        $_.pad = $s($_.barlen + 1); //#8638
        $put($_.pad, 0, 48); //#8639
        $puti($_.pad, 1, $_.barcode); //#8640
        $_.barcode = $_.pad; //#8641
        $_.barlen = $_.barlen + 1; //#8642
    } //#8642
    if ($_.includecheck) { //#8662
        $_.checksum = 0; //#8647
        for (var _J = 0, _I = $_.barlen - 1; _J <= _I; _J += 1) { //#8655
            $_.i = _J; //#8649
            if (($_.i % 2) == 0) { //#8653
                $_.checksum = $_.checksum + (($get($_.barcode, $_.i) - 48) * 3); //#8651
            } else { //#8653
                $_.checksum = $_.checksum + ($get($_.barcode, $_.i) - 48); //#8653
            } //#8653
        } //#8653
        $_.checksum = (10 - ($_.checksum % 10)) % 10; //#8656
        $_.pad = $s($_.barlen + 1); //#8657
        $puti($_.pad, 0, $_.barcode); //#8658
        $put($_.pad, $_.barlen, $_.checksum + 48); //#8659
        $_.barcode = $_.pad; //#8660
        $_.barlen = $_.barlen + 1; //#8661
    } //#8661
    $_.encs = $a(["11221", "21112", "12112", "22111", "11212", "21211", "12211", "11122", "21121", "12121", "1111", "2111"]); //#8670
    $_.barchars = "0123456789"; //#8673
    $_.sbs = $s(($_.barlen * 5) + 8); //#8676
    $_.txt = $a($_.barlen); //#8677
    $puti($_.sbs, 0, $get($_.encs, 10)); //#8680
    for (var _o = 0, _n = $_.barlen - 1; _o <= _n; _o += 2) { //#8714
        $_.i = _o; //#8683
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#8685
        $j--; //#8686
        $_.indx = $k[--$j].length; //#8687
        $j -= 2; //#8688
        $_.enca = $get($_.encs, $_.indx); //#8689
        $search($_.barchars, $geti($_.barcode, $_.i + 1, 1)); //#8691
        $j--; //#8692
        $_.indx = $k[--$j].length; //#8693
        $j -= 2; //#8694
        $_.encb = $get($_.encs, $_.indx); //#8695
        $_.intl = $s($_.enca.length * 2); //#8698
        for (var _19 = 0, _18 = $_.enca.length - 1; _19 <= _18; _19 += 1) { //#8705
            $_.j = _19; //#8700
            $_.achar = $get($_.enca, $_.j); //#8701
            $_.bchar = $get($_.encb, $_.j); //#8702
            $put($_.intl, $_.j * 2, $_.achar); //#8703
            $put($_.intl, ($_.j * 2) + 1, $_.bchar); //#8704
        } //#8704
        $puti($_.sbs, ($_.i * 5) + 4, $_.intl); //#8707
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 9) + 4, $_.textyoffset, $_.textfont, $_.textsize])); //#8708
        if (($_.includecheck && (!$_.includecheckintext)) && (($_.barlen - 2) == $_.i)) { //#8712
            $put($_.txt, $_.i + 1, $a([" ", (($_.i + 1) * 9) + 4, $_.textyoffset, $_.textfont, $_.textsize])); //#8710
        } else { //#8712
            $put($_.txt, $_.i + 1, $a([$geti($_.barcode, $_.i + 1, 1), (($_.i + 1) * 9) + 4, $_.textyoffset, $_.textfont, $_.textsize])); //#8712
        } //#8712
    } //#8712
    $puti($_.sbs, ($_.barlen * 5) + 4, $get($_.encs, 11)); //#8717
    $k[$j++] = Infinity; //#8720
    $k[$j++] = Infinity; //#8722
    var _1y = $_.sbs; //#8722
    for (var _1z = 0, _20 = _1y.length; _1z < _20; _1z++) { //#8722
        $k[$j++] = $get(_1y, _1z) - 48; //#8722
    } //#8722
    var _22 = $a(); //#8722
    $k[$j++] = Infinity; //#8723
    for (var _24 = 0, _25 = ~~(($_.sbs.length + 1) / 2); _24 < _25; _24++) { //#8723
        $k[$j++] = $_.height; //#8723
    } //#8723
    var _27 = $a(); //#8723
    $k[$j++] = Infinity; //#8724
    for (var _29 = 0, _2A = ~~(($_.sbs.length + 1) / 2); _29 < _2A; _29++) { //#8724
        $k[$j++] = 0; //#8724
    } //#8724
    var _2B = $a(); //#8724
    $k[$j++] = 'ren'; //#8727
    $k[$j++] = 'renlinear'; //#8727
    $k[$j++] = 'sbs'; //#8727
    $k[$j++] = _22; //#8727
    $k[$j++] = 'bhs'; //#8727
    $k[$j++] = _27; //#8727
    $k[$j++] = 'bbs'; //#8727
    $k[$j++] = _2B; //#8727
    if ($_.includetext) { //#8727
        $k[$j++] = 'txt'; //#8726
        $k[$j++] = $_.txt; //#8726
    } //#8726
    $k[$j++] = 'barratio'; //#8730
    $k[$j++] = 2; //#8730
    $k[$j++] = 'spaceratio'; //#8730
    $k[$j++] = 2; //#8730
    $k[$j++] = 'opt'; //#8730
    $k[$j++] = $_.options; //#8730
    var _2F = $d(); //#8730
    $k[$j++] = _2F; //#8733
    if (!$_.dontdraw) { //#8733
        bwipp_renlinear(); //#8733
    } //#8733
    $_ = $__; //#8737
} //bwipp_interleaved2of5
function bwipp_itf14() {
    var $__ = $_; //#8766
    $_ = Object.assign({}, $_); //#8766
    $_.dontdraw = false; //#8769
    $_.includetext = false; //#8770
    $_.includecheckintext = true; //#8771
    $_.textfont = "OCR-B"; //#8772
    $_.textsize = 10; //#8773
    $_.textxoffset = 0; //#8774
    $_.textyoffset = -3.5; //#8775
    $_.height = 0.5; //#8776
    $_.showborder = true; //#8777
    $_.borderwidth = 4; //#8778
    $_.borderleft = 10; //#8779
    $_.borderright = 10; //#8780
    $_.bordertop = -0.55; //#8781
    $_.borderbottom = -0.55; //#8782
    $k[$j++] = $_; //#8784
    bwipp_processoptions(); //#8784
    $_.options = $k[--$j]; //#8784
    $_.barcode = $k[--$j]; //#8785
    $_.text = $_.barcode; //#8788
    var _3 = $_.barcode; //#8789
    $k[$j++] = $s(_3.length); //#8790
    $k[$j++] = 0; //#8790
    $forall(_3, function() { //#8790
        var _5 = $k[$j - 1]; //#8790
        if (_5 != 32) { //#8790
            var _6 = $k[--$j]; //#8790
            var _7 = $k[--$j]; //#8790
            var _8 = $k[$j - 1]; //#8790
            $put(_8, _7, _6); //#8790
            $k[$j++] = _7 + 1; //#8790
        } else { //#8790
            $j--; //#8790
        } //#8790
    }); //#8790
    var _9 = $k[--$j]; //#8791
    $_.barcode = $geti($k[--$j], 0, _9); //#8791
    $_.hasspace = $_.text.length != $_.barcode.length; //#8792
    if (($_.barcode.length != 13) && ($_.barcode.length != 14)) { //#8797
        $k[$j++] = 'bwipp.itf14badLength#8796'; //#8796
        $k[$j++] = "ITF-14 must be 13 or 14 digits"; //#8796
        bwipp_raiseerror(); //#8796
    } //#8796
    var _G = $_.barcode; //#8798
    for (var _H = 0, _I = _G.length; _H < _I; _H++) { //#8802
        var _J = $get(_G, _H); //#8802
        if ((_J < 48) || (_J > 57)) { //#8801
            $k[$j++] = 'bwipp.itf14badCharacter#8800'; //#8800
            $k[$j++] = "ITF-14 must contain only digits"; //#8800
            bwipp_raiseerror(); //#8800
        } //#8800
    } //#8800
    $_.checksum = 0; //#8805
    for (var _K = 0; _K <= 12; _K += 1) { //#8809
        $_.i = _K; //#8807
        $k[$j++] = 'checksum'; //#8808
        $k[$j++] = $_.checksum; //#8808
        $k[$j++] = $get($_.barcode, $_.i) - 48; //#8808
        if (($_.i % 2) == 0) { //#8808
            var _Q = $k[--$j]; //#8808
            $k[$j++] = _Q * 3; //#8808
        } //#8808
        var _R = $k[--$j]; //#8808
        var _S = $k[--$j]; //#8808
        $_[$k[--$j]] = $f(_S + _R); //#8808
    } //#8808
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#8810
    if ($_.barcode.length == 14) { //#8824
        if ($get($_.barcode, 13) != ($_.checksum + 48)) { //#8814
            $k[$j++] = 'bwipp.itf14badCheckDigit#8813'; //#8813
            $k[$j++] = "Incorrect ITF-14 check digit provided"; //#8813
            bwipp_raiseerror(); //#8813
        } //#8813
    } else { //#8824
        var _Z = $s(14); //#8816
        $puti(_Z, 0, $_.barcode); //#8817
        $put(_Z, 13, $_.checksum + 48); //#8818
        $_.barcode = _Z; //#8819
        var _e = $_.hasspace ? 2 : 1; //#8820
        var _f = $s($_.text.length + _e); //#8820
        $puti(_f, _f.length - 2, " "); //#8821
        $put(_f, _f.length - 1, $_.checksum + 48); //#8822
        $puti(_f, 0, $_.text); //#8823
        $_.text = _f; //#8824
    } //#8824
    var _i = $_.options; //#8828
    $put(_i, "dontdraw", true); //#8828
    $put($_.options, "showborder", $_.showborder); //#8829
    $put($_.options, "borderwidth", $_.borderwidth); //#8830
    $put($_.options, "borderleft", $_.borderleft); //#8831
    $put($_.options, "borderright", $_.borderright); //#8832
    var _r = $_.options; //#8833
    $put(_r, "bordertop", $_.bordertop); //#8833
    $put($_.options, "borderbottom", $_.borderbottom); //#8834
    $put($_.options, "height", $_.height); //#8835
    $put($_.options, "textyoffset", $_.textyoffset); //#8836
    $k[$j++] = 'args'; //#8837
    $k[$j++] = $_.barcode; //#8837
    $k[$j++] = $_.options; //#8837
    bwipp_interleaved2of5(); //#8837
    var _11 = $k[--$j]; //#8837
    $_[$k[--$j]] = _11; //#8837
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])])); //#8839
    $put($_.args, "textxalign", "center"); //#8840
    $put($_.args, "opt", $_.options); //#8841
    $k[$j++] = $_.args; //#8844
    if (!$_.dontdraw) { //#8844
        bwipp_renlinear(); //#8844
    } //#8844
    $_ = $__; //#8846
} //bwipp_itf14
function bwipp_identcode() {
    var $__ = $_; //#8875
    $_ = Object.assign({}, $_); //#8875
    $_.dontdraw = false; //#8878
    $_.includetext = false; //#8879
    $_.textfont = "OCR-B"; //#8880
    $_.textsize = 10; //#8881
    $_.textxoffset = 0; //#8882
    $_.textyoffset = -8; //#8883
    $_.height = 1; //#8884
    $k[$j++] = $_; //#8886
    bwipp_processoptions(); //#8886
    $_.options = $k[--$j]; //#8886
    $_.barcode = $k[--$j]; //#8887
    if (($_.barcode.length != 11) && ($_.barcode.length != 12)) { //#8892
        $k[$j++] = 'bwipp.identcodeBadLength#8891'; //#8891
        $k[$j++] = "Deutsche Post Identcode must be 11 or 12 digits"; //#8891
        bwipp_raiseerror(); //#8891
    } //#8891
    $forall($_.barcode, function() { //#8897
        var _5 = $k[--$j]; //#8894
        if ((_5 < 48) || (_5 > 57)) { //#8896
            $k[$j++] = 'bwipp.identcodeBadCharacter#8895'; //#8895
            $k[$j++] = "Deutsche Post Identcode must contain only digits"; //#8895
            bwipp_raiseerror(); //#8895
        } //#8895
    }); //#8895
    $_.checksum = 0; //#8900
    for (var _6 = 0; _6 <= 10; _6 += 1) { //#8904
        $_.i = _6; //#8902
        $k[$j++] = 'checksum'; //#8903
        $k[$j++] = $_.checksum; //#8903
        $k[$j++] = $get($_.barcode, $_.i) - 48; //#8903
        if (($_.i % 2) == 0) { //#8903
            var _C = $k[--$j]; //#8903
            $k[$j++] = _C * 4; //#8903
        } else { //#8903
            var _D = $k[--$j]; //#8903
            $k[$j++] = _D * 9; //#8903
        } //#8903
        var _E = $k[--$j]; //#8903
        var _F = $k[--$j]; //#8903
        $_[$k[--$j]] = $f(_F + _E); //#8903
    } //#8903
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#8905
    if ($_.barcode.length == 12) { //#8910
        if ($get($_.barcode, 11) != ($_.checksum + 48)) { //#8909
            $k[$j++] = 'bwipp.identcodeBadCheckDigit#8908'; //#8908
            $k[$j++] = "Incorrect Deutsche Post Identcode check digit provided"; //#8908
            bwipp_raiseerror(); //#8908
        } //#8908
    } //#8908
    var _M = $s(12); //#8911
    $puti(_M, 0, $_.barcode); //#8912
    $put(_M, 11, $_.checksum + 48); //#8913
    $_.barcode = _M; //#8914
    $_.text = $strcpy($s(16), "  .       .     "); //#8917
    $puti($_.text, 0, $geti($_.barcode, 0, 2)); //#8918
    $puti($_.text, 3, $geti($_.barcode, 2, 3)); //#8919
    $puti($_.text, 7, $geti($_.barcode, 5, 3)); //#8920
    $puti($_.text, 11, $geti($_.barcode, 8, 3)); //#8921
    $puti($_.text, 15, $geti($_.barcode, 11, 1)); //#8922
    $put($_.options, "dontdraw", true); //#8925
    $put($_.options, "includecheck", false); //#8926
    var _i = $_.barcode; //#8927
    $k[$j++] = 'args'; //#8927
    $k[$j++] = _i; //#8927
    $k[$j++] = $_.options; //#8927
    bwipp_interleaved2of5(); //#8927
    var _k = $k[--$j]; //#8927
    $_[$k[--$j]] = _k; //#8927
    var _r = $_.textsize; //#8929
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, _r])])); //#8929
    $put($_.args, "textxalign", "center"); //#8930
    $put($_.args, "opt", $_.options); //#8931
    $k[$j++] = $_.args; //#8934
    if (!$_.dontdraw) { //#8934
        bwipp_renlinear(); //#8934
    } //#8934
    $_ = $__; //#8936
} //bwipp_identcode
function bwipp_leitcode() {
    var $__ = $_; //#8965
    $_ = Object.assign({}, $_); //#8965
    $_.dontdraw = false; //#8968
    $_.includetext = false; //#8969
    $_.textfont = "OCR-B"; //#8970
    $_.textsize = 10; //#8971
    $_.textxoffset = 0; //#8972
    $_.textyoffset = -8; //#8973
    $_.height = 1; //#8974
    $k[$j++] = $_; //#8976
    bwipp_processoptions(); //#8976
    $_.options = $k[--$j]; //#8976
    $_.barcode = $k[--$j]; //#8977
    if (($_.barcode.length != 13) && ($_.barcode.length != 14)) { //#8982
        $k[$j++] = 'bwipp.leitcodeBadLength#8981'; //#8981
        $k[$j++] = "Deutsche Post Leitcode must be 13 or 14 digits"; //#8981
        bwipp_raiseerror(); //#8981
    } //#8981
    $forall($_.barcode, function() { //#8987
        var _5 = $k[--$j]; //#8984
        if ((_5 < 48) || (_5 > 57)) { //#8986
            $k[$j++] = 'bwipp.leitcodeBadCharacter#8985'; //#8985
            $k[$j++] = "Deutsche Post Leitcode must contain only digits"; //#8985
            bwipp_raiseerror(); //#8985
        } //#8985
    }); //#8985
    $_.checksum = 0; //#8990
    for (var _6 = 0; _6 <= 12; _6 += 1) { //#8994
        $_.i = _6; //#8992
        $k[$j++] = 'checksum'; //#8993
        $k[$j++] = $_.checksum; //#8993
        $k[$j++] = $get($_.barcode, $_.i) - 48; //#8993
        if (($_.i % 2) == 0) { //#8993
            var _C = $k[--$j]; //#8993
            $k[$j++] = _C * 4; //#8993
        } else { //#8993
            var _D = $k[--$j]; //#8993
            $k[$j++] = _D * 9; //#8993
        } //#8993
        var _E = $k[--$j]; //#8993
        var _F = $k[--$j]; //#8993
        $_[$k[--$j]] = $f(_F + _E); //#8993
    } //#8993
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#8995
    if ($_.barcode.length == 14) { //#9000
        if ($get($_.barcode, 13) != ($_.checksum + 48)) { //#8999
            $k[$j++] = 'bwipp.leitcodeBadCheckDigit#8998'; //#8998
            $k[$j++] = "Incorrect Deutsche Post Leitcode check digit provided"; //#8998
            bwipp_raiseerror(); //#8998
        } //#8998
    } //#8998
    var _M = $s(14); //#9001
    $puti(_M, 0, $_.barcode); //#9002
    $put(_M, 13, $_.checksum + 48); //#9003
    $_.barcode = _M; //#9004
    $_.text = $strcpy($s(18), "     .   .   .    "); //#9007
    $puti($_.text, 0, $geti($_.barcode, 0, 5)); //#9008
    $puti($_.text, 6, $geti($_.barcode, 5, 3)); //#9009
    $puti($_.text, 10, $geti($_.barcode, 8, 3)); //#9010
    $puti($_.text, 14, $geti($_.barcode, 11, 2)); //#9011
    $puti($_.text, 17, $geti($_.barcode, 13, 1)); //#9012
    $put($_.options, "dontdraw", true); //#9015
    $put($_.options, "includecheck", false); //#9016
    var _i = $_.barcode; //#9017
    $k[$j++] = 'args'; //#9017
    $k[$j++] = _i; //#9017
    $k[$j++] = $_.options; //#9017
    bwipp_interleaved2of5(); //#9017
    var _k = $k[--$j]; //#9017
    $_[$k[--$j]] = _k; //#9017
    var _r = $_.textsize; //#9019
    $put($_.args, "txt", $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, _r])])); //#9019
    $put($_.args, "textxalign", "center"); //#9020
    $put($_.args, "opt", $_.options); //#9021
    $k[$j++] = $_.args; //#9024
    if (!$_.dontdraw) { //#9024
        bwipp_renlinear(); //#9024
    } //#9024
    $_ = $__; //#9026
} //bwipp_leitcode
function bwipp_databaromni() {
    var $__ = $_; //#9057
    $_ = Object.assign({}, $_); //#9057
    $_.dontdraw = false; //#9060
    $_.height = 33 / 72; //#9061
    $_.linkage = false; //#9062
    $_.format = "omni"; //#9063
    $_.barxmult = 33; //#9064
    $k[$j++] = $_; //#9066
    bwipp_processoptions(); //#9066
    $_.options = $k[--$j]; //#9066
    $_.barcode = $k[--$j]; //#9067
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#9071
        $k[$j++] = 'bwipp.databaromniBadLength#9070'; //#9070
        $k[$j++] = "GS1 DataBar Omnidirectional must be 13 or 14 digits"; //#9070
        bwipp_raiseerror(); //#9070
    } //#9070
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#9074
        $k[$j++] = 'bwipp.databaromniBadAI#9073'; //#9073
        $k[$j++] = "GS1 DataBar Omnidirectional must begin with (01) application identifier"; //#9073
        bwipp_raiseerror(); //#9073
    } //#9073
    var _8 = $geti($_.barcode, 4, $_.barcode.length - 4); //#9075
    for (var _9 = 0, _A = _8.length; _9 < _A; _9++) { //#9079
        var _B = $get(_8, _9); //#9079
        if ((_B < 48) || (_B > 57)) { //#9078
            $k[$j++] = 'bwipp.databaromniBadCharacter#9077'; //#9077
            $k[$j++] = "GS1 DataBar Omnidirectional must contain only digits"; //#9077
            bwipp_raiseerror(); //#9077
        } //#9077
    } //#9077
    if ($_.barxmult < 33) { //#9083
        $k[$j++] = 'bwipp.databarStackedOmniBarXmult#9082'; //#9082
        $k[$j++] = "GS1 DataBar Stacked Omnidirectional must have a barxmult of at least 33"; //#9082
        bwipp_raiseerror(); //#9082
    } //#9082
    if ((($ne($_.format, "omni") && $ne($_.format, "stacked")) && $ne($_.format, "stackedomni")) && $ne($_.format, "truncated")) { //#9087
        $k[$j++] = 'bwipp.databaromniBadFormat#9086'; //#9086
        $k[$j++] = "Valid formats are omni, stacked, stackedomni and truncated"; //#9086
        bwipp_raiseerror(); //#9086
    } //#9086
    if ($eq($_.format, "truncated")) { //#9089
        $_.height = 13 / 72; //#9089
    } //#9089
    $_.checksum = 0; //#9094
    for (var _I = 0; _I <= 12; _I += 1) { //#9098
        $_.i = _I; //#9096
        $k[$j++] = 'checksum'; //#9097
        $k[$j++] = $_.checksum; //#9097
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#9097
        if (($_.i % 2) == 0) { //#9097
            var _O = $k[--$j]; //#9097
            $k[$j++] = _O * 3; //#9097
        } //#9097
        var _P = $k[--$j]; //#9097
        var _Q = $k[--$j]; //#9097
        $_[$k[--$j]] = $f(_Q + _P); //#9097
    } //#9097
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#9099
    if ($_.barcode.length == 18) { //#9104
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#9103
            $k[$j++] = 'bwipp.databaromniBadCheckDigit#9102'; //#9102
            $k[$j++] = "Incorrect GS1 DataBar Omnidirectional check digit provided"; //#9102
            bwipp_raiseerror(); //#9102
        } //#9102
    } //#9102
    var _X = $s(18); //#9105
    $puti(_X, 0, $_.barcode); //#9106
    $put(_X, 17, $_.checksum + 48); //#9107
    $_.barcode = _X; //#9108
    $_.txt = $a($_.barcode.length); //#9111
    for (var _e = 0, _d = $_.barcode.length - 1; _e <= _d; _e += 1) { //#9115
        $_.i = _e; //#9113
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), 0, 0, "", 0])); //#9114
    } //#9114
    $_.ncr = function() {
        var _l = $k[--$j]; //#9118
        var _m = $k[--$j]; //#9118
        var _n = $f(_m - _l); //#9118
        if (_l < $f(_m - _l)) { //#9118
            var _ = _n; //#9118
            _n = _l; //#9118
            _l = _; //#9118
        } //#9118
        $k[$j++] = _n; //#9125
        $k[$j++] = 1; //#9125
        $k[$j++] = 1; //#9125
        for (var _p = _m, _o = _l + 1; _p >= _o; _p -= 1) { //#9125
            var _q = $k[--$j]; //#9121
            var _r = $k[$j - 1]; //#9122
            var _s = $k[$j - 2]; //#9122
            $k[$j++] = _q * _p; //#9124
            if ($le(_r, _s)) { //#9124
                var _t = $k[--$j]; //#9123
                var _u = $k[--$j]; //#9123
                $k[$j++] = _u + 1; //#9123
                $k[$j++] = ~~(_t / _u); //#9123
            } //#9123
        } //#9123
        for (;;) { //#9129
            var _w = $k[$j - 2]; //#9127
            var _x = $k[$j - 3]; //#9127
            if ($gt(_w, _x)) { //#9127
                break; //#9127
            } //#9127
            var _y = $k[--$j]; //#9128
            var _z = $k[--$j]; //#9128
            $k[$j++] = _z + 1; //#9128
            $k[$j++] = ~~(_y / _z); //#9128
        } //#9128
        var _10 = $k[--$j]; //#9130
        $k[$j - 1] = _10; //#9130
        var _12 = $k[--$j]; //#9130
        $k[$j - 1] = _12; //#9130
    }; //#9130
    $_.getRSSwidths = function() {
        $_.oe = $k[--$j]; //#9134
        $_.el = $k[--$j]; //#9135
        $_.mw = $k[--$j]; //#9136
        $_.nm = $k[--$j]; //#9137
        $_.val = $k[--$j]; //#9138
        $_.out = $a($_.el); //#9139
        $_.mask = 0; //#9140
        for (var _1D = 0, _1C = $_.el - 2; _1D <= _1C; _1D += 1) { //#9167
            $_.bar = _1D; //#9142
            $_.ew = 1; //#9143
            var _1F = $_.bar; //#9144
            $_.mask = $_.mask | ((_1F < 0 ? 1 >>> -_1F : 1 << _1F)); //#9144
            for (;;) { //#9163
                $k[$j++] = 'sval'; //#9145
                $k[$j++] = $f($_.nm - $_.ew) - 1; //#9145
                $k[$j++] = $f($_.el - $_.bar) - 2; //#9145
                $_.ncr(); //#9145
                var _1K = $k[--$j]; //#9145
                $_[$k[--$j]] = _1K; //#9145
                if ($_.oe && $_.mask == 0 && $f($_.nm - $_.ew - $_.el * 2 + $_.bar * 2) >= -2) { //#9148
                    $k[$j++] = 'sval'; //#9147
                    $k[$j++] = $_.sval; //#9147
                    $k[$j++] = $f($_.nm - $_.ew - $_.el + $_.bar); //#9147
                    $k[$j++] = $f($_.el - $_.bar) - 2; //#9147
                    $_.ncr(); //#9147
                    var _1Z = $k[--$j]; //#9147
                    var _1a = $k[--$j]; //#9147
                    $_[$k[--$j]] = $f(_1a - _1Z); //#9147
                } //#9147
                if ($f($_.el - $_.bar) > 2) { //#9157
                    $_.lval = 0; //#9150
                    for (var _1k = $f($_.nm - $_.ew - $_.el + $_.bar) + 2, _1j = $_.mw + 1; _1k >= _1j; _1k -= 1) { //#9154
                        $k[$j++] = $f($_.nm - _1k - $_.ew) - 1; //#9152
                        $k[$j++] = $f($_.el - $_.bar) - 3; //#9152
                        $_.ncr(); //#9152
                        $_.lval = $f($k[--$j] + $_.lval); //#9153
                    } //#9153
                    $_.sval = $f($_.sval - $_.lval * ($_.el - $_.bar - 1)); //#9155
                } else { //#9157
                    if ($f($_.nm - $_.ew) > $_.mw) { //#9157
                        $_.sval = $_.sval - 1; //#9157
                    } //#9157
                } //#9157
                $_.val = $f($_.val - $_.sval); //#9159
                if ($_.val < 0) { //#9160
                    break; //#9160
                } //#9160
                $_.ew = $_.ew + 1; //#9161
                var _24 = $_.bar; //#9162
                $_.mask = $_.mask & (~((_24 < 0 ? 1 >>> -_24 : 1 << _24))); //#9162
            } //#9162
            $_.val = $f($_.val + $_.sval); //#9164
            $_.nm = $f($_.nm - $_.ew); //#9165
            $put($_.out, $_.bar, $_.ew); //#9166
        } //#9166
        $put($_.out, $_.el - 1, $_.nm); //#9168
        $k[$j++] = $_.out; //#9169
    }; //#9169
    $k[$j++] = Infinity; //#9172
    var _2H = $_.linkage ? 1 : 0; //#9173
    var _2J = $geti($_.barcode, 4, 13); //#9174
    $k[$j++] = _2H; //#9174
    for (var _2K = 0, _2L = _2J.length; _2K < _2L; _2K++) { //#9174
        $k[$j++] = $get(_2J, _2K) - 48; //#9174
    } //#9174
    $_.binval = $a(); //#9174
    for (var _2O = 0; _2O <= 12; _2O += 1) { //#9181
        $_.i = _2O; //#9178
        var _2P = $_.binval; //#9179
        var _2Q = $_.i; //#9179
        $put(_2P, _2Q + 1, $f($get(_2P, _2Q + 1) + (($get($_.binval, $_.i) % 4537077) * 10))); //#9179
        $put($_.binval, $_.i, ~~($get($_.binval, $_.i) / 4537077)); //#9180
    } //#9180
    $_.right = $get($_.binval, 13) % 4537077; //#9182
    var _2c = $_.binval; //#9183
    $put(_2c, 13, ~~($get(_2c, 13) / 4537077)); //#9183
    $_.left = 0; //#9185
    $_.i = true; //#9186
    for (var _2e = 0; _2e <= 13; _2e += 1) { //#9196
        $_.j = _2e; //#9188
        var _2h = $get($_.binval, $_.j); //#9189
        $k[$j++] = _2h; //#9195
        if ((_2h == 0) && $_.i) { //#9194
            $j--; //#9191
        } else { //#9194
            $_.i = false; //#9193
            $k[$j++] = 'left'; //#9194
            $k[$j++] = $_.left; //#9194
            var _2l = $k[$j - 3]; //#9194
            var _2m = $k[$j - 1]; //#9194
            $_[$k[$j - 2]] = $f(_2m + (_2l * (~~$pow(10, 13 - $_.j)))); //#9194
            $j -= 3; //#9194
        } //#9194
    } //#9194
    $_.d1 = ~~($_.left / 1597); //#9198
    $_.d2 = $_.left % 1597; //#9199
    $_.d3 = ~~($_.right / 1597); //#9200
    $_.d4 = $_.right % 1597; //#9201
    $_.tab164 = $a([160, 0, 12, 4, 8, 1, 161, 1, 960, 161, 10, 6, 6, 3, 80, 10, 2014, 961, 8, 8, 4, 5, 31, 34, 2714, 2015, 6, 10, 3, 6, 10, 70, 2840, 2715, 4, 12, 1, 8, 1, 126]); //#9210
    $_.tab154 = $a([335, 0, 5, 10, 2, 7, 4, 84, 1035, 336, 7, 8, 4, 5, 20, 35, 1515, 1036, 9, 6, 6, 3, 48, 10, 1596, 1516, 11, 4, 8, 1, 81, 1]); //#9217
    $_.i = 0; //#9220
    for (;;) { //#9230
        if ($_.d1 <= $get($_.tab164, $_.i)) { //#9228
            $aload($geti($_.tab164, $_.i + 1, 7)); //#9222
            $_.d1te = $k[--$j]; //#9223
            $_.d1to = $k[--$j]; //#9223
            $_.d1mwe = $k[--$j]; //#9224
            $_.d1mwo = $k[--$j]; //#9224
            $_.d1ele = $k[--$j]; //#9225
            $_.d1elo = $k[--$j]; //#9225
            $_.d1gs = $k[--$j]; //#9226
            break; //#9227
        } //#9227
        $_.i = $_.i + 8; //#9229
    } //#9229
    $_.i = 0; //#9232
    for (;;) { //#9242
        if ($_.d2 <= $get($_.tab154, $_.i)) { //#9240
            $aload($geti($_.tab154, $_.i + 1, 7)); //#9234
            $_.d2te = $k[--$j]; //#9235
            $_.d2to = $k[--$j]; //#9235
            $_.d2mwe = $k[--$j]; //#9236
            $_.d2mwo = $k[--$j]; //#9236
            $_.d2ele = $k[--$j]; //#9237
            $_.d2elo = $k[--$j]; //#9237
            $_.d2gs = $k[--$j]; //#9238
            break; //#9239
        } //#9239
        $_.i = $_.i + 8; //#9241
    } //#9241
    $_.i = 0; //#9244
    for (;;) { //#9254
        if ($_.d3 <= $get($_.tab164, $_.i)) { //#9252
            $aload($geti($_.tab164, $_.i + 1, 7)); //#9246
            $_.d3te = $k[--$j]; //#9247
            $_.d3to = $k[--$j]; //#9247
            $_.d3mwe = $k[--$j]; //#9248
            $_.d3mwo = $k[--$j]; //#9248
            $_.d3ele = $k[--$j]; //#9249
            $_.d3elo = $k[--$j]; //#9249
            $_.d3gs = $k[--$j]; //#9250
            break; //#9251
        } //#9251
        $_.i = $_.i + 8; //#9253
    } //#9253
    $_.i = 0; //#9256
    for (;;) { //#9266
        if ($_.d4 <= $get($_.tab154, $_.i)) { //#9264
            $aload($geti($_.tab154, $_.i + 1, 7)); //#9258
            $_.d4te = $k[--$j]; //#9259
            $_.d4to = $k[--$j]; //#9259
            $_.d4mwe = $k[--$j]; //#9260
            $_.d4mwo = $k[--$j]; //#9260
            $_.d4ele = $k[--$j]; //#9261
            $_.d4elo = $k[--$j]; //#9261
            $_.d4gs = $k[--$j]; //#9262
            break; //#9263
        } //#9263
        $_.i = $_.i + 8; //#9265
    } //#9265
    $k[$j++] = 'd1wo'; //#9268
    $k[$j++] = ~~($f($_.d1 - $_.d1gs) / $_.d1te); //#9268
    $k[$j++] = $_.d1elo; //#9268
    $k[$j++] = $_.d1mwo; //#9268
    $k[$j++] = 4; //#9268
    $k[$j++] = false; //#9268
    $_.getRSSwidths(); //#9268
    var _3y = $k[--$j]; //#9268
    $_[$k[--$j]] = _3y; //#9268
    $k[$j++] = 'd1we'; //#9269
    $k[$j++] = $f($_.d1 - $_.d1gs) % $_.d1te; //#9269
    $k[$j++] = $_.d1ele; //#9269
    $k[$j++] = $_.d1mwe; //#9269
    $k[$j++] = 4; //#9269
    $k[$j++] = true; //#9269
    $_.getRSSwidths(); //#9269
    var _45 = $k[--$j]; //#9269
    $_[$k[--$j]] = _45; //#9269
    $k[$j++] = 'd2wo'; //#9270
    $k[$j++] = $f($_.d2 - $_.d2gs) % $_.d2to; //#9270
    $k[$j++] = $_.d2elo; //#9270
    $k[$j++] = $_.d2mwo; //#9270
    $k[$j++] = 4; //#9270
    $k[$j++] = true; //#9270
    $_.getRSSwidths(); //#9270
    var _4C = $k[--$j]; //#9270
    $_[$k[--$j]] = _4C; //#9270
    $k[$j++] = 'd2we'; //#9271
    $k[$j++] = ~~($f($_.d2 - $_.d2gs) / $_.d2to); //#9271
    $k[$j++] = $_.d2ele; //#9271
    $k[$j++] = $_.d2mwe; //#9271
    $k[$j++] = 4; //#9271
    $k[$j++] = false; //#9271
    $_.getRSSwidths(); //#9271
    var _4J = $k[--$j]; //#9271
    $_[$k[--$j]] = _4J; //#9271
    $k[$j++] = 'd3wo'; //#9272
    $k[$j++] = ~~($f($_.d3 - $_.d3gs) / $_.d3te); //#9272
    $k[$j++] = $_.d3elo; //#9272
    $k[$j++] = $_.d3mwo; //#9272
    $k[$j++] = 4; //#9272
    $k[$j++] = false; //#9272
    $_.getRSSwidths(); //#9272
    var _4Q = $k[--$j]; //#9272
    $_[$k[--$j]] = _4Q; //#9272
    $k[$j++] = 'd3we'; //#9273
    $k[$j++] = $f($_.d3 - $_.d3gs) % $_.d3te; //#9273
    $k[$j++] = $_.d3ele; //#9273
    $k[$j++] = $_.d3mwe; //#9273
    $k[$j++] = 4; //#9273
    $k[$j++] = true; //#9273
    $_.getRSSwidths(); //#9273
    var _4X = $k[--$j]; //#9273
    $_[$k[--$j]] = _4X; //#9273
    $k[$j++] = 'd4wo'; //#9274
    $k[$j++] = $f($_.d4 - $_.d4gs) % $_.d4to; //#9274
    $k[$j++] = $_.d4elo; //#9274
    $k[$j++] = $_.d4mwo; //#9274
    $k[$j++] = 4; //#9274
    $k[$j++] = true; //#9274
    $_.getRSSwidths(); //#9274
    var _4e = $k[--$j]; //#9274
    $_[$k[--$j]] = _4e; //#9274
    $k[$j++] = 'd4we'; //#9275
    $k[$j++] = ~~($f($_.d4 - $_.d4gs) / $_.d4to); //#9275
    $k[$j++] = $_.d4ele; //#9275
    $k[$j++] = $_.d4mwe; //#9275
    $k[$j++] = 4; //#9275
    $k[$j++] = false; //#9275
    $_.getRSSwidths(); //#9275
    var _4l = $k[--$j]; //#9275
    $_[$k[--$j]] = _4l; //#9275
    $_.d1w = $a(8); //#9277
    for (var _4o = 0; _4o <= 3; _4o += 1) { //#9282
        $_.i = _4o; //#9279
        $put($_.d1w, $_.i * 2, $get($_.d1wo, $_.i)); //#9280
        $put($_.d1w, ($_.i * 2) + 1, $get($_.d1we, $_.i)); //#9281
    } //#9281
    $_.d2w = $a(8); //#9284
    for (var _50 = 0; _50 <= 3; _50 += 1) { //#9289
        $_.i = _50; //#9286
        $put($_.d2w, 7 - ($_.i * 2), $get($_.d2wo, $_.i)); //#9287
        $put($_.d2w, 6 - ($_.i * 2), $get($_.d2we, $_.i)); //#9288
    } //#9288
    $_.d3w = $a(8); //#9291
    for (var _5C = 0; _5C <= 3; _5C += 1) { //#9296
        $_.i = _5C; //#9293
        $put($_.d3w, 7 - ($_.i * 2), $get($_.d3wo, $_.i)); //#9294
        $put($_.d3w, 6 - ($_.i * 2), $get($_.d3we, $_.i)); //#9295
    } //#9295
    $_.d4w = $a(8); //#9298
    for (var _5O = 0; _5O <= 3; _5O += 1) { //#9303
        $_.i = _5O; //#9300
        $put($_.d4w, $_.i * 2, $get($_.d4wo, $_.i)); //#9301
        $put($_.d4w, ($_.i * 2) + 1, $get($_.d4we, $_.i)); //#9302
    } //#9302
    $k[$j++] = Infinity; //#9305
    $aload($_.d1w); //#9306
    $aload($_.d2w); //#9307
    $aload($_.d3w); //#9308
    $aload($_.d4w); //#9309
    $_.widths = $a(); //#9309
    $_.checkweights = $a([1, 3, 9, 27, 2, 6, 18, 54, 58, 72, 24, 8, 29, 36, 12, 4, 74, 51, 17, 32, 37, 65, 48, 16, 64, 34, 23, 69, 49, 68, 46, 59]); //#9318
    $_.checkwidths = $a([3, 8, 2, 1, 1, 3, 5, 5, 1, 1, 3, 3, 7, 1, 1, 3, 1, 9, 1, 1, 2, 7, 4, 1, 1, 2, 5, 6, 1, 1, 2, 3, 8, 1, 1, 1, 5, 7, 1, 1, 1, 3, 9, 1, 1]); //#9324
    $_.checksum = 0; //#9327
    for (var _5h = 0; _5h <= 31; _5h += 1) { //#9331
        $_.i = _5h; //#9329
        $_.checksum = $f($_.checksum + ($get($_.widths, $_.i) * $get($_.checkweights, $_.i))); //#9330
    } //#9330
    $_.checksum = $_.checksum % 79; //#9332
    if ($_.checksum >= 8) { //#9333
        $_.checksum = $_.checksum + 1; //#9333
    } //#9333
    if ($_.checksum >= 72) { //#9334
        $_.checksum = $_.checksum + 1; //#9334
    } //#9334
    $_.checklt = $geti($_.checkwidths, (~~($_.checksum / 9)) * 5, 5); //#9335
    $_.checkrtrev = $geti($_.checkwidths, ($_.checksum % 9) * 5, 5); //#9336
    $_.checkrt = $a(5); //#9337
    for (var _61 = 0; _61 <= 4; _61 += 1) { //#9341
        $_.i = _61; //#9339
        $put($_.checkrt, $_.i, $get($_.checkrtrev, 4 - $_.i)); //#9340
    } //#9340
    if ($eq($_.format, "omni") || $eq($_.format, "truncated")) { //#9487
        $k[$j++] = Infinity; //#9346
        $k[$j++] = 1; //#9348
        $aload($_.d1w); //#9348
        $aload($_.checklt); //#9349
        $aload($_.d2w); //#9350
        $aload($_.d4w); //#9351
        $aload($_.checkrt); //#9352
        $aload($_.d3w); //#9353
        $k[$j++] = 1; //#9354
        $k[$j++] = 1; //#9354
        $_.sbs = $a(); //#9354
        $k[$j++] = Infinity; //#9357
        var _6G = $_.sbs; //#9359
        $k[$j++] = Infinity; //#9360
        for (var _6I = 0, _6J = ~~(($_.sbs.length + 1) / 2); _6I < _6J; _6I++) { //#9360
            $k[$j++] = $_.height; //#9360
        } //#9360
        var _6L = $a(); //#9360
        $k[$j++] = Infinity; //#9361
        for (var _6N = 0, _6O = ~~(($_.sbs.length + 1) / 2); _6N < _6O; _6N++) { //#9361
            $k[$j++] = 0; //#9361
        } //#9361
        var _6P = $a(); //#9361
        $k[$j++] = 'ren'; //#9366
        $k[$j++] = 'renlinear'; //#9366
        $k[$j++] = 'sbs'; //#9366
        $k[$j++] = _6G; //#9366
        $k[$j++] = 'bhs'; //#9366
        $k[$j++] = _6L; //#9366
        $k[$j++] = 'bbs'; //#9366
        $k[$j++] = _6P; //#9366
        $k[$j++] = 'txt'; //#9366
        $k[$j++] = $_.txt; //#9366
        $k[$j++] = 'textxalign'; //#9366
        $k[$j++] = "center"; //#9366
        $k[$j++] = 'borderleft'; //#9366
        $k[$j++] = 0; //#9366
        $k[$j++] = 'borderright'; //#9366
        $k[$j++] = 0; //#9366
        $k[$j++] = 'opt'; //#9366
        $k[$j++] = $_.options; //#9366
        var _6S = $d(); //#9366
        $k[$j++] = _6S; //#9369
        if (!$_.dontdraw) { //#9369
            bwipp_renlinear(); //#9369
        } //#9369
    } else { //#9487
        $k[$j++] = Infinity; //#9373
        $k[$j++] = 1; //#9375
        $k[$j++] = 1; //#9375
        $aload($_.d1w); //#9375
        $aload($_.checklt); //#9376
        $aload($_.d2w); //#9377
        $k[$j++] = 1; //#9378
        $k[$j++] = 1; //#9378
        $k[$j++] = 0; //#9378
        $_.top = $a(); //#9378
        $k[$j++] = Infinity; //#9380
        $k[$j++] = 1; //#9382
        $k[$j++] = 1; //#9382
        $aload($_.d4w); //#9382
        $aload($_.checkrt); //#9383
        $aload($_.d3w); //#9384
        $k[$j++] = 1; //#9385
        $k[$j++] = 1; //#9385
        $k[$j++] = 0; //#9385
        $_.bot = $a(); //#9385
        for (var _6c = 0; _6c <= 24; _6c += 2) { //#9391
            $_.i = _6c; //#9388
            for (var _6g = 0, _6h = $get($_.top, $_.i); _6g < _6h; _6g++) { //#9389
                $k[$j++] = 0; //#9389
            } //#9389
            for (var _6l = 0, _6m = $get($_.top, $_.i + 1); _6l < _6m; _6l++) { //#9390
                $k[$j++] = 1; //#9390
            } //#9390
        } //#9390
        $astore($a(50)); //#9392
        $_.top = $k[--$j]; //#9392
        for (var _6p = 0; _6p <= 24; _6p += 2) { //#9397
            $_.i = _6p; //#9394
            for (var _6t = 0, _6u = $get($_.bot, $_.i); _6t < _6u; _6t++) { //#9395
                $k[$j++] = 1; //#9395
            } //#9395
            for (var _6y = 0, _6z = $get($_.bot, $_.i + 1); _6y < _6z; _6y++) { //#9396
                $k[$j++] = 0; //#9396
            } //#9396
        } //#9396
        $astore($a(50)); //#9398
        $_.bot = $k[--$j]; //#9398
        if ($eq($_.format, "stacked")) { //#9420
            $_.sep = $a(50); //#9402
            $put($_.sep, 0, 0); //#9403
            for (var _75 = 1; _75 <= 49; _75 += 1) { //#9411
                $_.i = _75; //#9405
                if ($eq($get($_.top, $_.i), $get($_.bot, $_.i))) { //#9409
                    $put($_.sep, $_.i, 1 - $get($_.top, $_.i)); //#9407
                } else { //#9409
                    $put($_.sep, $_.i, 1 - $get($_.sep, $_.i - 1)); //#9409
                } //#9409
            } //#9409
            $puti($_.sep, 0, $a([0, 0, 0, 0])); //#9412
            $puti($_.sep, 46, $a([0, 0, 0, 0])); //#9413
            $k[$j++] = Infinity; //#9414
            for (var _7Q = 0; _7Q < 5; _7Q++) { //#9415
                $aload($_.top); //#9415
            } //#9415
            $aload($_.sep); //#9416
            for (var _7T = 0; _7T < 7; _7T++) { //#9417
                $aload($_.bot); //#9417
            } //#9417
            $_.pixs = $a(); //#9417
            $_.pixy = ~~($_.pixs.length / 50); //#9419
        } //#9419
        if ($eq($_.format, "stackedomni")) { //#9470
            $k[$j++] = Infinity; //#9424
            $forall($_.top, function() { //#9424
                var _7Z = $k[--$j]; //#9424
                $k[$j++] = 1 - _7Z; //#9424
            }); //#9424
            $_.sep1 = $a(); //#9424
            $puti($_.sep1, 0, $a([0, 0, 0, 0])); //#9425
            $puti($_.sep1, 46, $a([0, 0, 0, 0])); //#9426
            for (var _7f = 18; _7f <= 30; _7f += 1) { //#9439
                $_.i = _7f; //#9428
                if ($get($_.top, $_.i) == 0) { //#9436
                    if ($get($_.top, $_.i - 1) == 1) { //#9433
                        $k[$j++] = 1; //#9431
                    } else { //#9433
                        var _7p = ($get($_.sep1, $_.i - 1) == 0) ? 1 : 0; //#9433
                        $k[$j++] = _7p; //#9433
                    } //#9433
                } else { //#9436
                    $k[$j++] = 0; //#9436
                } //#9436
                $put($_.sep1, $_.i, $k[--$j]); //#9438
            } //#9438
            $k[$j++] = Infinity; //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            for (var _7t = 0; _7t < 21; _7t++) { //#9440
                $k[$j++] = 0; //#9440
                $k[$j++] = 1; //#9440
            } //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            $k[$j++] = 0; //#9440
            $_.sep2 = $a(); //#9440
            $k[$j++] = Infinity; //#9441
            $forall($_.bot, function() { //#9441
                var _7w = $k[--$j]; //#9441
                $k[$j++] = 1 - _7w; //#9441
            }); //#9441
            $_.sep3 = $a(); //#9441
            $puti($_.sep3, 0, $a([0, 0, 0, 0])); //#9442
            $puti($_.sep3, 46, $a([0, 0, 0, 0])); //#9443
            for (var _82 = 19; _82 <= 31; _82 += 1) { //#9456
                $_.i = _82; //#9445
                if ($get($_.bot, $_.i) == 0) { //#9453
                    if ($get($_.bot, $_.i - 1) == 1) { //#9450
                        $k[$j++] = 1; //#9448
                    } else { //#9450
                        var _8C = ($get($_.sep3, $_.i - 1) == 0) ? 1 : 0; //#9450
                        $k[$j++] = _8C; //#9450
                    } //#9450
                } else { //#9453
                    $k[$j++] = 0; //#9453
                } //#9453
                $put($_.sep3, $_.i, $k[--$j]); //#9455
            } //#9455
            $_.f3 = $a([1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1]); //#9457
            $k[$j++] = true; //#9458
            for (var _8H = 0; _8H <= 12; _8H += 1) { //#9458
                var _8M = $k[--$j]; //#9458
                $k[$j++] = _8M && $eq($get($_.bot, _8H + 19), $get($_.f3, _8H)); //#9458
            } //#9458
            if ($k[--$j]) { //#9461
                $puti($_.sep3, 19, $a([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0])); //#9460
            } //#9460
            $k[$j++] = Infinity; //#9462
            for (var _8R = 0, _8S = $_.barxmult; _8R < _8S; _8R++) { //#9463
                $aload($_.top); //#9463
            } //#9463
            $aload($_.sep1); //#9464
            $aload($_.sep2); //#9465
            $aload($_.sep3); //#9466
            for (var _8Y = 0, _8Z = $_.barxmult; _8Y < _8Z; _8Y++) { //#9467
                $aload($_.bot); //#9467
            } //#9467
            $_.pixs = $a(); //#9467
            $_.pixy = ~~($_.pixs.length / 50); //#9469
        } //#9469
        var _8h = new Map([
            ["ren", 'renmatrix'],
            ["pixs", $_.pixs],
            ["pixx", 50],
            ["pixy", $_.pixy],
            ["height", $_.pixy / 72],
            ["width", 50 / 72],
            ["borderleft", 0],
            ["borderright", 0],
            ["bordertop", 0],
            ["borderbottom", 0],
            ["opt", $_.options]
        ]); //#9484
        $k[$j++] = _8h; //#9487
        if (!$_.dontdraw) { //#9487
            bwipp_renmatrix(); //#9487
        } //#9487
    } //#9487
    $_ = $__; //#9493
} //bwipp_databaromni
function bwipp_databarstacked() {
    var $__ = $_; //#9522
    $_ = Object.assign({}, $_); //#9522
    $_.dontdraw = false; //#9525
    $k[$j++] = $_; //#9527
    bwipp_processoptions(); //#9527
    $_.options = $k[--$j]; //#9527
    $_.barcode = $k[--$j]; //#9528
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#9533
        $k[$j++] = 'bwipp.databarstackedBadLength#9532'; //#9532
        $k[$j++] = "GS1 DataBar Stacked must be 13 or 14 digits"; //#9532
        bwipp_raiseerror(); //#9532
    } //#9532
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#9536
        $k[$j++] = 'bwipp.databarstackedBadAI#9535'; //#9535
        $k[$j++] = "GS1 DataBar Stacked must begin with (01) application identifier"; //#9535
        bwipp_raiseerror(); //#9535
    } //#9535
    var _8 = $geti($_.barcode, 4, $_.barcode.length - 4); //#9537
    for (var _9 = 0, _A = _8.length; _9 < _A; _9++) { //#9541
        var _B = $get(_8, _9); //#9541
        if ((_B < 48) || (_B > 57)) { //#9540
            $k[$j++] = 'bwipp.databarstackedBadCharacter#9539'; //#9539
            $k[$j++] = "GS1 DataBar Stacked must contain only digits"; //#9539
            bwipp_raiseerror(); //#9539
        } //#9539
    } //#9539
    $_.checksum = 0; //#9544
    for (var _C = 0; _C <= 12; _C += 1) { //#9548
        $_.i = _C; //#9546
        $k[$j++] = 'checksum'; //#9547
        $k[$j++] = $_.checksum; //#9547
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#9547
        if (($_.i % 2) == 0) { //#9547
            var _I = $k[--$j]; //#9547
            $k[$j++] = _I * 3; //#9547
        } //#9547
        var _J = $k[--$j]; //#9547
        var _K = $k[--$j]; //#9547
        $_[$k[--$j]] = $f(_K + _J); //#9547
    } //#9547
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#9549
    if ($_.barcode.length == 18) { //#9554
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#9553
            $k[$j++] = 'bwipp.databarstackedBadCheckDigit#9552'; //#9552
            $k[$j++] = "Incorrect GS1 DataBar Stacked check digit provided"; //#9552
            bwipp_raiseerror(); //#9552
        } //#9552
    } //#9552
    var _R = $s(18); //#9555
    $puti(_R, 0, $_.barcode); //#9556
    $put(_R, 17, $_.checksum + 48); //#9557
    $_.barcode = _R; //#9558
    $put($_.options, "dontdraw", true); //#9561
    $put($_.options, "format", "stacked"); //#9562
    $k[$j++] = 'args'; //#9564
    $k[$j++] = $_.barcode; //#9564
    $k[$j++] = $_.options; //#9564
    bwipp_databaromni(); //#9564
    var _Y = $k[--$j]; //#9564
    $_[$k[--$j]] = _Y; //#9564
    $put($_.args, "opt", $_.options); //#9566
    $k[$j++] = $_.args; //#9569
    if (!$_.dontdraw) { //#9569
        bwipp_renmatrix(); //#9569
    } //#9569
    $_ = $__; //#9571
} //bwipp_databarstacked
function bwipp_databarstackedomni() {
    var $__ = $_; //#9600
    $_ = Object.assign({}, $_); //#9600
    $_.dontdraw = false; //#9603
    $k[$j++] = $_; //#9605
    bwipp_processoptions(); //#9605
    $_.options = $k[--$j]; //#9605
    $_.barcode = $k[--$j]; //#9606
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#9611
        $k[$j++] = 'bwipp.databarstackedomniBadAI#9610'; //#9610
        $k[$j++] = "GS1 DataBar Stacked Omnidirectional must begin with (01) application identifier"; //#9610
        bwipp_raiseerror(); //#9610
    } //#9610
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#9614
        $k[$j++] = 'bwipp.databarstackedomniBadLength#9613'; //#9613
        $k[$j++] = "GS1 DataBar Stacked Omnidirectional must be 13 or 14 digits"; //#9613
        bwipp_raiseerror(); //#9613
    } //#9613
    var _8 = $geti($_.barcode, 4, $_.barcode.length - 4); //#9615
    for (var _9 = 0, _A = _8.length; _9 < _A; _9++) { //#9619
        var _B = $get(_8, _9); //#9619
        if ((_B < 48) || (_B > 57)) { //#9618
            $k[$j++] = 'bwipp.databarstackedomniBadCharacter#9617'; //#9617
            $k[$j++] = "GS1 DataBar Stacked Omnidirectional must contain only digits"; //#9617
            bwipp_raiseerror(); //#9617
        } //#9617
    } //#9617
    $_.checksum = 0; //#9622
    for (var _C = 0; _C <= 12; _C += 1) { //#9626
        $_.i = _C; //#9624
        $k[$j++] = 'checksum'; //#9625
        $k[$j++] = $_.checksum; //#9625
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#9625
        if (($_.i % 2) == 0) { //#9625
            var _I = $k[--$j]; //#9625
            $k[$j++] = _I * 3; //#9625
        } //#9625
        var _J = $k[--$j]; //#9625
        var _K = $k[--$j]; //#9625
        $_[$k[--$j]] = $f(_K + _J); //#9625
    } //#9625
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#9627
    if ($_.barcode.length == 18) { //#9632
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#9631
            $k[$j++] = 'bwipp.databarstackedomniBadCheckDigit#9630'; //#9630
            $k[$j++] = "Incorrect GS1 DataBar Stacked Omnidirectional check digit provided"; //#9630
            bwipp_raiseerror(); //#9630
        } //#9630
    } //#9630
    var _R = $s(18); //#9633
    $puti(_R, 0, $_.barcode); //#9634
    $put(_R, 17, $_.checksum + 48); //#9635
    $_.barcode = _R; //#9636
    $put($_.options, "dontdraw", true); //#9639
    $put($_.options, "format", "stackedomni"); //#9640
    $k[$j++] = 'args'; //#9642
    $k[$j++] = $_.barcode; //#9642
    $k[$j++] = $_.options; //#9642
    bwipp_databaromni(); //#9642
    var _Y = $k[--$j]; //#9642
    $_[$k[--$j]] = _Y; //#9642
    $put($_.args, "opt", $_.options); //#9644
    $k[$j++] = $_.args; //#9647
    if (!$_.dontdraw) { //#9647
        bwipp_renmatrix(); //#9647
    } //#9647
    $_ = $__; //#9649
} //bwipp_databarstackedomni
function bwipp_databartruncated() {
    var $__ = $_; //#9678
    $_ = Object.assign({}, $_); //#9678
    $_.dontdraw = false; //#9681
    $k[$j++] = $_; //#9683
    bwipp_processoptions(); //#9683
    $_.options = $k[--$j]; //#9683
    $_.barcode = $k[--$j]; //#9684
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#9689
        $k[$j++] = 'bwipp.databartruncatedBadLength#9688'; //#9688
        $k[$j++] = "GS1 DataBar Truncated must be 13 or 14 digits"; //#9688
        bwipp_raiseerror(); //#9688
    } //#9688
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#9692
        $k[$j++] = 'bwipp.databartruncatedBadAI#9691'; //#9691
        $k[$j++] = "GS1 DataBar Truncated must begin with (01) application identifier"; //#9691
        bwipp_raiseerror(); //#9691
    } //#9691
    var _8 = $geti($_.barcode, 4, $_.barcode.length - 4); //#9693
    for (var _9 = 0, _A = _8.length; _9 < _A; _9++) { //#9697
        var _B = $get(_8, _9); //#9697
        if ((_B < 48) || (_B > 57)) { //#9696
            $k[$j++] = 'bwipp.databartruncatedBadCharacter#9695'; //#9695
            $k[$j++] = "GS1 DataBar Truncated must contain only digits"; //#9695
            bwipp_raiseerror(); //#9695
        } //#9695
    } //#9695
    $_.checksum = 0; //#9700
    for (var _C = 0; _C <= 12; _C += 1) { //#9704
        $_.i = _C; //#9702
        $k[$j++] = 'checksum'; //#9703
        $k[$j++] = $_.checksum; //#9703
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#9703
        if (($_.i % 2) == 0) { //#9703
            var _I = $k[--$j]; //#9703
            $k[$j++] = _I * 3; //#9703
        } //#9703
        var _J = $k[--$j]; //#9703
        var _K = $k[--$j]; //#9703
        $_[$k[--$j]] = $f(_K + _J); //#9703
    } //#9703
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#9705
    if ($_.barcode.length == 18) { //#9710
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#9709
            $k[$j++] = 'bwipp.databartruncatedBadCheckDigit#9708'; //#9708
            $k[$j++] = "Incorrect GS1 DataBar Truncated check digit provided"; //#9708
            bwipp_raiseerror(); //#9708
        } //#9708
    } //#9708
    var _R = $s(18); //#9711
    $puti(_R, 0, $_.barcode); //#9712
    $put(_R, 17, $_.checksum + 48); //#9713
    $_.barcode = _R; //#9714
    $put($_.options, "dontdraw", true); //#9717
    $put($_.options, "format", "truncated"); //#9718
    $k[$j++] = 'args'; //#9720
    $k[$j++] = $_.barcode; //#9720
    $k[$j++] = $_.options; //#9720
    bwipp_databaromni(); //#9720
    var _Y = $k[--$j]; //#9720
    $_[$k[--$j]] = _Y; //#9720
    $put($_.args, "opt", $_.options); //#9722
    $k[$j++] = $_.args; //#9725
    if (!$_.dontdraw) { //#9725
        bwipp_renlinear(); //#9725
    } //#9725
    $_ = $__; //#9727
} //bwipp_databartruncated
function bwipp_databarlimited() {
    var $__ = $_; //#9757
    $_ = Object.assign({}, $_); //#9757
    $_.dontdraw = false; //#9760
    $_.height = 10 / 72; //#9761
    $_.linkage = false; //#9762
    $k[$j++] = $_; //#9764
    bwipp_processoptions(); //#9764
    $_.options = $k[--$j]; //#9764
    $_.barcode = $k[--$j]; //#9765
    if (($_.barcode.length != 17) && ($_.barcode.length != 18)) { //#9772
        $k[$j++] = 'bwipp.databarlimitedBadLength#9771'; //#9771
        $k[$j++] = "GS1 DataBar Limited must be 13 or 14 digits"; //#9771
        bwipp_raiseerror(); //#9771
    } //#9771
    if ($ne($geti($_.barcode, 0, 4), "\(01\)")) { //#9775
        $k[$j++] = 'bwipp.databarlimitedBadAI#9774'; //#9774
        $k[$j++] = "GS1 DataBar Limited must begin with (01) application identifier"; //#9774
        bwipp_raiseerror(); //#9774
    } //#9774
    var _7 = $get($_.barcode, 4); //#9776
    if ((_7 < 48) || (_7 > 49)) { //#9778
        $k[$j++] = 'bwipp.databarlimitedBadStartDigit#9777'; //#9777
        $k[$j++] = "GS1 DataBar Limited must begin with 0 or 1"; //#9777
        bwipp_raiseerror(); //#9777
    } //#9777
    var _A = $geti($_.barcode, 5, $_.barcode.length - 5); //#9779
    for (var _B = 0, _C = _A.length; _B < _C; _B++) { //#9783
        var _D = $get(_A, _B); //#9783
        if ((_D < 48) || (_D > 57)) { //#9782
            $k[$j++] = 'bwipp.databarlimitedBadCharacter#9781'; //#9781
            $k[$j++] = "GS1 DataBar Limited must contain only digits"; //#9781
            bwipp_raiseerror(); //#9781
        } //#9781
    } //#9781
    $_.checksum = 0; //#9786
    for (var _E = 0; _E <= 12; _E += 1) { //#9790
        $_.i = _E; //#9788
        $k[$j++] = 'checksum'; //#9789
        $k[$j++] = $_.checksum; //#9789
        $k[$j++] = $get($_.barcode, $_.i + 4) - 48; //#9789
        if (($_.i % 2) == 0) { //#9789
            var _K = $k[--$j]; //#9789
            $k[$j++] = _K * 3; //#9789
        } //#9789
        var _L = $k[--$j]; //#9789
        var _M = $k[--$j]; //#9789
        $_[$k[--$j]] = $f(_M + _L); //#9789
    } //#9789
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#9791
    if ($_.barcode.length == 18) { //#9796
        if ($get($_.barcode, 17) != ($_.checksum + 48)) { //#9795
            $k[$j++] = 'bwipp.databarlimitedBadCheckDigit#9794'; //#9794
            $k[$j++] = "Incorrect GS1 DataBar Limited check digit provided"; //#9794
            bwipp_raiseerror(); //#9794
        } //#9794
    } //#9794
    var _T = $s(18); //#9797
    $puti(_T, 0, $_.barcode); //#9798
    $put(_T, 17, $_.checksum + 48); //#9799
    $_.barcode = _T; //#9800
    $_.linkval = $a([2, 0, 1, 5, 1, 3, 3, 5, 3, 1, 0, 9, 6]); //#9804
    $k[$j++] = Infinity; //#9806
    var _Z = $geti($_.barcode, 4, 13); //#9807
    for (var _a = 0, _b = _Z.length; _a < _b; _a++) { //#9807
        $k[$j++] = $get(_Z, _a) - 48; //#9807
    } //#9807
    $_.binval = $a(); //#9807
    if ($_.linkage) { //#9814
        for (var _h = 0, _g = $_.binval.length - 1; _h <= _g; _h += 1) { //#9813
            $_.i = _h; //#9811
            $put($_.binval, $_.i, $f($get($_.binval, $_.i) + $get($_.linkval, $_.i))); //#9812
        } //#9812
    } //#9812
    $_.txt = $a($_.barcode.length); //#9817
    for (var _u = 0, _t = $_.barcode.length - 1; _u <= _t; _u += 1) { //#9821
        $_.i = _u; //#9819
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), 0, 0, "", 0])); //#9820
    } //#9820
    $_.ncr = function() {
        var _11 = $k[--$j]; //#9824
        var _12 = $k[--$j]; //#9824
        var _13 = $f(_12 - _11); //#9824
        if (_11 < $f(_12 - _11)) { //#9824
            var _ = _13; //#9824
            _13 = _11; //#9824
            _11 = _; //#9824
        } //#9824
        $k[$j++] = _13; //#9831
        $k[$j++] = 1; //#9831
        $k[$j++] = 1; //#9831
        for (var _15 = _12, _14 = _11 + 1; _15 >= _14; _15 -= 1) { //#9831
            var _16 = $k[--$j]; //#9827
            var _17 = $k[$j - 1]; //#9828
            var _18 = $k[$j - 2]; //#9828
            $k[$j++] = _16 * _15; //#9830
            if ($le(_17, _18)) { //#9830
                var _19 = $k[--$j]; //#9829
                var _1A = $k[--$j]; //#9829
                $k[$j++] = _1A + 1; //#9829
                $k[$j++] = ~~(_19 / _1A); //#9829
            } //#9829
        } //#9829
        for (;;) { //#9835
            var _1C = $k[$j - 2]; //#9833
            var _1D = $k[$j - 3]; //#9833
            if ($gt(_1C, _1D)) { //#9833
                break; //#9833
            } //#9833
            var _1E = $k[--$j]; //#9834
            var _1F = $k[--$j]; //#9834
            $k[$j++] = _1F + 1; //#9834
            $k[$j++] = ~~(_1E / _1F); //#9834
        } //#9834
        var _1G = $k[--$j]; //#9836
        $k[$j - 1] = _1G; //#9836
        var _1I = $k[--$j]; //#9836
        $k[$j - 1] = _1I; //#9836
    }; //#9836
    $_.getRSSwidths = function() {
        $_.oe = $k[--$j]; //#9840
        $_.el = $k[--$j]; //#9841
        $_.mw = $k[--$j]; //#9842
        $_.nm = $k[--$j]; //#9843
        $_.val = $k[--$j]; //#9844
        $_.out = $a($_.el); //#9845
        $_.mask = 0; //#9846
        for (var _1T = 0, _1S = $_.el - 2; _1T <= _1S; _1T += 1) { //#9873
            $_.bar = _1T; //#9848
            $_.ew = 1; //#9849
            var _1V = $_.bar; //#9850
            $_.mask = $_.mask | ((_1V < 0 ? 1 >>> -_1V : 1 << _1V)); //#9850
            for (;;) { //#9869
                $k[$j++] = 'sval'; //#9851
                $k[$j++] = $f($_.nm - $_.ew) - 1; //#9851
                $k[$j++] = $f($_.el - $_.bar) - 2; //#9851
                $_.ncr(); //#9851
                var _1a = $k[--$j]; //#9851
                $_[$k[--$j]] = _1a; //#9851
                if ($_.oe && $_.mask == 0 && $f($_.nm - $_.ew - $_.el * 2 + $_.bar * 2) >= -2) { //#9854
                    $k[$j++] = 'sval'; //#9853
                    $k[$j++] = $_.sval; //#9853
                    $k[$j++] = $f($_.nm - $_.ew - $_.el + $_.bar); //#9853
                    $k[$j++] = $f($_.el - $_.bar) - 2; //#9853
                    $_.ncr(); //#9853
                    var _1p = $k[--$j]; //#9853
                    var _1q = $k[--$j]; //#9853
                    $_[$k[--$j]] = $f(_1q - _1p); //#9853
                } //#9853
                if ($f($_.el - $_.bar) > 2) { //#9863
                    $_.lval = 0; //#9856
                    for (var _20 = $f($_.nm - $_.ew - $_.el + $_.bar) + 2, _1z = $_.mw + 1; _20 >= _1z; _20 -= 1) { //#9860
                        $k[$j++] = $f($_.nm - _20 - $_.ew) - 1; //#9858
                        $k[$j++] = $f($_.el - $_.bar) - 3; //#9858
                        $_.ncr(); //#9858
                        $_.lval = $f($k[--$j] + $_.lval); //#9859
                    } //#9859
                    $_.sval = $f($_.sval - $_.lval * ($_.el - $_.bar - 1)); //#9861
                } else { //#9863
                    if ($f($_.nm - $_.ew) > $_.mw) { //#9863
                        $_.sval = $_.sval - 1; //#9863
                    } //#9863
                } //#9863
                $_.val = $f($_.val - $_.sval); //#9865
                if ($_.val < 0) { //#9866
                    break; //#9866
                } //#9866
                $_.ew = $_.ew + 1; //#9867
                var _2K = $_.bar; //#9868
                $_.mask = $_.mask & (~((_2K < 0 ? 1 >>> -_2K : 1 << _2K))); //#9868
            } //#9868
            $_.val = $f($_.val + $_.sval); //#9870
            $_.nm = $f($_.nm - $_.ew); //#9871
            $put($_.out, $_.bar, $_.ew); //#9872
        } //#9872
        $put($_.out, $_.el - 1, $_.nm); //#9874
        $k[$j++] = $_.out; //#9875
    }; //#9875
    for (var _2W = 0; _2W <= 11; _2W += 1) { //#9882
        $_.i = _2W; //#9879
        var _2X = $_.binval; //#9880
        var _2Y = $_.i; //#9880
        $put(_2X, _2Y + 1, $f($get(_2X, _2Y + 1) + (($get($_.binval, $_.i) % 2013571) * 10))); //#9880
        $put($_.binval, $_.i, ~~($get($_.binval, $_.i) / 2013571)); //#9881
    } //#9881
    $_.d2 = $get($_.binval, 12) % 2013571; //#9883
    var _2k = $_.binval; //#9884
    $put(_2k, 12, ~~($get(_2k, 12) / 2013571)); //#9884
    $_.d1 = 0; //#9886
    $_.i = true; //#9887
    for (var _2m = 0; _2m <= 12; _2m += 1) { //#9897
        $_.j = _2m; //#9889
        var _2p = $get($_.binval, $_.j); //#9890
        $k[$j++] = _2p; //#9896
        if ((_2p == 0) && $_.i) { //#9895
            $j--; //#9892
        } else { //#9895
            $_.i = false; //#9894
            $k[$j++] = 'd1'; //#9895
            $k[$j++] = $_.d1; //#9895
            var _2t = $k[$j - 3]; //#9895
            var _2u = $k[$j - 1]; //#9895
            $_[$k[$j - 2]] = $f(_2u + (_2t * (~~$pow(10, 12 - $_.j)))); //#9895
            $j -= 3; //#9895
        } //#9895
    } //#9895
    $_.tab267 = $a([183063, 0, 17, 9, 6, 3, 6538, 28, 820063, 183064, 13, 13, 5, 4, 875, 728, 1000775, 820064, 9, 17, 3, 6, 28, 6454, 1491020, 1000776, 15, 11, 5, 4, 2415, 203, 1979844, 1491021, 11, 15, 4, 5, 203, 2408, 1996938, 1979845, 19, 7, 8, 1, 17094, 1, 2013570, 1996939, 7, 19, 1, 8, 1, 16632]); //#9908
    $_.i = 0; //#9911
    for (;;) { //#9921
        if ($_.d1 <= $get($_.tab267, $_.i)) { //#9919
            $aload($geti($_.tab267, $_.i + 1, 7)); //#9913
            $_.d1te = $k[--$j]; //#9914
            $_.d1to = $k[--$j]; //#9914
            $_.d1mwe = $k[--$j]; //#9915
            $_.d1mwo = $k[--$j]; //#9915
            $_.d1ele = $k[--$j]; //#9916
            $_.d1elo = $k[--$j]; //#9916
            $_.d1gs = $k[--$j]; //#9917
            break; //#9918
        } //#9918
        $_.i = $_.i + 8; //#9920
    } //#9920
    $_.i = 0; //#9923
    for (;;) { //#9933
        if ($_.d2 <= $get($_.tab267, $_.i)) { //#9931
            $aload($geti($_.tab267, $_.i + 1, 7)); //#9925
            $_.d2te = $k[--$j]; //#9926
            $_.d2to = $k[--$j]; //#9926
            $_.d2mwe = $k[--$j]; //#9927
            $_.d2mwo = $k[--$j]; //#9927
            $_.d2ele = $k[--$j]; //#9928
            $_.d2elo = $k[--$j]; //#9928
            $_.d2gs = $k[--$j]; //#9929
            break; //#9930
        } //#9930
        $_.i = $_.i + 8; //#9932
    } //#9932
    $k[$j++] = 'd1wo'; //#9935
    $k[$j++] = ~~($f($_.d1 - $_.d1gs) / $_.d1te); //#9935
    $k[$j++] = $_.d1elo; //#9935
    $k[$j++] = $_.d1mwo; //#9935
    $k[$j++] = 7; //#9935
    $k[$j++] = false; //#9935
    $_.getRSSwidths(); //#9935
    var _3X = $k[--$j]; //#9935
    $_[$k[--$j]] = _3X; //#9935
    $k[$j++] = 'd1we'; //#9936
    $k[$j++] = $f($_.d1 - $_.d1gs) % $_.d1te; //#9936
    $k[$j++] = $_.d1ele; //#9936
    $k[$j++] = $_.d1mwe; //#9936
    $k[$j++] = 7; //#9936
    $k[$j++] = true; //#9936
    $_.getRSSwidths(); //#9936
    var _3e = $k[--$j]; //#9936
    $_[$k[--$j]] = _3e; //#9936
    $k[$j++] = 'd2wo'; //#9937
    $k[$j++] = ~~($f($_.d2 - $_.d2gs) / $_.d2te); //#9937
    $k[$j++] = $_.d2elo; //#9937
    $k[$j++] = $_.d2mwo; //#9937
    $k[$j++] = 7; //#9937
    $k[$j++] = false; //#9937
    $_.getRSSwidths(); //#9937
    var _3l = $k[--$j]; //#9937
    $_[$k[--$j]] = _3l; //#9937
    $k[$j++] = 'd2we'; //#9938
    $k[$j++] = $f($_.d2 - $_.d2gs) % $_.d2te; //#9938
    $k[$j++] = $_.d2ele; //#9938
    $k[$j++] = $_.d2mwe; //#9938
    $k[$j++] = 7; //#9938
    $k[$j++] = true; //#9938
    $_.getRSSwidths(); //#9938
    var _3s = $k[--$j]; //#9938
    $_[$k[--$j]] = _3s; //#9938
    $_.d1w = $a(14); //#9940
    for (var _3v = 0; _3v <= 6; _3v += 1) { //#9945
        $_.i = _3v; //#9942
        $put($_.d1w, $_.i * 2, $get($_.d1wo, $_.i)); //#9943
        $put($_.d1w, ($_.i * 2) + 1, $get($_.d1we, $_.i)); //#9944
    } //#9944
    $_.d2w = $a(14); //#9947
    for (var _47 = 0; _47 <= 6; _47 += 1) { //#9952
        $_.i = _47; //#9949
        $put($_.d2w, $_.i * 2, $get($_.d2wo, $_.i)); //#9950
        $put($_.d2w, ($_.i * 2) + 1, $get($_.d2we, $_.i)); //#9951
    } //#9951
    $k[$j++] = Infinity; //#9954
    $aload($_.d1w); //#9955
    $aload($_.d2w); //#9956
    $_.widths = $a(); //#9956
    $_.checkweights = $a([1, 3, 9, 27, 81, 65, 17, 51, 64, 14, 42, 37, 22, 66, 20, 60, 2, 6, 18, 54, 73, 41, 34, 13, 39, 28, 84, 74]); //#9963
    $k[$j++] = Infinity; //#9965
    for (var _4M = 0; _4M <= 43; _4M += 1) { //#9966
        $k[$j++] = _4M; //#9966
    } //#9966
    $k[$j++] = 45; //#9968
    $k[$j++] = 52; //#9968
    $k[$j++] = 57; //#9968
    for (var _4N = 63; _4N <= 66; _4N += 1) { //#9968
        $k[$j++] = _4N; //#9968
    } //#9968
    for (var _4O = 73; _4O <= 79; _4O += 1) { //#9969
        $k[$j++] = _4O; //#9969
    } //#9969
    $k[$j++] = 82; //#9971
    for (var _4P = 126; _4P <= 130; _4P += 1) { //#9971
        $k[$j++] = _4P; //#9971
    } //#9971
    $k[$j++] = 132; //#9973
    for (var _4Q = 141; _4Q <= 146; _4Q += 1) { //#9973
        $k[$j++] = _4Q; //#9973
    } //#9973
    for (var _4R = 210; _4R <= 217; _4R += 1) { //#9974
        $k[$j++] = _4R; //#9974
    } //#9974
    $k[$j++] = 220; //#9976
    for (var _4S = 316; _4S <= 320; _4S += 1) { //#9976
        $k[$j++] = _4S; //#9976
    } //#9976
    $k[$j++] = 322; //#9978
    $k[$j++] = 323; //#9978
    $k[$j++] = 326; //#9978
    $k[$j++] = 337; //#9978
    $_.checkseq = $a(); //#9978
    $_.checksum = 0; //#9982
    for (var _4V = 0; _4V <= 27; _4V += 1) { //#9986
        $_.i = _4V; //#9984
        $_.checksum = $f($_.checksum + ($get($_.widths, $_.i) * $get($_.checkweights, $_.i))); //#9985
    } //#9985
    $_.checksum = $_.checksum % 89; //#9987
    $_.seq = $get($_.checkseq, $_.checksum); //#9988
    $k[$j++] = 'swidths'; //#9989
    $k[$j++] = ~~($_.seq / 21); //#9989
    $k[$j++] = 8; //#9989
    $k[$j++] = 3; //#9989
    $k[$j++] = 6; //#9989
    $k[$j++] = false; //#9989
    $_.getRSSwidths(); //#9989
    var _4i = $k[--$j]; //#9989
    $_[$k[--$j]] = _4i; //#9989
    $k[$j++] = 'bwidths'; //#9990
    $k[$j++] = $_.seq % 21; //#9990
    $k[$j++] = 8; //#9990
    $k[$j++] = 3; //#9990
    $k[$j++] = 6; //#9990
    $k[$j++] = false; //#9990
    $_.getRSSwidths(); //#9990
    var _4l = $k[--$j]; //#9990
    $_[$k[--$j]] = _4l; //#9990
    $_.checkwidths = $a([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1]); //#9992
    for (var _4o = 0; _4o <= 5; _4o += 1) { //#9997
        $_.i = _4o; //#9994
        $put($_.checkwidths, $_.i * 2, $get($_.swidths, $_.i)); //#9995
        $put($_.checkwidths, ($_.i * 2) + 1, $get($_.bwidths, $_.i)); //#9996
    } //#9996
    $k[$j++] = Infinity; //#9999
    $k[$j++] = 1; //#10001
    $aload($_.d1w); //#10001
    $aload($_.checkwidths); //#10002
    $aload($_.d2w); //#10003
    $k[$j++] = 1; //#10004
    $k[$j++] = 1; //#10004
    $k[$j++] = 5; //#10004
    $_.sbs = $a(); //#10004
    $k[$j++] = Infinity; //#10008
    var _53 = $_.sbs; //#10010
    $k[$j++] = Infinity; //#10011
    for (var _55 = 0, _56 = ~~(($_.sbs.length + 1) / 2); _55 < _56; _55++) { //#10011
        $k[$j++] = $_.height; //#10011
    } //#10011
    var _58 = $a(); //#10011
    $k[$j++] = Infinity; //#10012
    for (var _5A = 0, _5B = ~~(($_.sbs.length + 1) / 2); _5A < _5B; _5A++) { //#10012
        $k[$j++] = 0; //#10012
    } //#10012
    var _5C = $a(); //#10012
    $k[$j++] = 'ren'; //#10017
    $k[$j++] = 'renlinear'; //#10017
    $k[$j++] = 'sbs'; //#10017
    $k[$j++] = _53; //#10017
    $k[$j++] = 'bhs'; //#10017
    $k[$j++] = _58; //#10017
    $k[$j++] = 'bbs'; //#10017
    $k[$j++] = _5C; //#10017
    $k[$j++] = 'txt'; //#10017
    $k[$j++] = $_.txt; //#10017
    $k[$j++] = 'textxalign'; //#10017
    $k[$j++] = "center"; //#10017
    $k[$j++] = 'borderleft'; //#10017
    $k[$j++] = 0; //#10017
    $k[$j++] = 'borderright'; //#10017
    $k[$j++] = 0; //#10017
    $k[$j++] = 'opt'; //#10017
    $k[$j++] = $_.options; //#10017
    var _5F = $d(); //#10017
    $k[$j++] = _5F; //#10020
    if (!$_.dontdraw) { //#10020
        bwipp_renlinear(); //#10020
    } //#10020
    $_ = $__; //#10024
} //bwipp_databarlimited
function bwipp_databarexpanded() {
    var $__ = $_; //#10056
    $_ = Object.assign({}, $_); //#10056
    $_.parse = false; //#10059
    $_.dontlint = false; //#10060
    $_.lintreqs = true; //#10061
    $_.dontdraw = false; //#10062
    $_.height = 34 / 72; //#10063
    $_.format = "expanded"; //#10064
    $_.segments = -1; //#10065
    $_.linkage = false; //#10066
    $_.barxmult = 34; //#10067
    $k[$j++] = $_; //#10069
    bwipp_processoptions(); //#10069
    $_.options = $k[--$j]; //#10069
    $_.barcode = $k[--$j]; //#10070
    if ($_.barxmult < 34) { //#10076
        $k[$j++] = 'bwipp.databarStackedOmniBarXmult#10075'; //#10075
        $k[$j++] = "GS1 DataBar Expanded Stacked must have a barxmult of at least 34"; //#10075
        bwipp_raiseerror(); //#10075
    } //#10075
    if ($ne($_.format, "expanded") && $ne($_.format, "expandedstacked")) { //#10080
        $k[$j++] = 'bwipp.databarexpandedBadFormat#10079'; //#10079
        $k[$j++] = "Valid formats are expanded and expandedstacked"; //#10079
        bwipp_raiseerror(); //#10079
    } //#10079
    if ($_.segments == -1) { //#10086
        var _7 = $eq($_.format, "expandedstacked") ? 4 : 22; //#10083
        $_.segments = _7; //#10083
    } else { //#10086
        if ((($_.segments < 2) || ($_.segments > 22)) || (($_.segments % 2) != 0)) { //#10087
            $k[$j++] = 'bwipp.gs1databarexpandedBadSegments#10086'; //#10086
            $k[$j++] = "The number of segments must be even from 2 to 22"; //#10086
            bwipp_raiseerror(); //#10086
        } //#10086
    } //#10086
    $k[$j++] = $_.barcode; //#10090
    $k[$j++] = 'ai'; //#10090
    bwipp_gs1process(); //#10090
    $_.fncs = $k[--$j]; //#10091
    $_.vals = $k[--$j]; //#10092
    $_.ais = $k[--$j]; //#10093
    for (;;) { //#10181
        if ($_.ais.length == 2) { //#10105
            if ($eq($get($_.ais, 0), "01") && $eq($get($_.ais, 1), "3103")) { //#10104
                if ($eq($geti($get($_.vals, 0), 0, 1), "9") && ($cvi($get($_.vals, 1)) <= 32767)) { //#10103
                    $k[$j++] = "0100"; //#10102
                    $k[$j++] = false; //#10102
                    break; //#10102
                } //#10102
            } //#10102
        } //#10102
        if ($_.ais.length == 2) { //#10114
            if ($eq($get($_.ais, 0), "01") && $eq($get($_.ais, 1), "3202")) { //#10113
                if ($eq($geti($get($_.vals, 0), 0, 1), "9") && ($cvi($get($_.vals, 1)) <= 9999)) { //#10112
                    $k[$j++] = "0101"; //#10111
                    $k[$j++] = false; //#10111
                    break; //#10111
                } //#10111
            } //#10111
        } //#10111
        if ($_.ais.length == 2) { //#10123
            if ($eq($get($_.ais, 0), "01") && $eq($get($_.ais, 1), "3203")) { //#10122
                if ($eq($geti($get($_.vals, 0), 0, 1), "9") && ($cvi($get($_.vals, 1)) <= 22767)) { //#10121
                    $k[$j++] = "0101"; //#10120
                    $k[$j++] = false; //#10120
                    break; //#10120
                } //#10120
            } //#10120
        } //#10120
        var _k = $_.ais.length; //#10126
        if ((_k == 2) || (_k == 3)) { //#10157
            $k[$j++] = 'ai310x'; //#10127
            $k[$j++] = false; //#10127
            for (var _l = 3100; _l <= 3109; _l += 1) { //#10127
                var _q = $k[--$j]; //#10127
                $k[$j++] = _q || $eq($cvrs($s(4), _l, 10), $get($_.ais, 1)); //#10127
            } //#10127
            var _r = $k[--$j]; //#10127
            $_[$k[--$j]] = _r; //#10127
            $k[$j++] = 'ai320x'; //#10128
            $k[$j++] = false; //#10128
            for (var _t = 3200; _t <= 3209; _t += 1) { //#10128
                var _y = $k[--$j]; //#10128
                $k[$j++] = _y || $eq($cvrs($s(4), _t, 10), $get($_.ais, 1)); //#10128
            } //#10128
            var _z = $k[--$j]; //#10128
            $_[$k[--$j]] = _z; //#10128
            if ($_.ais.length == 3) { //#10132
                var _12 = $a(["11", "13", "15", "17"]); //#10130
                $k[$j++] = 'aibad'; //#10130
                $k[$j++] = true; //#10130
                for (var _13 = 0, _14 = _12.length; _13 < _14; _13++) { //#10130
                    var _18 = $k[--$j]; //#10130
                    $k[$j++] = _18 && $ne($get(_12, _13), $get($_.ais, 2)); //#10130
                } //#10130
                var _19 = $k[--$j]; //#10130
                $_[$k[--$j]] = _19; //#10130
            } else { //#10132
                $_.aibad = false; //#10132
            } //#10132
            if ($eq($get($_.ais, 0), "01") && ($_.ai310x || $_.ai320x) && !$_.aibad) { //#10156
                if ($_.ais.length == 3) { //#10153
                    var _1P = $cvi($geti($get($_.vals, 2), 2, 2)); //#10138
                    var _1T = $cvi($geti($get($_.vals, 2), 4, 2)); //#10139
                    if ((($eq($geti($get($_.vals, 0), 0, 1), "9") && ($cvi($get($_.vals, 1)) <= 99999)) && ((_1P >= 1) && (_1P <= 12))) && ((_1T >= 0) && (_1T <= 31))) { //#10148
                        if ($_.ai310x && $eq($get($_.ais, 2), "11")) { //#10140
                            $k[$j++] = "0111000"; //#10140
                            $k[$j++] = false; //#10140
                            break; //#10140
                        } //#10140
                        if ($_.ai320x && $eq($get($_.ais, 2), "11")) { //#10141
                            $k[$j++] = "0111001"; //#10141
                            $k[$j++] = false; //#10141
                            break; //#10141
                        } //#10141
                        if ($_.ai310x && $eq($get($_.ais, 2), "13")) { //#10142
                            $k[$j++] = "0111010"; //#10142
                            $k[$j++] = false; //#10142
                            break; //#10142
                        } //#10142
                        if ($_.ai320x && $eq($get($_.ais, 2), "13")) { //#10143
                            $k[$j++] = "0111011"; //#10143
                            $k[$j++] = false; //#10143
                            break; //#10143
                        } //#10143
                        if ($_.ai310x && $eq($get($_.ais, 2), "15")) { //#10144
                            $k[$j++] = "0111100"; //#10144
                            $k[$j++] = false; //#10144
                            break; //#10144
                        } //#10144
                        if ($_.ai320x && $eq($get($_.ais, 2), "15")) { //#10145
                            $k[$j++] = "0111101"; //#10145
                            $k[$j++] = false; //#10145
                            break; //#10145
                        } //#10145
                        if ($_.ai310x && $eq($get($_.ais, 2), "17")) { //#10146
                            $k[$j++] = "0111110"; //#10146
                            $k[$j++] = false; //#10146
                            break; //#10146
                        } //#10146
                        if ($_.ai320x && $eq($get($_.ais, 2), "17")) { //#10147
                            $k[$j++] = "0111111"; //#10147
                            $k[$j++] = false; //#10147
                            break; //#10147
                        } //#10147
                    } //#10147
                } else { //#10153
                    if ($eq($geti($get($_.vals, 0), 0, 1), "9") && ($cvi($get($_.vals, 1)) <= 99999)) { //#10154
                        if ($_.ai310x) { //#10152
                            $k[$j++] = "0111000"; //#10152
                            $k[$j++] = false; //#10152
                            break; //#10152
                        } //#10152
                        if ($_.ai320x) { //#10153
                            $k[$j++] = "0111001"; //#10153
                            $k[$j++] = false; //#10153
                            break; //#10153
                        } //#10153
                    } //#10153
                } //#10153
            } //#10153
        } //#10153
        if ($_.ais.length >= 2) { //#10165
            $k[$j++] = 'ai392x'; //#10161
            $k[$j++] = false; //#10161
            for (var _20 = 3920; _20 <= 3923; _20 += 1) { //#10161
                var _25 = $k[--$j]; //#10161
                $k[$j++] = _25 || $eq($cvrs($s(4), _20, 10), $get($_.ais, 1)); //#10161
            } //#10161
            var _26 = $k[--$j]; //#10161
            $_[$k[--$j]] = _26; //#10161
            if ($eq($get($_.ais, 0), "01") && $_.ai392x) { //#10164
                if ($eq($geti($get($_.vals, 0), 0, 1), "9")) { //#10163
                    $k[$j++] = "01100"; //#10163
                    $k[$j++] = true; //#10163
                    break; //#10163
                } //#10163
            } //#10163
        } //#10163
        if ($_.ais.length >= 2) { //#10173
            $k[$j++] = 'ai393x'; //#10169
            $k[$j++] = false; //#10169
            for (var _2F = 3930; _2F <= 3933; _2F += 1) { //#10169
                var _2K = $k[--$j]; //#10169
                $k[$j++] = _2K || $eq($cvrs($s(4), _2F, 10), $get($_.ais, 1)); //#10169
            } //#10169
            var _2L = $k[--$j]; //#10169
            $_[$k[--$j]] = _2L; //#10169
            if ($eq($get($_.ais, 0), "01") && $_.ai393x) { //#10172
                if ($eq($geti($get($_.vals, 0), 0, 1), "9")) { //#10171
                    $k[$j++] = "01101"; //#10171
                    $k[$j++] = true; //#10171
                    break; //#10171
                } //#10171
            } //#10171
        } //#10171
        if ($eq($get($_.ais, 0), "01")) { //#10176
            $k[$j++] = "1"; //#10176
            $k[$j++] = true; //#10176
            break; //#10176
        } //#10176
        $k[$j++] = "00"; //#10179
        $k[$j++] = true; //#10179
        break; //#10179
    } //#10179
    $_.gpfallow = $k[--$j]; //#10182
    $_.method = $k[--$j]; //#10183
    $_.conv12to40 = function() {
        var _2Y = $strcpy($s(40), "0000000000000000000000000000000000000000"); //#10186
        var _2Z = $k[--$j]; //#10187
        var _2d = $cvrs($s(10), $cvi($geti(_2Z, 0, 3)), 2); //#10187
        $puti($geti(_2Y, 0, 10), 10 - _2d.length, _2d); //#10187
        var _2h = $cvrs($s(10), $cvi($geti(_2Z, 3, 3)), 2); //#10188
        $puti($geti(_2Y, 10, 10), 10 - _2h.length, _2h); //#10188
        var _2l = $cvrs($s(10), $cvi($geti(_2Z, 6, 3)), 2); //#10189
        $puti($geti(_2Y, 20, 10), 10 - _2l.length, _2l); //#10189
        var _2p = $cvrs($s(10), $cvi($geti(_2Z, 9, 3)), 2); //#10190
        $puti($geti(_2Y, 30, 10), 10 - _2p.length, _2p); //#10190
        $k[$j++] = _2Y; //#10191
    }; //#10191
    $_.conv13to44 = function() {
        var _2r = $strcpy($s(44), "00000000000000000000000000000000000000000000"); //#10195
        var _2s = $k[--$j]; //#10196
        var _2w = $cvrs($s(4), $cvi($geti(_2s, 0, 1)), 2); //#10196
        $puti($geti(_2r, 0, 4), 4 - _2w.length, _2w); //#10196
        $k[$j++] = _2r; //#10197
        $k[$j++] = _2r; //#10197
        $k[$j++] = $geti(_2s, 1, 12); //#10197
        $_.conv12to40(); //#10197
        var _2y = $k[--$j]; //#10197
        $puti($k[--$j], 4, _2y); //#10197
    }; //#10197
    $_.tobin = function() {
        var _31 = $s($k[--$j]); //#10201
        $k[$j++] = _31; //#10201
        for (var _33 = 0, _32 = _31.length - 1; _33 <= _32; _33 += 1) { //#10201
            var _34 = $k[$j - 1]; //#10201
            $put(_34, _33, 48); //#10201
        } //#10201
        var _35 = $k[$j - 1]; //#10202
        $k[$j++] = _35; //#10202
        var _36 = $k[$j - 3]; //#10202
        var _37 = $k[$j - 1]; //#10202
        $k[$j - 3] = $k[$j - 2]; //#10202
        $j -= 2; //#10202
        var _39 = $cvrs($s(_37.length), _36, 2); //#10202
        $puti(_37, _37.length - _39.length, _39); //#10202
    }; //#10202
    $_.fnc1 = -1; //#10205
    $_.lnumeric = -2; //#10205
    $_.lalphanumeric = -3; //#10205
    $_.liso646 = -4; //#10205
    if ($eq($_.method, "00")) { //#10210
        $_.cdf = $a([]); //#10208
        $_.gpf = $a([]); //#10209
    } //#10209
    if ($eq($_.method, "1")) { //#10219
        $k[$j++] = 'cdf'; //#10213
        $k[$j++] = $geti($get($_.vals, 0), 0, 13); //#10213
        $_.conv13to44(); //#10213
        var _3H = $k[--$j]; //#10213
        $_[$k[--$j]] = _3H; //#10213
        $k[$j++] = Infinity; //#10214
        var _3J = $_.cdf; //#10214
        for (var _3K = 0, _3L = _3J.length; _3K < _3L; _3K++) { //#10214
            $k[$j++] = $get(_3J, _3K) - 48; //#10214
        } //#10214
        $_.cdf = $a(); //#10214
        $_.gpf = $a([]); //#10215
        $_.ais = $geti($_.ais, 1, $_.ais.length - 1); //#10216
        $_.vals = $geti($_.vals, 1, $_.vals.length - 1); //#10217
        $_.fncs = $geti($_.fncs, 1, $_.fncs.length - 1); //#10218
    } //#10218
    if ($eq($_.method, "0100")) { //#10230
        $_.cdf = $s(55); //#10222
        $k[$j++] = $_.cdf; //#10223
        $k[$j++] = 0; //#10223
        $k[$j++] = $geti($get($_.vals, 0), 1, 12); //#10223
        $_.conv12to40(); //#10223
        var _3e = $k[--$j]; //#10223
        var _3f = $k[--$j]; //#10223
        $puti($k[--$j], _3f, _3e); //#10223
        $k[$j++] = $_.cdf; //#10224
        $k[$j++] = 40; //#10224
        $k[$j++] = $cvi($get($_.vals, 1)); //#10224
        $k[$j++] = 15; //#10224
        $_.tobin(); //#10224
        var _3k = $k[--$j]; //#10224
        var _3l = $k[--$j]; //#10224
        $puti($k[--$j], _3l, _3k); //#10224
        $k[$j++] = Infinity; //#10225
        var _3n = $_.cdf; //#10225
        for (var _3o = 0, _3p = _3n.length; _3o < _3p; _3o++) { //#10225
            $k[$j++] = $get(_3n, _3o) - 48; //#10225
        } //#10225
        $_.cdf = $a(); //#10225
        $_.gpf = $a([]); //#10226
        $_.ais = $a([]); //#10227
        $_.vals = $a([]); //#10228
        $_.fncs = $a([]); //#10229
    } //#10229
    if ($eq($_.method, "0101")) { //#10246
        $_.cdf = $s(55); //#10233
        $k[$j++] = $_.cdf; //#10234
        $k[$j++] = 0; //#10234
        $k[$j++] = $geti($get($_.vals, 0), 1, 12); //#10234
        $_.conv12to40(); //#10234
        var _42 = $k[--$j]; //#10234
        var _43 = $k[--$j]; //#10234
        $puti($k[--$j], _43, _42); //#10234
        if ($eq($get($_.ais, 1), "3202")) { //#10238
            $k[$j++] = $cvi($get($_.vals, 1)); //#10236
            $k[$j++] = 15; //#10236
            $_.tobin(); //#10236
        } else { //#10238
            $k[$j++] = $cvi($get($_.vals, 1)) + 10000; //#10238
            $k[$j++] = 15; //#10238
            $_.tobin(); //#10238
        } //#10238
        $puti($_.cdf, 40, $k[--$j]); //#10240
        $k[$j++] = Infinity; //#10241
        var _4D = $_.cdf; //#10241
        for (var _4E = 0, _4F = _4D.length; _4E < _4F; _4E++) { //#10241
            $k[$j++] = $get(_4D, _4E) - 48; //#10241
        } //#10241
        $_.cdf = $a(); //#10241
        $_.gpf = $a([]); //#10242
        $_.ais = $a([]); //#10243
        $_.vals = $a([]); //#10244
        $_.fncs = $a([]); //#10245
    } //#10245
    if ($_.method.length == 7) { //#10268
        $_.cdf = $s(76); //#10249
        $k[$j++] = $_.cdf; //#10250
        $k[$j++] = 0; //#10250
        $k[$j++] = $geti($get($_.vals, 0), 1, 12); //#10250
        $_.conv12to40(); //#10250
        var _4S = $k[--$j]; //#10250
        var _4T = $k[--$j]; //#10250
        $puti($k[--$j], _4T, _4S); //#10250
        var _4V = $s(6); //#10251
        $puti(_4V, 0, $geti($get($_.ais, 1), 3, 1)); //#10251
        $puti(_4V, 1, $geti($get($_.vals, 1), 1, 5)); //#10252
        $k[$j++] = $cvi(_4V); //#10252
        $k[$j++] = 20; //#10252
        $_.tobin(); //#10252
        $puti($_.cdf, 40, $k[--$j]); //#10253
        if ($_.ais.length == 3) { //#10260
            var _4g = $get($_.vals, 2); //#10255
            $k[$j++] = ($cvi($geti(_4g, 0, 2)) * 384) + ((($cvi($geti(_4g, 2, 2)) - 1) * 32) + $cvi($geti(_4g, 4, 2))); //#10258
        } else { //#10260
            $k[$j++] = 38400; //#10260
        } //#10260
        $k[$j++] = 16; //#10262
        $_.tobin(); //#10262
        $puti($_.cdf, 60, $k[--$j]); //#10262
        $k[$j++] = Infinity; //#10263
        var _4m = $_.cdf; //#10263
        for (var _4n = 0, _4o = _4m.length; _4n < _4o; _4n++) { //#10263
            $k[$j++] = $get(_4m, _4n) - 48; //#10263
        } //#10263
        $_.cdf = $a(); //#10263
        $_.gpf = $a([]); //#10264
        $_.ais = $a([]); //#10265
        $_.vals = $a([]); //#10266
        $_.fncs = $a([]); //#10267
    } //#10267
    if ($eq($_.method, "01100")) { //#10279
        $_.cdf = $s(42); //#10271
        $k[$j++] = $_.cdf; //#10272
        $k[$j++] = 0; //#10272
        $k[$j++] = $geti($get($_.vals, 0), 1, 12); //#10272
        $_.conv12to40(); //#10272
        var _51 = $k[--$j]; //#10272
        var _52 = $k[--$j]; //#10272
        $puti($k[--$j], _52, _51); //#10272
        $k[$j++] = $_.cdf; //#10273
        $k[$j++] = 40; //#10273
        $k[$j++] = $cvi($geti($get($_.ais, 1), 3, 1)); //#10273
        $k[$j++] = 2; //#10273
        $_.tobin(); //#10273
        var _58 = $k[--$j]; //#10273
        var _59 = $k[--$j]; //#10273
        $puti($k[--$j], _59, _58); //#10273
        $k[$j++] = Infinity; //#10274
        var _5B = $_.cdf; //#10274
        for (var _5C = 0, _5D = _5B.length; _5C < _5D; _5C++) { //#10274
            $k[$j++] = $get(_5B, _5C) - 48; //#10274
        } //#10274
        $_.cdf = $a(); //#10274
        $k[$j++] = Infinity; //#10275
        $forall($get($_.vals, 1)); //#10275
        if ($_.ais.length > 2) { //#10275
            $k[$j++] = $_.fnc1; //#10275
        } //#10275
        $_.gpf = $a(); //#10275
        $_.ais = $geti($_.ais, 2, $_.ais.length - 2); //#10276
        $_.vals = $geti($_.vals, 2, $_.vals.length - 2); //#10277
        $_.fncs = $geti($_.fncs, 2, $_.fncs.length - 2); //#10278
    } //#10278
    if ($eq($_.method, "01101")) { //#10291
        $_.cdf = $s(52); //#10282
        $k[$j++] = $_.cdf; //#10283
        $k[$j++] = 0; //#10283
        $k[$j++] = $geti($get($_.vals, 0), 1, 12); //#10283
        $_.conv12to40(); //#10283
        var _5a = $k[--$j]; //#10283
        var _5b = $k[--$j]; //#10283
        $puti($k[--$j], _5b, _5a); //#10283
        $k[$j++] = $_.cdf; //#10284
        $k[$j++] = 40; //#10284
        $k[$j++] = $cvi($geti($get($_.ais, 1), 3, 1)); //#10284
        $k[$j++] = 2; //#10284
        $_.tobin(); //#10284
        var _5h = $k[--$j]; //#10284
        var _5i = $k[--$j]; //#10284
        $puti($k[--$j], _5i, _5h); //#10284
        $k[$j++] = $_.cdf; //#10285
        $k[$j++] = 42; //#10285
        $k[$j++] = $cvi($geti($get($_.vals, 1), 0, 3)); //#10285
        $k[$j++] = 10; //#10285
        $_.tobin(); //#10285
        var _5o = $k[--$j]; //#10285
        var _5p = $k[--$j]; //#10285
        $puti($k[--$j], _5p, _5o); //#10285
        $k[$j++] = Infinity; //#10286
        var _5r = $_.cdf; //#10286
        for (var _5s = 0, _5t = _5r.length; _5s < _5t; _5s++) { //#10286
            $k[$j++] = $get(_5r, _5s) - 48; //#10286
        } //#10286
        $_.cdf = $a(); //#10286
        $k[$j++] = Infinity; //#10287
        var _5x = $get($_.vals, 1); //#10287
        var _5y = $geti(_5x, 3, _5x.length - 3); //#10287
        for (var _5z = 0, _60 = _5y.length; _5z < _60; _5z++) { //#10287
            $k[$j++] = $get(_5y, _5z); //#10287
        } //#10287
        if ($_.ais.length > 2) { //#10287
            $k[$j++] = $_.fnc1; //#10287
        } //#10287
        $_.gpf = $a(); //#10287
        $_.ais = $geti($_.ais, 2, $_.ais.length - 2); //#10288
        $_.vals = $geti($_.vals, 2, $_.vals.length - 2); //#10289
        $_.fncs = $geti($_.fncs, 2, $_.fncs.length - 2); //#10290
    } //#10290
    if ($_.gpfallow) { //#10297
        $_.vlf = $a(2); //#10295
    } else { //#10297
        $_.vlf = $a([]); //#10297
    } //#10297
    $k[$j++] = Infinity; //#10301
    for (var _6H = 0; _6H <= 119; _6H += 1) { //#10310
        var _6J = $strcpy($s(2), "00"); //#10303
        var _6L = $cvrs($s(2), _6H, 11); //#10303
        $puti(_6J, 2 - _6L.length, _6L); //#10304
        $k[$j++] = _6H; //#10305
        $k[$j++] = _6J; //#10305
        if ($get(_6J, 0) == 65) { //#10305
            var _6N = $k[$j - 1]; //#10305
            $put(_6N, 0, 94); //#10305
        } //#10305
        var _6O = $k[$j - 1]; //#10306
        if ($get(_6O, 1) == 65) { //#10306
            var _6Q = $k[$j - 1]; //#10306
            $put(_6Q, 1, 94); //#10306
        } //#10306
        var _6R = $k[--$j]; //#10307
        var _6U = $strcpy($s(7), "0000000"); //#10308
        var _6W = $cvrs($s(7), $k[--$j] + 8, 2); //#10308
        $puti(_6U, 7 - _6W.length, _6W); //#10309
        $k[$j++] = _6R; //#10309
        $k[$j++] = _6U; //#10309
    } //#10309
    $k[$j++] = $_.lalphanumeric; //#10311
    $k[$j++] = "0000"; //#10311
    $_.numeric = $d(); //#10312
    $k[$j++] = Infinity; //#10314
    for (var _6Z = 48; _6Z <= 57; _6Z += 1) { //#10315
        $k[$j++] = _6Z; //#10315
        $k[$j++] = _6Z - 43; //#10315
        $k[$j++] = 5; //#10315
        $_.tobin(); //#10315
    } //#10315
    $k[$j++] = $_.fnc1; //#10317
    $k[$j++] = "01111"; //#10317
    for (var _6b = 65; _6b <= 90; _6b += 1) { //#10317
        $k[$j++] = _6b; //#10317
        $k[$j++] = _6b - 33; //#10317
        $k[$j++] = 6; //#10317
        $_.tobin(); //#10317
    } //#10317
    $k[$j++] = 42; //#10319
    $k[$j++] = "111010"; //#10319
    for (var _6c = 44; _6c <= 47; _6c += 1) { //#10319
        $k[$j++] = _6c; //#10319
        $k[$j++] = _6c + 15; //#10319
        $k[$j++] = 6; //#10319
        $_.tobin(); //#10319
    } //#10319
    $k[$j++] = $_.lnumeric; //#10321
    $k[$j++] = "000"; //#10321
    $k[$j++] = $_.liso646; //#10321
    $k[$j++] = "00100"; //#10321
    $_.alphanumeric = $d(); //#10322
    $k[$j++] = Infinity; //#10324
    for (var _6g = 48; _6g <= 57; _6g += 1) { //#10325
        $k[$j++] = _6g; //#10325
        $k[$j++] = _6g - 43; //#10325
        $k[$j++] = 5; //#10325
        $_.tobin(); //#10325
    } //#10325
    $k[$j++] = $_.fnc1; //#10327
    $k[$j++] = "01111"; //#10327
    for (var _6i = 65; _6i <= 90; _6i += 1) { //#10327
        $k[$j++] = _6i; //#10327
        $k[$j++] = _6i - 1; //#10327
        $k[$j++] = 7; //#10327
        $_.tobin(); //#10327
    } //#10327
    for (var _6j = 97; _6j <= 122; _6j += 1) { //#10328
        $k[$j++] = _6j; //#10328
        $k[$j++] = _6j - 7; //#10328
        $k[$j++] = 7; //#10328
        $_.tobin(); //#10328
    } //#10328
    $k[$j++] = 33; //#10331
    $k[$j++] = "11101000"; //#10331
    $k[$j++] = 34; //#10331
    $k[$j++] = "11101001"; //#10331
    for (var _6k = 37; _6k <= 47; _6k += 1) { //#10331
        $k[$j++] = _6k; //#10331
        $k[$j++] = _6k + 197; //#10331
        $k[$j++] = 8; //#10331
        $_.tobin(); //#10331
    } //#10331
    for (var _6l = 58; _6l <= 63; _6l += 1) { //#10332
        $k[$j++] = _6l; //#10332
        $k[$j++] = _6l + 187; //#10332
        $k[$j++] = 8; //#10332
        $_.tobin(); //#10332
    } //#10332
    $k[$j++] = 95; //#10336
    $k[$j++] = "11111011"; //#10336
    $k[$j++] = 32; //#10336
    $k[$j++] = "11111100"; //#10336
    $k[$j++] = $_.lnumeric; //#10336
    $k[$j++] = "000"; //#10336
    $k[$j++] = $_.lalphanumeric; //#10336
    $k[$j++] = "00100"; //#10336
    $_.iso646 = $d(); //#10337
    for (var _6s = 0, _6r = $_.ais.length - 1; _6s <= _6r; _6s += 1) { //#10356
        $_.i = _6s; //#10342
        $_.ai = $get($_.ais, $_.i); //#10343
        $_.val = $get($_.vals, $_.i); //#10344
        var _72 = $a(($_.gpf.length + $_.ai.length) + $_.val.length); //#10345
        $puti(_72, 0, $_.gpf); //#10346
        $k[$j++] = _72; //#10347
        $k[$j++] = _72; //#10347
        $k[$j++] = $_.gpf.length; //#10347
        var _76 = Infinity; //#10347
        var _77 = $_.ai; //#10347
        $k[$j++] = _76; //#10347
        $forall(_77); //#10347
        var _78 = $a(); //#10347
        var _79 = $k[--$j]; //#10347
        $puti($k[--$j], _79, _78); //#10347
        var _7B = $k[$j - 1]; //#10348
        $k[$j++] = _7B; //#10348
        $k[$j++] = $_.gpf.length + $_.ai.length; //#10348
        var _7F = Infinity; //#10348
        var _7G = $_.val; //#10348
        $k[$j++] = _7F; //#10348
        $forall(_7G); //#10348
        var _7H = $a(); //#10348
        var _7I = $k[--$j]; //#10348
        $puti($k[--$j], _7I, _7H); //#10348
        $_.gpf = $k[--$j]; //#10349
        if (($_.i != ($_.ais.length - 1)) && $get($_.fncs, $_.i)) { //#10355
            var _7R = $a($_.gpf.length + 1); //#10351
            $puti(_7R, 0, $_.gpf); //#10352
            $put(_7R, $_.gpf.length, $_.fnc1); //#10353
            $_.gpf = _7R; //#10354
        } //#10354
    } //#10354
    $_.rembits = function() {
        var _7V = $k[--$j]; //#10360
        var _7W = 48; //#10362
        var _7X = (~~$ceil(_7V / 12)) * 12; //#10362
        if (((~~$ceil(_7V / 12)) * 12) < 48) { //#10362
            var _ = _7W; //#10362
            _7W = _7X; //#10362
            _7X = _; //#10362
        } //#10362
        var _7Y = ~~(_7X / 12); //#10363
        $k[$j++] = _7V; //#10367
        $k[$j++] = _7X; //#10367
        $k[$j++] = _7Y; //#10367
        if ((_7Y % $_.segments) == 1) { //#10366
            var _7a = $k[--$j]; //#10364
            $k[$j - 1] = (_7a + 1) * 12; //#10364
        } else { //#10366
            $j--; //#10366
        } //#10366
        var _7c = $k[--$j]; //#10368
        var _7d = $k[--$j]; //#10368
        $k[$j++] = $f(_7c - _7d); //#10368
    }; //#10368
    $_.encode = function() {
        var _7e = $k[$j - 1]; //#10372
        if ($ne(_7e, 'raw')) { //#10372
            var _7f = $k[--$j]; //#10372
            var _7h = $get(_7f, $k[--$j]); //#10372
            $k[$j++] = _7h; //#10372
        } else { //#10372
            $j--; //#10372
        } //#10372
        var _7i = $k[$j - 1]; //#10373
        if ((_7i.length + $_.j) >= 252) { //#10375
            $j--; //#10374
            $k[$j++] = 'bwipp.gs1databarexpandedTooLong#10374'; //#10374
            $k[$j++] = "Maximum length exceeded"; //#10374
            bwipp_raiseerror(); //#10374
        } //#10374
        $k[$j++] = Infinity; //#10376
        var _7k = $k[--$j]; //#10376
        var _7l = $k[--$j]; //#10376
        $k[$j++] = _7k; //#10376
        $forall(_7l, function() { //#10376
            var _7m = $k[--$j]; //#10376
            $k[$j++] = _7m - 48; //#10376
        }); //#10376
        var _7n = $a(); //#10376
        $puti($_.gpfenc, $_.j, _7n); //#10378
        $_.j = _7n.length + $_.j; //#10379
    }; //#10379
    $k[$j++] = Infinity; //#10383
    for (var _7s = 0, _7t = $_.gpf.length; _7s < _7t; _7s++) { //#10383
        $k[$j++] = 0; //#10383
    } //#10383
    $k[$j++] = 0; //#10383
    $k[$j++] = -1; //#10383
    $_.numericruns = $a(); //#10383
    $k[$j++] = Infinity; //#10384
    for (var _7w = 0, _7x = $_.gpf.length; _7w < _7x; _7w++) { //#10384
        $k[$j++] = 0; //#10384
    } //#10384
    $k[$j++] = 0; //#10384
    $_.alphanumericruns = $a(); //#10384
    $k[$j++] = Infinity; //#10385
    for (var _80 = 0, _81 = $_.gpf.length; _80 < _81; _80++) { //#10385
        $k[$j++] = 0; //#10385
    } //#10385
    $k[$j++] = 9999; //#10385
    $_.nextiso646only = $a(); //#10385
    for (var _84 = $_.gpf.length - 1; _84 >= 0; _84 -= 1) { //#10407
        $_.i = _84; //#10387
        var _89 = $strcpy($s(2), "00"); //#10389
        var _8C = $get($_.gpf, $_.i); //#10390
        $k[$j++] = $get($_.gpf, $_.i); //#10390
        $k[$j++] = _89; //#10390
        $k[$j++] = _89; //#10390
        $k[$j++] = 0; //#10390
        $k[$j++] = _8C; //#10390
        if (_8C == $_.fnc1) { //#10390
            $k[$j - 1] = 94; //#10390
        } //#10390
        var _8E = $k[--$j]; //#10390
        var _8F = $k[--$j]; //#10390
        $put($k[--$j], _8F, _8E); //#10390
        if ($_.i < ($_.gpf.length - 1)) { //#10391
            var _8J = $k[$j - 1]; //#10391
            var _8M = $get($_.gpf, $_.i + 1); //#10391
            $k[$j++] = _8J; //#10391
            $k[$j++] = 1; //#10391
            $k[$j++] = _8M; //#10391
            if (_8M == $_.fnc1) { //#10391
                $k[$j - 1] = 94; //#10391
            } //#10391
            var _8O = $k[--$j]; //#10391
            var _8P = $k[--$j]; //#10391
            $put($k[--$j], _8P, _8O); //#10391
        } //#10391
        var _8T = $get($_.numeric, $k[--$j]) !== undefined; //#10392
        if (_8T) { //#10395
            $put($_.numericruns, $_.i, $get($_.numericruns, $_.i + 2) + 2); //#10393
        } else { //#10395
            $put($_.numericruns, $_.i, 0); //#10395
        } //#10395
        var _8b = $k[$j - 1]; //#10397
        var _8d = $get($_.alphanumeric, _8b) !== undefined; //#10397
        if (_8d) { //#10400
            $put($_.alphanumericruns, $_.i, $get($_.alphanumericruns, $_.i + 1) + 1); //#10398
        } else { //#10400
            $put($_.alphanumericruns, $_.i, 0); //#10400
        } //#10400
        var _8l = $k[--$j]; //#10402
        var _8n = $get($_.iso646, _8l) !== undefined; //#10402
        var _8p = $get($_.alphanumeric, _8l) !== undefined; //#10402
        if (_8n && (!_8p)) { //#10405
            $put($_.nextiso646only, $_.i, 0); //#10403
        } else { //#10405
            $put($_.nextiso646only, $_.i, $get($_.nextiso646only, $_.i + 1) + 1); //#10405
        } //#10405
    } //#10405
    $_.gpfenc = $a(252); //#10410
    $_.i = 0; //#10411
    $_.j = 0; //#10411
    $_.mode = "numeric"; //#10411
    for (;;) { //#10501
        if ($_.i == $_.gpf.length) { //#10413
            break; //#10413
        } //#10413
        for (;;) { //#10500
            if ($eq($_.mode, "numeric")) { //#10449
                if ($_.i <= ($_.gpf.length - 2)) { //#10446
                    var _93 = $s(2); //#10418
                    var _96 = $get($_.gpf, $_.i); //#10419
                    $k[$j++] = _93; //#10419
                    $k[$j++] = _93; //#10419
                    $k[$j++] = 0; //#10419
                    $k[$j++] = _96; //#10419
                    if (_96 == $_.fnc1) { //#10419
                        $k[$j - 1] = 94; //#10419
                    } //#10419
                    var _98 = $k[--$j]; //#10419
                    var _99 = $k[--$j]; //#10419
                    $put($k[--$j], _99, _98); //#10419
                    var _9B = $k[$j - 1]; //#10420
                    var _9E = $get($_.gpf, $_.i + 1); //#10420
                    $k[$j++] = _9B; //#10420
                    $k[$j++] = 1; //#10420
                    $k[$j++] = _9E; //#10420
                    if (_9E == $_.fnc1) { //#10420
                        $k[$j - 1] = 94; //#10420
                    } //#10420
                    var _9G = $k[--$j]; //#10420
                    var _9H = $k[--$j]; //#10420
                    $put($k[--$j], _9H, _9G); //#10420
                    var _9J = $k[$j - 1]; //#10421
                    var _9L = $get($_.numeric, _9J) !== undefined; //#10421
                    if (_9L) { //#10425
                        $k[$j++] = $_.numeric; //#10422
                        $_.encode(); //#10422
                        $_.i = $_.i + 2; //#10423
                        break; //#10424
                    } //#10424
                    $k[$j - 1] = $_.lalphanumeric; //#10427
                    $k[$j++] = $_.numeric; //#10427
                    $_.encode(); //#10427
                    $_.mode = "alphanumeric"; //#10428
                    break; //#10429
                } else { //#10446
                    var _9S = $get($_.gpf, $_.i); //#10431
                    if ((_9S < 48) || (_9S > 57)) { //#10435
                        $k[$j++] = $_.lalphanumeric; //#10432
                        $k[$j++] = $_.numeric; //#10432
                        $_.encode(); //#10432
                        $_.mode = "alphanumeric"; //#10433
                        break; //#10434
                    } //#10434
                    $k[$j++] = 'rem'; //#10436
                    $k[$j++] = ((((12 + 1) + $_.method.length) + $_.vlf.length) + $_.cdf.length) + $_.j; //#10436
                    $_.rembits(); //#10436
                    var _9Z = $k[--$j]; //#10436
                    $_[$k[--$j]] = _9Z; //#10436
                    if (($_.rem >= 4) && ($_.rem <= 6)) { //#10446
                        var _9g = $geti($strcpy($s(6), "000000"), 0, $_.rem); //#10438
                        var _9l = $cvrs($s(4), $get($_.gpf, $_.i) - 47, 2); //#10439
                        $puti(_9g, 4 - _9l.length, _9l); //#10440
                        $k[$j++] = _9g; //#10440
                        $k[$j++] = 'raw'; //#10440
                        $_.encode(); //#10440
                        $_.i = $_.i + 1; //#10441
                        break; //#10442
                    } else { //#10446
                        var _9n = $s(2); //#10444
                        $put(_9n, 0, $get($_.gpf, $_.i)); //#10444
                        $put(_9n, 1, 94); //#10444
                        $k[$j++] = _9n; //#10444
                        $k[$j++] = $_.numeric; //#10444
                        $_.encode(); //#10444
                        $_.i = $_.i + 1; //#10445
                        break; //#10446
                    } //#10446
                } //#10446
            } //#10446
            if ($eq($_.mode, "alphanumeric")) { //#10476
                if ($get($_.gpf, $_.i) == $_.fnc1) { //#10457
                    $k[$j++] = $_.fnc1; //#10453
                    $k[$j++] = $_.alphanumeric; //#10453
                    $_.encode(); //#10453
                    $_.mode = "numeric"; //#10454
                    $_.i = $_.i + 1; //#10455
                    break; //#10456
                } //#10456
                var _A3 = $get($_.gpf, $_.i); //#10458
                var _A5 = $get($_.iso646, _A3) !== undefined; //#10458
                var _A7 = $get($_.alphanumeric, _A3) !== undefined; //#10458
                if (_A5 && (!_A7)) { //#10462
                    $k[$j++] = $_.liso646; //#10459
                    $k[$j++] = $_.alphanumeric; //#10459
                    $_.encode(); //#10459
                    $_.mode = "iso646"; //#10460
                    break; //#10461
                } //#10461
                if ($get($_.numericruns, $_.i) >= 6) { //#10467
                    $k[$j++] = $_.lnumeric; //#10464
                    $k[$j++] = $_.alphanumeric; //#10464
                    $_.encode(); //#10464
                    $_.mode = "numeric"; //#10465
                    break; //#10466
                } //#10466
                var _AH = $get($_.numericruns, $_.i); //#10468
                if ((_AH >= 4) && ($f(_AH + $_.i) == $_.gpf.length)) { //#10472
                    $k[$j++] = $_.lnumeric; //#10469
                    $k[$j++] = $_.alphanumeric; //#10469
                    $_.encode(); //#10469
                    $_.mode = "numeric"; //#10470
                    break; //#10471
                } //#10471
                $k[$j++] = $get($_.gpf, $_.i); //#10473
                $k[$j++] = $_.alphanumeric; //#10473
                $_.encode(); //#10473
                $_.i = $_.i + 1; //#10474
                break; //#10475
            } //#10475
            if ($eq($_.mode, "iso646")) { //#10498
                if ($get($_.gpf, $_.i) == $_.fnc1) { //#10484
                    $k[$j++] = $_.fnc1; //#10480
                    $k[$j++] = $_.iso646; //#10480
                    $_.encode(); //#10480
                    $_.mode = "numeric"; //#10481
                    $_.i = $_.i + 1; //#10482
                    break; //#10483
                } //#10483
                if (($get($_.numericruns, $_.i) >= 4) && ($get($_.nextiso646only, $_.i) >= 10)) { //#10489
                    $k[$j++] = $_.lnumeric; //#10486
                    $k[$j++] = $_.iso646; //#10486
                    $_.encode(); //#10486
                    $_.mode = "numeric"; //#10487
                    break; //#10488
                } //#10488
                if (($get($_.alphanumericruns, $_.i) >= 5) && ($get($_.nextiso646only, $_.i) >= 10)) { //#10494
                    $k[$j++] = $_.lalphanumeric; //#10491
                    $k[$j++] = $_.iso646; //#10491
                    $_.encode(); //#10491
                    $_.mode = "alphanumeric"; //#10492
                    break; //#10493
                } //#10493
                $k[$j++] = $get($_.gpf, $_.i); //#10495
                $k[$j++] = $_.iso646; //#10495
                $_.encode(); //#10495
                $_.i = $_.i + 1; //#10496
                break; //#10497
            } //#10497
        } //#10497
    } //#10497
    $_.gpf = $geti($_.gpfenc, 0, $_.j); //#10502
    var _B1 = ((((1 + 12) + $_.method.length) + $_.vlf.length) + $_.cdf.length) + $_.gpf.length; //#10506
    $k[$j++] = _B1; //#10506
    $k[$j++] = _B1; //#10506
    $_.rembits(); //#10506
    var _B2 = $k[--$j]; //#10506
    $_.pad = $a(_B2); //#10506
    $k[$j++] = _B2; //#10513
    if ($_.vlf.length != 0) { //#10512
        var _B5 = $k[--$j]; //#10508
        var _B7 = ~~($f($k[--$j] + _B5) / 12); //#10509
        $put($_.vlf, 0, _B7 % 2); //#10509
        var _B9 = (_B7 <= 14) ? 0 : 1; //#10510
        $put($_.vlf, 1, _B9); //#10510
    } else { //#10512
        $j -= 2; //#10512
    } //#10512
    if ($_.pad.length > 0) { //#10522
        for (var _BE = 0, _BD = $_.pad.length - 1; _BE <= _BD; _BE += 5) { //#10518
            $_.i = _BE; //#10516
            var _BF = $_.pad; //#10517
            var _BG = $_.i; //#10517
            var _BH = $a([0, 0, 1, 0, 0]); //#10517
            var _BI = $_.pad; //#10517
            var _BJ = $_.i; //#10517
            var _BK = 5; //#10517
            var _BL = _BI.length - _BJ; //#10517
            if ((_BI.length - _BJ) > 5) { //#10517
                var _ = _BK; //#10517
                _BK = _BL; //#10517
                _BL = _; //#10517
            } //#10517
            $puti(_BF, _BG, $geti(_BH, 0, _BL)); //#10517
        } //#10517
        if ($eq($_.mode, "numeric")) { //#10521
            $k[$j++] = Infinity; //#10520
            $k[$j++] = 0; //#10520
            $k[$j++] = 0; //#10520
            $k[$j++] = 0; //#10520
            $k[$j++] = 0; //#10520
            $aload($_.pad); //#10520
            $_.pad = $geti($a(), 0, $_.pad.length); //#10520
        } //#10520
    } //#10520
    $k[$j++] = Infinity; //#10525
    var _BT = $_.linkage ? 1 : 0; //#10526
    $k[$j++] = _BT; //#10527
    $forall($_.method, function() { //#10527
        var _BV = $k[--$j]; //#10527
        $k[$j++] = _BV - 48; //#10527
    }); //#10527
    $aload($_.vlf); //#10528
    $aload($_.cdf); //#10529
    $aload($_.gpf); //#10530
    $aload($_.pad); //#10531
    $_.binval = $a(); //#10531
    $_.datalen = ~~($_.binval.length / 12); //#10533
    $_.ncr = function() {
        var _Bc = $k[--$j]; //#10536
        var _Bd = $k[--$j]; //#10536
        var _Be = $f(_Bd - _Bc); //#10536
        if (_Bc < $f(_Bd - _Bc)) { //#10536
            var _ = _Be; //#10536
            _Be = _Bc; //#10536
            _Bc = _; //#10536
        } //#10536
        $k[$j++] = _Be; //#10543
        $k[$j++] = 1; //#10543
        $k[$j++] = 1; //#10543
        for (var _Bg = _Bd, _Bf = _Bc + 1; _Bg >= _Bf; _Bg -= 1) { //#10543
            var _Bh = $k[--$j]; //#10539
            var _Bi = $k[$j - 1]; //#10540
            var _Bj = $k[$j - 2]; //#10540
            $k[$j++] = _Bh * _Bg; //#10542
            if ($le(_Bi, _Bj)) { //#10542
                var _Bk = $k[--$j]; //#10541
                var _Bl = $k[--$j]; //#10541
                $k[$j++] = _Bl + 1; //#10541
                $k[$j++] = ~~(_Bk / _Bl); //#10541
            } //#10541
        } //#10541
        for (;;) { //#10547
            var _Bn = $k[$j - 2]; //#10545
            var _Bo = $k[$j - 3]; //#10545
            if ($gt(_Bn, _Bo)) { //#10545
                break; //#10545
            } //#10545
            var _Bp = $k[--$j]; //#10546
            var _Bq = $k[--$j]; //#10546
            $k[$j++] = _Bq + 1; //#10546
            $k[$j++] = ~~(_Bp / _Bq); //#10546
        } //#10546
        var _Br = $k[--$j]; //#10548
        $k[$j - 1] = _Br; //#10548
        var _Bt = $k[--$j]; //#10548
        $k[$j - 1] = _Bt; //#10548
    }; //#10548
    $_.getRSSwidths = function() {
        $_.oe = $k[--$j]; //#10552
        $_.el = $k[--$j]; //#10553
        $_.mw = $k[--$j]; //#10554
        $_.nm = $k[--$j]; //#10555
        $_.val = $k[--$j]; //#10556
        $_.out = $a($_.el); //#10557
        $_.mask = 0; //#10558
        for (var _C4 = 0, _C3 = $_.el - 2; _C4 <= _C3; _C4 += 1) { //#10585
            $_.bar = _C4; //#10560
            $_.ew = 1; //#10561
            var _C6 = $_.bar; //#10562
            $_.mask = $_.mask | ((_C6 < 0 ? 1 >>> -_C6 : 1 << _C6)); //#10562
            for (;;) { //#10581
                $k[$j++] = 'sval'; //#10563
                $k[$j++] = $f($_.nm - $_.ew) - 1; //#10563
                $k[$j++] = $f($_.el - $_.bar) - 2; //#10563
                $_.ncr(); //#10563
                var _CB = $k[--$j]; //#10563
                $_[$k[--$j]] = _CB; //#10563
                if ($_.oe && $_.mask == 0 && $f($_.nm - $_.ew - $_.el * 2 + $_.bar * 2) >= -2) { //#10566
                    $k[$j++] = 'sval'; //#10565
                    $k[$j++] = $_.sval; //#10565
                    $k[$j++] = $f($_.nm - $_.ew - $_.el + $_.bar); //#10565
                    $k[$j++] = $f($_.el - $_.bar) - 2; //#10565
                    $_.ncr(); //#10565
                    var _CQ = $k[--$j]; //#10565
                    var _CR = $k[--$j]; //#10565
                    $_[$k[--$j]] = $f(_CR - _CQ); //#10565
                } //#10565
                if ($f($_.el - $_.bar) > 2) { //#10575
                    $_.lval = 0; //#10568
                    for (var _Cb = $f($_.nm - $_.ew - $_.el + $_.bar) + 2, _Ca = $_.mw + 1; _Cb >= _Ca; _Cb -= 1) { //#10572
                        $k[$j++] = $f($_.nm - _Cb - $_.ew) - 1; //#10570
                        $k[$j++] = $f($_.el - $_.bar) - 3; //#10570
                        $_.ncr(); //#10570
                        $_.lval = $f($k[--$j] + $_.lval); //#10571
                    } //#10571
                    $_.sval = $f($_.sval - $_.lval * ($_.el - $_.bar - 1)); //#10573
                } else { //#10575
                    if ($f($_.nm - $_.ew) > $_.mw) { //#10575
                        $_.sval = $_.sval - 1; //#10575
                    } //#10575
                } //#10575
                $_.val = $f($_.val - $_.sval); //#10577
                if ($_.val < 0) { //#10578
                    break; //#10578
                } //#10578
                $_.ew = $_.ew + 1; //#10579
                var _Cv = $_.bar; //#10580
                $_.mask = $_.mask & (~((_Cv < 0 ? 1 >>> -_Cv : 1 << _Cv))); //#10580
            } //#10580
            $_.val = $f($_.val + $_.sval); //#10582
            $_.nm = $f($_.nm - $_.ew); //#10583
            $put($_.out, $_.bar, $_.ew); //#10584
        } //#10584
        $put($_.out, $_.el - 1, $_.nm); //#10586
        $k[$j++] = $_.out; //#10587
    }; //#10587
    $_.tab174 = $a([347, 0, 12, 5, 7, 2, 87, 4, 1387, 348, 10, 7, 5, 4, 52, 20, 2947, 1388, 8, 9, 4, 5, 30, 52, 3987, 2948, 6, 11, 3, 6, 10, 104, 4191, 3988, 4, 13, 1, 8, 1, 204]); //#10597
    $_.dxw = $a($_.datalen); //#10600
    for (var _DD = 0, _DC = $_.datalen - 1; _DD <= _DC; _DD += 1) { //#10641
        $_.x = _DD; //#10604
        $_.d = $geti($_.binval, $_.x * 12, 12); //#10606
        $k[$j++] = 'd'; //#10607
        $k[$j++] = 0; //#10607
        for (var _DH = 0; _DH <= 11; _DH += 1) { //#10607
            $_.j = _DH; //#10607
            var _DM = $k[--$j]; //#10607
            $k[$j++] = $f(_DM + ((~~$pow(2, 11 - $_.j)) * $get($_.d, $_.j))); //#10607
        } //#10607
        var _DN = $k[--$j]; //#10607
        $_[$k[--$j]] = _DN; //#10607
        $_.j = 0; //#10609
        for (;;) { //#10619
            if ($le($_.d, $get($_.tab174, $_.j))) { //#10617
                $aload($geti($_.tab174, $_.j + 1, 7)); //#10611
                $_.dte = $k[--$j]; //#10612
                $_.dto = $k[--$j]; //#10612
                $_.dmwe = $k[--$j]; //#10613
                $_.dmwo = $k[--$j]; //#10613
                $_.dele = $k[--$j]; //#10614
                $_.delo = $k[--$j]; //#10614
                $_.dgs = $k[--$j]; //#10615
                break; //#10616
            } //#10616
            $_.j = $_.j + 8; //#10618
        } //#10618
        $k[$j++] = 'dwo'; //#10621
        $k[$j++] = ~~($f($_.d - $_.dgs) / $_.dte); //#10621
        $k[$j++] = $_.delo; //#10621
        $k[$j++] = $_.dmwo; //#10621
        $k[$j++] = 4; //#10621
        $k[$j++] = true; //#10621
        $_.getRSSwidths(); //#10621
        var _Dj = $k[--$j]; //#10621
        $_[$k[--$j]] = _Dj; //#10621
        $k[$j++] = 'dwe'; //#10622
        $k[$j++] = $f($_.d - $_.dgs) % $_.dte; //#10622
        $k[$j++] = $_.dele; //#10622
        $k[$j++] = $_.dmwe; //#10622
        $k[$j++] = 4; //#10622
        $k[$j++] = false; //#10622
        $_.getRSSwidths(); //#10622
        var _Dq = $k[--$j]; //#10622
        $_[$k[--$j]] = _Dq; //#10622
        $_.dw = $a(8); //#10624
        if (($_.x % 2) == 0) { //#10635
            for (var _Du = 0; _Du <= 3; _Du += 1) { //#10630
                $_.j = _Du; //#10627
                $put($_.dw, 7 - ($_.j * 2), $get($_.dwo, $_.j)); //#10628
                $put($_.dw, 6 - ($_.j * 2), $get($_.dwe, $_.j)); //#10629
            } //#10629
        } else { //#10635
            for (var _E5 = 0; _E5 <= 3; _E5 += 1) { //#10636
                $_.j = _E5; //#10633
                $put($_.dw, $_.j * 2, $get($_.dwo, $_.j)); //#10634
                $put($_.dw, ($_.j * 2) + 1, $get($_.dwe, $_.j)); //#10635
            } //#10635
        } //#10635
        $put($_.dxw, $_.x, $_.dw); //#10639
    } //#10639
    $_.finderwidths = $a([1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1, 1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5, 6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2]); //#10651
    $_.finderseq = $a([$a([0, 1]), $a([0, 3, 2]), $a([0, 5, 2, 7]), $a([0, 9, 2, 7, 4]), $a([0, 9, 2, 7, 6, 11]), $a([0, 9, 2, 7, 8, 11, 10]), $a([0, 1, 2, 3, 4, 5, 6, 7]), $a([0, 1, 2, 3, 4, 5, 6, 9, 8]), $a([0, 1, 2, 3, 4, 5, 6, 9, 10, 11]), $a([0, 1, 2, 3, 4, 7, 6, 9, 8, 11, 10])]); //#10664
    $_.seq = $get($_.finderseq, ~~(($_.datalen - 2) / 2)); //#10667
    $_.fxw = $a($_.seq.length); //#10668
    for (var _Ed = 0, _Ec = $_.seq.length - 1; _Ed <= _Ec; _Ed += 1) { //#10672
        $_.x = _Ed; //#10670
        $put($_.fxw, $_.x, $geti($_.finderwidths, $get($_.seq, $_.x) * 5, 5)); //#10671
    } //#10671
    $_.checkweights = $a([-1, -1, -1, -1, -1, -1, -1, -1, 77, 96, 32, 81, 27, 9, 3, 1, 20, 60, 180, 118, 143, 7, 21, 63, 205, 209, 140, 117, 39, 13, 145, 189, 193, 157, 49, 147, 19, 57, 171, 91, 132, 44, 85, 169, 197, 136, 186, 62, 185, 133, 188, 142, 4, 12, 36, 108, 50, 87, 29, 80, 97, 173, 128, 113, 150, 28, 84, 41, 123, 158, 52, 156, 166, 196, 206, 139, 187, 203, 138, 46, 76, 17, 51, 153, 37, 111, 122, 155, 146, 119, 110, 107, 106, 176, 129, 43, 16, 48, 144, 10, 30, 90, 59, 177, 164, 125, 112, 178, 200, 137, 116, 109, 70, 210, 208, 202, 184, 130, 179, 115, 190, 204, 68, 93, 31, 151, 191, 134, 148, 22, 66, 198, 172, 94, 71, 2, 40, 154, 192, 64, 162, 54, 18, 6, 120, 149, 25, 75, 14, 42, 126, 167, 175, 199, 207, 69, 23, 78, 26, 79, 103, 98, 83, 38, 114, 131, 182, 124, 159, 53, 88, 170, 127, 183, 61, 161, 55, 165, 73, 8, 24, 72, 5, 15, 89, 100, 174, 58, 160, 194, 135, 45]); //#10700
    $k[$j++] = Infinity; //#10703
    $forall($_.seq, function() { //#10704
        $aload($geti($_.checkweights, $k[--$j] * 16, 16)); //#10704
    }); //#10704
    var _Er = $a(); //#10704
    $_.checkweightseq = $geti(_Er, 8, _Er.length - 8); //#10705
    $k[$j++] = Infinity; //#10707
    var _Et = $_.dxw; //#10708
    for (var _Eu = 0, _Ev = _Et.length; _Eu < _Ev; _Eu++) { //#10708
        $aload($get(_Et, _Eu)); //#10708
    } //#10708
    $_.widths = $a(); //#10708
    $_.checksum = 0; //#10711
    for (var _F0 = 0, _Ez = $_.widths.length - 1; _F0 <= _Ez; _F0 += 1) { //#10715
        $_.i = _F0; //#10713
        $_.checksum = $f($_.checksum + ($get($_.widths, $_.i) * $get($_.checkweightseq, $_.i))); //#10714
    } //#10714
    $_.checksum = $f(($_.checksum % 211) + (($_.datalen - 3) * 211)); //#10716
    $_.i = 0; //#10718
    for (;;) { //#10728
        if ($_.checksum <= $get($_.tab174, $_.i)) { //#10726
            $aload($geti($_.tab174, $_.i + 1, 7)); //#10720
            $_.cte = $k[--$j]; //#10721
            $_.cto = $k[--$j]; //#10721
            $_.cmwe = $k[--$j]; //#10722
            $_.cmwo = $k[--$j]; //#10722
            $_.cele = $k[--$j]; //#10723
            $_.celo = $k[--$j]; //#10723
            $_.cgs = $k[--$j]; //#10724
            break; //#10725
        } //#10725
        $_.i = $_.i + 8; //#10727
    } //#10727
    $k[$j++] = 'cwo'; //#10730
    $k[$j++] = ~~($f($_.checksum - $_.cgs) / $_.cte); //#10730
    $k[$j++] = $_.celo; //#10730
    $k[$j++] = $_.cmwo; //#10730
    $k[$j++] = 4; //#10730
    $k[$j++] = true; //#10730
    $_.getRSSwidths(); //#10730
    var _FU = $k[--$j]; //#10730
    $_[$k[--$j]] = _FU; //#10730
    $k[$j++] = 'cwe'; //#10731
    $k[$j++] = $f($_.checksum - $_.cgs) % $_.cte; //#10731
    $k[$j++] = $_.cele; //#10731
    $k[$j++] = $_.cmwe; //#10731
    $k[$j++] = 4; //#10731
    $k[$j++] = false; //#10731
    $_.getRSSwidths(); //#10731
    var _Fb = $k[--$j]; //#10731
    $_[$k[--$j]] = _Fb; //#10731
    $_.cw = $a(8); //#10733
    for (var _Fe = 0; _Fe <= 3; _Fe += 1) { //#10738
        $_.i = _Fe; //#10735
        $put($_.cw, $_.i * 2, $get($_.cwo, $_.i)); //#10736
        $put($_.cw, ($_.i * 2) + 1, $get($_.cwe, $_.i)); //#10737
    } //#10737
    var _Fp = $a(22); //#10740
    $put(_Fp, 0, $_.cw); //#10741
    $puti(_Fp, 1, $_.dxw); //#10741
    $_.dxw = $geti(_Fp, 0, $_.datalen + 1); //#10742
    $_.datalen = $_.dxw.length; //#10743
    $_.rows = $a(~~$ceil($_.datalen / $_.segments)); //#10746
    $_.numrows = $_.rows.length; //#10747
    for (var _G1 = 0, _G0 = $_.numrows - 1; _G1 <= _G0; _G1 += 1) { //#10762
        $_.r = _G1; //#10749
        $k[$j++] = Infinity; //#10750
        if ((($_.segments % 4) != 0) && (($_.r % 2) == 1)) { //#10751
            $k[$j++] = 0; //#10751
        } //#10751
        $k[$j++] = 1; //#10759
        $k[$j++] = 1; //#10759
        for (var _G6 = 0, _G5 = $_.segments - 1; _G6 <= _G5; _G6 += 1) { //#10759
            $_.pos = _G6 + ($_.r * $_.segments); //#10754
            if ($_.pos < $_.datalen) { //#10758
                $aload($get($_.dxw, $_.pos)); //#10756
                if (($_.pos % 2) == 0) { //#10757
                    $aload($get($_.fxw, ~~($_.pos / 2))); //#10757
                } //#10757
            } //#10757
        } //#10757
        var _GI = $counttomark() + 2; //#10761
        $k[$j++] = 1; //#10761
        $k[$j++] = 1; //#10761
        $astore($a(_GI)); //#10761
        $put($_.rows, $_.r, $k[--$j]); //#10761
        $j--; //#10761
    } //#10761
    if ($ne($_.format, "expandedstacked")) { //#10894
        var _GP = $get($_.rows, 0); //#10767
        $_.sbs = $geti(_GP, 1, _GP.length - 1); //#10767
        $k[$j++] = Infinity; //#10769
        $k[$j++] = 1; //#10775
        for (var _GT = 0, _GS = $_.datalen - 1; _GT <= _GS; _GT += 1) { //#10775
            $_.i = _GT; //#10772
            $aload($get($_.dxw, $_.i)); //#10773
            if (($_.i % 2) == 0) { //#10774
                $aload($get($_.fxw, ~~($_.i / 2))); //#10774
            } //#10774
        } //#10774
        $k[$j++] = 1; //#10776
        $k[$j++] = 1; //#10776
        $_.sbs = $a(); //#10776
        $k[$j++] = Infinity; //#10779
        var _Gc = $_.sbs; //#10781
        $k[$j++] = Infinity; //#10782
        for (var _Ge = 0, _Gf = ~~(($_.sbs.length + 1) / 2); _Ge < _Gf; _Ge++) { //#10782
            $k[$j++] = $_.height; //#10782
        } //#10782
        var _Gh = $a(); //#10782
        $k[$j++] = Infinity; //#10783
        for (var _Gj = 0, _Gk = ~~(($_.sbs.length + 1) / 2); _Gj < _Gk; _Gj++) { //#10783
            $k[$j++] = 0; //#10783
        } //#10783
        var _Gl = $a(); //#10783
        $k[$j++] = 'ren'; //#10788
        $k[$j++] = 'renlinear'; //#10788
        $k[$j++] = 'sbs'; //#10788
        $k[$j++] = _Gc; //#10788
        $k[$j++] = 'bhs'; //#10788
        $k[$j++] = _Gh; //#10788
        $k[$j++] = 'bbs'; //#10788
        $k[$j++] = _Gl; //#10788
        $k[$j++] = 'borderleft'; //#10788
        $k[$j++] = 0; //#10788
        $k[$j++] = 'borderright'; //#10788
        $k[$j++] = 0; //#10788
        $k[$j++] = 'bordertop'; //#10788
        $k[$j++] = 0; //#10788
        $k[$j++] = 'borderbottom'; //#10788
        $k[$j++] = 0; //#10788
        $k[$j++] = 'opt'; //#10788
        $k[$j++] = $_.options; //#10788
        var _Gn = $d(); //#10788
        $k[$j++] = _Gn; //#10791
        if (!$_.dontdraw) { //#10791
            bwipp_renlinear(); //#10791
        } //#10791
    } else { //#10894
        $_.seps = $a($_.numrows); //#10795
        for (var _Gt = 0, _Gs = $_.numrows - 1; _Gt <= _Gs; _Gt += 1) { //#10851
            $_.r = _Gt; //#10798
            $_.row = $get($_.rows, $_.r); //#10799
            $k[$j++] = Infinity; //#10802
            for (var _Gz = 0, _Gy = $_.row.length - 1; _Gz <= _Gy; _Gz += 2) { //#10807
                $_.i = _Gz; //#10804
                for (var _H3 = 0, _H4 = $get($_.row, $_.i); _H3 < _H4; _H3++) { //#10805
                    $k[$j++] = 0; //#10805
                } //#10805
                if ($_.i < ($_.row.length - 1)) { //#10806
                    for (var _HA = 0, _HB = $get($_.row, $_.i + 1); _HA < _HB; _HA++) { //#10806
                        $k[$j++] = 1; //#10806
                    } //#10806
                } //#10806
            } //#10806
            $astore($a($counttomark())); //#10808
            $_.row = $k[--$j]; //#10808
            $k[$j - 1] = Infinity; //#10811
            $forall($_.row, function() { //#10811
                var _HG = $k[--$j]; //#10811
                $k[$j++] = 1 - _HG; //#10811
            }); //#10811
            $_.sep = $a(); //#10811
            $k[$j++] = Infinity; //#10812
            for (var _HK = 19, _HJ = $_.row.length - 13; _HK <= _HJ; _HK += 98) { //#10813
                $k[$j++] = _HK; //#10813
            } //#10813
            for (var _HN = 68, _HM = $_.row.length - 13; _HN <= _HM; _HN += 98) { //#10814
                $k[$j++] = _HN; //#10814
            } //#10814
            $_.finderpos = $a(); //#10814
            var _HP = $_.finderpos; //#10816
            for (var _HQ = 0, _HR = _HP.length; _HQ < _HR; _HQ++) { //#10830
                var _HS = $get(_HP, _HQ); //#10830
                for (var _HU = _HS, _HT = _HS + 14; _HU <= _HT; _HU += 1) { //#10829
                    $_.i = _HU; //#10818
                    if ($get($_.row, $_.i) == 0) { //#10826
                        if ($get($_.row, $_.i - 1) == 1) { //#10823
                            $k[$j++] = 1; //#10821
                        } else { //#10823
                            var _He = ($get($_.sep, $_.i - 1) == 0) ? 1 : 0; //#10823
                            $k[$j++] = _He; //#10823
                        } //#10823
                    } else { //#10826
                        $k[$j++] = 0; //#10826
                    } //#10826
                    $put($_.sep, $_.i, $k[--$j]); //#10828
                } //#10828
            } //#10828
            $puti($_.sep, 0, $a([0, 0, 0, 0])); //#10831
            $puti($_.sep, $_.row.length - 4, $a([0, 0, 0, 0])); //#10832
            if ((($_.segments % 4) == 0) && (($_.r % 2) == 1)) { //#10846
                if (($get($_.rows, $_.r).length != $get($_.rows, 0).length) && (($_.finderpos.length % 2) == 1)) { //#10844
                    $k[$j++] = Infinity; //#10838
                    $k[$j++] = 0; //#10838
                    $aload($_.row); //#10838
                    $_.row = $a(); //#10838
                    $k[$j++] = Infinity; //#10839
                    $k[$j++] = 0; //#10839
                    $aload($_.sep); //#10839
                    $_.sep = $a(); //#10839
                } else { //#10844
                    for (var _I0 = $_.row.length - 1; _I0 >= 0; _I0 -= 1) { //#10841
                        $k[$j++] = $get($_.row, _I0); //#10841
                    } //#10841
                    $astore($_.row); //#10842
                    $j--; //#10842
                    for (var _I5 = $_.sep.length - 1; _I5 >= 0; _I5 -= 1) { //#10843
                        $k[$j++] = $get($_.sep, _I5); //#10843
                    } //#10843
                    $astore($_.sep); //#10844
                    $j--; //#10844
                } //#10844
            } //#10844
            $put($_.rows, $_.r, $_.row); //#10848
            $put($_.seps, $_.r, $_.sep); //#10849
        } //#10849
        $_.pixx = $get($_.rows, 0).length; //#10854
        $k[$j++] = Infinity; //#10855
        for (var _II = 0, _IJ = $_.pixx; _II < _IJ; _II++) { //#10855
            $k[$j++] = 0; //#10855
        } //#10855
        var _IK = $a(); //#10855
        $puti(_IK, 0, $get($_.rows, $_.numrows - 1)); //#10855
        $put($_.rows, $_.numrows - 1, _IK); //#10856
        $k[$j++] = Infinity; //#10857
        for (var _IR = 0, _IS = $_.pixx; _IR < _IS; _IR++) { //#10857
            $k[$j++] = 0; //#10857
        } //#10857
        var _IT = $a(); //#10857
        $puti(_IT, 0, $get($_.seps, $_.numrows - 1)); //#10857
        $put($_.seps, $_.numrows - 1, _IT); //#10858
        $k[$j++] = Infinity; //#10861
        for (var _Ia = 0, _Ib = (~~($_.pixx / 2)) + 1; _Ia < _Ib; _Ia++) { //#10861
            $k[$j++] = 0; //#10861
            $k[$j++] = 1; //#10861
        } //#10861
        $_.sep = $geti($a(), 0, $_.pixx); //#10861
        $puti($_.sep, 0, $a([0, 0, 0, 0])); //#10862
        $puti($_.sep, $_.pixx - 4, $a([0, 0, 0, 0])); //#10863
        $k[$j++] = Infinity; //#10866
        for (var _Im = 0, _Il = $_.numrows - 1; _Im <= _Il; _Im += 1) { //#10877
            $_.r = _Im; //#10868
            if ($_.r != 0) { //#10871
                $aload($get($_.seps, $_.r)); //#10870
            } //#10870
            for (var _Is = 0, _It = $_.barxmult; _Is < _It; _Is++) { //#10872
                $aload($get($_.rows, $_.r)); //#10872
            } //#10872
            if ($_.r != ($_.numrows - 1)) { //#10876
                $aload($get($_.seps, $_.r)); //#10874
                $aload($_.sep); //#10875
            } //#10875
        } //#10875
        $_.pixs = $a(); //#10875
        var _JC = new Map([
            ["ren", 'renmatrix'],
            ["pixs", $_.pixs],
            ["pixx", $_.pixx],
            ["pixy", ~~($_.pixs.length / $_.pixx)],
            ["height", (~~($_.pixs.length / $_.pixx)) / 72],
            ["width", $_.pixx / 72],
            ["borderleft", 0],
            ["borderright", 0],
            ["bordertop", 0],
            ["borderbottom", 0],
            ["opt", $_.options]
        ]); //#10891
        $k[$j++] = _JC; //#10894
        if (!$_.dontdraw) { //#10894
            bwipp_renmatrix(); //#10894
        } //#10894
    } //#10894
    $_ = $__; //#10900
} //bwipp_databarexpanded
function bwipp_databarexpandedstacked() {
    var $__ = $_; //#10929
    $_ = Object.assign({}, $_); //#10929
    $_.dontdraw = false; //#10932
    $k[$j++] = $_; //#10934
    bwipp_processoptions(); //#10934
    $_.options = $k[--$j]; //#10934
    $_.barcode = $k[--$j]; //#10935
    $put($_.options, "dontdraw", true); //#10938
    $put($_.options, "format", "expandedstacked"); //#10939
    $k[$j++] = 'args'; //#10941
    $k[$j++] = $_.barcode; //#10941
    $k[$j++] = $_.options; //#10941
    bwipp_databarexpanded(); //#10941
    var _6 = $k[--$j]; //#10941
    $_[$k[--$j]] = _6; //#10941
    $put($_.args, "opt", $_.options); //#10943
    $k[$j++] = $_.args; //#10946
    if (!$_.dontdraw) { //#10946
        bwipp_renmatrix(); //#10946
    } //#10946
    $_ = $__; //#10948
} //bwipp_databarexpandedstacked
function bwipp_gs1northamericancoupon() {
    var $__ = $_; //#10978
    $_ = Object.assign({}, $_); //#10978
    $_.dontdraw = false; //#10981
    $_.includetext = false; //#10982
    $_.coupontextfont = "OCR-B"; //#10983
    $_.coupontextsize = 9; //#10984
    $_.coupontextxoffset = null; //#10985
    $_.coupontextyoffset = null; //#10986
    $_.parse = false; //#10987
    $_.dontlint = false; //#10988
    $_.lintreqs = true; //#10989
    $k[$j++] = $_; //#10991
    bwipp_processoptions(); //#10991
    $_.options = $k[--$j]; //#10991
    $_.barcode = $k[--$j]; //#10992
    $k[$j++] = $_.barcode; //#10994
    $k[$j++] = 'ai'; //#10994
    bwipp_gs1process(); //#10994
    $j--; //#10995
    $_.vals = $k[--$j]; //#10996
    $_.ais = $k[--$j]; //#10997
    if ($_.ais.length != 1) { //#11002
        $k[$j++] = 'bwipp.gs1northamericancouponBadAIStructure#11001'; //#11001
        $k[$j++] = "A GS1 North American Coupon should consist of a single AI (8110)"; //#11001
        bwipp_raiseerror(); //#11001
    } //#11001
    if ($ne($get($_.ais, 0), "8110")) { //#11005
        $k[$j++] = 'bwipp.gs1northamericancouponBadAIStructure#11004'; //#11004
        $k[$j++] = "A GS1 North American Coupon should consist of a single AI (8110)"; //#11004
        bwipp_raiseerror(); //#11004
    } //#11004
    $_.val = $get($_.vals, 0); //#11008
    $_.vli = $get($_.val, 0) - 48; //#11009
    if (($_.vli < 0) || ($_.vli > 6)) { //#11012
        $k[$j++] = 'bwipp.gs1northamericancouponBadVLI#11011'; //#11011
        $k[$j++] = "The AI (8110) data should start with a Company Prefix length indicator in the range 0 to 6"; //#11011
        bwipp_raiseerror(); //#11011
    } //#11011
    $_.gcp = $geti($_.val, 1, $_.vli + 6); //#11013
    $_.cod = $geti($_.val, $_.vli + 7, 6); //#11014
    $_.coupontext = $s($_.gcp.length + 7); //#11015
    $puti($_.coupontext, 0, $_.gcp); //#11016
    $puti($_.coupontext, $_.gcp.length, "-"); //#11017
    $puti($_.coupontext, $_.gcp.length + 1, $_.cod); //#11018
    $put($_.options, "dontdraw", true); //#11021
    $k[$j++] = 'args'; //#11023
    $k[$j++] = $_.barcode; //#11023
    $k[$j++] = $_.options; //#11023
    bwipp_databarexpandedstacked(); //#11023
    var _W = $k[--$j]; //#11023
    $_[$k[--$j]] = _W; //#11023
    if ($_.includetext) { //#11042
        if ($_.coupontextxoffset == null) { //#11029
            $_.coupontextxoffset = 0; //#11028
        } //#11028
        if ($_.coupontextyoffset == null) { //#11032
            $_.coupontextyoffset = ($get($_.args, "height") * 72) + 3; //#11031
        } //#11031
        var _e = $get($_.args, "txt") !== undefined; //#11033
        if (_e) { //#11040
            $_.txt = $get($_.args, "txt"); //#11034
            $_.newtxt = $a($_.txt.length + 1); //#11035
            $puti($_.newtxt, 0, $_.txt); //#11036
            $put($_.newtxt, $_.newtxt.length - 1, $a([$_.coupontext, $_.coupontextxoffset, $_.coupontextyoffset, $_.coupontextfont, $_.coupontextsize])); //#11037
            $put($_.args, "txt", $_.newtxt); //#11038
        } else { //#11040
            $put($_.args, "txt", $a([$a([$_.coupontext, $_.coupontextxoffset, $_.coupontextyoffset, $_.coupontextfont, $_.coupontextsize])])); //#11040
        } //#11040
    } //#11040
    $put($_.args, "opt", $_.options); //#11044
    $k[$j++] = $_.args; //#11047
    if (!$_.dontdraw) { //#11047
        bwipp_renmatrix(); //#11047
    } //#11047
    $_ = $__; //#11049
} //bwipp_gs1northamericancoupon
function bwipp_pharmacode() {
    var $__ = $_; //#11077
    $_ = Object.assign({}, $_); //#11077
    $_.dontdraw = false; //#11080
    $_.height = (8 * 2.835) / 72; //#11081
    $_.nwidth = 0.5 * 2.835; //#11082
    $_.wwidth = 1.5 * 2.835; //#11083
    $_.swidth = 1 * 2.835; //#11084
    $k[$j++] = $_; //#11086
    bwipp_processoptions(); //#11086
    $_.options = $k[--$j]; //#11086
    $_.barcode = $k[--$j]; //#11087
    if (($_.barcode.length < 1) || ($_.barcode.length > 6)) { //#11092
        $k[$j++] = 'bwipp.pharmacodeBadLength#11091'; //#11091
        $k[$j++] = "Pharmacode must be 1 to 6 digits"; //#11091
        bwipp_raiseerror(); //#11091
    } //#11091
    $forall($_.barcode, function() { //#11097
        var _5 = $k[--$j]; //#11094
        if ((_5 < 48) || (_5 > 57)) { //#11096
            $k[$j++] = 'bwipp.pharmacodeBadCharacter#11095'; //#11095
            $k[$j++] = "Pharmacode must contain only digits"; //#11095
            bwipp_raiseerror(); //#11095
        } //#11095
    }); //#11095
    var _7 = $cvi($_.barcode); //#11098
    if ((_7 < 3) || (_7 > 131070)) { //#11100
        $k[$j++] = 'bwipp.pharmacodeBadValue#11099'; //#11099
        $k[$j++] = "Pharmacode value must be between 3 and 131070"; //#11099
        bwipp_raiseerror(); //#11099
    } //#11099
    $_.txt = $a($_.barcode.length); //#11103
    for (var _C = 0, _B = $_.barcode.length - 1; _C <= _B; _C += 1) { //#11107
        $_.i = _C; //#11105
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), 0, 0, "", 0])); //#11106
    } //#11106
    $_.barcode = $cvrs($s(17), $cvi($_.barcode) + 1, 2); //#11110
    $_.barcode = $geti($_.barcode, 1, $_.barcode.length - 1); //#11111
    $_.barlen = $_.barcode.length; //#11113
    $_.sbs = $a($_.barlen * 2); //#11114
    for (var _U = 0, _T = $_.barlen - 1; _U <= _T; _U += 1) { //#11125
        $_.i = _U; //#11117
        $_.enc = $geti($_.barcode, $_.i, 1); //#11118
        if ($eq($_.enc, "0")) { //#11122
            $put($_.sbs, $_.i * 2, $_.nwidth); //#11120
        } else { //#11122
            $put($_.sbs, $_.i * 2, $_.wwidth); //#11122
        } //#11122
        $put($_.sbs, ($_.i * 2) + 1, $_.swidth); //#11124
    } //#11124
    $k[$j++] = Infinity; //#11128
    var _i = $_.sbs; //#11130
    $k[$j++] = Infinity; //#11131
    for (var _k = 0, _l = ~~(($_.sbs.length + 1) / 2); _k < _l; _k++) { //#11131
        $k[$j++] = $_.height; //#11131
    } //#11131
    var _n = $a(); //#11131
    $k[$j++] = Infinity; //#11132
    for (var _p = 0, _q = ~~(($_.sbs.length + 1) / 2); _p < _q; _p++) { //#11132
        $k[$j++] = 0; //#11132
    } //#11132
    var _r = $a(); //#11132
    $k[$j++] = 'ren'; //#11135
    $k[$j++] = 'renlinear'; //#11135
    $k[$j++] = 'sbs'; //#11135
    $k[$j++] = _i; //#11135
    $k[$j++] = 'bhs'; //#11135
    $k[$j++] = _n; //#11135
    $k[$j++] = 'bbs'; //#11135
    $k[$j++] = _r; //#11135
    $k[$j++] = 'txt'; //#11135
    $k[$j++] = $_.txt; //#11135
    $k[$j++] = 'textxalign'; //#11135
    $k[$j++] = "center"; //#11135
    $k[$j++] = 'opt'; //#11135
    $k[$j++] = $_.options; //#11135
    var _u = $d(); //#11135
    $k[$j++] = _u; //#11138
    if (!$_.dontdraw) { //#11138
        bwipp_renlinear(); //#11138
    } //#11138
    $_ = $__; //#11140
} //bwipp_pharmacode
function bwipp_pharmacode2() {
    var $__ = $_; //#11168
    $_ = Object.assign({}, $_); //#11168
    $_.dontdraw = false; //#11171
    $_.includetext = false; //#11172
    $_.height = 4; //#11173
    $k[$j++] = $_; //#11175
    bwipp_processoptions(); //#11175
    $_.options = $k[--$j]; //#11175
    $_.barcode = $k[--$j]; //#11176
    if (($_.barcode.length < 1) || ($_.barcode.length > 8)) { //#11181
        $k[$j++] = 'bwipp.pharmacode2BadLength#11180'; //#11180
        $k[$j++] = "Two-track Pharmacode must be 1 to 6 digits"; //#11180
        bwipp_raiseerror(); //#11180
    } //#11180
    $forall($_.barcode, function() { //#11186
        var _5 = $k[--$j]; //#11183
        if ((_5 < 48) || (_5 > 57)) { //#11185
            $k[$j++] = 'bwipp.pharmacode2badCharacter#11184'; //#11184
            $k[$j++] = "Two-track Pharmacode must contain only digits"; //#11184
            bwipp_raiseerror(); //#11184
        } //#11184
    }); //#11184
    var _7 = $cvi($_.barcode); //#11187
    if ((_7 < 4) || (_7 > 64570080)) { //#11189
        $k[$j++] = 'bwipp.pharmacode2badValue#11188'; //#11188
        $k[$j++] = "Two-track Pharmacode value must be between 4 and 64570080"; //#11188
        bwipp_raiseerror(); //#11188
    } //#11188
    $_.txt = $a($_.barcode.length); //#11192
    for (var _C = 0, _B = $_.barcode.length - 1; _C <= _B; _C += 1) { //#11196
        $_.i = _C; //#11194
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), 0, 0, "", 0])); //#11195
    } //#11195
    $_.encstr = $s(16); //#11198
    $_.sbs = $a(32); //#11199
    $_.bar = (1 / 25.4) * $_.height; //#11200
    $_.spc = (1 / 25.4) * 72; //#11201
    $_.i = 15; //#11204
    $_.val = $cvi($_.barcode); //#11205
    for (;;) { //#11212
        if ($_.val == 0) { //#11207
            break; //#11207
        } //#11207
        var _P = $_.val % 3; //#11208
        $_.val = ~~($f($_.val - $get($a([3, 1, 2]), _P)) / 3); //#11209
        $put($_.encstr, $_.i, $get($a([2, 0, 1]), _P)); //#11210
        $_.i = $_.i - 1; //#11211
    } //#11211
    $_.encstr = $geti($_.encstr, $_.i + 1, 15 - $_.i); //#11213
    $_.bhs = $a($_.encstr.length); //#11216
    $_.bbs = $a($_.encstr.length); //#11217
    for (var _i = 0, _h = $_.encstr.length - 1; _i <= _h; _i += 1) { //#11223
        $_.i = _i; //#11219
        var _l = $get($_.encstr, $_.i); //#11220
        $put($_.bhs, $_.i, $_.bar * $get($a([1, 1, 2]), _l)); //#11221
        $put($_.bbs, $_.i, $get($a([0, $_.bar, 0]), _l)); //#11222
    } //#11222
    $k[$j++] = Infinity; //#11226
    var _w = $_.bhs; //#11228
    var _x = $_.bbs; //#11229
    $k[$j++] = Infinity; //#11230
    for (var _z = 0, _10 = $_.encstr.length * 2; _z < _10; _z++) { //#11230
        $k[$j++] = $_.spc; //#11230
    } //#11230
    var _12 = $a(); //#11230
    $k[$j++] = 'ren'; //#11234
    $k[$j++] = 'renlinear'; //#11234
    $k[$j++] = 'bhs'; //#11234
    $k[$j++] = _w; //#11234
    $k[$j++] = 'bbs'; //#11234
    $k[$j++] = _x; //#11234
    $k[$j++] = 'sbs'; //#11234
    $k[$j++] = _12; //#11234
    $k[$j++] = 'txt'; //#11234
    $k[$j++] = $_.txt; //#11234
    $k[$j++] = 'textxalign'; //#11234
    $k[$j++] = "center"; //#11234
    $k[$j++] = 'textyoffset'; //#11234
    $k[$j++] = 4; //#11234
    $k[$j++] = 'opt'; //#11234
    $k[$j++] = $_.options; //#11234
    var _15 = $d(); //#11234
    $k[$j++] = _15; //#11237
    if (!$_.dontdraw) { //#11237
        bwipp_renlinear(); //#11237
    } //#11237
    $_ = $__; //#11239
} //bwipp_pharmacode2
function bwipp_code2of5() {
    var $__ = $_; //#11269
    $_ = Object.assign({}, $_); //#11269
    $_.dontdraw = false; //#11272
    $_.includecheck = false; //#11273
    $_.validatecheck = false; //#11274
    $_.includetext = false; //#11275
    $_.includecheckintext = false; //#11276
    $_.textfont = "OCR-B"; //#11277
    $_.textsize = 10; //#11278
    $_.textyoffset = -8; //#11279
    $_.height = 1; //#11280
    $_.version = "industrial"; //#11281
    $k[$j++] = $_; //#11283
    bwipp_processoptions(); //#11283
    $_.options = $k[--$j]; //#11283
    $_.barcode = $k[--$j]; //#11284
    if ($_.barcode.length == 0) { //#11288
        $k[$j++] = 'bwipp.code2of5emptyData#11287'; //#11287
        $k[$j++] = "The data must not be empty"; //#11287
        bwipp_raiseerror(); //#11287
    } //#11287
    $forall($_.barcode, function() { //#11297
        var _4 = $k[--$j]; //#11294
        if ((_4 < 48) || (_4 > 57)) { //#11296
            $k[$j++] = 'bwipp.code2of5badCharacter#11295'; //#11295
            $k[$j++] = "Code 25 must contain only digits"; //#11295
            bwipp_raiseerror(); //#11295
        } //#11295
    }); //#11295
    $k[$j++] = 'barlen'; //#11299
    $k[$j++] = $_.barcode.length; //#11299
    if ($_.validatecheck) { //#11299
        var _7 = $k[--$j]; //#11299
        $k[$j++] = _7 - 1; //#11299
    } //#11299
    var _8 = $k[--$j]; //#11299
    $_[$k[--$j]] = _8; //#11299
    $_.checksum = 0; //#11301
    for (var _C = 0, _B = $_.barlen - 1; _C <= _B; _C += 1) { //#11305
        $_.i = _C; //#11303
        $k[$j++] = 'checksum'; //#11304
        $k[$j++] = $_.checksum; //#11304
        $k[$j++] = $get($_.barcode, $_.i) - 48; //#11304
        if (($f($_.barlen - $_.i) % 2) != 0) { //#11304
            var _J = $k[--$j]; //#11304
            $k[$j++] = _J * 3; //#11304
        } //#11304
        var _K = $k[--$j]; //#11304
        var _L = $k[--$j]; //#11304
        $_[$k[--$j]] = $f(_L + _K); //#11304
    } //#11304
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#11306
    if ($_.validatecheck) { //#11313
        if ($get($_.barcode, $_.barlen) != ($_.checksum + 48)) { //#11310
            $k[$j++] = 'bwipp.code2of5badCheckDigit#11309'; //#11309
            $k[$j++] = "Incorrect Code 25 check digit provided"; //#11309
            bwipp_raiseerror(); //#11309
        } //#11309
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#11311
        $_.includecheck = true; //#11312
    } //#11312
    var _b = new Map([
        ["industrial", $a(["1111313111", "3111111131", "1131111131", "3131111111", "1111311131", "3111311111", "1131311111", "1111113131", "3111113111", "1131113111", "313111", "31113"])],
        ["iata", $a(["1111313111", "3111111131", "1131111131", "3131111111", "1111311131", "3111311111", "1131311111", "1111113131", "3111113111", "1131113111", "1111", "311"])],
        ["matrix", $a(["113311", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "131311", "311111", "31111"])],
        ["coop", $a(["331111", "111331", "113131", "113311", "131131", "131311", "133111", "311131", "311311", "313111", "3131", "133"])],
        ["datalogic", $a(["113311", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "131311", "1111", "311"])]
    ]); //#11338
    $_.versions = _b; //#11340
    var _f = $get($_.versions, $_.version) !== undefined; //#11343
    if (!_f) { //#11345
        $k[$j++] = 'bwipp.code2of5badVersion#11344'; //#11344
        $k[$j++] = "Unrecognised Code 25 version"; //#11344
        bwipp_raiseerror(); //#11344
    } //#11344
    $_.encs = $get($_.versions, $_.version); //#11346
    $_.cs = $get($_.encs, 0).length; //#11348
    $k[$j++] = 'cw'; //#11349
    $k[$j++] = 0; //#11349
    $forall($get($_.encs, 0), function() { //#11349
        var _n = $k[--$j]; //#11349
        var _o = $k[--$j]; //#11349
        $k[$j++] = $f(_o + (_n - 48)); //#11349
    }); //#11349
    var _p = $k[--$j]; //#11349
    $_[$k[--$j]] = _p; //#11349
    $_.ss = $get($_.encs, 10).length; //#11350
    $k[$j++] = 'sw'; //#11351
    $k[$j++] = 0; //#11351
    $forall($get($_.encs, 10), function() { //#11351
        var _v = $k[--$j]; //#11351
        var _w = $k[--$j]; //#11351
        $k[$j++] = $f(_w + (_v - 48)); //#11351
    }); //#11351
    var _x = $k[--$j]; //#11351
    $_[$k[--$j]] = _x; //#11351
    $_.es = $get($_.encs, 11).length; //#11352
    $_.barchars = "0123456789"; //#11356
    $k[$j++] = 'sbs'; //#11359
    $k[$j++] = $_.barlen; //#11359
    if ($_.includecheck) { //#11359
        var _14 = $k[--$j]; //#11359
        $k[$j++] = _14 + 1; //#11359
    } //#11359
    var _19 = $s($f($k[--$j] * $_.cs + $_.ss + $_.es));; //#11359
    $_[$k[--$j]] = _19; //#11359
    $k[$j++] = 'txt'; //#11360
    $k[$j++] = $_.barlen; //#11360
    if ($_.includecheck) { //#11360
        var _1D = $k[--$j]; //#11360
        $k[$j++] = _1D + 1; //#11360
    } //#11360
    var _1F = $a($k[--$j]); //#11360
    $_[$k[--$j]] = _1F; //#11360
    $puti($_.sbs, 0, $get($_.encs, 10)); //#11363
    for (var _1M = 0, _1L = $_.barlen - 1; _1M <= _1L; _1M += 1) { //#11374
        $_.i = _1M; //#11365
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#11367
        $j--; //#11368
        $_.indx = $k[--$j].length; //#11369
        $j -= 2; //#11370
        $_.enc = $get($_.encs, $_.indx); //#11371
        $puti($_.sbs, ($_.i * $_.cs) + $_.ss, $_.enc); //#11372
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), $f(($_.i * $_.cw) + $_.sw), $_.textyoffset, $_.textfont, $_.textsize])); //#11373
    } //#11373
    if ($_.includecheck) { //#11386
        $puti($_.sbs, $f(($_.barlen * $_.cs) + $_.ss), $get($_.encs, $_.checksum)); //#11378
        $puti($_.sbs, $f($_.barlen * $_.cs + $_.cs + $_.ss), $get($_.encs, 11)); //#11379
        if ($_.includecheckintext) { //#11383
            $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum, 1), $f(($_.barlen * $_.cw) + $_.sw), $_.textyoffset, $_.textfont, $_.textsize])); //#11381
        } else { //#11383
            $put($_.txt, $_.barlen, $a([" ", $f(($_.barlen * $_.cw) + $_.sw), $_.textyoffset, $_.textfont, $_.textsize])); //#11383
        } //#11383
    } else { //#11386
        $puti($_.sbs, $f(($_.barlen * $_.cs) + $_.ss), $get($_.encs, 11)); //#11386
    } //#11386
    $k[$j++] = Infinity; //#11390
    $k[$j++] = Infinity; //#11392
    $forall($_.sbs, function() { //#11392
        var _2U = $k[--$j]; //#11392
        $k[$j++] = _2U - 48; //#11392
    }); //#11392
    var _2V = $a(); //#11392
    $k[$j++] = Infinity; //#11393
    for (var _2X = 0, _2Y = ~~(($_.sbs.length + 1) / 2); _2X < _2Y; _2X++) { //#11393
        $k[$j++] = $_.height; //#11393
    } //#11393
    var _2a = $a(); //#11393
    $k[$j++] = Infinity; //#11394
    for (var _2c = 0, _2d = ~~(($_.sbs.length + 1) / 2); _2c < _2d; _2c++) { //#11394
        $k[$j++] = 0; //#11394
    } //#11394
    var _2e = $a(); //#11394
    $k[$j++] = 'ren'; //#11397
    $k[$j++] = 'renlinear'; //#11397
    $k[$j++] = 'sbs'; //#11397
    $k[$j++] = _2V; //#11397
    $k[$j++] = 'bhs'; //#11397
    $k[$j++] = _2a; //#11397
    $k[$j++] = 'bbs'; //#11397
    $k[$j++] = _2e; //#11397
    if ($_.includetext) { //#11397
        $k[$j++] = 'txt'; //#11396
        $k[$j++] = $_.txt; //#11396
    } //#11396
    $k[$j++] = 'opt'; //#11398
    $k[$j++] = $_.options; //#11398
    var _2i = $d(); //#11398
    $k[$j++] = _2i; //#11401
    if (!$_.dontdraw) { //#11401
        bwipp_renlinear(); //#11401
    } //#11401
    $_ = $__; //#11405
} //bwipp_code2of5
function bwipp_industrial2of5() {
    var $__ = $_; //#11434
    $_ = Object.assign({}, $_); //#11434
    $_.dontdraw = false; //#11437
    $k[$j++] = $_; //#11439
    bwipp_processoptions(); //#11439
    $_.options = $k[--$j]; //#11439
    $_.barcode = $k[--$j]; //#11440
    $put($_.options, "dontdraw", true); //#11443
    $put($_.options, "version", "industrial"); //#11444
    $k[$j++] = 'args'; //#11446
    $k[$j++] = $_.barcode; //#11446
    $k[$j++] = $_.options; //#11446
    bwipp_code2of5(); //#11446
    var _6 = $k[--$j]; //#11446
    $_[$k[--$j]] = _6; //#11446
    $put($_.args, "opt", $_.options); //#11448
    $k[$j++] = $_.args; //#11451
    if (!$_.dontdraw) { //#11451
        bwipp_renlinear(); //#11451
    } //#11451
    $_ = $__; //#11453
} //bwipp_industrial2of5
function bwipp_iata2of5() {
    var $__ = $_; //#11482
    $_ = Object.assign({}, $_); //#11482
    $_.dontdraw = false; //#11485
    $k[$j++] = $_; //#11487
    bwipp_processoptions(); //#11487
    $_.options = $k[--$j]; //#11487
    $_.barcode = $k[--$j]; //#11488
    $put($_.options, "dontdraw", true); //#11491
    $put($_.options, "version", "iata"); //#11492
    $k[$j++] = 'args'; //#11494
    $k[$j++] = $_.barcode; //#11494
    $k[$j++] = $_.options; //#11494
    bwipp_code2of5(); //#11494
    var _6 = $k[--$j]; //#11494
    $_[$k[--$j]] = _6; //#11494
    $put($_.args, "opt", $_.options); //#11496
    $k[$j++] = $_.args; //#11499
    if (!$_.dontdraw) { //#11499
        bwipp_renlinear(); //#11499
    } //#11499
    $_ = $__; //#11501
} //bwipp_iata2of5
function bwipp_matrix2of5() {
    var $__ = $_; //#11530
    $_ = Object.assign({}, $_); //#11530
    $_.dontdraw = false; //#11533
    $k[$j++] = $_; //#11535
    bwipp_processoptions(); //#11535
    $_.options = $k[--$j]; //#11535
    $_.barcode = $k[--$j]; //#11536
    $put($_.options, "dontdraw", true); //#11539
    $put($_.options, "version", "matrix"); //#11540
    $k[$j++] = 'args'; //#11542
    $k[$j++] = $_.barcode; //#11542
    $k[$j++] = $_.options; //#11542
    bwipp_code2of5(); //#11542
    var _6 = $k[--$j]; //#11542
    $_[$k[--$j]] = _6; //#11542
    $put($_.args, "opt", $_.options); //#11544
    $k[$j++] = $_.args; //#11547
    if (!$_.dontdraw) { //#11547
        bwipp_renlinear(); //#11547
    } //#11547
    $_ = $__; //#11549
} //bwipp_matrix2of5
function bwipp_coop2of5() {
    var $__ = $_; //#11578
    $_ = Object.assign({}, $_); //#11578
    $_.dontdraw = false; //#11581
    $k[$j++] = $_; //#11583
    bwipp_processoptions(); //#11583
    $_.options = $k[--$j]; //#11583
    $_.barcode = $k[--$j]; //#11584
    $put($_.options, "dontdraw", true); //#11587
    $put($_.options, "version", "coop"); //#11588
    $k[$j++] = 'args'; //#11590
    $k[$j++] = $_.barcode; //#11590
    $k[$j++] = $_.options; //#11590
    bwipp_code2of5(); //#11590
    var _6 = $k[--$j]; //#11590
    $_[$k[--$j]] = _6; //#11590
    $put($_.args, "opt", $_.options); //#11592
    $k[$j++] = $_.args; //#11595
    if (!$_.dontdraw) { //#11595
        bwipp_renlinear(); //#11595
    } //#11595
    $_ = $__; //#11597
} //bwipp_coop2of5
function bwipp_datalogic2of5() {
    var $__ = $_; //#11626
    $_ = Object.assign({}, $_); //#11626
    $_.dontdraw = false; //#11629
    $k[$j++] = $_; //#11631
    bwipp_processoptions(); //#11631
    $_.options = $k[--$j]; //#11631
    $_.barcode = $k[--$j]; //#11632
    $put($_.options, "dontdraw", true); //#11635
    $put($_.options, "version", "datalogic"); //#11636
    $k[$j++] = 'args'; //#11638
    $k[$j++] = $_.barcode; //#11638
    $k[$j++] = $_.options; //#11638
    bwipp_code2of5(); //#11638
    var _6 = $k[--$j]; //#11638
    $_[$k[--$j]] = _6; //#11638
    $put($_.args, "opt", $_.options); //#11640
    $k[$j++] = $_.args; //#11643
    if (!$_.dontdraw) { //#11643
        bwipp_renlinear(); //#11643
    } //#11643
    $_ = $__; //#11645
} //bwipp_datalogic2of5
function bwipp_code11() {
    var $__ = $_; //#11675
    $_ = Object.assign({}, $_); //#11675
    $_.dontdraw = false; //#11678
    $_.includecheck = false; //#11679
    $_.validatecheck = false; //#11680
    $_.includetext = false; //#11681
    $_.includecheckintext = false; //#11682
    $_.textfont = "OCR-B"; //#11683
    $_.textsize = 10; //#11684
    $_.textyoffset = -8; //#11685
    $_.height = 1; //#11686
    $k[$j++] = $_; //#11688
    bwipp_processoptions(); //#11688
    $_.options = $k[--$j]; //#11688
    $_.barcode = $k[--$j]; //#11689
    if ($_.barcode.length == 0) { //#11693
        $k[$j++] = 'bwipp.code11emptyData#11692'; //#11692
        $k[$j++] = "The data must not be empty"; //#11692
        bwipp_raiseerror(); //#11692
    } //#11692
    $_.encs = $a(["111131", "311131", "131131", "331111", "113131", "313111", "133111", "111331", "311311", "311111", "113111", "113311"]); //#11703
    $_.barchars = "0123456789-"; //#11706
    $_.charvals = new Map; //#11707
    for (var _4 = 0; _4 <= 10; _4 += 1) { //#11708
        $put($_.charvals, $geti($_.barchars, _4, 1), _4); //#11708
    } //#11708
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#11716
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#11713
        if (!_F) { //#11715
            $k[$j++] = 'bwipp.code11badCharacter#11714'; //#11714
            $k[$j++] = "Code 11 must contain only digits and dashes"; //#11714
            bwipp_raiseerror(); //#11714
        } //#11714
    } //#11714
    $_.barlen = $_.barcode.length; //#11718
    if ($_.validatecheck) { //#11724
        if ($_.barlen == 11) { //#11722
            $k[$j++] = 'bwipp.code11badLength#11721'; //#11721
            $k[$j++] = "Code 11 cannot be 11 characters using check digits"; //#11721
            bwipp_raiseerror(); //#11721
        } //#11721
        var _L = ($_.barlen <= 10) ? 1 : 2; //#11723
        $_.barlen = $_.barlen - _L; //#11723
    } //#11723
    $k[$j++] = 'numchecks'; //#11726
    if ($_.includecheck || $_.validatecheck) { //#11726
        var _P = ($_.barlen >= 10) ? 2 : 1; //#11726
        $k[$j++] = _P; //#11726
    } else { //#11726
        $k[$j++] = 0; //#11726
    } //#11726
    var _Q = $k[--$j]; //#11726
    $_[$k[--$j]] = _Q; //#11726
    $_.checksum1 = 0; //#11727
    $_.checksum2 = 0; //#11727
    for (var _U = 0, _T = $_.barlen - 1; _U <= _T; _U += 1) { //#11733
        $_.i = _U; //#11729
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#11730
        $_.checksum1 = $f($_.checksum1 + ((((($_.barlen - $_.i) - 1) % 10) + 1) * $_.indx)); //#11731
        $_.checksum2 = $f($_.checksum2 + (((($_.barlen - $_.i) % 9) + 1) * $_.indx)); //#11732
    } //#11732
    $_.checksum1 = $_.checksum1 % 11; //#11734
    $_.checksum2 = $f($_.checksum2 + $_.checksum1) % 11; //#11735
    if ($_.validatecheck) { //#11749
        if ($_.numchecks == 1) { //#11744
            var _r = $_.checksum1; //#11738
            if ($get($_.barcode, $_.barlen) != $get($_.barchars, _r)) { //#11740
                $k[$j++] = 'bwipp.code11badCheckDigit#11739'; //#11739
                $k[$j++] = "Incorrect Code 11 check digit provided"; //#11739
                bwipp_raiseerror(); //#11739
            } //#11739
        } else { //#11744
            if (($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum1)) || ($get($_.barcode, $_.barlen + 1) != $get($_.barchars, $_.checksum2))) { //#11745
                $k[$j++] = 'bwipp.code11badCheckDigits#11744'; //#11744
                $k[$j++] = "Incorrect Code 11 check digits provided"; //#11744
                bwipp_raiseerror(); //#11744
            } //#11744
        } //#11744
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#11747
        $_.includecheck = true; //#11748
    } //#11748
    $_.sbs = $s(($f($_.barlen + $_.numchecks) * 6) + 12); //#11751
    $_.txt = $a($f($_.barlen + $_.numchecks)); //#11752
    $puti($_.sbs, 0, $get($_.encs, 11)); //#11755
    $_.xpos = 8; //#11757
    for (var _1J = 0, _1I = $_.barlen - 1; _1J <= _1I; _1J += 1) { //#11767
        $_.i = _1J; //#11759
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#11760
        $_.enc = $get($_.encs, $_.indx); //#11761
        $puti($_.sbs, ($_.i * 6) + 6, $_.enc); //#11762
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11763
        for (var _1f = 0; _1f <= 5; _1f += 1) { //#11766
            $_.xpos = $f(($get($_.enc, _1f) - 48) + $_.xpos); //#11765
        } //#11765
    } //#11765
    if ($_.includecheck) { //#11796
        if ($_.barlen >= 10) { //#11793
            $puti($_.sbs, ($_.barlen * 6) + 6, $get($_.encs, $_.checksum1)); //#11772
            $puti($_.sbs, ($_.barlen * 6) + 12, $get($_.encs, $_.checksum2)); //#11773
            if ($_.includecheckintext) { //#11783
                $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum1, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11775
                $_.enc = $get($_.encs, $_.checksum1); //#11776
                for (var _29 = 0; _29 <= 5; _29 += 1) { //#11779
                    $_.xpos = $f(($get($_.enc, _29) - 48) + $_.xpos); //#11778
                } //#11778
                $put($_.txt, $_.barlen + 1, $a([$geti($_.barchars, $_.checksum2, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11780
            } else { //#11783
                $put($_.txt, $_.barlen, $a(["", $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11782
                $put($_.txt, $_.barlen + 1, $a(["", $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11783
            } //#11783
            $puti($_.sbs, ($_.barlen * 6) + 18, $get($_.encs, 11)); //#11785
        } else { //#11793
            $puti($_.sbs, ($_.barlen * 6) + 6, $get($_.encs, $_.checksum1)); //#11787
            if ($_.includecheckintext) { //#11791
                $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum1, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11789
            } else { //#11791
                $put($_.txt, $_.barlen, $a(["", $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11791
            } //#11791
            $puti($_.sbs, ($_.barlen * 6) + 12, $get($_.encs, 11)); //#11793
        } //#11793
    } else { //#11796
        $puti($_.sbs, ($_.barlen * 6) + 6, $get($_.encs, 11)); //#11796
    } //#11796
    $k[$j++] = Infinity; //#11800
    $k[$j++] = Infinity; //#11802
    var _3A = $_.sbs; //#11802
    for (var _3B = 0, _3C = _3A.length; _3B < _3C; _3B++) { //#11802
        $k[$j++] = $get(_3A, _3B) - 48; //#11802
    } //#11802
    var _3E = $a(); //#11802
    $k[$j++] = Infinity; //#11803
    for (var _3G = 0, _3H = ~~(($_.sbs.length + 1) / 2); _3G < _3H; _3G++) { //#11803
        $k[$j++] = $_.height; //#11803
    } //#11803
    var _3J = $a(); //#11803
    $k[$j++] = Infinity; //#11804
    for (var _3L = 0, _3M = ~~(($_.sbs.length + 1) / 2); _3L < _3M; _3L++) { //#11804
        $k[$j++] = 0; //#11804
    } //#11804
    var _3N = $a(); //#11804
    $k[$j++] = 'ren'; //#11807
    $k[$j++] = 'renlinear'; //#11807
    $k[$j++] = 'sbs'; //#11807
    $k[$j++] = _3E; //#11807
    $k[$j++] = 'bhs'; //#11807
    $k[$j++] = _3J; //#11807
    $k[$j++] = 'bbs'; //#11807
    $k[$j++] = _3N; //#11807
    if ($_.includetext) { //#11807
        $k[$j++] = 'txt'; //#11806
        $k[$j++] = $_.txt; //#11806
    } //#11806
    $k[$j++] = 'opt'; //#11808
    $k[$j++] = $_.options; //#11808
    var _3R = $d(); //#11808
    $k[$j++] = _3R; //#11811
    if (!$_.dontdraw) { //#11811
        bwipp_renlinear(); //#11811
    } //#11811
    $_ = $__; //#11815
} //bwipp_code11
function bwipp_bc412() {
    var $__ = $_; //#11845
    $_ = Object.assign({}, $_); //#11845
    $_.dontdraw = false; //#11848
    $_.includecheck = false; //#11849
    $_.validatecheck = false; //#11850
    $_.includetext = false; //#11851
    $_.includecheckintext = false; //#11852
    $_.includestartstop = false; //#11853
    $_.semi = false; //#11854
    $_.textfont = "OCR-B"; //#11855
    $_.textsize = 10; //#11856
    $_.textyoffset = -8; //#11857
    $_.height = 1; //#11858
    $k[$j++] = $_; //#11860
    bwipp_processoptions(); //#11860
    $_.options = $k[--$j]; //#11860
    $_.barcode = $k[--$j]; //#11861
    if ($_.barcode.length == 0) { //#11865
        $k[$j++] = 'bwipp.bc412emptyData#11864'; //#11864
        $k[$j++] = "The data must not be empty"; //#11864
        bwipp_raiseerror(); //#11864
    } //#11864
    if ($_.semi) { //#11870
        $_.includestartstop = true; //#11869
    } //#11869
    $_.barchars = "0R9GLVHA8EZ4NTS1J2Q6C7DYKBUIX3FWP5M"; //#11876
    $_.charvals = new Map; //#11877
    for (var _4 = 0; _4 <= 34; _4 += 1) { //#11878
        $put($_.charvals, $geti($_.barchars, _4, 1), _4); //#11878
    } //#11878
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#11886
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#11883
        if (!_F) { //#11885
            $k[$j++] = 'bwipp.bc412badCharacter#11884'; //#11884
            $k[$j++] = "BC412 must contain only digits and capital letters except O"; //#11884
            bwipp_raiseerror(); //#11884
        } //#11884
    } //#11884
    if ($_.semi) { //#11942
        if ($_.validatecheck) { //#11896
            if (($_.barcode.length < 8) || ($_.barcode.length > 19)) { //#11893
                $k[$j++] = 'bwipp.bc412semivalidateBadLength#11892'; //#11892
                $k[$j++] = "BC412 semi with check digit must be 8 to 19 characters long"; //#11892
                bwipp_raiseerror(); //#11892
            } //#11892
        } else { //#11896
            if (($_.barcode.length < 7) || ($_.barcode.length > 18)) { //#11897
                $k[$j++] = 'bwipp.bc412semiBadLength#11896'; //#11896
                $k[$j++] = "BC412 semi must be 7 to 18 characters long"; //#11896
                bwipp_raiseerror(); //#11896
            } //#11896
        } //#11896
        $_.barlen = $_.barcode.length; //#11900
        if (!$_.validatecheck) { //#11908
            $_.barlen = $_.barlen + 1; //#11902
            $_.sbarcode = $s($_.barlen); //#11903
            $put($_.sbarcode, 0, $get($_.barcode, 0)); //#11904
            $put($_.sbarcode, 1, 48); //#11905
            $puti($_.sbarcode, 2, $geti($_.barcode, 1, $_.barlen - 2)); //#11906
            $_.barcode = $_.sbarcode; //#11907
        } //#11907
        $_.sumodd = 0; //#11909
        $_.sumeven = 0; //#11909
        for (var _c = 0, _b = $_.barlen - 1; _c <= _b; _c += 1) { //#11918
            $_.i = _c; //#11911
            $k[$j++] = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#11917
            if (($_.i % 2) == 0) { //#11916
                $_.sumodd = $f($k[--$j] + $_.sumodd); //#11914
            } else { //#11916
                $_.sumeven = $f($k[--$j] + $_.sumeven); //#11916
            } //#11916
        } //#11916
        $_.checksum = ($f(($_.sumodd % 35) + (($_.sumeven % 35) * 2))) % 35; //#11919
        if ($_.validatecheck) { //#11926
            if ($_.checksum != 0) { //#11923
                $k[$j++] = 'bwipp.bc412semiBadCheckDigit#11922'; //#11922
                $k[$j++] = "Incorrect BC412 semi check digit provided"; //#11922
                bwipp_raiseerror(); //#11922
            } //#11922
        } else { //#11926
            $_.checksum = ($_.checksum * 17) % 35; //#11925
            $put($_.barcode, 1, $get($_.barchars, $_.checksum)); //#11926
        } //#11926
    } else { //#11942
        $k[$j++] = 'barlen'; //#11929
        $k[$j++] = $_.barcode.length; //#11929
        if ($_.validatecheck) { //#11929
            var _y = $k[--$j]; //#11929
            $k[$j++] = _y - 1; //#11929
        } //#11929
        var _z = $k[--$j]; //#11929
        $_[$k[--$j]] = _z; //#11929
        $_.checksum = 0; //#11931
        for (var _13 = 0, _12 = $_.barlen - 1; _13 <= _12; _13 += 1) { //#11935
            $_.checksum = $f($get($_.charvals, $geti($_.barcode, _13, 1)) + $_.checksum); //#11934
        } //#11934
        $_.checksum = $_.checksum % 35; //#11936
        if ($_.validatecheck) { //#11943
            if ($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum)) { //#11940
                $k[$j++] = 'bwipp.bc412badCheckDigit#11939'; //#11939
                $k[$j++] = "Incorrect BC412 check digit provided"; //#11939
                bwipp_raiseerror(); //#11939
            } //#11939
            $_.barcode = $geti($_.barcode, 0, $_.barlen); //#11941
            $_.includecheck = true; //#11942
        } //#11942
    } //#11942
    $_.encs = $a(["11111115", "13111212", "11131113", "12111213", "12121311", "13131111", "12111312", "11131212", "11121411", "11151111", "15111111", "11111511", "12131211", "13121112", "13111311", "11111214", "12121113", "11111313", "13111113", "11121213", "11141112", "11121312", "11141211", "14121111", "12121212", "11131311", "13121211", "12111411", "14111211", "11111412", "12111114", "14111112", "12141111", "11121114", "12131112", "12", "111"]); //#11957
    $_.sbs = $s((($_.barlen + 1) * 8) + 5); //#11960
    if ($_.includecheck) { //#11964
        $_.txt = $a($_.barlen + 1); //#11962
    } else { //#11964
        $_.txt = $a($_.barlen); //#11964
    } //#11964
    if ($_.includestartstop) { //#11974
        $puti($_.sbs, 0, $get($_.encs, 35)); //#11969
        $_.pos = 2; //#11970
        $_.txtpos = 3; //#11971
    } else { //#11974
        $_.pos = 0; //#11973
        $_.txtpos = 0; //#11974
    } //#11974
    for (var _1Z = 0, _1Y = $_.barlen - 1; _1Z <= _1Y; _1Z += 1) { //#11983
        $_.i = _1Z; //#11978
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#11979
        $puti($_.sbs, $_.pos, $get($_.encs, $_.indx)); //#11980
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 12) + $_.txtpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11981
        $_.pos = $_.pos + 8; //#11982
    } //#11982
    if ($_.includecheck) { //#11994
        $puti($_.sbs, $_.pos, $get($_.encs, $_.checksum)); //#11987
        if ($_.includecheckintext) { //#11991
            $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum, 1), ($_.barlen * 12) + $_.txtpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11989
        } else { //#11991
            $put($_.txt, $_.barlen, $a(["", ($_.barlen * 12) + $_.txtpos, $_.textyoffset, $_.textfont, $_.textsize])); //#11991
        } //#11991
        $_.pos = $_.pos + 8; //#11993
    } //#11993
    if ($_.includestartstop) { //#12000
        $puti($_.sbs, $_.pos, $get($_.encs, 36)); //#11998
        $_.pos = $_.pos + 3; //#11999
    } //#11999
    $_.sbs = $geti($_.sbs, 0, $_.pos); //#12003
    $k[$j++] = Infinity; //#12004
    $k[$j++] = Infinity; //#12006
    var _2W = $_.sbs; //#12006
    for (var _2X = 0, _2Y = _2W.length; _2X < _2Y; _2X++) { //#12006
        $k[$j++] = $get(_2W, _2X) - 48; //#12006
    } //#12006
    var _2a = $a(); //#12006
    $k[$j++] = Infinity; //#12007
    for (var _2c = 0, _2d = ~~(($_.sbs.length + 1) / 2); _2c < _2d; _2c++) { //#12007
        $k[$j++] = $_.height; //#12007
    } //#12007
    var _2f = $a(); //#12007
    $k[$j++] = Infinity; //#12008
    for (var _2h = 0, _2i = ~~(($_.sbs.length + 1) / 2); _2h < _2i; _2h++) { //#12008
        $k[$j++] = 0; //#12008
    } //#12008
    var _2j = $a(); //#12008
    $k[$j++] = 'ren'; //#12011
    $k[$j++] = 'renlinear'; //#12011
    $k[$j++] = 'sbs'; //#12011
    $k[$j++] = _2a; //#12011
    $k[$j++] = 'bhs'; //#12011
    $k[$j++] = _2f; //#12011
    $k[$j++] = 'bbs'; //#12011
    $k[$j++] = _2j; //#12011
    if ($_.includetext) { //#12011
        $k[$j++] = 'txt'; //#12010
        $k[$j++] = $_.txt; //#12010
    } //#12010
    $k[$j++] = 'opt'; //#12012
    $k[$j++] = $_.options; //#12012
    var _2n = $d(); //#12012
    $k[$j++] = _2n; //#12015
    if (!$_.dontdraw) { //#12015
        bwipp_renlinear(); //#12015
    } //#12015
    $_ = $__; //#12019
} //bwipp_bc412
function bwipp_rationalizedCodabar() {
    var $__ = $_; //#12049
    $_ = Object.assign({}, $_); //#12049
    $_.dontdraw = false; //#12052
    $_.altstartstop = false; //#12053
    $_.includecheck = false; //#12054
    $_.validatecheck = false; //#12055
    $_.includetext = false; //#12056
    $_.includecheckintext = false; //#12057
    $_.textfont = "OCR-B"; //#12058
    $_.textsize = 10; //#12059
    $_.textyoffset = -8; //#12060
    $_.height = 1; //#12061
    $k[$j++] = $_; //#12063
    bwipp_processoptions(); //#12063
    $_.options = $k[--$j]; //#12063
    $_.barcode = $k[--$j]; //#12064
    if (($_.barcode.length < 2) || (($_.barcode.length < 3) && $_.validatecheck)) { //#12068
        $k[$j++] = 'bwipp.rationalizedCodabarBadLength#12067'; //#12067
        $k[$j++] = "Codabar must be at least 2 characters in length excluding any check digit"; //#12067
        bwipp_raiseerror(); //#12067
    } //#12067
    $_.encs = $a(["11111331", "11113311", "11131131", "33111111", "11311311", "31111311", "13111131", "13113111", "13311111", "31131111", "11133111", "11331111", "31113131", "31311131", "31313111", "11313131", "11331311", "13131131", "11131331", "11133311"]); //#12079
    $_.barcharsnormal = "0123456789-$:/.+ABCD"; //#12082
    $_.barcharsalt = "0123456789-$:/.+TN*E"; //#12083
    var _8 = $_.altstartstop ? $_.barcharsalt : $_.barcharsnormal; //#12086
    $_.barchars = _8; //#12086
    $_.charvals = new Map; //#12088
    for (var _9 = 0; _9 <= 19; _9 += 1) { //#12089
        $put($_.charvals, $geti($_.barchars, _9, 1), _9); //#12089
    } //#12089
    $_.bodyvals = new Map; //#12090
    for (var _D = 0; _D <= 15; _D += 1) { //#12091
        $put($_.bodyvals, $geti($_.barchars, _D, 1), _D); //#12091
    } //#12091
    $_.ssvals = new Map; //#12092
    for (var _H = 16; _H <= 19; _H += 1) { //#12093
        $put($_.ssvals, $geti($_.barchars, _H, 1), _H); //#12093
    } //#12093
    var _O = $get($_.ssvals, $geti($_.barcode, 0, 1)) !== undefined; //#12096
    var _T = $get($_.ssvals, $geti($_.barcode, $_.barcode.length - 1, 1)) !== undefined; //#12097
    if ((!_O) || (!_T)) { //#12103
        if ($_.altstartstop) { //#12101
            $k[$j++] = 'bwipp.rationalizedCodabarBadAltStartStop#12099'; //#12099
            $k[$j++] = "Codabar start and stop characters must be one of E N T or *"; //#12099
            bwipp_raiseerror(); //#12099
        } else { //#12101
            $k[$j++] = 'bwipp.rationalizedCodabarBadStartStop#12101'; //#12101
            $k[$j++] = "Codabar start and stop characters must be one of A B C or D"; //#12101
            bwipp_raiseerror(); //#12101
        } //#12101
    } //#12101
    for (var _X = 1, _W = $_.barcode.length - 2; _X <= _W; _X += 1) { //#12108
        var _b = $get($_.bodyvals, $geti($_.barcode, _X, 1)) !== undefined; //#12105
        if (!_b) { //#12107
            $k[$j++] = 'bwipp.rationalizedCodabarBadCharacter#12106'; //#12106
            $k[$j++] = "Codabar body must contain only digits and symbols - $ : / . +"; //#12106
            bwipp_raiseerror(); //#12106
        } //#12106
    } //#12106
    $k[$j++] = 'barlen'; //#12110
    $k[$j++] = $_.barcode.length; //#12110
    if ($_.validatecheck) { //#12110
        var _e = $k[--$j]; //#12110
        $k[$j++] = _e - 1; //#12110
    } //#12110
    var _f = $k[--$j]; //#12110
    $_[$k[--$j]] = _f; //#12110
    $_.checksum = 0; //#12112
    for (var _j = 0, _i = $_.barlen - 2; _j <= _i; _j += 1) { //#12116
        $_.checksum = $f($get($_.charvals, $geti($_.barcode, _j, 1)) + $_.checksum); //#12115
    } //#12115
    var _r = $geti($_.barcode, $_.barcode.length - 1, 1); //#12117
    $_.checksum = $f($get($_.charvals, _r) + $_.checksum); //#12118
    $_.checksum = (16 - ($_.checksum % 16)) % 16; //#12119
    if ($_.validatecheck) { //#12129
        if ($get($_.barcode, $_.barlen - 1) != $get($_.barchars, $_.checksum)) { //#12123
            $k[$j++] = 'bwipp.rationalizedCodabarBadCheckDigit#12122'; //#12122
            $k[$j++] = "Incorrect Codabar check digit provided"; //#12122
            bwipp_raiseerror(); //#12122
        } //#12122
        var _14 = $s($_.barlen); //#12124
        $puti(_14, 0, $geti($_.barcode, 0, $_.barlen - 1)); //#12125
        $puti(_14, $_.barlen - 1, $geti($_.barcode, $_.barlen, 1)); //#12126
        $_.barcode = _14; //#12127
        $_.includecheck = true; //#12128
    } //#12128
    $k[$j++] = 'sbs'; //#12131
    $k[$j++] = $_.barlen; //#12131
    if ($_.includecheck) { //#12131
        var _1E = $k[--$j]; //#12131
        $k[$j++] = _1E + 1; //#12131
    } //#12131
    var _1G = $s($k[--$j] * 8); //#12131
    $_[$k[--$j]] = _1G; //#12131
    $k[$j++] = 'txt'; //#12132
    $k[$j++] = $_.barlen; //#12132
    if ($_.includecheck) { //#12132
        var _1K = $k[--$j]; //#12132
        $k[$j++] = _1K + 1; //#12132
    } //#12132
    var _1M = $a($k[--$j]); //#12132
    $_[$k[--$j]] = _1M; //#12132
    $_.xpos = 0; //#12134
    for (var _1Q = 0, _1P = $_.barlen - 2; _1Q <= _1P; _1Q += 1) { //#12144
        $_.i = _1Q; //#12136
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#12137
        $_.enc = $get($_.encs, $_.indx); //#12138
        $puti($_.sbs, $_.i * 8, $_.enc); //#12139
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#12140
        for (var _1m = 0; _1m <= 7; _1m += 1) { //#12143
            $_.xpos = $f(($get($_.enc, _1m) - 48) + $_.xpos); //#12142
        } //#12142
    } //#12142
    if ($_.includecheck) { //#12168
        $puti($_.sbs, ($_.barlen * 8) - 8, $get($_.encs, $_.checksum)); //#12149
        if ($_.includecheckintext) { //#12153
            $put($_.txt, $_.barlen - 1, $a([$geti($_.barchars, $_.checksum, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#12151
        } else { //#12153
            $put($_.txt, $_.barlen - 1, $a([" ", $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#12153
        } //#12153
        for (var _2E = 0; _2E <= 7; _2E += 1) { //#12157
            $_.xpos = $f(($get($get($_.encs, $_.checksum), _2E) - 48) + $_.xpos); //#12156
        } //#12156
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.barlen - 1, 1)); //#12159
        $_.enc = $get($_.encs, $_.indx); //#12160
        $puti($_.sbs, $_.barlen * 8, $_.enc); //#12161
        $put($_.txt, $_.barlen, $a([$geti($_.barcode, $_.barlen - 1, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#12162
    } else { //#12168
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.barlen - 1, 1)); //#12165
        $_.enc = $get($_.encs, $_.indx); //#12166
        $puti($_.sbs, ($_.barlen * 8) - 8, $_.enc); //#12167
        $put($_.txt, $_.barlen - 1, $a([$geti($_.barcode, $_.barlen - 1, 1), $_.xpos, $_.textyoffset, $_.textfont, $_.textsize])); //#12168
    } //#12168
    $k[$j++] = Infinity; //#12172
    $k[$j++] = Infinity; //#12174
    $forall($_.sbs, function() { //#12174
        var _31 = $k[--$j]; //#12174
        $k[$j++] = _31 - 48; //#12174
    }); //#12174
    var _32 = $a(); //#12174
    $k[$j++] = Infinity; //#12175
    for (var _34 = 0, _35 = ~~(($_.sbs.length + 1) / 2); _34 < _35; _34++) { //#12175
        $k[$j++] = $_.height; //#12175
    } //#12175
    var _37 = $a(); //#12175
    $k[$j++] = Infinity; //#12176
    for (var _39 = 0, _3A = ~~(($_.sbs.length + 1) / 2); _39 < _3A; _39++) { //#12176
        $k[$j++] = 0; //#12176
    } //#12176
    var _3B = $a(); //#12176
    $k[$j++] = 'ren'; //#12179
    $k[$j++] = 'renlinear'; //#12179
    $k[$j++] = 'sbs'; //#12179
    $k[$j++] = _32; //#12179
    $k[$j++] = 'bhs'; //#12179
    $k[$j++] = _37; //#12179
    $k[$j++] = 'bbs'; //#12179
    $k[$j++] = _3B; //#12179
    if ($_.includetext) { //#12179
        $k[$j++] = 'txt'; //#12178
        $k[$j++] = $_.txt; //#12178
    } //#12178
    $k[$j++] = 'opt'; //#12180
    $k[$j++] = $_.options; //#12180
    var _3F = $d(); //#12180
    $k[$j++] = _3F; //#12183
    if (!$_.dontdraw) { //#12183
        bwipp_renlinear(); //#12183
    } //#12183
    $_ = $__; //#12187
} //bwipp_rationalizedCodabar
function bwipp_onecode() {
    var $__ = $_; //#12217
    $_ = Object.assign({}, $_); //#12217
    $_.dontdraw = false; //#12220
    $_.height = 0.15; //#12221
    $k[$j++] = $_; //#12223
    bwipp_processoptions(); //#12223
    $_.options = $k[--$j]; //#12223
    $_.barcode = $k[--$j]; //#12224
    $_.barlen = $_.barcode.length; //#12228
    $forall($_.barcode, function() { //#12234
        var _4 = $k[--$j]; //#12231
        if ((_4 < 48) || (_4 > 57)) { //#12233
            $k[$j++] = 'bwipp.onecodeBadCharacter#12232'; //#12232
            $k[$j++] = "The data must contain only digits"; //#12232
            bwipp_raiseerror(); //#12232
        } //#12232
    }); //#12232
    if (((($_.barlen != 20) && ($_.barlen != 25)) && ($_.barlen != 29)) && ($_.barlen != 31)) { //#12238
        $k[$j++] = 'bwipp.onecodeInvalidLength#12237'; //#12237
        $k[$j++] = "The data must be 20, 25, 29 or 31 digits"; //#12237
        bwipp_raiseerror(); //#12237
    } //#12237
    $k[$j++] = Infinity; //#12241
    var _B = $eq($geti($_.barcode, 5, 1), "9") ? 14 : 11; //#12244
    $k[$j++] = 2; //#12247
    $k[$j++] = 2; //#12247
    $k[$j++] = 5; //#12247
    $k[$j++] = 5; //#12247
    $k[$j++] = _B; //#12247
    $k[$j++] = _B; //#12247
    $k[$j++] = 20; //#12247
    $k[$j++] = 20; //#12247
    $k[$j++] = 25; //#12247
    $k[$j++] = 25; //#12247
    $k[$j++] = 29; //#12247
    $k[$j++] = 29; //#12247
    $_.txtdict = $d(); //#12248
    $_.txt = $a($_.barlen + 6); //#12249
    $_.spacecnt = 0; //#12250
    for (var _H = 0, _G = $_.barlen - 1; _H <= _G; _H += 1) { //#12258
        $_.i = _H; //#12252
        var _K = $get($_.txtdict, $_.i) !== undefined; //#12253
        if (_K) { //#12256
            $put($_.txt, $_.i + $_.spacecnt, $a([" ", 0, 0, "", 0])); //#12254
            $_.spacecnt = $_.spacecnt + 1; //#12255
        } //#12255
        $put($_.txt, $_.i + $_.spacecnt, $a([$geti($_.barcode, $_.i, 1), 0, 0, "", 0])); //#12257
    } //#12257
    $_.txt = $geti($_.txt, 0, $_.barcode.length + $_.spacecnt); //#12259
    $_.normalize = function() {
        $_.base = $k[--$j]; //#12262
        $_.num = $k[--$j]; //#12263
        for (var _e = $_.num.length - 1; _e >= 1; _e -= 1) { //#12268
            $_.i = _e; //#12265
            var _f = $_.num; //#12266
            var _g = $_.i; //#12266
            $put(_f, _g - 1, $f($get(_f, _g - 1) + (~~($get($_.num, $_.i) / $_.base)))); //#12266
            $put($_.num, $_.i, $get($_.num, $_.i) % $_.base); //#12267
        } //#12267
        for (;;) { //#12274
            if ($lt($get($_.num, 0), $_.base)) { //#12270
                break; //#12270
            } //#12270
            $k[$j++] = Infinity; //#12271
            $k[$j++] = 0; //#12271
            $aload($_.num); //#12271
            $_.num = $a(); //#12271
            $put($_.num, 0, $f($get($_.num, 0) + (~~($get($_.num, 1) / $_.base)))); //#12272
            $put($_.num, 1, $get($_.num, 1) % $_.base); //#12273
        } //#12273
        $k[$j++] = Infinity; //#12276
        $_.i = true; //#12276
        var _17 = $_.num; //#12276
        for (var _18 = 0, _19 = _17.length; _18 < _19; _18++) { //#12276
            var _1A = $get(_17, _18); //#12276
            $k[$j++] = _1A; //#12276
            if ((_1A == 0) && $_.i) { //#12276
                $j--; //#12276
            } else { //#12276
                $_.i = false; //#12276
            } //#12276
        } //#12276
        $_.num = $a(); //#12276
        if ($_.num.length == 0) { //#12277
            $_.num = $a([0]); //#12277
        } //#12277
        $k[$j++] = $_.num; //#12278
    }; //#12278
    $_.bigadd = function() {
        var _1G = $k[--$j]; //#12282
        var _1H = $k[--$j]; //#12282
        $_.offset = $abs(_1G.length - _1H.length); //#12283
        if (_1G.length < _1H.length) { //#12284
            var _ = _1G; //#12284
            _1G = _1H; //#12284
            _1H = _; //#12284
        } //#12284
        $_.a = _1G; //#12285
        $_.b = _1H; //#12285
        for (var _1K = 0, _1J = $_.b.length - 1; _1K <= _1J; _1K += 1) { //#12288
            var _1L = $_.a; //#12287
            var _1M = $_.offset; //#12287
            $put(_1L, _1K + _1M, $f($get(_1L, _1K + _1M) + $get($_.b, _1K))); //#12287
        } //#12287
        $k[$j++] = $_.a; //#12289
    }; //#12289
    var _1V = new Map([
        [20, $a([0])],
        [25, $a([1])],
        [29, $a([1, 0, 0, 0, 0, 1])],
        [31, $a([1, 0, 0, 0, 1, 0, 0, 0, 0, 1])]
    ]); //#12298
    $_.startvals = _1V; //#12299
    $k[$j++] = 'binval'; //#12302
    $k[$j++] = $get($_.startvals, $_.barlen); //#12302
    $k[$j++] = Infinity; //#12302
    var _1c = $geti($_.barcode, 20, $_.barlen - 20); //#12302
    for (var _1d = 0, _1e = _1c.length; _1d < _1e; _1d++) { //#12302
        $k[$j++] = $get(_1c, _1d) - 48; //#12302
    } //#12302
    var _1g = $a(); //#12302
    $k[$j++] = _1g; //#12302
    $_.bigadd(); //#12302
    var _1h = $k[--$j]; //#12302
    $_[$k[--$j]] = _1h; //#12302
    $k[$j++] = Infinity; //#12303
    $aload($_.binval); //#12303
    $k[$j++] = $get($_.barcode, 0) - 48; //#12303
    $_.binval = $a(); //#12303
    $k[$j++] = Infinity; //#12304
    var _1n = $_.binval; //#12304
    for (var _1o = 0, _1p = _1n.length; _1o < _1p; _1o++) { //#12304
        $k[$j++] = $get(_1n, _1o) * 5; //#12304
    } //#12304
    var _1r = $a(); //#12304
    $k[$j++] = 'binval'; //#12304
    $k[$j++] = _1r; //#12304
    $k[$j++] = $a([$get($_.barcode, 1) - 48]); //#12304
    $_.bigadd(); //#12304
    $k[$j++] = 10; //#12304
    $_.normalize(); //#12304
    var _1v = $k[--$j]; //#12304
    $_[$k[--$j]] = _1v; //#12304
    $k[$j++] = Infinity; //#12305
    $aload($_.binval); //#12305
    var _1z = $geti($_.barcode, 2, 18); //#12305
    for (var _20 = 0, _21 = _1z.length; _20 < _21; _20++) { //#12305
        $k[$j++] = $get(_1z, _20) - 48; //#12305
    } //#12305
    $_.binval = $a(); //#12305
    $_.bytes = $a(13); //#12308
    var _25 = $_.binval; //#12309
    $_.bintmp = $arrcpy($a(_25.length), _25); //#12309
    for (var _28 = 12; _28 >= 0; _28 -= 1) { //#12319
        $_.i = _28; //#12311
        for (var _2B = 0, _2A = $_.bintmp.length - 2; _2B <= _2A; _2B += 1) { //#12316
            $_.j = _2B; //#12313
            var _2C = $_.bintmp; //#12314
            var _2D = $_.j; //#12314
            $put(_2C, _2D + 1, $f($get(_2C, _2D + 1) + (($get($_.bintmp, $_.j) % 256) * 10))); //#12314
            $put($_.bintmp, $_.j, ~~($get($_.bintmp, $_.j) / 256)); //#12315
        } //#12315
        $put($_.bytes, $_.i, $get($_.bintmp, $_.bintmp.length - 1) % 256); //#12317
        var _2S = $_.bintmp; //#12318
        var _2T = $_.bintmp; //#12318
        $put(_2S, _2T.length - 1, ~~($get(_2S, _2T.length - 1) / 256)); //#12318
    } //#12318
    $_.fcs = 2047; //#12322
    $_.dat = $get($_.bytes, 0) << 5; //#12323
    for (var _2X = 0; _2X < 6; _2X++) { //#12332
        if ((($_.fcs ^ $_.dat) & 1024) != 0) { //#12328
            $_.fcs = ($_.fcs << 1) ^ 3893; //#12326
        } else { //#12328
            $_.fcs = $_.fcs << 1; //#12328
        } //#12328
        $_.fcs = $_.fcs & 2047; //#12330
        $_.dat = $_.dat << 1; //#12331
    } //#12331
    for (var _2e = 1; _2e <= 12; _2e += 1) { //#12344
        $_.dat = $get($_.bytes, _2e) << 3; //#12334
        for (var _2h = 0; _2h < 8; _2h++) { //#12343
            if ((($_.fcs ^ $_.dat) & 1024) != 0) { //#12339
                $_.fcs = ($_.fcs << 1) ^ 3893; //#12337
            } else { //#12339
                $_.fcs = $_.fcs << 1; //#12339
            } //#12339
            $_.fcs = $_.fcs & 2047; //#12341
            $_.dat = $_.dat << 1; //#12342
        } //#12342
    } //#12342
    $_.codewords = $a(10); //#12347
    for (var _2p = 9; _2p >= 0; _2p -= 1) { //#12358
        $_.i = _2p; //#12349
        var _2r = ($_.i == 9) ? 636 : 1365; //#12350
        $_.b = _2r; //#12350
        for (var _2u = 0, _2t = $_.binval.length - 2; _2u <= _2t; _2u += 1) { //#12355
            $_.j = _2u; //#12352
            var _2v = $_.binval; //#12353
            var _2w = $_.j; //#12353
            $put(_2v, _2w + 1, $f($get(_2v, _2w + 1) + (($get($_.binval, $_.j) % $_.b) * 10))); //#12353
            $put($_.binval, $_.j, ~~($get($_.binval, $_.j) / $_.b)); //#12354
        } //#12354
        $put($_.codewords, $_.i, $get($_.binval, $_.binval.length - 1) % $_.b); //#12356
        var _3E = $_.binval; //#12357
        var _3F = $_.binval; //#12357
        $put(_3E, _3F.length - 1, ~~($get(_3E, _3F.length - 1) / $_.b)); //#12357
    } //#12357
    $put($_.codewords, 9, $get($_.codewords, 9) * 2); //#12361
    if (($_.fcs & 1024) != 0) { //#12364
        $put($_.codewords, 0, $get($_.codewords, 0) + 659); //#12363
    } //#12363
    $_.tab513 = $a([31, 7936, 47, 7808, 55, 7552, 59, 7040, 61, 6016, 62, 3968, 79, 7744, 87, 7488, 91, 6976, 93, 5952, 94, 3904, 103, 7360, 107, 6848, 109, 5824, 110, 3776, 115, 6592, 117, 5568, 118, 3520, 121, 5056, 122, 3008, 124, 1984, 143, 7712, 151, 7456, 155, 6944, 157, 5920, 158, 3872, 167, 7328, 171, 6816, 173, 5792, 174, 3744, 179, 6560, 181, 5536, 182, 3488, 185, 5024, 186, 2976, 188, 1952, 199, 7264, 203, 6752, 205, 5728, 206, 3680, 211, 6496, 213, 5472, 214, 3424, 217, 4960, 218, 2912, 220, 1888, 227, 6368, 229, 5344, 230, 3296, 233, 4832, 234, 2784, 236, 1760, 241, 4576, 242, 2528, 244, 1504, 248, 992, 271, 7696, 279, 7440, 283, 6928, 285, 5904, 286, 3856, 295, 7312, 299, 6800, 301, 5776, 302, 3728, 307, 6544, 309, 5520, 310, 3472, 313, 5008, 314, 2960, 316, 1936, 327, 7248, 331, 6736, 333, 5712, 334, 3664, 339, 6480, 341, 5456, 342, 3408, 345, 4944, 346, 2896, 348, 1872, 355, 6352, 357, 5328, 358, 3280, 361, 4816, 362, 2768, 364, 1744, 369, 4560, 370, 2512, 372, 1488, 376, 976, 391, 7216, 395, 6704, 397, 5680, 398, 3632, 403, 6448, 405, 5424, 406, 3376, 409, 4912, 410, 2864, 412, 1840, 419, 6320, 421, 5296, 422, 3248, 425, 4784, 426, 2736, 428, 1712, 433, 4528, 434, 2480, 436, 1456, 440, 944, 451, 6256, 453, 5232, 454, 3184, 457, 4720, 458, 2672, 460, 1648, 465, 4464, 466, 2416, 468, 1392, 472, 880, 481, 4336, 482, 2288, 484, 1264, 488, 752, 527, 7688, 535, 7432, 539, 6920, 541, 5896, 542, 3848, 551, 7304, 555, 6792, 557, 5768, 558, 3720, 563, 6536, 565, 5512, 566, 3464, 569, 5000, 570, 2952, 572, 1928, 583, 7240, 587, 6728, 589, 5704, 590, 3656, 595, 6472, 597, 5448, 598, 3400, 601, 4936, 602, 2888, 604, 1864, 611, 6344, 613, 5320, 614, 3272, 617, 4808, 618, 2760, 620, 1736, 625, 4552, 626, 2504, 628, 1480, 632, 968, 647, 7208, 651, 6696, 653, 5672, 654, 3624, 659, 6440, 661, 5416, 662, 3368, 665, 4904, 666, 2856, 668, 1832, 675, 6312, 677, 5288, 678, 3240, 681, 4776, 682, 2728, 684, 1704, 689, 4520, 690, 2472, 692, 1448, 696, 936, 707, 6248, 709, 5224, 710, 3176, 713, 4712, 714, 2664, 716, 1640, 721, 4456, 722, 2408, 724, 1384, 728, 872, 737, 4328, 738, 2280, 740, 1256, 775, 7192, 779, 6680, 781, 5656, 782, 3608, 787, 6424, 789, 5400, 790, 3352, 793, 4888, 794, 2840, 796, 1816, 803, 6296, 805, 5272, 806, 3224, 809, 4760, 810, 2712, 812, 1688, 817, 4504, 818, 2456, 820, 1432, 824, 920, 835, 6232, 837, 5208, 838, 3160, 841, 4696, 842, 2648, 844, 1624, 849, 4440, 850, 2392, 852, 1368, 865, 4312, 866, 2264, 868, 1240, 899, 6200, 901, 5176, 902, 3128, 905, 4664, 906, 2616, 908, 1592, 913, 4408, 914, 2360, 916, 1336, 929, 4280, 930, 2232, 932, 1208, 961, 4216, 962, 2168, 964, 1144, 1039, 7684, 1047, 7428, 1051, 6916, 1053, 5892, 1054, 3844, 1063, 7300, 1067, 6788, 1069, 5764, 1070, 3716, 1075, 6532, 1077, 5508, 1078, 3460, 1081, 4996, 1082, 2948, 1084, 1924, 1095, 7236, 1099, 6724, 1101, 5700, 1102, 3652, 1107, 6468, 1109, 5444, 1110, 3396, 1113, 4932, 1114, 2884, 1116, 1860, 1123, 6340, 1125, 5316, 1126, 3268, 1129, 4804, 1130, 2756, 1132, 1732, 1137, 4548, 1138, 2500, 1140, 1476, 1159, 7204, 1163, 6692, 1165, 5668, 1166, 3620, 1171, 6436, 1173, 5412, 1174, 3364, 1177, 4900, 1178, 2852, 1180, 1828, 1187, 6308, 1189, 5284, 1190, 3236, 1193, 4772, 1194, 2724, 1196, 1700, 1201, 4516, 1202, 2468, 1204, 1444, 1219, 6244, 1221, 5220, 1222, 3172, 1225, 4708, 1226, 2660, 1228, 1636, 1233, 4452, 1234, 2404, 1236, 1380, 1249, 4324, 1250, 2276, 1287, 7188, 1291, 6676, 1293, 5652, 1294, 3604, 1299, 6420, 1301, 5396, 1302, 3348, 1305, 4884, 1306, 2836, 1308, 1812, 1315, 6292, 1317, 5268, 1318, 3220, 1321, 4756, 1322, 2708, 1324, 1684, 1329, 4500, 1330, 2452, 1332, 1428, 1347, 6228, 1349, 5204, 1350, 3156, 1353, 4692, 1354, 2644, 1356, 1620, 1361, 4436, 1362, 2388, 1377, 4308, 1378, 2260, 1411, 6196, 1413, 5172, 1414, 3124, 1417, 4660, 1418, 2612, 1420, 1588, 1425, 4404, 1426, 2356, 1441, 4276, 1442, 2228, 1473, 4212, 1474, 2164, 1543, 7180, 1547, 6668, 1549, 5644, 1550, 3596, 1555, 6412, 1557, 5388, 1558, 3340, 1561, 4876, 1562, 2828, 1564, 1804, 1571, 6284, 1573, 5260, 1574, 3212, 1577, 4748, 1578, 2700, 1580, 1676, 1585, 4492, 1586, 2444, 1603, 6220, 1605, 5196, 1606, 3148, 1609, 4684, 1610, 2636, 1617, 4428, 1618, 2380, 1633, 4300, 1634, 2252, 1667, 6188, 1669, 5164, 1670, 3116, 1673, 4652, 1674, 2604, 1681, 4396, 1682, 2348, 1697, 4268, 1698, 2220, 1729, 4204, 1730, 2156, 1795, 6172, 1797, 5148, 1798, 3100, 1801, 4636, 1802, 2588, 1809, 4380, 1810, 2332, 1825, 4252, 1826, 2204, 1857, 4188, 1858, 2140, 1921, 4156, 1922, 2108, 2063, 7682, 2071, 7426, 2075, 6914, 2077, 5890, 2078, 3842, 2087, 7298, 2091, 6786, 2093, 5762, 2094, 3714, 2099, 6530, 2101, 5506, 2102, 3458, 2105, 4994, 2106, 2946, 2119, 7234, 2123, 6722, 2125, 5698, 2126, 3650, 2131, 6466, 2133, 5442, 2134, 3394, 2137, 4930, 2138, 2882, 2147, 6338, 2149, 5314, 2150, 3266, 2153, 4802, 2154, 2754, 2161, 4546, 2162, 2498, 2183, 7202, 2187, 6690, 2189, 5666, 2190, 3618, 2195, 6434, 2197, 5410, 2198, 3362, 2201, 4898, 2202, 2850, 2211, 6306, 2213, 5282, 2214, 3234, 2217, 4770, 2218, 2722, 2225, 4514, 2226, 2466, 2243, 6242, 2245, 5218, 2246, 3170, 2249, 4706, 2250, 2658, 2257, 4450, 2258, 2402, 2273, 4322, 2311, 7186, 2315, 6674, 2317, 5650, 2318, 3602, 2323, 6418, 2325, 5394, 2326, 3346, 2329, 4882, 2330, 2834, 2339, 6290, 2341, 5266, 2342, 3218, 2345, 4754, 2346, 2706, 2353, 4498, 2354, 2450, 2371, 6226, 2373, 5202, 2374, 3154, 2377, 4690, 2378, 2642, 2385, 4434, 2401, 4306, 2435, 6194, 2437, 5170, 2438, 3122, 2441, 4658, 2442, 2610, 2449, 4402, 2465, 4274, 2497, 4210, 2567, 7178, 2571, 6666, 2573, 5642, 2574, 3594, 2579, 6410, 2581, 5386, 2582, 3338, 2585, 4874, 2586, 2826, 2595, 6282, 2597, 5258, 2598, 3210, 2601, 4746, 2602, 2698, 2609, 4490, 2627, 6218, 2629, 5194, 2630, 3146, 2633, 4682, 2641, 4426, 2657, 4298, 2691, 6186, 2693, 5162, 2694, 3114, 2697, 4650, 2705, 4394, 2721, 4266, 2753, 4202, 2819, 6170, 2821, 5146, 2822, 3098, 2825, 4634, 2833, 4378, 2849, 4250, 2881, 4186, 2945, 4154, 3079, 7174, 3083, 6662, 3085, 5638, 3086, 3590, 3091, 6406, 3093, 5382, 3094, 3334, 3097, 4870, 3107, 6278, 3109, 5254, 3110, 3206, 3113, 4742, 3121, 4486, 3139, 6214, 3141, 5190, 3145, 4678, 3153, 4422, 3169, 4294, 3203, 6182, 3205, 5158, 3209, 4646, 3217, 4390, 3233, 4262, 3265, 4198, 3331, 6166, 3333, 5142, 3337, 4630, 3345, 4374, 3361, 4246, 3393, 4182, 3457, 4150, 3587, 6158, 3589, 5134, 3593, 4622, 3601, 4366, 3617, 4238, 3649, 4174, 3713, 4142, 3841, 4126, 4111, 7681, 4119, 7425, 4123, 6913, 4125, 5889, 4135, 7297, 4139, 6785, 4141, 5761, 4147, 6529, 4149, 5505, 4153, 4993, 4167, 7233, 4171, 6721, 4173, 5697, 4179, 6465, 4181, 5441, 4185, 4929, 4195, 6337, 4197, 5313, 4201, 4801, 4209, 4545, 4231, 7201, 4235, 6689, 4237, 5665, 4243, 6433, 4245, 5409, 4249, 4897, 4259, 6305, 4261, 5281, 4265, 4769, 4273, 4513, 4291, 6241, 4293, 5217, 4297, 4705, 4305, 4449, 4359, 7185, 4363, 6673, 4365, 5649, 4371, 6417, 4373, 5393, 4377, 4881, 4387, 6289, 4389, 5265, 4393, 4753, 4401, 4497, 4419, 6225, 4421, 5201, 4425, 4689, 4483, 6193, 4485, 5169, 4489, 4657, 4615, 7177, 4619, 6665, 4621, 5641, 4627, 6409, 4629, 5385, 4633, 4873, 4643, 6281, 4645, 5257, 4649, 4745, 4675, 6217, 4677, 5193, 4739, 6185, 4741, 5161, 4867, 6169, 4869, 5145, 5127, 7173, 5131, 6661, 5133, 5637, 5139, 6405, 5141, 5381, 5155, 6277, 5157, 5253, 5187, 6213, 5251, 6181, 5379, 6165, 5635, 6157, 6151, 7171, 6155, 6659, 6163, 6403, 6179, 6275, 6211, 5189, 4681, 4433, 4321, 3142, 2634, 2386, 2274, 1612, 1364, 1252, 856, 744, 496]); //#12455
    $_.tab213 = $a([3, 6144, 5, 5120, 6, 3072, 9, 4608, 10, 2560, 12, 1536, 17, 4352, 18, 2304, 20, 1280, 24, 768, 33, 4224, 34, 2176, 36, 1152, 40, 640, 48, 384, 65, 4160, 66, 2112, 68, 1088, 72, 576, 80, 320, 96, 192, 129, 4128, 130, 2080, 132, 1056, 136, 544, 144, 288, 257, 4112, 258, 2064, 260, 1040, 264, 528, 513, 4104, 514, 2056, 516, 1032, 1025, 4100, 1026, 2052, 2049, 4098, 4097, 2050, 1028, 520, 272, 160]); //#12464
    $_.chars = $a(10); //#12467
    for (var _3T = 0; _3T <= 9; _3T += 1) { //#12476
        $_.i = _3T; //#12469
        var _3W = $get($_.codewords, $_.i); //#12470
        $k[$j++] = _3W; //#12474
        if (_3W <= 1286) { //#12473
            var _3Z = $get($_.tab513, $k[--$j]); //#12471
            $k[$j++] = _3Z; //#12471
        } else { //#12473
            var _3c = $get($_.tab213, $k[--$j] - 1287); //#12473
            $k[$j++] = _3c; //#12473
        } //#12473
        $k[$j++] = $_.chars; //#12475
        $k[$j++] = $_.i; //#12475
        var _3f = $k[$j - 3]; //#12475
        var _3g = $k[$j - 1]; //#12475
        $put($k[$j - 2], _3g, _3f); //#12475
        $j -= 3; //#12475
    } //#12475
    for (var _3i = 9; _3i >= 0; _3i -= 1) { //#12483
        $_.i = _3i; //#12479
        if (((~~$pow(2, $_.i)) & $_.fcs) != 0) { //#12482
            $put($_.chars, $_.i, $get($_.chars, $_.i) ^ 8191); //#12481
        } //#12481
    } //#12481
    $_.barmap = $a([7, 2, 4, 3, 1, 10, 0, 0, 9, 12, 2, 8, 5, 5, 6, 11, 8, 9, 3, 1, 0, 1, 5, 12, 2, 5, 1, 8, 4, 4, 9, 11, 6, 3, 8, 10, 3, 9, 7, 6, 5, 11, 1, 4, 8, 5, 2, 12, 9, 10, 0, 2, 7, 1, 6, 7, 3, 6, 4, 9, 0, 3, 8, 6, 6, 4, 2, 7, 1, 1, 9, 9, 7, 10, 5, 2, 4, 0, 3, 8, 6, 2, 0, 4, 8, 11, 1, 0, 9, 8, 3, 12, 2, 6, 7, 7, 5, 1, 4, 10, 1, 12, 6, 9, 7, 3, 8, 0, 5, 8, 9, 7, 4, 6, 2, 10, 3, 4, 0, 5, 8, 4, 5, 7, 7, 11, 1, 9, 6, 0, 9, 6, 0, 6, 4, 8, 2, 1, 3, 2, 5, 9, 8, 12, 4, 11, 6, 1, 9, 5, 7, 4, 3, 3, 1, 2, 0, 7, 2, 0, 1, 3, 4, 1, 6, 10, 3, 5, 8, 7, 9, 4, 2, 11, 5, 6, 0, 8, 7, 12, 4, 2, 8, 1, 5, 10, 3, 0, 9, 3, 0, 9, 6, 5, 2, 4, 7, 8, 1, 7, 5, 0, 4, 5, 2, 3, 0, 10, 6, 12, 9, 2, 3, 11, 1, 6, 8, 8, 7, 9, 5, 4, 0, 11, 1, 5, 2, 2, 9, 1, 4, 12, 8, 3, 6, 6, 7, 0, 3, 7, 4, 7, 7, 5, 0, 12, 1, 11, 2, 9, 9, 0, 6, 8, 5, 3, 3, 10, 8, 2]); //#12501
    $_.bbs = $a(65); //#12504
    $_.bhs = $a(65); //#12505
    for (var _3u = 0; _3u <= 64; _3u += 1) { //#12526
        $_.i = _3u; //#12507
        $_.dec = ($get($_.chars, $get($_.barmap, $_.i * 4)) & (~~$pow(2, $get($_.barmap, ($_.i * 4) + 1)))) != 0; //#12508
        $_.asc = ($get($_.chars, $get($_.barmap, ($_.i * 4) + 2)) & (~~$pow(2, $get($_.barmap, ($_.i * 4) + 3)))) != 0; //#12509
        if ((!$_.dec) && (!$_.asc)) { //#12513
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#12511
            $put($_.bhs, $_.i, (2 * $_.height) / 8); //#12512
        } //#12512
        if ((!$_.dec) && $_.asc) { //#12517
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#12515
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#12516
        } //#12516
        if ($_.dec && (!$_.asc)) { //#12521
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#12519
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#12520
        } //#12520
        if ($_.dec && $_.asc) { //#12525
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#12523
            $put($_.bhs, $_.i, (8 * $_.height) / 8); //#12524
        } //#12524
    } //#12524
    $k[$j++] = Infinity; //#12529
    var _4h = $_.bbs; //#12531
    var _4i = $_.bhs; //#12532
    $k[$j++] = Infinity; //#12533
    for (var _4k = 0, _4l = $_.bhs.length - 1; _4k < _4l; _4k++) { //#12533
        $k[$j++] = 1.44; //#12533
        $k[$j++] = 1.872; //#12533
    } //#12533
    $k[$j++] = 1.44; //#12533
    var _4m = $a(); //#12533
    $k[$j++] = 'ren'; //#12539
    $k[$j++] = 'renlinear'; //#12539
    $k[$j++] = 'bbs'; //#12539
    $k[$j++] = _4h; //#12539
    $k[$j++] = 'bhs'; //#12539
    $k[$j++] = _4i; //#12539
    $k[$j++] = 'sbs'; //#12539
    $k[$j++] = _4m; //#12539
    $k[$j++] = 'txt'; //#12539
    $k[$j++] = $_.txt; //#12539
    $k[$j++] = 'textxalign'; //#12539
    $k[$j++] = "left"; //#12539
    $k[$j++] = 'textfont'; //#12539
    $k[$j++] = "OCR-B"; //#12539
    $k[$j++] = 'textyoffset'; //#12539
    $k[$j++] = 1; //#12539
    $k[$j++] = 'textxoffset'; //#12539
    $k[$j++] = -0.3; //#12539
    $k[$j++] = 'opt'; //#12539
    $k[$j++] = $_.options; //#12539
    var _4p = $d(); //#12539
    $k[$j++] = _4p; //#12542
    if (!$_.dontdraw) { //#12542
        bwipp_renlinear(); //#12542
    } //#12542
    $_ = $__; //#12546
} //bwipp_onecode
function bwipp_postnet() {
    var $__ = $_; //#12576
    $_ = Object.assign({}, $_); //#12576
    $_.dontdraw = false; //#12579
    $_.includetext = false; //#12580
    $_.validatecheck = false; //#12581
    $_.includecheckintext = false; //#12582
    $_.textfont = "OCR-B"; //#12583
    $_.textsize = 10; //#12584
    $_.textyoffset = -8; //#12585
    $_.height = 0.125; //#12586
    $k[$j++] = $_; //#12588
    bwipp_processoptions(); //#12588
    $_.options = $k[--$j]; //#12588
    $_.barcode = $k[--$j]; //#12589
    $k[$j++] = 'barlen'; //#12593
    $k[$j++] = $_.barcode.length; //#12593
    if ($_.validatecheck) { //#12593
        var _4 = $k[--$j]; //#12593
        $k[$j++] = _4 - 1; //#12593
    } //#12593
    var _5 = $k[--$j]; //#12593
    $_[$k[--$j]] = _5; //#12593
    if ((($_.barlen != 5) && ($_.barlen != 9)) && ($_.barlen != 11)) { //#12598
        $k[$j++] = 'bwipp.postnetBadLength#12597'; //#12597
        $k[$j++] = "USPS POSTNET must be 5, 9 or 11 digits excluding check digit"; //#12597
        bwipp_raiseerror(); //#12597
    } //#12597
    $forall($_.barcode, function() { //#12603
        var _B = $k[--$j]; //#12600
        if ((_B < 48) || (_B > 57)) { //#12602
            $k[$j++] = 'bwipp.postnetBadCharacter#12601'; //#12601
            $k[$j++] = "USPS POSTNET must contain only digits"; //#12601
            bwipp_raiseerror(); //#12601
        } //#12601
    }); //#12601
    $_.barchars = "0123456789"; //#12607
    $_.encs = $a(["55222", "22255", "22525", "22552", "25225", "25252", "25522", "52225", "52252", "52522", "5", "5"]); //#12614
    $_.checksum = 0; //#12617
    for (var _G = 0, _F = $_.barlen - 1; _G <= _F; _G += 1) { //#12621
        $_.i = _G; //#12619
        $_.checksum = $f($_.checksum + ($get($_.barcode, $_.i) - 48)); //#12620
    } //#12620
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#12622
    if ($_.validatecheck) { //#12628
        if ($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum)) { //#12626
            $k[$j++] = 'bwipp.postnetBadCheckDigit#12625'; //#12625
            $k[$j++] = "Incorrect USPS POSTNET check digit provided"; //#12625
            bwipp_raiseerror(); //#12625
        } //#12625
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#12627
    } //#12627
    $_.bhs = $a(($_.barlen * 5) + 7); //#12630
    $_.txt = $a($_.barlen + 1); //#12631
    $_.enc = $get($_.encs, 10); //#12634
    $_.heights = $a($_.enc.length); //#12635
    for (var _g = 0, _f = $_.enc.length - 1; _g <= _f; _g += 1) { //#12639
        $_.j = _g; //#12637
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12638
    } //#12638
    $puti($_.bhs, 0, $_.heights); //#12640
    for (var _r = 0, _q = $_.barlen - 1; _r <= _q; _r += 1) { //#12652
        $_.i = _r; //#12643
        $_.enc = $get($_.encs, $get($_.barcode, $_.i) - 48); //#12644
        $_.heights = $a($_.enc.length); //#12645
        for (var _11 = 0, _10 = $_.enc.length - 1; _11 <= _10; _11 += 1) { //#12649
            $_.j = _11; //#12647
            $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12648
        } //#12648
        $puti($_.bhs, ($_.i * 5) + 1, $_.heights); //#12650
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 5) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12651
    } //#12651
    $_.enc = $get($_.encs, $_.checksum); //#12655
    $_.heights = $a($_.enc.length); //#12656
    for (var _1S = 0, _1R = $_.enc.length - 1; _1S <= _1R; _1S += 1) { //#12660
        $_.j = _1S; //#12658
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12659
    } //#12659
    $puti($_.bhs, ($_.barlen * 5) + 1, $_.heights); //#12661
    if ($_.includecheckintext) { //#12666
        $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum, 1), (($_.barlen * 5) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12664
    } else { //#12666
        $put($_.txt, $_.barlen, $a([" ", ((($_.barlen * 5) + 1) * 72) / 25, $_.textyoffset, $_.textfont, $_.textsize])); //#12666
    } //#12666
    $_.enc = $get($_.encs, 11); //#12670
    $_.heights = $a($_.enc.length); //#12671
    for (var _20 = 0, _1z = $_.enc.length - 1; _20 <= _1z; _20 += 1) { //#12675
        $_.j = _20; //#12673
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12674
    } //#12674
    $puti($_.bhs, ($_.barlen * 5) + 6, $_.heights); //#12676
    $k[$j++] = Infinity; //#12679
    var _2A = $_.bhs; //#12681
    $k[$j++] = Infinity; //#12682
    for (var _2C = 0, _2D = $_.bhs.length; _2C < _2D; _2C++) { //#12682
        $k[$j++] = 0; //#12682
    } //#12682
    var _2E = $a(); //#12682
    $k[$j++] = Infinity; //#12683
    for (var _2G = 0, _2H = $_.bhs.length - 1; _2G < _2H; _2G++) { //#12683
        $k[$j++] = 1.44; //#12683
        $k[$j++] = 1.872; //#12683
    } //#12683
    $k[$j++] = 1.44; //#12683
    var _2I = $a(); //#12683
    $k[$j++] = 'ren'; //#12686
    $k[$j++] = 'renlinear'; //#12686
    $k[$j++] = 'bhs'; //#12686
    $k[$j++] = _2A; //#12686
    $k[$j++] = 'bbs'; //#12686
    $k[$j++] = _2E; //#12686
    $k[$j++] = 'sbs'; //#12686
    $k[$j++] = _2I; //#12686
    if ($_.includetext) { //#12686
        $k[$j++] = 'txt'; //#12685
        $k[$j++] = $_.txt; //#12685
    } //#12685
    $k[$j++] = 'opt'; //#12687
    $k[$j++] = $_.options; //#12687
    var _2M = $d(); //#12687
    $k[$j++] = _2M; //#12690
    if (!$_.dontdraw) { //#12690
        bwipp_renlinear(); //#12690
    } //#12690
    $_ = $__; //#12694
} //bwipp_postnet
function bwipp_planet() {
    var $__ = $_; //#12724
    $_ = Object.assign({}, $_); //#12724
    $_.dontdraw = false; //#12727
    $_.includetext = false; //#12728
    $_.validatecheck = false; //#12729
    $_.includecheckintext = false; //#12730
    $_.textfont = "OCR-B"; //#12731
    $_.textsize = 10; //#12732
    $_.textyoffset = -8; //#12733
    $_.height = 0.125; //#12734
    $k[$j++] = $_; //#12736
    bwipp_processoptions(); //#12736
    $_.options = $k[--$j]; //#12736
    $_.barcode = $k[--$j]; //#12737
    $k[$j++] = 'barlen'; //#12741
    $k[$j++] = $_.barcode.length; //#12741
    if ($_.validatecheck) { //#12741
        var _4 = $k[--$j]; //#12741
        $k[$j++] = _4 - 1; //#12741
    } //#12741
    var _5 = $k[--$j]; //#12741
    $_[$k[--$j]] = _5; //#12741
    if (($_.barlen != 11) && ($_.barlen != 13)) { //#12746
        $k[$j++] = 'bwipp.planetBadLength#12745'; //#12745
        $k[$j++] = "USPS PLANET must be 11 or 13 digits excluding check digit"; //#12745
        bwipp_raiseerror(); //#12745
    } //#12745
    $forall($_.barcode, function() { //#12751
        var _A = $k[--$j]; //#12748
        if ((_A < 48) || (_A > 57)) { //#12750
            $k[$j++] = 'bwipp.planetBadCharacter#12749'; //#12749
            $k[$j++] = "USPS PLANET must contain only digits"; //#12749
            bwipp_raiseerror(); //#12749
        } //#12749
    }); //#12749
    $_.barchars = "0123456789"; //#12755
    $_.encs = $a(["22555", "55522", "55252", "55225", "52552", "52525", "52255", "25552", "25525", "25255", "5", "5"]); //#12762
    $_.checksum = 0; //#12766
    for (var _F = 0, _E = $_.barlen - 1; _F <= _E; _F += 1) { //#12770
        $_.i = _F; //#12768
        $_.checksum = $f($_.checksum + ($get($_.barcode, $_.i) - 48)); //#12769
    } //#12769
    $_.checksum = (10 - ($_.checksum % 10)) % 10; //#12771
    if ($_.validatecheck) { //#12777
        if ($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum)) { //#12775
            $k[$j++] = 'bwipp.planetBadCheckDigit#12774'; //#12774
            $k[$j++] = "Incorrect USPS PLANET check digit provided"; //#12774
            bwipp_raiseerror(); //#12774
        } //#12774
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#12776
    } //#12776
    $_.bhs = $a(($_.barlen * 5) + 7); //#12779
    $_.txt = $a($_.barlen + 1); //#12780
    $_.enc = $get($_.encs, 10); //#12783
    $_.heights = $a($_.enc.length); //#12784
    for (var _f = 0, _e = $_.enc.length - 1; _f <= _e; _f += 1) { //#12788
        $_.j = _f; //#12786
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12787
    } //#12787
    $puti($_.bhs, 0, $_.heights); //#12789
    for (var _q = 0, _p = $_.barlen - 1; _q <= _p; _q += 1) { //#12801
        $_.i = _q; //#12792
        $_.enc = $get($_.encs, $get($_.barcode, $_.i) - 48); //#12793
        $_.heights = $a($_.enc.length); //#12794
        for (var _10 = 0, _z = $_.enc.length - 1; _10 <= _z; _10 += 1) { //#12798
            $_.j = _10; //#12796
            $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12797
        } //#12797
        $puti($_.bhs, ($_.i * 5) + 1, $_.heights); //#12799
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 5) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12800
    } //#12800
    $_.enc = $get($_.encs, $_.checksum); //#12804
    $_.heights = $a($_.enc.length); //#12805
    for (var _1R = 0, _1Q = $_.enc.length - 1; _1R <= _1Q; _1R += 1) { //#12809
        $_.j = _1R; //#12807
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12808
    } //#12808
    $puti($_.bhs, ($_.barlen * 5) + 1, $_.heights); //#12810
    if ($_.includecheckintext) { //#12815
        $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum, 1), (($_.barlen * 5) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12813
    } else { //#12815
        $put($_.txt, $_.barlen, $a([" ", ((($_.barlen * 5) + 1) * 72) / 25, $_.textyoffset, $_.textfont, $_.textsize])); //#12815
    } //#12815
    $_.enc = $get($_.encs, 11); //#12819
    $_.heights = $a($_.enc.length); //#12820
    for (var _1z = 0, _1y = $_.enc.length - 1; _1z <= _1y; _1z += 1) { //#12824
        $_.j = _1z; //#12822
        $put($_.heights, $_.j, ($cvi($geti($_.enc, $_.j, 1)) * $_.height) / 5); //#12823
    } //#12823
    $puti($_.bhs, ($_.barlen * 5) + 6, $_.heights); //#12825
    $k[$j++] = Infinity; //#12828
    var _29 = $_.bhs; //#12830
    $k[$j++] = Infinity; //#12831
    for (var _2B = 0, _2C = $_.bhs.length; _2B < _2C; _2B++) { //#12831
        $k[$j++] = 0; //#12831
    } //#12831
    var _2D = $a(); //#12831
    $k[$j++] = Infinity; //#12832
    for (var _2F = 0, _2G = $_.bhs.length - 1; _2F < _2G; _2F++) { //#12832
        $k[$j++] = 1.44; //#12832
        $k[$j++] = 1.872; //#12832
    } //#12832
    $k[$j++] = 1.44; //#12832
    var _2H = $a(); //#12832
    $k[$j++] = 'ren'; //#12835
    $k[$j++] = 'renlinear'; //#12835
    $k[$j++] = 'bhs'; //#12835
    $k[$j++] = _29; //#12835
    $k[$j++] = 'bbs'; //#12835
    $k[$j++] = _2D; //#12835
    $k[$j++] = 'sbs'; //#12835
    $k[$j++] = _2H; //#12835
    if ($_.includetext) { //#12835
        $k[$j++] = 'txt'; //#12834
        $k[$j++] = $_.txt; //#12834
    } //#12834
    $k[$j++] = 'opt'; //#12836
    $k[$j++] = $_.options; //#12836
    var _2L = $d(); //#12836
    $k[$j++] = _2L; //#12839
    if (!$_.dontdraw) { //#12839
        bwipp_renlinear(); //#12839
    } //#12839
    $_ = $__; //#12843
} //bwipp_planet
function bwipp_royalmail() {
    var $__ = $_; //#12873
    $_ = Object.assign({}, $_); //#12873
    $_.dontdraw = false; //#12876
    $_.includetext = false; //#12877
    $_.validatecheck = false; //#12878
    $_.includecheckintext = false; //#12879
    $_.textfont = "OCR-B"; //#12880
    $_.textsize = 10; //#12881
    $_.textyoffset = -8; //#12882
    $_.height = 0.175; //#12883
    $k[$j++] = $_; //#12885
    bwipp_processoptions(); //#12885
    $_.options = $k[--$j]; //#12885
    $_.barcode = $k[--$j]; //#12886
    if ($_.barcode.length == 0) { //#12890
        $k[$j++] = 'bwipp.royalmailEmptyData#12889'; //#12889
        $k[$j++] = "The data must not be empty"; //#12889
        bwipp_raiseerror(); //#12889
    } //#12889
    $_.barchars = "ZUVWXY501234B6789AHCDEFGNIJKLMTOPQRS"; //#12896
    $_.charvals = new Map; //#12897
    for (var _3 = 0; _3 <= 35; _3 += 1) { //#12898
        $put($_.charvals, $geti($_.barchars, _3, 1), _3); //#12898
    } //#12898
    $_.encs = $a(["3300", "2211", "2301", "2310", "3201", "3210", "1122", "0033", "0123", "0132", "1023", "1032", "1302", "0213", "0303", "0312", "1203", "1212", "1320", "0231", "0321", "0330", "1221", "1230", "3102", "2013", "2103", "2112", "3003", "3012", "3120", "2031", "2121", "2130", "3021", "3030", "2", "3"]); //#12909
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#12917
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#12914
        if (!_F) { //#12916
            $k[$j++] = 'bwipp.royalmailBadCharacter#12915'; //#12915
            $k[$j++] = "RM4SCC must contain only capital letters and digits"; //#12915
            bwipp_raiseerror(); //#12915
        } //#12915
    } //#12915
    $k[$j++] = 'barlen'; //#12919
    $k[$j++] = $_.barcode.length; //#12919
    if ($_.validatecheck) { //#12919
        var _I = $k[--$j]; //#12919
        $k[$j++] = _I - 1; //#12919
    } //#12919
    var _J = $k[--$j]; //#12919
    $_[$k[--$j]] = _J; //#12919
    $_.checksumrow = 0; //#12921
    $_.checksumcol = 0; //#12922
    for (var _N = 0, _M = $_.barlen - 1; _N <= _M; _N += 1) { //#12928
        $_.i = _N; //#12924
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#12925
        $_.checksumrow = $_.checksumrow + (~~($_.indx / 6)); //#12926
        $_.checksumcol = $f($_.checksumcol + ($_.indx % 6)); //#12927
    } //#12927
    $_.checksum = $f((($_.checksumrow % 6) * 6) + ($_.checksumcol % 6)); //#12929
    if ($_.validatecheck) { //#12935
        if ($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum)) { //#12933
            $k[$j++] = 'bwipp.royalmailBadCheckDigit#12932'; //#12932
            $k[$j++] = "Incorrect RM4SCC check digit provided"; //#12932
            bwipp_raiseerror(); //#12932
        } //#12932
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#12934
    } //#12934
    $_.encstr = $s(($_.barlen * 4) + 6); //#12937
    $_.txt = $a($_.barlen + 1); //#12938
    $puti($_.encstr, 0, $get($_.encs, 36)); //#12941
    for (var _s = 0, _r = $_.barlen - 1; _s <= _r; _s += 1) { //#12948
        $_.i = _s; //#12944
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#12945
        $puti($_.encstr, ($_.i * 4) + 1, $get($_.encs, $_.indx)); //#12946
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.i * 4) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12947
    } //#12947
    $puti($_.encstr, ($_.barlen * 4) + 1, $get($_.encs, $_.checksum)); //#12951
    if ($_.includecheckintext) { //#12955
        $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum, 1), (($_.barlen * 4) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12953
    } else { //#12955
        $put($_.txt, $_.barlen, $a([" ", (($_.barlen * 4) + 1) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#12955
    } //#12955
    $puti($_.encstr, ($_.barlen * 4) + 5, $get($_.encs, 37)); //#12959
    $_.bbs = $a($_.encstr.length); //#12961
    $_.bhs = $a($_.encstr.length); //#12962
    for (var _1k = 0, _1j = $_.encstr.length - 1; _1k <= _1j; _1k += 1) { //#12982
        $_.i = _1k; //#12964
        $_.enc = $geti($_.encstr, $_.i, 1); //#12965
        if ($eq($_.enc, "0")) { //#12969
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#12967
            $put($_.bhs, $_.i, (2 * $_.height) / 8); //#12968
        } //#12968
        if ($eq($_.enc, "1")) { //#12973
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#12971
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#12972
        } //#12972
        if ($eq($_.enc, "2")) { //#12977
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#12975
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#12976
        } //#12976
        if ($eq($_.enc, "3")) { //#12981
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#12979
            $put($_.bhs, $_.i, (8 * $_.height) / 8); //#12980
        } //#12980
    } //#12980
    $k[$j++] = Infinity; //#12985
    var _2G = $_.bbs; //#12987
    var _2H = $_.bhs; //#12988
    $k[$j++] = Infinity; //#12989
    for (var _2J = 0, _2K = $_.bhs.length - 1; _2J < _2K; _2J++) { //#12989
        $k[$j++] = 1.44; //#12989
        $k[$j++] = 1.872; //#12989
    } //#12989
    $k[$j++] = 1.44; //#12989
    var _2L = $a(); //#12989
    $k[$j++] = 'ren'; //#12992
    $k[$j++] = 'renlinear'; //#12992
    $k[$j++] = 'bbs'; //#12992
    $k[$j++] = _2G; //#12992
    $k[$j++] = 'bhs'; //#12992
    $k[$j++] = _2H; //#12992
    $k[$j++] = 'sbs'; //#12992
    $k[$j++] = _2L; //#12992
    if ($_.includetext) { //#12992
        $k[$j++] = 'txt'; //#12991
        $k[$j++] = $_.txt; //#12991
    } //#12991
    $k[$j++] = 'opt'; //#12993
    $k[$j++] = $_.options; //#12993
    var _2P = $d(); //#12993
    $k[$j++] = _2P; //#12996
    if (!$_.dontdraw) { //#12996
        bwipp_renlinear(); //#12996
    } //#12996
    $_ = $__; //#13000
} //bwipp_royalmail
function bwipp_auspost() {
    var $__ = $_; //#13030
    $_ = Object.assign({}, $_); //#13030
    $_.dontdraw = false; //#13033
    $_.includetext = false; //#13034
    $_.textfont = "OCR-B"; //#13035
    $_.textsize = 10; //#13036
    $_.textyoffset = -8; //#13037
    $_.height = 0.175; //#13038
    $_.custinfoenc = "character"; //#13039
    $k[$j++] = $_; //#13041
    bwipp_processoptions(); //#13041
    $_.options = $k[--$j]; //#13041
    $_.barcode = $k[--$j]; //#13042
    if ($ne($_.custinfoenc, "character") && $ne($_.custinfoenc, "numeric")) { //#13046
        $k[$j++] = 'bwipp.auspostBadCustinfoEncoding#13045'; //#13045
        $k[$j++] = "Customer information encoding must be either character or numeric"; //#13045
        bwipp_raiseerror(); //#13045
    } //#13045
    $_.encs = $a(["000", "001", "002", "010", "011", "012", "020", "021", "022", "100", "101", "102", "110", "111", "112", "120", "121", "122", "200", "201", "202", "210", "211", "212", "220", "221", "222", "300", "301", "302", "310", "311", "312", "320", "321", "322", "023", "030", "031", "032", "033", "103", "113", "123", "130", "131", "132", "133", "203", "213", "223", "230", "231", "232", "233", "303", "313", "323", "330", "331", "332", "333", "003", "013", "00", "01", "02", "10", "11", "12", "20", "21", "22", "30", "13", "3"]); //#13063
    $_.barchars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789abcdefghijklmnopqrstuvwxyz #"; //#13066
    var _5 = new Map([
        ["11", 37],
        ["45", 37],
        ["59", 52],
        ["62", 67]
    ]); //#13069
    $_.fcclen = _5; //#13069
    $_.barlen = $_.barcode.length; //#13073
    if ($_.barlen < 10) { //#13076
        $k[$j++] = 'bwipp.auspostTooShort#13075'; //#13075
        $k[$j++] = "The data length must be at least 10"; //#13075
        bwipp_raiseerror(); //#13075
    } //#13075
    $_.txt = $a($_.barlen - 2); //#13078
    $k[$j++] = $_.fcclen; //#13080
    if ($_.barlen >= 2) { //#13080
        $k[$j++] = $geti($_.barcode, 0, 2); //#13080
    } else { //#13080
        $k[$j++] = ""; //#13080
    } //#13080
    var _F = $k[$j - 1]; //#13081
    var _G = $k[$j - 2]; //#13081
    var _H = $get(_G, _F) !== undefined; //#13081
    if (!_H) { //#13083
        $j -= 2; //#13082
        $k[$j++] = 'bwipp.auspostBadFCC#13082'; //#13082
        $k[$j++] = "Must begin with an FCC, either 11, 45, 59 or 62"; //#13082
        bwipp_raiseerror(); //#13082
    } //#13082
    var _I = $k[--$j]; //#13084
    $_.encstr = $s($get($k[--$j], _I)); //#13084
    $k[$j++] = true; //#13091
    for (var _M = 0; _M < 1; _M++) { //#13091
        if ($_.barlen < 10) { //#13089
            $k[$j - 1] = false; //#13089
            break; //#13089
        } //#13089
        var _P = $geti($_.barcode, 2, 8); //#13090
        for (var _Q = 0, _R = _P.length; _Q < _R; _Q++) { //#13090
            var _S = $get(_P, _Q); //#13090
            if ((_S < 48) || (_S > 57)) { //#13090
                $k[$j - 1] = false; //#13090
                break; //#13090
            } //#13090
        } //#13090
    } //#13090
    if (!$k[--$j]) { //#13094
        $k[$j++] = 'bwipp.auspostIncompleteDPID#13093'; //#13093
        $k[$j++] = "The DPID must be 8 digits"; //#13093
        bwipp_raiseerror(); //#13093
    } //#13093
    var _W = $eq($_.custinfoenc, "numeric") ? 2 : 3; //#13097
    if ((($_.barlen - 10) * _W) > (($_.encstr.length - 22) - 14)) { //#13100
        $k[$j++] = 'bwipp.auspostTooLong#13099'; //#13099
        $k[$j++] = "The message is too long"; //#13099
        bwipp_raiseerror(); //#13099
    } //#13099
    $puti($_.encstr, 0, $get($_.encs, 74)); //#13103
    for (var _b = 0; _b <= 1; _b += 1) { //#13110
        $_.i = _b; //#13107
        $puti($_.encstr, ($_.i * 2) + 2, $get($_.encs, $cvi($geti($_.barcode, $_.i, 1)) + 64)); //#13109
    } //#13109
    for (var _j = 2; _j <= 9; _j += 1) { //#13118
        $_.i = _j; //#13114
        $puti($_.encstr, ($_.i * 2) + 2, $get($_.encs, $cvi($geti($_.barcode, $_.i, 1)) + 64)); //#13116
        $put($_.txt, $_.i - 2, $a([$geti($_.barcode, $_.i, 1), ((($_.i - 2) * 2) + 6) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13117
    } //#13117
    if ($eq($_.custinfoenc, "numeric")) { //#13146
        for (var _14 = 0, _13 = $_.barlen - 11; _14 <= _13; _14 += 1) { //#13131
            $_.i = _14; //#13123
            var _18 = $get($_.barcode, $_.i + 10); //#13124
            $k[$j++] = $_.encs; //#13127
            $k[$j++] = _18; //#13127
            if ((_18 < 48) || (_18 > 57)) { //#13127
                $k[$j++] = 'bwipp.auspostInvalidCustinfoDigit#13126'; //#13126
                $k[$j++] = "The customer information data contains a non-digit character"; //#13126
                bwipp_raiseerror(); //#13126
            } //#13126
            var _19 = $k[--$j]; //#13128
            $puti($_.encstr, ($_.i * 2) + 22, $get($k[--$j], (_19 - 48) + 64)); //#13129
            $put($_.txt, $_.i + 8, $a([$geti($_.barcode, $_.i + 10, 1), (($_.i * 2) + 22) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13130
        } //#13130
        $_.ciflen = ($_.barlen - 10) * 2; //#13132
    } else { //#13146
        for (var _1R = 0, _1Q = $_.barlen - 11; _1R <= _1Q; _1R += 1) { //#13145
            $_.i = _1R; //#13135
            $search($_.barchars, $geti($_.barcode, $_.i + 10, 1)); //#13136
            if (!$k[--$j]) { //#13139
                $j--; //#13138
                $k[$j++] = 'bwipp.auspostInvalidCustinfoCharacter#13138'; //#13138
                $k[$j++] = "The customer information contains an invalid character"; //#13138
                bwipp_raiseerror(); //#13138
            } //#13138
            $_.indx = $k[--$j].length; //#13140
            $j -= 2; //#13141
            $_.enc = $get($_.encs, $_.indx); //#13142
            $puti($_.encstr, ($_.i * 3) + 22, $_.enc); //#13143
            $put($_.txt, $_.i + 8, $a([$geti($_.barcode, $_.i + 10, 1), (($_.i * 3) + 22) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13144
        } //#13144
        $_.ciflen = ($_.barlen - 10) * 3; //#13146
    } //#13146
    for (var _1s = 22 + $_.ciflen, _1r = $_.encstr.length - 14; _1s <= _1r; _1s += 1) { //#13152
        $puti($_.encstr, _1s, $get($_.encs, 75)); //#13151
    } //#13151
    $_.rstable = $a(64 * 64); //#13156
    $k[$j++] = $_.rstable; //#13157
    $k[$j++] = 0; //#13157
    $k[$j++] = Infinity; //#13157
    for (var _1y = 0; _1y < 64; _1y++) { //#13157
        $k[$j++] = 0; //#13157
    } //#13157
    var _1z = $a(); //#13157
    var _20 = $k[--$j]; //#13157
    $puti($k[--$j], _20, _1z); //#13157
    $k[$j++] = $_.rstable; //#13158
    $k[$j++] = 64; //#13158
    $k[$j++] = Infinity; //#13158
    for (var _23 = 0; _23 <= 63; _23 += 1) { //#13158
        $k[$j++] = _23; //#13158
    } //#13158
    var _24 = $a(); //#13158
    var _25 = $k[--$j]; //#13158
    $puti($k[--$j], _25, _24); //#13158
    $_.prev = 1; //#13159
    for (var _27 = 0; _27 < 64; _27++) { //#13174
        $_.next = $_.prev << 1; //#13161
        if (($_.next & 64) != 0) { //#13164
            $_.next = $_.next ^ 67; //#13163
        } //#13163
        for (var _2B = 0; _2B <= 63; _2B += 1) { //#13172
            $_.j = _2B; //#13166
            $_.nextcell = function() {
                $k[$j++] = $_.rstable; //#13167
                $k[$j++] = (64 * $_.next) + $_.j; //#13167
            }; //#13167
            $_.nextcell(); //#13168
            var _2J = $k[--$j]; //#13168
            $put($k[--$j], _2J, $get($_.rstable, (64 * $_.prev) + $_.j) << 1); //#13168
            $_.nextcell(); //#13169
            var _2L = $k[--$j]; //#13169
            if (($get($k[--$j], _2L) & 64) != 0) { //#13171
                $_.nextcell(); //#13170
                $_.nextcell(); //#13170
                var _2O = $k[--$j]; //#13170
                var _2Q = $get($k[--$j], _2O); //#13170
                var _2R = $k[--$j]; //#13170
                $put($k[--$j], _2R, _2Q ^ 67); //#13170
            } //#13170
        } //#13170
        $_.prev = $_.next; //#13173
    } //#13173
    $_.rscodes = $a((~~(($_.encstr.length - 16) / 3)) + 4); //#13178
    $k[$j++] = $_.rscodes; //#13179
    $k[$j++] = 0; //#13179
    $k[$j++] = Infinity; //#13179
    for (var _2Y = 0; _2Y < 4; _2Y++) { //#13179
        $k[$j++] = 0; //#13179
    } //#13179
    var _2Z = $a(); //#13179
    var _2a = $k[--$j]; //#13179
    $puti($k[--$j], _2a, _2Z); //#13179
    for (var _2e = 2, _2d = $_.encstr.length - 16; _2e <= _2d; _2e += 3) { //#13187
        $_.i = _2e; //#13181
        $put($_.rscodes, ($_.rscodes.length - (~~(($_.i - 2) / 3))) - 1, (($cvi($geti($_.encstr, $_.i, 1)) * 16) + ($cvi($geti($_.encstr, $_.i + 1, 1)) * 4)) + $cvi($geti($_.encstr, $_.i + 2, 1))); //#13186
    } //#13186
    for (var _2s = $_.rscodes.length - 5; _2s >= 0; _2s -= 1) { //#13196
        $_.i = _2s; //#13189
        for (var _2t = 0; _2t <= 4; _2t += 1) { //#13195
            $_.j = _2t; //#13191
            $put($_.rscodes, $_.i + $_.j, $xo($get($_.rscodes, $_.i + $_.j), $get($_.rstable, $f((64 * $get($a([48, 17, 29, 30, 1]), $_.j)) + $get($_.rscodes, $_.i + 4))))); //#13194
        } //#13194
    } //#13194
    $_.checkcode = $strcpy($s(12), "000000000000"); //#13197
    for (var _3B = 0; _3B <= 3; _3B += 1) { //#13202
        $_.i = _3B; //#13199
        $_.enc = $cvrs($s(3), $get($_.rscodes, 3 - $_.i), 4); //#13200
        $puti($_.checkcode, ($_.i * 3) + (3 - $_.enc.length), $_.enc); //#13201
    } //#13201
    $puti($_.encstr, $_.encstr.length - 14, $_.checkcode); //#13205
    $puti($_.encstr, $_.encstr.length - 2, $get($_.encs, 74)); //#13206
    $_.bbs = $a($_.encstr.length); //#13208
    $_.bhs = $a($_.encstr.length); //#13209
    for (var _3Y = 0, _3X = $_.encstr.length - 1; _3Y <= _3X; _3Y += 1) { //#13229
        $_.i = _3Y; //#13211
        $_.enc = $geti($_.encstr, $_.i, 1); //#13212
        if ($eq($_.enc, "0")) { //#13216
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#13214
            $put($_.bhs, $_.i, (8 * $_.height) / 8); //#13215
        } //#13215
        if ($eq($_.enc, "1")) { //#13220
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#13218
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#13219
        } //#13219
        if ($eq($_.enc, "2")) { //#13224
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#13222
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#13223
        } //#13223
        if ($eq($_.enc, "3")) { //#13228
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#13226
            $put($_.bhs, $_.i, (2 * $_.height) / 8); //#13227
        } //#13227
    } //#13227
    $k[$j++] = Infinity; //#13232
    var _44 = $_.bbs; //#13234
    var _45 = $_.bhs; //#13235
    $k[$j++] = Infinity; //#13236
    for (var _47 = 0, _48 = $_.bhs.length - 1; _47 < _48; _47++) { //#13236
        $k[$j++] = 1.44; //#13236
        $k[$j++] = 1.872; //#13236
    } //#13236
    $k[$j++] = 1.44; //#13236
    var _49 = $a(); //#13236
    $k[$j++] = 'ren'; //#13239
    $k[$j++] = 'renlinear'; //#13239
    $k[$j++] = 'bbs'; //#13239
    $k[$j++] = _44; //#13239
    $k[$j++] = 'bhs'; //#13239
    $k[$j++] = _45; //#13239
    $k[$j++] = 'sbs'; //#13239
    $k[$j++] = _49; //#13239
    if ($_.includetext) { //#13239
        $k[$j++] = 'txt'; //#13238
        $k[$j++] = $_.txt; //#13238
    } //#13238
    $k[$j++] = 'opt'; //#13240
    $k[$j++] = $_.options; //#13240
    var _4D = $d(); //#13240
    $k[$j++] = _4D; //#13243
    if (!$_.dontdraw) { //#13243
        bwipp_renlinear(); //#13243
    } //#13243
    $_ = $__; //#13247
} //bwipp_auspost
function bwipp_kix() {
    var $__ = $_; //#13277
    $_ = Object.assign({}, $_); //#13277
    $_.dontdraw = false; //#13280
    $_.includetext = false; //#13281
    $_.textfont = "OCR-B"; //#13282
    $_.textsize = 10; //#13283
    $_.textyoffset = -8; //#13284
    $_.height = 0.175; //#13285
    $k[$j++] = $_; //#13287
    bwipp_processoptions(); //#13287
    $_.options = $k[--$j]; //#13287
    $_.barcode = $k[--$j]; //#13288
    if ($_.barcode.length == 0) { //#13292
        $k[$j++] = 'bwipp.kixEmptyData#13291'; //#13291
        $k[$j++] = "The data must not be empty"; //#13291
        bwipp_raiseerror(); //#13291
    } //#13291
    $_.encs = $a(["0033", "0123", "0132", "1023", "1032", "1122", "0213", "0303", "0312", "1203", "1212", "1302", "0231", "0321", "0330", "1221", "1230", "1320", "2013", "2103", "2112", "3003", "3012", "3102", "2031", "2121", "2130", "3021", "3030", "3120", "2211", "2301", "2310", "3201", "3210", "3300"]); //#13305
    $_.barchars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"; //#13308
    $_.charvals = new Map; //#13309
    for (var _4 = 0; _4 <= 35; _4 += 1) { //#13310
        $put($_.charvals, $geti($_.barchars, _4, 1), _4); //#13310
    } //#13310
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#13318
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#13315
        if (!_F) { //#13317
            $k[$j++] = 'bwipp.kixBadCharacter#13316'; //#13316
            $k[$j++] = "KIX must contain only capital letters and digits"; //#13316
            bwipp_raiseerror(); //#13316
        } //#13316
    } //#13316
    $_.barlen = $_.barcode.length; //#13320
    $_.encstr = $s($_.barlen * 4); //#13321
    $_.txt = $a($_.barlen); //#13322
    for (var _N = 0, _M = $_.barlen - 1; _N <= _M; _N += 1) { //#13329
        $_.i = _N; //#13325
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#13326
        $puti($_.encstr, $_.i * 4, $get($_.encs, $_.indx)); //#13327
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 4) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13328
    } //#13328
    $_.bbs = $a($_.encstr.length); //#13331
    $_.bhs = $a($_.encstr.length); //#13332
    for (var _o = 0, _n = $_.encstr.length - 1; _o <= _n; _o += 1) { //#13352
        $_.i = _o; //#13334
        $_.enc = $geti($_.encstr, $_.i, 1); //#13335
        if ($eq($_.enc, "0")) { //#13339
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#13337
            $put($_.bhs, $_.i, (2 * $_.height) / 8); //#13338
        } //#13338
        if ($eq($_.enc, "1")) { //#13343
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#13341
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#13342
        } //#13342
        if ($eq($_.enc, "2")) { //#13347
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#13345
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#13346
        } //#13346
        if ($eq($_.enc, "3")) { //#13351
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#13349
            $put($_.bhs, $_.i, (8 * $_.height) / 8); //#13350
        } //#13350
    } //#13350
    $k[$j++] = Infinity; //#13355
    var _1K = $_.bbs; //#13357
    var _1L = $_.bhs; //#13358
    $k[$j++] = Infinity; //#13359
    for (var _1N = 0, _1O = $_.bhs.length - 1; _1N < _1O; _1N++) { //#13359
        $k[$j++] = 1.44; //#13359
        $k[$j++] = 1.872; //#13359
    } //#13359
    $k[$j++] = 1.44; //#13359
    var _1P = $a(); //#13359
    $k[$j++] = 'ren'; //#13362
    $k[$j++] = 'renlinear'; //#13362
    $k[$j++] = 'bbs'; //#13362
    $k[$j++] = _1K; //#13362
    $k[$j++] = 'bhs'; //#13362
    $k[$j++] = _1L; //#13362
    $k[$j++] = 'sbs'; //#13362
    $k[$j++] = _1P; //#13362
    if ($_.includetext) { //#13362
        $k[$j++] = 'txt'; //#13361
        $k[$j++] = $_.txt; //#13361
    } //#13361
    $k[$j++] = 'opt'; //#13363
    $k[$j++] = $_.options; //#13363
    var _1T = $d(); //#13363
    $k[$j++] = _1T; //#13366
    if (!$_.dontdraw) { //#13366
        bwipp_renlinear(); //#13366
    } //#13366
    $_ = $__; //#13370
} //bwipp_kix
function bwipp_japanpost() {
    var $__ = $_; //#13400
    $_ = Object.assign({}, $_); //#13400
    $_.dontdraw = false; //#13403
    $_.includetext = false; //#13404
    $_.includecheckintext = false; //#13405
    $_.textfont = "OCR-B"; //#13406
    $_.textsize = 10; //#13407
    $_.textyoffset = -8; //#13408
    $_.height = 0.175; //#13409
    $k[$j++] = $_; //#13411
    bwipp_processoptions(); //#13411
    $_.options = $k[--$j]; //#13411
    $_.barcode = $k[--$j]; //#13412
    if ($_.barcode.length == 0) { //#13416
        $k[$j++] = 'bwipp.japanpostEmptyData#13415'; //#13415
        $k[$j++] = "The data must not be empty"; //#13415
        bwipp_raiseerror(); //#13415
    } //#13415
    $forall($_.barcode, function() { //#13427
        var _4 = $k[--$j]; //#13422
        if (!(((_4 >= 48) && (_4 <= 57)) || (((_4 >= 65) && (_4 <= 90)) || (_4 == 45)))) { //#13426
            $k[$j++] = 'bwipp.japanpostBadCharacter#13425'; //#13425
            $k[$j++] = "Japan Post must contain only digits, capital letters and the dash symbol"; //#13425
            bwipp_raiseerror(); //#13425
        } //#13425
    }); //#13425
    $_.encs = $a(["300", "330", "312", "132", "321", "303", "123", "231", "213", "033", "030", "120", "102", "210", "012", "201", "021", "003", "333", "31", "13"]); //#13435
    $_.barchars = "0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ"; //#13438
    $_.barlen = $_.barcode.length; //#13441
    $_.encstr = $s(((20 + 1) * 3) + 4); //#13443
    $_.digits = $a(20); //#13444
    $_.txt = $a($_.barlen + 1); //#13445
    $puti($_.encstr, 0, $get($_.encs, 19)); //#13448
    $_.checksum = 0; //#13450
    $_.j = 0; //#13451
    $_.i = 0; //#13451
    $k[$j++] = true; //#13486
    for (var _H = 0, _G = $_.barlen - 1; _H <= _G; _H += 1) { //#13486
        $_.i = _H; //#13454
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#13455
        $j--; //#13456
        $_.indx = $k[--$j].length; //#13457
        $j -= 2; //#13458
        if (($_.indx >= 11) && ($_.indx < 37)) { //#13469
            if ($_.j > 18) { //#13460
                $k[$j - 1] = false; //#13460
                break; //#13460
            } //#13460
            $_.digit = (~~(($_.indx - 1) / 10)) + 10; //#13464
            $puti($_.encstr, ($_.j * 3) + 2, $get($_.encs, $_.digit)); //#13465
            $_.checksum = $_.checksum + $_.digit; //#13466
            $put($_.digits, $_.j, $_.digit); //#13467
            $_.j = $_.j + 1; //#13468
        } //#13468
        if ($_.j > 19) { //#13470
            $k[$j - 1] = false; //#13470
            break; //#13470
        } //#13470
        $_.digit = 0; //#13471
        if ($_.indx == 37) { //#13474
            $_.digit = 14; //#13473
        } //#13473
        if (($_.indx >= 11) && ($_.indx < 37)) { //#13477
            $_.digit = ($_.indx - 1) % 10; //#13476
        } //#13476
        if ($_.indx < 11) { //#13480
            $_.digit = $_.indx; //#13479
        } //#13479
        $puti($_.encstr, ($_.j * 3) + 2, $get($_.encs, $_.digit)); //#13481
        $_.checksum = $_.checksum + $_.digit; //#13482
        $put($_.digits, $_.j, $_.digit); //#13483
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), (($_.j * 3) + 2) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13484
        $_.j = $_.j + 1; //#13485
    } //#13485
    if (!$k[--$j]) { //#13490
        $k[$j++] = 'bwipp.japanpostTooLong#13489'; //#13489
        $k[$j++] = "The input is too long"; //#13489
        bwipp_raiseerror(); //#13489
    } //#13489
    for (var _17 = $_.j, _16 = 20 - 1; _17 <= _16; _17 += 1) { //#13498
        $_.k = _17; //#13494
        $puti($_.encstr, ($_.k * 3) + 2, $get($_.encs, 14)); //#13495
        $_.checksum = $_.checksum + 14; //#13496
        $put($_.digits, $_.k, 14); //#13497
    } //#13497
    $_.checksum = 19 - ($_.checksum % 19); //#13501
    $puti($_.encstr, 2 + (20 * 3), $get($_.encs, $_.checksum)); //#13502
    $puti($_.encstr, 2 + (21 * 3), $get($_.encs, 20)); //#13505
    $_.checkdigit = " "; //#13506
    if ($_.includecheckintext) { //#13509
        $_.checkdigit = $geti($_.barchars, $_.checksum, 1); //#13508
    } //#13508
    $put($_.txt, $_.i + 1, $a([$_.checkdigit, ((20 * 3) + 2) * 3.312, $_.textyoffset, $_.textfont, $_.textsize])); //#13510
    $puti($_.encstr, ((20 + 1) * 3) + 2, $get($_.encs, 20)); //#13513
    $_.bbs = $a((21 * 3) + 4); //#13515
    $_.bhs = $a($_.bbs.length); //#13516
    for (var _1g = 0, _1f = $_.bbs.length - 1; _1g <= _1f; _1g += 1) { //#13540
        $_.i = _1g; //#13518
        $_.enc = $geti($_.encstr, $_.i, 1); //#13519
        $_.bunit = 0; //#13520
        $_.hunit = 0; //#13521
        if ($eq($_.enc, "0")) { //#13525
            $_.bunit = 3; //#13523
            $_.hunit = 2; //#13524
        } //#13524
        if ($eq($_.enc, "1")) { //#13529
            $_.bunit = 0; //#13527
            $_.hunit = 5; //#13528
        } //#13528
        if ($eq($_.enc, "2")) { //#13533
            $_.bunit = 3; //#13531
            $_.hunit = 5; //#13532
        } //#13532
        if ($eq($_.enc, "3")) { //#13537
            $_.bunit = 0; //#13535
            $_.hunit = 8; //#13536
        } //#13536
        $put($_.bbs, $_.i, ($_.bunit * $_.height) / 8); //#13538
        $put($_.bhs, $_.i, ($_.hunit * $_.height) / 8); //#13539
    } //#13539
    $k[$j++] = Infinity; //#13543
    var _1w = $_.bbs; //#13545
    var _1x = $_.bhs; //#13546
    $k[$j++] = Infinity; //#13547
    for (var _1z = 0, _20 = $_.bhs.length - 1; _1z < _20; _1z++) { //#13547
        $k[$j++] = 1.44; //#13547
        $k[$j++] = 1.872; //#13547
    } //#13547
    $k[$j++] = 1.44; //#13547
    var _21 = $a(); //#13547
    $k[$j++] = 'ren'; //#13550
    $k[$j++] = 'renlinear'; //#13550
    $k[$j++] = 'bbs'; //#13550
    $k[$j++] = _1w; //#13550
    $k[$j++] = 'bhs'; //#13550
    $k[$j++] = _1x; //#13550
    $k[$j++] = 'sbs'; //#13550
    $k[$j++] = _21; //#13550
    if ($_.includetext) { //#13550
        $k[$j++] = 'txt'; //#13549
        $k[$j++] = $_.txt; //#13549
    } //#13549
    $k[$j++] = 'opt'; //#13551
    $k[$j++] = $_.options; //#13551
    var _25 = $d(); //#13551
    $k[$j++] = _25; //#13554
    if (!$_.dontdraw) { //#13554
        bwipp_renlinear(); //#13554
    } //#13554
    $_ = $__; //#13558
} //bwipp_japanpost
function bwipp_msi() {
    var $__ = $_; //#13588
    $_ = Object.assign({}, $_); //#13588
    $_.dontdraw = false; //#13591
    $_.includecheck = false; //#13592
    $_.includetext = false; //#13593
    $_.includecheckintext = false; //#13594
    $_.checktype = "unset"; //#13595
    $_.badmod11 = false; //#13596
    $_.textfont = "OCR-B"; //#13597
    $_.textsize = 10; //#13598
    $_.textyoffset = -8; //#13599
    $_.height = 1; //#13600
    $k[$j++] = $_; //#13602
    bwipp_processoptions(); //#13602
    $_.options = $k[--$j]; //#13602
    $_.barcode = $k[--$j]; //#13603
    if ($ne($_.checktype, "unset") && (!$_.includecheck)) { //#13607
        $k[$j++] = 'bwipp.msiCheckTypeWithoutCheck#13606'; //#13606
        $k[$j++] = "checktype requires includecheck"; //#13606
        bwipp_raiseerror(); //#13606
    } //#13606
    if ($_.badmod11 && ((($ne($_.checktype, "mod11") && $ne($_.checktype, "ncrmod11")) && $ne($_.checktype, "mod1110")) && $ne($_.checktype, "ncrmod1110"))) { //#13611
        $k[$j++] = 'bwipp.msiBadMod11Mismatch#13610'; //#13610
        $k[$j++] = "badmod11 requires checktype with mod11"; //#13610
        bwipp_raiseerror(); //#13610
    } //#13610
    if ($eq($_.checktype, "unset")) { //#13613
        $_.checktype = "mod10"; //#13613
    } //#13613
    $forall($_.barcode, function() { //#13622
        var _B = $k[--$j]; //#13619
        if ((_B < 48) || (_B > 57)) { //#13621
            $k[$j++] = 'bwipp.msiBadCharacter#13620'; //#13620
            $k[$j++] = "MSI must contain only digits"; //#13620
            bwipp_raiseerror(); //#13620
        } //#13620
    }); //#13620
    $_.encs = $a(["12121212", "12121221", "12122112", "12122121", "12211212", "12211221", "12212112", "12212121", "21121212", "21121221", "21", "121"]); //#13630
    $_.barchars = "0123456789"; //#13633
    $_.barlen = $_.barcode.length; //#13636
    $_.txtlen = $_.barlen; //#13637
    $_.mod10 = function() {
        $_.code = $k[--$j]; //#13640
        $k[$j++] = 0; //#13641
        $k[$j++] = 0; //#13641
        $forall($_.code, function() { //#13641
            var _I = $k[--$j]; //#13641
            var _J = $k[--$j]; //#13641
            var _K = $k[--$j]; //#13641
            $k[$j++] = $f((_I - 48) + (_J * 10)); //#13641
            $k[$j++] = _K; //#13641
        }); //#13641
        $j--; //#13641
        var _L = $k[--$j]; //#13641
        $k[$j++] = _L * 2; //#13642
        $k[$j++] = 0; //#13642
        for (;;) { //#13642
            var _M = $k[--$j]; //#13642
            var _N = $k[--$j]; //#13642
            var _O = ~~(_N / 10); //#13642
            $k[$j++] = $f(_M + (_N % 10)); //#13642
            $k[$j++] = _O; //#13642
            if (_O == 0) { //#13642
                $j--; //#13642
                break; //#13642
            } //#13642
            var _P = $k[--$j]; //#13642
            var _Q = $k[--$j]; //#13642
            $k[$j++] = _P; //#13642
            $k[$j++] = _Q; //#13642
        } //#13642
        $k[$j++] = 0; //#13643
        $k[$j++] = 0; //#13643
        $forall($_.code, function() { //#13643
            var _S = $k[--$j]; //#13643
            var _T = $k[--$j]; //#13643
            var _U = $k[--$j]; //#13643
            $k[$j++] = $f(_T + (_S - 48)); //#13643
            $k[$j++] = _U; //#13643
        }); //#13643
        var _V = $k[--$j]; //#13643
        $k[$j - 1] = _V; //#13643
        var _X = $k[--$j]; //#13643
        var _a = $s($_.code.length + 1); //#13645
        $puti(_a, 0, $_.code); //#13645
        $put(_a, $_.code.length, ((10 - ($f($k[--$j] + _X) % 10)) % 10) + 48); //#13645
        $k[$j++] = _a; //#13645
    }; //#13645
    $_.mod11 = function() {
        $_.code = $k[--$j]; //#13649
        $k[$j++] = $_.code.length - 1; //#13650
        $forall($_.code, function() { //#13650
            var _g = $k[--$j]; //#13650
            var _h = $k[--$j]; //#13650
            $k[$j++] = _g - 48; //#13650
            $k[$j++] = (_h % 6) + 2; //#13650
            $k[$j++] = _h - 1; //#13650
        }); //#13650
        $k[$j - 1] = 0; //#13651
        for (var _j = 0, _k = $_.code.length; _j < _k; _j++) { //#13651
            var _l = $k[$j - 2]; //#13651
            var _m = $k[$j - 3]; //#13651
            var _n = $k[$j - 1]; //#13651
            $j -= 3; //#13651
            $k[$j++] = $f(_n + (_m * _l)); //#13651
        } //#13651
        var _p = (11 - ($k[--$j] % 11)) % 11; //#13653
        $k[$j++] = _p; //#13661
        if (_p == 10) { //#13660
            if ($_.badmod11) { //#13657
                $j--; //#13655
                var _s = $s($_.code.length + 2); //#13655
                $puti(_s, 0, $_.code); //#13655
                $puti(_s, $_.code.length, "10"); //#13655
                $k[$j++] = _s; //#13655
            } else { //#13657
                $j--; //#13657
                $k[$j++] = 'bwipp.msiBadMod11NotSpecified#13657'; //#13657
                $k[$j++] = "mod11 check digit is 10 but badmod11 not specified"; //#13657
                bwipp_raiseerror(); //#13657
            } //#13657
        } else { //#13660
            var _w = $s($_.code.length + 1); //#13660
            $puti(_w, 0, $_.code); //#13660
            $k[$j++] = _w; //#13660
            $k[$j++] = _w; //#13660
            $k[$j++] = $_.code.length; //#13660
            $r(4, -1); //#13660
            var _z = $k[--$j]; //#13660
            var _10 = $k[--$j]; //#13660
            $put($k[--$j], _10, _z + 48); //#13660
        } //#13660
    }; //#13660
    $_.ncrmod11 = function() {
        $_.code = $k[--$j]; //#13665
        $k[$j++] = $_.code.length - 1; //#13666
        $forall($_.code, function() { //#13666
            var _15 = $k[--$j]; //#13666
            var _16 = $k[--$j]; //#13666
            $k[$j++] = _15 - 48; //#13666
            $k[$j++] = (_16 % 8) + 2; //#13666
            $k[$j++] = _16 - 1; //#13666
        }); //#13666
        $k[$j - 1] = 0; //#13667
        for (var _18 = 0, _19 = $_.code.length; _18 < _19; _18++) { //#13667
            var _1A = $k[$j - 2]; //#13667
            var _1B = $k[$j - 3]; //#13667
            var _1C = $k[$j - 1]; //#13667
            $j -= 3; //#13667
            $k[$j++] = $f(_1C + (_1B * _1A)); //#13667
        } //#13667
        var _1E = (11 - ($k[--$j] % 11)) % 11; //#13669
        $k[$j++] = _1E; //#13677
        if (_1E == 10) { //#13676
            if ($_.badmod11) { //#13673
                $j--; //#13671
                var _1H = $s($_.code.length + 2); //#13671
                $puti(_1H, 0, $_.code); //#13671
                $puti(_1H, $_.code.length, "10"); //#13671
                $k[$j++] = _1H; //#13671
            } else { //#13673
                $j--; //#13673
                $k[$j++] = 'bwipp.msiBadMod11NotSpecified#13673'; //#13673
                $k[$j++] = "mod11 check digit is 10 but badmod11 not specified"; //#13673
                bwipp_raiseerror(); //#13673
            } //#13673
        } else { //#13676
            var _1L = $s($_.code.length + 1); //#13676
            $puti(_1L, 0, $_.code); //#13676
            $k[$j++] = _1L; //#13676
            $k[$j++] = _1L; //#13676
            $k[$j++] = $_.code.length; //#13676
            $r(4, -1); //#13676
            var _1O = $k[--$j]; //#13676
            var _1P = $k[--$j]; //#13676
            $put($k[--$j], _1P, _1O + 48); //#13676
        } //#13676
    }; //#13676
    var _1R = new Map([
        ["mod10", function() {
            $_.mod10(); //#13682
        }],
        ["mod1010", function() {
            $_.mod10(); //#13683
            $_.mod10(); //#13683
        }],
        ["mod11", function() {
            $_.mod11(); //#13684
        }],
        ["ncrmod11", function() {
            $_.ncrmod11(); //#13685
        }],
        ["mod1110", function() {
            $_.mod11(); //#13686
            $_.mod10(); //#13686
        }],
        ["ncrmod1110", function() {
            $_.ncrmod11(); //#13687
            $_.mod10(); //#13687
        }]
    ]); //#13687
    $_.checkfunc = _1R; //#13688
    if ($_.includecheck) { //#13699
        var _1U = $_.checkfunc; //#13693
        var _1V = $_.checktype; //#13693
        var _1W = $get(_1U, _1V) !== undefined; //#13693
        $k[$j++] = _1U; //#13695
        $k[$j++] = _1V; //#13695
        if (!_1W) { //#13695
            $j -= 2; //#13694
            $k[$j++] = 'bwipp.msiBadCharacter#13694'; //#13694
            $k[$j++] = "MSI checktype must be mod10, mod1010, mod11, ncrmod11, mod1110 or ncrmod1110"; //#13694
            bwipp_raiseerror(); //#13694
        } //#13694
        var _1X = $k[--$j]; //#13696
        var _1Z = $get($k[--$j], _1X); //#13696
        $k[$j++] = $_.barcode; //#13696
        if (_1Z() === true) {
            return true;
        } //#13696
        $_.barcode = $k[--$j]; //#13696
        $_.barlen = $_.barcode.length; //#13697
        if ($_.includecheckintext) { //#13698
            $_.txtlen = $_.barlen; //#13698
        } //#13698
    } //#13698
    $_.sbs = $s(($_.barlen * 8) + 5); //#13701
    $_.txt = $a($_.barlen); //#13702
    $puti($_.sbs, 0, $get($_.encs, 10)); //#13705
    for (var _1o = 0, _1n = $_.barlen - 1; _1o <= _1n; _1o += 1) { //#13717
        $_.i = _1o; //#13708
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#13710
        $j--; //#13711
        $_.indx = $k[--$j].length; //#13712
        $j -= 2; //#13713
        $_.enc = $get($_.encs, $_.indx); //#13714
        $puti($_.sbs, ($_.i * 8) + 2, $_.enc); //#13715
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 12) + 3, $_.textyoffset, $_.textfont, $_.textsize])); //#13716
    } //#13716
    $puti($_.sbs, ($_.barlen * 8) + 2, $get($_.encs, 11)); //#13720
    $k[$j++] = Infinity; //#13723
    $k[$j++] = Infinity; //#13725
    var _2E = $_.sbs; //#13725
    for (var _2F = 0, _2G = _2E.length; _2F < _2G; _2F++) { //#13725
        $k[$j++] = $get(_2E, _2F) - 48; //#13725
    } //#13725
    var _2I = $a(); //#13725
    $k[$j++] = Infinity; //#13726
    for (var _2K = 0, _2L = ~~(($_.sbs.length + 1) / 2); _2K < _2L; _2K++) { //#13726
        $k[$j++] = $_.height; //#13726
    } //#13726
    var _2N = $a(); //#13726
    $k[$j++] = Infinity; //#13727
    for (var _2P = 0, _2Q = ~~(($_.sbs.length + 1) / 2); _2P < _2Q; _2P++) { //#13727
        $k[$j++] = 0; //#13727
    } //#13727
    var _2R = $a(); //#13727
    $k[$j++] = 'ren'; //#13730
    $k[$j++] = 'renlinear'; //#13730
    $k[$j++] = 'sbs'; //#13730
    $k[$j++] = _2I; //#13730
    $k[$j++] = 'bhs'; //#13730
    $k[$j++] = _2N; //#13730
    $k[$j++] = 'bbs'; //#13730
    $k[$j++] = _2R; //#13730
    if ($_.includetext) { //#13730
        $k[$j++] = 'txt'; //#13729
        $k[$j++] = $geti($_.txt, 0, $_.txtlen); //#13729
    } //#13729
    $k[$j++] = 'opt'; //#13731
    $k[$j++] = $_.options; //#13731
    var _2X = $d(); //#13731
    $k[$j++] = _2X; //#13734
    if (!$_.dontdraw) { //#13734
        bwipp_renlinear(); //#13734
    } //#13734
    $_ = $__; //#13738
} //bwipp_msi
function bwipp_plessey() {
    var $__ = $_; //#13768
    $_ = Object.assign({}, $_); //#13768
    $_.dontdraw = false; //#13771
    $_.includetext = false; //#13772
    $_.validatecheck = false; //#13773
    $_.includecheckintext = false; //#13774
    $_.unidirectional = false; //#13775
    $_.textfont = "OCR-B"; //#13776
    $_.textsize = 10; //#13777
    $_.textyoffset = -8; //#13778
    $_.height = 1; //#13779
    $k[$j++] = $_; //#13781
    bwipp_processoptions(); //#13781
    $_.options = $k[--$j]; //#13781
    $_.barcode = $k[--$j]; //#13782
    if ($_.barcode.length == 0) { //#13786
        $k[$j++] = 'bwipp.plesseyEmptyData#13785'; //#13785
        $k[$j++] = "The data must not be empty"; //#13785
        bwipp_raiseerror(); //#13785
    } //#13785
    $_.barchars = "0123456789ABCDEF"; //#13792
    $_.charvals = new Map; //#13793
    for (var _3 = 0; _3 <= 15; _3 += 1) { //#13794
        $put($_.charvals, $geti($_.barchars, _3, 1), _3); //#13794
    } //#13794
    $_.encs = $a(["14141414", "32141414", "14321414", "32321414", "14143214", "32143214", "14323214", "32323214", "14141432", "32141432", "14321432", "32321432", "14143232", "32143232", "14323232", "32323232", "32321432", "541412323", "323"]); //#13803
    for (var _B = 0, _A = $_.barcode.length - 1; _B <= _A; _B += 1) { //#13811
        var _F = $get($_.charvals, $geti($_.barcode, _B, 1)) !== undefined; //#13808
        if (!_F) { //#13810
            $k[$j++] = 'bwipp.plesseyBadCharacter#13809'; //#13809
            $k[$j++] = "Plessey must contain only digits and letters A B C D E F"; //#13809
            bwipp_raiseerror(); //#13809
        } //#13809
    } //#13809
    $k[$j++] = 'barlen'; //#13813
    $k[$j++] = $_.barcode.length; //#13813
    if ($_.validatecheck) { //#13813
        var _I = $k[--$j]; //#13813
        $k[$j++] = _I - 2; //#13813
    } //#13813
    var _J = $k[--$j]; //#13813
    $_[$k[--$j]] = _J; //#13813
    $_.checkbits = $a(($_.barlen * 4) + 8); //#13816
    $puti($_.checkbits, $_.barlen * 4, $a([0, 0, 0, 0, 0, 0, 0, 0])); //#13817
    for (var _S = 0, _R = $_.barlen - 1; _S <= _R; _S += 1) { //#13827
        $_.i = _S; //#13819
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#13820
        $puti($_.checkbits, $_.i * 4, $a([$_.indx & 1, ($_.indx >>> 1) & 1, ($_.indx >>> 2) & 1, $_.indx >>> 3])); //#13826
    } //#13826
    $_.checksalt = $a([1, 1, 1, 1, 0, 1, 0, 0, 1]); //#13828
    for (var _i = 0, _h = ($_.barlen * 4) - 1; _i <= _h; _i += 1) { //#13837
        $_.i = _i; //#13830
        if ($get($_.checkbits, $_.i) == 1) { //#13836
            for (var _m = 0; _m <= 8; _m += 1) { //#13835
                $_.j = _m; //#13833
                $put($_.checkbits, $_.i + $_.j, $xo($get($_.checkbits, $_.i + $_.j), $get($_.checksalt, $_.j))); //#13834
            } //#13834
        } //#13834
    } //#13834
    $_.checkval = 0; //#13838
    for (var _x = 0; _x <= 7; _x += 1) { //#13842
        $_.i = _x; //#13840
        $_.checkval = $f($_.checkval + ~~$pow(2, $_.i) * $get($_.checkbits, $f($_.barlen * 4 + $_.i))); //#13841
    } //#13841
    $_.checksum1 = $_.checkval & 15; //#13843
    $_.checksum2 = $_.checkval >>> 4; //#13844
    if ($_.validatecheck) { //#13851
        if (($get($_.barcode, $_.barlen) != $get($_.barchars, $_.checksum1)) || ($get($_.barcode, $_.barlen + 1) != $get($_.barchars, $_.checksum2))) { //#13849
            $k[$j++] = 'bwipp.plesseyBadCheckDigits#13848'; //#13848
            $k[$j++] = "Incorrect Plessey check digits provided"; //#13848
            bwipp_raiseerror(); //#13848
        } //#13848
        $_.barcode = $geti($_.barcode, 0, $_.barlen); //#13850
    } //#13850
    var _1O = $_.unidirectional ? 27 : 33; //#13853
    $_.sbs = $s($f(($_.barlen * 8) + _1O)); //#13853
    $_.txt = $a($_.barlen + 2); //#13854
    $puti($_.sbs, 0, $get($_.encs, 16)); //#13857
    for (var _1X = 0, _1W = $_.barlen - 1; _1X <= _1W; _1X += 1) { //#13864
        $_.i = _1X; //#13860
        $_.indx = $get($_.charvals, $geti($_.barcode, $_.i, 1)); //#13861
        $puti($_.sbs, ($_.i * 8) + 8, $get($_.encs, $_.indx)); //#13862
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), ($_.i * 20) + 20, $_.textyoffset, $_.textfont, $_.textsize])); //#13863
    } //#13863
    $puti($_.sbs, ($_.barlen * 8) + 8, $get($_.encs, $_.checksum1)); //#13867
    $puti($_.sbs, ($_.barlen * 8) + 16, $get($_.encs, $_.checksum2)); //#13868
    if ($_.includecheckintext) { //#13874
        $put($_.txt, $_.barlen, $a([$geti($_.barchars, $_.checksum1, 1), ($_.barlen * 20) + 20, $_.textyoffset, $_.textfont, $_.textsize])); //#13870
        $put($_.txt, $_.barlen + 1, $a([$geti($_.barchars, $_.checksum2, 1), (($_.barlen + 1) * 20) + 20, $_.textyoffset, $_.textfont, $_.textsize])); //#13871
    } else { //#13874
        $put($_.txt, $_.barlen, $a([" ", ($_.barlen * 20) + 20, $_.textyoffset, $_.textfont, $_.textsize])); //#13873
        $put($_.txt, $_.barlen + 1, $a([" ", (($_.barlen + 1) * 20) + 20, $_.textyoffset, $_.textfont, $_.textsize])); //#13874
    } //#13874
    var _2f = $_.unidirectional ? 18 : 17; //#13878
    $puti($_.sbs, ($_.barlen * 8) + 24, $get($_.encs, _2f)); //#13878
    $k[$j++] = Infinity; //#13881
    $k[$j++] = Infinity; //#13883
    var _2h = $_.sbs; //#13883
    for (var _2i = 0, _2j = _2h.length; _2i < _2j; _2i++) { //#13883
        $k[$j++] = $get(_2h, _2i) - 48; //#13883
    } //#13883
    var _2l = $a(); //#13883
    $k[$j++] = Infinity; //#13884
    for (var _2n = 0, _2o = ~~(($_.sbs.length + 1) / 2); _2n < _2o; _2n++) { //#13884
        $k[$j++] = $_.height; //#13884
    } //#13884
    var _2q = $a(); //#13884
    $k[$j++] = Infinity; //#13885
    for (var _2s = 0, _2t = ~~(($_.sbs.length + 1) / 2); _2s < _2t; _2s++) { //#13885
        $k[$j++] = 0; //#13885
    } //#13885
    var _2u = $a(); //#13885
    $k[$j++] = 'ren'; //#13888
    $k[$j++] = 'renlinear'; //#13888
    $k[$j++] = 'sbs'; //#13888
    $k[$j++] = _2l; //#13888
    $k[$j++] = 'bhs'; //#13888
    $k[$j++] = _2q; //#13888
    $k[$j++] = 'bbs'; //#13888
    $k[$j++] = _2u; //#13888
    if ($_.includetext) { //#13888
        $k[$j++] = 'txt'; //#13887
        $k[$j++] = $_.txt; //#13887
    } //#13887
    $k[$j++] = 'opt'; //#13889
    $k[$j++] = $_.options; //#13889
    var _2y = $d(); //#13889
    $k[$j++] = _2y; //#13892
    if (!$_.dontdraw) { //#13892
        bwipp_renlinear(); //#13892
    } //#13892
    $_ = $__; //#13896
} //bwipp_plessey
function bwipp_telepen() {
    var $__ = $_; //#13927
    $_ = Object.assign({}, $_); //#13927
    $_.numeric = false; //#13930
    $_.dontdraw = false; //#13931
    $_.includetext = false; //#13932
    $_.textfont = "OCR-B"; //#13933
    $_.textsize = 10; //#13934
    $_.textyoffset = -8; //#13935
    $_.height = 1; //#13936
    $_.parse = false; //#13937
    $k[$j++] = $_; //#13939
    bwipp_processoptions(); //#13939
    $_.options = $k[--$j]; //#13939
    $_.barcode = $k[--$j]; //#13940
    var _3 = new Map([
        ["parse", $_.parse],
        ["parseonly", true],
        ["parsefnc", false]
    ]); //#13948
    $_.fncvals = _3; //#13949
    $k[$j++] = 'barcode'; //#13950
    $k[$j++] = $_.barcode; //#13950
    $k[$j++] = $_.fncvals; //#13950
    bwipp_parseinput(); //#13950
    var _6 = $k[--$j]; //#13950
    $_[$k[--$j]] = _6; //#13950
    $_.barlen = $_.barcode.length; //#13951
    delete $_.options["parse"]; //#13952
    if ($_.numeric && (($_.barlen % 2) != 0)) { //#13956
        $k[$j++] = 'bwipp.telepenNumericOddLength#13955'; //#13955
        $k[$j++] = "Telepen Numeric must have an even length"; //#13955
        bwipp_raiseerror(); //#13955
    } //#13955
    $_.encs = $a(["31313131", "1131313111", "33313111", "1111313131", "3111313111", "11333131", "13133131", "111111313111", "31333111", "1131113131", "33113131", "1111333111", "3111113131", "1113133111", "1311133111", "111111113131", "3131113111", "11313331", "333331", "111131113111", "31113331", "1133113111", "1313113111", "1111113331", "31131331", "113111113111", "3311113111", "1111131331", "311111113111", "1113111331", "1311111331", "11111111113111", "31313311", "1131311131", "33311131", "1111313311", "3111311131", "11333311", "13133311", "111111311131", "31331131", "1131113311", "33113311", "1111331131", "3111113311", "1113131131", "1311131131", "111111113311", "3131111131", "1131131311", "33131311", "111131111131", "3111131311", "1133111131", "1313111131", "111111131311", "3113111311", "113111111131", "3311111131", "111113111311", "311111111131", "111311111311", "131111111311", "11111111111131", "3131311111", "11313133", "333133", "111131311111", "31113133", "1133311111", "1313311111", "1111113133", "313333", "113111311111", "3311311111", "11113333", "311111311111", "11131333", "13111333", "11111111311111", "31311133", "1131331111", "33331111", "1111311133", "3111331111", "11331133", "13131133", "111111331111", "3113131111", "1131111133", "33111133", "111113131111", "3111111133", "111311131111", "131111131111", "111111111133", "31311313", "113131111111", "3331111111", "1111311313", "311131111111", "11331313", "13131313", "11111131111111", "3133111111", "1131111313", "33111313", "111133111111", "3111111313", "111313111111", "131113111111", "111111111313", "313111111111", "1131131113", "33131113", "11113111111111", "3111131113", "113311111111", "131311111111", "111111131113", "3113111113", "11311111111111", "331111111111", "111113111113", "31111111111111", "111311111113", "131111111113", "1111111111111111"]); //#13993
    $_.sbs = $s(($_.barlen * 16) + 48); //#13996
    $_.txt = $a($_.barlen); //#13997
    $_.enc = $get($_.encs, 95); //#14000
    $puti($_.sbs, 0, $_.enc); //#14001
    $_.l = $_.enc.length; //#14002
    $_.checksum = 0; //#14004
    $_.i = 0; //#14005
    $_.j = 0; //#14005
    for (;;) { //#14048
        if ($_.i == $_.barlen) { //#14007
            break; //#14007
        } //#14007
        if ($_.numeric) { //#14041
            if ($get($_.barcode, $_.i) > 16) { //#14029
                $_.np = $geti($_.barcode, $_.i, 2); //#14010
                var _X = $get($_.np, 0); //#14011
                var _Z = $get($_.np, 1); //#14012
                if (((_X < 48) || (_X > 57)) || (((_Z < 48) || (_Z > 57)) && (!(_Z == 88)))) { //#14018
                    $k[$j++] = -1; //#14014
                } else { //#14018
                    var _e = $get($_.np, 1) - 48; //#14017
                    $k[$j++] = $get($_.np, 0) - 48; //#14017
                    $k[$j++] = _e; //#14017
                    if (_e != 40) { //#14017
                        var _f = $k[--$j]; //#14017
                        var _g = $k[--$j]; //#14017
                        $k[$j++] = ($f(_f + (_g * 10))) + 10; //#14017
                    } else { //#14017
                        $j--; //#14017
                    } //#14017
                    var _h = $k[--$j]; //#14018
                    $k[$j++] = _h + 17; //#14018
                } //#14018
                var _i = $k[$j - 1]; //#14020
                if (_i == -1) { //#14022
                    $k[$j++] = 'bwipp.telepenInvalidNumericCharacter#14021'; //#14021
                    $k[$j++] = "Telepen Numeric may contain only digits, or X in even positions"; //#14021
                    bwipp_raiseerror(); //#14021
                } //#14021
                $_.indx = $k[--$j]; //#14023
                $put($_.txt, $_.j, $a([$_.np, ($_.j * 16) + 16, $_.textyoffset, $_.textfont, $_.textsize])); //#14024
                $_.i = $_.i + 2; //#14025
            } else { //#14029
                $_.indx = $get($_.barcode, $_.i); //#14027
                $put($_.txt, $_.j, $a([" ", ($_.j * 16) + 16, $_.textyoffset, $_.textfont, $_.textsize])); //#14028
                $_.i = $_.i + 1; //#14029
            } //#14029
        } else { //#14041
            $_.indx = $get($_.barcode, $_.i); //#14032
            if ($_.indx > 127) { //#14035
                $k[$j++] = 'bwipp.telepenInvalidAlphaCharacter#14034'; //#14034
                $k[$j++] = "Telepen Alpha characters must have ordinal values 0 to 127"; //#14034
                bwipp_raiseerror(); //#14034
            } //#14034
            if (($_.indx >= 32) && ($_.indx <= 126)) { //#14039
                $put($_.txt, $_.j, $a([$geti($_.barcode, $_.i, 1), ($_.j * 16) + 16, $_.textyoffset, $_.textfont, $_.textsize])); //#14037
            } else { //#14039
                $put($_.txt, $_.j, $a([" ", ($_.j * 16) + 16, $_.textyoffset, $_.textfont, $_.textsize])); //#14039
            } //#14039
            $_.i = $_.i + 1; //#14041
        } //#14041
        $_.checksum = $f($_.checksum + $_.indx); //#14043
        $_.enc = $get($_.encs, $_.indx); //#14044
        $puti($_.sbs, $_.l, $_.enc); //#14045
        $_.l = $_.l + $_.enc.length; //#14046
        $_.j = $_.j + 1; //#14047
    } //#14047
    $_.checksum = (127 - ($_.checksum % 127)) % 127; //#14051
    $_.enc = $get($_.encs, $_.checksum); //#14052
    $puti($_.sbs, $_.l, $_.enc); //#14053
    $_.l = $_.l + $_.enc.length; //#14054
    $_.enc = $get($_.encs, 122); //#14057
    $puti($_.sbs, $_.l, $_.enc); //#14058
    $_.l = $_.l + $_.enc.length; //#14059
    $_.sbs = $geti($_.sbs, 0, $_.l); //#14062
    $_.txt = $geti($_.txt, 0, $_.j); //#14063
    $k[$j++] = Infinity; //#14066
    $k[$j++] = Infinity; //#14068
    var _1z = $_.sbs; //#14068
    for (var _20 = 0, _21 = _1z.length; _20 < _21; _20++) { //#14068
        $k[$j++] = $get(_1z, _20) - 48; //#14068
    } //#14068
    var _23 = $a(); //#14068
    $k[$j++] = Infinity; //#14069
    for (var _25 = 0, _26 = ~~(($_.sbs.length + 1) / 2); _25 < _26; _25++) { //#14069
        $k[$j++] = $_.height; //#14069
    } //#14069
    var _28 = $a(); //#14069
    $k[$j++] = Infinity; //#14070
    for (var _2A = 0, _2B = ~~(($_.sbs.length + 1) / 2); _2A < _2B; _2A++) { //#14070
        $k[$j++] = 0; //#14070
    } //#14070
    var _2C = $a(); //#14070
    $k[$j++] = 'ren'; //#14073
    $k[$j++] = 'renlinear'; //#14073
    $k[$j++] = 'sbs'; //#14073
    $k[$j++] = _23; //#14073
    $k[$j++] = 'bhs'; //#14073
    $k[$j++] = _28; //#14073
    $k[$j++] = 'bbs'; //#14073
    $k[$j++] = _2C; //#14073
    if ($_.includetext) { //#14073
        $k[$j++] = 'txt'; //#14072
        $k[$j++] = $_.txt; //#14072
    } //#14072
    $k[$j++] = 'opt'; //#14074
    $k[$j++] = $_.options; //#14074
    var _2G = $d(); //#14074
    $k[$j++] = _2G; //#14077
    if (!$_.dontdraw) { //#14077
        bwipp_renlinear(); //#14077
    } //#14077
    $_ = $__; //#14081
} //bwipp_telepen
function bwipp_telepennumeric() {
    var $__ = $_; //#14110
    $_ = Object.assign({}, $_); //#14110
    $_.dontdraw = false; //#14113
    $k[$j++] = $_; //#14115
    bwipp_processoptions(); //#14115
    $_.options = $k[--$j]; //#14115
    $_.barcode = $k[--$j]; //#14116
    $put($_.options, "dontdraw", true); //#14119
    $put($_.options, "numeric", true); //#14120
    $k[$j++] = 'args'; //#14122
    $k[$j++] = $_.barcode; //#14122
    $k[$j++] = $_.options; //#14122
    bwipp_telepen(); //#14122
    var _6 = $k[--$j]; //#14122
    $_[$k[--$j]] = _6; //#14122
    $put($_.args, "opt", $_.options); //#14124
    $k[$j++] = $_.args; //#14127
    if (!$_.dontdraw) { //#14127
        bwipp_renlinear(); //#14127
    } //#14127
    $_ = $__; //#14129
} //bwipp_telepennumeric
function bwipp_posicode() {
    var $__ = $_; //#14160
    $_ = Object.assign({}, $_); //#14160
    $_.dontdraw = false; //#14163
    $_.includetext = false; //#14164
    $_.textfont = "OCR-B"; //#14165
    $_.textsize = 10; //#14166
    $_.textxoffset = 0; //#14167
    $_.textyoffset = -8; //#14168
    $_.height = 1; //#14169
    $_.encoding = "auto"; //#14170
    $_.version = "a"; //#14171
    $_.checkoffset = 0; //#14172
    $_.raw = false; //#14173
    $_.parse = false; //#14174
    $_.parsefnc = false; //#14175
    $k[$j++] = $_; //#14177
    bwipp_processoptions(); //#14177
    $_.options = $k[--$j]; //#14177
    var _1 = $k[--$j]; //#14178
    $_.barcode = _1; //#14178
    $_.la0 = -1; //#14184
    $_.la1 = -2; //#14184
    $_.la2 = -3; //#14184
    $_.sf0 = -4; //#14185
    $_.sf1 = -5; //#14185
    $_.sf2 = -6; //#14185
    $_.fn1 = -7; //#14186
    $_.fn2 = -8; //#14186
    $_.fn3 = -9; //#14186
    $_.fn4 = -10; //#14186
    var _r = $a([$_.sf1, $_.sf0, $_.fn3]); //#14205
    $_.charmapsnormal = $a([$a(["0", "^", "'"]), $a(["1", ";", 27]), $a(["2", "<", 28]), $a(["3", "=", 29]), $a(["4", ">", 30]), $a(["5", "?", 31]), $a(["6", "@", "!"]), $a(["7", "[", "\""]), $a(["8", 92, "#"]), $a(["9", "]", "&"]), $a(["A", "a", 1]), $a(["B", "b", 2]), $a(["C", "c", 3]), $a(["D", "d", 4]), $a(["E", "e", 5]), $a(["F", "f", 6]), $a(["G", "g", 7]), $a(["H", "h", 8]), $a(["I", "i", 9]), $a(["J", "j", 10]), $a(["K", "k", 11]), $a(["L", "l", 12]), $a(["M", "m", 13]), $a(["N", "n", 14]), $a(["O", "o", 15]), $a(["P", "p", 16]), $a(["Q", "q", 17]), $a(["R", "r", 18]), $a(["S", "s", 19]), $a(["T", "t", 20]), $a(["U", "u", 21]), $a(["V", "v", 22]), $a(["W", "w", 23]), $a(["X", "x", 24]), $a(["Y", "y", 25]), $a(["Z", "z", 26]), $a(["-", "_", 40]), $a([".", "`", 41]), $a([" ", 127, 0]), $a(["$", "{", "*"]), $a(["/", "|", ","]), $a(["+", "}", ":"]), $a(["%", "~", $_.fn1]), $a([$_.la1, $_.la0, $_.fn2]), _r, $a([$_.sf2, $_.sf2, $_.fn4])]); //#14207
    $_.chapmapslimited = $a([$a(["0", -98, -98]), $a(["1", -98, -98]), $a(["2", -98, -98]), $a(["3", -98, -98]), $a(["4", -98, -98]), $a(["5", -98, -98]), $a(["6", -98, -98]), $a(["7", -98, -98]), $a(["8", -98, -98]), $a(["9", -98, -98]), $a(["A", -98, -98]), $a(["B", -98, -98]), $a(["C", -98, -98]), $a(["D", -98, -98]), $a(["E", -98, -98]), $a(["F", -98, -98]), $a(["G", -98, -98]), $a(["H", -98, -98]), $a(["I", -98, -98]), $a(["J", -98, -98]), $a(["K", -98, -98]), $a(["L", -98, -98]), $a(["M", -98, -98]), $a(["N", -98, -98]), $a(["O", -98, -98]), $a(["P", -98, -98]), $a(["Q", -98, -98]), $a(["R", -98, -98]), $a(["S", -98, -98]), $a(["T", -98, -98]), $a(["U", -98, -98]), $a(["V", -98, -98]), $a(["W", -98, -98]), $a(["X", -98, -98]), $a(["Y", -98, -98]), $a(["Z", -98, -98]), $a(["-", -98, -98]), $a([".", -98, -98])]); //#14222
    if ((($ne($_.version, "a") && $ne($_.version, "b")) && $ne($_.version, "limiteda")) && $ne($_.version, "limitedb")) { //#14227
        $k[$j++] = 'bwipp.posicodeInvalidVersion#14226'; //#14226
        $k[$j++] = "The version must be either a, b, limiteda or limitedb"; //#14226
        bwipp_raiseerror(); //#14226
    } //#14226
    $k[$j++] = 'charmaps'; //#14229
    if ($eq($_.version, "a") || $eq($_.version, "b")) { //#14229
        $k[$j++] = $_.charmapsnormal; //#14229
    } else { //#14229
        $k[$j++] = $_.chapmapslimited; //#14229
    } //#14229
    var _1j = $k[--$j]; //#14229
    $_[$k[--$j]] = _1j; //#14229
    var _1l = $_.charmaps; //#14233
    var _1m = $_.charmaps; //#14233
    var _1n = $_.charmaps; //#14233
    $_.charvals = $a([new Map, new Map, new Map]); //#14234
    for (var _1r = 0, _1q = $_.charmaps.length - 1; _1r <= _1q; _1r += 1) { //#14243
        $_.i = _1r; //#14236
        $_.encs = $get($_.charmaps, $_.i); //#14237
        for (var _1v = 0; _1v <= 2; _1v += 1) { //#14242
            $_.j = _1v; //#14239
            var _1y = $get($_.encs, $_.j); //#14240
            $k[$j++] = _1y; //#14240
            if ($eq($type(_1y), 'stringtype')) { //#14240
                var _21 = $get($k[--$j], 0); //#14240
                $k[$j++] = _21; //#14240
            } //#14240
            $put($get($_.charvals, $_.j), $k[--$j], $_.i); //#14241
        } //#14241
    } //#14241
    $_.set0 = $get($_.charvals, 0); //#14244
    $_.set1 = $get($_.charvals, 1); //#14245
    $_.set2 = $get($_.charvals, 2); //#14246
    if ($eq($_.version, "limiteda") || $eq($_.version, "limitedb")) { //#14255
        for (var _2H = 0, _2G = $_.barcode.length - 1; _2H <= _2G; _2H += 1) { //#14254
            var _2M = $get($_.set0, $get($geti($_.barcode, _2H, 1), 0)) !== undefined; //#14251
            if (!_2M) { //#14253
                $k[$j++] = 'bwipp.posicodeBadCharacter#14252'; //#14252
                $k[$j++] = "Posicode limited must contain only digits, capital letters, and the symbols - and ."; //#14252
                bwipp_raiseerror(); //#14252
            } //#14252
        } //#14252
    } //#14252
    if ($_.raw) { //#14257
        $_.encoding = "raw"; //#14257
    } //#14257
    if ($eq($_.encoding, "raw")) { //#14271
        $_.cws = $a($_.barcode.length); //#14260
        $_.i = 0; //#14261
        $_.j = 0; //#14261
        for (;;) { //#14268
            if ($_.i == $_.barcode.length) { //#14263
                break; //#14263
            } //#14263
            $_.cw = $cvi($geti($_.barcode, $_.i + 1, 3)); //#14264
            $put($_.cws, $_.j, $_.cw); //#14265
            $_.i = $_.i + 4; //#14266
            $_.j = $_.j + 1; //#14267
        } //#14267
        $_.cws = $geti($_.cws, 0, $_.j); //#14269
        $_.text = ""; //#14270
    } //#14270
    if ($eq($_.encoding, "auto")) { //#14386
        var _2k = new Map([
            ["parse", $_.parse],
            ["parsefnc", $_.parsefnc],
            ["FNC1", $_.fn1],
            ["FNC2", $_.fn2],
            ["FNC3", $_.fn3]
        ]); //#14281
        $_.fncvals = _2k; //#14283
        $k[$j++] = 'msg'; //#14284
        $k[$j++] = $_.barcode; //#14284
        $k[$j++] = $_.fncvals; //#14284
        bwipp_parseinput(); //#14284
        var _2n = $k[--$j]; //#14284
        $_[$k[--$j]] = _2n; //#14284
        $_.msglen = $_.msg.length; //#14285
        $_.text = $s($_.msglen); //#14287
        for (var _2u = 0, _2t = $_.msglen - 1; _2u <= _2t; _2u += 1) { //#14291
            $_.i = _2u; //#14289
            var _2z = $get($_.msg, $_.i); //#14290
            $k[$j++] = $_.text; //#14290
            $k[$j++] = $_.i; //#14290
            $k[$j++] = _2z; //#14290
            if (_2z < 0) { //#14290
                $k[$j - 1] = 32; //#14290
            } //#14290
            var _30 = $k[--$j]; //#14290
            var _31 = $k[--$j]; //#14290
            $put($k[--$j], _31, _30); //#14290
        } //#14290
        $k[$j++] = Infinity; //#14294
        for (var _34 = 0, _35 = $_.msglen; _34 < _35; _34++) { //#14294
            $k[$j++] = 0; //#14294
        } //#14294
        $k[$j++] = 0; //#14294
        $_.numSA = $a(); //#14294
        $k[$j++] = Infinity; //#14295
        for (var _38 = 0, _39 = $_.msglen; _38 < _39; _38++) { //#14295
            $k[$j++] = 0; //#14295
        } //#14295
        $k[$j++] = 0; //#14295
        $_.numEA = $a(); //#14295
        for (var _3C = $_.msglen - 1; _3C >= 0; _3C -= 1) { //#14305
            $_.i = _3C; //#14297
            if ($get($_.msg, $_.i) >= 0) { //#14304
                if ($get($_.msg, $_.i) >= 128) { //#14302
                    $put($_.numEA, $_.i, $get($_.numEA, $_.i + 1) + 1); //#14300
                } else { //#14302
                    $put($_.numSA, $_.i, $get($_.numSA, $_.i + 1) + 1); //#14302
                } //#14302
            } //#14302
        } //#14302
        $_.ea = false; //#14308
        $_.msgtmp = $a([]); //#14308
        for (var _3W = 0, _3V = $_.msglen - 1; _3W <= _3V; _3W += 1) { //#14323
            $_.i = _3W; //#14310
            $_.c = $get($_.msg, $_.i); //#14311
            if ((!($_.ea != ($_.c < 128))) && ($_.c >= 0)) { //#14321
                if ($_.ea) { //#14313
                    $k[$j++] = $_.numSA; //#14313
                } else { //#14313
                    $k[$j++] = $_.numEA; //#14313
                } //#14313
                var _3i = $get($k[--$j], $_.i); //#14313
                var _3l = ($f(_3i + $_.i) == $_.msglen) ? 3 : 5; //#14314
                if (_3i < _3l) { //#14319
                    $k[$j++] = Infinity; //#14316
                    $aload($_.msgtmp); //#14316
                    $k[$j++] = $_.fn4; //#14316
                    $_.msgtmp = $a(); //#14316
                } else { //#14319
                    $k[$j++] = Infinity; //#14318
                    $aload($_.msgtmp); //#14318
                    $k[$j++] = $_.fn4; //#14318
                    $k[$j++] = $_.fn4; //#14318
                    $_.msgtmp = $a(); //#14318
                    $_.ea = !$_.ea; //#14319
                } //#14319
            } //#14319
            $k[$j++] = Infinity; //#14322
            $aload($_.msgtmp); //#14322
            if ($_.c >= 0) { //#14322
                $k[$j++] = $_.c & 127; //#14322
            } else { //#14322
                $k[$j++] = $_.c; //#14322
            } //#14322
            $_.msgtmp = $a(); //#14322
        } //#14322
        $_.msg = $_.msgtmp; //#14324
        $_.msglen = $_.msg.length; //#14325
        $_.enc = function() {
            var _41 = $k[--$j]; //#14329
            $put($_.cws, $_.j, $get(_41, $k[--$j])); //#14329
            $_.j = $_.j + 1; //#14330
        }; //#14330
        $_.cws = $a($_.msglen * 2); //#14333
        $_.i = 0; //#14334
        $_.j = 0; //#14334
        $_.cset = "set0"; //#14334
        for (;;) { //#14384
            if ($_.i == $_.msglen) { //#14336
                break; //#14336
            } //#14336
            for (;;) { //#14382
                $_.char1 = $get($_.msg, $_.i); //#14341
                $k[$j++] = 'char2'; //#14342
                if (($_.i + 1) < $_.msglen) { //#14342
                    $k[$j++] = $get($_.msg, $_.i + 1); //#14342
                } else { //#14342
                    $k[$j++] = -99; //#14342
                } //#14342
                var _4J = $k[--$j]; //#14342
                $_[$k[--$j]] = _4J; //#14342
                var _4O = $get($_[$_.cset], $_.char1) !== undefined; //#14345
                if (_4O) { //#14349
                    $k[$j++] = $_.char1; //#14346
                    $k[$j++] = $_[$_.cset]; //#14346
                    $_.enc(); //#14346
                    $_.i = $_.i + 1; //#14347
                    break; //#14348
                } //#14348
                var _4V = $get($_.set2, $_.char1) !== undefined; //#14352
                if (_4V) { //#14357
                    $k[$j++] = $_.sf2; //#14353
                    $k[$j++] = $_[$_.cset]; //#14353
                    $_.enc(); //#14353
                    $k[$j++] = $_.char1; //#14354
                    $k[$j++] = $_.set2; //#14354
                    $_.enc(); //#14354
                    $_.i = $_.i + 1; //#14355
                    break; //#14356
                } //#14356
                var _4f = $get($_[$_.cset], $_.char2) !== undefined; //#14360
                if (!_4f) { //#14378
                    if ($eq($_.cset, "set0")) { //#14366
                        $k[$j++] = $_.la1; //#14362
                        $k[$j++] = $_[$_.cset]; //#14362
                        $_.enc(); //#14362
                        $_.cset = "set1"; //#14363
                    } else { //#14366
                        $k[$j++] = $_.la0; //#14365
                        $k[$j++] = $_[$_.cset]; //#14365
                        $_.enc(); //#14365
                        $_.cset = "set0"; //#14366
                    } //#14366
                    break; //#14368
                } else { //#14378
                    if ($eq($_.cset, "set0")) { //#14375
                        $k[$j++] = $_.sf1; //#14371
                        $k[$j++] = $_[$_.cset]; //#14371
                        $_.enc(); //#14371
                        $k[$j++] = $_.char1; //#14372
                        $k[$j++] = $_.set1; //#14372
                        $_.enc(); //#14372
                    } else { //#14375
                        $k[$j++] = $_.sf0; //#14374
                        $k[$j++] = $_[$_.cset]; //#14374
                        $_.enc(); //#14374
                        $k[$j++] = $_.char1; //#14375
                        $k[$j++] = $_.set0; //#14375
                        $_.enc(); //#14375
                    } //#14375
                    $_.i = $_.i + 1; //#14377
                    break; //#14378
                } //#14378
                break; //#14381
            } //#14381
        } //#14381
        $_.cws = $geti($_.cws, 0, $_.j); //#14385
    } //#14385
    var _56 = new Map([
        ["a", $a(["141112", "131212", "121312", "111412", "131113", "121213", "111313", "121114", "111214", "111115", "181111", "171211", "161311", "151411", "141511", "131611", "121711", "111811", "171112", "161212", "151312", "141412", "131512", "121612", "111712", "161113", "151213", "141313", "131413", "121513", "111613", "151114", "141214", "131314", "121414", "111514", "141115", "131215", "121315", "111415", "131116", "121216", "111316", "121117", "111217", "111118", "1<111112", "111111111;1"])],
        ["b", $a(["151213", "141313", "131413", "121513", "141214", "131314", "121414", "131215", "121315", "121216", "191212", "181312", "171412", "161512", "151612", "141712", "131812", "121912", "181213", "171313", "161413", "151513", "141613", "131713", "121813", "171214", "161314", "151414", "141514", "131614", "121714", "161215", "151315", "141415", "131515", "121615", "151216", "141316", "131416", "121516", "141217", "131317", "121417", "131218", "121318", "121219", "1<121312", "121212121<1"])],
        ["limiteda", $a(["111411", "111312", "111213", "111114", "121311", "121212", "121113", "141111", "131211", "131112", "171111", "161211", "151311", "141411", "131511", "121611", "111711", "161112", "151212", "141312", "131412", "121512", "111612", "151113", "141213", "131313", "121413", "111513", "141114", "131214", "121314", "111414", "131115", "121215", "111315", "121116", "111216", "111117", "151111", "1"])],
        ["limitedb", $a(["121512", "121413", "121314", "121215", "131412", "131313", "131214", "151212", "141312", "141213", "181212", "171312", "161412", "151512", "141612", "131712", "121812", "171213", "161313", "151413", "141513", "131613", "121713", "161214", "151314", "141414", "131514", "121614", "151215", "141315", "131415", "121515", "141216", "131316", "121416", "131217", "121317", "121218", "141212", "1"])]
    ]); //#14433
    $_.encmaps = _56; //#14435
    $_.encs = $get($_.encmaps, $_.version); //#14437
    $_.c2w = $a([$a([495, 330, 210, 126, 70, 35, 15, 5]), $a([165, 120, 84, 56, 35, 20, 10, 4]), $a([45, 36, 28, 21, 15, 10, 6, 3]), $a([9, 8, 7, 6, 5, 4, 3, 2]), $a([1, 1, 1, 1, 1, 1, 1, 1])]); //#14446
    $_.v = 0; //#14448
    var _5I = $_.cws; //#14449
    for (var _5J = 0, _5K = _5I.length; _5J < _5K; _5J++) { //#14458
        $_.cw = $get(_5I, _5J); //#14450
        for (var _5M = 0; _5M < 6; _5M++) { //#14457
            if ((($_.cw ^ $_.v) & 1) != 0) { //#14454
                $_.v = $_.v ^ 7682; //#14453
            } //#14453
            $_.v = $_.v >>> 1; //#14455
            $_.cw = $_.cw >>> 1; //#14456
        } //#14456
    } //#14456
    $_.v = $_.v + $_.checkoffset; //#14459
    if ($eq($_.version, "limiteda") || $eq($_.version, "limitedb")) { //#14467
        $_.v = $_.v & 1023; //#14462
        if (($_.v > 824) && ($_.v < 853)) { //#14465
            $_.v = $_.v + 292; //#14464
        } //#14464
    } else { //#14467
        $_.v = ($_.v & 1023) + 45; //#14467
    } //#14467
    $_.d = $a([2, 2, 2, 2, 2, 2]); //#14470
    $_.r = 0; //#14471
    $_.c = 0; //#14471
    $_.w = 0; //#14471
    $_.sum = 0; //#14471
    for (;;) { //#14490
        if ($_.sum == $_.v) { //#14473
            break; //#14473
        } //#14473
        $_.t = $f($_.sum + $get($get($_.c2w, $_.r), $_.c)); //#14474
        if ($_.t == $_.v) { //#14479
            $_.w = $_.w + 1; //#14476
            $put($_.d, $_.r, $_.w + 2); //#14477
            $_.sum = $_.t; //#14478
        } //#14478
        if ($_.t > $_.v) { //#14484
            $put($_.d, $_.r, $_.w + 2); //#14481
            $_.r = $_.r + 1; //#14482
            $_.w = 0; //#14483
        } //#14483
        if ($_.t < $_.v) { //#14489
            $_.c = $_.c + 1; //#14486
            $_.w = $_.w + 1; //#14487
            $_.sum = $_.t; //#14488
        } //#14488
    } //#14488
    $k[$j++] = 20; //#14491
    for (var _62 = 0; _62 <= 4; _62 += 1) { //#14491
        var _65 = $k[--$j]; //#14491
        $k[$j++] = $f(_65 - $get($_.d, _62)); //#14491
    } //#14491
    $put($_.d, 5, $k[--$j]); //#14492
    if ($eq($_.version, "b") || $eq($_.version, "limitedb")) { //#14495
        $k[$j++] = Infinity; //#14494
        var _6A = $_.d; //#14494
        for (var _6B = 0, _6C = _6A.length; _6B < _6C; _6B++) { //#14494
            $k[$j++] = $get(_6A, _6B) + 1; //#14494
        } //#14494
        $_.d = $a(); //#14494
    } //#14494
    $_.cbs = $strcpy($s(12), "111111111111"); //#14496
    for (var _6H = 5; _6H >= 0; _6H -= 1) { //#14500
        $_.i = _6H; //#14498
        $put($_.cbs, ((5 - $_.i) * 2) + 1, $get($_.d, $_.i) + 47); //#14499
    } //#14499
    $_.sbs = $s(($_.cws.length * 6) + 31); //#14503
    var _6R = $get($_.encs, $_.encs.length - 2); //#14504
    $puti($_.sbs, 0, _6R); //#14505
    $_.j = _6R.length; //#14506
    for (var _6V = 0, _6U = $_.cws.length - 1; _6V <= _6U; _6V += 1) { //#14511
        $_.i = _6V; //#14508
        $puti($_.sbs, $_.j, $get($_.encs, $get($_.cws, $_.i))); //#14509
        $_.j = $_.j + 6; //#14510
    } //#14510
    $puti($_.sbs, $_.j, $_.cbs); //#14512
    $_.j = $_.j + 12; //#14513
    var _6k = $get($_.encs, $_.encs.length - 1); //#14514
    $puti($_.sbs, $_.j, _6k); //#14515
    $_.j = _6k.length + $_.j; //#14516
    $_.sbs = $geti($_.sbs, 0, $_.j); //#14517
    $k[$j++] = Infinity; //#14520
    $k[$j++] = Infinity; //#14522
    var _6r = $_.sbs; //#14522
    for (var _6s = 0, _6t = _6r.length; _6s < _6t; _6s++) { //#14522
        $k[$j++] = $get(_6r, _6s) - 48; //#14522
    } //#14522
    var _6v = $a(); //#14522
    $k[$j++] = Infinity; //#14523
    for (var _6x = 0, _6y = ~~(($_.sbs.length + 1) / 2); _6x < _6y; _6x++) { //#14523
        $k[$j++] = $_.height; //#14523
    } //#14523
    var _70 = $a(); //#14523
    $k[$j++] = Infinity; //#14524
    for (var _72 = 0, _73 = ~~(($_.sbs.length + 1) / 2); _72 < _73; _72++) { //#14524
        $k[$j++] = 0; //#14524
    } //#14524
    var _74 = $a(); //#14524
    var _7C = new Map([
        ["a", 0],
        ["b", 0],
        ["limiteda", 12],
        ["limitedb", 13]
    ]); //#14527
    var _7F = new Map([
        ["a", 0],
        ["b", 0],
        ["limiteda", 12],
        ["limitedb", 13]
    ]); //#14528
    $k[$j++] = 'ren'; //#14529
    $k[$j++] = 'renlinear'; //#14529
    $k[$j++] = 'sbs'; //#14529
    $k[$j++] = _6v; //#14529
    $k[$j++] = 'bhs'; //#14529
    $k[$j++] = _70; //#14529
    $k[$j++] = 'bbs'; //#14529
    $k[$j++] = _74; //#14529
    $k[$j++] = 'txt'; //#14529
    $k[$j++] = $a([$a([$_.text, $_.textxoffset, $_.textyoffset, $_.textfont, $_.textsize])]); //#14529
    $k[$j++] = 'textxalign'; //#14529
    $k[$j++] = "center"; //#14529
    $k[$j++] = 'borderleft'; //#14529
    $k[$j++] = $get(_7C, $_.version); //#14529
    $k[$j++] = 'borderright'; //#14529
    $k[$j++] = $get(_7F, $_.version); //#14529
    $k[$j++] = 'opt'; //#14529
    $k[$j++] = $_.options; //#14529
    var _7J = $d(); //#14529
    $k[$j++] = _7J; //#14532
    if (!$_.dontdraw) { //#14532
        bwipp_renlinear(); //#14532
    } //#14532
    $_ = $__; //#14536
} //bwipp_posicode
function bwipp_codablockf() {
    var $__ = $_; //#14567
    $_ = Object.assign({}, $_); //#14567
    $_.dontdraw = false; //#14570
    $_.rows = -1; //#14571
    $_.columns = 8; //#14572
    $_.rowheight = 10; //#14573
    $_.sepheight = 1; //#14574
    $_.parse = false; //#14575
    $_.parsefnc = false; //#14576
    $k[$j++] = $_; //#14578
    bwipp_processoptions(); //#14578
    $_.options = $k[--$j]; //#14578
    $_.barcode = $k[--$j]; //#14579
    if (($_.columns < 4) || ($_.columns > 62)) { //#14583
        $k[$j++] = 'bwipp.codablockfBadColumns#14582'; //#14582
        $k[$j++] = "Codablock F must have 4 to 62 columns"; //#14582
        bwipp_raiseerror(); //#14582
    } //#14582
    if (($_.rows != -1) && (($_.rows < 2) || ($_.rows > 44))) { //#14587
        $k[$j++] = 'bwipp.codablockfBadRows#14586'; //#14586
        $k[$j++] = "Codablock F must have 2 to 44 rows"; //#14586
        bwipp_raiseerror(); //#14586
    } //#14586
    if ($_.rowheight < 8) { //#14591
        $k[$j++] = 'bwipp.codablockfBadRowHeight#14590'; //#14590
        $k[$j++] = "Codablock F must have rowheight of at least 8"; //#14590
        bwipp_raiseerror(); //#14590
    } //#14590
    if ($_.sepheight < 1) { //#14595
        $k[$j++] = 'bwipp.codablockfBadSepHeight#14594'; //#14594
        $k[$j++] = "Codablock F must have sepheight of at least 1"; //#14594
        bwipp_raiseerror(); //#14594
    } //#14594
    $k[$j++] = 'c'; //#14599
    if (($_.columns >= 4) && ($_.columns <= 62)) { //#14599
        $k[$j++] = $_.columns; //#14599
    } else { //#14599
        $k[$j++] = 8; //#14599
    } //#14599
    var _C = $k[--$j]; //#14599
    $_[$k[--$j]] = _C; //#14599
    $k[$j++] = 'rows'; //#14600
    if (($_.rows >= 2) && ($_.rows <= 44)) { //#14600
        $k[$j++] = $_.rows; //#14600
    } else { //#14600
        $k[$j++] = -1; //#14600
    } //#14600
    var _H = $k[--$j]; //#14600
    $_[$k[--$j]] = _H; //#14600
    $_.swa = -1; //#14603
    $_.swb = -2; //#14603
    $_.swc = -3; //#14603
    $_.sft = -4; //#14603
    $_.fn1 = -5; //#14604
    $_.fn2 = -6; //#14604
    $_.fn3 = -7; //#14604
    $_.fn4 = -8; //#14604
    $_.sta = -9; //#14605
    $_.stp = -10; //#14605
    var _N = new Map([
        ["parse", $_.parse],
        ["parsefnc", $_.parsefnc],
        ["FNC1", $_.fn1],
        ["FNC3", $_.fn3]
    ]); //#14613
    $_.fncvals = _N; //#14615
    $k[$j++] = 'msg'; //#14616
    $k[$j++] = $_.barcode; //#14616
    $k[$j++] = $_.fncvals; //#14616
    bwipp_parseinput(); //#14616
    var _Q = $k[--$j]; //#14616
    $_[$k[--$j]] = _Q; //#14616
    $_.msglen = $_.msg.length; //#14617
    $_.msgtmp = $a([]); //#14620
    var _U = $_.msg; //#14621
    for (var _V = 0, _W = _U.length; _V < _W; _V++) { //#14624
        $_.char = $get(_U, _V); //#14622
        $k[$j++] = Infinity; //#14623
        $aload($_.msgtmp); //#14623
        if ($_.char < 128) { //#14623
            $k[$j++] = $_.char; //#14623
        } else { //#14623
            $k[$j++] = $_.fn4; //#14623
            $k[$j++] = $_.char & 127; //#14623
        } //#14623
        $_.msgtmp = $a(); //#14623
    } //#14623
    $_.kmsg = $_.msg; //#14625
    $_.msg = $_.msgtmp; //#14626
    $_.msglen = $_.msg.length; //#14627
    $_.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$_.fn3, $_.fn3, "96"]), $a([$_.fn2, $_.fn2, "97"]), $a([$_.sft, $_.sft, "98"]), $a([$_.swc, $_.swc, "99"]), $a([$_.swb, $_.fn4, $_.swb]), $a([$_.fn4, $_.swa, $_.swa]), $a([$_.fn1, $_.fn1, $_.fn1]), $a([$_.sta, $_.sta, $_.sta]), $a([$_.stp, $_.stp, $_.stp])]); //#14668
    $_.charvals = $a([new Map, new Map, new Map]); //#14671
    for (var _2p = 0, _2o = $_.charmaps.length - 1; _2p <= _2o; _2p += 1) { //#14680
        $_.i = _2p; //#14673
        $_.encs = $get($_.charmaps, $_.i); //#14674
        for (var _2t = 0; _2t <= 2; _2t += 1) { //#14679
            $_.j = _2t; //#14676
            var _2w = $get($_.encs, $_.j); //#14677
            $k[$j++] = _2w; //#14677
            if ($eq($type(_2w), 'stringtype')) { //#14677
                var _2z = $get($k[--$j], 0); //#14677
                $k[$j++] = _2z; //#14677
            } //#14677
            $put($get($_.charvals, $_.j), $k[--$j], $_.i); //#14678
        } //#14678
    } //#14678
    $_.seta = $get($_.charvals, 0); //#14681
    $_.setb = $get($_.charvals, 1); //#14682
    $_.setc = $get($_.charvals, 2); //#14683
    $_.numsscr = function() {
        $_.n = 0; //#14688
        $_.s = 0; //#14688
        $_.p = $k[--$j]; //#14689
        for (;;) { //#14703
            if ($_.p >= $_.msglen) { //#14690
                break; //#14690
            } //#14690
            if ($_.p != 0) { //#14693
                if ($get($_.msg, $_.p - 1) == $_.fn4) { //#14692
                    break; //#14692
                } //#14692
            } //#14692
            var _3M = $get($_.msg, $_.p); //#14694
            var _3O = $get($_.setc, _3M) !== undefined; //#14695
            $k[$j++] = _3M; //#14695
            if (!_3O) { //#14695
                $j--; //#14695
                break; //#14695
            } //#14695
            if ($k[--$j] == $_.fn1) { //#14699
                if (($_.s % 2) == 0) { //#14698
                    $_.s = $_.s + 1; //#14698
                } else { //#14698
                    break; //#14698
                } //#14698
            } //#14698
            $_.n = $_.n + 1; //#14700
            $_.s = $_.s + 1; //#14701
            $_.p = $_.p + 1; //#14702
        } //#14702
        $k[$j++] = $_.n; //#14704
        $k[$j++] = $_.s; //#14704
    }; //#14704
    $_.enca = function() {
        $put($_.cws, $_.j, $get($_.seta, $k[--$j])); //#14709
        $_.j = $_.j + 1; //#14710
    }; //#14710
    $_.encb = function() {
        $put($_.cws, $_.j, $get($_.setb, $k[--$j])); //#14713
        $_.j = $_.j + 1; //#14714
    }; //#14714
    $_.encc = function() {
        var _3k = $k[$j - 1]; //#14717
        if ($ne($type(_3k), 'arraytype')) { //#14720
            var _3o = $get($_.setc, $k[--$j]); //#14718
            $k[$j++] = _3o; //#14718
        } else { //#14720
            $aload($k[--$j]); //#14720
            var _3q = $k[--$j]; //#14720
            var _3r = $k[--$j]; //#14720
            $k[$j++] = $f((_3q - 48) + ((_3r - 48) * 10)); //#14720
        } //#14720
        $put($_.cws, $_.j, $k[--$j]); //#14722
        $_.j = $_.j + 1; //#14723
    }; //#14723
    $_.anotb = function() {
        var _3w = $k[--$j]; //#14727
        var _3y = $get($_.seta, _3w) !== undefined; //#14727
        var _40 = $get($_.setb, _3w) !== undefined; //#14727
        $k[$j++] = _3y && (!_40); //#14727
    }; //#14727
    $_.bnota = function() {
        var _41 = $k[--$j]; //#14728
        var _43 = $get($_.setb, _41) !== undefined; //#14728
        var _45 = $get($_.seta, _41) !== undefined; //#14728
        $k[$j++] = _43 && (!_45); //#14728
    }; //#14728
    $k[$j++] = Infinity; //#14731
    for (var _47 = 0, _48 = $_.msg.length; _47 < _48; _47++) { //#14731
        $k[$j++] = 0; //#14731
    } //#14731
    $k[$j++] = 9999; //#14731
    $_.nextanotb = $a(); //#14731
    $k[$j++] = Infinity; //#14732
    for (var _4B = 0, _4C = $_.msg.length; _4B < _4C; _4B++) { //#14732
        $k[$j++] = 0; //#14732
    } //#14732
    $k[$j++] = 9999; //#14732
    $_.nextbnota = $a(); //#14732
    for (var _4F = $_.msg.length - 1; _4F >= 0; _4F -= 1) { //#14745
        $_.i = _4F; //#14734
        $k[$j++] = $get($_.msg, $_.i); //#14735
        $_.anotb(); //#14735
        if ($k[--$j]) { //#14738
            $put($_.nextanotb, $_.i, 0); //#14736
        } else { //#14738
            $put($_.nextanotb, $_.i, $get($_.nextanotb, $_.i + 1) + 1); //#14738
        } //#14738
        $k[$j++] = $get($_.msg, $_.i); //#14740
        $_.bnota(); //#14740
        if ($k[--$j]) { //#14743
            $put($_.nextbnota, $_.i, 0); //#14741
        } else { //#14743
            $put($_.nextbnota, $_.i, $get($_.nextbnota, $_.i + 1) + 1); //#14743
        } //#14743
    } //#14743
    $_.abeforeb = function() {
        var _4c = $k[--$j]; //#14748
        $k[$j++] = $lt($get($_.nextanotb, _4c), $get($_.nextbnota, _4c)); //#14748
    }; //#14748
    $_.bbeforea = function() {
        var _4h = $k[--$j]; //#14749
        $k[$j++] = $lt($get($_.nextbnota, _4h), $get($_.nextanotb, _4h)); //#14749
    }; //#14749
    $_.padrow = function() {
        for (var _4n = 0, _4o = $k[--$j]; _4n < _4o; _4n++) { //#14759
            for (;;) { //#14758
                if ($eq($_.cset, "seta")) { //#14755
                    $k[$j++] = $_.swc; //#14755
                    $_.enca(); //#14755
                    $_.cset = "setc"; //#14755
                    break; //#14755
                } //#14755
                if ($eq($_.cset, "setb")) { //#14756
                    $k[$j++] = $_.swc; //#14756
                    $_.encb(); //#14756
                    $_.cset = "setc"; //#14756
                    break; //#14756
                } //#14756
                if ($eq($_.cset, "setc")) { //#14757
                    $k[$j++] = $_.swb; //#14757
                    $_.encc(); //#14757
                    $_.cset = "setb"; //#14757
                    break; //#14757
                } //#14757
            } //#14757
        } //#14757
    }; //#14757
    $_.encafitsrow = function() {
        if (($_.rem <= 2) && ($get($_.msg, $_.i) == $_.fn4)) { //#14774
            var _54 = ($_.rem == 2) && ($get($_.msg, $_.i + 1) <= 95); //#14766
            $k[$j++] = _54; //#14770
            if (_54) { //#14770
                $k[$j++] = $_.fn4; //#14767
                $_.enca(); //#14767
                $k[$j++] = $get($_.msg, $_.i + 1); //#14768
                $_.enca(); //#14768
                $_.i = $_.i + 2; //#14769
            } //#14769
        } else { //#14774
            $k[$j++] = $get($_.msg, $_.i); //#14772
            $_.enca(); //#14772
            $_.i = $_.i + 1; //#14773
            $k[$j++] = true; //#14774
        } //#14774
    }; //#14774
    $_.encbfitsrow = function() {
        if (($_.rem <= 2) && ($get($_.msg, $_.i) == $_.fn4)) { //#14790
            var _5N = ($_.rem == 2) && ($get($_.msg, $_.i + 1) >= 32); //#14782
            $k[$j++] = _5N; //#14786
            if (_5N) { //#14786
                $k[$j++] = $_.fn4; //#14783
                $_.encb(); //#14783
                $k[$j++] = $get($_.msg, $_.i + 1); //#14784
                $_.encb(); //#14784
                $_.i = $_.i + 2; //#14785
            } //#14785
        } else { //#14790
            $k[$j++] = $get($_.msg, $_.i); //#14788
            $_.encb(); //#14788
            $_.i = $_.i + 1; //#14789
            $k[$j++] = true; //#14790
        } //#14790
    }; //#14790
    $_.cws = $a(($_.c + 5) * 44); //#14795
    $_.i = 0; //#14797
    $_.j = 0; //#14797
    $_.r = 1; //#14797
    $_.lastrow = false; //#14798
    for (;;) { //#14965
        if ($_.lastrow) { //#14800
            break; //#14800
        } //#14800
        if ($_.r > 44) { //#14804
            $k[$j++] = 'bwipp.codablockfTooBig#14803'; //#14803
            $k[$j++] = "Maximum length exceeded"; //#14803
            bwipp_raiseerror(); //#14803
        } //#14803
        $k[$j++] = $_.sta; //#14807
        $_.enca(); //#14807
        if ($_.i < $_.msglen) { //#14808
            $k[$j++] = $_.i; //#14808
            $_.numsscr(); //#14808
        } else { //#14808
            $k[$j++] = -1; //#14808
            $k[$j++] = -1; //#14808
        } //#14808
        $_.nums = $k[--$j]; //#14808
        $_.nchars = $k[--$j]; //#14808
        for (;;) { //#14828
            if ($_.msglen == 0) { //#14814
                $k[$j++] = $_.swb; //#14811
                $_.enca(); //#14811
                $_.cset = "setb"; //#14812
                break; //#14813
            } //#14813
            if ($_.nums >= 2) { //#14819
                $k[$j++] = $_.swc; //#14816
                $_.enca(); //#14816
                $_.cset = "setc"; //#14817
                break; //#14818
            } //#14818
            $k[$j++] = $_.i; //#14820
            $_.abeforeb(); //#14820
            if ($k[--$j]) { //#14824
                $k[$j++] = $_.sft; //#14821
                $_.enca(); //#14821
                $_.cset = "seta"; //#14822
                break; //#14823
            } //#14823
            $k[$j++] = $_.swb; //#14825
            $_.enca(); //#14825
            $_.cset = "setb"; //#14826
            break; //#14827
        } //#14827
        $_.j = $_.j + 1; //#14829
        $_.endofrow = false; //#14832
        for (;;) { //#14950
            $_.rem = $f(($_.c + 3) - ($_.j % ($_.c + 5))); //#14833
            if (($_.i == $_.msglen) || $_.endofrow) { //#14835
                break; //#14835
            } //#14835
            $k[$j++] = $_.i; //#14837
            $_.numsscr(); //#14837
            $_.nums = $k[--$j]; //#14837
            $_.nchars = $k[--$j]; //#14837
            $k[$j++] = 'remnums'; //#14838
            if ($_.nums > ($_.rem * 2)) { //#14838
                $k[$j++] = $_.rem * 2; //#14838
            } else { //#14838
                $k[$j++] = $_.nums; //#14838
            } //#14838
            var _63 = $k[--$j]; //#14838
            $_[$k[--$j]] = _63; //#14838
            for (;;) { //#14948
                if ((($eq($_.cset, "seta") || $eq($_.cset, "setb")) && ($_.remnums >= 4)) && ($get($_.msg, $_.i) != $_.fn1)) { //#14876
                    if ((($_.remnums % 2) == 0) && ($_.rem >= 3)) { //#14858
                        $k[$j++] = $_.swc; //#14846
                        if ($eq($_.cset, "seta")) { //#14846
                            $_.enca(); //#14846
                        } else { //#14846
                            $_.encb(); //#14846
                        } //#14846
                        $_.cset = "setc"; //#14847
                        for (var _6G = 0; _6G < 2; _6G++) { //#14856
                            if ($get($_.msg, $_.i) == $_.fn1) { //#14854
                                $k[$j++] = $_.fn1; //#14850
                                $_.encc(); //#14850
                                $_.i = $_.i + 1; //#14851
                            } else { //#14854
                                $k[$j++] = $geti($_.msg, $_.i, 2); //#14853
                                $_.encc(); //#14853
                                $_.i = $_.i + 2; //#14854
                            } //#14854
                        } //#14854
                        break; //#14857
                    } //#14857
                    if ((($_.remnums % 2) != 0) && ($_.rem >= 4)) { //#14875
                        $k[$j++] = $get($_.msg, $_.i); //#14861
                        if ($eq($_.cset, "seta")) { //#14861
                            $_.enca(); //#14861
                        } else { //#14861
                            $_.encb(); //#14861
                        } //#14861
                        $_.i = $_.i + 1; //#14862
                        $k[$j++] = $_.swc; //#14863
                        if ($eq($_.cset, "seta")) { //#14863
                            $_.enca(); //#14863
                        } else { //#14863
                            $_.encb(); //#14863
                        } //#14863
                        $_.cset = "setc"; //#14864
                        for (var _6a = 0; _6a < 2; _6a++) { //#14873
                            if ($get($_.msg, $_.i) == $_.fn1) { //#14871
                                $k[$j++] = $_.fn1; //#14867
                                $_.encc(); //#14867
                                $_.i = $_.i + 1; //#14868
                            } else { //#14871
                                $k[$j++] = $geti($_.msg, $_.i, 2); //#14870
                                $_.encc(); //#14870
                                $_.i = $_.i + 2; //#14871
                            } //#14871
                        } //#14871
                        break; //#14874
                    } //#14874
                } //#14874
                $k[$j++] = $eq($_.cset, "setb"); //#14877
                $k[$j++] = $get($_.msg, $_.i); //#14877
                $_.anotb(); //#14877
                var _6p = $k[--$j]; //#14877
                var _6q = $k[--$j]; //#14877
                if (_6q && _6p && $_.rem >= 2) { //#14892
                    if ($_.i < ($_.msglen - 1)) { //#14886
                        $k[$j++] = $_.i + 1; //#14880
                        $_.bbeforea(); //#14880
                        if ($k[--$j]) { //#14885
                            $k[$j++] = $_.sft; //#14881
                            $_.encb(); //#14881
                            $k[$j++] = $get($_.msg, $_.i); //#14882
                            $_.enca(); //#14882
                            $_.i = $_.i + 1; //#14883
                            break; //#14884
                        } //#14884
                    } //#14884
                    $k[$j++] = $_.swa; //#14887
                    $_.encb(); //#14887
                    $_.cset = "seta"; //#14888
                    $k[$j++] = $get($_.msg, $_.i); //#14889
                    $_.enca(); //#14889
                    $_.i = $_.i + 1; //#14890
                    break; //#14891
                } //#14891
                $k[$j++] = $eq($_.cset, "seta"); //#14893
                $k[$j++] = $get($_.msg, $_.i); //#14893
                $_.bnota(); //#14893
                var _7A = $k[--$j]; //#14893
                var _7B = $k[--$j]; //#14893
                if (_7B && _7A && $_.rem >= 2) { //#14908
                    if ($_.i < ($_.msglen - 1)) { //#14902
                        $k[$j++] = $_.i + 1; //#14896
                        $_.abeforeb(); //#14896
                        if ($k[--$j]) { //#14901
                            $k[$j++] = $_.sft; //#14897
                            $_.enca(); //#14897
                            $k[$j++] = $get($_.msg, $_.i); //#14898
                            $_.encb(); //#14898
                            $_.i = $_.i + 1; //#14899
                            break; //#14900
                        } //#14900
                    } //#14900
                    $k[$j++] = $_.swb; //#14903
                    $_.enca(); //#14903
                    $_.cset = "setb"; //#14904
                    $k[$j++] = $get($_.msg, $_.i); //#14905
                    $_.encb(); //#14905
                    $_.i = $_.i + 1; //#14906
                    break; //#14907
                } //#14907
                if (($eq($_.cset, "setc") && ($_.remnums < 2)) && ($_.rem >= 2)) { //#14921
                    $_.rem = $_.rem - 1; //#14911
                    $k[$j++] = $_.i; //#14912
                    $_.abeforeb(); //#14912
                    if ($k[--$j]) { //#14919
                        $k[$j++] = $_.swa; //#14913
                        $_.encc(); //#14913
                        $_.cset = "seta"; //#14914
                        $_.encafitsrow(); //#14915
                        if ($k[--$j]) { //#14915
                            break; //#14915
                        } //#14915
                    } else { //#14919
                        $k[$j++] = $_.swb; //#14917
                        $_.encc(); //#14917
                        $_.cset = "setb"; //#14918
                        $_.encbfitsrow(); //#14919
                        if ($k[--$j]) { //#14919
                            break; //#14919
                        } //#14919
                    } //#14919
                } //#14919
                var _7g = $get($_.seta, $get($_.msg, $_.i)) !== undefined; //#14924
                if (($eq($_.cset, "seta") && _7g) && ($_.rem >= 1)) { //#14927
                    $_.encafitsrow(); //#14926
                    if ($k[--$j]) { //#14926
                        break; //#14926
                    } //#14926
                } //#14926
                var _7o = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#14928
                if (($eq($_.cset, "setb") && _7o) && ($_.rem >= 1)) { //#14931
                    $_.encbfitsrow(); //#14930
                    if ($k[--$j]) { //#14930
                        break; //#14930
                    } //#14930
                } //#14930
                if (($eq($_.cset, "setc") && ($_.remnums >= 2)) && ($_.rem >= 1)) { //#14942
                    if ($get($_.msg, $_.i) == $_.fn1) { //#14939
                        $k[$j++] = $_.fn1; //#14935
                        $_.encc(); //#14935
                        $_.i = $_.i + 1; //#14936
                    } else { //#14939
                        $k[$j++] = $geti($_.msg, $_.i, 2); //#14938
                        $_.encc(); //#14938
                        $_.i = $_.i + 2; //#14939
                    } //#14939
                    break; //#14941
                } //#14941
                $_.endofrow = true; //#14945
                break; //#14946
            } //#14946
        } //#14946
        if ((((($_.r >= $_.rows) || ($_.rows == -1)) && ($_.r > 1)) && ($_.i == $_.msglen)) && ($_.rem >= 2)) { //#14962
            $k[$j++] = $_.rem - 2; //#14954
            $_.padrow(); //#14954
            $_.j = $_.j + 3; //#14955
            $k[$j++] = $_.stp; //#14956
            $_.enca(); //#14956
            $_.lastrow = true; //#14957
        } else { //#14962
            $k[$j++] = $_.rem; //#14959
            $_.padrow(); //#14959
            $_.j = $_.j + 1; //#14960
            $k[$j++] = $_.stp; //#14961
            $_.enca(); //#14961
            $_.r = $_.r + 1; //#14962
        } //#14962
    } //#14962
    $_.cws = $geti($_.cws, 0, $_.j); //#14966
    $k[$j++] = Infinity; //#14970
    for (var _8L = 64; _8L <= 95; _8L += 1) { //#14971
        $k[$j++] = _8L; //#14971
    } //#14971
    for (var _8M = 0; _8M <= 15; _8M += 1) { //#14972
        $k[$j++] = _8M; //#14972
    } //#14972
    for (var _8N = 26; _8N <= 63; _8N += 1) { //#14973
        $k[$j++] = _8N; //#14973
    } //#14973
    $_.abmap = $a(); //#14973
    $k[$j++] = Infinity; //#14975
    for (var _8P = 0; _8P <= 85; _8P += 1) { //#14975
        $k[$j++] = _8P; //#14975
    } //#14975
    $_.cmap = $a(); //#14975
    $_.chkmsg = $a($_.kmsg.length); //#14979
    $_.j = 0; //#14980
    for (var _8W = 0, _8V = $_.kmsg.length - 1; _8W <= _8V; _8W += 1) { //#14986
        $_.i = _8W; //#14982
        $_.char = $get($_.kmsg, $_.i); //#14983
        if ($_.char >= 0) { //#14984
            $put($_.chkmsg, $_.j, $_.char); //#14984
            $_.j = $_.j + 1; //#14984
        } //#14984
        if (($_.char == $_.fn1) && ($_.i != 0)) { //#14985
            $put($_.chkmsg, $_.j, 29); //#14985
            $_.j = $_.j + 1; //#14985
        } //#14985
    } //#14985
    $_.t1 = 0; //#14987
    $_.t2 = 0; //#14987
    $_.k1 = 0; //#14987
    $_.k2 = 0; //#14987
    for (var _8n = 0, _8m = $_.j - 1; _8n <= _8m; _8n += 1) { //#14994
        $_.i = _8n; //#14989
        $_.t1 = ($get($_.chkmsg, $_.i) * $_.i) % 86; //#14990
        $_.t2 = $f($_.t1 + $get($_.chkmsg, $_.i)) % 86; //#14991
        $_.k1 = $f($_.k1 + $_.t2) % 86; //#14992
        $_.k2 = $f($_.k2 + $_.t1) % 86; //#14993
    } //#14993
    $k[$j++] = $_.cws; //#14995
    $k[$j++] = $_.cws.length - 4; //#14995
    if ($ne($_.cset, "setc")) { //#14995
        $k[$j++] = $_.abmap; //#14995
    } else { //#14995
        $k[$j++] = $_.cmap; //#14995
    } //#14995
    var _97 = $get($k[--$j], $_.k1); //#14995
    var _98 = $k[--$j]; //#14995
    $put($k[--$j], _98, _97); //#14995
    $k[$j++] = $_.cws; //#14996
    $k[$j++] = $_.cws.length - 3; //#14996
    if ($ne($_.cset, "setc")) { //#14996
        $k[$j++] = $_.abmap; //#14996
    } else { //#14996
        $k[$j++] = $_.cmap; //#14996
    } //#14996
    var _9H = $get($k[--$j], $_.k2); //#14996
    var _9I = $k[--$j]; //#14996
    $put($k[--$j], _9I, _9H); //#14996
    var _9K = $_.cws; //#14999
    $k[$j++] = _9K; //#14999
    $k[$j++] = 2; //#14999
    if ($get(_9K, 2 - 1) != 99) { //#14999
        $k[$j++] = $_.abmap; //#14999
    } else { //#14999
        $k[$j++] = $_.cmap; //#14999
    } //#14999
    var _9Q = $get($k[--$j], $_.r - 2); //#14999
    var _9R = $k[--$j]; //#14999
    $put($k[--$j], _9R, _9Q); //#14999
    for (var _9V = 1, _9U = $_.r - 1; _9V <= _9U; _9V += 1) { //#15003
        $_.i = _9V; //#15001
        var _9W = $_.cws; //#15002
        var _9X = $_.i; //#15002
        var _9Y = $_.c; //#15002
        $k[$j++] = _9W; //#15002
        $k[$j++] = (_9X * (_9Y + 5)) + 2; //#15002
        if ($get(_9W, ((_9X * (_9Y + 5)) + 2) - 1) != 99) { //#15002
            $k[$j++] = $_.abmap; //#15002
        } else { //#15002
            $k[$j++] = $_.cmap; //#15002
        } //#15002
        var _9e = $get($k[--$j], $_.i + 42); //#15002
        var _9f = $k[--$j]; //#15002
        $put($k[--$j], _9f, _9e); //#15002
    } //#15002
    for (var _9j = 0, _9i = $_.r - 1; _9j <= _9i; _9j += 1) { //#15014
        $_.rcws = $geti($_.cws, _9j * ($_.c + 5), $_.c + 4); //#15007
        $_.csum = $get($_.rcws, 0); //#15008
        for (var _9s = 1, _9r = $_.rcws.length - 2; _9s <= _9r; _9s += 1) { //#15012
            $_.i = _9s; //#15010
            $_.csum = $f($_.csum + ($get($_.rcws, $_.i) * $_.i)); //#15011
        } //#15011
        $put($_.rcws, $_.rcws.length - 1, $_.csum % 103); //#15013
    } //#15013
    var _A2 = $get($_.options, 'debugcws') !== undefined; //#15016
    if (_A2) { //#15016
        $k[$j++] = 'bwipp.debugcws#15016'; //#15016
        $k[$j++] = $_.cws; //#15016
        bwipp_raiseerror(); //#15016
    } //#15016
    $_.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "2331112"]); //#15036
    $_.rowbits = $a($_.r); //#15040
    for (var _AA = 0, _A9 = $_.r - 1; _AA <= _A9; _AA += 1) { //#15052
        $_.i = _AA; //#15042
        $k[$j++] = Infinity; //#15043
        var _AF = $geti($_.cws, $_.i * ($_.c + 5), $_.c + 5); //#15044
        for (var _AG = 0, _AH = _AF.length; _AG < _AH; _AG++) { //#15046
            $forall($get($_.encs, $get(_AF, _AG)), function() { //#15045
                var _AL = $k[--$j]; //#15045
                $k[$j++] = _AL - 48; //#15045
            }); //#15045
        } //#15045
        $_.sbs = $a(); //#15045
        $k[$j++] = Infinity; //#15048
        var _AN = $_.sbs; //#15049
        $k[$j++] = 0; //#15049
        for (var _AO = 0, _AP = _AN.length; _AO < _AP; _AO++) { //#15049
            var _AR = $k[$j - 1]; //#15049
            var _AS = (_AR == 1) ? 0 : 1; //#15049
            for (var _AT = 0, _AU = $get(_AN, _AO); _AT < _AU; _AT++) { //#15049
                $k[$j++] = _AS //#15049
            } //#15049
        } //#15049
        $astore($a($counttomark() - 1)); //#15050
        var _AX = $k[--$j]; //#15050
        $k[$j - 1] = _AX; //#15050
        var _AZ = $k[--$j]; //#15050
        $k[$j - 1] = _AZ; //#15050
        $k[$j++] = $_.rowbits; //#15051
        $k[$j++] = $_.i; //#15051
        var _Ad = $k[$j - 3]; //#15051
        var _Ae = $k[$j - 1]; //#15051
        $put($k[$j - 2], _Ae, _Ad); //#15051
        $j -= 3; //#15051
    } //#15051
    $_.symwid = ($_.c * 11) + 57; //#15055
    $k[$j++] = Infinity; //#15056
    for (var _Aj = 0, _Ak = $_.symwid * $_.sepheight; _Aj < _Ak; _Aj++) { //#15057
        $k[$j++] = 1; //#15057
    } //#15057
    for (var _An = 0, _Am = $_.r - 2; _An <= _Am; _An += 1) { //#15066
        $_.i = _An; //#15059
        for (var _Ap = 0, _Aq = $_.rowheight; _Ap < _Aq; _Ap++) { //#15060
            $aload($get($_.rowbits, $_.i)); //#15060
        } //#15060
        for (var _Av = 0, _Aw = $_.sepheight; _Av < _Aw; _Av++) { //#15065
            $k[$j++] = 1; //#15063
            $k[$j++] = 1; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 1; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 1; //#15063
            $k[$j++] = 0; //#15063
            $k[$j++] = 0; //#15063
            for (var _Ay = 0, _Az = $_.symwid - 24; _Ay < _Az; _Ay++) { //#15063
                $k[$j++] = 1; //#15063
            } //#15063
            $k[$j++] = 1; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 0; //#15064
            $k[$j++] = 0; //#15064
            $k[$j++] = 0; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 0; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 0; //#15064
            $k[$j++] = 1; //#15064
            $k[$j++] = 1; //#15064
        } //#15064
    } //#15064
    for (var _B1 = 0, _B2 = $_.rowheight; _B1 < _B2; _B1++) { //#15067
        $aload($get($_.rowbits, $_.r - 1)); //#15067
    } //#15067
    for (var _B8 = 0, _B9 = $_.symwid * $_.sepheight; _B8 < _B9; _B8++) { //#15068
        $k[$j++] = 1; //#15068
    } //#15068
    $_.pixs = $a(); //#15068
    var _BJ = new Map([
        ["ren", 'renmatrix'],
        ["pixs", $_.pixs],
        ["pixx", $_.symwid],
        ["pixy", ~~($_.pixs.length / $_.symwid)],
        ["height", (~~($_.pixs.length / $_.symwid)) / 72],
        ["width", $_.symwid / 72],
        ["opt", $_.options]
    ]); //#15079
    $k[$j++] = _BJ; //#15082
    if (!$_.dontdraw) { //#15082
        bwipp_renmatrix(); //#15082
    } //#15082
    $_ = $__; //#15086
} //bwipp_codablockf
function bwipp_code16k() {
    var $__ = $_; //#15117
    $_ = Object.assign({}, $_); //#15117
    $_.dontdraw = false; //#15120
    $_.sam = -1; //#15121
    $_.rows = 0; //#15122
    $_.rowheight = 8; //#15123
    $_.sepheight = 1; //#15124
    $_.raw = false; //#15125
    $_.parse = false; //#15126
    $_.parsefnc = false; //#15127
    $k[$j++] = $_; //#15129
    bwipp_processoptions(); //#15129
    $_.options = $k[--$j]; //#15129
    $_.barcode = $k[--$j]; //#15130
    if (($_.rows != 0) && (($_.rows < 2) || ($_.rows > 16))) { //#15134
        $k[$j++] = 'bwipp.code16kBadRows#15133'; //#15133
        $k[$j++] = "Code 16K must have 2 to 16 rows"; //#15133
        bwipp_raiseerror(); //#15133
    } //#15133
    if ($_.rowheight < 8) { //#15138
        $k[$j++] = 'bwipp.code16kBadRowHeight#15137'; //#15137
        $k[$j++] = "Code 16K must have rowheight of at least 8"; //#15137
        bwipp_raiseerror(); //#15137
    } //#15137
    if ($_.sepheight < 1) { //#15142
        $k[$j++] = 'bwipp.code16kBadSepHeight#15141'; //#15141
        $k[$j++] = "Code 16K must have sepheight of at least 1"; //#15141
        bwipp_raiseerror(); //#15141
    } //#15141
    if ($_.sam != -1) { //#15154
        $k[$j++] = true; //#15150
        for (var _8 = 0; _8 < 1; _8++) { //#15150
            var _9 = $_.sam; //#15147
            if ((_9 < 12) || (_9 > 99)) { //#15147
                $k[$j - 1] = false; //#15147
                break; //#15147
            } //#15147
            if (($_.sam % 10) < 2) { //#15148
                $k[$j - 1] = false; //#15148
                break; //#15148
            } //#15148
            if ((~~($_.sam / 10)) > ($_.sam % 10)) { //#15149
                $k[$j - 1] = false; //#15149
                break; //#15149
            } //#15149
        } //#15149
        if (!$k[--$j]) { //#15153
            $k[$j++] = 'bwipp.code16kBadSAM#15152'; //#15152
            $k[$j++] = "SAM must be formatted as \"NM\" for Nth of M symbols, from 2 to 9 symbols"; //#15152
            bwipp_raiseerror(); //#15152
        } //#15152
    } //#15152
    if ($_.sam != -1) { //#15158
        $_.rows = 16; //#15158
    } //#15158
    $_.swa = -1; //#15162
    $_.swb = -2; //#15162
    $_.swc = -3; //#15162
    $_.sa1 = -4; //#15163
    $_.sb1 = -5; //#15163
    $_.sc1 = -6; //#15163
    $_.sa2 = -7; //#15164
    $_.sb2 = -8; //#15164
    $_.sc2 = -9; //#15164
    $_.pad = -10; //#15165
    $_.sb3 = -11; //#15165
    $_.sc3 = -12; //#15165
    $_.fn1 = -13; //#15166
    $_.fn2 = -14; //#15166
    $_.fn3 = -15; //#15166
    $_.fn4 = -16; //#15166
    $_.charmaps = $a([$a([32, 32, "00"]), $a(["!", "!", "01"]), $a(["\"", "\"", "02"]), $a(["#", "#", "03"]), $a(["$", "$", "04"]), $a(["%", "%", "05"]), $a(["&", "&", "06"]), $a(["'", "'", "07"]), $a([40, 40, "08"]), $a([41, 41, "09"]), $a(["*", "*", "10"]), $a(["+", "+", "11"]), $a([",", ",", "12"]), $a(["-", "-", "13"]), $a([".", ".", "14"]), $a(["/", "/", "15"]), $a(["0", "0", "16"]), $a(["1", "1", "17"]), $a(["2", "2", "18"]), $a(["3", "3", "19"]), $a(["4", "4", "20"]), $a(["5", "5", "21"]), $a(["6", "6", "22"]), $a(["7", "7", "23"]), $a(["8", "8", "24"]), $a(["9", "9", "25"]), $a([":", ":", "26"]), $a([";", ";", "27"]), $a(["<", "<", "28"]), $a(["=", "=", "29"]), $a([">", ">", "30"]), $a(["?", "?", "31"]), $a(["@", "@", "32"]), $a(["A", "A", "33"]), $a(["B", "B", "34"]), $a(["C", "C", "35"]), $a(["D", "D", "36"]), $a(["E", "E", "37"]), $a(["F", "F", "38"]), $a(["G", "G", "39"]), $a(["H", "H", "40"]), $a(["I", "I", "41"]), $a(["J", "J", "42"]), $a(["K", "K", "43"]), $a(["L", "L", "44"]), $a(["M", "M", "45"]), $a(["N", "N", "46"]), $a(["O", "O", "47"]), $a(["P", "P", "48"]), $a(["Q", "Q", "49"]), $a(["R", "R", "50"]), $a(["S", "S", "51"]), $a(["T", "T", "52"]), $a(["U", "U", "53"]), $a(["V", "V", "54"]), $a(["W", "W", "55"]), $a(["X", "X", "56"]), $a(["Y", "Y", "57"]), $a(["Z", "Z", "58"]), $a(["[", "[", "59"]), $a([92, 92, "60"]), $a(["]", "]", "61"]), $a(["^", "^", "62"]), $a(["_", "_", "63"]), $a([0, "`", "64"]), $a([1, "a", "65"]), $a([2, "b", "66"]), $a([3, "c", "67"]), $a([4, "d", "68"]), $a([5, "e", "69"]), $a([6, "f", "70"]), $a([7, "g", "71"]), $a([8, "h", "72"]), $a([9, "i", "73"]), $a([10, "j", "74"]), $a([11, "k", "75"]), $a([12, "l", "76"]), $a([13, "m", "77"]), $a([14, "n", "78"]), $a([15, "o", "79"]), $a([16, "p", "80"]), $a([17, "q", "81"]), $a([18, "r", "82"]), $a([19, "s", "83"]), $a([20, "t", "84"]), $a([21, "u", "85"]), $a([22, "v", "86"]), $a([23, "w", "87"]), $a([24, "x", "88"]), $a([25, "y", "89"]), $a([26, "z", "90"]), $a([27, "{", "91"]), $a([28, "|", "92"]), $a([29, "}", "93"]), $a([30, "~", "94"]), $a([31, 127, "95"]), $a([$_.fn3, $_.fn3, "96"]), $a([$_.fn2, $_.fn2, "97"]), $a([$_.sb1, $_.sa1, "98"]), $a([$_.swc, $_.swc, "99"]), $a([$_.swb, $_.fn4, $_.swb]), $a([$_.fn4, $_.swa, $_.swa]), $a([$_.fn1, $_.fn1, $_.fn1]), $a([$_.pad, $_.pad, $_.pad]), $a([$_.sb2, $_.sa2, $_.sb1]), $a([$_.sc2, $_.sc2, $_.sb2]), $a([$_.sc3, $_.sc3, $_.sb3])]); //#15207
    $_.charvals = $a([new Map, new Map, new Map]); //#15210
    for (var _2V = 0, _2U = $_.charmaps.length - 1; _2V <= _2U; _2V += 1) { //#15219
        $_.i = _2V; //#15212
        $_.encs = $get($_.charmaps, $_.i); //#15213
        for (var _2Z = 0; _2Z <= 2; _2Z += 1) { //#15218
            $_.j = _2Z; //#15215
            var _2c = $get($_.encs, $_.j); //#15216
            $k[$j++] = _2c; //#15216
            if ($eq($type(_2c), 'stringtype')) { //#15216
                var _2f = $get($k[--$j], 0); //#15216
                $k[$j++] = _2f; //#15216
            } //#15216
            $put($get($_.charvals, $_.j), $k[--$j], $_.i); //#15217
        } //#15217
    } //#15217
    $_.seta = $get($_.charvals, 0); //#15220
    $_.setb = $get($_.charvals, 1); //#15221
    $_.setc = $get($_.charvals, 2); //#15222
    if ($_.raw) { //#15693
        $_.cws = $a($_.barcode.length); //#15227
        $_.i = 0; //#15228
        $_.j = 0; //#15228
        for (;;) { //#15239
            if ($_.i >= ($_.barcode.length - 3)) { //#15230
                break; //#15230
            } //#15230
            if ($get($_.barcode, $_.i) != 94) { //#15231
                break; //#15231
            } //#15231
            var _32 = $geti($_.barcode, $_.i + 1, 3); //#15232
            $k[$j++] = _32; //#15234
            $k[$j++] = false; //#15234
            for (var _33 = 0, _34 = _32.length; _33 < _34; _33++) { //#15234
                var _35 = $get(_32, _33); //#15234
                if ((_35 < 48) || (_35 > 57)) { //#15233
                    $k[$j - 1] = true; //#15233
                } //#15233
            } //#15233
            if ($k[--$j]) { //#15234
                $j--; //#15234
                break; //#15234
            } //#15234
            $_.cw = $cvi($k[--$j]); //#15235
            $put($_.cws, $_.j, $_.cw); //#15236
            $_.i = $_.i + 4; //#15237
            $_.j = $_.j + 1; //#15238
        } //#15238
        if ($_.i != $_.barcode.length) { //#15242
            $k[$j++] = 'bwipp.code16kBadRawFormat#15241'; //#15241
            $k[$j++] = "raw data must be formatted as ^NNN"; //#15241
            bwipp_raiseerror(); //#15241
        } //#15241
        $_.cws = $geti($_.cws, 0, $_.j); //#15243
    } else { //#15693
        var _3N = new Map([
            ["parse", $_.parse],
            ["parsefnc", $_.parsefnc],
            ["eci", true],
            ["FNC1", $_.fn1],
            ["FNC2", $_.fn2],
            ["FNC3", $_.fn3]
        ]); //#15254
        $_.fncvals = _3N; //#15256
        $k[$j++] = 'msg'; //#15257
        $k[$j++] = $_.barcode; //#15257
        $k[$j++] = $_.fncvals; //#15257
        bwipp_parseinput(); //#15257
        var _3Q = $k[--$j]; //#15257
        $_[$k[--$j]] = _3Q; //#15257
        $_.msglen = $_.msg.length; //#15258
        $k[$j++] = Infinity; //#15261
        for (var _3U = 0, _3V = $_.msglen; _3U < _3V; _3U++) { //#15261
            $k[$j++] = 0; //#15261
        } //#15261
        $k[$j++] = 0; //#15261
        $_.numSA = $a(); //#15261
        $k[$j++] = Infinity; //#15262
        for (var _3Y = 0, _3Z = $_.msglen; _3Y < _3Z; _3Y++) { //#15262
            $k[$j++] = 0; //#15262
        } //#15262
        $k[$j++] = 0; //#15262
        $_.numEA = $a(); //#15262
        for (var _3c = $_.msglen - 1; _3c >= 0; _3c -= 1) { //#15272
            $_.i = _3c; //#15264
            if ($get($_.msg, $_.i) >= 0) { //#15271
                if ($get($_.msg, $_.i) >= 128) { //#15269
                    $put($_.numEA, $_.i, $get($_.numEA, $_.i + 1) + 1); //#15267
                } else { //#15269
                    $put($_.numSA, $_.i, $get($_.numSA, $_.i + 1) + 1); //#15269
                } //#15269
            } //#15269
        } //#15269
        $_.ea = false; //#15275
        $_.msgtmp = $a([]); //#15275
        for (var _3w = 0, _3v = $_.msglen - 1; _3w <= _3v; _3w += 1) { //#15290
            $_.i = _3w; //#15277
            $_.c = $get($_.msg, $_.i); //#15278
            if ((!($_.ea != ($_.c < 128))) && ($_.c >= 0)) { //#15288
                if ($_.ea) { //#15280
                    $k[$j++] = $_.numSA; //#15280
                } else { //#15280
                    $k[$j++] = $_.numEA; //#15280
                } //#15280
                var _48 = $get($k[--$j], $_.i); //#15280
                var _4B = ($f(_48 + $_.i) == $_.msglen) ? 3 : 5; //#15281
                if (_48 < _4B) { //#15286
                    $k[$j++] = Infinity; //#15283
                    $aload($_.msgtmp); //#15283
                    $k[$j++] = $_.fn4; //#15283
                    $_.msgtmp = $a(); //#15283
                } else { //#15286
                    $k[$j++] = Infinity; //#15285
                    $aload($_.msgtmp); //#15285
                    $k[$j++] = $_.fn4; //#15285
                    $k[$j++] = $_.fn4; //#15285
                    $_.msgtmp = $a(); //#15285
                    $_.ea = !$_.ea; //#15286
                } //#15286
            } //#15286
            $k[$j++] = Infinity; //#15289
            $aload($_.msgtmp); //#15289
            if ($_.c >= 0) { //#15289
                $k[$j++] = $_.c & 127; //#15289
            } else { //#15289
                $k[$j++] = $_.c; //#15289
            } //#15289
            $_.msgtmp = $a(); //#15289
        } //#15289
        $_.msg = $_.msgtmp; //#15291
        $_.msglen = $_.msg.length; //#15292
        $_.numsscr = function() {
            $_.n = 0; //#15296
            $_.s = 0; //#15296
            $_.p = $k[--$j]; //#15297
            for (;;) { //#15308
                if ($_.p >= $_.msglen) { //#15298
                    break; //#15298
                } //#15298
                var _4W = $get($_.msg, $_.p); //#15299
                var _4Y = $get($_.setc, _4W) !== undefined; //#15300
                $k[$j++] = _4W; //#15300
                if (!_4Y) { //#15300
                    $j--; //#15300
                    break; //#15300
                } //#15300
                if ($k[--$j] == $_.fn1) { //#15304
                    if (($_.s % 2) == 0) { //#15303
                        $_.s = $_.s + 1; //#15303
                    } else { //#15303
                        break; //#15303
                    } //#15303
                } //#15303
                $_.n = $_.n + 1; //#15305
                $_.s = $_.s + 1; //#15306
                $_.p = $_.p + 1; //#15307
            } //#15307
            $k[$j++] = $_.n; //#15309
            $k[$j++] = $_.s; //#15309
        }; //#15309
        $_.enca = function() {
            $put($_.cws, $_.j, $get($_.seta, $k[--$j])); //#15314
            $_.j = $_.j + 1; //#15315
        }; //#15315
        $_.encb = function() {
            $put($_.cws, $_.j, $get($_.setb, $k[--$j])); //#15318
            $_.j = $_.j + 1; //#15319
        }; //#15319
        $_.encc = function() {
            var _4u = $k[$j - 1]; //#15322
            if ($ne($type(_4u), 'arraytype')) { //#15325
                var _4y = $get($_.setc, $k[--$j]); //#15323
                $k[$j++] = _4y; //#15323
            } else { //#15325
                $aload($k[--$j]); //#15325
                var _50 = $k[--$j]; //#15325
                var _51 = $k[--$j]; //#15325
                $k[$j++] = $f((_50 - 48) + ((_51 - 48) * 10)); //#15325
            } //#15325
            $put($_.cws, $_.j, $k[--$j]); //#15327
            $_.j = $_.j + 1; //#15328
        }; //#15328
        $_.anotb = function() {
            var _56 = $k[--$j]; //#15332
            var _58 = $get($_.seta, _56) !== undefined; //#15332
            var _5A = $get($_.setb, _56) !== undefined; //#15332
            $k[$j++] = _58 && (!_5A); //#15332
        }; //#15332
        $_.bnota = function() {
            var _5B = $k[--$j]; //#15333
            var _5D = $get($_.setb, _5B) !== undefined; //#15333
            var _5F = $get($_.seta, _5B) !== undefined; //#15333
            $k[$j++] = _5D && (!_5F); //#15333
        }; //#15333
        $k[$j++] = Infinity; //#15336
        for (var _5H = 0, _5I = $_.msg.length; _5H < _5I; _5H++) { //#15336
            $k[$j++] = 0; //#15336
        } //#15336
        $k[$j++] = 9999; //#15336
        $_.nextanotb = $a(); //#15336
        $k[$j++] = Infinity; //#15337
        for (var _5L = 0, _5M = $_.msg.length; _5L < _5M; _5L++) { //#15337
            $k[$j++] = 0; //#15337
        } //#15337
        $k[$j++] = 9999; //#15337
        $_.nextbnota = $a(); //#15337
        for (var _5P = $_.msg.length - 1; _5P >= 0; _5P -= 1) { //#15350
            $_.i = _5P; //#15339
            $k[$j++] = $get($_.msg, $_.i); //#15340
            $_.anotb(); //#15340
            if ($k[--$j]) { //#15343
                $put($_.nextanotb, $_.i, 0); //#15341
            } else { //#15343
                $put($_.nextanotb, $_.i, $get($_.nextanotb, $_.i + 1) + 1); //#15343
            } //#15343
            $k[$j++] = $get($_.msg, $_.i); //#15345
            $_.bnota(); //#15345
            if ($k[--$j]) { //#15348
                $put($_.nextbnota, $_.i, 0); //#15346
            } else { //#15348
                $put($_.nextbnota, $_.i, $get($_.nextbnota, $_.i + 1) + 1); //#15348
            } //#15348
        } //#15348
        $_.abeforeb = function() {
            var _5m = $k[--$j]; //#15353
            $k[$j++] = $lt($get($_.nextanotb, _5m), $get($_.nextbnota, _5m)); //#15353
        }; //#15353
        $_.bbeforea = function() {
            var _5r = $k[--$j]; //#15354
            $k[$j++] = $lt($get($_.nextbnota, _5r), $get($_.nextanotb, _5r)); //#15354
        }; //#15354
        $_.cws = $a(($_.barcode.length * 2) + 3); //#15356
        $_.i = 0; //#15359
        $_.j = 0; //#15359
        for (;;) { //#15449
            if ($_.sam != -1) { //#15370
                $put($_.cws, 0, (~~($_.sam / 10)) - 1); //#15364
                $put($_.cws, 1, ($_.sam % 10) - 1); //#15365
                $_.j = 2; //#15366
                $_.cset = "setb"; //#15367
                $_.mode = 7; //#15368
                break; //#15369
            } //#15369
            if ($_.msglen == 0) { //#15376
                $_.cset = "setb"; //#15373
                $_.mode = 1; //#15374
                break; //#15375
            } //#15375
            if ($_.msglen >= 2) { //#15385
                $k[$j++] = $get($_.msg, 0) == $_.fn1; //#15379
                $k[$j++] = 1; //#15379
                $_.numsscr(); //#15379
                var _68 = $k[--$j]; //#15379
                $k[$j - 1] = _68; //#15379
                var _6A = $k[--$j]; //#15379
                var _6B = $k[--$j]; //#15379
                if (_6B && (_6A >= 2)) { //#15384
                    $_.cset = "setc"; //#15380
                    $_.mode = 4; //#15381
                    $_.i = 1; //#15382
                    break; //#15383
                } //#15383
            } //#15383
            if ($get($_.msg, 0) == $_.fn1) { //#15392
                $_.cset = "setb"; //#15388
                $_.mode = 3; //#15389
                $_.i = 1; //#15390
                break; //#15391
            } //#15391
            if ($_.msglen >= 2) { //#15438
                $k[$j++] = 0; //#15395
                $_.numsscr(); //#15395
                var _6G = $k[--$j]; //#15395
                $k[$j - 1] = _6G; //#15395
                var _6I = $k[--$j]; //#15395
                if ((_6I >= 2) && ((_6I % 2) == 0)) { //#15399
                    $_.cset = "setc"; //#15396
                    $_.mode = 2; //#15397
                    break; //#15398
                } //#15398
                $k[$j++] = 0; //#15401
                $_.numsscr(); //#15401
                var _6J = $k[--$j]; //#15401
                $k[$j - 1] = _6J; //#15401
                var _6L = $k[--$j]; //#15401
                if ((_6L >= 3) && ((_6L % 2) == 1)) { //#15407
                    $k[$j++] = $get($_.msg, 0); //#15402
                    $_.encb(); //#15402
                    $_.cset = "setc"; //#15403
                    $_.mode = 5; //#15404
                    $_.i = 1; //#15405
                    break; //#15406
                } //#15406
                var _6R = $get($_.setb, $get($_.msg, 0)) !== undefined; //#15409
                $k[$j++] = _6R; //#15410
                $k[$j++] = 1; //#15410
                $_.numsscr(); //#15410
                var _6S = $k[--$j]; //#15410
                $k[$j - 1] = _6S; //#15410
                var _6U = $k[--$j]; //#15410
                var _6V = $k[--$j]; //#15410
                if (_6V && ((_6U >= 2) && ((_6U % 2) == 0))) { //#15416
                    $k[$j++] = $get($_.msg, 0); //#15411
                    $_.encb(); //#15411
                    $_.cset = "setc"; //#15412
                    $_.mode = 5; //#15413
                    $_.i = 1; //#15414
                    break; //#15415
                } //#15415
                var _6b = $get($_.setb, $get($_.msg, 0)) !== undefined; //#15418
                $k[$j++] = _6b; //#15419
                $k[$j++] = 1; //#15419
                $_.numsscr(); //#15419
                var _6c = $k[--$j]; //#15419
                $k[$j - 1] = _6c; //#15419
                var _6e = $k[--$j]; //#15419
                var _6f = $k[--$j]; //#15419
                if (_6f && ((_6e >= 3) && ((_6e % 2) == 1))) { //#15426
                    $k[$j++] = $get($_.msg, 0); //#15420
                    $_.encb(); //#15420
                    $k[$j++] = $get($_.msg, 1); //#15421
                    $_.encb(); //#15421
                    $_.cset = "setc"; //#15422
                    $_.mode = 6; //#15423
                    $_.i = 2; //#15424
                    break; //#15425
                } //#15425
                var _6n = $get($_.setb, $get($_.msg, 0)) !== undefined; //#15428
                var _6r = $get($_.setb, $get($_.msg, 1)) !== undefined; //#15429
                $k[$j++] = _6n && _6r; //#15430
                $k[$j++] = 2; //#15430
                $_.numsscr(); //#15430
                var _6s = $k[--$j]; //#15430
                $k[$j - 1] = _6s; //#15430
                var _6u = $k[--$j]; //#15430
                var _6v = $k[--$j]; //#15430
                if (_6v && ((_6u >= 2) && ((_6u % 2) == 0))) { //#15437
                    $k[$j++] = $get($_.msg, 0); //#15431
                    $_.encb(); //#15431
                    $k[$j++] = $get($_.msg, 1); //#15432
                    $_.encb(); //#15432
                    $_.cset = "setc"; //#15433
                    $_.mode = 6; //#15434
                    $_.i = 2; //#15435
                    break; //#15436
                } //#15436
            } //#15436
            $k[$j++] = 0; //#15440
            $_.abeforeb(); //#15440
            if ($k[--$j]) { //#15444
                $_.cset = "seta"; //#15441
                $_.mode = 0; //#15442
                break; //#15443
            } //#15443
            $_.cset = "setb"; //#15446
            $_.mode = 1; //#15447
            break; //#15448
        } //#15448
        for (;;) { //#15692
            if ($_.i == $_.msglen) { //#15452
                break; //#15452
            } //#15452
            $k[$j++] = $_.i; //#15454
            $_.numsscr(); //#15454
            $_.nums = $k[--$j]; //#15454
            $_.nchars = $k[--$j]; //#15454
            for (;;) { //#15690
                if ($eq($_.cset, "seta")) { //#15531
                    if ($_.i < ($_.msglen - 1)) { //#15468
                        $k[$j++] = $get($_.msg, $_.i); //#15461
                        $_.bnota(); //#15461
                        $k[$j++] = $_.i + 1; //#15462
                        $_.abeforeb(); //#15462
                        var _7D = $k[--$j]; //#15462
                        if ($k[--$j] && _7D) { //#15467
                            $k[$j++] = $_.sb1; //#15463
                            $_.enca(); //#15463
                            $k[$j++] = $get($_.msg, $_.i); //#15464
                            $_.encb(); //#15464
                            $_.i = $_.i + 1; //#15465
                            break; //#15466
                        } //#15466
                    } //#15466
                    if ($_.i < ($_.msglen - 2)) { //#15480
                        $k[$j++] = $get($_.msg, $_.i); //#15471
                        $_.bnota(); //#15471
                        $k[$j++] = $get($_.msg, $_.i + 1); //#15472
                        $_.bnota(); //#15472
                        var _7S = $k[--$j]; //#15472
                        var _7T = $k[--$j]; //#15472
                        $k[$j++] = $an(_7T, _7S); //#15473
                        $k[$j++] = $_.i + 2; //#15473
                        $_.abeforeb(); //#15473
                        var _7V = $k[--$j]; //#15473
                        if ($k[--$j] && _7V) { //#15479
                            $k[$j++] = $_.sb2; //#15474
                            $_.enca(); //#15474
                            $k[$j++] = $get($_.msg, $_.i); //#15475
                            $_.encb(); //#15475
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15476
                            $_.encb(); //#15476
                            $_.i = $_.i + 2; //#15477
                            break; //#15478
                        } //#15478
                    } //#15478
                    $k[$j++] = $get($_.msg, $_.i); //#15482
                    $_.bnota(); //#15482
                    if ($k[--$j]) { //#15486
                        $k[$j++] = $_.swb; //#15483
                        $_.enca(); //#15483
                        $_.cset = "setb"; //#15484
                        break; //#15485
                    } //#15485
                    if ($_.i < ($_.msglen - 4)) { //#15503
                        var _7r = $get($_.seta, $get($_.msg, $_.i + 4)) !== undefined; //#15490
                        if (($_.nums == 4) && _7r) { //#15502
                            $k[$j++] = $_.sc2; //#15491
                            $_.enca(); //#15491
                            for (var _7t = 0; _7t < 2; _7t++) { //#15500
                                if ($get($_.msg, $_.i) == $_.fn1) { //#15498
                                    $k[$j++] = $_.fn1; //#15494
                                    $_.encc(); //#15494
                                    $_.i = $_.i + 1; //#15495
                                } else { //#15498
                                    $k[$j++] = $geti($_.msg, $_.i, 2); //#15497
                                    $_.encc(); //#15497
                                    $_.i = $_.i + 2; //#15498
                                } //#15498
                            } //#15498
                            break; //#15501
                        } //#15501
                    } //#15501
                    if ($_.i < ($_.msglen - 6)) { //#15520
                        var _8B = $get($_.seta, $get($_.msg, $_.i + 6)) !== undefined; //#15507
                        if (($_.nums == 6) && _8B) { //#15519
                            $k[$j++] = $_.sc3; //#15508
                            $_.enca(); //#15508
                            for (var _8D = 0; _8D < 3; _8D++) { //#15517
                                if ($get($_.msg, $_.i) == $_.fn1) { //#15515
                                    $k[$j++] = $_.fn1; //#15511
                                    $_.encc(); //#15511
                                    $_.i = $_.i + 1; //#15512
                                } else { //#15515
                                    $k[$j++] = $geti($_.msg, $_.i, 2); //#15514
                                    $_.encc(); //#15514
                                    $_.i = $_.i + 2; //#15515
                                } //#15515
                            } //#15515
                            break; //#15518
                        } //#15518
                    } //#15518
                    if (($_.nums >= 4) && (($_.nums % 2) == 0)) { //#15526
                        $k[$j++] = $_.swc; //#15523
                        $_.enca(); //#15523
                        $_.cset = "setc"; //#15524
                        break; //#15525
                    } //#15525
                    $k[$j++] = $get($_.msg, $_.i); //#15528
                    $_.enca(); //#15528
                    $_.i = $_.i + 1; //#15529
                    break; //#15530
                } //#15530
                if ($eq($_.cset, "setb")) { //#15605
                    if ($_.i < ($_.msglen - 1)) { //#15542
                        $k[$j++] = $get($_.msg, $_.i); //#15535
                        $_.anotb(); //#15535
                        $k[$j++] = $_.i + 1; //#15536
                        $_.bbeforea(); //#15536
                        var _8c = $k[--$j]; //#15536
                        if ($k[--$j] && _8c) { //#15541
                            $k[$j++] = $_.sa1; //#15537
                            $_.encb(); //#15537
                            $k[$j++] = $get($_.msg, $_.i); //#15538
                            $_.enca(); //#15538
                            $_.i = $_.i + 1; //#15539
                            break; //#15540
                        } //#15540
                    } //#15540
                    if ($_.i < ($_.msglen - 2)) { //#15554
                        $k[$j++] = $get($_.msg, $_.i); //#15545
                        $_.anotb(); //#15545
                        $k[$j++] = $get($_.msg, $_.i + 1); //#15546
                        $_.anotb(); //#15546
                        var _8r = $k[--$j]; //#15546
                        var _8s = $k[--$j]; //#15546
                        $k[$j++] = $an(_8s, _8r); //#15547
                        $k[$j++] = $_.i + 2; //#15547
                        $_.bbeforea(); //#15547
                        var _8u = $k[--$j]; //#15547
                        if ($k[--$j] && _8u) { //#15553
                            $k[$j++] = $_.sa2; //#15548
                            $_.encb(); //#15548
                            $k[$j++] = $get($_.msg, $_.i); //#15549
                            $_.enca(); //#15549
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15550
                            $_.enca(); //#15550
                            $_.i = $_.i + 2; //#15551
                            break; //#15552
                        } //#15552
                    } //#15552
                    $k[$j++] = $get($_.msg, $_.i); //#15556
                    $_.anotb(); //#15556
                    if ($k[--$j]) { //#15560
                        $k[$j++] = $_.swa; //#15557
                        $_.encb(); //#15557
                        $_.cset = "seta"; //#15558
                        break; //#15559
                    } //#15559
                    if ($_.i < ($_.msglen - 4)) { //#15577
                        var _9G = $get($_.setb, $get($_.msg, $_.i + 4)) !== undefined; //#15564
                        if (($_.nums == 4) && _9G) { //#15576
                            $k[$j++] = $_.sc2; //#15565
                            $_.encb(); //#15565
                            for (var _9I = 0; _9I < 2; _9I++) { //#15574
                                if ($get($_.msg, $_.i) == $_.fn1) { //#15572
                                    $k[$j++] = $_.fn1; //#15568
                                    $_.encc(); //#15568
                                    $_.i = $_.i + 1; //#15569
                                } else { //#15572
                                    $k[$j++] = $geti($_.msg, $_.i, 2); //#15571
                                    $_.encc(); //#15571
                                    $_.i = $_.i + 2; //#15572
                                } //#15572
                            } //#15572
                            break; //#15575
                        } //#15575
                    } //#15575
                    if ($_.i < ($_.msglen - 6)) { //#15594
                        var _9a = $get($_.setb, $get($_.msg, $_.i + 6)) !== undefined; //#15581
                        if (($_.nums == 6) && _9a) { //#15593
                            $k[$j++] = $_.sc3; //#15582
                            $_.encb(); //#15582
                            for (var _9c = 0; _9c < 3; _9c++) { //#15591
                                if ($get($_.msg, $_.i) == $_.fn1) { //#15589
                                    $k[$j++] = $_.fn1; //#15585
                                    $_.encc(); //#15585
                                    $_.i = $_.i + 1; //#15586
                                } else { //#15589
                                    $k[$j++] = $geti($_.msg, $_.i, 2); //#15588
                                    $_.encc(); //#15588
                                    $_.i = $_.i + 2; //#15589
                                } //#15589
                            } //#15589
                            break; //#15592
                        } //#15592
                    } //#15592
                    if (($_.nums >= 4) && (($_.nums % 2) == 0)) { //#15600
                        $k[$j++] = $_.swc; //#15597
                        $_.encb(); //#15597
                        $_.cset = "setc"; //#15598
                        break; //#15599
                    } //#15599
                    $k[$j++] = $get($_.msg, $_.i); //#15602
                    $_.encb(); //#15602
                    $_.i = $_.i + 1; //#15603
                    break; //#15604
                } //#15604
                if ($eq($_.cset, "setc")) { //#15688
                    if ($_.nums >= 2) { //#15617
                        if ($get($_.msg, $_.i) == $_.fn1) { //#15614
                            $k[$j++] = $_.fn1; //#15610
                            $_.encc(); //#15610
                            $_.i = $_.i + 1; //#15611
                        } else { //#15614
                            $k[$j++] = $geti($_.msg, $_.i, 2); //#15613
                            $_.encc(); //#15613
                            $_.i = $_.i + 2; //#15614
                        } //#15614
                        break; //#15616
                    } //#15616
                    if ($_.i < ($_.msglen - 1)) { //#15627
                        var _AC = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#15620
                        $k[$j++] = _AC; //#15621
                        $k[$j++] = $_.i + 1; //#15621
                        $_.numsscr(); //#15621
                        var _AE = $k[--$j]; //#15621
                        $k[$j - 1] = _AE; //#15621
                        var _AG = $k[--$j]; //#15621
                        var _AH = $k[--$j]; //#15621
                        if (_AH && ((_AG >= 2) && ((_AG % 2) == 0))) { //#15626
                            $k[$j++] = $_.sb1; //#15622
                            $_.encc(); //#15622
                            $k[$j++] = $get($_.msg, $_.i); //#15623
                            $_.encb(); //#15623
                            $_.i = $_.i + 1; //#15624
                            break; //#15625
                        } //#15625
                    } //#15625
                    if ($_.i < ($_.msglen - 1)) { //#15638
                        var _AT = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#15630
                        $k[$j++] = _AT; //#15631
                        $k[$j++] = $_.i + 1; //#15631
                        $_.numsscr(); //#15631
                        var _AV = $k[--$j]; //#15631
                        $k[$j - 1] = _AV; //#15631
                        var _AX = $k[--$j]; //#15631
                        var _AY = $k[--$j]; //#15631
                        if (_AY && ((_AX >= 3) && ((_AX % 2) == 1))) { //#15637
                            $k[$j++] = $_.sb2; //#15632
                            $_.encc(); //#15632
                            $k[$j++] = $get($_.msg, $_.i); //#15633
                            $_.encb(); //#15633
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15634
                            $_.encb(); //#15634
                            $_.i = $_.i + 2; //#15635
                            break; //#15636
                        } //#15636
                    } //#15636
                    if ($_.i < ($_.msglen - 2)) { //#15650
                        var _An = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#15641
                        var _As = $get($_.setb, $get($_.msg, $_.i + 1)) !== undefined; //#15642
                        $k[$j++] = _An && _As; //#15643
                        $k[$j++] = $_.i + 2; //#15643
                        $_.numsscr(); //#15643
                        var _Au = $k[--$j]; //#15643
                        $k[$j - 1] = _Au; //#15643
                        var _Aw = $k[--$j]; //#15643
                        var _Ax = $k[--$j]; //#15643
                        if (_Ax && ((_Aw >= 2) && ((_Aw % 2) == 0))) { //#15649
                            $k[$j++] = $_.sb2; //#15644
                            $_.encc(); //#15644
                            $k[$j++] = $get($_.msg, $_.i); //#15645
                            $_.encb(); //#15645
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15646
                            $_.encb(); //#15646
                            $_.i = $_.i + 2; //#15647
                            break; //#15648
                        } //#15648
                    } //#15648
                    if ($_.i < ($_.msglen - 3)) { //#15663
                        var _BC = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#15653
                        var _BH = $get($_.setb, $get($_.msg, $_.i + 1)) !== undefined; //#15654
                        $k[$j++] = _BC && _BH; //#15655
                        $k[$j++] = $_.i + 2; //#15655
                        $_.numsscr(); //#15655
                        var _BJ = $k[--$j]; //#15655
                        $k[$j - 1] = _BJ; //#15655
                        var _BL = $k[--$j]; //#15655
                        var _BM = $k[--$j]; //#15655
                        if (_BM && ((_BL >= 3) && ((_BL % 2) == 1))) { //#15662
                            $k[$j++] = $_.sb3; //#15656
                            $_.encc(); //#15656
                            $k[$j++] = $get($_.msg, $_.i); //#15657
                            $_.encb(); //#15657
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15658
                            $_.encb(); //#15658
                            $k[$j++] = $get($_.msg, $_.i + 2); //#15659
                            $_.encb(); //#15659
                            $_.i = $_.i + 3; //#15660
                            break; //#15661
                        } //#15661
                    } //#15661
                    if ($_.i < ($_.msglen - 3)) { //#15677
                        var _Be = $get($_.setb, $get($_.msg, $_.i)) !== undefined; //#15666
                        var _Bj = $get($_.setb, $get($_.msg, $_.i + 1)) !== undefined; //#15667
                        var _Bo = $get($_.setb, $get($_.msg, $_.i + 2)) !== undefined; //#15668
                        $k[$j++] = (_Be && _Bj) && _Bo; //#15669
                        $k[$j++] = $_.i + 3; //#15669
                        $_.numsscr(); //#15669
                        var _Bq = $k[--$j]; //#15669
                        $k[$j - 1] = _Bq; //#15669
                        var _Bs = $k[--$j]; //#15669
                        var _Bt = $k[--$j]; //#15669
                        if (_Bt && ((_Bs >= 2) && ((_Bs % 2) == 0))) { //#15676
                            $k[$j++] = $_.sb3; //#15670
                            $_.encc(); //#15670
                            $k[$j++] = $get($_.msg, $_.i); //#15671
                            $_.encb(); //#15671
                            $k[$j++] = $get($_.msg, $_.i + 1); //#15672
                            $_.encb(); //#15672
                            $k[$j++] = $get($_.msg, $_.i + 2); //#15673
                            $_.encb(); //#15673
                            $_.i = $_.i + 3; //#15674
                            break; //#15675
                        } //#15675
                    } //#15675
                    $k[$j++] = $_.i; //#15679
                    $_.abeforeb(); //#15679
                    if ($k[--$j]) { //#15683
                        $k[$j++] = $_.swa; //#15680
                        $_.encc(); //#15680
                        $_.cset = "seta"; //#15681
                        break; //#15682
                    } //#15682
                    $k[$j++] = $_.swb; //#15685
                    $_.encc(); //#15685
                    $_.cset = "setb"; //#15686
                    break; //#15687
                } //#15687
                break; //#15689
            } //#15689
        } //#15689
        $_.cws = $geti($_.cws, 0, $_.j); //#15693
    } //#15693
    $_.metrics = $a([$a([2, 7]), $a([3, 12]), $a([4, 17]), $a([5, 22]), $a([6, 27]), $a([7, 32]), $a([8, 37]), $a([9, 42]), $a([10, 47]), $a([11, 52]), $a([12, 57]), $a([13, 62]), $a([14, 67]), $a([15, 72]), $a([16, 77])]); //#15715
    $_.urows = $_.rows; //#15719
    var _CU = $_.metrics; //#15720
    for (var _CV = 0, _CW = _CU.length; _CV < _CW; _CV++) { //#15728
        $_.m = $get(_CU, _CV); //#15721
        $_.r = $get($_.m, 0); //#15722
        $_.dcws = $get($_.m, 1); //#15723
        $_.okay = true; //#15724
        if (($_.urows != 0) && ($_.urows != $_.r)) { //#15725
            $_.okay = false; //#15725
        } //#15725
        if ($_.cws.length > $_.dcws) { //#15726
            $_.okay = false; //#15726
        } //#15726
        if ($_.okay) { //#15727
            break; //#15727
        } //#15727
    } //#15727
    if (!$_.okay) { //#15732
        $k[$j++] = 'bwipp.code16kNoValidSymbol#15731'; //#15731
        $k[$j++] = "Maximum length exceeded or data too large for given options"; //#15731
        bwipp_raiseerror(); //#15731
    } //#15731
    $k[$j++] = Infinity; //#15735
    $aload($_.cws); //#15735
    for (var _Cm = 0, _Cn = $f($_.dcws - $_.cws.length); _Cm < _Cn; _Cm++) { //#15735
        $k[$j++] = 103; //#15735
    } //#15735
    $_.cws = $a(); //#15735
    $k[$j++] = Infinity; //#15738
    $k[$j++] = $f((($_.r - 2) * 7) + $_.mode); //#15738
    $aload($_.cws); //#15738
    $_.cws = $a(); //#15738
    $k[$j++] = 0; //#15741
    for (var _Cv = 0, _Cu = $_.dcws; _Cv <= _Cu; _Cv += 1) { //#15741
        var _Cy = $k[--$j]; //#15741
        $k[$j++] = $f(_Cy + ((_Cv + 2) * $get($_.cws, _Cv))); //#15741
    } //#15741
    $_.c1 = $k[--$j] % 107; //#15742
    $k[$j++] = 0; //#15743
    for (var _D2 = 0, _D1 = $_.dcws; _D2 <= _D1; _D2 += 1) { //#15743
        var _D5 = $k[--$j]; //#15743
        $k[$j++] = $f(_D5 + ((_D2 + 1) * $get($_.cws, _D2))); //#15743
    } //#15743
    $_.c2 = ($f($k[--$j] + ($_.c1 * ($_.dcws + 2)))) % 107; //#15744
    $k[$j++] = Infinity; //#15745
    $aload($_.cws); //#15745
    $k[$j++] = $_.c1; //#15745
    $k[$j++] = $_.c2; //#15745
    $_.cws = $a(); //#15745
    var _DE = $get($_.options, 'debugcws') !== undefined; //#15747
    if (_DE) { //#15747
        $k[$j++] = 'bwipp.debugcws#15747'; //#15747
        $k[$j++] = $_.cws; //#15747
        bwipp_raiseerror(); //#15747
    } //#15747
    $_.encs = $a(["212222", "222122", "222221", "121223", "121322", "131222", "122213", "122312", "132212", "221213", "221312", "231212", "112232", "122132", "122231", "113222", "123122", "123221", "223211", "221132", "221231", "213212", "223112", "312131", "311222", "321122", "321221", "312212", "322112", "322211", "212123", "212321", "232121", "111323", "131123", "131321", "112313", "132113", "132311", "211313", "231113", "231311", "112133", "112331", "132131", "113123", "113321", "133121", "313121", "211331", "231131", "213113", "213311", "213131", "311123", "311321", "331121", "312113", "312311", "332111", "314111", "221411", "431111", "111224", "111422", "121124", "121421", "141122", "141221", "112214", "112412", "122114", "122411", "142112", "142211", "241211", "221114", "413111", "241112", "134111", "111242", "121142", "121241", "114212", "124112", "124211", "411212", "421112", "421211", "212141", "214121", "412121", "111143", "111341", "131141", "114113", "114311", "411113", "411311", "113141", "114131", "311141", "411131", "211412", "211214", "211232", "211133"]); //#15768
    $_.startencs = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112"]); //#15774
    $_.stopencsodd = $a(["3211", "2221", "2122", "1411", "1132", "1231", "1114", "3112", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411"]); //#15778
    $_.stopencseven = $a(["2122", "1411", "1132", "1231", "1114", "3112", "1132", "1231", "1114", "3112", "3211", "2221", "2122", "1411", "3211", "2221"]); //#15782
    if (($_.sam == -1) || (((~~($_.sam / 10)) % 2) == 1)) { //#15788
        $_.stopencs = $_.stopencsodd; //#15786
    } else { //#15788
        $_.stopencs = $_.stopencseven; //#15788
    } //#15788
    $_.rowbits = $a($_.r); //#15792
    for (var _DT = 0, _DS = $_.r - 1; _DT <= _DS; _DT += 1) { //#15809
        $_.i = _DT; //#15794
        $k[$j++] = Infinity; //#15795
        $k[$j++] = 10; //#15797
        $forall($get($_.startencs, $_.i), function() { //#15797
            var _DX = $k[--$j]; //#15797
            $k[$j++] = _DX - 48; //#15797
        }); //#15797
        var _Da = $geti($_.cws, $_.i * 5, 5); //#15799
        $k[$j++] = 1; //#15801
        for (var _Db = 0, _Dc = _Da.length; _Db < _Dc; _Db++) { //#15801
            $forall($get($_.encs, $get(_Da, _Db)), function() { //#15800
                var _Dg = $k[--$j]; //#15800
                $k[$j++] = _Dg - 48; //#15800
            }); //#15800
        } //#15800
        $forall($get($_.stopencs, $_.i), function() { //#15802
            var _Dk = $k[--$j]; //#15802
            $k[$j++] = _Dk - 48; //#15802
        }); //#15802
        $k[$j++] = 1; //#15803
        $_.sbs = $a(); //#15803
        $k[$j++] = Infinity; //#15805
        var _Dm = $_.sbs; //#15806
        $k[$j++] = 1; //#15806
        for (var _Dn = 0, _Do = _Dm.length; _Dn < _Do; _Dn++) { //#15806
            var _Dq = $k[$j - 1]; //#15806
            var _Dr = (_Dq == 0) ? 1 : 0; //#15806
            for (var _Ds = 0, _Dt = $get(_Dm, _Dn); _Ds < _Dt; _Ds++) { //#15806
                $k[$j++] = _Dr //#15806
            } //#15806
        } //#15806
        $astore($a($counttomark() - 1)); //#15807
        var _Dw = $k[--$j]; //#15807
        $k[$j - 1] = _Dw; //#15807
        var _Dy = $k[--$j]; //#15807
        $k[$j - 1] = _Dy; //#15807
        $k[$j++] = $_.rowbits; //#15808
        $k[$j++] = $_.i; //#15808
        var _E2 = $k[$j - 3]; //#15808
        var _E3 = $k[$j - 1]; //#15808
        $put($k[$j - 2], _E3, _E2); //#15808
        $j -= 3; //#15808
    } //#15808
    $k[$j++] = Infinity; //#15812
    for (var _E6 = 0, _E7 = 81 * $_.sepheight; _E6 < _E7; _E6++) { //#15813
        $k[$j++] = 1; //#15813
    } //#15813
    for (var _EA = 0, _E9 = $_.r - 2; _EA <= _E9; _EA += 1) { //#15818
        $_.i = _EA; //#15815
        for (var _EC = 0, _ED = $_.rowheight; _EC < _ED; _EC++) { //#15816
            $aload($get($_.rowbits, $_.i)); //#15816
        } //#15816
        for (var _EI = 0, _EJ = $_.sepheight; _EI < _EJ; _EI++) { //#15817
            for (var _EK = 0; _EK < 10; _EK++) { //#15817
                $k[$j++] = 0; //#15817
            } //#15817
            for (var _EL = 0; _EL < 70; _EL++) { //#15817
                $k[$j++] = 1; //#15817
            } //#15817
            $k[$j++] = 0; //#15817
        } //#15817
    } //#15817
    for (var _EN = 0, _EO = $_.rowheight; _EN < _EO; _EN++) { //#15819
        $aload($get($_.rowbits, $_.r - 1)); //#15819
    } //#15819
    for (var _ET = 0, _EU = 81 * $_.sepheight; _ET < _EU; _ET++) { //#15820
        $k[$j++] = 1; //#15820
    } //#15820
    $_.pixs = $a(); //#15820
    var _Ea = new Map([
        ["ren", 'renmatrix'],
        ["pixs", $_.pixs],
        ["pixx", 81],
        ["pixy", ~~($_.pixs.length / 81)],
        ["height", (~~($_.pixs.length / 81)) / 72],
        ["width", 81 / 72],
        ["opt", $_.options]
    ]); //#15831
    $k[$j++] = _Ea; //#15834
    if (!$_.dontdraw) { //#15834
        bwipp_renmatrix(); //#15834
    } //#15834
    $_ = $__; //#15838
} //bwipp_code16k
function bwipp_code49() {
    var $__ = $_; //#15869
    $_ = Object.assign({}, $_); //#15869
    $_.dontdraw = false; //#15872
    $_.sam = -1; //#15873
    $_.append = false; //#15874
    $_.rows = 0; //#15875
    $_.rowheight = 8; //#15876
    $_.sepheight = 1; //#15877
    $_.parse = false; //#15878
    $_.parsefnc = false; //#15879
    $k[$j++] = $_; //#15881
    bwipp_processoptions(); //#15881
    $_.options = $k[--$j]; //#15881
    $_.barcode = $k[--$j]; //#15882
    if (($_.rows != 0) && (($_.rows < 2) || ($_.rows > 8))) { //#15886
        $k[$j++] = 'bwipp.code49badRows#15885'; //#15885
        $k[$j++] = "Code 49 must have 2 to 8 rows"; //#15885
        bwipp_raiseerror(); //#15885
    } //#15885
    if ($_.rowheight < 8) { //#15890
        $k[$j++] = 'bwipp.code49badRowHeight#15889'; //#15889
        $k[$j++] = "Code 49 must have rowheight of at least 8"; //#15889
        bwipp_raiseerror(); //#15889
    } //#15889
    if ($_.sepheight < 1) { //#15894
        $k[$j++] = 'bwipp.code49badSepHeight#15893'; //#15893
        $k[$j++] = "Code 49 must have sepheight of at least 1"; //#15893
        bwipp_raiseerror(); //#15893
    } //#15893
    if ($_.append && ($_.sam != -1)) { //#15898
        $k[$j++] = 'bwipp.code49samAndAppend#15897'; //#15897
        $k[$j++] = "sam and append cannot be specified together"; //#15897
        bwipp_raiseerror(); //#15897
    } //#15897
    if ($_.sam != -1) { //#15910
        $k[$j++] = true; //#15906
        for (var _A = 0; _A < 1; _A++) { //#15906
            var _B = $_.sam; //#15903
            if ((_B < 12) || (_B > 99)) { //#15903
                $k[$j - 1] = false; //#15903
                break; //#15903
            } //#15903
            if (($_.sam % 10) < 2) { //#15904
                $k[$j - 1] = false; //#15904
                break; //#15904
            } //#15904
            if ((~~($_.sam / 10)) > ($_.sam % 10)) { //#15905
                $k[$j - 1] = false; //#15905
                break; //#15905
            } //#15905
        } //#15905
        if (!$k[--$j]) { //#15909
            $k[$j++] = 'bwipp.code49badSAM#15908'; //#15908
            $k[$j++] = "SAM must be formatted as \"NM\" for Nth of M symbols, from 2 to 9 symbols"; //#15908
            bwipp_raiseerror(); //#15908
        } //#15908
    } //#15908
    $_.s1 = -1; //#15915
    $_.s2 = -2; //#15915
    $_.fn1 = -3; //#15915
    $_.fn2 = -4; //#15915
    $_.fn3 = -5; //#15915
    $_.ns = -6; //#15915
    var _M = new Map([
        ["parse", $_.parse],
        ["parsefnc", $_.parsefnc],
        ["FNC1", $_.fn1],
        ["FNC2", $_.fn2],
        ["FNC3", $_.fn3]
    ]); //#15924
    $_.fncvals = _M; //#15925
    $k[$j++] = 'msg'; //#15926
    $k[$j++] = $_.barcode; //#15926
    $k[$j++] = $_.fncvals; //#15926
    bwipp_parseinput(); //#15926
    var _P = $k[--$j]; //#15926
    $_[$k[--$j]] = _P; //#15926
    $_.msglen = $_.msg.length; //#15927
    $forall($_.msg, function() { //#15933
        if ($k[--$j] > 127) { //#15932
            $k[$j++] = 'bwipp.code49badCharacter#15931'; //#15931
            $k[$j++] = "Code 49 can only support ASCII characters with values 0 to 127"; //#15931
            bwipp_raiseerror(); //#15931
        } //#15931
    }); //#15931
    $_.charmap = $a(["0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "-", ".", " ", "$", "/", "+", "%", $_.s1, $_.s2, $_.fn1, $_.fn2, $_.fn3, $_.ns]); //#15943
    $_.charvals = new Map; //#15946
    for (var _b = 0; _b <= 48; _b += 1) { //#15952
        $_.i = _b; //#15948
        var _e = $get($_.charmap, $_.i); //#15949
        $k[$j++] = _e; //#15950
        if ($eq($type(_e), 'stringtype')) { //#15950
            var _h = $get($k[--$j], 0); //#15950
            $k[$j++] = _h; //#15950
        } //#15950
        $put($_.charvals, $k[--$j], $_.i); //#15951
    } //#15951
    $_.combos = $a(["1 ", "1A", "1B", "1C", "1D", "1E", "1F", "1G", "1H", "1I", "1J", "1K", "1L", "1M", "1N", "1O", "1P", "1Q", "1R", "1S", "1T", "1U", "1V", "1W", "1X", "1Y", "1Z", "11", "12", "13", "14", "15", "  ", "16", "17", "18", " $", " %", "19", "10", "1-", "1.", "1$", " +", "1/", " -", " .", " /", " 0", " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9", "1+", "21", "22", "23", "24", "25", "26", " A", " B", " C", " D", " E", " F", " G", " H", " I", " J", " K", " L", " M", " N", " O", " P", " Q", " R", " S", " T", " U", " V", " W", " X", " Y", " Z", "27", "28", "29", "20", "2-", "2.", "2A", "2B", "2C", "2D", "2E", "2F", "2G", "2H", "2I", "2J", "2K", "2L", "2M", "2N", "2O", "2P", "2Q", "2R", "2S", "2T", "2U", "2V", "2W", "2X", "2Y", "2Z", "2$", "2/", "2+", "2%", "2 "]); //#15964
    for (var _m = 0; _m <= 127; _m += 1) { //#15974
        $_.i = _m; //#15966
        var _p = $get($_.combos, $_.i); //#15967
        $_.c1 = $get(_p, 0); //#15967
        $_.c2 = $get(_p, 1); //#15967
        if ($_.c1 == 49) { //#15970
            $put($_.charvals, $_.i, $a([$get($_.charvals, $_.s1), $get($_.charvals, $_.c2)])); //#15969
        } //#15969
        if ($_.c1 == 50) { //#15973
            $put($_.charvals, $_.i, $a([$get($_.charvals, $_.s2), $get($_.charvals, $_.c2)])); //#15972
        } //#15972
    } //#15972
    $_.encodealpha = function() {
        var _1F = $get($_.charvals, $k[--$j]); //#15978
        $k[$j++] = _1F; //#15979
        if ($ne($type(_1F), 'arraytype')) { //#15979
            $astore($a(1)); //#15979
        } //#15979
        var _1I = $k[--$j]; //#15980
        $puti($_.cws, $_.j, _1I); //#15980
        $_.j = _1I.length + $_.j; //#15981
    }; //#15981
    $_.base48 = function() {
        var _1M = $k[--$j]; //#15985
        $k[$j++] = 0; //#15985
        $forall(_1M, function() { //#15985
            var _1N = $k[--$j]; //#15985
            var _1O = $k[--$j]; //#15985
            $k[$j++] = $f((_1N - 48) + (_1O * 10)); //#15985
        }); //#15985
        $k[$j++] = Infinity; //#15986
        var _1P = $k[$j - 2]; //#15986
        var _1Q = $k[$j - 3]; //#15986
        $k[$j - 3] = $k[$j - 1]; //#15986
        $j -= 2; //#15986
        $k[$j++] = _1P; //#15986
        for (var _1R = 0, _1S = _1Q - 1; _1R < _1S; _1R++) { //#15986
            var _1T = $k[--$j]; //#15986
            $k[$j++] = _1T % 48; //#15986
            $k[$j++] = ~~(_1T / 48); //#15986
        } //#15986
        var _1U = $a(); //#15986
        var _1V = Infinity; //#15987
        var _1W = _1U; //#15987
        $k[$j++] = _1V; //#15987
        $k[$j++] = _1W; //#15987
        for (var _1X = _1W.length - 1; _1X >= 0; _1X -= 1) { //#15987
            var _1Y = $k[--$j]; //#15987
            $k[$j++] = $get(_1Y, _1X); //#15987
            $k[$j++] = _1Y; //#15987
        } //#15987
        $j--; //#15987
        var _1a = $a(); //#15987
        $puti($_.cws, $_.j, _1a); //#15988
        $_.j = _1a.length + $_.j; //#15989
    }; //#15989
    $_.encodenumeric = function() {
        $_.nums = $k[--$j]; //#15993
        var _1g = $_.nums.length; //#15994
        var _1h = _1g % 5; //#15994
        $k[$j++] = 'pre'; //#15994
        $k[$j++] = _1g; //#15994
        $k[$j++] = _1h; //#15994
        if (_1h != 2) { //#15994
            var _1i = $k[--$j]; //#15994
            var _1j = $k[--$j]; //#15994
            $k[$j++] = $f(_1j - _1i); //#15994
        } else { //#15994
            var _1k = $k[--$j]; //#15994
            var _1l = $k[--$j]; //#15994
            $k[$j++] = $f(_1l - _1k) - 5; //#15994
        } //#15994
        var _1m = $k[--$j]; //#15994
        $_[$k[--$j]] = _1m; //#15994
        for (var _1q = 0, _1p = $_.pre - 1; _1q <= _1p; _1q += 5) { //#15995
            $k[$j++] = 3; //#15995
            $k[$j++] = $geti($_.nums, _1q, 5); //#15995
            $_.base48(); //#15995
        } //#15995
        $_.nums = $geti($_.nums, $_.pre, $f($_.nums.length - $_.pre)); //#15996
        var _1z = $_.nums.length; //#15998
        $k[$j++] = _1z; //#15998
        if (_1z == 1) { //#15998
            $k[$j++] = $get($_.nums, $_.i); //#15998
            $_.encodealpha(); //#15998
        } //#15998
        var _23 = $k[$j - 1]; //#15999
        if (_23 == 3) { //#15999
            $k[$j++] = 2; //#15999
            $k[$j++] = $_.nums; //#15999
            $_.base48(); //#15999
        } //#15999
        var _25 = $k[$j - 1]; //#16000
        if (_25 == 4) { //#16000
            $k[$j++] = 3; //#16000
            $k[$j++] = Infinity; //#16000
            $k[$j++] = 49; //#16000
            $k[$j++] = 48; //#16000
            $aload($_.nums); //#16000
            var _27 = $a(); //#16000
            $k[$j++] = _27; //#16000
            $_.base48(); //#16000
        } //#16000
        if ($k[--$j] == 7) { //#16004
            $k[$j++] = 3; //#16002
            $k[$j++] = Infinity; //#16002
            $k[$j++] = 49; //#16002
            $k[$j++] = 48; //#16002
            $aload($geti($_.nums, 0, 4)); //#16002
            var _2B = $a(); //#16002
            $k[$j++] = _2B; //#16002
            $_.base48(); //#16002
            $k[$j++] = 2; //#16003
            $k[$j++] = $geti($_.nums, 4, 3); //#16003
            $_.base48(); //#16003
        } //#16003
    }; //#16003
    $k[$j++] = Infinity; //#16008
    for (var _2F = 0, _2G = $_.msglen; _2F < _2G; _2F++) { //#16008
        $k[$j++] = 0; //#16008
    } //#16008
    $k[$j++] = 0; //#16008
    $_.numericruns = $a(); //#16008
    for (var _2J = $_.msglen - 1; _2J >= 0; _2J -= 1) { //#16017
        $_.i = _2J; //#16010
        var _2M = $get($_.msg, $_.i); //#16011
        if ((_2M >= 48) && (_2M <= 57)) { //#16015
            $put($_.numericruns, $_.i, $get($_.numericruns, $_.i + 1) + 1); //#16013
        } else { //#16015
            $put($_.numericruns, $_.i, 0); //#16015
        } //#16015
    } //#16015
    $_.cws = $a(($_.msglen * 2) + 1); //#16019
    $_.samval = $a([12, 22, 13, 23, 33, 14, 24, 34, 44, 15, 25, 35, 45, 55, 16, 26, 36, 46, 56, 66, 17, 27, 37, 47, 57, 67, 77, 18, 28, 38, 48, 58, 68, 78, 88, 19, 29, 39, 49, 59, 69, 79, 89, 99]); //#16031
    for (;;) { //#16068
        if ($_.append) { //#16041
            $_.mode = 1; //#16037
            $_.method = "alpha"; //#16038
            $_.i = 0; //#16039
            $_.j = 0; //#16039
            break; //#16040
        } //#16040
        if ($_.sam != -1) { //#16048
            $_.mode = 3; //#16043
            $k[$j++] = $_.cws; //#16044
            $k[$j++] = 0; //#16044
            for (var _2b = 0; _2b <= 43; _2b += 1) { //#16044
                $k[$j++] = _2b; //#16044
                if ($get($_.samval, _2b) != $_.sam) { //#16044
                    $j--; //#16044
                } //#16044
            } //#16044
            var _2f = $k[--$j]; //#16044
            var _2g = $k[--$j]; //#16044
            $put($k[--$j], _2g, _2f + 1); //#16044
            $_.method = "alpha"; //#16045
            $_.i = 0; //#16046
            $_.j = 1; //#16046
            break; //#16047
        } //#16047
        if ($get($_.numericruns, 0) >= 5) { //#16053
            $_.mode = 2; //#16050
            $_.method = "numeric"; //#16051
            $_.i = 0; //#16051
            $_.j = 0; //#16051
            break; //#16052
        } //#16052
        var _2n = $get($_.charvals, $get($_.msg, 0)); //#16054
        $k[$j++] = _2n; //#16061
        if ($ne($type(_2n), 'arraytype')) { //#16061
            $j--; //#16056
            $_.mode = 0; //#16057
            $_.method = "alpha"; //#16058
            $_.i = 0; //#16059
            $_.j = 0; //#16059
            break; //#16060
        } //#16060
        var _2r = ($get($k[--$j], 0) == 43) ? 4 : 5; //#16063
        $_.mode = _2r; //#16063
        $put($_.cws, 0, $get($get($_.charvals, $get($_.msg, 0)), 1)); //#16064
        $_.method = "alpha"; //#16065
        $_.i = 1; //#16066
        $_.j = 1; //#16066
        break; //#16067
    } //#16067
    for (;;) { //#16095
        if ($_.i == $_.msglen) { //#16072
            break; //#16072
        } //#16072
        for (;;) { //#16094
            if ($eq($_.method, "alpha")) { //#16083
                if ($get($_.numericruns, $_.i) >= 5) { //#16079
                    $k[$j++] = $_.ns; //#16076
                    $_.encodealpha(); //#16076
                    $_.method = "numeric"; //#16077
                    break; //#16078
                } //#16078
                $k[$j++] = $get($_.msg, $_.i); //#16080
                $_.encodealpha(); //#16080
                $_.i = $_.i + 1; //#16081
                break; //#16082
            } //#16082
            if ($eq($_.method, "numeric")) { //#16093
                if ($get($_.numericruns, $_.i) < 5) { //#16089
                    $k[$j++] = $_.ns; //#16086
                    $_.encodealpha(); //#16086
                    $_.method = "alpha"; //#16087
                    break; //#16088
                } //#16088
                $k[$j++] = $geti($_.msg, $_.i, $get($_.numericruns, $_.i)); //#16090
                $_.encodenumeric(); //#16090
                $_.i = $f($_.i + $get($_.numericruns, $_.i)); //#16091
                break; //#16092
            } //#16092
        } //#16092
    } //#16092
    $_.cws = $geti($_.cws, 0, $_.j); //#16096
    $_.metrics = $a([$a([2, 9]), $a([3, 16]), $a([4, 23]), $a([5, 30]), $a([6, 37]), $a([7, 42]), $a([8, 49])]); //#16109
    $_.urows = $_.rows; //#16113
    var _3b = $_.metrics; //#16114
    for (var _3c = 0, _3d = _3b.length; _3c < _3d; _3c++) { //#16122
        $_.m = $get(_3b, _3c); //#16115
        $_.r = $get($_.m, 0); //#16116
        $_.dcws = $get($_.m, 1); //#16117
        $_.okay = true; //#16118
        if (($_.urows != 0) && ($_.urows != $_.r)) { //#16119
            $_.okay = false; //#16119
        } //#16119
        if ($_.cws.length > $_.dcws) { //#16120
            $_.okay = false; //#16120
        } //#16120
        if ($_.okay) { //#16121
            break; //#16121
        } //#16121
    } //#16121
    if (!$_.okay) { //#16126
        $k[$j++] = 'bwipp.code49noValidSymbol#16125'; //#16125
        $k[$j++] = "Maximum length exceeded"; //#16125
        bwipp_raiseerror(); //#16125
    } //#16125
    $k[$j++] = Infinity; //#16129
    $aload($_.cws); //#16129
    for (var _3t = 0, _3u = $f($_.dcws - $_.cws.length); _3t < _3u; _3t++) { //#16129
        $k[$j++] = 48; //#16129
    } //#16129
    $_.cws = $a(); //#16129
    $_.ccs = $a($_.r * 8); //#16132
    $_.j = 0; //#16133
    for (var _40 = 0, _3z = $_.r - 2; _40 <= _3z; _40 += 1) { //#16140
        $_.i = _40; //#16135
        $_.cc = $geti($_.cws, $_.j, 7); //#16136
        $puti($_.ccs, $_.i * 8, $_.cc); //#16137
        var _49 = $_.cc; //#16138
        $k[$j++] = $_.ccs; //#16138
        $k[$j++] = ($_.i * 8) + 7; //#16138
        $k[$j++] = 0; //#16138
        for (var _4A = 0, _4B = _49.length; _4A < _4B; _4A++) { //#16138
            var _4D = $k[--$j]; //#16138
            $k[$j++] = $f(_4D + $get(_49, _4A)); //#16138
        } //#16138
        var _4E = $k[--$j]; //#16138
        var _4F = $k[--$j]; //#16138
        $put($k[--$j], _4F, _4E % 49); //#16138
        $_.j = $_.j + 7; //#16139
    } //#16139
    if ($_.j < $_.dcws) { //#16143
        $puti($_.ccs, $_.ccs.length - 8, $geti($_.cws, $_.j, $f($_.dcws - $_.j))); //#16142
    } //#16142
    $_.cr7 = $f((($_.r - 2) * 7) + $_.mode); //#16146
    $put($_.ccs, $_.ccs.length - 2, $_.cr7); //#16147
    var _4W = $a([1, 9, 31, 26, 2, 12, 17, 23, 37, 18, 22, 6, 27, 44, 15, 43, 39, 11, 13, 5, 41, 33, 36, 8, 4, 32, 3, 19, 40, 25, 29, 10, 24, 30]); //#16152
    $k[$j++] = _4W; //#16153
    $k[$j++] = _4W; //#16153
    $k[$j++] = Infinity; //#16153
    var _4X = $k[--$j]; //#16153
    var _4Z = $geti($k[--$j], 0, 32); //#16153
    $k[$j++] = _4X; //#16153
    $k[$j++] = 20; //#16153
    $aload(_4Z); //#16153
    $_.weightx = $a(); //#16153
    var _4b = $k[$j - 1]; //#16154
    $k[$j++] = _4b; //#16154
    $k[$j++] = Infinity; //#16154
    var _4c = $k[--$j]; //#16154
    var _4e = $geti($k[--$j], 1, 32); //#16154
    $k[$j++] = _4c; //#16154
    $k[$j++] = 16; //#16154
    $aload(_4e); //#16154
    $_.weighty = $a(); //#16154
    $k[$j++] = Infinity; //#16155
    var _4g = $k[--$j]; //#16155
    var _4i = $geti($k[--$j], 2, 32); //#16155
    $k[$j++] = _4g; //#16155
    $k[$j++] = 38; //#16155
    $aload(_4i); //#16155
    $_.weightz = $a(); //#16155
    $_.calccheck = function() {
        $_.weights = $k[--$j]; //#16158
        $_.score = 0; //#16159
        for (var _4o = 0, _4n = (~~((($_.r - 1) * 8) / 2)) - 1; _4o <= _4n; _4o += 1) { //#16164
            $_.i = _4o; //#16161
            $_.score = $f(($get($_.ccs, $_.i * 2) * 49 + $get($_.ccs, $_.i * 2 + 1)) * $get($_.weights, $_.i + 1) + $_.score); //#16163
        } //#16163
        $k[$j++] = $_.score; //#16165
    }; //#16165
    $_.lastrow = $geti($_.ccs, $_.ccs.length - 8, 8); //#16167
    if ($_.r >= 7) { //#16174
        $k[$j++] = $_.cr7 * $get($_.weightz, 0); //#16171
        $k[$j++] = $_.weightz; //#16171
        $_.calccheck(); //#16171
        var _58 = $k[--$j]; //#16171
        var _5A = $f($k[--$j] + _58) % 2401; //#16172
        $k[$j++] = ~~(_5A / 49); //#16172
        $k[$j++] = _5A % 49; //#16172
        $astore($a(2)); //#16172
        $k[$j++] = $_.lastrow; //#16173
        $k[$j++] = 0; //#16173
        var _5D = $k[$j - 3]; //#16173
        var _5E = $k[$j - 1]; //#16173
        $puti($k[$j - 2], _5E, _5D); //#16173
        $j -= 3; //#16173
    } //#16173
    $_.wr1 = $f(($get($_.lastrow, 0) * 49) + $get($_.lastrow, 1)); //#16175
    $k[$j++] = $_.cr7 * $get($_.weighty, 0); //#16178
    $k[$j++] = $_.weighty; //#16178
    $_.calccheck(); //#16178
    var _5O = $k[--$j]; //#16178
    var _5U = $f($k[--$j] + _5O + $_.wr1 * $get($_.weighty, $_.r * 4 - 3)) % 2401;; //#16180
    $_.wr2 = _5U; //#16180
    $k[$j++] = ~~(_5U / 49); //#16181
    $k[$j++] = _5U % 49; //#16181
    $astore($a(2)); //#16181
    $k[$j++] = $_.lastrow; //#16182
    $k[$j++] = 2; //#16182
    var _5X = $k[$j - 3]; //#16182
    var _5Y = $k[$j - 1]; //#16182
    $puti($k[$j - 2], _5Y, _5X); //#16182
    $j -= 3; //#16182
    $k[$j++] = $_.cr7 * $get($_.weightx, 0); //#16185
    $k[$j++] = $_.weightx; //#16185
    $_.calccheck(); //#16185
    var _5e = $k[--$j]; //#16185
    var _5o = $f($k[--$j] + _5e + $_.wr1 * $get($_.weightx, $_.r * 4 - 3) + $_.wr2 * $get($_.weightx, $_.r * 4 - 2)) % 2401;; //#16189
    $k[$j++] = ~~(_5o / 49); //#16189
    $k[$j++] = _5o % 49; //#16189
    $astore($a(2)); //#16189
    $k[$j++] = $_.lastrow; //#16190
    $k[$j++] = 4; //#16190
    var _5r = $k[$j - 3]; //#16190
    var _5s = $k[$j - 1]; //#16190
    $puti($k[$j - 2], _5s, _5r); //#16190
    $j -= 3; //#16190
    var _5w = $geti($_.ccs, $_.ccs.length - 8, 7); //#16193
    $k[$j++] = 0; //#16193
    for (var _5x = 0, _5y = _5w.length; _5x < _5y; _5x++) { //#16193
        var _60 = $k[--$j]; //#16193
        $k[$j++] = $f(_60 + $get(_5w, _5x)); //#16193
    } //#16193
    $put($_.ccs, $_.ccs.length - 1, $k[--$j] % 49); //#16194
    var _65 = $get($_.options, 'debugcws') !== undefined; //#16196
    if (_65) { //#16196
        $k[$j++] = 'bwipp.debugcws#16196'; //#16196
        $k[$j++] = $_.ccs; //#16196
        bwipp_raiseerror(); //#16196
    } //#16196
    $_.patterns = $a([$a(["11521132", "25112131", "14212132", "25121221", "14221222", "12412132", "23321221", "12421222", "21521221", "15112222", "15121312", "13312222", "24221311", "13321312", "11512222", "22421311", "11521312", "25112311", "14212312", "23312311", "12412312", "21512311", "16121131", "14321131", "12521131", "15212131", "15221221", "13412131", "13421221", "11612131", "16112221", "16121311", "14312221", "14321311", "12512221", "12521311", "15212311", "13412311", "11612311", "11131135", "31131133", "51131131", "21122134", "41122132", "21131224", "41131222", "11113135", "31113133", "51113131", "11122225", "31122223", "51122221", "11131315", "31131313", "51131311", "21113224", "41113222", "21122314", "41122312", "11113315", "31113313", "51113311", "12131134", "32131132", "21231133", "41231131", "22122133", "42122131", "11222134", "22131223", "42131221", "11231224", "31231222", "12113134", "32113132", "12122224", "32122222", "12131314", "32131312", "21231313", "41231311", "22113223", "42113221", "11213224", "22122313", "42122311", "11222314", "31222312", "12113314", "32113312", "21213313", "41213311", "13131133", "33131131", "22231132", "11331133", "31331131", "23122132", "12222133", "23131222", "12231223", "32231221", "21331222", "13113133", "33113131", "13122223", "33122221", "11313133", "13131313", "33131311", "11322223", "22231312", "11331313", "31331311", "23113222", "12213223", "23122312", "12222313", "32222311", "21322312", "13113313", "33113311", "22213312", "11313313", "31313311", "14131132", "23231131", "12331132", "21431131", "24122131", "13222132", "24131221", "13231222", "11422132", "22331221", "11431222", "14113132", "14122222", "12313132", "14131312", "12322222", "23231311", "12331312", "21431311", "24113221", "13213222", "24122311", "13222312", "11413222", "22322311", "11422312", "14113312", "23213311", "12313312", "21413311", "15131131", "13331131", "14222131", "14231221", "12422131", "12431221", "15113131", "15122221", "13313131", "15131311", "13322221", "11513131", "13331311", "11522221", "14213221", "14222311", "12413221", "12422311", "15113311", "13313311", "11513311", "11141134", "31141132", "21132133", "41132131", "21141223", "41141221", "11123134", "31123132", "11132224", "31132222", "11141314", "31141312", "21114133", "41114131", "21123223", "41123221", "21132313", "41132311", "11114224", "31114222", "11123314", "31123312", "21114313", "41114311", "12141133", "32141131", "21241132", "22132132", "11232133", "22141222", "11241223", "31241221", "12123133", "32123131", "12132223", "32132221", "12141313", "32141311", "21241312", "22114132", "11214133", "22123222", "11223223", "22132312", "11232313", "31232311", "12114223", "32114221", "12123313", "32123311", "21223312", "22114312", "11214313", "31214311", "13141132", "22241131", "11341132", "23132131", "12232132", "23141221", "12241222", "21341221", "13123132", "13132222", "11323132", "13141312", "11332222", "22241311", "11341312", "23114131", "12214132", "23123221", "12223222", "23132311", "12232312", "21332311", "13114222", "13123312", "11314222", "22223311", "11323312", "23114311", "12214312", "21314311", "14141131", "12341131", "13232131", "13241221", "11432131", "14123131", "14132221", "12323131", "14141311", "12332221", "12341311", "13214131", "13223221", "11414131", "13232311", "11423221", "11432311", "14114221", "14123311", "12314221", "12323311", "13214311", "11414311", "11151133", "31151131", "21142132", "21151222", "11133133", "31133131", "11142223", "31142221", "11151313", "31151311", "21124132", "21133222", "21142312", "11115133", "31115131", "11124223", "31124221", "11133313", "31133311", "21115222", "21124312", "12151132", "21251131", "22142131", "11242132", "22151221", "11251222", "12133132", "12142222", "12151312", "21251311", "22124131", "11224132", "22133221", "11233222", "22142311", "11242312", "12115132", "12124222", "12133312", "21233311", "22115221", "11215222", "22124311", "11224312", "13151131", "12242131", "12251221", "13133131", "13142221", "11333131", "13151311", "11342221", "12224131", "12233221", "12242311", "13115131", "13124221", "11315131", "13133311", "11324221", "11333311", "12215221", "12224311", "11161132", "21152131", "21161221", "11143132", "11152222", "11161312", "21134131", "21143221", "21152311", "11125132", "11134222", "11143312", "21116131", "21125221", "21134311", "12161131", "11252131", "12143131", "12152221", "12161311", "11234131", "11243221", "11252311", "12125131", "12134221", "12143311", "11216131", "11225221", "11234311", "11111236", "31111234", "51111232", "21111325", "41111323", "61111321", "11111416", "31111414", "51111412", "31211143", "51211141", "12111235", "32111233", "52111231", "21211234", "41211232", "22111324", "42111322", "11211325", "31211323", "51211321", "12111415", "32111413", "52111411", "21211414", "41211412", "12211144", "32211142", "21311143", "41311141", "13111234", "33111232", "22211233", "42211231", "11311234", "31311232", "23111323", "43111321", "12211324", "32211322", "21311323", "41311321", "13111414", "33111412", "22211413", "42211411", "11311414", "31311412", "13211143", "33211141", "22311142", "11411143", "31411141", "14111233", "34111231", "23211232", "12311233", "32311231", "21411232", "24111322", "13211323", "33211321", "22311322", "11411323", "31411321", "14111413", "34111411", "23211412", "12311413", "32311411", "21411412", "14211142", "23311141", "12411142", "21511141", "15111232", "24211231", "13311232", "22411231", "11511232", "25111321", "14211322", "23311321", "12411322", "21511321", "15111412", "24211411", "13311412", "22411411", "11511412", "15211141", "13411141", "11611141", "16111231", "14311231", "12511231", "15211321", "13411321", "11611321", "16111411", "14311411", "12511411", "21121144", "41121142", "11112145", "31112143", "51112141", "11121235", "31121233", "51121231", "21112234", "41112232", "21121324", "41121322", "11112325", "31112323", "51112321", "11121415", "31121413", "51121411", "21112414", "41112412", "22121143", "42121141", "11221144", "31221142", "12112144", "32112142", "12121234", "32121232", "21221233", "41221231", "22112233", "42112231", "11212234", "22121323", "42121321", "11221324", "31221322", "12112324", "32112322", "12121414", "32121412", "21221413", "41221411", "22112413", "42112411", "11212414", "31212412", "23121142", "12221143", "32221141", "21321142", "13112143", "33112141", "13121233", "33121231", "11312143", "22221232", "11321233", "31321231", "23112232", "12212233", "23121322", "12221323", "32221321", "21321322", "13112323", "33112321", "13121413", "33121411", "11312323", "22221412", "11321413", "31321411", "23112412", "12212413", "32212411", "21312412", "24121141", "13221142", "22321141", "11421142", "14112142", "14121232", "12312142", "23221231", "12321232", "21421231", "24112231", "13212232", "24121321", "13221322", "11412232", "22321321", "11421322", "14112322", "14121412", "12312322", "23221411", "12321412", "21421411", "24112411", "13212412", "22312411", "11412412", "14221141", "12421141", "15112141", "15121231", "13312141", "13321231", "11512141", "11521231", "14212231", "14221321", "12412231", "12421321", "15112321", "15121411", "13312321", "13321411", "11512321", "11521411", "14212411", "12412411", "21131143", "41131141", "11122144", "31122142", "11131234", "31131232", "21113143", "41113141", "21122233", "41122231", "21131323", "41131321", "11113234", "31113232", "11122324", "31122322", "11131414", "31131412", "21113323", "41113321", "21122413", "41122411", "11113414", "31113412", "22131142", "11231143", "31231141", "12122143", "32122141", "12131233", "32131231", "21231232", "22113142", "11213143", "22122232", "11222233", "22131322", "11231323", "31231321", "12113233", "32113231", "12122323", "32122321", "12131413", "32131411", "21231412", "22113322", "11213323", "22122412", "11222413", "31222411", "12113413", "32113411", "21213412", "23131141", "12231142", "21331141", "13122142", "13131232", "11322142", "22231231", "11331232", "23113141", "12213142", "23122231", "12222232", "23131321", "12231322", "21331321", "13113232", "13122322", "11313232", "13131412", "11322322", "22231411", "11331412", "23113321", "12213322", "23122411", "12222412", "21322411", "13113412", "22213411", "11313412", "13231141", "11431141", "14122141", "14131231", "12322141", "12331231", "13213141", "13222231", "11413141", "13231321", "11422231", "11431321", "14113231", "14122321", "12313231", "14131411", "12322321", "12331411", "13213321", "13222411", "11413321", "11422411", "14113411", "12313411", "21141142", "11132143", "31132141", "11141233", "31141231", "21123142", "21132232", "21141322", "11114143", "31114141", "11123233", "31123231", "11132323", "31132321", "11141413", "31141411", "21114232", "21123322", "21132412", "11114323", "31114321", "11123413", "31123411", "22141141", "11241142", "12132142", "12141232", "21241231", "22123141", "11223142", "22132231", "11232232", "22141321", "11241322", "12114142", "12123232", "12132322", "12141412", "21241411", "22114231", "11214232", "22123321", "11223322", "22132411", "11232412", "12114322", "12123412", "21223411", "12241141", "13132141", "13141231", "11332141", "11341231", "12223141", "12232231", "12241321", "13114141", "13123231", "11314141", "13132321", "11323231", "13141411", "11332321", "11341411", "12214231", "12223321", "12232411", "13114321", "13123411", "11314321", "11323411", "21151141", "11142142", "11151232", "21133141", "21142231", "21151321", "11124142", "11133232", "11142322", "11151412", "21115141", "21124231", "21133321", "21142411", "11115232", "11124322", "11133412", "11251141", "12142141", "12151231", "11233141", "11242231", "11251321", "12124141", "12133231", "12142321", "12151411", "11215141", "11224231", "11233321", "11242411", "12115231", "12124321", "12133411", "11152141", "11161231", "11134141", "11143231", "11152321", "11161411", "11116141", "11125231", "11134321", "11143411", "21111244", "41111242", "11111335", "31111333", "51111331", "21111424", "41111422", "11111515", "31111513", "51111511", "21211153", "41211151", "22111243", "42111241", "11211244", "31211242", "12111334", "32111332", "21211333", "41211331", "22111423", "42111421", "11211424", "31211422", "12111514", "32111512", "21211513", "41211511", "22211152", "11311153", "31311151", "23111242", "12211243", "32211241", "21311242", "13111333", "33111331", "22211332", "11311333", "31311331", "23111422", "12211423", "32211421", "21311422", "13111513", "33111511", "22211512", "11311513", "31311511", "23211151", "12311152", "21411151", "24111241", "13211242", "22311241", "11411242", "14111332", "23211331", "12311332", "21411331", "24111421", "13211422", "22311421", "11411422", "14111512", "23211511", "12311512", "21411511", "13311151", "11511151", "14211241", "12411241", "15111331", "13311331", "11511331", "14211421", "12411421", "15111511", "13311511", "11511511", "31121152", "21112153", "41112151", "21121243", "41121241", "11112244", "31112242", "11121334", "31121332", "21112333", "41112331", "21121423", "41121421", "11112424", "31112422", "11121514", "31121512", "21112513", "41112511", "12121153", "32121151", "21221152", "22112152", "11212153", "22121242", "11221243", "31221241", "12112243", "32112241", "12121333", "32121331", "21221332", "22112332", "11212333", "22121422", "11221423", "31221421", "12112423", "32112421", "12121513", "32121511", "21221512", "22112512", "11212513", "31212511", "13121152", "22221151", "11321152", "23112151", "12212152", "23121241", "12221242", "21321241", "13112242", "13121332", "11312242", "22221331", "11321332", "23112331", "12212332", "23121421", "12221422", "21321421", "13112422", "13121512", "11312422", "22221511", "11321512", "23112511", "12212512", "21312511", "14121151", "12321151", "13212151", "13221241", "11412151", "11421241", "14112241", "14121331", "12312241", "12321331", "13212331", "13221421", "11412331", "11421421", "14112421", "14121511", "12312421", "12321511", "13212511", "11412511", "11131153", "31131151", "21122152", "21131242", "11113153", "31113151", "11122243", "31122241", "11131333", "31131331", "21113242", "21122332", "21131422", "11113333", "31113331", "11122423", "31122421", "11131513", "31131511", "21113422", "21122512", "12131152", "21231151", "22122151", "11222152", "22131241", "11231242", "12113152", "12122242", "12131332", "21231331", "22113241", "11213242", "22122331", "11222332", "22131421", "11231422", "12113332", "12122422", "12131512", "21231511", "22113421", "11213422", "22122511", "11222512", "13131151", "11331151", "12222151", "12231241", "13113151", "13122241", "11313151", "13131331", "11322241", "11331331", "12213241", "12222331", "12231421", "13113331", "13122421", "11313331", "13131511", "11322421", "11331511", "12213421", "12222511", "11141152", "21132151", "21141241", "11123152", "11132242", "11141332", "21114151", "21123241", "21132331", "21141421", "11114242", "11123332", "11132422", "11141512", "21114331", "21123421", "21132511", "12141151", "11232151", "11241241", "12123151", "12132241", "12141331", "11214151", "11223241", "11232331", "11241421", "12114241", "12123331", "12132421", "12141511", "11214331", "11223421", "11232511", "11151151", "11133151", "11142241", "11151331", "11115151", "11124241", "11133331", "11142421", "11151511", "11111254", "31111252", "21111343", "41111341", "11111434", "31111432", "21111523", "41111521", "11111614", "31111612", "31211161", "12111253", "32111251", "21211252", "22111342", "11211343", "31211341", "12111433", "32111431", "21211432", "22111522", "11211523", "31211521", "12111613", "32111611", "21211612", "12211162", "21311161", "13111252", "22211251", "11311252", "23111341", "12211342", "21311341", "13111432", "22211431", "11311432", "23111521", "12211522", "21311521", "13111612", "22211611", "11311612", "13211161", "11411161", "14111251", "12311251", "13211341", "11411341", "14111431", "12311431", "13211521", "11411521", "14111611", "12311611", "21121162", "11112163", "31112161", "11121253", "31121251", "21112252", "21121342", "11112343", "31112341", "11121433", "31121431", "21112432", "21121522", "11112523", "31112521", "11121613", "31121611", "22121161", "11221162", "12112162", "12121252", "21221251", "22112251", "11212252", "22121341", "11221342", "12112342", "12121432", "21221431", "22112431", "11212432", "22121521", "11221522", "12112522", "12121612", "21221611", "12221161", "13112161", "13121251", "11312161", "11321251", "32121115", "52121113", "21221116", "41221114", "61221112", "22112116", "42112114", "31212115", "51212113", "13121116", "33121114", "22221115", "42221113", "11321116", "31321114", "51321112", "23112115", "43112113", "12212116", "32212114", "52212112", "21312115", "41312113", "61312111", "14121115", "34121113", "23221114", "43221112", "12321115", "32321113", "52321111", "21421114", "41421112", "24112114", "13212115", "33212113", "22312114", "42312112", "11412115", "31412113", "51412111", "15121114", "24221113", "13321114", "33321112", "22421113", "42421111", "11521114", "31521112", "25112113", "14212114", "34212112", "23312113", "43312111", "12412114", "32412112", "21512113", "41512111", "16121113", "25221112", "14321113", "34321111", "23421112", "12521113", "32521111", "15212113", "24312112", "13412113", "33412111", "22512112", "11612113", "31612111", "31131115", "51131113", "21122116", "41122114", "61122112", "31113115", "51113113", "12131116", "32131114", "52131112", "21231115", "41231113", "61231111", "22122115", "42122113", "11222116", "31222114", "51222112", "12113116", "32113114", "52113112", "21213115", "41213113", "61213111", "13131115", "33131113", "22231114", "42231112", "11331115", "31331113", "51331111", "23122114", "43122112", "12222115", "32222113", "52222111", "21322114", "41322112", "13113115", "33113113", "22213114", "42213112", "11313115", "31313113", "51313111", "14131114", "34131112", "23231113", "43231111", "12331114", "32331112", "21431113", "41431111", "24122113", "13222114", "33222112", "22322113", "42322111", "11422114", "31422112", "14113114", "34113112", "23213113", "43213111", "12313114", "32313112", "21413113", "41413111", "15131113", "24231112", "13331113", "33331111", "22431112", "25122112", "14222113", "34222111", "23322112", "12422113", "32422111", "21522112", "15113113", "24213112", "13313113", "33313111", "22413112", "11513113", "31513111", "16131112", "25231111", "14331112", "23431111", "15222112", "24322111", "13422112", "22522111", "16113112", "25213111", "14313112", "23413111", "12513112", "21613111", "11141116", "31141114", "51141112", "21132115", "41132113", "61132111", "11123116", "31123114", "51123112", "21114115", "41114113", "61114111", "12141115", "32141113", "52141111", "21241114", "41241112", "22132114", "42132112", "11232115", "31232113", "51232111", "12123115", "32123113", "52123111", "21223114", "41223112", "22114114", "42114112", "11214115", "31214113", "51214111", "13141114", "33141112", "22241113", "42241111", "11341114", "31341112", "23132113", "43132111", "12232114", "32232112", "21332113", "41332111", "13123114", "33123112", "22223113", "42223111", "11323114", "31323112", "23114113", "43114111", "12214114", "32214112", "21314113", "41314111", "14141113", "34141111", "23241112", "12341113", "32341111", "24132112", "13232113", "33232111", "22332112", "11432113", "31432111", "14123113", "34123111", "23223112", "12323113", "32323111", "21423112", "24114112", "13214113", "33214111", "22314112", "11414113", "31414111", "15141112", "24241111", "13341112", "25132111", "14232112", "23332111", "12432112", "15123112", "24223111", "13323112", "22423111", "11523112", "25114111", "14214112", "23314111", "12414112", "21514111", "16141111", "14341111", "15232111", "13432111", "16123111", "14323111", "12523111", "15214111", "13414111", "11614111", "11151115", "31151113", "51151111", "21142114", "41142112", "11133115", "31133113", "51133111", "21124114", "41124112", "11115115", "31115113", "51115111", "12151114", "32151112", "21251113", "41251111", "22142113", "42142111", "11242114", "31242112", "12133114", "32133112", "21233113", "41233111", "22124113", "42124111", "11224114", "31224112", "12115114", "32115112", "21215113", "41215111", "13151113", "33151111", "22251112", "23142112", "12242113", "32242111", "21342112", "13133113", "33133111", "22233112", "11333113", "31333111", "23124112", "12224113", "32224111", "21324112", "13115113", "33115111", "22215112", "11315113", "31315111", "14151112", "23251111", "24142111", "13242112", "22342111", "14133112", "23233111", "12333112", "21433111", "24124111", "13224112", "22324111", "11424112", "14115112", "23215111", "12315112", "21415111", "15151111", "14242111", "15133111", "13333111", "14224111", "12424111", "15115111", "13315111", "11515111", "11161114", "31161112", "21152113", "41152111", "11143114", "31143112", "21134113", "41134111", "11125114", "31125112", "21116113", "41116111", "12161113", "32161111", "22152112", "11252113", "31252111", "12143113", "32143111", "21243112", "22134112", "11234113", "31234111", "12125113", "32125111", "21225112", "22116112", "11216113", "31216111", "13161112", "23152111", "12252112", "13143112", "22243111", "11343112", "23134111", "12234112", "21334111", "13125112", "22225111", "11325112", "23116111", "12216112", "21316111", "14161111", "13252111", "14143111", "12343111", "13234111", "11434111", "14125111", "12325111", "13216111", "11416111", "31111216", "51111214", "31211125", "51211123", "32111215", "52111213", "21211216", "41211214", "61211212", "12211126", "32211124", "52211122", "21311125", "41311123", "61311121", "13111216", "33111214", "22211215", "42211213", "11311216", "31311214", "51311212", "13211125", "33211123", "22311124", "42311122", "11411125", "31411123", "51411121", "14111215", "34111213", "23211214", "43211212", "12311215", "32311213", "52311211", "21411214", "41411212", "14211124", "34211122", "23311123", "43311121", "12411124", "32411122", "21511123", "41511121", "15111214", "24211213", "13311214", "33311212", "22411213", "42411211", "11511214", "31511212", "15211123", "24311122", "13411123", "33411121", "22511122", "11611123", "31611121", "16111213", "25211212", "14311213", "34311211", "23411212", "12511213", "32511211", "21611212", "21121126", "41121124", "61121122", "31112125", "51112123", "31121215", "51121213", "21112216", "41112214", "61112212", "22121125", "42121123", "11221126", "31221124", "51221122", "12112126", "32112124", "52112122", "12121216", "32121214", "52121212", "21221215", "41221213", "61221211", "22112215", "42112213", "11212216", "31212214", "51212212", "23121124", "43121122", "12221125", "32221123", "52221121", "21321124", "41321122", "13112125", "33112123", "13121215", "33121213", "11312125", "22221214", "42221212", "11321215", "31321213", "51321211", "23112214", "43112212", "12212215", "32212213", "52212211", "21312214", "41312212", "24121123", "13221124", "33221122", "22321123", "42321121", "11421124", "31421122", "14112124", "34112122", "14121214", "34121212", "12312124", "23221213", "43221211", "12321214", "32321212", "21421213", "41421211", "24112213", "13212214", "33212212", "22312213", "42312211", "11412214", "31412212", "25121122", "14221123", "34221121", "23321122", "12421123", "32421121", "21521122", "15112123", "15121213", "13312123", "24221212", "13321213", "33321211", "11512123", "22421212", "11521213", "31521211", "25112212", "14212213", "34212211", "23312212", "12412213", "32412211", "21512212", "15221122", "24321121", "13421122", "22521121", "16112122", "16121212", "14312122", "25221211", "14321212", "12512122", "23421211", "12521212", "15212212", "24312211", "13412212", "22512211", "11612212", "21131125", "41131123", "61131121", "11122126", "31122124", "51122122", "11131216", "31131214", "51131212", "21113125", "41113123", "61113121", "21122215", "41122213", "61122211", "11113216", "31113214", "51113212", "22131124", "42131122", "11231125", "31231123", "51231121", "12122125", "32122123", "52122121", "12131215", "32131213", "52131211", "21231214", "41231212", "22113124", "42113122", "11213125", "22122214", "42122212", "11222215", "31222213", "51222211", "12113215", "32113213", "52113211", "21213214", "41213212", "23131123", "43131121", "12231124", "32231122", "21331123", "41331121", "13122124", "33122122", "13131214", "33131212", "11322124", "22231213", "42231211", "11331214", "31331212", "23113123", "43113121", "12213124", "23122213", "43122211", "12222214", "32222212", "21322213", "41322211", "13113214", "33113212", "22213213", "42213211", "11313214", "31313212", "24131122", "13231123", "33231121", "22331122", "11431123", "31431121", "14122123", "34122121", "14131213", "34131211", "12322123", "23231212", "12331213", "32331211", "21431212", "24113122", "13213123", "24122212", "13222213", "33222211", "11413123", "22322212", "11422213", "31422211", "14113213", "34113211", "23213212", "12313213", "32313211", "21413212", "25131121", "14231122", "23331121", "12431122", "15122122", "15131212", "13322122", "24231211", "13331212", "11522122", "22431211", "25113121", "14213122", "25122211", "14222212", "12413122", "23322211", "12422212", "21522211", "15113212", "24213211", "13313212", "22413211", "11513212", "15231121", "13431121", "16122121", "16131211", "14322121", "14331211", "12522121", "15213121", "15222211", "13413121", "13422211", "11613121", "16113211", "14313211", "12513211", "21141124", "41141122", "11132125", "31132123", "51132121", "11141215", "31141213", "51141211", "21123124", "41123122", "21132214", "41132212", "11114125", "31114123", "51114121", "11123215", "31123213", "51123211", "21114214", "41114212", "22141123", "42141121", "11241124", "31241122", "12132124", "32132122", "12141214", "32141212", "21241213", "41241211", "22123123", "42123121", "11223124", "22132213", "42132211", "11232214", "31232212", "12114124", "32114122", "12123214", "32123212", "21223213", "41223211", "22114213", "42114211", "11214214", "31214212", "23141122", "12241123", "32241121", "21341122", "13132123", "33132121", "13141213", "33141211", "11332123", "22241212", "11341213", "31341211", "23123122", "12223123", "23132212", "12232213", "32232211", "21332212", "13114123", "33114121", "13123213", "33123211", "11314123", "22223212", "11323213", "31323211", "23114212", "12214213", "32214211", "21314212", "24141121", "13241122", "22341121", "14132122", "14141212", "12332122", "23241211", "12341212", "24123121", "13223122", "24132211", "13232212", "11423122", "22332211", "11432212", "14114122", "14123212", "12314122", "23223211", "12323212", "21423211", "24114211", "13214212", "22314211", "11414212", "14241121", "15132121", "15141211", "13332121", "13341211", "14223121", "14232211", "12423121", "12432211", "15114121", "15123211", "13314121", "13323211", "11514121", "11523211", "14214211", "12414211", "21151123", "41151121", "11142124", "31142122", "11151214", "31151212", "21133123", "41133121", "21142213", "41142211", "11124124", "31124122", "11133214", "31133212", "21115123", "41115121", "21124213", "41124211", "11115214", "31115212", "22151122", "11251123", "31251121", "12142123", "32142121", "12151213", "32151211", "21251212", "22133122", "11233123", "22142212", "11242213", "31242211", "12124123", "32124121", "12133213", "32133211", "21233212", "22115122", "11215123", "22124212", "11224213", "31224211", "12115213", "32115211", "21215212", "23151121", "12251122", "13142122", "13151212", "11342122", "22251211", "23133121", "12233122", "23142211", "12242212", "21342211", "13124122", "13133212", "11324122", "22233211", "11333212", "23115121", "12215122", "23124211", "12224212", "21324211", "13115212", "22215211", "11315212", "13251121", "14142121", "14151211", "12342121", "13233121", "13242211", "11433121", "14124121", "14133211", "12324121", "12333211", "13215121", "13224211", "11415121", "11424211", "14115211", "12315211", "21161122", "11152123", "31152121", "11161213", "31161211", "21143122", "21152212", "11134123", "31134121", "11143213", "31143211", "21125122", "21134212", "11116123", "31116121", "11125213", "31125211", "22161121", "12152122", "12161212", "22143121", "11243122", "22152211", "11252212", "12134122", "12143212", "21243211", "22125121", "11225122", "22134211", "11234212", "12116122", "12125212", "21225211", "13152121", "13161211", "12243121", "12252211", "13134121", "13143211", "11334121", "11343211", "12225121", "12234211", "13116121", "13125211", "11316121", "11325211", "21111226", "41111224", "61111222", "31111315", "51111313", "21211135", "41211133", "61211131", "22111225", "42111223", "11211226", "31211224", "51211222", "12111316", "32111314", "52111312", "21211315", "41211313", "61211311", "22211134", "42211132", "11311135", "31311133", "51311131", "23111224", "43111222", "12211225", "32211223", "52211221", "21311224", "41311222", "13111315", "33111313", "22211314", "42211312", "11311315", "31311313", "51311311", "23211133", "43211131", "12311134", "32311132", "21411133", "41411131", "24111223", "13211224", "33211222", "22311223", "42311221", "11411224", "31411222", "14111314", "34111312", "23211313", "43211311", "12311314", "32311312", "21411313", "41411311", "24211132", "13311133", "33311131", "22411132", "11511133", "31511131", "25111222", "14211223", "34211221", "23311222", "12411223", "32411221", "21511222", "15111313", "24211312", "13311313", "33311311", "22411312", "11511313", "31511311", "25211131", "14311132", "23411131", "12511132", "21611131", "15211222", "24311221", "13411222", "22511221", "11611222", "16111312", "25211311", "14311312", "23411311", "12511312", "21611311", "31121134", "51121132", "21112135", "41112133", "61112131", "21121225", "41121223", "61121221", "11112226", "31112224", "51112222", "11121316", "31121314", "51121312", "21112315", "41112313", "61112311", "12121135", "32121133", "52121131", "21221134", "41221132", "22112134", "42112132", "11212135", "22121224", "42121222", "11221225", "31221223", "51221221", "12112225", "32112223", "52112221", "12121315", "32121313", "52121311", "21221314", "41221312", "22112314", "42112312", "11212315", "31212313", "51212311", "13121134", "33121132", "22221133", "42221131", "11321134", "31321132", "23112133", "43112131", "12212134", "23121223", "43121221", "12221224", "32221222", "21321223", "41321221", "13112224", "33112222", "13121314", "33121312", "11312224", "22221313", "42221311", "11321314", "31321312", "23112313", "43112311", "12212314", "32212312", "21312313", "41312311", "14121133", "34121131", "23221132", "12321133", "32321131", "21421132", "24112132", "13212133", "24121222", "13221223", "33221221", "11412133", "22321222", "11421223", "31421221", "14112223", "34112221", "14121313", "34121311", "12312223", "23221312", "12321313", "32321311", "21421312", "24112312", "13212313", "33212311", "22312312", "11412313", "31412311", "15121132", "24221131", "13321132", "22421131"]), $a(["22121116", "42121114", "31221115", "51221113", "32112115", "52112113", "21212116", "41212114", "61212112", "23121115", "43121113", "12221116", "32221114", "52221112", "21321115", "41321113", "61321111", "13112116", "33112114", "22212115", "42212113", "11312116", "31312114", "51312112", "24121114", "13221115", "33221113", "22321114", "42321112", "11421115", "31421113", "51421111", "14112115", "34112113", "23212114", "43212112", "12312115", "32312113", "52312111", "21412114", "41412112", "25121113", "14221114", "34221112", "23321113", "43321111", "12421114", "32421112", "21521113", "41521111", "15112114", "24212113", "13312114", "33312112", "22412113", "42412111", "11512114", "31512112", "15221113", "24321112", "13421113", "33421111", "22521112", "16112113", "25212112", "14312113", "34312111", "23412112", "12512113", "32512111", "21612112", "21131116", "41131114", "61131112", "31122115", "51122113", "21113116", "41113114", "61113112", "22131115", "42131113", "11231116", "31231114", "51231112", "12122116", "32122114", "52122112", "21222115", "41222113", "61222111", "22113115", "42113113", "11213116", "31213114", "51213112", "23131114", "43131112", "12231115", "32231113", "52231111", "21331114", "41331112", "13122115", "33122113", "22222114", "42222112", "11322115", "31322113", "51322111", "23113114", "43113112", "12213115", "32213113", "52213111", "21313114", "41313112", "24131113", "13231114", "33231112", "22331113", "42331111", "11431114", "31431112", "14122114", "34122112", "23222113", "43222111", "12322114", "32322112", "21422113", "41422111", "24113113", "13213114", "33213112", "22313113", "42313111", "11413114", "31413112", "25131112", "14231113", "34231111", "23331112", "12431113", "32431111", "15122113", "24222112", "13322113", "33322111", "22422112", "11522113", "31522111", "25113112", "14213113", "34213111", "23313112", "12413113", "32413111", "21513112", "15231112", "24331111", "13431112", "16122112", "25222111", "14322112", "23422111", "12522112", "15213112", "24313111", "13413112", "22513111", "11613112", "21141115", "41141113", "61141111", "11132116", "31132114", "51132112", "21123115", "41123113", "61123111", "11114116", "31114114", "51114112", "22141114", "42141112", "11241115", "31241113", "51241111", "12132115", "32132113", "52132111", "21232114", "41232112", "22123114", "42123112", "11223115", "31223113", "51223111", "12114115", "32114113", "52114111", "21214114", "41214112", "23141113", "43141111", "12241114", "32241112", "21341113", "41341111", "13132114", "33132112", "22232113", "42232111", "11332114", "31332112", "23123113", "43123111", "12223114", "32223112", "21323113", "41323111", "13114114", "33114112", "22214113", "42214111", "11314114", "31314112", "24141112", "13241113", "33241111", "22341112", "14132113", "34132111", "23232112", "12332113", "32332111", "21432112", "24123112", "13223113", "33223111", "22323112", "11423113", "31423111", "14114113", "34114111", "23214112", "12314113", "32314111", "21414112", "25141111", "14241112", "23341111", "15132112", "24232111", "13332112", "22432111", "25123111", "14223112", "23323111", "12423112", "21523111", "15114112", "24214111", "13314112", "22414111", "11514112", "15241111", "16132111", "14332111", "15223111", "13423111", "16114111", "14314111", "12514111", "21151114", "41151112", "11142115", "31142113", "51142111", "21133114", "41133112", "11124115", "31124113", "51124111", "21115114", "41115112", "22151113", "42151111", "11251114", "31251112", "12142114", "32142112", "21242113", "41242111", "22133113", "42133111", "11233114", "31233112", "12124114", "32124112", "21224113", "41224111", "22115113", "42115111", "11215114", "31215112", "23151112", "12251113", "32251111", "13142113", "33142111", "22242112", "11342113", "31342111", "23133112", "12233113", "32233111", "21333112", "13124113", "33124111", "22224112", "11324113", "31324111", "23115112", "12215113", "32215111", "21315112", "24151111", "13251112", "14142112", "23242111", "12342112", "24133111", "13233112", "22333111", "11433112", "14124112", "23224111", "12324112", "21424111", "24115111", "13215112", "22315111", "11415112", "14251111", "15142111", "13342111", "14233111", "12433111", "15124111", "13324111", "11524111", "14215111", "12415111", "21161113", "41161111", "11152114", "31152112", "21143113", "41143111", "11134114", "31134112", "21125113", "41125111", "11116114", "31116112", "22161112", "12152113", "32152111", "21252112", "22143112", "11243113", "31243111", "12134113", "32134111", "21234112", "22125112", "11225113", "31225111", "12116113", "32116111", "21216112", "23161111", "13152112", "22252111", "23143111", "12243112", "21343111", "13134112", "22234111", "11334112", "23125111", "12225112", "21325111", "13116112", "22216111", "11316112", "14152111", "13243111", "14134111", "12334111", "13225111", "11425111", "14116111", "12316111", "41111215", "61111213", "21211126", "41211124", "61211122", "22111216", "42111214", "31211215", "51211213", "22211125", "42211123", "11311126", "31311124", "51311122", "23111215", "43111213", "12211216", "32211214", "52211212", "21311215", "41311213", "61311211", "23211124", "43211122", "12311125", "32311123", "52311121", "21411124", "41411122", "24111214", "13211215", "33211213", "22311214", "42311212", "11411215", "31411213", "51411211", "24211123", "13311124", "33311122", "22411123", "42411121", "11511124", "31511122", "25111213", "14211214", "34211212", "23311213", "43311211", "12411214", "32411212", "21511213", "41511211", "25211122", "14311123", "34311121", "23411122", "12511123", "32511121", "21611122", "15211213", "24311212", "13411213", "33411211", "22511212", "11611213", "31611211", "31121125", "51121123", "21112126", "41112124", "61112122", "21121216", "41121214", "61121212", "31112215", "51112213", "12121126", "32121124", "52121122", "21221125", "41221123", "61221121", "22112125", "42112123", "11212126", "22121215", "42121213", "11221216", "31221214", "51221212", "12112216", "32112214", "52112212", "21212215", "41212213", "61212211", "13121125", "33121123", "22221124", "42221122", "11321125", "31321123", "51321121", "23112124", "43112122", "12212125", "23121214", "43121212", "12221215", "32221213", "52221211", "21321214", "41321212", "13112215", "33112213", "22212214", "42212212", "11312215", "31312213", "51312211", "14121124", "34121122", "23221123", "43221121", "12321124", "32321122", "21421123", "41421121", "24112123", "13212124", "24121213", "13221214", "33221212", "11412124", "22321213", "42321211", "11421214", "31421212", "14112214", "34112212", "23212213", "43212211", "12312214", "32312212", "21412213", "41412211", "15121123", "24221122", "13321123", "33321121", "22421122", "11521123", "31521121", "25112122", "14212123", "25121212", "14221213", "34221211", "12412123", "23321212", "12421213", "32421211", "21521212", "15112213", "24212212", "13312213", "33312211", "22412212", "11512213", "31512211", "16121122", "25221121", "14321122", "23421121", "12521122", "15212122", "15221212", "13412122", "24321211", "13421212", "11612122", "22521211", "16112212", "25212211", "14312212", "23412211", "12512212", "21612211", "11131126", "31131124", "51131122", "21122125", "41122123", "61122121", "21131215", "41131213", "61131211", "11113126", "31113124", "51113122", "11122216", "31122214", "51122212", "21113215", "41113213", "61113211", "12131125", "32131123", "52131121", "21231124", "41231122", "22122124", "42122122", "11222125", "22131214", "42131212", "11231215", "31231213", "51231211", "12113125", "32113123", "52113121", "12122215", "32122213", "52122211", "21222214", "41222212", "22113214", "42113212", "11213215", "31213213", "51213211", "13131124", "33131122", "22231123", "42231121", "11331124", "31331122", "23122123", "43122121", "12222124", "23131213", "43131211", "12231214", "32231212", "21331213", "41331211", "13113124", "33113122", "13122214", "33122212", "11313124", "22222213", "42222211", "11322214", "31322212", "23113213", "43113211", "12213214", "32213212", "21313213", "41313211", "14131123", "34131121", "23231122", "12331123", "32331121", "21431122", "24122122", "13222123", "24131212", "13231213", "33231211", "11422123", "22331212", "11431213", "31431211", "14113123", "34113121", "14122213", "34122211", "12313123", "23222212", "12322213", "32322211", "21422212", "24113212", "13213213", "33213211", "22313212", "11413213", "31413211", "15131122", "24231121", "13331122", "22431121", "25122121", "14222122", "25131211", "14231212", "12422122", "23331211", "12431212", "15113122", "15122212", "13313122", "24222211", "13322212", "11513122", "22422211", "11522212", "25113211", "14213212", "23313211", "12413212", "21513211", "16131121", "14331121", "15222121", "15231211", "13422121", "13431211", "16113121", "16122211", "14313121", "14322211", "12513121", "12522211", "15213211", "13413211", "11613211", "11141125", "31141123", "51141121", "21132124", "41132122", "21141214", "41141212", "11123125", "31123123", "51123121", "11132215", "31132213", "51132211", "21114124", "41114122", "21123214", "41123212", "11114215", "31114213", "51114211", "12141124", "32141122", "21241123", "41241121", "22132123", "42132121", "11232124", "22141213", "42141211", "11241214", "31241212", "12123124", "32123122", "12132214", "32132212", "21232213", "41232211", "22114123", "42114121", "11214124", "22123213", "42123211", "11223214", "31223212", "12114214", "32114212", "21214213", "41214211", "13141123", "33141121", "22241122", "11341123", "31341121", "23132122", "12232123", "23141212", "12241213", "32241211", "21341212", "13123123", "33123121", "13132213", "33132211", "11323123", "22232212", "11332213", "31332211", "23114122", "12214123", "23123212", "12223213", "32223211", "21323212", "13114213", "33114211", "22214212", "11314213", "31314211", "14141122", "23241121", "12341122", "24132121", "13232122", "24141211", "13241212", "11432122", "22341211", "14123122", "14132212", "12323122", "23232211", "12332212", "21432211", "24114121", "13214122", "24123211", "13223212", "11414122", "22323211", "11423212", "14114212", "23214211", "12314212", "21414211", "15141121", "13341121", "14232121", "14241211", "12432121", "15123121", "15132211", "13323121", "13332211", "11523121", "14214121", "14223211", "12414121", "12423211", "15114211", "13314211", "11514211", "11151124", "31151122", "21142123", "41142121", "21151213", "41151211", "11133124", "31133122", "11142214", "31142212", "21124123", "41124121", "21133213", "41133211", "11115124", "31115122", "11124214", "31124212", "21115213", "41115211", "12151123", "32151121", "21251122", "22142122", "11242123", "22151212", "11251213", "31251211", "12133123", "32133121", "12142213", "32142211", "21242212", "22124122", "11224123", "22133212", "11233213", "31233211", "12115123", "32115121", "12124213", "32124211", "21224212", "22115212", "11215213", "31215211", "13151122", "22251121", "23142121", "12242122", "23151211", "12251212", "13133122", "13142212", "11333122", "22242211", "11342212", "23124121", "12224122", "23133211", "12233212", "21333211", "13115122", "13124212", "11315122", "22224211", "11324212", "23115211", "12215212", "21315211", "14151121", "13242121", "13251211", "14133121", "14142211", "12333121", "12342211", "13224121", "13233211", "11424121", "11433211", "14115121", "14124211", "12315121", "12324211", "13215211", "11415211", "11161123", "31161121", "21152122", "21161212", "11143123", "31143121", "11152213", "31152211", "21134122", "21143212", "11125123", "31125121", "11134213", "31134211", "21116122", "21125212", "12161122", "22152121", "11252122", "22161211", "12143122", "12152212", "21252211", "22134121", "11234122", "22143211", "11243212", "12125122", "12134212", "21234211", "22116121", "11216122", "22125211", "11225212", "13161121", "12252121", "13143121", "13152211", "11343121", "12234121", "12243211", "13125121", "13134211", "11325121", "11334211", "12216121", "12225211", "31111225", "51111223", "21111316", "41111314", "61111312", "31211134", "51211132", "12111226", "32111224", "52111222", "21211225", "41211223", "61211221", "22111315", "42111313", "11211316", "31211314", "51211312", "12211135", "32211133", "52211131", "21311134", "41311132", "13111225", "33111223", "22211224", "42211222", "11311225", "31311223", "51311221", "23111314", "43111312", "12211315", "32211313", "52211311", "21311314", "41311312", "13211134", "33211132", "22311133", "42311131", "11411134", "31411132", "14111224", "34111222", "23211223", "43211221", "12311224", "32311222", "21411223", "41411221", "24111313", "13211314", "33211312", "22311313", "42311311", "11411314", "31411312", "14211133", "34211131", "23311132", "12411133", "32411131", "21511132", "15111223", "24211222", "13311223", "33311221", "22411222", "11511223", "31511221", "25111312", "14211313", "34211311", "23311312", "12411313", "32411311", "21511312", "15211132", "24311131", "13411132", "22511131", "11611132", "16111222", "25211221", "14311222", "23411221", "12511222", "21611221", "15211312", "24311311", "13411312", "22511311", "11611312", "21121135", "41121133", "61121131", "11112136", "31112134", "51112132", "11121226", "31121224", "51121222", "21112225", "41112223", "61112221", "21121315", "41121313", "61121311", "11112316", "31112314", "51112312", "22121134", "42121132", "11221135", "31221133", "51221131", "12112135", "32112133", "52112131", "12121225", "32121223", "52121221", "21221224", "41221222", "22112224", "42112222", "11212225", "22121314", "42121312", "11221315", "31221313", "51221311", "12112315", "32112313", "52112311", "21212314", "41212312", "23121133", "43121131", "12221134", "32221132", "21321133", "41321131", "13112134", "33112132", "13121224", "33121222", "11312134", "22221223", "42221221", "11321224", "31321222", "23112223", "43112221", "12212224", "23121313", "43121311", "12221314", "32221312", "21321313", "41321311", "13112314", "33112312", "22212313", "42212311", "11312314", "31312312", "24121132", "13221133", "33221131", "22321132", "11421133", "31421131", "14112133", "34112131", "14121223", "34121221", "12312133", "23221222", "12321223", "32321221", "21421222", "24112222", "13212223", "24121312", "13221313", "33221311", "11412223", "22321312", "11421313", "31421311", "14112313", "34112311", "23212312", "12312313", "32312311", "21412312", "25121131", "14221132", "23321131", "12421132", "21521131", "15112132", "15121222", "13312132", "24221221", "13321222", "11512132", "22421221", "11521222", "25112221", "14212222", "25121311", "14221312", "12412222", "23321311", "12421312", "21521311", "15112312", "24212311", "13312312", "22412311", "11512312", "15221131", "13421131", "16112131", "16121221", "14312131", "14321221", "12512131", "12521221", "15212221", "15221311", "13412221", "13421311", "11612221", "16112311", "14312311", "12512311", "21131134", "41131132", "11122135", "31122133", "51122131", "11131225", "31131223", "51131221", "21113134", "41113132", "21122224", "41122222", "21131314", "41131312", "11113225", "31113223", "51113221", "11122315", "31122313", "51122311", "21113314", "41113312", "22131133", "42131131", "11231134", "31231132", "12122134", "32122132", "12131224", "32131222", "21231223", "41231221", "22113133", "42113131", "11213134", "22122223", "42122221", "11222224", "22131313", "42131311", "11231314", "31231312", "12113224", "32113222", "12122314", "32122312", "21222313", "41222311", "22113313", "42113311", "11213314", "31213312", "23131132", "12231133", "32231131", "21331132", "13122133", "33122131", "13131223", "33131221", "11322133", "22231222", "11331223", "31331221", "23113132", "12213133", "23122222", "12222223", "23131312", "12231313", "32231311", "21331312", "13113223", "33113221", "13122313", "33122311", "11313223", "22222312", "11322313", "31322311", "23113312", "12213313", "32213311", "21313312", "24131131", "13231132", "22331131", "11431132", "14122132", "14131222", "12322132", "23231221", "12331222", "21431221", "24113131", "13213132", "24122221", "13222222", "24131311", "11413132", "13231312", "11422222", "22331311", "11431312", "14113222", "14122312", "12313222", "23222311", "12322312", "21422311", "24113311", "13213312", "22313311", "11413312", "14231131", "12431131", "15122131", "15131221", "13322131", "13331221", "11522131", "14213131", "14222221", "12413131", "14231311", "12422221", "12431311", "15113221", "15122311", "13313221", "13322311", "11513221", "11522311", "14213311", "12413311", "21141133", "41141131", "11132134", "31132132", "11141224", "31141222", "21123133", "41123131", "21132223", "41132221", "21141313", "41141311", "11114134", "31114132", "11123224", "31123222", "11132314", "31132312", "21114223", "41114221", "21123313", "41123311", "11114314", "31114312", "22141132", "11241133", "31241131", "12132133", "32132131", "12141223", "32141221", "21241222", "22123132", "11223133", "22132222", "11232223", "22141312", "11241313", "31241311", "12114133", "32114131", "12123223", "32123221", "12132313", "32132311", "21232312", "22114222", "11214223", "22123312", "11223313", "31223311", "12114313", "32114311", "21214312", "23141131", "12241132", "21341131", "13132132", "13141222", "11332132", "22241221", "11341222", "23123131", "12223132", "23132221", "12232222", "23141311", "12241312", "21341311", "13114132", "13123222", "11314132", "13132312", "11323222", "22232311", "11332312", "23114221", "12214222", "23123311", "12223312", "21323311", "13114312", "22214311", "11314312", "13241131", "14132131", "14141221", "12332131", "12341221", "13223131", "13232221", "11423131", "13241311", "11432221", "14114131", "14123221", "12314131", "14132311", "12323221", "12332311", "13214221", "13223311", "11414221", "11423311", "14114311", "12314311", "21151132", "11142133", "31142131", "11151223", "31151221", "21133132", "21142222", "21151312", "11124133", "31124131", "11133223", "31133221", "11142313", "31142311", "21115132", "21124222", "21133312", "11115223", "31115221", "11124313", "31124311", "22151131", "11251132", "12142132", "12151222", "21251221", "22133131", "11233132", "22142221", "11242222", "22151311", "11251312", "12124132", "12133222", "12142312", "21242311", "22115131", "11215132", "22124221", "11224222", "22133311", "11233312", "12115222", "12124312", "21224311", "12251131", "13142131", "13151221", "11342131", "12233131", "12242221", "12251311", "13124131", "13133221", "11324131", "13142311", "11333221", "11342311", "12215131", "12224221", "12233311", "13115221", "13124311", "11315221", "11324311", "21161131", "11152132", "11161222", "21143131", "21152221", "21161311", "11134132", "11143222", "11152312", "21125131", "21134221", "21143311", "11116132", "11125222", "11134312", "12152131", "12161221", "11243131", "11252221", "12134131", "12143221", "12152311", "11225131", "11234221", "11243311", "12116131", "12125221", "12134311", "21111235", "41111233", "61111231", "11111326", "31111324", "51111322", "21111415", "41111413", "61111411", "21211144", "41211142", "22111234", "42111232", "11211235", "31211233", "51211231", "12111325", "32111323", "52111321", "21211324", "41211322", "22111414", "42111412", "11211415", "31211413", "51211411", "22211143", "42211141", "11311144", "31311142", "23111233", "43111231", "12211234", "32211232", "21311233", "41311231", "13111324", "33111322", "22211323", "42211321", "11311324", "31311322", "23111413", "43111411", "12211414", "32211412", "21311413", "41311411", "23211142", "12311143", "32311141", "21411142", "24111232", "13211233", "33211231", "22311232", "11411233", "31411231", "14111323", "34111321", "23211322", "12311323", "32311321", "21411322", "24111412", "13211413", "33211411", "22311412", "11411413", "31411411", "24211141", "13311142", "22411141", "11511142", "25111231", "14211232", "23311231", "12411232", "21511231", "15111322", "24211321", "13311322", "22411321", "11511322", "25111411", "14211412", "23311411", "12411412", "21511411", "14311141", "12511141", "15211231", "13411231", "11611231", "16111321", "14311321", "12511321", "15211411", "13411411", "11611411", "31121143", "51121141", "21112144", "41112142", "21121234", "41121232", "11112235", "31112233", "51112231", "11121325", "31121323", "51121321", "21112324", "41112322", "21121414", "41121412", "11112415", "31112413", "51112411", "12121144", "32121142", "21221143", "41221141", "22112143", "42112141", "11212144", "22121233", "42121231", "11221234", "31221232", "12112234", "32112232", "12121324", "32121322", "21221323", "41221321", "22112323", "42112321", "11212324", "22121413", "42121411", "11221414", "31221412", "12112414", "32112412", "21212413", "41212411", "13121143", "33121141", "22221142", "11321143", "31321141", "23112142", "12212143", "23121232", "12221233", "32221231", "21321232", "13112233", "33112231", "13121323", "33121321", "11312233", "22221322", "11321323", "31321321", "23112322", "12212323", "23121412", "12221413", "32221411", "21321412", "13112413", "33112411", "22212412", "11312413", "31312411", "14121142", "23221141", "12321142", "21421141", "24112141", "13212142", "24121231", "13221232", "11412142", "22321231", "11421232", "14112232", "14121322", "12312232", "23221321", "12321322", "21421321", "24112321", "13212322", "24121411", "13221412", "11412322", "22321411", "11421412", "14112412", "23212411", "12312412", "21412411", "15121141", "13321141", "11521141", "14212141", "14221231", "12412141", "12421231", "15112231", "15121321", "13312231", "13321321", "11512231", "11521321", "14212321", "14221411", "12412321", "12421411", "15112411", "13312411", "11512411", "11131144", "31131142", "21122143", "41122141", "21131233", "41131231", "11113144", "31113142", "11122234", "31122232", "11131324", "31131322", "21113233", "41113231", "21122323", "41122321", "21131413", "41131411", "11113324", "31113322", "11122414", "31122412", "21113413", "41113411", "12131143", "32131141", "21231142", "22122142", "11222143", "22131232", "11231233", "31231231", "12113143", "32113141", "12122233", "32122231", "12131323", "32131321", "21231322", "22113232", "11213233", "22122322", "11222323", "22131412", "11231413", "31231411", "12113323", "32113321", "12122413", "32122411", "21222412", "22113412", "11213413", "31213411", "13131142", "22231141", "11331142", "23122141", "12222142", "23131231", "12231232", "21331231", "13113142", "13122232", "11313142", "13131322", "11322232", "22231321", "11331322", "23113231", "12213232", "23122321", "12222322", "23131411", "12231412", "21331411", "13113322", "13122412", "11313322", "22222411", "11322412", "23113411", "12213412", "21313411", "14131141", "12331141", "13222141", "13231231", "11422141", "11431231", "14113141", "14122231", "12313141", "14131321", "12322231", "12331321", "13213231", "13222321", "11413231", "13231411", "11422321", "11431411", "14113321", "14122411", "12313321", "12322411", "13213411", "11413411", "11141143", "31141141", "21132142", "21141232", "11123143", "31123141", "11132233", "31132231", "11141323", "31141321", "21114142", "21123232", "21132322", "21141412", "11114233", "31114231", "11123323", "31123321", "11132413", "31132411", "21114322", "21123412", "12141142", "21241141", "22132141", "11232142", "22141231", "11241232", "12123142", "12132232", "12141322", "21241321", "22114141", "11214142", "22123231", "11223232", "22132321", "11232322", "22141411", "11241412", "12114232", "12123322", "12132412", "21232411", "22114321", "11214322", "22123411", "11223412", "13141141", "11341141", "12232141", "12241231", "13123141", "13132231", "11323141", "13141321", "11332231", "11341321", "12214141", "12223231", "12232321", "12241411", "13114231", "13123321", "11314231", "13132411", "11323321", "11332411", "12214321", "12223411", "11151142", "21142141", "21151231", "11133142", "11142232", "11151322", "21124141", "21133231", "21142321", "21151411", "11115142", "11124232", "11133322", "11142412", "21115231", "21124321", "21133411", "12151141", "11242141", "11251231", "12133141", "12142231", "12151321", "11224141", "11233231", "11242321", "11251411", "12115141", "12124231", "12133321", "12142411", "11215231", "11224321", "11233411", "11161141", "11143141", "11152231", "11161321", "11125141", "11134231", "11143321", "11152411", "11111245", "31111243", "51111241", "21111334", "41111332", "11111425", "31111423", "51111421", "21111514", "41111512", "31211152", "12111244", "32111242", "21211243", "41211241", "22111333", "42111331", "11211334", "31211332", "12111424", "32111422", "21211423", "41211421", "22111513", "42111511", "11211514", "31211512", "12211153", "32211151", "21311152", "13111243", "33111241", "22211242", "11311243", "31311241", "23111332", "12211333", "32211331", "21311332", "13111423", "33111421", "22211422", "11311423", "31311421", "23111512", "12211513", "32211511", "21311512", "13211152", "22311151", "11411152", "14111242", "23211241", "12311242", "21411241", "24111331", "13211332", "22311331", "11411332", "14111422", "23211421", "12311422", "21411421", "24111511", "13211512", "22311511", "11411512", "14211151", "12411151", "15111241", "13311241", "11511241", "14211331", "12411331", "15111421", "13311421", "11511421", "14211511", "12411511", "21121153", "41121151", "11112154", "31112152", "11121244", "31121242", "21112243", "41112241", "21121333", "41121331", "11112334", "31112332", "11121424", "31121422", "21112423", "41112421", "21121513", "41121511", "11112514", "31112512", "22121152", "11221153", "31221151", "12112153", "32112151", "12121243", "32121241", "21221242", "22112242", "11212243", "22121332", "11221333", "31221331", "12112333", "32112331", "12121423", "32121421", "21221422", "22112422", "11212423", "22121512", "11221513", "31221511", "12112513", "32112511", "21212512", "23121151", "12221152", "21321151", "13112152", "13121242", "11312152", "22221241", "11321242", "23112241", "12212242", "23121331", "12221332", "21321331", "13112332", "13121422", "11312332", "22221421", "11321422", "23112421", "12212422", "23121511", "12221512", "21321511", "13112512", "22212511", "11312512", "13221151", "11421151", "14112151", "14121241", "12312151", "12321241", "13212241", "13221331", "11412241", "11421331", "14112331", "14121421", "12312331", "12321421", "13212421", "13221511", "11412421", "11421511", "14112511", "12312511", "21131152", "11122153", "31122151", "11131243", "31131241", "21113152", "21122242", "21131332", "11113243", "31113241", "11122333", "31122331", "11131423", "31131421", "21113332", "21122422", "21131512", "11113423", "31113421", "11122513", "31122511", "22131151", "11231152", "12122152", "12131242", "21231241", "22113151", "11213152", "22122241", "11222242", "22131331", "11231332", "12113242", "12122332", "12131422", "21231421", "22113331", "11213332", "22122421", "11222422", "22131511", "11231512", "12113422", "12122512", "21222511", "12231151", "13122151", "13131241", "11322151", "11331241", "12213151", "12222241", "12231331", "13113241", "13122331", "11313241", "13131421", "11322331", "11331421", "12213331", "12222421", "12231511", "13113421", "13122511", "11313421", "11322511", "21141151", "11132152", "11141242", "21123151", "21132241", "21141331", "11114152", "11123242", "11132332", "11141422", "21114241", "21123331", "21132421", "21141511", "11114332", "11123422", "11132512", "11241151", "12132151", "12141241", "11223151", "11232241", "11241331", "12114151", "12123241", "12132331", "12141421", "11214241", "11223331", "11232421", "11241511", "12114331", "12123421", "12132511", "11142151", "11151241", "11124151", "11133241", "11142331", "11151421", "11115241", "11124331", "11133421", "11142511", "21111253", "41111251", "11111344", "31111342", "21111433", "41111431", "11111524", "31111522", "21111613", "41111611", "21211162", "22111252", "11211253", "31211251", "12111343", "32111341", "21211342", "22111432", "11211433", "31211431", "12111523", "32111521", "21211522", "22111612", "11211613", "31211611", "22211161", "11311162", "23111251", "12211252", "21311251", "13111342", "22211341", "11311342", "23111431", "12211432", "21311431", "13111522", "22211521", "11311522", "23111611", "12211612", "21311611", "12311161", "13211251", "11411251", "14111341", "12311341", "13211431", "11411431", "14111521", "12311521", "13211611", "11411611", "31121161", "21112162", "21121252", "11112253", "31112251", "11121343", "31121341", "21112342", "21121432", "11112433", "31112431", "11121523", "31121521", "21112522", "21121612", "12121162", "21221161", "22112161", "11212162", "22121251", "11221252", "12112252", "12121342", "21221341", "22112341", "11212342", "22121431", "11221432", "12112432", "12121522", "21221521", "22112521", "11212522", "22121611", "11221612", "13121161", "11321161", "12212161", "12221251", "13112251", "13121341", "11312251", "11321341", "12212341", "12221431", "13112431", "13121521", "11312431", "11321521", "12212521", "12221611", "11131162", "21122161", "21131251", "11113162"])]); //#16889
    $_.parity = $a(["1001", "0101", "1100", "0011", "1010", "0110", "1111", "0000"]); //#16893
    $_.rowbits = $a($_.r); //#16897
    for (var _6G = 0, _6F = $_.r - 1; _6G <= _6F; _6G += 1) { //#16917
        $_.i = _6G; //#16899
        $k[$j++] = 'p'; //#16900
        if ($_.i != ($_.r - 1)) { //#16900
            $k[$j++] = $get($_.parity, $_.i); //#16900
        } else { //#16900
            $k[$j++] = "0000"; //#16900
        } //#16900
        var _6M = $k[--$j]; //#16900
        $_[$k[--$j]] = _6M; //#16900
        $_.ccrow = $geti($_.ccs, $_.i * 8, 8); //#16901
        $k[$j++] = Infinity; //#16902
        for (var _6R = 0; _6R <= 7; _6R += 2) { //#16903
            $aload($geti($_.ccrow, _6R, 2)); //#16903
            var _6U = $k[--$j]; //#16903
            var _6V = $k[--$j]; //#16903
            $k[$j++] = $f(_6U + (_6V * 49)); //#16903
        } //#16903
        $_.scrow = $a(); //#16903
        $k[$j++] = Infinity; //#16905
        $k[$j++] = 10; //#16910
        $k[$j++] = 1; //#16910
        $k[$j++] = 1; //#16910
        for (var _6X = 0; _6X <= 3; _6X += 1) { //#16910
            $_.j = _6X; //#16908
            $forall($get($get($_.patterns, $get($_.p, $_.j) - 48), $get($_.scrow, $_.j)), function() { //#16909
                var _6h = $k[--$j]; //#16909
                $k[$j++] = _6h - 48; //#16909
            }); //#16909
        } //#16909
        $k[$j++] = 4; //#16911
        $k[$j++] = 1; //#16911
        $_.sbs = $a(); //#16911
        $k[$j++] = Infinity; //#16913
        var _6j = $_.sbs; //#16914
        $k[$j++] = 1; //#16914
        for (var _6k = 0, _6l = _6j.length; _6k < _6l; _6k++) { //#16914
            var _6n = $k[$j - 1]; //#16914
            var _6o = (_6n == 0) ? 1 : 0; //#16914
            for (var _6p = 0, _6q = $get(_6j, _6k); _6p < _6q; _6p++) { //#16914
                $k[$j++] = _6o //#16914
            } //#16914
        } //#16914
        $astore($a($counttomark() - 1)); //#16915
        var _6t = $k[--$j]; //#16915
        $k[$j - 1] = _6t; //#16915
        var _6v = $k[--$j]; //#16915
        $k[$j - 1] = _6v; //#16915
        $k[$j++] = $_.rowbits; //#16916
        $k[$j++] = $_.i; //#16916
        var _6z = $k[$j - 3]; //#16916
        var _70 = $k[$j - 1]; //#16916
        $put($k[$j - 2], _70, _6z); //#16916
        $j -= 3; //#16916
    } //#16916
    $k[$j++] = Infinity; //#16920
    for (var _73 = 0, _74 = 81 * $_.sepheight; _73 < _74; _73++) { //#16921
        $k[$j++] = 1; //#16921
    } //#16921
    for (var _77 = 0, _76 = $_.r - 2; _77 <= _76; _77 += 1) { //#16926
        $_.i = _77; //#16923
        for (var _79 = 0, _7A = $_.rowheight; _79 < _7A; _79++) { //#16924
            $aload($get($_.rowbits, $_.i)); //#16924
        } //#16924
        for (var _7F = 0, _7G = $_.sepheight; _7F < _7G; _7F++) { //#16925
            for (var _7H = 0; _7H < 10; _7H++) { //#16925
                $k[$j++] = 0; //#16925
            } //#16925
            for (var _7I = 0; _7I < 70; _7I++) { //#16925
                $k[$j++] = 1; //#16925
            } //#16925
            $k[$j++] = 0; //#16925
        } //#16925
    } //#16925
    for (var _7K = 0, _7L = $_.rowheight; _7K < _7L; _7K++) { //#16927
        $aload($get($_.rowbits, $_.r - 1)); //#16927
    } //#16927
    for (var _7Q = 0, _7R = 81 * $_.sepheight; _7Q < _7R; _7Q++) { //#16928
        $k[$j++] = 1; //#16928
    } //#16928
    $_.pixs = $a(); //#16928
    var _7X = new Map([
        ["ren", 'renmatrix'],
        ["pixs", $_.pixs],
        ["pixx", 81],
        ["pixy", ~~($_.pixs.length / 81)],
        ["height", (~~($_.pixs.length / 81)) / 72],
        ["width", 81 / 72],
        ["opt", $_.options]
    ]); //#16939
    $k[$j++] = _7X; //#16942
    if (!$_.dontdraw) { //#16942
        bwipp_renmatrix(); //#16942
    } //#16942
    $_ = $__; //#16946
} //bwipp_code49
function bwipp_flattermarken() {
    var $__ = $_; //#17240
    $_ = Object.assign({}, $_); //#17240
    $_.dontdraw = false; //#17243
    $_.includetext = false; //#17244
    $_.textfont = "OCR-B"; //#17245
    $_.textsize = 10; //#17246
    $_.textyoffset = -8; //#17247
    $_.height = 0.3; //#17248
    $k[$j++] = $_; //#17250
    bwipp_processoptions(); //#17250
    $_.options = $k[--$j]; //#17250
    $_.barcode = $k[--$j]; //#17251
    $forall($_.barcode, function() { //#17260
        var _3 = $k[--$j]; //#17257
        if ((_3 < 48) || (_3 > 57)) { //#17259
            $k[$j++] = 'bwipp.flattermarkenBadCharacter#17258'; //#17258
            $k[$j++] = "Flattermarken must contain only digits"; //#17258
            bwipp_raiseerror(); //#17258
        } //#17258
    }); //#17258
    $_.encs = $a(["0018", "0117", "0216", "0315", "0414", "0513", "0612", "0711", "0810", "0900"]); //#17267
    $_.barchars = "1234567890"; //#17270
    $_.barlen = $_.barcode.length; //#17273
    $_.sbs = $s($_.barlen * 4); //#17275
    $_.txt = $a($_.barlen); //#17276
    for (var _D = 0, _C = $_.barlen - 1; _D <= _C; _D += 1) { //#17288
        $_.i = _D; //#17279
        $search($_.barchars, $geti($_.barcode, $_.i, 1)); //#17281
        $j--; //#17282
        $_.indx = $k[--$j].length; //#17283
        $j -= 2; //#17284
        $_.enc = $get($_.encs, $_.indx); //#17285
        $puti($_.sbs, $_.i * 4, $_.enc); //#17286
        $put($_.txt, $_.i, $a([$geti($_.barcode, $_.i, 1), $_.i * 9, $_.textyoffset, $_.textfont, $_.textsize])); //#17287
    } //#17287
    $k[$j++] = Infinity; //#17291
    $k[$j++] = Infinity; //#17293
    var _Z = $_.sbs; //#17293
    for (var _a = 0, _b = _Z.length; _a < _b; _a++) { //#17293
        $k[$j++] = $get(_Z, _a) - 48; //#17293
    } //#17293
    var _d = $a(); //#17293
    $k[$j++] = Infinity; //#17294
    for (var _f = 0, _g = ~~(($_.sbs.length + 1) / 2); _f < _g; _f++) { //#17294
        $k[$j++] = $_.height; //#17294
    } //#17294
    var _i = $a(); //#17294
    $k[$j++] = Infinity; //#17295
    for (var _k = 0, _l = ~~(($_.sbs.length + 1) / 2); _k < _l; _k++) { //#17295
        $k[$j++] = 0; //#17295
    } //#17295
    var _m = $a(); //#17295
    $k[$j++] = 'ren'; //#17297
    $k[$j++] = 'renlinear'; //#17297
    $k[$j++] = 'sbs'; //#17297
    $k[$j++] = _d; //#17297
    $k[$j++] = 'bhs'; //#17297
    $k[$j++] = _i; //#17297
    $k[$j++] = 'bbs'; //#17297
    $k[$j++] = _m; //#17297
    $k[$j++] = 'txt'; //#17297
    $k[$j++] = $_.txt; //#17297
    $k[$j++] = 'opt'; //#17297
    $k[$j++] = $_.options; //#17297
    var _p = $d(); //#17297
    $k[$j++] = _p; //#17300
    if (!$_.dontdraw) { //#17300
        bwipp_renlinear(); //#17300
    } //#17300
    $_ = $__; //#17304
} //bwipp_flattermarken
function bwipp_raw() {
    var $__ = $_; //#17332
    $_ = Object.assign({}, $_); //#17332
    $_.dontdraw = false; //#17335
    $_.height = 1; //#17336
    $k[$j++] = $_; //#17338
    bwipp_processoptions(); //#17338
    $_.options = $k[--$j]; //#17338
    $_.barcode = $k[--$j]; //#17339
    $forall($_.barcode, function() { //#17346
        var _3 = $k[--$j]; //#17343
        if ((_3 < 49) || (_3 > 57)) { //#17345
            $k[$j++] = 'bwipp.rawBadCharacter#17344'; //#17344
            $k[$j++] = "Raw must contain only digits 1 to 9"; //#17344
            bwipp_raiseerror(); //#17344
        } //#17344
    }); //#17344
    $k[$j++] = Infinity; //#17349
    $k[$j++] = Infinity; //#17351
    $forall($_.barcode, function() { //#17351
        var _5 = $k[--$j]; //#17351
        $k[$j++] = _5 - 48; //#17351
    }); //#17351
    var _6 = $a(); //#17351
    $k[$j++] = Infinity; //#17352
    for (var _8 = 0, _9 = ~~(($_.barcode.length + 1) / 2); _8 < _9; _8++) { //#17352
        $k[$j++] = $_.height; //#17352
    } //#17352
    var _B = $a(); //#17352
    $k[$j++] = Infinity; //#17353
    for (var _D = 0, _E = ~~(($_.barcode.length + 1) / 2); _D < _E; _D++) { //#17353
        $k[$j++] = 0; //#17353
    } //#17353
    var _F = $a(); //#17353
    $k[$j++] = 'ren'; //#17354
    $k[$j++] = 'renlinear'; //#17354
    $k[$j++] = 'sbs'; //#17354
    $k[$j++] = _6; //#17354
    $k[$j++] = 'bhs'; //#17354
    $k[$j++] = _B; //#17354
    $k[$j++] = 'bbs'; //#17354
    $k[$j++] = _F; //#17354
    $k[$j++] = 'opt'; //#17354
    $k[$j++] = $_.options; //#17354
    var _H = $d(); //#17354
    $k[$j++] = _H; //#17357
    if (!$_.dontdraw) { //#17357
        bwipp_renlinear(); //#17357
    } //#17357
    $_ = $__; //#17359
} //bwipp_raw
function bwipp_daft() {
    var $__ = $_; //#17387
    $_ = Object.assign({}, $_); //#17387
    $_.dontdraw = false; //#17390
    $_.height = 0.175; //#17391
    $k[$j++] = $_; //#17393
    bwipp_processoptions(); //#17393
    $_.options = $k[--$j]; //#17393
    $_.barcode = $k[--$j]; //#17394
    if ($_.barcode.length == 0) { //#17398
        $k[$j++] = 'bwipp.daftEmptyData#17397'; //#17397
        $k[$j++] = "The data must not be empty"; //#17397
        bwipp_raiseerror(); //#17397
    } //#17397
    $forall($_.barcode, function() { //#17404
        var _4 = $k[--$j]; //#17401
        if ((_4 != 68) && ((_4 != 65) && ((_4 != 70) && (_4 != 84)))) { //#17403
            $k[$j++] = 'bwipp.daftBadCharacter#17402'; //#17402
            $k[$j++] = "DAFT must contain only characters D, A, F and T"; //#17402
            bwipp_raiseerror(); //#17402
        } //#17402
    }); //#17402
    $_.barlen = $_.barcode.length; //#17406
    $_.bbs = $a($_.barlen); //#17408
    $_.bhs = $a($_.barlen); //#17409
    for (var _C = 0, _B = $_.barlen - 1; _C <= _B; _C += 1) { //#17429
        $_.i = _C; //#17411
        $_.enc = $geti($_.barcode, $_.i, 1); //#17412
        if ($eq($_.enc, "D")) { //#17416
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#17414
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#17415
        } //#17415
        if ($eq($_.enc, "A")) { //#17420
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#17418
            $put($_.bhs, $_.i, (5 * $_.height) / 8); //#17419
        } //#17419
        if ($eq($_.enc, "F")) { //#17424
            $put($_.bbs, $_.i, (0 * $_.height) / 8); //#17422
            $put($_.bhs, $_.i, (8 * $_.height) / 8); //#17423
        } //#17423
        if ($eq($_.enc, "T")) { //#17428
            $put($_.bbs, $_.i, (3 * $_.height) / 8); //#17426
            $put($_.bhs, $_.i, (2 * $_.height) / 8); //#17427
        } //#17427
    } //#17427
    $k[$j++] = Infinity; //#17432
    var _i = $_.bbs; //#17434
    var _j = $_.bhs; //#17435
    $k[$j++] = Infinity; //#17436
    for (var _l = 0, _m = $_.bhs.length - 1; _l < _m; _l++) { //#17436
        $k[$j++] = 1.44; //#17436
        $k[$j++] = 1.872; //#17436
    } //#17436
    $k[$j++] = 1.44; //#17436
    var _n = $a(); //#17436
    $k[$j++] = 'ren'; //#17437
    $k[$j++] = 'renlinear'; //#17437
    $k[$j++] = 'bbs'; //#17437
    $k[$j++] = _i; //#17437
    $k[$j++] = 'bhs'; //#17437
    $k[$j++] = _j; //#17437
    $k[$j++] = 'sbs'; //#17437
    $k[$j++] = _n; //#17437
    $k[$j++] = 'opt'; //#17437
    $k[$j++] = $_.options; //#17437
    var _p = $d(); //#17437
    $k[$j++] = _p; //#17440
    if (!$_.dontdraw) { //#17440
        bwipp_renlinear(); //#17440
    } //#17440
    $_ = $__; //#17442
} //bwipp_daft
function bwipp_symbol() {
    var $__ = $_; //#17473
    $_ = Object.assign({}, $_); //#17473
    $_.dontdraw = false; //#17476
    $k[$j++] = $_; //#17478
    bwipp_processoptions(); //#17478
    $_.options = $k[--$j]; //#17478
    $_.barcode = $k[--$j]; //#17479
    $_.fimasbs = $a([2.25, 2.25, 2.25, 11.25, 2.25, 11.25, 2.25, 2.25, 2.25]); //#17485
    $k[$j++] = Infinity; //#17486
    for (var _4 = 0, _5 = (~~($_.fimasbs.length / 2)) + 1; _4 < _5; _4++) { //#17486
        $k[$j++] = 0.625; //#17486
    } //#17486
    $_.fimabhs = $a(); //#17486
    $k[$j++] = Infinity; //#17487
    for (var _8 = 0, _9 = (~~($_.fimasbs.length / 2)) + 1; _8 < _9; _8++) { //#17487
        $k[$j++] = 0; //#17487
    } //#17487
    $_.fimabbs = $a(); //#17487
    $_.fimbsbs = $a([2.25, 6.75, 2.25, 2.25, 2.25, 6.25, 2.25, 2.25, 2.25, 6.75, 2.25]); //#17489
    $k[$j++] = Infinity; //#17490
    for (var _D = 0, _E = (~~($_.fimbsbs.length / 2)) + 1; _D < _E; _D++) { //#17490
        $k[$j++] = 0.625; //#17490
    } //#17490
    $_.fimbbhs = $a(); //#17490
    $k[$j++] = Infinity; //#17491
    for (var _H = 0, _I = (~~($_.fimbsbs.length / 2)) + 1; _H < _I; _H++) { //#17491
        $k[$j++] = 0; //#17491
    } //#17491
    $_.fimbbbs = $a(); //#17491
    $_.fimcsbs = $a([2.25, 2.25, 2.25, 6.75, 2.25, 6.75, 2.25, 6.75, 2.25, 2.25, 2.25]); //#17493
    $k[$j++] = Infinity; //#17494
    for (var _M = 0, _N = (~~($_.fimcsbs.length / 2)) + 1; _M < _N; _M++) { //#17494
        $k[$j++] = 0.625; //#17494
    } //#17494
    $_.fimcbhs = $a(); //#17494
    $k[$j++] = Infinity; //#17495
    for (var _Q = 0, _R = (~~($_.fimcsbs.length / 2)) + 1; _Q < _R; _Q++) { //#17495
        $k[$j++] = 0; //#17495
    } //#17495
    $_.fimcbbs = $a(); //#17495
    $_.fimdsbs = $a([2.25, 2.25, 2.25, 2.25, 2.25, 6.75, 2.25, 6.75, 2.25, 2.25, 2.25, 2.25, 2.25]); //#17497
    $k[$j++] = Infinity; //#17498
    for (var _V = 0, _W = (~~($_.fimdsbs.length / 2)) + 1; _V < _W; _V++) { //#17498
        $k[$j++] = 0.625; //#17498
    } //#17498
    $_.fimdbhs = $a(); //#17498
    $k[$j++] = Infinity; //#17499
    for (var _Z = 0, _a = (~~($_.fimdsbs.length / 2)) + 1; _Z < _a; _Z++) { //#17499
        $k[$j++] = 0; //#17499
    } //#17499
    $_.fimdbbs = $a(); //#17499
    $_.fimesbs = $a([2.25, 6.75, 2.25, 15.75, 2.25, 6.75, 2.25]); //#17501
    $k[$j++] = Infinity; //#17502
    for (var _e = 0, _f = (~~($_.fimesbs.length / 2)) + 1; _e < _f; _e++) { //#17502
        $k[$j++] = 0.625; //#17502
    } //#17502
    $_.fimebhs = $a(); //#17502
    $k[$j++] = Infinity; //#17503
    for (var _i = 0, _j = (~~($_.fimesbs.length / 2)) + 1; _i < _j; _i++) { //#17503
        $k[$j++] = 0; //#17503
    } //#17503
    $_.fimebbs = $a(); //#17503
    $k[$j++] = Infinity; //#17505
    for (var _l = 0; _l < 100; _l++) { //#17505
        $k[$j++] = 1; //#17505
        $k[$j++] = 1; //#17505
    } //#17505
    $k[$j++] = 1; //#17505
    $_.zebrasbs = $a(); //#17505
    $k[$j++] = Infinity; //#17506
    for (var _o = 0, _p = (~~($_.zebrasbs.length / 2)) + 1; _o < _p; _o++) { //#17506
        $k[$j++] = 1; //#17506
    } //#17506
    $_.zebrabhs = $a(); //#17506
    $k[$j++] = Infinity; //#17507
    for (var _s = 0, _t = (~~($_.zebrasbs.length / 2)) + 1; _s < _t; _s++) { //#17507
        $k[$j++] = 0; //#17507
    } //#17507
    $_.zebrabbs = $a(); //#17507
    $_.checksize = 199; //#17509
    $k[$j++] = Infinity; //#17510
    var _v = $_.checksize; //#17510
    for (var _w = 0, _x = ~~((_v * _v) / 2); _w < _x; _w++) { //#17510
        $k[$j++] = 1; //#17510
        $k[$j++] = 0; //#17510
    } //#17510
    $k[$j++] = 1; //#17510
    $_.checkpixs = $a(); //#17510
    var _z = new Map([
        ["renlinear", bwipp_renlinear],
        ["renmatrix", bwipp_renmatrix]
    ]); //#17514
    $_.renmap = _z; //#17515
    var _1V = new Map([
        ["fima", function() {
            var _14 = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.fimasbs],
                ["bhs", $_.fimabhs],
                ["bbs", $_.fimabbs]
            ]); //#17520
            $k[$j++] = _14; //#17520
        }],
        ["fimb", function() {
            var _18 = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.fimbsbs],
                ["bhs", $_.fimbbhs],
                ["bbs", $_.fimbbbs]
            ]); //#17521
            $k[$j++] = _18; //#17521
        }],
        ["fimc", function() {
            var _1C = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.fimcsbs],
                ["bhs", $_.fimcbhs],
                ["bbs", $_.fimcbbs]
            ]); //#17522
            $k[$j++] = _1C; //#17522
        }],
        ["fimd", function() {
            var _1G = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.fimdsbs],
                ["bhs", $_.fimdbhs],
                ["bbs", $_.fimdbbs]
            ]); //#17523
            $k[$j++] = _1G; //#17523
        }],
        ["fime", function() {
            var _1K = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.fimesbs],
                ["bhs", $_.fimebhs],
                ["bbs", $_.fimebbs]
            ]); //#17524
            $k[$j++] = _1K; //#17524
        }],
        ["zebra", function() {
            var _1O = new Map([
                ["ren", 'renlinear'],
                ["sbs", $_.zebrasbs],
                ["bhs", $_.zebrabhs],
                ["bbs", $_.zebrabbs]
            ]); //#17525
            $k[$j++] = _1O; //#17525
        }],
        ["check", function() {
            var _1U = new Map([
                ["ren", 'renmatrix'],
                ["pixs", $_.checkpixs],
                ["pixx", $_.checksize],
                ["pixy", $_.checksize],
                ["height", $_.checksize / 72],
                ["width", $_.checksize / 72]
            ]); //#17526
            $k[$j++] = _1U; //#17526
        }]
    ]); //#17526
    $_.symbols = _1V; //#17527
    var _1Y = $get($_.symbols, $_.barcode) !== undefined; //#17530
    if (!_1Y) { //#17532
        $k[$j++] = 'bwipp.symbolUnknownSymbol#17531'; //#17531
        $k[$j++] = "Unknown symbol name provided"; //#17531
        bwipp_raiseerror(); //#17531
    } //#17531
    if ($get($_.symbols, $_.barcode)() === true) {
        return true;
    } //#17534
    var _1c = $k[$j - 1]; //#17535
    $put(_1c, 'opt', $_.options); //#17535
    if (!$_.dontdraw) { //#17537
        var _1f = $k[$j - 1]; //#17537
        if ($get($_.renmap, $get(_1f, 'ren'))() === true) {
            return true;
        } //#17537
    } //#17537
    $_ = $__; //#17541
} //bwipp_symbol
function bwipp_pdf417() {
    var $__ = $_; //#17572
    $_ = Object.assign({}, $_); //#17572
    $_.dontdraw = false; //#17575
    $_.compact = false; //#17576
    $_.eclevel = -1; //#17577
    $_.fixedeclevel = false; //#17578
    $_.columns = 0; //#17579
    $_.rows = 0; //#17580
    $_.rowmult = 3; //#17581
    $_.ccc = false; //#17582
    $_.raw = false; //#17583
    $_.parse = false; //#17584
    $_.parsefnc = false; //#17585
    $k[$j++] = $_; //#17587
    bwipp_processoptions(); //#17587
    $_.options = $k[--$j]; //#17587
    $_.barcode = $k[--$j]; //#17588
    if ($_.barcode.length == 0) { //#17592
        $k[$j++] = 'bwipp.pdf417emptyData#17591'; //#17591
        $k[$j++] = "The data must not be empty"; //#17591
        bwipp_raiseerror(); //#17591
    } //#17591
    if (($_.eclevel != -1) && (($_.eclevel < 0) || ($_.eclevel > 8))) { //#17596
        $k[$j++] = 'bwipp.pdf417InvalidErrorCorrectionLevel#17595'; //#17595
        $k[$j++] = "Valid error correction levels are 0 to 8"; //#17595
        bwipp_raiseerror(); //#17595
    } //#17595
    if ($_.ccc && $_.raw) { //#17600
        $k[$j++] = 'bwipp.pdf417cccAndRaw#17599'; //#17599
        $k[$j++] = "Cannot combine ccc and raw"; //#17599
        bwipp_raiseerror(); //#17599
    } //#17599
    if ($_.ccc && $_.compact) { //#17604
        $k[$j++] = 'bwipp.pdf417cccAndCompact#17603'; //#17603
        $k[$j++] = "Cannot combine ccc and compact"; //#17603
        bwipp_raiseerror(); //#17603
    } //#17603
    if (($_.rows != 0) && (($_.rows < 3) || ($_.rows > 90))) { //#17608
        $k[$j++] = 'bwipp.pdf417invalidRows#17607'; //#17607
        $k[$j++] = "There must be between 3 and 90 rows"; //#17607
        bwipp_raiseerror(); //#17607
    } //#17607
    if (($_.columns != 0) && (($_.columns < 1) || ($_.columns > 30))) { //#17612
        $k[$j++] = 'bwipp.pdf417invalidColumns#17611'; //#17611
        $k[$j++] = "There must be between 1 and 30 columns"; //#17611
        bwipp_raiseerror(); //#17611
    } //#17611
    if ($_.rowmult <= 0) { //#17616
        $k[$j++] = 'bwipp.pdf417badRowMult#17615'; //#17615
        $k[$j++] = "The row multiplier must be greater than zero"; //#17615
        bwipp_raiseerror(); //#17615
    } //#17615
    $_.T = 0; //#17622
    $_.N = 1; //#17622
    $_.B = 2; //#17622
    $_.A = 0; //#17623
    $_.L = 1; //#17623
    $_.M = 2; //#17623
    $_.P = 3; //#17623
    $_.tl = -1; //#17626
    $_.nl = -2; //#17626
    $_.bl = -3; //#17626
    $_.bl6 = -4; //#17626
    $_.bs = -5; //#17626
    $_.al = -6; //#17629
    $_.ll = -7; //#17629
    $_.ml = -8; //#17629
    $_.pl = -9; //#17629
    $_.as = -10; //#17629
    $_.ps = -11; //#17629
    $_.charmaps = $a([$a(["A", "a", "0", ";"]), $a(["B", "b", "1", "<"]), $a(["C", "c", "2", ">"]), $a(["D", "d", "3", "@"]), $a(["E", "e", "4", "["]), $a(["F", "f", "5", 92]), $a(["G", "g", "6", "]"]), $a(["H", "h", "7", "_"]), $a(["I", "i", "8", "`"]), $a(["J", "j", "9", "~"]), $a(["K", "k", "&", "!"]), $a(["L", "l", 13, 13]), $a(["M", "m", 9, 9]), $a(["N", "n", ",", ","]), $a(["O", "o", ":", ":"]), $a(["P", "p", "#", 10]), $a(["Q", "q", "-", "-"]), $a(["R", "r", ".", "."]), $a(["S", "s", "$", "$"]), $a(["T", "t", "/", "/"]), $a(["U", "u", "+", "\""]), $a(["V", "v", "%", "|"]), $a(["W", "w", "*", "*"]), $a(["X", "x", "=", 40]), $a(["Y", "y", "^", 41]), $a(["Z", "z", $_.pl, "?"]), $a([" ", " ", " ", "{"]), $a([$_.ll, $_.as, $_.ll, "}"]), $a([$_.ml, $_.ml, $_.al, "'"]), $a([$_.ps, $_.ps, $_.ps, $_.al])]); //#17664
    $_.charvals = $a([new Map, new Map, new Map, new Map]); //#17667
    $_.alltext = new Map; //#17668
    for (var _10 = 0, _z = $_.charmaps.length - 1; _10 <= _z; _10 += 1) { //#17678
        $_.i = _10; //#17670
        $_.encs = $get($_.charmaps, $_.i); //#17671
        for (var _14 = 0; _14 <= 3; _14 += 1) { //#17677
            $_.j = _14; //#17673
            var _17 = $get($_.encs, $_.j); //#17674
            $k[$j++] = _17; //#17674
            if ($eq($type(_17), 'stringtype')) { //#17674
                var _1A = $get($k[--$j], 0); //#17674
                $k[$j++] = _1A; //#17674
            } //#17674
            var _1B = $k[--$j]; //#17675
            $put($get($_.charvals, $_.j), _1B, $_.i); //#17675
            $put($_.alltext, _1B, -1); //#17676
        } //#17676
    } //#17676
    $_.e = 10000; //#17680
    $_.latlen = $a([$a([0, 1, 1, 2]), $a([2, 0, 1, 2]), $a([1, 1, 0, 1]), $a([1, 2, 2, 0])]); //#17688
    $_.latseq = $a([$a([$a([]), $a([$_.ll]), $a([$_.ml]), $a([$_.ml, $_.pl])]), $a([$a([$_.ml, $_.al]), $a([]), $a([$_.ml]), $a([$_.ml, $_.pl])]), $a([$a([$_.al]), $a([$_.ll]), $a([]), $a([$_.pl])]), $a([$a([$_.al]), $a([$_.al, $_.ll]), $a([$_.al, $_.ml]), $a([])])]); //#17696
    $_.shftlen = $a([$a([$_.e, $_.e, $_.e, 1]), $a([1, $_.e, $_.e, 1]), $a([$_.e, $_.e, $_.e, 1]), $a([$_.e, $_.e, $_.e, $_.e])]); //#17704
    $k[$j++] = Infinity; //#17706
    $k[$j++] = $_.tl; //#17707
    $k[$j++] = 900; //#17707
    $k[$j++] = $_.bl; //#17707
    $k[$j++] = 901; //#17707
    $k[$j++] = $_.bl6; //#17707
    $k[$j++] = 924; //#17707
    $k[$j++] = $_.nl; //#17707
    $k[$j++] = 902; //#17707
    $k[$j++] = $_.bs; //#17707
    $k[$j++] = 913; //#17707
    $_.latchcws = $d(); //#17708
    if ($_.raw) { //#17730
        $_.datcws = $a($_.barcode.length); //#17713
        $_.i = 0; //#17714
        $_.j = 0; //#17714
        for (;;) { //#17725
            if ($_.i >= ($_.barcode.length - 3)) { //#17716
                break; //#17716
            } //#17716
            if ($get($_.barcode, $_.i) != 94) { //#17717
                break; //#17717
            } //#17717
            var _2W = $geti($_.barcode, $_.i + 1, 3); //#17718
            $k[$j++] = _2W; //#17720
            $k[$j++] = false; //#17720
            for (var _2X = 0, _2Y = _2W.length; _2X < _2Y; _2X++) { //#17720
                var _2Z = $get(_2W, _2X); //#17720
                if ((_2Z < 48) || (_2Z > 57)) { //#17719
                    $k[$j - 1] = true; //#17719
                } //#17719
            } //#17719
            if ($k[--$j]) { //#17720
                $j--; //#17720
                break; //#17720
            } //#17720
            $_.cw = $cvi($k[--$j]); //#17721
            $put($_.datcws, $_.j, $_.cw); //#17722
            $_.i = $_.i + 4; //#17723
            $_.j = $_.j + 1; //#17724
        } //#17724
        if ($_.i != $_.barcode.length) { //#17728
            $k[$j++] = 'bwipp.pdf417badRawFormat#17727'; //#17727
            $k[$j++] = "raw data must be formatted as ^NNN"; //#17727
            bwipp_raiseerror(); //#17727
        } //#17727
        $_.datcws = $geti($_.datcws, 0, $_.j); //#17729
    } //#17729
    $_.encb = function() {
        $_.in = $k[--$j]; //#17733
        $_.inlen = $_.in.length; //#17734
        $_.out = $a(((~~($_.inlen / 6)) * 5) + ($_.inlen % 6)); //#17735
        for (var _2t = 0, _2s = (~~($_.inlen / 6)) - 1; _2t <= _2s; _2t += 1) { //#17758
            $_.k = _2t; //#17737
            $k[$j++] = Infinity; //#17738
            $aload($geti($_.in, $_.k * 6, 3)); //#17738
            $_.msbs = $a(); //#17738
            $k[$j++] = Infinity; //#17739
            $aload($_.msbs); //#17740
            var _2z = $k[--$j]; //#17740
            var _30 = $k[--$j]; //#17740
            var _31 = $k[--$j]; //#17740
            $k[$j++] = $f(_2z + _30 * 256 + _31 * 65536); //#17741
            for (var _32 = 0; _32 < 3; _32++) { //#17741
                var _33 = $k[--$j]; //#17741
                $k[$j++] = _33 % 900; //#17741
                $k[$j++] = ~~(_33 / 900); //#17741
            } //#17741
            $_.mscs = $a(); //#17741
            $k[$j++] = Infinity; //#17743
            $aload($geti($_.in, ($_.k * 6) + 3, 3)); //#17743
            $_.lsbs = $a(); //#17743
            $k[$j++] = Infinity; //#17744
            $aload($_.lsbs); //#17745
            var _3A = $k[--$j]; //#17745
            var _3B = $k[--$j]; //#17745
            var _3C = $k[--$j]; //#17745
            $k[$j++] = $f(_3A + _3B * 256 + _3C * 65536); //#17746
            for (var _3D = 0; _3D < 3; _3D++) { //#17746
                var _3E = $k[--$j]; //#17746
                $k[$j++] = _3E % 900; //#17746
                $k[$j++] = ~~(_3E / 900); //#17746
            } //#17746
            $_.lscs = $a(); //#17746
            var _3H = $get($_.lscs, 0); //#17748
            var _3J = $get($_.mscs, 0); //#17748
            $put($_.out, ($_.k * 5) + 4, ($f(_3H + (_3J * 316))) % 900); //#17749
            var _3N = $get($_.lscs, 1); //#17750
            var _3P = $get($_.mscs, 0); //#17750
            var _3R = $get($_.mscs, 1); //#17750
            $put($_.out, $_.k * 5 + 3, $f(~~((_3H + _3J * 316) / 900) + _3N + _3P * 641 + _3R * 316) % 900); //#17751
            var _3V = $get($_.lscs, 2); //#17752
            var _3X = $get($_.mscs, 0); //#17752
            var _3Z = $get($_.mscs, 1); //#17752
            var _3b = $get($_.mscs, 2); //#17752
            $put($_.out, $_.k * 5 + 2, $f(~~((~~((_3H + _3J * 316) / 900) + _3N + _3P * 641 + _3R * 316) / 900) + _3V + _3X * 20 + _3Z * 641 + _3b * 316) % 900); //#17753
            var _3f = $get($_.lscs, 3); //#17754
            var _3h = $get($_.mscs, 1); //#17754
            var _3j = $get($_.mscs, 2); //#17754
            $put($_.out, $_.k * 5 + 1, $f(~~((~~((~~((_3H + _3J * 316) / 900) + _3N + _3P * 641 + _3R * 316) / 900) + _3V + _3X * 20 + _3Z * 641 + _3b * 316) / 900) + _3f + _3h * 20 + _3j * 641) % 900); //#17755
            $put($_.out, $_.k * 5, $f(~~((~~((~~((~~((_3H + _3J * 316) / 900) + _3N + _3P * 641 + _3R * 316) / 900) + _3V + _3X * 20 + _3Z * 641 + _3b * 316) / 900) + _3f + _3h * 20 + _3j * 641) / 900) + $get($_.mscs, 2) * 20) % 900); //#17757
        } //#17757
        $_.rem = $_.inlen % 6; //#17759
        if ($_.rem != 0) { //#17764
            $k[$j++] = $_.out; //#17762
            $k[$j++] = $_.out.length - $_.rem; //#17762
            $k[$j++] = Infinity; //#17762
            $aload($geti($_.in, $_.inlen - $_.rem, $_.rem)); //#17762
            var _40 = $a(); //#17762
            var _41 = $k[--$j]; //#17763
            $puti($k[--$j], _41, _40); //#17763
        } //#17763
        $k[$j++] = $_.out; //#17765
    }; //#17765
    if ($_.ccc) { //#17774
        $_.barlen = $_.barcode.length; //#17769
        $_.datcws = $a((((~~($_.barlen / 6)) * 5) + ($_.barlen % 6)) + 2); //#17770
        $put($_.datcws, 0, 920); //#17771
        var _4C = (($_.barlen % 6) == 0) ? 924 : 901; //#17772
        $put($_.datcws, 1, _4C); //#17772
        $k[$j++] = $_.datcws; //#17773
        $k[$j++] = 2; //#17773
        $k[$j++] = Infinity; //#17773
        $forall($_.barcode); //#17773
        var _4F = $a(); //#17773
        $k[$j++] = _4F; //#17773
        $_.encb(); //#17773
        var _4G = $k[--$j]; //#17773
        var _4H = $k[--$j]; //#17773
        $puti($k[--$j], _4H, _4G); //#17773
    } //#17773
    if ((!$_.raw) && (!$_.ccc)) { //#18125
        var _4N = new Map([
            ["parse", $_.parse],
            ["parsefnc", $_.parsefnc],
            ["eci", true]
        ]); //#17782
        $_.fncvals = _4N; //#17784
        $k[$j++] = 'msg'; //#17785
        $k[$j++] = $_.barcode; //#17785
        $k[$j++] = $_.fncvals; //#17785
        bwipp_parseinput(); //#17785
        var _4Q = $k[--$j]; //#17785
        $_[$k[--$j]] = _4Q; //#17785
        $_.msglen = $_.msg.length; //#17786
        $k[$j++] = Infinity; //#17789
        for (var _4U = 0, _4V = $_.msglen; _4U < _4V; _4U++) { //#17789
            $k[$j++] = 0; //#17789
        } //#17789
        $k[$j++] = 0; //#17789
        $_.numdigits = $a(); //#17789
        $k[$j++] = Infinity; //#17790
        for (var _4Y = 0, _4Z = $_.msglen; _4Y < _4Z; _4Y++) { //#17790
            $k[$j++] = 0; //#17790
        } //#17790
        $k[$j++] = 0; //#17790
        $_.numtext = $a(); //#17790
        $k[$j++] = Infinity; //#17791
        for (var _4c = 0, _4d = $_.msglen; _4c < _4d; _4c++) { //#17791
            $k[$j++] = 0; //#17791
        } //#17791
        $k[$j++] = 0; //#17791
        $_.numbytes = $a(); //#17791
        $k[$j++] = Infinity; //#17792
        for (var _4g = 0, _4h = $_.msglen; _4g < _4h; _4g++) { //#17792
            $k[$j++] = 0; //#17792
        } //#17792
        $k[$j++] = 0; //#17792
        $_.iseci = $a(); //#17792
        for (var _4k = $_.msglen - 1; _4k >= 0; _4k -= 1) { //#17805
            $_.i = _4k; //#17794
            var _4n = $get($_.msg, $_.i); //#17795
            if ((_4n >= 48) && (_4n <= 57)) { //#17797
                $put($_.numdigits, $_.i, $get($_.numdigits, $_.i + 1) + 1); //#17796
            } //#17796
            var _4x = $get($_.alltext, $get($_.msg, $_.i)) !== undefined; //#17798
            if (_4x && ($get($_.numdigits, $_.i) < 13)) { //#17800
                $put($_.numtext, $_.i, $get($_.numtext, $_.i + 1) + 1); //#17799
            } //#17799
            if ((($get($_.msg, $_.i) >= 0) && ($get($_.numtext, $_.i) < 5)) && ($get($_.numdigits, $_.i) < 13)) { //#17803
                $put($_.numbytes, $_.i, $get($_.numbytes, $_.i + 1) + 1); //#17802
            } //#17802
            $put($_.iseci, $_.i, $get($_.msg, $_.i) <= -1000000); //#17804
        } //#17804
        $_.numdigits = $geti($_.numdigits, 0, $_.msglen); //#17806
        $_.numtext = $geti($_.numtext, 0, $_.msglen); //#17807
        $_.numbytes = $geti($_.numbytes, 0, $_.msglen); //#17808
        $_.seq = $a([]); //#17810
        $_.seqlen = 0; //#17810
        $_.state = $_.T; //#17810
        $_.p = 0; //#17810
        for (;;) { //#17864
            if ($_.p == $_.msglen) { //#17811
                break; //#17811
            } //#17811
            if ($get($_.iseci, $_.p)) { //#17862
                $_.eci = $get($_.msg, $_.p); //#17813
                $k[$j++] = Infinity; //#17814
                $aload($_.seq); //#17815
                $k[$j++] = $a([$_.eci]); //#17816
                $_.seq = $a(); //#17816
                $_.p = $_.p + 1; //#17818
                $k[$j++] = 'seqlen'; //#17819
                $k[$j++] = $_.seqlen; //#17819
                if ($_.eci <= -1810900) { //#17819
                    $k[$j++] = 2; //#17819
                } else { //#17819
                    var _5q = ($_.eci <= -1000900) ? 3 : 2; //#17819
                    $k[$j++] = _5q; //#17819
                } //#17819
                var _5r = $k[--$j]; //#17819
                var _5s = $k[--$j]; //#17819
                $_[$k[--$j]] = $f(_5s + _5r); //#17819
            } else { //#17862
                $_.n = $get($_.numdigits, $_.p); //#17821
                if (($_.n >= 13) || (($_.n == $_.msglen) && ($_.n >= 8))) { //#17862
                    $k[$j++] = Infinity; //#17823
                    $aload($_.seq); //#17824
                    $k[$j++] = $_.nl; //#17826
                    $k[$j++] = Infinity; //#17826
                    $aload($geti($_.msg, $_.p, $_.n)); //#17826
                    var _67 = $a(); //#17826
                    $k[$j++] = _67; //#17826
                    $_.seq = $a(); //#17826
                    $_.state = $_.N; //#17828
                    $_.p = $f($_.p + $_.n); //#17829
                    $_.seqlen = $f(($_.seqlen + 1) + $_.n); //#17830
                } else { //#17862
                    $_.t = $get($_.numtext, $_.p); //#17832
                    if ($_.t >= 5) { //#17862
                        $k[$j++] = Infinity; //#17834
                        $aload($_.seq); //#17835
                        if ($_.state != $_.T) { //#17838
                            $k[$j++] = $_.tl; //#17837
                        } //#17837
                        $k[$j++] = Infinity; //#17839
                        $aload($geti($_.msg, $_.p, $_.t)); //#17839
                        var _6Q = $a(); //#17839
                        $k[$j++] = _6Q; //#17839
                        $_.seq = $a(); //#17839
                        $_.state = $_.T; //#17841
                        $_.p = $f($_.p + $_.t); //#17842
                        $_.seqlen = $f(($_.seqlen + 1) + $_.t); //#17843
                    } else { //#17862
                        $_.b = $get($_.numbytes, $_.p); //#17845
                        if (($_.b == 1) && ($_.state == $_.T)) { //#17862
                            $k[$j++] = Infinity; //#17847
                            $aload($_.seq); //#17848
                            $k[$j++] = $_.bs; //#17850
                            $k[$j++] = $a([$get($_.msg, $_.p)]); //#17850
                            $_.seq = $a(); //#17850
                            $_.p = $f($_.p + $_.b); //#17852
                            $_.seqlen = $_.seqlen + 2; //#17853
                        } else { //#17862
                            $k[$j++] = Infinity; //#17855
                            $aload($_.seq); //#17856
                            var _6p = (($_.b % 6) != 0) ? $_.bl : $_.bl6; //#17857
                            $k[$j++] = _6p; //#17858
                            $k[$j++] = Infinity; //#17858
                            $aload($geti($_.msg, $_.p, $_.b)); //#17858
                            var _6u = $a(); //#17858
                            $k[$j++] = _6u; //#17858
                            $_.seq = $a(); //#17858
                            $_.state = $_.B; //#17860
                            $_.p = $f($_.p + $_.b); //#17861
                            $_.seqlen = $f(($_.seqlen + 1) + $_.b); //#17862
                        } //#17862
                    } //#17862
                } //#17862
            } //#17862
        } //#17862
        $_.enca = function() {
            var _75 = $get($get($_.charvals, $_.A), $k[--$j]); //#17867
            $k[$j++] = _75; //#17867
        }; //#17867
        $_.encl = function() {
            var _7A = $get($get($_.charvals, $_.L), $k[--$j]); //#17868
            $k[$j++] = _7A; //#17868
        }; //#17868
        $_.encm = function() {
            var _7F = $get($get($_.charvals, $_.M), $k[--$j]); //#17869
            $k[$j++] = _7F; //#17869
        }; //#17869
        $_.encp = function() {
            var _7K = $get($get($_.charvals, $_.P), $k[--$j]); //#17870
            $k[$j++] = _7K; //#17870
        }; //#17870
        $_.textencfuncs = $a(['enca', 'encl', 'encm', 'encp']); //#17871
        $_.addtotext = function() {
            $put($_.text, $_.l, $k[--$j]); //#17874
            $_.l = $_.l + 1; //#17875
        }; //#17875
        $_.enct = function() {
            $_.in = $k[--$j]; //#17879
            $_.curlen = $a([$_.e, $_.e, $_.e, $_.e]); //#17881
            $put($_.curlen, $_.submode, 0); //#17882
            $_.curseq = $a([$a([]), $a([]), $a([]), $a([])]); //#17883
            $forall($_.in, function() { //#17953
                $_.char = $k[--$j]; //#17888
                for (;;) { //#17909
                    $_.imp = false; //#17892
                    var _7j = $a([$_.A, $_.L, $_.M, $_.P]); //#17893
                    for (var _7k = 0, _7l = _7j.length; _7k < _7l; _7k++) { //#17907
                        $_.x = $get(_7j, _7k); //#17894
                        var _7r = $a([$_.A, $_.L, $_.M, $_.P]); //#17895
                        for (var _7s = 0, _7t = _7r.length; _7s < _7t; _7s++) { //#17906
                            $_.y = $get(_7r, _7s); //#17896
                            $_.cost = $f($get($_.curlen, $_.x) + $get($get($_.latlen, $_.x), $_.y)); //#17897
                            if ($_.cost < $get($_.curlen, $_.y)) { //#17905
                                $put($_.curlen, $_.y, $_.cost); //#17899
                                $k[$j++] = $_.curseq; //#17902
                                $k[$j++] = $_.y; //#17902
                                $k[$j++] = Infinity; //#17900
                                $aload($get($_.curseq, $_.x)); //#17901
                                $aload($get($get($_.latseq, $_.x), $_.y)); //#17902
                                var _8K = $a(); //#17902
                                var _8L = $k[--$j]; //#17903
                                $put($k[--$j], _8L, _8K); //#17903
                                $_.imp = true; //#17904
                            } //#17904
                        } //#17904
                    } //#17904
                    if (!$_.imp) { //#17908
                        break; //#17908
                    } //#17908
                } //#17908
                $_.nxtlen = $a([$_.e, $_.e, $_.e, $_.e]); //#17912
                $_.nxtseq = $a(4); //#17913
                var _8Y = $a([$_.A, $_.L, $_.M, $_.P]); //#17915
                for (var _8Z = 0, _8a = _8Y.length; _8Z < _8a; _8Z++) { //#17948
                    $_.x = $get(_8Y, _8Z); //#17916
                    for (;;) { //#17946
                        var _8g = $get($get($_.charvals, $_.x), $_.char) !== undefined; //#17920
                        if (!_8g) { //#17920
                            break; //#17920
                        } //#17920
                        $_.cost = $get($_.curlen, $_.x) + 1; //#17923
                        if ($_.cost < $get($_.nxtlen, $_.x)) { //#17927
                            $put($_.nxtlen, $_.x, $_.cost); //#17925
                            $k[$j++] = $_.nxtseq; //#17926
                            $k[$j++] = $_.x; //#17926
                            $k[$j++] = Infinity; //#17926
                            $aload($get($_.curseq, $_.x)); //#17926
                            $k[$j++] = $_.char; //#17926
                            var _8x = $a(); //#17926
                            var _8y = $k[--$j]; //#17926
                            $put($k[--$j], _8y, _8x); //#17926
                        } //#17926
                        var _94 = $a([$_.A, $_.L, $_.M, $_.P]); //#17930
                        for (var _95 = 0, _96 = _94.length; _95 < _96; _95++) { //#17943
                            $_.y = $get(_94, _95); //#17931
                            if ($ne($_.x, $_.y)) { //#17942
                                $_.cost = $f($get($_.curlen, $_.y) + $get($get($_.shftlen, $_.y), $_.x)) + 1; //#17933
                                if ($_.cost < $get($_.nxtlen, $_.y)) { //#17941
                                    $put($_.nxtlen, $_.y, $_.cost); //#17935
                                    $k[$j++] = $_.nxtseq; //#17939
                                    $k[$j++] = $_.y; //#17939
                                    $k[$j++] = Infinity; //#17936
                                    $aload($get($_.curseq, $_.y)); //#17937
                                    var _9W = ($_.x == $_.A) ? $_.as : $_.ps; //#17938
                                    $k[$j++] = _9W; //#17939
                                    $k[$j++] = $_.char; //#17939
                                    var _9Y = $a(); //#17939
                                    var _9Z = $k[--$j]; //#17940
                                    $put($k[--$j], _9Z, _9Y); //#17940
                                } //#17940
                            } //#17940
                        } //#17940
                        break; //#17945
                    } //#17945
                } //#17945
                $_.curlen = $_.nxtlen; //#17950
                $_.curseq = $_.nxtseq; //#17951
            }); //#17951
            $_.minseq = $_.e; //#17956
            var _9i = $a([$_.A, $_.L, $_.M, $_.P]); //#17957
            for (var _9j = 0, _9k = _9i.length; _9j < _9k; _9j++) { //#17963
                $_.k = $get(_9i, _9j); //#17958
                if ($get($_.curlen, $_.k) < $_.minseq) { //#17962
                    $_.minseq = $get($_.curlen, $_.k); //#17960
                    $_.txtseq = $get($_.curseq, $_.k); //#17961
                } //#17961
            } //#17961
            $_.text = $a($_.minseq); //#17966
            $_.k = 0; //#17967
            $_.l = 0; //#17967
            for (;;) { //#17987
                if ($_.k >= $_.txtseq.length) { //#17968
                    break; //#17968
                } //#17968
                $_.char = $get($_.txtseq, $_.k); //#17970
                $k[$j++] = $_.char; //#17973
                if ($_[$get($_.textencfuncs, $_.submode)]() === true) {
                    break;
                } //#17973
                $_.addtotext(); //#17973
                $_.k = $_.k + 1; //#17974
                if (($_.char == $_.as) || ($_.char == $_.ps)) { //#17980
                    $k[$j++] = $get($_.txtseq, $_.k); //#17978
                    if ($_.char == $_.as) { //#17978
                        $_.enca(); //#17978
                    } else { //#17978
                        $_.encp(); //#17978
                    } //#17978
                    $_.addtotext(); //#17978
                    $_.k = $_.k + 1; //#17979
                } //#17979
                if ($_.char == $_.al) { //#17983
                    $_.submode = $_.A; //#17983
                } //#17983
                if ($_.char == $_.ll) { //#17984
                    $_.submode = $_.L; //#17984
                } //#17984
                if ($_.char == $_.ml) { //#17985
                    $_.submode = $_.M; //#17985
                } //#17985
                if ($_.char == $_.pl) { //#17986
                    $_.submode = $_.P; //#17986
                } //#17986
            } //#17986
            if (($_.text.length % 2) == 1) { //#18034
                if ($_.submode == $_.P) { //#18030
                    $k[$j++] = 'pad'; //#17991
                    $k[$j++] = $_.al; //#17991
                    $_.encp(); //#17991
                    var _AZ = $k[--$j]; //#17991
                    $_[$k[--$j]] = _AZ; //#17991
                    $_.submode = $_.A; //#17992
                } else { //#18030
                    $_.nextecitext = false; //#17995
                    for (var _Af = $_.i + 1, _Ae = $_.seq.length - 1; _Af <= _Ae; _Af += 1) { //#18002
                        $_.ti = _Af; //#17998
                        if ($ne($type($get($_.seq, $_.ti)), 'arraytype')) { //#17999
                            $_.nextecitext = false; //#17999
                            break; //#17999
                        } //#17999
                        if ($get($get($_.seq, $_.ti), 0) > -1000000) { //#18000
                            break; //#18000
                        } //#18000
                        $_.nextecitext = true; //#18001
                    } //#18001
                    if ($_.nextecitext) { //#18030
                        if ($_.submode == $_.A) { //#18027
                            var _Ay = $get($get($_.charvals, $_.L), $get($get($_.seq, $_.ti), 0)) !== undefined; //#18005
                            if (_Ay) { //#18010
                                $k[$j++] = 'pad'; //#18006
                                $k[$j++] = $_.ll; //#18006
                                $_.enca(); //#18006
                                var _B0 = $k[--$j]; //#18006
                                $_[$k[--$j]] = _B0; //#18006
                                $_.submode = $_.L; //#18007
                            } else { //#18010
                                $k[$j++] = 'pad'; //#18009
                                $k[$j++] = $_.ml; //#18009
                                $_.enca(); //#18009
                                var _B4 = $k[--$j]; //#18009
                                $_[$k[--$j]] = _B4; //#18009
                                $_.submode = $_.M; //#18010
                            } //#18010
                        } else { //#18027
                            if ($_.submode == $_.M) { //#18027
                                var _BG = $get($get($_.charvals, $_.L), $get($get($_.seq, $_.ti), 0)) !== undefined; //#18014
                                if (_BG) { //#18023
                                    $k[$j++] = 'pad'; //#18015
                                    $k[$j++] = $_.ll; //#18015
                                    $_.encm(); //#18015
                                    var _BI = $k[--$j]; //#18015
                                    $_[$k[--$j]] = _BI; //#18015
                                    $_.submode = $_.L; //#18016
                                } else { //#18023
                                    var _BS = $get($get($_.charvals, $_.P), $get($get($_.seq, $_.ti), 0)) !== undefined; //#18018
                                    if (_BS) { //#18023
                                        $k[$j++] = 'pad'; //#18019
                                        $k[$j++] = $_.pl; //#18019
                                        $_.encm(); //#18019
                                        var _BU = $k[--$j]; //#18019
                                        $_[$k[--$j]] = _BU; //#18019
                                        $_.submode = $_.P; //#18020
                                    } else { //#18023
                                        $k[$j++] = 'pad'; //#18022
                                        $k[$j++] = $_.al; //#18022
                                        $_.encm(); //#18022
                                        var _BY = $k[--$j]; //#18022
                                        $_[$k[--$j]] = _BY; //#18022
                                        $_.submode = $_.A; //#18023
                                    } //#18023
                                } //#18023
                            } else { //#18027
                                $k[$j++] = 'pad'; //#18026
                                $k[$j++] = $_.ml; //#18026
                                $_.encl(); //#18026
                                var _Bc = $k[--$j]; //#18026
                                $_[$k[--$j]] = _Bc; //#18026
                                $_.submode = $_.M; //#18027
                            } //#18027
                        } //#18027
                    } else { //#18030
                        $k[$j++] = 'pad'; //#18030
                        $k[$j++] = $_.ps; //#18030
                        if ($_[$get($_.textencfuncs, $_.submode)]() === true) {
                            return true;
                        } //#18030
                        var _Bk = $k[--$j]; //#18030
                        $_[$k[--$j]] = _Bk; //#18030
                    } //#18030
                } //#18030
                $k[$j++] = Infinity; //#18033
                $aload($_.text); //#18033
                $k[$j++] = $_.pad; //#18033
                $_.text = $a(); //#18033
            } //#18033
            $_.out = $a(~~($_.text.length / 2)); //#18036
            for (var _Bt = 0, _Bs = $_.out.length - 1; _Bt <= _Bs; _Bt += 1) { //#18040
                $_.k = _Bt; //#18038
                $put($_.out, $_.k, $f(($get($_.text, $_.k * 2) * 30) + $get($_.text, ($_.k * 2) + 1))); //#18039
            } //#18039
            $k[$j++] = $_.out; //#18042
        }; //#18042
        $_.encn = function() {
            $_.in = $k[--$j]; //#18047
            $_.out = $a([]); //#18048
            for (var _C7 = 0, _C6 = $_.in.length - 1; _C7 <= _C6; _C7 += 44) { //#18076
                $_.k = _C7; //#18050
                $k[$j++] = Infinity; //#18051
                var _CC = $_.in.length - $_.k; //#18052
                $k[$j++] = 1; //#18052
                $k[$j++] = $_.in; //#18052
                $k[$j++] = $_.k; //#18052
                $k[$j++] = _CC; //#18052
                if (_CC > 44) { //#18052
                    $k[$j - 1] = 44; //#18052
                } //#18052
                var _CD = $k[--$j]; //#18052
                var _CE = $k[--$j]; //#18052
                var _CG = $geti($k[--$j], _CE, _CD); //#18052
                for (var _CH = 0, _CI = _CG.length; _CH < _CI; _CH++) { //#18053
                    $k[$j++] = $get(_CG, _CH) - 48; //#18053
                } //#18053
                $_.gmod = $a(); //#18053
                $_.cwn = $a([]); //#18055
                for (;;) { //#18074
                    $_.dv = 900; //#18056
                    $_.gmul = $a([]); //#18057
                    $_.val = 0; //#18057
                    for (;;) { //#18069
                        if ($_.gmod.length == 0) { //#18058
                            break; //#18058
                        } //#18058
                        $_.val = $f(($_.val * 10) + $get($_.gmod, 0)); //#18059
                        $_.gmod = $geti($_.gmod, 1, $_.gmod.length - 1); //#18060
                        if ($_.val < $_.dv) { //#18066
                            if ($_.gmul.length != 0) { //#18064
                                $k[$j++] = Infinity; //#18063
                                $aload($_.gmul); //#18063
                                $k[$j++] = 0; //#18063
                                $_.gmul = $a(); //#18063
                            } //#18063
                        } else { //#18066
                            $k[$j++] = Infinity; //#18066
                            $aload($_.gmul); //#18066
                            $k[$j++] = ~~($_.val / $_.dv); //#18066
                            $_.gmul = $a(); //#18066
                        } //#18066
                        $_.val = $_.val % $_.dv; //#18068
                    } //#18068
                    $_.dv = $_.val; //#18070
                    $k[$j++] = Infinity; //#18071
                    $k[$j++] = $_.dv; //#18071
                    $aload($_.cwn); //#18071
                    $_.cwn = $a(); //#18071
                    $_.gmod = $_.gmul; //#18072
                    if ($_.gmul.length == 0) { //#18073
                        break; //#18073
                    } //#18073
                } //#18073
                $k[$j++] = Infinity; //#18075
                $aload($_.out); //#18075
                $aload($_.cwn); //#18075
                $_.out = $a(); //#18075
            } //#18075
            $k[$j++] = $_.out; //#18077
        }; //#18077
        $_.ence = function() {
            var _Cr = (-$get($k[--$j], 0)) - 1000000; //#18082
            $k[$j++] = _Cr; //#18090
            if (_Cr <= 899) { //#18089
                var _Cs = $k[--$j]; //#18083
                $k[$j++] = 927; //#18083
                $k[$j++] = _Cs; //#18083
                $astore($a(2)); //#18083
            } else { //#18089
                var _Cu = $k[$j - 1]; //#18084
                if (_Cu <= 810899) { //#18089
                    var _Cv = $k[--$j]; //#18085
                    $k[$j++] = 926; //#18085
                    $k[$j++] = (~~(_Cv / 900)) - 1; //#18085
                    $k[$j++] = _Cv % 900; //#18085
                    $astore($a(3)); //#18085
                } else { //#18089
                    var _Cx = $k[$j - 1]; //#18086
                    if (_Cx <= 811799) { //#18089
                        var _Cy = $k[--$j]; //#18087
                        $k[$j++] = 925; //#18087
                        $k[$j++] = _Cy - 810900; //#18087
                        $astore($a(2)); //#18087
                    } else { //#18089
                        $k[$j++] = 'bwipp.pdf417badECI#18089'; //#18089
                        $k[$j++] = "PDF417 supports ECIs 000000 to 811799"; //#18089
                        bwipp_raiseerror(); //#18089
                    } //#18089
                } //#18089
            } //#18089
        }; //#18089
        $_.encfuncs = $a(['enct', 'encn', 'encb']); //#18093
        $_.addtocws = function() {
            var _D1 = $k[--$j]; //#18096
            $puti($_.datcws, $_.j, _D1); //#18096
            $_.j = _D1.length + $_.j; //#18097
        }; //#18097
        $_.state = $_.T; //#18101
        $_.submode = $_.A; //#18101
        $_.datcws = $a($_.seqlen); //#18102
        $_.i = 0; //#18103
        $_.j = 0; //#18103
        for (;;) { //#18123
            if ($_.i >= $_.seq.length) { //#18104
                break; //#18104
            } //#18104
            $_.chars = $get($_.seq, $_.i); //#18105
            if ($eq($type($_.chars), 'arraytype')) { //#18119
                if ($get($_.chars, 0) <= -1000000) { //#18110
                    $k[$j++] = $_.chars; //#18108
                    $_.ence(); //#18108
                    $_.addtocws(); //#18108
                } else { //#18110
                    $k[$j++] = $_.chars; //#18110
                    if ($_[$get($_.encfuncs, $_.state)]() === true) {
                        break;
                    } //#18110
                    $_.addtocws(); //#18110
                } //#18110
            } else { //#18119
                $k[$j++] = $a([$get($_.latchcws, $_.chars)]); //#18113
                $_.addtocws(); //#18113
                if ($_.chars == $_.tl) { //#18114
                    $_.state = $_.T; //#18114
                    $_.submode = $_.A; //#18114
                } //#18114
                if ($_.chars == $_.nl) { //#18115
                    $_.state = $_.N; //#18115
                } //#18115
                if (($_.chars == $_.bl) || ($_.chars == $_.bl6)) { //#18116
                    $_.state = $_.B; //#18116
                } //#18116
                if ($_.chars == $_.bs) { //#18120
                    $_.i = $_.i + 1; //#18118
                    $k[$j++] = $get($_.seq, $_.i); //#18119
                    $_.encb(); //#18119
                    $_.addtocws(); //#18119
                } //#18119
            } //#18119
            $_.i = $_.i + 1; //#18122
        } //#18122
        $_.datcws = $geti($_.datcws, 0, $_.j); //#18124
    } //#18124
    $_.m = $_.datcws.length; //#18126
    if ($_.m > 926) { //#18130
        $k[$j++] = 'bwipp.pdf417dataTooLong#18129'; //#18129
        $k[$j++] = "The data is too long"; //#18129
        bwipp_raiseerror(); //#18129
    } //#18129
    if ($_.eclevel == -1) { //#18138
        if ($_.m <= 40) { //#18134
            $_.eclevel = 2; //#18134
        } //#18134
        if (($_.m >= 41) && ($_.m <= 160)) { //#18135
            $_.eclevel = 3; //#18135
        } //#18135
        if (($_.m >= 161) && ($_.m <= 320)) { //#18136
            $_.eclevel = 4; //#18136
        } //#18136
        if ($_.m >= 321) { //#18137
            $_.eclevel = 5; //#18137
        } //#18137
    } //#18137
    if (!$_.fixedeclevel) { //#18144
        $_.maxeclevel = (~~(($log((928 - 1) - $_.m)) / $log(2))) - 1; //#18142
        if ($_.eclevel > $_.maxeclevel) { //#18143
            $_.eclevel = $_.maxeclevel; //#18143
        } //#18143
    } //#18143
    $_.k = ~~$pow(2, $_.eclevel + 1); //#18147
    if ($_.columns == 0) { //#18148
        $_.columns = ~~($round($sqrt(($_.m + $_.k) / 3))); //#18148
    } //#18148
    $k[$j++] = 'c'; //#18149
    if ($_.columns >= 1) { //#18149
        $k[$j++] = $_.columns; //#18149
    } else { //#18149
        $k[$j++] = 1; //#18149
    } //#18149
    var _E8 = $k[--$j]; //#18149
    $_[$k[--$j]] = _E8; //#18149
    $_.r = ~~($ceil((($_.m + $_.k) + 1) / $_.columns)); //#18152
    if (($_.r < $_.rows) && ($_.rows <= 90)) { //#18153
        $_.r = $_.rows; //#18153
    } //#18153
    if ($_.r < 3) { //#18154
        $_.r = 3; //#18154
    } //#18154
    if ($_.r > 90) { //#18159
        $k[$j++] = 'bwipp.pdf417insufficientCapacity#18158'; //#18158
        $k[$j++] = "Insufficient capacity in the symbol"; //#18158
        bwipp_raiseerror(); //#18158
    } //#18158
    if (!$_.fixedeclevel) { //#18168
        var _EK = $_.c; //#18163
        var _EL = $_.r; //#18163
        var _EM = $_.m; //#18163
        var _EN = 8; //#18163
        var _EO = (~~(($log($f(((_EK * _EL) - 1) - _EM))) / $log(2))) - 1; //#18163
        if (((~~(($log($f(((_EK * _EL) - 1) - _EM))) / $log(2))) - 1) > 8) { //#18163
            var _ = _EN; //#18163
            _EN = _EO; //#18163
            _EO = _; //#18163
        } //#18163
        $_.maxeclevel = _EO; //#18163
        if ($_.maxeclevel > $_.eclevel) { //#18167
            $_.eclevel = $_.maxeclevel; //#18165
            $_.k = ~~$pow(2, $_.eclevel + 1); //#18166
        } //#18166
    } //#18166
    $_.n = $f(($_.c * $_.r) - $_.k); //#18171
    $_.cws = $a(($_.c * $_.r) + 1); //#18172
    $put($_.cws, 0, $_.n); //#18173
    $puti($_.cws, 1, $_.datcws); //#18174
    $k[$j++] = $_.cws; //#18175
    $k[$j++] = $_.m + 1; //#18175
    $k[$j++] = Infinity; //#18175
    for (var _Eh = 0, _Ei = $f($_.n - $_.m) - 1; _Eh < _Ei; _Eh++) { //#18175
        $k[$j++] = 900; //#18175
    } //#18175
    var _Ej = $a(); //#18175
    var _Ek = $k[--$j]; //#18175
    $puti($k[--$j], _Ek, _Ej); //#18175
    $k[$j++] = $_.cws; //#18176
    $k[$j++] = $_.n; //#18176
    $k[$j++] = Infinity; //#18176
    for (var _Ep = 0, _Eq = $_.k; _Ep < _Eq; _Ep++) { //#18176
        $k[$j++] = 0; //#18176
    } //#18176
    $k[$j++] = 0; //#18176
    var _Er = $a(); //#18176
    var _Es = $k[--$j]; //#18176
    $puti($k[--$j], _Es, _Er); //#18176
    $k[$j++] = Infinity; //#18180
    $k[$j++] = 1; //#18180
    for (var _Eu = 0; _Eu < 928; _Eu++) { //#18180
        var _Ev = $k[$j - 1]; //#18180
        $k[$j++] = (_Ev * 3) % 929; //#18180
    } //#18180
    $_.rsalog = $a(); //#18180
    $_.rslog = $a(929); //#18181
    for (var _Ey = 1; _Ey <= 928; _Ey += 1) { //#18182
        $put($_.rslog, $get($_.rsalog, _Ey), _Ey); //#18182
    } //#18182
    $_.rsprod = function() {
        var _F3 = $k[$j - 1]; //#18187
        var _F4 = $k[$j - 2]; //#18187
        if ((_F3 != 0) && (_F4 != 0)) { //#18190
            var _F7 = $get($_.rslog, $k[--$j]); //#18188
            var _FC = $get($_.rsalog, $f(_F7 + $get($_.rslog, $k[--$j])) % 928); //#18188
            $k[$j++] = _FC; //#18188
        } else { //#18190
            $j -= 2; //#18190
            $k[$j++] = 0; //#18190
        } //#18190
    }; //#18190
    $k[$j++] = Infinity; //#18195
    $k[$j++] = 1; //#18195
    for (var _FE = 0, _FF = $_.k; _FE < _FF; _FE++) { //#18195
        $k[$j++] = 0; //#18195
    } //#18195
    $_.coeffs = $a(); //#18195
    for (var _FJ = 1, _FI = $_.k; _FJ <= _FI; _FJ += 1) { //#18204
        $_.i = _FJ; //#18197
        $put($_.coeffs, $_.i, $get($_.coeffs, $_.i - 1)); //#18198
        for (var _FQ = $_.i - 1; _FQ >= 1; _FQ -= 1) { //#18202
            $_.j = _FQ; //#18200
            $k[$j++] = $_.coeffs; //#18201
            $k[$j++] = $_.j; //#18201
            $k[$j++] = $get($_.coeffs, $_.j - 1); //#18201
            $k[$j++] = $get($_.coeffs, $_.j); //#18201
            $k[$j++] = $get($_.rsalog, $_.i); //#18201
            $_.rsprod(); //#18201
            var _Fc = $k[--$j]; //#18201
            var _Fd = $k[--$j]; //#18201
            var _Fe = $k[--$j]; //#18201
            $put($k[--$j], _Fe, $f(_Fd + _Fc) % 929); //#18201
        } //#18201
        $k[$j++] = $_.coeffs; //#18203
        $k[$j++] = 0; //#18203
        $k[$j++] = $get($_.coeffs, 0); //#18203
        $k[$j++] = $get($_.rsalog, $_.i); //#18203
        $_.rsprod(); //#18203
        var _Fm = $k[--$j]; //#18203
        var _Fn = $k[--$j]; //#18203
        $put($k[--$j], _Fn, _Fm); //#18203
    } //#18203
    $_.coeffs = $geti($_.coeffs, 0, $_.coeffs.length - 1); //#18205
    for (var _Ft = $_.coeffs.length - 1; _Ft >= 0; _Ft -= 2) { //#18206
        var _Fu = $_.coeffs; //#18206
        $put(_Fu, _Ft, 929 - $get(_Fu, _Ft)); //#18206
    } //#18206
    for (var _Fy = 0, _Fx = $_.n - 1; _Fy <= _Fx; _Fy += 1) { //#18215
        $_.t = $f($get($_.cws, _Fy) + $get($_.cws, $_.n)) % 929; //#18210
        for (var _G6 = 0, _G5 = $_.k - 1; _G6 <= _G5; _G6 += 1) { //#18214
            $_.j = _G6; //#18212
            $put($_.cws, $f($_.n + $_.j), $f($get($_.cws, $f($_.n + $_.j) + 1) + (929 - $_.t * $get($_.coeffs, $_.k - $_.j - 1) % 929)) % 929); //#18213
        } //#18213
    } //#18213
    for (var _GN = $_.n, _GM = $f($_.n + $_.k); _GN <= _GM; _GN += 1) { //#18216
        $put($_.cws, _GN, (929 - $get($_.cws, _GN)) % 929); //#18216
    } //#18216
    $_.cws = $geti($_.cws, 0, $_.cws.length - 1); //#18219
    var _GV = $get($_.options, 'debugcws') !== undefined; //#18221
    if (_GV) { //#18221
        $k[$j++] = 'bwipp.debugcws#18221'; //#18221
        $k[$j++] = $_.cws; //#18221
        bwipp_raiseerror(); //#18221
    } //#18221
    $_.clusters = $a([$a([120256, 125680, 128380, 120032, 125560, 128318, 108736, 119920, 108640, 86080, 108592, 86048, 110016, 120560, 125820, 109792, 120440, 125758, 88256, 109680, 88160, 89536, 110320, 120700, 89312, 110200, 120638, 89200, 110140, 89840, 110460, 89720, 110398, 89980, 128506, 119520, 125304, 128190, 107712, 119408, 125244, 107616, 119352, 84032, 107568, 119324, 84000, 107544, 83984, 108256, 119672, 125374, 85184, 108144, 119612, 85088, 108088, 119582, 85040, 108060, 85728, 108408, 119742, 85616, 108348, 85560, 108318, 85880, 108478, 85820, 85790, 107200, 119152, 125116, 107104, 119096, 125086, 83008, 107056, 119068, 82976, 107032, 82960, 82952, 83648, 107376, 119228, 83552, 107320, 119198, 83504, 107292, 83480, 83468, 83824, 107452, 83768, 107422, 83740, 83900, 106848, 118968, 125022, 82496, 106800, 118940, 82464, 106776, 118926, 82448, 106764, 82440, 106758, 82784, 106936, 119006, 82736, 106908, 82712, 106894, 82700, 82694, 106974, 82830, 82240, 106672, 118876, 82208, 106648, 118862, 82192, 106636, 82184, 106630, 82180, 82352, 82328, 82316, 82080, 118830, 106572, 106566, 82050, 117472, 124280, 127678, 103616, 117360, 124220, 103520, 117304, 124190, 75840, 103472, 75808, 104160, 117624, 124350, 76992, 104048, 117564, 76896, 103992, 76848, 76824, 77536, 104312, 117694, 77424, 104252, 77368, 77340, 77688, 104382, 77628, 77758, 121536, 126320, 128700, 121440, 126264, 128670, 111680, 121392, 126236, 111648, 121368, 126222, 111632, 121356, 103104, 117104, 124092, 112320, 103008, 117048, 124062, 112224, 121656, 126366, 93248, 74784, 102936, 117006, 93216, 112152, 93200, 75456, 103280, 117180, 93888, 75360, 103224, 117150, 93792, 112440, 121758, 93744, 75288, 93720, 75632, 103356, 94064, 75576, 103326, 94008, 112542, 93980, 75708, 94140, 75678, 94110, 121184, 126136, 128606, 111168, 121136, 126108, 111136, 121112, 126094, 111120, 121100, 111112, 111108, 102752, 116920, 123998, 111456, 102704, 116892, 91712, 74272, 121244, 116878, 91680, 74256, 102668, 91664, 111372, 102662, 74244, 74592, 102840, 116958, 92000, 74544, 102812, 91952, 111516, 102798, 91928, 74508, 74502, 74680, 102878, 92088, 74652, 92060, 74638, 92046, 92126, 110912, 121008, 126044, 110880, 120984, 126030, 110864, 120972, 110856, 120966, 110852, 110850, 74048, 102576, 116828, 90944, 74016, 102552, 116814, 90912, 111000, 121038, 90896, 73992, 102534, 90888, 110982, 90884, 74160, 102620, 91056, 74136, 102606, 91032, 111054, 91020, 74118, 91014, 91100, 91086, 110752, 120920, 125998, 110736, 120908, 110728, 120902, 110724, 110722, 73888, 102488, 116782, 90528, 73872, 102476, 90512, 110796, 102470, 90504, 73860, 90500, 73858, 73944, 90584, 90572, 90566, 120876, 120870, 110658, 102444, 73800, 90312, 90308, 90306, 101056, 116080, 123580, 100960, 116024, 70720, 100912, 115996, 70688, 100888, 70672, 70664, 71360, 101232, 116156, 71264, 101176, 116126, 71216, 101148, 71192, 71180, 71536, 101308, 71480, 101278, 71452, 71612, 71582, 118112, 124600, 127838, 105024, 118064, 124572, 104992, 118040, 124558, 104976, 118028, 104968, 118022, 100704, 115896, 123486, 105312, 100656, 115868, 79424, 70176, 118172, 115854, 79392, 105240, 100620, 79376, 70152, 79368, 70496, 100792, 115934, 79712, 70448, 118238, 79664, 105372, 100750, 79640, 70412, 79628, 70584, 100830, 79800, 70556, 79772, 70542, 70622, 79838, 122176, 126640, 128860, 122144, 126616, 128846, 122128, 126604, 122120, 126598, 122116, 104768, 117936, 124508, 113472, 104736, 126684, 124494, 113440, 122264, 126670, 113424, 104712, 117894, 113416, 122246, 104706, 69952, 100528, 115804, 78656, 69920, 100504, 115790, 96064, 78624, 104856, 117966, 96032, 113560, 122318, 100486, 96016, 78600, 104838, 96008, 69890, 70064, 100572, 78768, 70040, 100558, 96176, 78744, 104910, 96152, 113614, 70022, 78726, 70108, 78812, 70094, 96220, 78798, 122016, 126552, 128814, 122000, 126540, 121992, 126534, 121988, 121986, 104608, 117848, 124462, 113056, 104592, 126574, 113040, 122060, 117830, 113032, 104580, 113028, 104578, 113026, 69792, 100440, 115758, 78240, 69776, 100428, 95136, 78224, 104652, 100422, 95120, 113100, 69764, 95112, 78212, 69762, 78210, 69848, 100462, 78296, 69836, 95192, 78284, 69830, 95180, 78278, 69870, 95214, 121936, 126508, 121928, 126502, 121924, 121922, 104528, 117804, 112848, 104520, 117798, 112840, 121958, 112836, 104514, 112834, 69712, 100396, 78032, 69704, 100390, 94672, 78024, 104550, 94664, 112870, 69698, 94660, 78018, 94658, 78060, 94700, 94694, 126486, 121890, 117782, 104484, 104482, 69672, 77928, 94440, 69666, 77922, 99680, 68160, 99632, 68128, 99608, 115342, 68112, 99596, 68104, 99590, 68448, 99768, 115422, 68400, 99740, 68376, 99726, 68364, 68358, 68536, 99806, 68508, 68494, 68574, 101696, 116400, 123740, 101664, 116376, 101648, 116364, 101640, 116358, 101636, 67904, 99504, 115292, 72512, 67872, 116444, 115278, 72480, 101784, 116430, 72464, 67848, 99462, 72456, 101766, 67842, 68016, 99548, 72624, 67992, 99534, 72600, 101838, 72588, 67974, 68060, 72668, 68046, 72654, 118432, 124760, 127918, 118416, 124748, 118408, 124742, 118404, 118402, 101536, 116312, 105888, 101520, 116300, 105872, 118476, 116294, 105864, 101508, 105860, 101506, 105858, 67744, 99416, 72096, 67728, 116334, 80800, 72080, 101580, 99398, 80784, 105932, 67716, 80776, 72068, 67714, 72066, 67800, 99438, 72152, 67788, 80856, 72140, 67782, 80844, 72134, 67822, 72174, 80878, 126800, 128940, 126792, 128934, 126788, 126786, 118352, 124716, 122576, 126828, 124710, 122568, 126822, 122564, 118338, 122562, 101456, 116268, 105680, 101448, 116262, 114128, 105672, 118374, 114120, 122598, 101442, 114116, 105666, 114114, 67664, 99372, 71888, 67656, 99366, 80336, 71880, 101478, 97232, 80328, 105702, 67650, 97224, 114150, 71874, 97220, 67692, 71916, 67686, 80364, 71910, 97260, 80358, 97254, 126760, 128918, 126756, 126754, 118312, 124694, 122472, 126774, 122468, 118306, 122466, 101416, 116246, 105576, 101412, 113896, 105572, 101410, 113892, 105570, 113890, 67624, 99350, 71784, 101430, 80104, 71780, 67618, 96744, 80100, 71778, 96740, 80098, 96738, 71798, 96758, 126738, 122420, 122418, 105524, 113780, 113778, 71732, 79988, 96500, 96498, 66880, 66848, 98968, 66832, 66824, 66820, 66992, 66968, 66956, 66950, 67036, 67022, 100000, 99984, 115532, 99976, 115526, 99972, 99970, 66720, 98904, 69024, 100056, 98892, 69008, 100044, 69000, 100038, 68996, 66690, 68994, 66776, 98926, 69080, 100078, 69068, 66758, 69062, 66798, 69102, 116560, 116552, 116548, 116546, 99920, 102096, 116588, 115494, 102088, 116582, 102084, 99906, 102082, 66640, 68816, 66632, 98854, 73168, 68808, 66628, 73160, 68804, 66626, 73156, 68802, 66668, 68844, 66662, 73196, 68838, 73190, 124840, 124836, 124834, 116520, 118632, 124854, 118628, 116514, 118626, 99880, 115478, 101992, 116534, 106216, 101988, 99874, 106212, 101986, 106210, 66600, 98838, 68712, 99894, 72936, 68708, 66594, 81384, 72932, 68706, 81380, 72930, 66614, 68726, 72950, 81398, 128980, 128978, 124820, 126900, 124818, 126898, 116500, 118580, 116498, 122740, 118578, 122738, 99860, 101940, 99858, 106100, 101938, 114420]), $a([128352, 129720, 125504, 128304, 129692, 125472, 128280, 129678, 125456, 128268, 125448, 128262, 125444, 125792, 128440, 129758, 120384, 125744, 128412, 120352, 125720, 128398, 120336, 125708, 120328, 125702, 120324, 120672, 125880, 128478, 110144, 120624, 125852, 110112, 120600, 125838, 110096, 120588, 110088, 120582, 110084, 110432, 120760, 125918, 89664, 110384, 120732, 89632, 110360, 120718, 89616, 110348, 89608, 110342, 89952, 110520, 120798, 89904, 110492, 89880, 110478, 89868, 90040, 110558, 90012, 89998, 125248, 128176, 129628, 125216, 128152, 129614, 125200, 128140, 125192, 128134, 125188, 125186, 119616, 125360, 128220, 119584, 125336, 128206, 119568, 125324, 119560, 125318, 119556, 119554, 108352, 119728, 125404, 108320, 119704, 125390, 108304, 119692, 108296, 119686, 108292, 108290, 85824, 108464, 119772, 85792, 108440, 119758, 85776, 108428, 85768, 108422, 85764, 85936, 108508, 85912, 108494, 85900, 85894, 85980, 85966, 125088, 128088, 129582, 125072, 128076, 125064, 128070, 125060, 125058, 119200, 125144, 128110, 119184, 125132, 119176, 125126, 119172, 119170, 107424, 119256, 125166, 107408, 119244, 107400, 119238, 107396, 107394, 83872, 107480, 119278, 83856, 107468, 83848, 107462, 83844, 83842, 83928, 107502, 83916, 83910, 83950, 125008, 128044, 125000, 128038, 124996, 124994, 118992, 125036, 118984, 125030, 118980, 118978, 106960, 119020, 106952, 119014, 106948, 106946, 82896, 106988, 82888, 106982, 82884, 82882, 82924, 82918, 124968, 128022, 124964, 124962, 118888, 124982, 118884, 118882, 106728, 118902, 106724, 106722, 82408, 106742, 82404, 82402, 124948, 124946, 118836, 118834, 106612, 106610, 124224, 127664, 129372, 124192, 127640, 129358, 124176, 127628, 124168, 127622, 124164, 124162, 117568, 124336, 127708, 117536, 124312, 127694, 117520, 124300, 117512, 124294, 117508, 117506, 104256, 117680, 124380, 104224, 117656, 124366, 104208, 117644, 104200, 117638, 104196, 104194, 77632, 104368, 117724, 77600, 104344, 117710, 77584, 104332, 77576, 104326, 77572, 77744, 104412, 77720, 104398, 77708, 77702, 77788, 77774, 128672, 129880, 93168, 128656, 129868, 92664, 128648, 129862, 92412, 128644, 128642, 124064, 127576, 129326, 126368, 124048, 129902, 126352, 128716, 127558, 126344, 124036, 126340, 124034, 126338, 117152, 124120, 127598, 121760, 117136, 124108, 121744, 126412, 124102, 121736, 117124, 121732, 117122, 121730, 103328, 117208, 124142, 112544, 103312, 117196, 112528, 121804, 117190, 112520, 103300, 112516, 103298, 112514, 75680, 103384, 117230, 94112, 75664, 103372, 94096, 112588, 103366, 94088, 75652, 94084, 75650, 75736, 103406, 94168, 75724, 94156, 75718, 94150, 75758, 128592, 129836, 91640, 128584, 129830, 91388, 128580, 91262, 128578, 123984, 127532, 126160, 123976, 127526, 126152, 128614, 126148, 123970, 126146, 116944, 124012, 121296, 116936, 124006, 121288, 126182, 121284, 116930, 121282, 102864, 116972, 111568, 102856, 116966, 111560, 121318, 111556, 102850, 111554, 74704, 102892, 92112, 74696, 102886, 92104, 111590, 92100, 74690, 92098, 74732, 92140, 74726, 92134, 128552, 129814, 90876, 128548, 90750, 128546, 123944, 127510, 126056, 128566, 126052, 123938, 126050, 116840, 123958, 121064, 116836, 121060, 116834, 121058, 102632, 116854, 111080, 121078, 111076, 102626, 111074, 74216, 102646, 91112, 74212, 91108, 74210, 91106, 74230, 91126, 128532, 90494, 128530, 123924, 126004, 123922, 126002, 116788, 120948, 116786, 120946, 102516, 110836, 102514, 110834, 73972, 90612, 73970, 90610, 128522, 123914, 125978, 116762, 120890, 102458, 110714, 123552, 127320, 129198, 123536, 127308, 123528, 127302, 123524, 123522, 116128, 123608, 127342, 116112, 123596, 116104, 123590, 116100, 116098, 101280, 116184, 123630, 101264, 116172, 101256, 116166, 101252, 101250, 71584, 101336, 116206, 71568, 101324, 71560, 101318, 71556, 71554, 71640, 101358, 71628, 71622, 71662, 127824, 129452, 79352, 127816, 129446, 79100, 127812, 78974, 127810, 123472, 127276, 124624, 123464, 127270, 124616, 127846, 124612, 123458, 124610, 115920, 123500, 118224, 115912, 123494, 118216, 124646, 118212, 115906, 118210, 100816, 115948, 105424, 100808, 115942, 105416, 118246, 105412, 100802, 105410, 70608, 100844, 79824, 70600, 100838, 79816, 105446, 79812, 70594, 79810, 70636, 79852, 70630, 79846, 129960, 95728, 113404, 129956, 95480, 113278, 129954, 95356, 95294, 127784, 129430, 78588, 128872, 129974, 95996, 78462, 128868, 127778, 95870, 128866, 123432, 127254, 124520, 123428, 126696, 128886, 123426, 126692, 124514, 126690, 115816, 123446, 117992, 115812, 122344, 117988, 115810, 122340, 117986, 122338, 100584, 115830, 104936, 100580, 113640, 104932, 100578, 113636, 104930, 113634, 70120, 100598, 78824, 70116, 96232, 78820, 70114, 96228, 78818, 96226, 70134, 78838, 129940, 94968, 113022, 129938, 94844, 94782, 127764, 78206, 128820, 127762, 95102, 128818, 123412, 124468, 123410, 126580, 124466, 126578, 115764, 117876, 115762, 122100, 117874, 122098, 100468, 104692, 100466, 113140, 104690, 113138, 69876, 78324, 69874, 95220, 78322, 95218, 129930, 94588, 94526, 127754, 128794, 123402, 124442, 126522, 115738, 117818, 121978, 100410, 104570, 112890, 69754, 78074, 94714, 94398, 123216, 127148, 123208, 127142, 123204, 123202, 115408, 123244, 115400, 123238, 115396, 115394, 99792, 115436, 99784, 115430, 99780, 99778, 68560, 99820, 68552, 99814, 68548, 68546, 68588, 68582, 127400, 129238, 72444, 127396, 72318, 127394, 123176, 127126, 123752, 123172, 123748, 123170, 123746, 115304, 123190, 116456, 115300, 116452, 115298, 116450, 99560, 115318, 101864, 99556, 101860, 99554, 101858, 68072, 99574, 72680, 68068, 72676, 68066, 72674, 68086, 72694, 129492, 80632, 105854, 129490, 80508, 80446, 127380, 72062, 127924, 127378, 80766, 127922, 123156, 123700, 123154, 124788, 123698, 124786, 115252, 116340, 115250, 118516, 116338, 118514, 99444, 101620, 99442, 105972, 101618, 105970, 67828, 72180, 67826, 80884, 72178, 80882, 97008, 114044, 96888, 113982, 96828, 96798, 129482, 80252, 130010, 97148, 80190, 97086, 127370, 127898, 128954, 123146, 123674, 124730, 126842, 115226, 116282, 118394, 122618, 99386, 101498, 105722, 114170, 67706, 71930, 80378, 96632, 113854, 96572, 96542, 80062, 96702, 96444, 96414, 96350, 123048, 123044, 123042, 115048, 123062, 115044, 115042, 99048, 115062, 99044, 99042, 67048, 99062, 67044, 67042, 67062, 127188, 68990, 127186, 123028, 123316, 123026, 123314, 114996, 115572, 114994, 115570, 98932, 100084, 98930, 100082, 66804, 69108, 66802, 69106, 129258, 73084, 73022, 127178, 127450, 123018, 123290, 123834, 114970, 115514, 116602, 98874, 99962, 102138, 66682, 68858, 73210, 81272, 106174, 81212, 81182, 72894, 81342, 97648, 114364, 97592, 114334, 97564, 97550, 81084, 97724, 81054, 97694, 97464, 114270, 97436, 97422, 80990, 97502, 97372, 97358, 97326, 114868, 114866, 98676, 98674, 66292, 66290, 123098, 114842, 115130, 98618, 99194, 66170, 67322, 69310, 73404, 73374, 81592, 106334, 81564, 81550, 73310, 81630, 97968, 114524, 97944, 114510, 97932, 97926, 81500, 98012, 81486, 97998, 97880, 114478, 97868, 97862, 81454, 97902, 97836, 97830, 69470, 73564, 73550, 81752, 106414, 81740, 81734, 73518, 81774, 81708, 81702]), $a([109536, 120312, 86976, 109040, 120060, 86496, 108792, 119934, 86256, 108668, 86136, 129744, 89056, 110072, 129736, 88560, 109820, 129732, 88312, 109694, 129730, 88188, 128464, 129772, 89592, 128456, 129766, 89340, 128452, 89214, 128450, 125904, 128492, 125896, 128486, 125892, 125890, 120784, 125932, 120776, 125926, 120772, 120770, 110544, 120812, 110536, 120806, 110532, 84928, 108016, 119548, 84448, 107768, 119422, 84208, 107644, 84088, 107582, 84028, 129640, 85488, 108284, 129636, 85240, 108158, 129634, 85116, 85054, 128232, 129654, 85756, 128228, 85630, 128226, 125416, 128246, 125412, 125410, 119784, 125430, 119780, 119778, 108520, 119798, 108516, 108514, 83424, 107256, 119166, 83184, 107132, 83064, 107070, 83004, 82974, 129588, 83704, 107390, 129586, 83580, 83518, 128116, 83838, 128114, 125172, 125170, 119284, 119282, 107508, 107506, 82672, 106876, 82552, 106814, 82492, 82462, 129562, 82812, 82750, 128058, 125050, 119034, 82296, 106686, 82236, 82206, 82366, 82108, 82078, 76736, 103920, 117500, 76256, 103672, 117374, 76016, 103548, 75896, 103486, 75836, 129384, 77296, 104188, 129380, 77048, 104062, 129378, 76924, 76862, 127720, 129398, 77564, 127716, 77438, 127714, 124392, 127734, 124388, 124386, 117736, 124406, 117732, 117730, 104424, 117750, 104420, 104418, 112096, 121592, 126334, 92608, 111856, 121468, 92384, 111736, 121406, 92272, 111676, 92216, 111646, 92188, 75232, 103160, 117118, 93664, 74992, 103036, 93424, 112252, 102974, 93304, 74812, 93244, 74782, 93214, 129332, 75512, 103294, 129908, 129330, 93944, 75388, 129906, 93820, 75326, 93758, 127604, 75646, 128756, 127602, 94078, 128754, 124148, 126452, 124146, 126450, 117236, 121844, 117234, 121842, 103412, 103410, 91584, 111344, 121212, 91360, 111224, 121150, 91248, 111164, 91192, 111134, 91164, 91150, 74480, 102780, 91888, 74360, 102718, 91768, 111422, 91708, 74270, 91678, 129306, 74620, 129850, 92028, 74558, 91966, 127546, 128634, 124026, 126202, 116986, 121338, 102906, 90848, 110968, 121022, 90736, 110908, 90680, 110878, 90652, 90638, 74104, 102590, 91000, 74044, 90940, 74014, 90910, 74174, 91070, 90480, 110780, 90424, 110750, 90396, 90382, 73916, 90556, 73886, 90526, 90296, 110686, 90268, 90254, 73822, 90334, 90204, 90190, 71136, 101112, 116094, 70896, 100988, 70776, 100926, 70716, 70686, 129204, 71416, 101246, 129202, 71292, 71230, 127348, 71550, 127346, 123636, 123634, 116212, 116210, 101364, 101362, 79296, 105200, 118140, 79072, 105080, 118078, 78960, 105020, 78904, 104990, 78876, 78862, 70384, 100732, 79600, 70264, 100670, 79480, 105278, 79420, 70174, 79390, 129178, 70524, 129466, 79740, 70462, 79678, 127290, 127866, 123514, 124666, 115962, 118266, 100858, 113376, 122232, 126654, 95424, 113264, 122172, 95328, 113208, 122142, 95280, 113180, 95256, 113166, 95244, 78560, 104824, 117950, 95968, 78448, 104764, 95856, 113468, 104734, 95800, 78364, 95772, 78350, 95758, 70008, 100542, 78712, 69948, 96120, 78652, 69918, 96060, 78622, 96030, 70078, 78782, 96190, 94912, 113008, 122044, 94816, 112952, 122014, 94768, 112924, 94744, 112910, 94732, 94726, 78192, 104636, 95088, 78136, 104606, 95032, 113054, 95004, 78094, 94990, 69820, 78268, 69790, 95164, 78238, 95134, 94560, 112824, 121950, 94512, 112796, 94488, 112782, 94476, 94470, 78008, 104542, 94648, 77980, 94620, 77966, 94606, 69726, 78046, 94686, 94384, 112732, 94360, 112718, 94348, 94342, 77916, 94428, 77902, 94414, 94296, 112686, 94284, 94278, 77870, 94318, 94252, 94246, 68336, 99708, 68216, 99646, 68156, 68126, 68476, 68414, 127162, 123258, 115450, 99834, 72416, 101752, 116414, 72304, 101692, 72248, 101662, 72220, 72206, 67960, 99518, 72568, 67900, 72508, 67870, 72478, 68030, 72638, 80576, 105840, 118460, 80480, 105784, 118430, 80432, 105756, 80408, 105742, 80396, 80390, 72048, 101564, 80752, 71992, 101534, 80696, 71964, 80668, 71950, 80654, 67772, 72124, 67742, 80828, 72094, 80798, 114016, 122552, 126814, 96832, 113968, 122524, 96800, 113944, 122510, 96784, 113932, 96776, 113926, 96772, 80224, 105656, 118366, 97120, 80176, 105628, 97072, 114076, 105614, 97048, 80140, 97036, 80134, 97030, 71864, 101470, 80312, 71836, 97208, 80284, 71822, 97180, 80270, 97166, 67678, 71902, 80350, 97246, 96576, 113840, 122460, 96544, 113816, 122446, 96528, 113804, 96520, 113798, 96516, 96514, 80048, 105564, 96688, 80024, 105550, 96664, 113870, 96652, 80006, 96646, 71772, 80092, 71758, 96732, 80078, 96718, 96416, 113752, 122414, 96400, 113740, 96392, 113734, 96388, 96386, 79960, 105518, 96472, 79948, 96460, 79942, 96454, 71726, 79982, 96494, 96336, 113708, 96328, 113702, 96324, 96322, 79916, 96364, 79910, 96358, 96296, 113686, 96292, 96290, 79894, 96310, 66936, 99006, 66876, 66846, 67006, 68976, 100028, 68920, 99998, 68892, 68878, 66748, 69052, 66718, 69022, 73056, 102072, 116574, 73008, 102044, 72984, 102030, 72972, 72966, 68792, 99934, 73144, 68764, 73116, 68750, 73102, 66654, 68830, 73182, 81216, 106160, 118620, 81184, 106136, 118606, 81168, 106124, 81160, 106118, 81156, 81154, 72880, 101980, 81328, 72856, 101966, 81304, 106190, 81292, 72838, 81286, 68700, 72924, 68686, 81372, 72910, 81358, 114336, 122712, 126894, 114320, 122700, 114312, 122694, 114308, 114306, 81056, 106072, 118574, 97696, 81040, 106060, 97680, 114380, 106054, 97672, 81028, 97668, 81026, 97666, 72792, 101934, 81112, 72780, 97752, 81100, 72774, 97740, 81094, 97734, 68654, 72814, 81134, 97774, 114256, 122668, 114248, 122662, 114244, 114242, 80976, 106028, 97488, 80968, 106022, 97480, 114278, 97476, 80962, 97474, 72748, 81004, 72742, 97516, 80998, 97510, 114216, 122646, 114212, 114210, 80936, 106006, 97384, 80932, 97380, 80930, 97378, 72726, 80950, 97398, 114196, 114194, 80916, 97332, 80914, 97330, 66236, 66206, 67256, 99166, 67228, 67214, 66142, 67294, 69296, 100188, 69272, 100174, 69260, 69254, 67164, 69340, 67150, 69326, 73376, 102232, 116654, 73360, 102220, 73352, 102214, 73348, 73346, 69208, 100142, 73432, 102254, 73420, 69190, 73414, 67118, 69230, 73454, 106320, 118700, 106312, 118694, 106308, 106306, 73296, 102188, 81616, 106348, 102182, 81608, 73284, 81604, 73282, 81602, 69164, 73324, 69158, 81644, 73318, 81638, 122792, 126934, 122788, 122786, 106280, 118678, 114536, 106276, 114532, 106274, 114530, 73256, 102166, 81512, 73252, 98024, 81508, 73250, 98020, 81506, 98018, 69142, 73270, 81526, 98038, 122772, 122770, 106260, 114484, 106258, 114482, 73236, 81460, 73234, 97908, 81458, 97906, 122762, 106250, 114458, 73226, 81434, 97850, 66396, 66382, 67416, 99246, 67404, 67398, 66350, 67438, 69456, 100268, 69448, 100262, 69444, 69442, 67372, 69484, 67366, 69478, 102312, 116694, 102308, 102306, 69416, 100246, 73576, 102326, 73572, 69410, 73570, 67350, 69430, 73590, 118740, 118738, 102292, 106420, 102290, 106418, 69396, 73524, 69394, 81780, 73522, 81778, 118730, 102282, 106394, 69386, 73498, 81722, 66476, 66470, 67496, 99286, 67492, 67490, 66454, 67510, 100308, 100306, 67476, 69556, 67474, 69554, 116714])]); //#18464
    $_.cwtobits = function() {
        var _Ge = $get($_.clusters, $k[--$j]); //#18469
        $_.v = $get(_Ge, $k[--$j]); //#18469
        $k[$j++] = Infinity; //#18470
        for (var _Gh = 0; _Gh < 17; _Gh++) { //#18470
            $k[$j++] = 0; //#18470
        } //#18470
        var _Gk = $cvrs($s(17), $_.v, 2); //#18470
        for (var _Gl = 0, _Gm = _Gk.length; _Gl < _Gm; _Gl++) { //#18470
            $k[$j++] = $get(_Gk, _Gl) - 48; //#18470
        } //#18470
        var _Go = $a(); //#18470
        $k[$j++] = $geti(_Go, _Go.length - 17, 17); //#18471
    }; //#18471
    if ($_.compact) { //#18478
        $_.rwid = (((17 * $_.c) + 17) + 17) + 1; //#18476
    } else { //#18478
        $_.rwid = ((((17 * $_.c) + 17) + 17) + 17) + 18; //#18478
    } //#18478
    $_.pixs = $a(($_.rwid * $_.r) * $_.rowmult); //#18480
    for (var _Gz = 0, _Gy = $_.r - 1; _Gz <= _Gy; _Gz += 1) { //#18513
        $_.i = _Gz; //#18483
        if (($_.i % 3) == 0) { //#18488
            $_.lcw = ((~~($_.i / 3)) * 30) + (~~(($_.r - 1) / 3)); //#18486
            $_.rcw = ($f(((~~($_.i / 3)) * 30) + $_.c)) - 1; //#18487
        } //#18487
        if (($_.i % 3) == 1) { //#18492
            $_.lcw = (((~~($_.i / 3)) * 30) + ($_.eclevel * 3)) + (($_.r - 1) % 3); //#18490
            $_.rcw = ((~~($_.i / 3)) * 30) + (~~(($_.r - 1) / 3)); //#18491
        } //#18491
        if (($_.i % 3) == 2) { //#18496
            $_.lcw = ($f(((~~($_.i / 3)) * 30) + $_.c)) - 1; //#18494
            $_.rcw = (((~~($_.i / 3)) * 30) + ($_.eclevel * 3)) + (($_.r - 1) % 3); //#18495
        } //#18495
        $k[$j++] = Infinity; //#18498
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = 1; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = 0; //#18500
        $k[$j++] = $_.lcw; //#18500
        $k[$j++] = $_.i % 3; //#18500
        $_.cwtobits(); //#18500
        $aload($k[--$j]); //#18500
        var _HO = $geti($_.cws, $_.c * $_.i, $_.c); //#18501
        for (var _HP = 0, _HQ = _HO.length; _HP < _HQ; _HP++) { //#18501
            $k[$j++] = $get(_HO, _HP); //#18501
            $k[$j++] = $_.i % 3; //#18501
            $_.cwtobits(); //#18501
            $aload($k[--$j]); //#18501
        } //#18501
        if ($_.compact) { //#18506
            $k[$j++] = 1; //#18503
        } else { //#18506
            $k[$j++] = $_.rcw; //#18505
            $k[$j++] = $_.i % 3; //#18505
            $_.cwtobits(); //#18505
            $aload($k[--$j]); //#18505
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 1; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 0; //#18506
            $k[$j++] = 1; //#18506
        } //#18506
        var _HY = $a(); //#18506
        $k[$j++] = _HY; //#18511
        for (var _Hb = 0, _Ha = $_.rowmult - 1; _Hb <= _Ha; _Hb += 1) { //#18511
            var _Hg = $k[$j - 1]; //#18510
            $puti($_.pixs, (_Hb + ($_.i * $_.rowmult)) * $_.rwid, _Hg); //#18510
        } //#18510
        $j--; //#18512
    } //#18512
    var _Hp = new Map([
        ["ren", 'renmatrix'],
        ["pixs", $_.pixs],
        ["pixx", $_.rwid],
        ["pixy", $_.r * $_.rowmult],
        ["height", ($_.r * $_.rowmult) / 72],
        ["width", $_.rwid / 72],
        ["borderleft", 2],
        ["borderright", 2],
        ["bordertop", 2],
        ["borderbottom", 2],
        ["opt", $_.options]
    ]); //#18527
    $k[$j++] = _Hp; //#18530
    if (!$_.dontdraw) { //#18530
        bwipp_renmatrix(); //#18530
    } //#18530
    $_ = $__; //#18534
} //bwipp_pdf417
function bwipp_pdf417compact() {
    var $__ = $_; //#18563
    $_ = Object.assign({}, $_); //#18563
    $_.dontdraw = false; //#18566
    $k[$j++] = $_; //#18568
    bwipp_processoptions(); //#18568
    $_.options = $k[--$j]; //#18568
    $_.barcode = $k[--$j]; //#18569
    $put($_.options, "dontdraw", true); //#18572
    $put($_.options, "compact", true); //#18573
    $k[$j++] = 'args'; //#18575
    $k[$j++] = $_.barcode; //#18575
    $k[$j++] = $_.options; //#18575
    bwipp_pdf417(); //#18575
    var _6 = $k[--$j]; //#18575
    $_[$k[--$j]] = _6; //#18575
    $put($_.args, "opt", $_.options); //#18577
    $k[$j++] = $_.args; //#18580
    if (!$_.dontdraw) { //#18580
        bwipp_renmatrix(); //#18580
    } //#18580
    $_ = $__; //#18582
} //bwipp_pdf417compact
function bwipp_micropdf417() {
    var $__ = $_; //#18613
    $_ = Object.assign({}, $_); //#18613
    $_.dontdraw = false; //#18616
    $_.version = "unset"; //#18617
    $_.columns = 0; //#18618
    $_.rows = 0; //#18619
    $_.rowmult = 2; //#18620
    $_.cca = false; //#18621
    $_.ccb = false; //#18622
    $_.raw = false; //#18623
    $_.parse = false; //#18624
    $_.parsefnc = false; //#18625
    $k[$j++] = $_; //#18627
    bwipp_processoptions(); //#18627
    $_.options = $k[--$j]; //#18627
    $_.barcode = $k[--$j]; //#18628
    if ($_.barcode.length == 0) { //#18632
        $k[$j++] = 'bwipp.micropdf417emptyData#18631'; //#18631
        $k[$j++] = "The data must not be empty"; //#18631
        bwipp_raiseerror(); //#18631
    } //#18631
    if ($_.rowmult <= 0) { //#18636
        $k[$j++] = 'bwipp.micropdf417badRowMult#18635'; //#18635
        $k[$j++] = "The row multiplier must be greater than zero"; //#18635
        bwipp_raiseerror(); //#18635
    } //#18635
    if (($_.cca && $_.ccb) || (($_.cca && $_.raw) || ($_.ccb && $_.raw))) { //#18640
        $k[$j++] = 'bwipp.micropdf417ccaAndCcb#18639'; //#18639
        $k[$j++] = "Cannot combine cca, ccb and raw"; //#18639
        bwipp_raiseerror(); //#18639
    } //#18639
    if ($ne($_.version, "unset")) { //#18661
        $search($_.version, "x"); //#18644
        if ($k[--$j]) { //#18659
            var _D = $k[--$j]; //#18645
            $k[$j - 1] = _D; //#18645
            var _F = $k[$j - 1]; //#18646
            var _G = $k[$j - 2]; //#18646
            $k[$j++] = _G; //#18646
            $k[$j++] = _F; //#18646
            $astore($a(2)); //#18646
            $forall($k[--$j], function() { //#18655
                var _J = $k[$j - 1]; //#18648
                $k[$j++] = true; //#18648
                if (_J.length == 0) { //#18648
                    $k[$j - 1] = false; //#18648
                } //#18648
                var _K = $k[--$j]; //#18649
                var _L = $k[--$j]; //#18649
                $k[$j++] = _K; //#18651
                $forall(_L, function() { //#18651
                    var _M = $k[--$j]; //#18650
                    if ((_M < 48) || (_M > 57)) { //#18650
                        $k[$j - 1] = false; //#18650
                    } //#18650
                }); //#18650
                if (!$k[--$j]) { //#18654
                    $k[$j++] = 'bwipp.micropdf417BadVersionBadRowOrColumn#18653'; //#18653
                    $k[$j++] = "version must be formatted as RxC"; //#18653
                    bwipp_raiseerror(); //#18653
                } //#18653
            }); //#18653
            $_.rows = $cvi($k[--$j]); //#18656
            $_.columns = $cvi($k[--$j]); //#18657
        } else { //#18659
            $j--; //#18659
            $k[$j++] = 'bwipp.micropdf417badVersionFormat#18659'; //#18659
            $k[$j++] = "version must be formatted as RxC"; //#18659
            bwipp_raiseerror(); //#18659
        } //#18659
    } //#18659
    $_.T = 0; //#18667
    $_.N = 1; //#18667
    $_.B = 2; //#18667
    $_.A = 0; //#18668
    $_.L = 1; //#18668
    $_.M = 2; //#18668
    $_.P = 3; //#18668
    $_.tl = -1; //#18671
    $_.nl = -2; //#18671
    $_.bl = -3; //#18671
    $_.bl6 = -4; //#18671
    $_.bs = -5; //#18671
    $_.al = -6; //#18674
    $_.ll = -7; //#18674
    $_.ml = -8; //#18674
    $_.pl = -9; //#18674
    $_.as = -10; //#18674
    $_.ps = -11; //#18674
    $_.m05 = -12; //#18677
    $_.m06 = -13; //#18677
    $_.charmaps = $a([$a(["A", "a", "0", ";"]), $a(["B", "b", "1", "<"]), $a(["C", "c", "2", ">"]), $a(["D", "d", "3", "@"]), $a(["E", "e", "4", "["]), $a(["F", "f", "5", 92]), $a(["G", "g", "6", "]"]), $a(["H", "h", "7", "_"]), $a(["I", "i", "8", "`"]), $a(["J", "j", "9", "~"]), $a(["K", "k", "&", "!"]), $a(["L", "l", 13, 13]), $a(["M", "m", 9, 9]), $a(["N", "n", ",", ","]), $a(["O", "o", ":", ":"]), $a(["P", "p", "#", 10]), $a(["Q", "q", "-", "-"]), $a(["R", "r", ".", "."]), $a(["S", "s", "$", "$"]), $a(["T", "t", "/", "/"]), $a(["U", "u", "+", "\""]), $a(["V", "v", "%", "|"]), $a(["W", "w", "*", "*"]), $a(["X", "x", "=", 40]), $a(["Y", "y", "^", 41]), $a(["Z", "z", $_.pl, "?"]), $a([" ", " ", " ", "{"]), $a([$_.ll, $_.as, $_.ll, "}"]), $a([$_.ml, $_.ml, $_.al, "'"]), $a([$_.ps, $_.ps, $_.ps, $_.al])]); //#18712
    $_.charvals = $a([new Map, new Map, new Map, new Map]); //#18715
    $_.alltext = new Map; //#18716
    for (var _19 = 0, _18 = $_.charmaps.length - 1; _19 <= _18; _19 += 1) { //#18726
        $_.i = _19; //#18718
        $_.encs = $get($_.charmaps, $_.i); //#18719
        for (var _1D = 0; _1D <= 3; _1D += 1) { //#18725
            $_.j = _1D; //#18721
            var _1G = $get($_.encs, $_.j); //#18722
            $k[$j++] = _1G; //#18722
            if ($eq($type(_1G), 'stringtype')) { //#18722
                var _1J = $get($k[--$j], 0); //#18722
                $k[$j++] = _1J; //#18722
            } //#18722
            var _1K = $k[--$j]; //#18723
            $put($get($_.charvals, $_.j), _1K, $_.i); //#18723
            $put($_.alltext, _1K, -1); //#18724
        } //#18724
    } //#18724
    $_.e = 10000; //#18728
    $_.latlen = $a([$a([0, 1, 1, 2]), $a([2, 0, 1, 2]), $a([1, 1, 0, 1]), $a([1, 2, 2, 0])]); //#18736
    $_.latseq = $a([$a([$a([]), $a([$_.ll]), $a([$_.ml]), $a([$_.ml, $_.pl])]), $a([$a([$_.ml, $_.al]), $a([]), $a([$_.ml]), $a([$_.ml, $_.pl])]), $a([$a([$_.al]), $a([$_.ll]), $a([]), $a([$_.pl])]), $a([$a([$_.al]), $a([$_.al, $_.ll]), $a([$_.al, $_.ml]), $a([])])]); //#18744
    $_.shftlen = $a([$a([$_.e, $_.e, $_.e, 1]), $a([1, $_.e, $_.e, 1]), $a([$_.e, $_.e, $_.e, 1]), $a([$_.e, $_.e, $_.e, $_.e])]); //#18752
    if ($_.raw || $_.cca) { //#18774
        $_.datcws = $a($_.barcode.length); //#18757
        $_.i = 0; //#18758
        $_.j = 0; //#18758
        for (;;) { //#18769
            if ($_.i >= ($_.barcode.length - 3)) { //#18760
                break; //#18760
            } //#18760
            if ($get($_.barcode, $_.i) != 94) { //#18761
                break; //#18761
            } //#18761
            var _2a = $geti($_.barcode, $_.i + 1, 3); //#18762
            $k[$j++] = _2a; //#18764
            $k[$j++] = false; //#18764
            for (var _2b = 0, _2c = _2a.length; _2b < _2c; _2b++) { //#18764
                var _2d = $get(_2a, _2b); //#18764
                if ((_2d < 48) || (_2d > 57)) { //#18763
                    $k[$j - 1] = true; //#18763
                } //#18763
            } //#18763
            if ($k[--$j]) { //#18764
                $j--; //#18764
                break; //#18764
            } //#18764
            $_.cw = $cvi($k[--$j]); //#18765
            $put($_.datcws, $_.j, $_.cw); //#18766
            $_.i = $_.i + 4; //#18767
            $_.j = $_.j + 1; //#18768
        } //#18768
        if ($_.i != $_.barcode.length) { //#18772
            $k[$j++] = 'bwipp.micropdf417badCcaRawFormat#18771'; //#18771
            $k[$j++] = "cca and raw data must be formatted as ^NNN"; //#18771
            bwipp_raiseerror(); //#18771
        } //#18771
        $_.datcws = $geti($_.datcws, 0, $_.j); //#18773
    } //#18773
    $_.encb = function() {
        $_.in = $k[--$j]; //#18777
        $_.inlen = $_.in.length; //#18778
        $_.out = $a(((~~($_.inlen / 6)) * 5) + ($_.inlen % 6)); //#18779
        for (var _2x = 0, _2w = (~~($_.inlen / 6)) - 1; _2x <= _2w; _2x += 1) { //#18802
            $_.k = _2x; //#18781
            $k[$j++] = Infinity; //#18782
            $aload($geti($_.in, $_.k * 6, 3)); //#18782
            $_.msbs = $a(); //#18782
            $k[$j++] = Infinity; //#18783
            $aload($_.msbs); //#18784
            var _33 = $k[--$j]; //#18784
            var _34 = $k[--$j]; //#18784
            var _35 = $k[--$j]; //#18784
            $k[$j++] = $f(_33 + _34 * 256 + _35 * 65536); //#18785
            for (var _36 = 0; _36 < 3; _36++) { //#18785
                var _37 = $k[--$j]; //#18785
                $k[$j++] = _37 % 900; //#18785
                $k[$j++] = ~~(_37 / 900); //#18785
            } //#18785
            $_.mscs = $a(); //#18785
            $k[$j++] = Infinity; //#18787
            $aload($geti($_.in, ($_.k * 6) + 3, 3)); //#18787
            $_.lsbs = $a(); //#18787
            $k[$j++] = Infinity; //#18788
            $aload($_.lsbs); //#18789
            var _3E = $k[--$j]; //#18789
            var _3F = $k[--$j]; //#18789
            var _3G = $k[--$j]; //#18789
            $k[$j++] = $f(_3E + _3F * 256 + _3G * 65536); //#18790
            for (var _3H = 0; _3H < 3; _3H++) { //#18790
                var _3I = $k[--$j]; //#18790
                $k[$j++] = _3I % 900; //#18790
                $k[$j++] = ~~(_3I / 900); //#18790
            } //#18790
            $_.lscs = $a(); //#18790
            var _3L = $get($_.lscs, 0); //#18792
            var _3N = $get($_.mscs, 0); //#18792
            $put($_.out, ($_.k * 5) + 4, ($f(_3L + (_3N * 316))) % 900); //#18793
            var _3R = $get($_.lscs, 1); //#18794
            var _3T = $get($_.mscs, 0); //#18794
            var _3V = $get($_.mscs, 1); //#18794
            $put($_.out, $_.k * 5 + 3, $f(~~((_3L + _3N * 316) / 900) + _3R + _3T * 641 + _3V * 316) % 900); //#18795
            var _3Z = $get($_.lscs, 2); //#18796
            var _3b = $get($_.mscs, 0); //#18796
            var _3d = $get($_.mscs, 1); //#18796
            var _3f = $get($_.mscs, 2); //#18796
            $put($_.out, $_.k * 5 + 2, $f(~~((~~((_3L + _3N * 316) / 900) + _3R + _3T * 641 + _3V * 316) / 900) + _3Z + _3b * 20 + _3d * 641 + _3f * 316) % 900); //#18797
            var _3j = $get($_.lscs, 3); //#18798
            var _3l = $get($_.mscs, 1); //#18798
            var _3n = $get($_.mscs, 2); //#18798
            $put($_.out, $_.k * 5 + 1, $f(~~((~~((~~((_3L + _3N * 316) / 900) + _3R + _3T * 641 + _3V * 316) / 900) + _3Z + _3b * 20 + _3d * 641 + _3f * 316) / 900) + _3j + _3l * 20 + _3n * 641) % 900); //#18799
            $put($_.out, $_.k * 5, $f(~~((~~((~~((~~((_3L + _3N * 316) / 900) + _3R + _3T * 641 + _3V * 316) / 900) + _3Z + _3b * 20 + _3d * 641 + _3f * 316) / 900) + _3j + _3l * 20 + _3n * 641) / 900) + $get($_.mscs, 2) * 20) % 900); //#18801
        } //#18801
        $_.rem = $_.inlen % 6; //#18803
        if ($_.rem != 0) { //#18808
            $k[$j++] = $_.out; //#18806
            $k[$j++] = $_.out.length - $_.rem; //#18806
            $k[$j++] = Infinity; //#18806
            $aload($geti($_.in, $_.inlen - $_.rem, $_.rem)); //#18806
            var _44 = $a(); //#18806
            var _45 = $k[--$j]; //#18807
            $puti($k[--$j], _45, _44); //#18807
        } //#18807
        $k[$j++] = $_.out; //#18809
    }; //#18809
    if ($_.ccb) { //#18818
        $_.barlen = $_.barcode.length; //#18813
        $_.datcws = $a((((~~($_.barlen / 6)) * 5) + ($_.barlen % 6)) + 2); //#18814
        $put($_.datcws, 0, 920); //#18815
        var _4G = (($_.barlen % 6) == 0) ? 924 : 901; //#18816
        $put($_.datcws, 1, _4G); //#18816
        $k[$j++] = $_.datcws; //#18817
        $k[$j++] = 2; //#18817
        $k[$j++] = Infinity; //#18817
        $forall($_.barcode); //#18817
        var _4J = $a(); //#18817
        $k[$j++] = _4J; //#18817
        $_.encb(); //#18817
        var _4K = $k[--$j]; //#18817
        var _4L = $k[--$j]; //#18817
        $puti($k[--$j], _4L, _4K); //#18817
    } //#18817
    if (((!$_.raw) && (!$_.cca)) && (!$_.ccb)) { //#19219
        var _4S = new Map([
            ["parse", $_.parse],
            ["parsefnc", $_.parsefnc],
            ["eci", true]
        ]); //#18829
        $_.fncvals = _4S; //#18830
        $k[$j++] = 'msg'; //#18831
        $k[$j++] = $_.barcode; //#18831
        $k[$j++] = $_.fncvals; //#18831
        bwipp_parseinput(); //#18831
        var _4V = $k[--$j]; //#18831
        $_[$k[--$j]] = _4V; //#18831
        $_.msglen = $_.msg.length; //#18832
        if ($_.msglen >= 9) { //#18850
            $aload($geti($_.msg, 0, 7)); //#18836
            var _4b = $k[--$j]; //#18837
            $k[$j - 1] = _4b == 29; //#18837
            var _4d = $k[--$j]; //#18837
            var _4e = $k[--$j]; //#18837
            var _4f = $k[--$j]; //#18837
            var _4g = $k[--$j]; //#18838
            var _4h = $k[--$j]; //#18838
            var _4i = $k[--$j]; //#18838
            if (((((_4d && (_4e == 48)) && (_4f == 30)) && (_4g == 62)) && (_4h == 41)) && (_4i == 91)) { //#18849
                $aload($geti($_.msg, $_.msglen - 2, 2)); //#18839
                var _4m = $k[--$j]; //#18840
                var _4n = $k[--$j]; //#18840
                if ((_4m == 4) && (_4n == 30)) { //#18848
                    if ($get($_.msg, 5) == 53) { //#18845
                        $k[$j++] = Infinity; //#18842
                        $k[$j++] = $_.m05; //#18842
                        $aload($geti($_.msg, 7, $_.msg.length - 9)); //#18842
                        $_.msg = $a(); //#18842
                    } else { //#18845
                        if ($get($_.msg, 5) == 54) { //#18846
                            $k[$j++] = Infinity; //#18845
                            $k[$j++] = $_.m06; //#18845
                            $aload($geti($_.msg, 7, $_.msg.length - 9)); //#18845
                            $_.msg = $a(); //#18845
                        } //#18845
                    } //#18845
                } //#18845
            } //#18845
        } //#18845
        $_.msglen = $_.msg.length; //#18851
        $k[$j++] = Infinity; //#18854
        for (var _54 = 0, _55 = $_.msglen; _54 < _55; _54++) { //#18854
            $k[$j++] = 0; //#18854
        } //#18854
        $k[$j++] = 0; //#18854
        $_.numdigits = $a(); //#18854
        $k[$j++] = Infinity; //#18855
        for (var _58 = 0, _59 = $_.msglen; _58 < _59; _58++) { //#18855
            $k[$j++] = 0; //#18855
        } //#18855
        $k[$j++] = 0; //#18855
        $_.numtext = $a(); //#18855
        $k[$j++] = Infinity; //#18856
        for (var _5C = 0, _5D = $_.msglen; _5C < _5D; _5C++) { //#18856
            $k[$j++] = 0; //#18856
        } //#18856
        $k[$j++] = 0; //#18856
        $_.numbytes = $a(); //#18856
        $k[$j++] = Infinity; //#18857
        for (var _5G = 0, _5H = $_.msglen; _5G < _5H; _5G++) { //#18857
            $k[$j++] = 0; //#18857
        } //#18857
        $_.iseci = $a(); //#18857
        for (var _5K = $_.msglen - 1; _5K >= 0; _5K -= 1) { //#18870
            $_.i = _5K; //#18859
            var _5N = $get($_.msg, $_.i); //#18860
            if ((_5N >= 48) && (_5N <= 57)) { //#18862
                $put($_.numdigits, $_.i, $get($_.numdigits, $_.i + 1) + 1); //#18861
            } //#18861
            var _5X = $get($_.alltext, $get($_.msg, $_.i)) !== undefined; //#18863
            if (_5X && ($get($_.numdigits, $_.i) < 13)) { //#18865
                $put($_.numtext, $_.i, $get($_.numtext, $_.i + 1) + 1); //#18864
            } //#18864
            if ((($get($_.msg, $_.i) >= 0) && ($get($_.numtext, $_.i) < 5)) && ($get($_.numdigits, $_.i) < 13)) { //#18868
                $put($_.numbytes, $_.i, $get($_.numbytes, $_.i + 1) + 1); //#18867
            } //#18867
            $put($_.iseci, $_.i, $get($_.msg, $_.i) <= -1000000); //#18869
        } //#18869
        $_.numdigits = $geti($_.numdigits, 0, $_.msglen); //#18871
        $_.numtext = $geti($_.numtext, 0, $_.msglen); //#18872
        $_.numbytes = $geti($_.numbytes, 0, $_.msglen); //#18873
        $_.seq = $a([]); //#18875
        $_.seqlen = 0; //#18875
        $_.state = $_.B; //#18875
        $_.p = 0; //#18875
        for (;;) { //#18937
            if ($_.p == $_.msglen) { //#18876
                break; //#18876
            } //#18876
            var _6F = $get($_.msg, $_.p); //#18877
            if (($_.p == 0) && ((_6F == $_.m05) || (_6F == $_.m06))) { //#18935
                $k[$j++] = Infinity; //#18878
                $aload($_.seq); //#18879
                var _6N = ($get($_.msg, $_.p) == $_.m05) ? $_.m05 : $_.m06; //#18880
                $k[$j++] = _6N; //#18880
                $_.seq = $a(); //#18880
                $_.p = $_.p + 1; //#18882
                $_.seqlen = $_.seqlen + 1; //#18883
            } else { //#18935
                if ($get($_.iseci, $_.p)) { //#18935
                    $_.eci = $get($_.msg, $_.p); //#18886
                    $k[$j++] = Infinity; //#18887
                    $aload($_.seq); //#18888
                    $k[$j++] = $a([$_.eci]); //#18889
                    $_.seq = $a(); //#18889
                    $_.p = $_.p + 1; //#18891
                    $k[$j++] = 'seqlen'; //#18892
                    $k[$j++] = $_.seqlen; //#18892
                    if ($_.eci <= -1810900) { //#18892
                        $k[$j++] = 2; //#18892
                    } else { //#18892
                        var _6f = ($_.eci <= -1000900) ? 3 : 2; //#18892
                        $k[$j++] = _6f; //#18892
                    } //#18892
                    var _6g = $k[--$j]; //#18892
                    var _6h = $k[--$j]; //#18892
                    $_[$k[--$j]] = $f(_6h + _6g); //#18892
                } else { //#18935
                    $_.n = $get($_.numdigits, $_.p); //#18894
                    if (($_.n >= 13) || (($_.n == $_.msglen) && ($_.n >= 8))) { //#18935
                        $k[$j++] = Infinity; //#18896
                        $aload($_.seq); //#18897
                        $k[$j++] = $_.nl; //#18899
                        $k[$j++] = Infinity; //#18899
                        $aload($geti($_.msg, $_.p, $_.n)); //#18899
                        var _6w = $a(); //#18899
                        $k[$j++] = _6w; //#18899
                        $_.seq = $a(); //#18899
                        $_.state = $_.N; //#18901
                        $_.p = $f($_.p + $_.n); //#18902
                        $_.seqlen = $f(($_.seqlen + 1) + $_.n); //#18903
                    } else { //#18935
                        $_.t = $get($_.numtext, $_.p); //#18905
                        if ($_.t >= 5) { //#18935
                            $k[$j++] = Infinity; //#18907
                            $aload($_.seq); //#18908
                            if ($_.state != $_.T) { //#18911
                                $k[$j++] = $_.tl; //#18910
                            } //#18910
                            $k[$j++] = Infinity; //#18912
                            $aload($geti($_.msg, $_.p, $_.t)); //#18912
                            var _7F = $a(); //#18912
                            $k[$j++] = _7F; //#18912
                            $_.seq = $a(); //#18912
                            $_.state = $_.T; //#18914
                            $_.p = $f($_.p + $_.t); //#18915
                            $_.seqlen = $f(($_.seqlen + 1) + $_.t); //#18916
                        } else { //#18935
                            $_.b = $get($_.numbytes, $_.p); //#18918
                            if (($_.b == 1) && ($_.state == $_.T)) { //#18935
                                $k[$j++] = Infinity; //#18920
                                $aload($_.seq); //#18921
                                $k[$j++] = $_.bs; //#18923
                                $k[$j++] = $a([$get($_.msg, $_.p)]); //#18923
                                $_.seq = $a(); //#18923
                                $_.p = $f($_.p + $_.b); //#18925
                                $_.seqlen = $_.seqlen + 2; //#18926
                            } else { //#18935
                                $k[$j++] = Infinity; //#18928
                                $aload($_.seq); //#18929
                                var _7e = (($_.b % 6) != 0) ? $_.bl : $_.bl6; //#18930
                                $k[$j++] = _7e; //#18931
                                $k[$j++] = Infinity; //#18931
                                $aload($geti($_.msg, $_.p, $_.b)); //#18931
                                var _7j = $a(); //#18931
                                $k[$j++] = _7j; //#18931
                                $_.seq = $a(); //#18931
                                $_.state = $_.B; //#18933
                                $_.p = $f($_.p + $_.b); //#18934
                                $_.seqlen = $f(($_.seqlen + 1) + $_.b); //#18935
                            } //#18935
                        } //#18935
                    } //#18935
                } //#18935
            } //#18935
        } //#18935
        $k[$j++] = Infinity; //#18939
        $k[$j++] = $_.tl; //#18940
        $k[$j++] = 900; //#18940
        $k[$j++] = $_.bl; //#18940
        $k[$j++] = 901; //#18940
        $k[$j++] = $_.bl6; //#18940
        $k[$j++] = 924; //#18940
        $k[$j++] = $_.nl; //#18940
        $k[$j++] = 902; //#18940
        $k[$j++] = $_.bs; //#18940
        $k[$j++] = 913; //#18940
        $k[$j++] = $_.m05; //#18940
        $k[$j++] = 916; //#18940
        $k[$j++] = $_.m06; //#18940
        $k[$j++] = 917; //#18940
        $_.latchcws = $d(); //#18941
        $_.enca = function() {
            var _82 = $get($get($_.charvals, $_.A), $k[--$j]); //#18944
            $k[$j++] = _82; //#18944
        }; //#18944
        $_.encl = function() {
            var _87 = $get($get($_.charvals, $_.L), $k[--$j]); //#18945
            $k[$j++] = _87; //#18945
        }; //#18945
        $_.encm = function() {
            var _8C = $get($get($_.charvals, $_.M), $k[--$j]); //#18946
            $k[$j++] = _8C; //#18946
        }; //#18946
        $_.encp = function() {
            var _8H = $get($get($_.charvals, $_.P), $k[--$j]); //#18947
            $k[$j++] = _8H; //#18947
        }; //#18947
        $_.textencfuncs = $a(['enca', 'encl', 'encm', 'encp']); //#18948
        $_.addtotext = function() {
            $put($_.text, $_.l, $k[--$j]); //#18951
            $_.l = $_.l + 1; //#18952
        }; //#18952
        $_.enct = function() {
            $_.in = $k[--$j]; //#18957
            $_.curlen = $a([$_.e, $_.e, $_.e, $_.e]); //#18959
            $put($_.curlen, $_.submode, 0); //#18960
            $_.curseq = $a([$a([]), $a([]), $a([]), $a([])]); //#18961
            $forall($_.in, function() { //#19031
                $_.char = $k[--$j]; //#18966
                for (;;) { //#18987
                    $_.imp = false; //#18970
                    var _8g = $a([$_.A, $_.L, $_.M, $_.P]); //#18971
                    for (var _8h = 0, _8i = _8g.length; _8h < _8i; _8h++) { //#18985
                        $_.x = $get(_8g, _8h); //#18972
                        var _8o = $a([$_.A, $_.L, $_.M, $_.P]); //#18973
                        for (var _8p = 0, _8q = _8o.length; _8p < _8q; _8p++) { //#18984
                            $_.y = $get(_8o, _8p); //#18974
                            $_.cost = $f($get($_.curlen, $_.x) + $get($get($_.latlen, $_.x), $_.y)); //#18975
                            if ($_.cost < $get($_.curlen, $_.y)) { //#18983
                                $put($_.curlen, $_.y, $_.cost); //#18977
                                $k[$j++] = $_.curseq; //#18980
                                $k[$j++] = $_.y; //#18980
                                $k[$j++] = Infinity; //#18978
                                $aload($get($_.curseq, $_.x)); //#18979
                                $aload($get($get($_.latseq, $_.x), $_.y)); //#18980
                                var _9H = $a(); //#18980
                                var _9I = $k[--$j]; //#18981
                                $put($k[--$j], _9I, _9H); //#18981
                                $_.imp = true; //#18982
                            } //#18982
                        } //#18982
                    } //#18982
                    if (!$_.imp) { //#18986
                        break; //#18986
                    } //#18986
                } //#18986
                $_.nxtlen = $a([$_.e, $_.e, $_.e, $_.e]); //#18990
                $_.nxtseq = $a(4); //#18991
                var _9V = $a([$_.A, $_.L, $_.M, $_.P]); //#18993
                for (var _9W = 0, _9X = _9V.length; _9W < _9X; _9W++) { //#19026
                    $_.x = $get(_9V, _9W); //#18994
                    for (;;) { //#19024
                        var _9d = $get($get($_.charvals, $_.x), $_.char) !== undefined; //#18998
                        if (!_9d) { //#18998
                            break; //#18998
                        } //#18998
                        $_.cost = $get($_.curlen, $_.x) + 1; //#19001
                        if ($_.cost < $get($_.nxtlen, $_.x)) { //#19005
                            $put($_.nxtlen, $_.x, $_.cost); //#19003
                            $k[$j++] = $_.nxtseq; //#19004
                            $k[$j++] = $_.x; //#19004
                            $k[$j++] = Infinity; //#19004
                            $aload($get($_.curseq, $_.x)); //#19004
                            $k[$j++] = $_.char; //#19004
                            var _9u = $a(); //#19004
                            var _9v = $k[--$j]; //#19004
                            $put($k[--$j], _9v, _9u); //#19004
                        } //#19004
                        var _A1 = $a([$_.A, $_.L, $_.M, $_.P]); //#19008
                        for (var _A2 = 0, _A3 = _A1.length; _A2 < _A3; _A2++) { //#19021
                            $_.y = $get(_A1, _A2); //#19009
                            if ($ne($_.x, $_.y)) { //#19020
                                $_.cost = $f($get($_.curlen, $_.y) + $get($get($_.shftlen, $_.y), $_.x)) + 1; //#19011
                                if ($_.cost < $get($_.nxtlen, $_.y)) { //#19019
                                    $put($_.nxtlen, $_.y, $_.cost); //#19013
                                    $k[$j++] = $_.nxtseq; //#19017
                                    $k[$j++] = $_.y; //#19017
                                    $k[$j++] = Infinity; //#19014
                                    $aload($get($_.curseq, $_.y)); //#19015
                                    var _AT = ($_.x == $_.A) ? $_.as : $_.ps; //#19016
                                    $k[$j++] = _AT; //#19017
                                    $k[$j++] = $_.char; //#19017
                                    var _AV = $a(); //#19017
                                    var _AW = $k[--$j]; //#19018
                                    $put($k[--$j], _AW, _AV); //#19018
                                } //#19018
                            } //#19018
                        } //#19018
                        break; //#19023
                    } //#19023
                } //#19023
                $_.curlen = $_.nxtlen; //#19028
                $_.curseq = $_.nxtseq; //#19029
            }); //#19029
            $_.minseq = $_.e; //#19034
            var _Af = $a([$_.A, $_.L, $_.M, $_.P]); //#19035
            for (var _Ag = 0, _Ah = _Af.length; _Ag < _Ah; _Ag++) { //#19041
                $_.k = $get(_Af, _Ag); //#19036
                if ($get($_.curlen, $_.k) < $_.minseq) { //#19040
                    $_.minseq = $get($_.curlen, $_.k); //#19038
                    $_.txtseq = $get($_.curseq, $_.k); //#19039
                } //#19039
            } //#19039
            $_.text = $a($_.minseq); //#19044
            $_.k = 0; //#19045
            $_.l = 0; //#19045
            for (;;) { //#19065
                if ($_.k >= $_.txtseq.length) { //#19046
                    break; //#19046
                } //#19046
                $_.char = $get($_.txtseq, $_.k); //#19048
                $k[$j++] = $_.char; //#19051
                if ($_[$get($_.textencfuncs, $_.submode)]() === true) {
                    break;
                } //#19051
                $_.addtotext(); //#19051
                $_.k = $_.k + 1; //#19052
                if (($_.char == $_.as) || ($_.char == $_.ps)) { //#19058
                    $k[$j++] = $get($_.txtseq, $_.k); //#19056
                    if ($_.char == $_.as) { //#19056
                        $_.enca(); //#19056
                    } else { //#19056
                        $_.encp(); //#19056
                    } //#19056
                    $_.addtotext(); //#19056
                    $_.k = $_.k + 1; //#19057
                } //#19057
                if ($_.char == $_.al) { //#19061
                    $_.submode = $_.A; //#19061
                } //#19061
                if ($_.char == $_.ll) { //#19062
                    $_.submode = $_.L; //#19062
                } //#19062
                if ($_.char == $_.ml) { //#19063
                    $_.submode = $_.M; //#19063
                } //#19063
                if ($_.char == $_.pl) { //#19064
                    $_.submode = $_.P; //#19064
                } //#19064
            } //#19064
            if (($_.text.length % 2) == 1) { //#19112
                if ($_.submode == $_.P) { //#19108
                    $k[$j++] = 'pad'; //#19069
                    $k[$j++] = $_.al; //#19069
                    $_.encp(); //#19069
                    var _BW = $k[--$j]; //#19069
                    $_[$k[--$j]] = _BW; //#19069
                    $_.submode = $_.A; //#19070
                } else { //#19108
                    $_.nextecitext = false; //#19073
                    for (var _Bc = $_.i + 1, _Bb = $_.seq.length - 1; _Bc <= _Bb; _Bc += 1) { //#19080
                        $_.ti = _Bc; //#19076
                        if ($ne($type($get($_.seq, $_.ti)), 'arraytype')) { //#19077
                            $_.nextecitext = false; //#19077
                            break; //#19077
                        } //#19077
                        if ($get($get($_.seq, $_.ti), 0) > -1000000) { //#19078
                            break; //#19078
                        } //#19078
                        $_.nextecitext = true; //#19079
                    } //#19079
                    if ($_.nextecitext) { //#19108
                        if ($_.submode == $_.A) { //#19105
                            var _Bv = $get($get($_.charvals, $_.L), $get($get($_.seq, $_.ti), 0)) !== undefined; //#19083
                            if (_Bv) { //#19088
                                $k[$j++] = 'pad'; //#19084
                                $k[$j++] = $_.ll; //#19084
                                $_.enca(); //#19084
                                var _Bx = $k[--$j]; //#19084
                                $_[$k[--$j]] = _Bx; //#19084
                                $_.submode = $_.L; //#19085
                            } else { //#19088
                                $k[$j++] = 'pad'; //#19087
                                $k[$j++] = $_.ml; //#19087
                                $_.enca(); //#19087
                                var _C1 = $k[--$j]; //#19087
                                $_[$k[--$j]] = _C1; //#19087
                                $_.submode = $_.M; //#19088
                            } //#19088
                        } else { //#19105
                            if ($_.submode == $_.M) { //#19105
                                var _CD = $get($get($_.charvals, $_.L), $get($get($_.seq, $_.ti), 0)) !== undefined; //#19092
                                if (_CD) { //#19101
                                    $k[$j++] = 'pad'; //#19093
                                    $k[$j++] = $_.ll; //#19093
                                    $_.encm(); //#19093
                                    var _CF = $k[--$j]; //#19093
                                    $_[$k[--$j]] = _CF; //#19093
                                    $_.submode = $_.L; //#19094
                                } else { //#19101
                                    var _CP = $get($get($_.charvals, $_.P), $get($get($_.seq, $_.ti), 0)) !== undefined; //#19096
                                    if (_CP) { //#19101
                                        $k[$j++] = 'pad'; //#19097
                                        $k[$j++] = $_.pl; //#19097
                                        $_.encm(); //#19097
                                        var _CR = $k[--$j]; //#19097
                                        $_[$k[--$j]] = _CR; //#19097
                                        $_.submode = $_.P; //#19098
                                    } else { //#19101
                                        $k[$j++] = 'pad'; //#19100
                                        $k[$j++] = $_.al; //#19100
                                        $_.encm(); //#19100
                                        var _CV = $k[--$j]; //#19100
                                        $_[$k[--$j]] = _CV; //#19100
                                        $_.submode = $_.A; //#19101
                                    } //#19101
                                } //#19101
                            } else { //#19105
                                $k[$j++] = 'pad'; //#19104
                                $k[$j++] = $_.ml; //#19104
                                $_.encl(); //#19104
                                var _CZ = $k[--$j]; //#19104
                                $_[$k[--$j]] = _CZ; //#19104
                                $_.submode = $_.M; //#19105
                            } //#19105
                        } //#19105
                    } else { //#19108
                        $k[$j++] = 'pad'; //#19108
                        $k[$j++] = $_.ps; //#19108
                        if ($_[$get($_.textencfuncs, $_.submode)]() === true) {
                            return true;
                        } //#19108
                        var _Ch = $k[--$j]; //#19108
                        $_[$k[--$j]] = _Ch; //#19108
                    } //#19108
                } //#19108
                $k[$j++] = Infinity; //#19111
                $aload($_.text); //#19111
                $k[$j++] = $_.pad; //#19111
                $_.text = $a(); //#19111
            } //#19111
            $_.out = $a(~~($_.text.length / 2)); //#19114
            for (var _Cq = 0, _Cp = $_.out.length - 1; _Cq <= _Cp; _Cq += 1) { //#19118
                $_.k = _Cq; //#19116
                $put($_.out, $_.k, $f(($get($_.text, $_.k * 2) * 30) + $get($_.text, ($_.k * 2) + 1))); //#19117
            } //#19117
            $k[$j++] = $_.out; //#19120
        }; //#19120
        $_.encn = function() {
            $_.in = $k[--$j]; //#19125
            $_.out = $a([]); //#19126
            for (var _D4 = 0, _D3 = $_.in.length - 1; _D4 <= _D3; _D4 += 44) { //#19154
                $_.k = _D4; //#19128
                $k[$j++] = Infinity; //#19129
                var _D9 = $_.in.length - $_.k; //#19130
                $k[$j++] = 1; //#19130
                $k[$j++] = $_.in; //#19130
                $k[$j++] = $_.k; //#19130
                $k[$j++] = _D9; //#19130
                if (_D9 > 44) { //#19130
                    $k[$j - 1] = 44; //#19130
                } //#19130
                var _DA = $k[--$j]; //#19130
                var _DB = $k[--$j]; //#19130
                var _DD = $geti($k[--$j], _DB, _DA); //#19130
                for (var _DE = 0, _DF = _DD.length; _DE < _DF; _DE++) { //#19131
                    $k[$j++] = $get(_DD, _DE) - 48; //#19131
                } //#19131
                $_.gmod = $a(); //#19131
                $_.cwn = $a([]); //#19133
                for (;;) { //#19152
                    $_.dv = 900; //#19134
                    $_.gmul = $a([]); //#19135
                    $_.val = 0; //#19135
                    for (;;) { //#19147
                        if ($_.gmod.length == 0) { //#19136
                            break; //#19136
                        } //#19136
                        $_.val = $f(($_.val * 10) + $get($_.gmod, 0)); //#19137
                        $_.gmod = $geti($_.gmod, 1, $_.gmod.length - 1); //#19138
                        if ($_.val < $_.dv) { //#19144
                            if ($_.gmul.length != 0) { //#19142
                                $k[$j++] = Infinity; //#19141
                                $aload($_.gmul); //#19141
                                $k[$j++] = 0; //#19141
                                $_.gmul = $a(); //#19141
                            } //#19141
                        } else { //#19144
                            $k[$j++] = Infinity; //#19144
                            $aload($_.gmul); //#19144
                            $k[$j++] = ~~($_.val / $_.dv); //#19144
                            $_.gmul = $a(); //#19144
                        } //#19144
                        $_.val = $_.val % $_.dv; //#19146
                    } //#19146
                    $_.dv = $_.val; //#19148
                    $k[$j++] = Infinity; //#19149
                    $k[$j++] = $_.dv; //#19149
                    $aload($_.cwn); //#19149
                    $_.cwn = $a(); //#19149
                    $_.gmod = $_.gmul; //#19150
                    if ($_.gmul.length == 0) { //#19151
                        break; //#19151
                    } //#19151
                } //#19151
                $k[$j++] = Infinity; //#19153
                $aload($_.out); //#19153
                $aload($_.cwn); //#19153
                $_.out = $a(); //#19153
            } //#19153
            $k[$j++] = $_.out; //#19155
        }; //#19155
        $_.ence = function() {
            var _Do = (-$get($k[--$j], 0)) - 1000000; //#19160
            $k[$j++] = _Do; //#19168
            if (_Do <= 899) { //#19167
                var _Dp = $k[--$j]; //#19161
                $k[$j++] = 927; //#19161
                $k[$j++] = _Dp; //#19161
                $astore($a(2)); //#19161
            } else { //#19167
                var _Dr = $k[$j - 1]; //#19162
                if (_Dr <= 810899) { //#19167
                    var _Ds = $k[--$j]; //#19163
                    $k[$j++] = 926; //#19163
                    $k[$j++] = (~~(_Ds / 900)) - 1; //#19163
                    $k[$j++] = _Ds % 900; //#19163
                    $astore($a(3)); //#19163
                } else { //#19167
                    var _Du = $k[$j - 1]; //#19164
                    if (_Du <= 811799) { //#19167
                        var _Dv = $k[--$j]; //#19165
                        $k[$j++] = 925; //#19165
                        $k[$j++] = _Dv - 810900; //#19165
                        $astore($a(2)); //#19165
                    } else { //#19167
                        $k[$j++] = 'bwipp.pdf417badECI#19167'; //#19167
                        $k[$j++] = "PDF417 supports ECIs 000000 to 811799"; //#19167
                        bwipp_raiseerror(); //#19167
                    } //#19167
                } //#19167
            } //#19167
        }; //#19167
        $_.encfuncs = $a(['enct', 'encn', 'encb']); //#19171
        $_.addtocws = function() {
            var _Dy = $k[--$j]; //#19174
            $puti($_.datcws, $_.j, _Dy); //#19174
            $_.j = _Dy.length + $_.j; //#19175
        }; //#19175
        $_.datcws = $a($_.seqlen); //#19180
        $_.i = 0; //#19181
        $_.j = 0; //#19181
        for (;;) { //#19217
            if ($_.i >= $_.seq.length) { //#19182
                break; //#19182
            } //#19182
            $_.chars = $get($_.seq, $_.i); //#19183
            if ($eq($type($_.chars), 'arraytype')) { //#19213
                if ($get($_.chars, 0) <= -1000000) { //#19188
                    $k[$j++] = $_.chars; //#19186
                    $_.ence(); //#19186
                    $_.addtocws(); //#19186
                } else { //#19188
                    $k[$j++] = $_.chars; //#19188
                    if ($_[$get($_.encfuncs, $_.state)]() === true) {
                        break;
                    } //#19188
                    $_.addtocws(); //#19188
                } //#19188
            } else { //#19213
                $k[$j++] = $a([$get($_.latchcws, $_.chars)]); //#19191
                $_.addtocws(); //#19191
                if ($_.chars == $_.tl) { //#19192
                    $_.state = $_.T; //#19192
                    $_.submode = $_.A; //#19192
                } //#19192
                if ($_.chars == $_.nl) { //#19193
                    $_.state = $_.N; //#19193
                } //#19193
                if (($_.chars == $_.bl) || ($_.chars == $_.bl6)) { //#19194
                    $_.state = $_.B; //#19194
                } //#19194
                if ($_.chars == $_.bs) { //#19198
                    $_.i = $_.i + 1; //#19196
                    $k[$j++] = $get($_.seq, $_.i); //#19197
                    $_.encb(); //#19197
                    $_.addtocws(); //#19197
                } //#19197
                if ($_.chars == $_.m05) { //#19206
                    if (($_.i + 1) < $_.seq.length) { //#19204
                        if ($get($_.seq, $_.i + 1) == $_.nl) { //#19203
                            $_.i = $_.i + 1; //#19202
                        } //#19202
                    } //#19202
                    $_.state = $_.N; //#19205
                } //#19205
                if ($_.chars == $_.m06) { //#19214
                    if (($_.i + 1) < $_.seq.length) { //#19212
                        if ($get($_.seq, $_.i + 1) == $_.tl) { //#19211
                            $_.i = $_.i + 1; //#19210
                        } //#19210
                    } //#19210
                    $_.state = $_.T; //#19213
                    $_.submode = $_.M; //#19213
                } //#19213
            } //#19213
            $_.i = $_.i + 1; //#19216
        } //#19216
        $_.datcws = $geti($_.datcws, 0, $_.j); //#19218
    } //#19218
    $_.ccametrics = $a([$a([2, 5, 4, 39, 0, 19]), $a([2, 6, 4, 1, 0, 33]), $a([2, 7, 5, 32, 0, 12]), $a([2, 8, 5, 8, 0, 40]), $a([2, 9, 6, 14, 0, 46]), $a([2, 10, 6, 43, 0, 23]), $a([2, 12, 7, 20, 0, 52]), $a([3, 4, 4, 11, 43, 23]), $a([3, 5, 5, 1, 33, 13]), $a([3, 6, 6, 5, 37, 17]), $a([3, 7, 7, 15, 47, 27]), $a([3, 8, 7, 21, 1, 33]), $a([4, 3, 4, 40, 20, 52]), $a([4, 4, 5, 43, 23, 3]), $a([4, 5, 6, 46, 26, 6]), $a([4, 6, 7, 34, 14, 46]), $a([4, 7, 8, 29, 9, 41])]); //#19242
    $_.nonccametrics = $a([$a([1, 11, 7, 1, 0, 9]), $a([1, 14, 7, 8, 0, 8]), $a([1, 17, 7, 36, 0, 36]), $a([1, 20, 8, 19, 0, 19]), $a([1, 24, 8, 9, 0, 17]), $a([1, 28, 8, 25, 0, 33]), $a([2, 8, 8, 1, 0, 1]), $a([2, 11, 9, 1, 0, 9]), $a([2, 14, 9, 8, 0, 8]), $a([2, 17, 10, 36, 0, 36]), $a([2, 20, 11, 19, 0, 19]), $a([2, 23, 13, 9, 0, 17]), $a([2, 26, 15, 27, 0, 35]), $a([3, 6, 12, 1, 1, 1]), $a([3, 8, 14, 7, 7, 7]), $a([3, 10, 16, 15, 15, 15]), $a([3, 12, 18, 25, 25, 25]), $a([3, 15, 21, 37, 37, 37]), $a([3, 20, 26, 1, 17, 33]), $a([3, 26, 32, 1, 9, 17]), $a([3, 32, 38, 21, 29, 37]), $a([3, 38, 44, 15, 31, 47]), $a([3, 44, 50, 1, 25, 49]), $a([4, 4, 8, 47, 19, 43]), $a([4, 6, 12, 1, 1, 1]), $a([4, 8, 14, 7, 7, 7]), $a([4, 10, 16, 15, 15, 15]), $a([4, 12, 18, 25, 25, 25]), $a([4, 15, 21, 37, 37, 37]), $a([4, 20, 26, 1, 17, 33]), $a([4, 26, 32, 1, 9, 17]), $a([4, 32, 38, 21, 29, 37]), $a([4, 38, 44, 15, 31, 47]), $a([4, 44, 50, 1, 25, 49])]); //#19281
    $k[$j++] = 'metrics'; //#19284
    if ($_.cca) { //#19284
        $k[$j++] = $_.ccametrics; //#19284
    } else { //#19284
        $k[$j++] = $_.nonccametrics; //#19284
    } //#19284
    var _Fz = $k[--$j]; //#19284
    $_[$k[--$j]] = _Fz; //#19284
    $_.urows = $_.rows; //#19287
    $_.ucols = $_.columns; //#19288
    $forall($_.metrics, function() { //#19303
        $_.m = $k[--$j]; //#19290
        $_.c = $get($_.m, 0); //#19291
        $_.r = $get($_.m, 1); //#19292
        $_.k = $get($_.m, 2); //#19293
        $_.rapl = $get($_.m, 3); //#19294
        $_.rapc = $get($_.m, 4); //#19295
        $_.rapr = $get($_.m, 5); //#19296
        $_.ncws = $f(($_.r * $_.c) - $_.k); //#19297
        $_.okay = true; //#19298
        if ($_.datcws.length > $_.ncws) { //#19299
            $_.okay = false; //#19299
        } //#19299
        if (($_.urows != 0) && ($_.urows != $_.r)) { //#19300
            $_.okay = false; //#19300
        } //#19300
        if (($_.ucols != 0) && ($_.ucols != $_.c)) { //#19301
            $_.okay = false; //#19301
        } //#19301
        if ($_.okay) { //#19302
            return true; //#19302
        } //#19302
    }); //#19302
    if (!$_.okay) { //#19307
        $k[$j++] = 'bwipp.micropdf417noValidSymbol#19306'; //#19306
        $k[$j++] = "Maximum length exceeded or invalid size"; //#19306
        bwipp_raiseerror(); //#19306
    } //#19306
    $_.m = $_.datcws.length; //#19310
    $_.n = $f(($_.c * $_.r) - $_.k); //#19311
    $_.cws = $a(($_.c * $_.r) + 1); //#19312
    $puti($_.cws, 0, $_.datcws); //#19313
    $k[$j++] = $_.cws; //#19314
    $k[$j++] = $_.m; //#19314
    $k[$j++] = Infinity; //#19314
    for (var _Gh = 0, _Gi = $f($_.n - $_.m); _Gh < _Gi; _Gh++) { //#19314
        $k[$j++] = 900; //#19314
    } //#19314
    var _Gj = $a(); //#19314
    var _Gk = $k[--$j]; //#19314
    $puti($k[--$j], _Gk, _Gj); //#19314
    $k[$j++] = $_.cws; //#19315
    $k[$j++] = $_.n; //#19315
    $k[$j++] = Infinity; //#19315
    for (var _Gp = 0, _Gq = $_.k; _Gp < _Gq; _Gp++) { //#19315
        $k[$j++] = 0; //#19315
    } //#19315
    $k[$j++] = 0; //#19315
    var _Gr = $a(); //#19315
    var _Gs = $k[--$j]; //#19315
    $puti($k[--$j], _Gs, _Gr); //#19315
    $k[$j++] = Infinity; //#19319
    $k[$j++] = 1; //#19319
    for (var _Gu = 0; _Gu < 928; _Gu++) { //#19319
        var _Gv = $k[$j - 1]; //#19319
        $k[$j++] = (_Gv * 3) % 929; //#19319
    } //#19319
    $_.rsalog = $a(); //#19319
    $_.rslog = $a(929); //#19320
    for (var _Gy = 1; _Gy <= 928; _Gy += 1) { //#19321
        $put($_.rslog, $get($_.rsalog, _Gy), _Gy); //#19321
    } //#19321
    $_.rsprod = function() {
        var _H3 = $k[$j - 1]; //#19326
        var _H4 = $k[$j - 2]; //#19326
        if ((_H3 != 0) && (_H4 != 0)) { //#19329
            var _H7 = $get($_.rslog, $k[--$j]); //#19327
            var _HC = $get($_.rsalog, $f(_H7 + $get($_.rslog, $k[--$j])) % 928); //#19327
            $k[$j++] = _HC; //#19327
        } else { //#19329
            $j -= 2; //#19329
            $k[$j++] = 0; //#19329
        } //#19329
    }; //#19329
    $k[$j++] = Infinity; //#19334
    $k[$j++] = 1; //#19334
    for (var _HE = 0, _HF = $_.k; _HE < _HF; _HE++) { //#19334
        $k[$j++] = 0; //#19334
    } //#19334
    $_.coeffs = $a(); //#19334
    for (var _HJ = 1, _HI = $_.k; _HJ <= _HI; _HJ += 1) { //#19343
        $_.i = _HJ; //#19336
        $put($_.coeffs, $_.i, $get($_.coeffs, $_.i - 1)); //#19337
        for (var _HQ = $_.i - 1; _HQ >= 1; _HQ -= 1) { //#19341
            $_.j = _HQ; //#19339
            $k[$j++] = $_.coeffs; //#19340
            $k[$j++] = $_.j; //#19340
            $k[$j++] = $get($_.coeffs, $_.j - 1); //#19340
            $k[$j++] = $get($_.coeffs, $_.j); //#19340
            $k[$j++] = $get($_.rsalog, $_.i); //#19340
            $_.rsprod(); //#19340
            var _Hc = $k[--$j]; //#19340
            var _Hd = $k[--$j]; //#19340
            var _He = $k[--$j]; //#19340
            $put($k[--$j], _He, $f(_Hd + _Hc) % 929); //#19340
        } //#19340
        $k[$j++] = $_.coeffs; //#19342
        $k[$j++] = 0; //#19342
        $k[$j++] = $get($_.coeffs, 0); //#19342
        $k[$j++] = $get($_.rsalog, $_.i); //#19342
        $_.rsprod(); //#19342
        var _Hm = $k[--$j]; //#19342
        var _Hn = $k[--$j]; //#19342
        $put($k[--$j], _Hn, _Hm); //#19342
    } //#19342
    $_.coeffs = $geti($_.coeffs, 0, $_.coeffs.length - 1); //#19344
    for (var _Ht = $_.coeffs.length - 1; _Ht >= 0; _Ht -= 2) { //#19345
        var _Hu = $_.coeffs; //#19345
        $put(_Hu, _Ht, 929 - $get(_Hu, _Ht)); //#19345
    } //#19345
    for (var _Hy = 0, _Hx = $_.n - 1; _Hy <= _Hx; _Hy += 1) { //#19354
        $_.t = $f($get($_.cws, _Hy) + $get($_.cws, $_.n)) % 929; //#19349
        for (var _I6 = 0, _I5 = $_.k - 1; _I6 <= _I5; _I6 += 1) { //#19353
            $_.j = _I6; //#19351
            $put($_.cws, $f($_.n + $_.j), $f($get($_.cws, $f($_.n + $_.j) + 1) + (929 - $_.t * $get($_.coeffs, $f($_.k - $_.j) - 1) % 929)) % 929); //#19352
        } //#19352
    } //#19352
    for (var _IN = $_.n, _IM = $f($_.n + $_.k); _IN <= _IM; _IN += 1) { //#19355
        $put($_.cws, _IN, (929 - $get($_.cws, _IN)) % 929); //#19355
    } //#19355
    $_.cws = $geti($_.cws, 0, $_.cws.length - 1); //#19358
    var _IV = $get($_.options, 'debugcws') !== undefined; //#19360
    if (_IV) { //#19360
        $k[$j++] = 'bwipp.debugcws#19360'; //#19360
        $k[$j++] = $_.cws; //#19360
        bwipp_raiseerror(); //#19360
    } //#19360
    $_.clusters = $a([$a([120256, 125680, 128380, 120032, 125560, 128318, 108736, 119920, 108640, 86080, 108592, 86048, 110016, 120560, 125820, 109792, 120440, 125758, 88256, 109680, 88160, 89536, 110320, 120700, 89312, 110200, 120638, 89200, 110140, 89840, 110460, 89720, 110398, 89980, 128506, 119520, 125304, 128190, 107712, 119408, 125244, 107616, 119352, 84032, 107568, 119324, 84000, 107544, 83984, 108256, 119672, 125374, 85184, 108144, 119612, 85088, 108088, 119582, 85040, 108060, 85728, 108408, 119742, 85616, 108348, 85560, 108318, 85880, 108478, 85820, 85790, 107200, 119152, 125116, 107104, 119096, 125086, 83008, 107056, 119068, 82976, 107032, 82960, 82952, 83648, 107376, 119228, 83552, 107320, 119198, 83504, 107292, 83480, 83468, 83824, 107452, 83768, 107422, 83740, 83900, 106848, 118968, 125022, 82496, 106800, 118940, 82464, 106776, 118926, 82448, 106764, 82440, 106758, 82784, 106936, 119006, 82736, 106908, 82712, 106894, 82700, 82694, 106974, 82830, 82240, 106672, 118876, 82208, 106648, 118862, 82192, 106636, 82184, 106630, 82180, 82352, 82328, 82316, 82080, 118830, 106572, 106566, 82050, 117472, 124280, 127678, 103616, 117360, 124220, 103520, 117304, 124190, 75840, 103472, 75808, 104160, 117624, 124350, 76992, 104048, 117564, 76896, 103992, 76848, 76824, 77536, 104312, 117694, 77424, 104252, 77368, 77340, 77688, 104382, 77628, 77758, 121536, 126320, 128700, 121440, 126264, 128670, 111680, 121392, 126236, 111648, 121368, 126222, 111632, 121356, 103104, 117104, 124092, 112320, 103008, 117048, 124062, 112224, 121656, 126366, 93248, 74784, 102936, 117006, 93216, 112152, 93200, 75456, 103280, 117180, 93888, 75360, 103224, 117150, 93792, 112440, 121758, 93744, 75288, 93720, 75632, 103356, 94064, 75576, 103326, 94008, 112542, 93980, 75708, 94140, 75678, 94110, 121184, 126136, 128606, 111168, 121136, 126108, 111136, 121112, 126094, 111120, 121100, 111112, 111108, 102752, 116920, 123998, 111456, 102704, 116892, 91712, 74272, 121244, 116878, 91680, 74256, 102668, 91664, 111372, 102662, 74244, 74592, 102840, 116958, 92000, 74544, 102812, 91952, 111516, 102798, 91928, 74508, 74502, 74680, 102878, 92088, 74652, 92060, 74638, 92046, 92126, 110912, 121008, 126044, 110880, 120984, 126030, 110864, 120972, 110856, 120966, 110852, 110850, 74048, 102576, 116828, 90944, 74016, 102552, 116814, 90912, 111000, 121038, 90896, 73992, 102534, 90888, 110982, 90884, 74160, 102620, 91056, 74136, 102606, 91032, 111054, 91020, 74118, 91014, 91100, 91086, 110752, 120920, 125998, 110736, 120908, 110728, 120902, 110724, 110722, 73888, 102488, 116782, 90528, 73872, 102476, 90512, 110796, 102470, 90504, 73860, 90500, 73858, 73944, 90584, 90572, 90566, 120876, 120870, 110658, 102444, 73800, 90312, 90308, 90306, 101056, 116080, 123580, 100960, 116024, 70720, 100912, 115996, 70688, 100888, 70672, 70664, 71360, 101232, 116156, 71264, 101176, 116126, 71216, 101148, 71192, 71180, 71536, 101308, 71480, 101278, 71452, 71612, 71582, 118112, 124600, 127838, 105024, 118064, 124572, 104992, 118040, 124558, 104976, 118028, 104968, 118022, 100704, 115896, 123486, 105312, 100656, 115868, 79424, 70176, 118172, 115854, 79392, 105240, 100620, 79376, 70152, 79368, 70496, 100792, 115934, 79712, 70448, 118238, 79664, 105372, 100750, 79640, 70412, 79628, 70584, 100830, 79800, 70556, 79772, 70542, 70622, 79838, 122176, 126640, 128860, 122144, 126616, 128846, 122128, 126604, 122120, 126598, 122116, 104768, 117936, 124508, 113472, 104736, 126684, 124494, 113440, 122264, 126670, 113424, 104712, 117894, 113416, 122246, 104706, 69952, 100528, 115804, 78656, 69920, 100504, 115790, 96064, 78624, 104856, 117966, 96032, 113560, 122318, 100486, 96016, 78600, 104838, 96008, 69890, 70064, 100572, 78768, 70040, 100558, 96176, 78744, 104910, 96152, 113614, 70022, 78726, 70108, 78812, 70094, 96220, 78798, 122016, 126552, 128814, 122000, 126540, 121992, 126534, 121988, 121986, 104608, 117848, 124462, 113056, 104592, 126574, 113040, 122060, 117830, 113032, 104580, 113028, 104578, 113026, 69792, 100440, 115758, 78240, 69776, 100428, 95136, 78224, 104652, 100422, 95120, 113100, 69764, 95112, 78212, 69762, 78210, 69848, 100462, 78296, 69836, 95192, 78284, 69830, 95180, 78278, 69870, 95214, 121936, 126508, 121928, 126502, 121924, 121922, 104528, 117804, 112848, 104520, 117798, 112840, 121958, 112836, 104514, 112834, 69712, 100396, 78032, 69704, 100390, 94672, 78024, 104550, 94664, 112870, 69698, 94660, 78018, 94658, 78060, 94700, 94694, 126486, 121890, 117782, 104484, 104482, 69672, 77928, 94440, 69666, 77922, 99680, 68160, 99632, 68128, 99608, 115342, 68112, 99596, 68104, 99590, 68448, 99768, 115422, 68400, 99740, 68376, 99726, 68364, 68358, 68536, 99806, 68508, 68494, 68574, 101696, 116400, 123740, 101664, 116376, 101648, 116364, 101640, 116358, 101636, 67904, 99504, 115292, 72512, 67872, 116444, 115278, 72480, 101784, 116430, 72464, 67848, 99462, 72456, 101766, 67842, 68016, 99548, 72624, 67992, 99534, 72600, 101838, 72588, 67974, 68060, 72668, 68046, 72654, 118432, 124760, 127918, 118416, 124748, 118408, 124742, 118404, 118402, 101536, 116312, 105888, 101520, 116300, 105872, 118476, 116294, 105864, 101508, 105860, 101506, 105858, 67744, 99416, 72096, 67728, 116334, 80800, 72080, 101580, 99398, 80784, 105932, 67716, 80776, 72068, 67714, 72066, 67800, 99438, 72152, 67788, 80856, 72140, 67782, 80844, 72134, 67822, 72174, 80878, 126800, 128940, 126792, 128934, 126788, 126786, 118352, 124716, 122576, 126828, 124710, 122568, 126822, 122564, 118338, 122562, 101456, 116268, 105680, 101448, 116262, 114128, 105672, 118374, 114120, 122598, 101442, 114116, 105666, 114114, 67664, 99372, 71888, 67656, 99366, 80336, 71880, 101478, 97232, 80328, 105702, 67650, 97224, 114150, 71874, 97220, 67692, 71916, 67686, 80364, 71910, 97260, 80358, 97254, 126760, 128918, 126756, 126754, 118312, 124694, 122472, 126774, 122468, 118306, 122466, 101416, 116246, 105576, 101412, 113896, 105572, 101410, 113892, 105570, 113890, 67624, 99350, 71784, 101430, 80104, 71780, 67618, 96744, 80100, 71778, 96740, 80098, 96738, 71798, 96758, 126738, 122420, 122418, 105524, 113780, 113778, 71732, 79988, 96500, 96498, 66880, 66848, 98968, 66832, 66824, 66820, 66992, 66968, 66956, 66950, 67036, 67022, 100000, 99984, 115532, 99976, 115526, 99972, 99970, 66720, 98904, 69024, 100056, 98892, 69008, 100044, 69000, 100038, 68996, 66690, 68994, 66776, 98926, 69080, 100078, 69068, 66758, 69062, 66798, 69102, 116560, 116552, 116548, 116546, 99920, 102096, 116588, 115494, 102088, 116582, 102084, 99906, 102082, 66640, 68816, 66632, 98854, 73168, 68808, 66628, 73160, 68804, 66626, 73156, 68802, 66668, 68844, 66662, 73196, 68838, 73190, 124840, 124836, 124834, 116520, 118632, 124854, 118628, 116514, 118626, 99880, 115478, 101992, 116534, 106216, 101988, 99874, 106212, 101986, 106210, 66600, 98838, 68712, 99894, 72936, 68708, 66594, 81384, 72932, 68706, 81380, 72930, 66614, 68726, 72950, 81398, 128980, 128978, 124820, 126900, 124818, 126898, 116500, 118580, 116498, 122740, 118578, 122738, 99860, 101940, 99858, 106100, 101938, 114420]), $a([128352, 129720, 125504, 128304, 129692, 125472, 128280, 129678, 125456, 128268, 125448, 128262, 125444, 125792, 128440, 129758, 120384, 125744, 128412, 120352, 125720, 128398, 120336, 125708, 120328, 125702, 120324, 120672, 125880, 128478, 110144, 120624, 125852, 110112, 120600, 125838, 110096, 120588, 110088, 120582, 110084, 110432, 120760, 125918, 89664, 110384, 120732, 89632, 110360, 120718, 89616, 110348, 89608, 110342, 89952, 110520, 120798, 89904, 110492, 89880, 110478, 89868, 90040, 110558, 90012, 89998, 125248, 128176, 129628, 125216, 128152, 129614, 125200, 128140, 125192, 128134, 125188, 125186, 119616, 125360, 128220, 119584, 125336, 128206, 119568, 125324, 119560, 125318, 119556, 119554, 108352, 119728, 125404, 108320, 119704, 125390, 108304, 119692, 108296, 119686, 108292, 108290, 85824, 108464, 119772, 85792, 108440, 119758, 85776, 108428, 85768, 108422, 85764, 85936, 108508, 85912, 108494, 85900, 85894, 85980, 85966, 125088, 128088, 129582, 125072, 128076, 125064, 128070, 125060, 125058, 119200, 125144, 128110, 119184, 125132, 119176, 125126, 119172, 119170, 107424, 119256, 125166, 107408, 119244, 107400, 119238, 107396, 107394, 83872, 107480, 119278, 83856, 107468, 83848, 107462, 83844, 83842, 83928, 107502, 83916, 83910, 83950, 125008, 128044, 125000, 128038, 124996, 124994, 118992, 125036, 118984, 125030, 118980, 118978, 106960, 119020, 106952, 119014, 106948, 106946, 82896, 106988, 82888, 106982, 82884, 82882, 82924, 82918, 124968, 128022, 124964, 124962, 118888, 124982, 118884, 118882, 106728, 118902, 106724, 106722, 82408, 106742, 82404, 82402, 124948, 124946, 118836, 118834, 106612, 106610, 124224, 127664, 129372, 124192, 127640, 129358, 124176, 127628, 124168, 127622, 124164, 124162, 117568, 124336, 127708, 117536, 124312, 127694, 117520, 124300, 117512, 124294, 117508, 117506, 104256, 117680, 124380, 104224, 117656, 124366, 104208, 117644, 104200, 117638, 104196, 104194, 77632, 104368, 117724, 77600, 104344, 117710, 77584, 104332, 77576, 104326, 77572, 77744, 104412, 77720, 104398, 77708, 77702, 77788, 77774, 128672, 129880, 93168, 128656, 129868, 92664, 128648, 129862, 92412, 128644, 128642, 124064, 127576, 129326, 126368, 124048, 129902, 126352, 128716, 127558, 126344, 124036, 126340, 124034, 126338, 117152, 124120, 127598, 121760, 117136, 124108, 121744, 126412, 124102, 121736, 117124, 121732, 117122, 121730, 103328, 117208, 124142, 112544, 103312, 117196, 112528, 121804, 117190, 112520, 103300, 112516, 103298, 112514, 75680, 103384, 117230, 94112, 75664, 103372, 94096, 112588, 103366, 94088, 75652, 94084, 75650, 75736, 103406, 94168, 75724, 94156, 75718, 94150, 75758, 128592, 129836, 91640, 128584, 129830, 91388, 128580, 91262, 128578, 123984, 127532, 126160, 123976, 127526, 126152, 128614, 126148, 123970, 126146, 116944, 124012, 121296, 116936, 124006, 121288, 126182, 121284, 116930, 121282, 102864, 116972, 111568, 102856, 116966, 111560, 121318, 111556, 102850, 111554, 74704, 102892, 92112, 74696, 102886, 92104, 111590, 92100, 74690, 92098, 74732, 92140, 74726, 92134, 128552, 129814, 90876, 128548, 90750, 128546, 123944, 127510, 126056, 128566, 126052, 123938, 126050, 116840, 123958, 121064, 116836, 121060, 116834, 121058, 102632, 116854, 111080, 121078, 111076, 102626, 111074, 74216, 102646, 91112, 74212, 91108, 74210, 91106, 74230, 91126, 128532, 90494, 128530, 123924, 126004, 123922, 126002, 116788, 120948, 116786, 120946, 102516, 110836, 102514, 110834, 73972, 90612, 73970, 90610, 128522, 123914, 125978, 116762, 120890, 102458, 110714, 123552, 127320, 129198, 123536, 127308, 123528, 127302, 123524, 123522, 116128, 123608, 127342, 116112, 123596, 116104, 123590, 116100, 116098, 101280, 116184, 123630, 101264, 116172, 101256, 116166, 101252, 101250, 71584, 101336, 116206, 71568, 101324, 71560, 101318, 71556, 71554, 71640, 101358, 71628, 71622, 71662, 127824, 129452, 79352, 127816, 129446, 79100, 127812, 78974, 127810, 123472, 127276, 124624, 123464, 127270, 124616, 127846, 124612, 123458, 124610, 115920, 123500, 118224, 115912, 123494, 118216, 124646, 118212, 115906, 118210, 100816, 115948, 105424, 100808, 115942, 105416, 118246, 105412, 100802, 105410, 70608, 100844, 79824, 70600, 100838, 79816, 105446, 79812, 70594, 79810, 70636, 79852, 70630, 79846, 129960, 95728, 113404, 129956, 95480, 113278, 129954, 95356, 95294, 127784, 129430, 78588, 128872, 129974, 95996, 78462, 128868, 127778, 95870, 128866, 123432, 127254, 124520, 123428, 126696, 128886, 123426, 126692, 124514, 126690, 115816, 123446, 117992, 115812, 122344, 117988, 115810, 122340, 117986, 122338, 100584, 115830, 104936, 100580, 113640, 104932, 100578, 113636, 104930, 113634, 70120, 100598, 78824, 70116, 96232, 78820, 70114, 96228, 78818, 96226, 70134, 78838, 129940, 94968, 113022, 129938, 94844, 94782, 127764, 78206, 128820, 127762, 95102, 128818, 123412, 124468, 123410, 126580, 124466, 126578, 115764, 117876, 115762, 122100, 117874, 122098, 100468, 104692, 100466, 113140, 104690, 113138, 69876, 78324, 69874, 95220, 78322, 95218, 129930, 94588, 94526, 127754, 128794, 123402, 124442, 126522, 115738, 117818, 121978, 100410, 104570, 112890, 69754, 78074, 94714, 94398, 123216, 127148, 123208, 127142, 123204, 123202, 115408, 123244, 115400, 123238, 115396, 115394, 99792, 115436, 99784, 115430, 99780, 99778, 68560, 99820, 68552, 99814, 68548, 68546, 68588, 68582, 127400, 129238, 72444, 127396, 72318, 127394, 123176, 127126, 123752, 123172, 123748, 123170, 123746, 115304, 123190, 116456, 115300, 116452, 115298, 116450, 99560, 115318, 101864, 99556, 101860, 99554, 101858, 68072, 99574, 72680, 68068, 72676, 68066, 72674, 68086, 72694, 129492, 80632, 105854, 129490, 80508, 80446, 127380, 72062, 127924, 127378, 80766, 127922, 123156, 123700, 123154, 124788, 123698, 124786, 115252, 116340, 115250, 118516, 116338, 118514, 99444, 101620, 99442, 105972, 101618, 105970, 67828, 72180, 67826, 80884, 72178, 80882, 97008, 114044, 96888, 113982, 96828, 96798, 129482, 80252, 130010, 97148, 80190, 97086, 127370, 127898, 128954, 123146, 123674, 124730, 126842, 115226, 116282, 118394, 122618, 99386, 101498, 105722, 114170, 67706, 71930, 80378, 96632, 113854, 96572, 96542, 80062, 96702, 96444, 96414, 96350, 123048, 123044, 123042, 115048, 123062, 115044, 115042, 99048, 115062, 99044, 99042, 67048, 99062, 67044, 67042, 67062, 127188, 68990, 127186, 123028, 123316, 123026, 123314, 114996, 115572, 114994, 115570, 98932, 100084, 98930, 100082, 66804, 69108, 66802, 69106, 129258, 73084, 73022, 127178, 127450, 123018, 123290, 123834, 114970, 115514, 116602, 98874, 99962, 102138, 66682, 68858, 73210, 81272, 106174, 81212, 81182, 72894, 81342, 97648, 114364, 97592, 114334, 97564, 97550, 81084, 97724, 81054, 97694, 97464, 114270, 97436, 97422, 80990, 97502, 97372, 97358, 97326, 114868, 114866, 98676, 98674, 66292, 66290, 123098, 114842, 115130, 98618, 99194, 66170, 67322, 69310, 73404, 73374, 81592, 106334, 81564, 81550, 73310, 81630, 97968, 114524, 97944, 114510, 97932, 97926, 81500, 98012, 81486, 97998, 97880, 114478, 97868, 97862, 81454, 97902, 97836, 97830, 69470, 73564, 73550, 81752, 106414, 81740, 81734, 73518, 81774, 81708, 81702]), $a([109536, 120312, 86976, 109040, 120060, 86496, 108792, 119934, 86256, 108668, 86136, 129744, 89056, 110072, 129736, 88560, 109820, 129732, 88312, 109694, 129730, 88188, 128464, 129772, 89592, 128456, 129766, 89340, 128452, 89214, 128450, 125904, 128492, 125896, 128486, 125892, 125890, 120784, 125932, 120776, 125926, 120772, 120770, 110544, 120812, 110536, 120806, 110532, 84928, 108016, 119548, 84448, 107768, 119422, 84208, 107644, 84088, 107582, 84028, 129640, 85488, 108284, 129636, 85240, 108158, 129634, 85116, 85054, 128232, 129654, 85756, 128228, 85630, 128226, 125416, 128246, 125412, 125410, 119784, 125430, 119780, 119778, 108520, 119798, 108516, 108514, 83424, 107256, 119166, 83184, 107132, 83064, 107070, 83004, 82974, 129588, 83704, 107390, 129586, 83580, 83518, 128116, 83838, 128114, 125172, 125170, 119284, 119282, 107508, 107506, 82672, 106876, 82552, 106814, 82492, 82462, 129562, 82812, 82750, 128058, 125050, 119034, 82296, 106686, 82236, 82206, 82366, 82108, 82078, 76736, 103920, 117500, 76256, 103672, 117374, 76016, 103548, 75896, 103486, 75836, 129384, 77296, 104188, 129380, 77048, 104062, 129378, 76924, 76862, 127720, 129398, 77564, 127716, 77438, 127714, 124392, 127734, 124388, 124386, 117736, 124406, 117732, 117730, 104424, 117750, 104420, 104418, 112096, 121592, 126334, 92608, 111856, 121468, 92384, 111736, 121406, 92272, 111676, 92216, 111646, 92188, 75232, 103160, 117118, 93664, 74992, 103036, 93424, 112252, 102974, 93304, 74812, 93244, 74782, 93214, 129332, 75512, 103294, 129908, 129330, 93944, 75388, 129906, 93820, 75326, 93758, 127604, 75646, 128756, 127602, 94078, 128754, 124148, 126452, 124146, 126450, 117236, 121844, 117234, 121842, 103412, 103410, 91584, 111344, 121212, 91360, 111224, 121150, 91248, 111164, 91192, 111134, 91164, 91150, 74480, 102780, 91888, 74360, 102718, 91768, 111422, 91708, 74270, 91678, 129306, 74620, 129850, 92028, 74558, 91966, 127546, 128634, 124026, 126202, 116986, 121338, 102906, 90848, 110968, 121022, 90736, 110908, 90680, 110878, 90652, 90638, 74104, 102590, 91000, 74044, 90940, 74014, 90910, 74174, 91070, 90480, 110780, 90424, 110750, 90396, 90382, 73916, 90556, 73886, 90526, 90296, 110686, 90268, 90254, 73822, 90334, 90204, 90190, 71136, 101112, 116094, 70896, 100988, 70776, 100926, 70716, 70686, 129204, 71416, 101246, 129202, 71292, 71230, 127348, 71550, 127346, 123636, 123634, 116212, 116210, 101364, 101362, 79296, 105200, 118140, 79072, 105080, 118078, 78960, 105020, 78904, 104990, 78876, 78862, 70384, 100732, 79600, 70264, 100670, 79480, 105278, 79420, 70174, 79390, 129178, 70524, 129466, 79740, 70462, 79678, 127290, 127866, 123514, 124666, 115962, 118266, 100858, 113376, 122232, 126654, 95424, 113264, 122172, 95328, 113208, 122142, 95280, 113180, 95256, 113166, 95244, 78560, 104824, 117950, 95968, 78448, 104764, 95856, 113468, 104734, 95800, 78364, 95772, 78350, 95758, 70008, 100542, 78712, 69948, 96120, 78652, 69918, 96060, 78622, 96030, 70078, 78782, 96190, 94912, 113008, 122044, 94816, 112952, 122014, 94768, 112924, 94744, 112910, 94732, 94726, 78192, 104636, 95088, 78136, 104606, 95032, 113054, 95004, 78094, 94990, 69820, 78268, 69790, 95164, 78238, 95134, 94560, 112824, 121950, 94512, 112796, 94488, 112782, 94476, 94470, 78008, 104542, 94648, 77980, 94620, 77966, 94606, 69726, 78046, 94686, 94384, 112732, 94360, 112718, 94348, 94342, 77916, 94428, 77902, 94414, 94296, 112686, 94284, 94278, 77870, 94318, 94252, 94246, 68336, 99708, 68216, 99646, 68156, 68126, 68476, 68414, 127162, 123258, 115450, 99834, 72416, 101752, 116414, 72304, 101692, 72248, 101662, 72220, 72206, 67960, 99518, 72568, 67900, 72508, 67870, 72478, 68030, 72638, 80576, 105840, 118460, 80480, 105784, 118430, 80432, 105756, 80408, 105742, 80396, 80390, 72048, 101564, 80752, 71992, 101534, 80696, 71964, 80668, 71950, 80654, 67772, 72124, 67742, 80828, 72094, 80798, 114016, 122552, 126814, 96832, 113968, 122524, 96800, 113944, 122510, 96784, 113932, 96776, 113926, 96772, 80224, 105656, 118366, 97120, 80176, 105628, 97072, 114076, 105614, 97048, 80140, 97036, 80134, 97030, 71864, 101470, 80312, 71836, 97208, 80284, 71822, 97180, 80270, 97166, 67678, 71902, 80350, 97246, 96576, 113840, 122460, 96544, 113816, 122446, 96528, 113804, 96520, 113798, 96516, 96514, 80048, 105564, 96688, 80024, 105550, 96664, 113870, 96652, 80006, 96646, 71772, 80092, 71758, 96732, 80078, 96718, 96416, 113752, 122414, 96400, 113740, 96392, 113734, 96388, 96386, 79960, 105518, 96472, 79948, 96460, 79942, 96454, 71726, 79982, 96494, 96336, 113708, 96328, 113702, 96324, 96322, 79916, 96364, 79910, 96358, 96296, 113686, 96292, 96290, 79894, 96310, 66936, 99006, 66876, 66846, 67006, 68976, 100028, 68920, 99998, 68892, 68878, 66748, 69052, 66718, 69022, 73056, 102072, 116574, 73008, 102044, 72984, 102030, 72972, 72966, 68792, 99934, 73144, 68764, 73116, 68750, 73102, 66654, 68830, 73182, 81216, 106160, 118620, 81184, 106136, 118606, 81168, 106124, 81160, 106118, 81156, 81154, 72880, 101980, 81328, 72856, 101966, 81304, 106190, 81292, 72838, 81286, 68700, 72924, 68686, 81372, 72910, 81358, 114336, 122712, 126894, 114320, 122700, 114312, 122694, 114308, 114306, 81056, 106072, 118574, 97696, 81040, 106060, 97680, 114380, 106054, 97672, 81028, 97668, 81026, 97666, 72792, 101934, 81112, 72780, 97752, 81100, 72774, 97740, 81094, 97734, 68654, 72814, 81134, 97774, 114256, 122668, 114248, 122662, 114244, 114242, 80976, 106028, 97488, 80968, 106022, 97480, 114278, 97476, 80962, 97474, 72748, 81004, 72742, 97516, 80998, 97510, 114216, 122646, 114212, 114210, 80936, 106006, 97384, 80932, 97380, 80930, 97378, 72726, 80950, 97398, 114196, 114194, 80916, 97332, 80914, 97330, 66236, 66206, 67256, 99166, 67228, 67214, 66142, 67294, 69296, 100188, 69272, 100174, 69260, 69254, 67164, 69340, 67150, 69326, 73376, 102232, 116654, 73360, 102220, 73352, 102214, 73348, 73346, 69208, 100142, 73432, 102254, 73420, 69190, 73414, 67118, 69230, 73454, 106320, 118700, 106312, 118694, 106308, 106306, 73296, 102188, 81616, 106348, 102182, 81608, 73284, 81604, 73282, 81602, 69164, 73324, 69158, 81644, 73318, 81638, 122792, 126934, 122788, 122786, 106280, 118678, 114536, 106276, 114532, 106274, 114530, 73256, 102166, 81512, 73252, 98024, 81508, 73250, 98020, 81506, 98018, 69142, 73270, 81526, 98038, 122772, 122770, 106260, 114484, 106258, 114482, 73236, 81460, 73234, 97908, 81458, 97906, 122762, 106250, 114458, 73226, 81434, 97850, 66396, 66382, 67416, 99246, 67404, 67398, 66350, 67438, 69456, 100268, 69448, 100262, 69444, 69442, 67372, 69484, 67366, 69478, 102312, 116694, 102308, 102306, 69416, 100246, 73576, 102326, 73572, 69410, 73570, 67350, 69430, 73590, 118740, 118738, 102292, 106420, 102290, 106418, 69396, 73524, 69394, 81780, 73522, 81778, 118730, 102282, 106394, 69386, 73498, 81722, 66476, 66470, 67496, 99286, 67492, 67490, 66454, 67510, 100308, 100306, 67476, 69556, 67474, 69554, 116714])]); //#19603
    $_.raps = $a([$a([802, 930, 946, 818, 882, 890, 826, 954, 922, 986, 970, 906, 778, 794, 786, 914, 978, 982, 980, 916, 948, 932, 934, 942, 940, 936, 808, 812, 814, 806, 822, 950, 918, 790, 788, 820, 884, 868, 870, 878, 876, 872, 840, 856, 860, 862, 846, 844, 836, 838, 834, 866]), $a([718, 590, 622, 558, 550, 566, 534, 530, 538, 570, 562, 546, 610, 626, 634, 762, 754, 758, 630, 628, 612, 614, 582, 578, 706, 738, 742, 740, 748, 620, 556, 552, 616, 744, 712, 716, 708, 710, 646, 654, 652, 668, 664, 696, 688, 656, 720, 592, 600, 604, 732, 734])]); //#19618
    $_.cwtobits = function() {
        var _Ih = $get($_.clusters, $k[--$j]); //#19623
        $_.v = $get(_Ih, $k[--$j]); //#19623
        $k[$j++] = Infinity; //#19624
        for (var _Ik = 0; _Ik < 17; _Ik++) { //#19624
            $k[$j++] = 0; //#19624
        } //#19624
        var _In = $cvrs($s(17), $_.v, 2); //#19624
        for (var _Io = 0, _Ip = _In.length; _Io < _Ip; _Io++) { //#19624
            $k[$j++] = $get(_In, _Io) - 48; //#19624
        } //#19624
        var _Ir = $a(); //#19624
        $k[$j++] = $geti(_Ir, _Ir.length - 17, 17); //#19625
    }; //#19625
    $_.raptobits = function() {
        var _Iv = $get($_.raps, $k[--$j]); //#19630
        $_.v = $get(_Iv, $k[--$j]); //#19630
        $k[$j++] = Infinity; //#19631
        for (var _Iy = 0; _Iy < 10; _Iy++) { //#19631
            $k[$j++] = 0; //#19631
        } //#19631
        var _J1 = $cvrs($s(10), $_.v, 2); //#19631
        for (var _J2 = 0, _J3 = _J1.length; _J2 < _J3; _J2++) { //#19631
            $k[$j++] = $get(_J1, _J2) - 48; //#19631
        } //#19631
        var _J5 = $a(); //#19631
        $k[$j++] = $geti(_J5, _J5.length - 10, 10); //#19632
    }; //#19632
    $_.rwid = $get($a([38, 55, 82, 99]), $_.c - 1); //#19636
    if (($_.c == 3) && $_.cca) { //#19637
        $_.rwid = 72; //#19637
    } //#19637
    $_.pixs = $a(($_.rwid * $_.r) * $_.rowmult); //#19638
    for (var _JI = 0, _JH = $_.r - 1; _JI <= _JH; _JI += 1) { //#19680
        $_.i = _JI; //#19641
        $_.clst = ($f($_.i + $_.rapl) - 1) % 3;