import { createContext, useContext, useEffect, useState } from "react";
import { toast } from "react-toastify";

const SupplierProductContext = createContext(null);

/* ---------- Hook ---------- */
export const useSupplierCart = () => {
  const context = useContext(SupplierProductContext);

  if (!context) {
    toast.error("باید داخل SupplierProductProvider استفاده شود");
    throw new Error("SupplierProductContext error");
  }

  return context;
};

/* ---------- Provider ---------- */
export const SupplierProductProvider = ({ children }) => {
  const [supplierCart, setSupplierCart] = useState([]);

  /* Load from localStorage */
  useEffect(() => {
    const savedCart = localStorage.getItem("supplierCart");
    if (savedCart) {
      try {
        setSupplierCart(JSON.parse(savedCart));
      } catch (error) {
        console.log(error);
      }
    }
  }, []);

  /* Save to localStorage */
  useEffect(() => {
    localStorage.setItem("supplierCart", JSON.stringify(supplierCart));
  }, [supplierCart]);

  /* ---------- Functions ---------- */

  const addToCart = (product, price, quantity) => {
    setCart(prev => {
      const index = prev.findIndex(
        item =>
          item.product._id === product._id &&
          item.price === price
      );

      if (index !== -1) {
        const updated = [...prev];
        updated[index].quantity += quantity;
        return updated;
      }

      return [...prev, { product, price, quantity }];
    });
  };

  const removeFromCart = (name, karat) => {
  setSupplierCart(prev =>
    prev.filter(item => !(item.name === name && item.karat === karat))
  );
};

  const updateQuantity = (productId, quantity) => {
    if (quantity < 1) {
      removeFromCart(productId);
      return;
    }

    supplierCart(prev =>
      prev.map(item =>
        item.product._id === productId
          ? { ...item, quantity }
          : item
      )
    );
  };

  const clearCart = () => {
    setSupplierCart([]);
    localStorage.removeItem("supplierCart");
  };

  const getTotalPrice = () => {
    return supplierCart.reduce(
      (total, item) => total + item.price * item.quantity,
      0
    );
  };

  const getTotalItems = () => supplierCart.length;

  const savePrice = (productId, price, discount = 0) => {
    setCart(prev =>
      prev.map(item =>
        item.product._id === productId
          ? { ...item, price, discount }
          : item
      )
    );
  };

  return (
    <SupplierProductContext.Provider
      value={{
        supplierCart,
        setSupplierCart,
        addToCart,
        removeFromCart,
        updateQuantity,
        clearCart,
        getTotalPrice,
        getTotalItems,
        savePrice
      }}
    >
      {children}
    </SupplierProductContext.Provider>
  );
};
