import React, { useContext, useEffect, useState } from 'react'
import { AppContext } from '../context/AppContext'
import Barcode from 'react-barcode'
import { assets } from '../assets/assets'
import { toast } from 'react-toastify'
import axios from 'axios'

const DailyRegistredProduct = () => {
  const { backendUrl } = useContext(AppContext)
  const [existProduct, setExistProduct] = useState([])
  const [editId, setEditId] = useState(null)

  const getExistProduct = async () => {
    try {
      const { data } = await axios.get(backendUrl + '/api/product/today')
      if (!data.success) {
        toast.warn(data.message)
      } else {
        setExistProduct(data.dailyProduct)
      }
    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  useEffect(() => {
    getExistProduct()
  }, [])

  const toPersianDate = (date) => {
    if (!date) return "";
    const d = new Date(date);
    return d.toLocaleDateString("fa-IR");
  };

  const handleChange = (id, field, value) => {
    setExistProduct(prev =>
      prev.map(p => p._id === id ? { ...p, [field]: value } : p)
    )
  }

  const updateProduct = async (id) => {
    const product = existProduct.find(p => p._id === id)
    if (!product) return toast.error("محصول وجود ندارد")

    try {
      const { data } = await axios.put(`${backendUrl}/api/product/${id}`, product, {
        headers: { "Content-Type": "application/json" }
      })

      if (data.success) {
        toast.success("محصول ویرایش شد")
        setEditId(null)
        getExistProduct()
      } else {
        toast.error(data.message)
      }
    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  const deleteProduct = async (productId) => {
    try {
      const { data } = await axios.delete(backendUrl + `/api/product/delete-product/${productId}`)
      if (data.success) {
        toast.success(data.message)
        setExistProduct(prev => prev.filter(item => item._id !== productId))
      }
    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  // تابع چاپ دو ستونه
  const printTwoSidedBarcode = async (product) => {
  const printWindow = window.open('', '_blank');
  
  // بارگذاری کتابخانه bwip-js
  const html = `
    <!DOCTYPE html>
    <html dir="rtl">
    <head>
      <title>چاپ بارکود - ${product.barcode}</title>
      <script src="https://cdn.jsdelivr.net/npm/bwip-js@3.3.1/dist/bwip-js.min.js"></script>
      <style>
        @media print {
          @page {
            size: 60mm 15mm;
            margin: 0.5mm;
          }
          body {
            margin: 0;
            padding-top: 4px;
            width: 75mm;
            height: 12mm;
            font-family: Tahoma, Arial, sans-serif;
          }
          .container {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: end;
            justify-content: space-around;
            padding: 0 0.5mm;
          }
          .left-column {
            width: 40%;
            height: 100%;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            text-align: center;
            font-size: 6pt;
            line-height: 1;
          }
          .right-column {
            width: 60%;
            height: 100%;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
          }
          .shop-name {
            font-size: 10pt;
            margin-bottom: 0.5mm;
            color: #000;
            font-weight:500;
          }

          .product-info {
            font-size: 8pt;
            margin: 0.2mm 0;
          }
          .barcode-text {
            font-family: monospace;
            font-size: 9pt;
            text-align: center;
            margin-top: 0.5mm;
          }
          #barcode-canvas {
            width: 20mm;
            height: 6mm;
          }
        }
        
        @media screen {
          body {
            background: #f0f0f0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
          }
          .container {
            background: white;
            border: 1px solid #ccc;
          }
        }
      </style>
    </head>
    <body>
      <div class="container">
        <!-- ستون چپ: اطلاعات مغازه -->
        <div class="left-column">
          <div class="shop-name">زرگری صداقت</div>
          <div class="product-info">${product.gram} گرام</div>
          <div class="product-info">${product.karat} عیار</div>
          <div class="product-info">${product.wage}</div>
        </div>
        
        <!-- ستون راست: بارکود -->
        <div class="right-column">
          <canvas id="barcode-canvas"></canvas>
          <div class="barcode-text">${product.barcode}</div>
        </div>
      </div>
      
      <script>
        // تولید بارکود روی کانوا
        try {
          bwipjs.toCanvas('barcode-canvas', {
            bcid: 'code128',        // نوع بارکود
            text: '${product.barcode}',  // متن بارکود
            scale: 0.7,               // مقیاس
            height: 3,              // ارتفاع
            includetext: false,     // نمایش متن
            textxalign: 'center',   // تراز متن
          });
          
          // چاپ بعد از تولید بارکود
          setTimeout(() => {
            window.print();
          }, 300);
        } catch (e) {
          console.error('خطا در تولید بارکود:', e);
          // اگر بارکود تولید نشد، باز هم چاپ کن
          setTimeout(() => {
            window.print();
          }, 300);
        }
        
        // بستن پنجره بعد از چاپ
        window.onafterprint = function() {
          setTimeout(() => {
            window.close();
          }, 100);
        };
      </script>
    </body>
    </html>
  `;
  
  printWindow.document.open();
  printWindow.document.write(html);
  printWindow.document.close();
};
  // تابع ساده‌تر برای تست سریع

  const filteredExistProduct = existProduct.filter(product => product.isSold === false)

  return (
    <div className='w-full px-20 py-8 h-[90vh] overflow-y-scroll'>
      <div className='flex justify-center pb-2 font-bold text-3xl'>
        <h1>اجناس موجود</h1>
      </div>

      {/* table header */}
      <div className='grid grid-cols-[1fr_3fr_3fr_2fr_2fr_2fr_2fr_2fr_4fr_4fr_1.5fr] text-center justify-center items-center text-white'>
        <p className='border bg-sky-800 border-white py-2 text-sm'>شماره</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>نام جنس</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>نوعیت جنس</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>وزن</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>عیار</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>خرید</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>واحد پول</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>تاریخ</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>عکس</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>بارکود</p>
        <p className='border bg-sky-800 border-white py-2 text-sm'>ویرایش</p>
      </div>

      {/* table rows */}
      {existProduct.map((product, index) => (
        <div key={product._id} className='grid grid-cols-[1fr_3fr_3fr_2fr_2fr_2fr_2fr_2fr_4fr_4fr_1.5fr] text-center items-center'>
          <p className='border border-gray-300 py-12 text-sm'>{index + 1}</p>

          {/* editable fields */}
          {editId === product._id ? (
            <>
              <input className='w-full py-1.5 px-4 border border-gray-300' value={product.productName} onChange={e => handleChange(product._id, "productName", e.target.value)} />
              <input className='w-full py-1.5 px-4 border border-gray-300' value={product.type} onChange={e => handleChange(product._id, "type", e.target.value)} />
              <input className='w-full py-1.5 px-4 border border-gray-300' type='number' value={product.weight} onChange={e => handleChange(product._id, "weight", e.target.value)} />
              <input className='w-full py-1.5 px-4 border border-gray-300' type='number' value={product.karat} onChange={e => handleChange(product._id, "karat", e.target.value)} />
              <input className='w-full py-1.5 px-4 border border-gray-300' type='number' value={product.purchasePrice} onChange={e => handleChange(product._id, "purchasePrice", e.target.value)} />
              <input className='w-full py-1.5 px-4 border border-gray-300' value={product.currency} onChange={e => handleChange(product._id, "currency", e.target.value)} />
            </>
          ) : (
            <>
              <p className='border border-gray-300 py-12 text-sm'>{product.productName}</p>
              <p className='border border-gray-300 py-12 text-sm'>{product.type}</p>
              <p className='border border-gray-300 py-12 text-sm'>{product.gram}</p>
              <p className='border border-gray-300 py-12 text-sm'>{product.karat}</p>
              <p className='border border-gray-300 py-12 text-sm'>{parseFloat(product.purchasePriceToAfn).toFixed(0)}</p>
              <p className='border border-gray-300 py-12 text-sm'>افغانی</p>
            </>
          )}

          <p className='border border-gray-300 py-12 text-sm'>{toPersianDate(product.createdAt)}</p>
          <div className='flex justify-center'>
            <img className='w-28 h-28 object-cover' src={`http://localhost:3000/${product.image}`} alt='' />
          </div>

          <div className='h-28 text-sm'>
            <Barcode
              value={product.barcode}
              format="CODE128"
              width={0.8}
              height={50}
              displayValue
              fontSize={16}
              margin={2}
            />
            <div className='flex flex-col gap-1 justify-center mt-1'>
              <button
                className='bg-green-600 py-1 px-2 text-white rounded cursor-pointer text-xs hover:bg-green-700'
                onClick={() => printTwoSidedBarcode(product)}
              >
                چاپ دو ستونه
              </button>

            </div>
          </div>

          {/* edit buttons */}
          <div className='flex justify-center items-center border border-gray-300 py-6'>
            {editId === product._id ? (
              <div className='flex flex-col items-center gap-2'>
                <img src={assets.sell_icon} className='w-6 cursor-pointer hover:opacity-80' onClick={() => updateProduct(editId)} alt='' />
                <img src={assets.back_icon} className='w-6 cursor-pointer hover:opacity-80' onClick={() => setEditId(null)} alt='' />
              </div>
            ) : (
              <div className='flex flex-col items-center gap-2'>
                <img src={assets.edit_icon} className='w-6 cursor-pointer hover:opacity-80' onClick={() => setEditId(product._id)} alt='' />
              </div>
            )}
          </div>
        </div>
      ))}
    </div>
  )
}

export default DailyRegistredProduct