import React, { useState, useEffect, useContext } from "react";
import { AppContext } from "../context/AppContext";
import { assets } from "../assets/assets";
import { useNavigate } from "react-router-dom";
import axios from "axios";
import { toast } from "react-toastify";
import Barcode from "react-barcode";

const LoanManagement = () => {
  const { getLoan, loanData, backendUrl } = useContext(AppContext);

  const [isToggle, setIsToggle] = useState(false);
  const [showPay, setShowPay] = useState(false);
  const [customerLoan, setCustomerLoan] = useState([]);
  const [loading, setLoading] = useState(false);
  const [payToggle, setPayToggle] = useState(false)
  const [transactionId, setTransactionId] = useState("")
  const [amount, setAmount] = useState("")
  const [currency, setCurrency] = useState("")

  console.log("transactiob Id is :", transactionId)

  const navigate = useNavigate();

  useEffect(() => {
    getLoan();
  }, []);

  const toPersianDate = (date) =>
    new Date(date).toLocaleDateString("fa-IR");


  const getLoanAccordingPerson = async (customerId) => {
    try {
      setLoading(true);
      const { data } = await axios.get(
        backendUrl + "/api/transaction/to-pay",
        { params: { customerId } }
      );

      if (data.success) {
        console.log(data)
        setCustomerLoan(data.loan || []);
      }
    } catch (error) {
      toast.error("خطا در دریافت اطلاعات");
    } finally {
      setLoading(false);
    }
  };

  const payLoan = async () => {

    if (currency === "" || !currency) {
      return toast.error("واحد پولی را انتخاب کنید")
    }

    try {

      const { data } = await axios.post(backendUrl + `/api/transaction/pay-loan?transactionId=${transactionId}`, { amount: Number(amount), currency })
      if (data.success) {
        toast.success(data.message)
      }
    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  const convertToEnglish = (str) => {
    const map = {
      "۰": "0", "۱": "1", "۲": "2", "۳": "3", "۴": "4",
      "۵": "5", "۶": "6", "۷": "7", "۸": "8", "۹": "9",
      "٠": "0", "١": "1", "٢": "2", "٣": "3", "٤": "4",
      "٥": "5", "٦": "6", "٧": "7", "٨": "8", "٩": "9"
    };
    return str.replace(/[۰-۹٠-٩]/g, d => map[d]);
  };

  return (
    <div className="w-full py-8 px-16 relative">
      <div>
        <img onClick={() => setIsToggle(!isToggle)} src={assets.drop_down_icon} className='w-8 cursor-pointer' alt="" />
        <div className={` bg-sky-700 absolute top-18 text-sm rounded-xs right-20 py-2 ${isToggle ? 'block' : 'hidden'}`}>
          <p onClick={()=>navigate('/loan')} className="py-2 px-4 text-white hover:bg-sky-800 cursor-pointer ">افزودن گزارش قرض</p>
          <p onClick={()=>navigate('/report-list')} className="py-2 px-4 text-white hover:bg-sky-800 cursor-pointer ">لیست گزارش قرض</p>
        </div>
      </div>

      <div className={`w-lg pb-12 bg-white transition-all duration-500 absolute z-20 rounded shadow-2xl ${payToggle ? 'top-16 right-72' : 'top-16 -right-140'}`}>
        <div><img onClick={() => setPayToggle(false)} src={assets.cancel_icon} className="w-6 m-2 cursor-pointer" alt="" /></div>
        <div className="w-full flex flex-col items-center gap-4">
          <p className="text-xl">رسید حساب مشتری</p>
          <div className="flex gap-1">
            <input type="text" onChange={e => setAmount(convertToEnglish(e.target.value))} value={amount} inputMode="numeric" className="bg-blue-600 rounded-xs border border-gray-400 py-1 px-2 w-72 text-white" />
            <select onChange={e => setCurrency(e.target.value)} value={currency} name="" id="" className="bg-blue-600 border border-blue-600 text-white rounded-xs cursor-pointer">
              <option className="cursor-pointer" value="">واحد پولی</option>
              <option className="cursor-pointer" value="دالر">دالر</option>
              <option className="cursor-pointer" value="افغانی">افغانی</option>
            </select>
          </div>
          <button onClick={payLoan} className="bg-green-600 text-white py-1 w-40 rounded-xs cursor-pointer">ذخیره</button>
        </div>
      </div>

      {/* ---- menu ---- */}
      {/* <img
        src={assets.drop_down_icon}
        onClick={() => setIsToggle(!isToggle)}
        className="w-6 absolute top-8 right-28 cursor-pointer"
        alt=""
      /> */}


      {/* ---- header ---- */}
      <div className="flex justify-center">
        <h1 className="text-2xl font-semibold">قرضداران</h1>
      </div>

      {/* ---- table header ---- */}
      <ul className="grid grid-cols-[2fr_4fr_4fr_3fr_3fr_3fr_3fr_3fr] text-center mt-6">
        {[
          "شماره",
          "اسم مشتری",
          "شماره مشتری",
          "مجموع",
          "باقی",
          "رسید",
          "دفعات خرید",
          "#",
        ].map((h) => (
          <li key={h} className="bg-sky-700 border border-white py-1 text-white">
            {h}
          </li>
        ))}
      </ul>

      {/* ---- loan rows ---- */}
      {Array.isArray(loanData) &&
        loanData.map((item, index) => (
          <ul
            key={item._id}
            className="grid grid-cols-[2fr_4fr_4fr_3fr_3fr_3fr_3fr_3fr] text-center"
          >
            <li className="border border-gray-500 py-1">{index + 1}</li>
            <li className="border border-gray-500 py-1">{item.customerName}</li>
            <li className="border border-gray-500 py-1">{item.customerPhone}</li>
            <li className="border border-gray-500 py-1">
              {Number(item.totalAmount || 0).toFixed(0)}
            </li>
            <li className="border border-gray-500 py-1">
              {Number(item.totalLoan || 0).toFixed(0)}
            </li>
            <li className="border border-gray-500 py-1">
              {Number(item.totalPaid || 0).toFixed(0)}
            </li>
            <li className="border border-gray-500 py-1">
              {Number(item.transactionsCount || 0)}
            </li>
            <li className="border border-gray-500 py-1 flex justify-center gap-2">
              <img src={assets.delete_icon} className="w-6 cursor-pointer" />
              <img src={assets.edit_icon} className="w-6 cursor-pointer" />
              <img
                src={assets.add_icon}
                className="w-6 cursor-pointer"
                onClick={() => {
                  setShowPay(true);
                  getLoanAccordingPerson(item.customerId);
                }}
              />
            </li>
          </ul>
        ))}

      {/* ---- modal ---- */}
      {showPay && (
        <div className="fixed inset-0 bg-black/50 flex justify-center items-center">
          <div className="bg-white px-8 pt-12 pb-2 rounded w-[90%] max-h-[80%] overflow-y-auto relative">
            <img
              src={assets.cancel_icon}
              className="w-6 cursor-pointer mb-4 absolute top-2 right-2"
              onClick={() => setShowPay(false)}
            />

            {loading && <p className="text-center">در حال دریافت اطلاعات...</p>}

            {!loading &&
              customerLoan.filter(l => l.remainingAmount > 0).map((p, index) => (
                <div key={index} className="mb-12 border border-gray-400/80 rounded-xs">

                  {/* ---- info ---- */}
                  <ul className="grid grid-cols-6 text-center gap-2 p-4 gap-x-12 w-full bg-sky-600  rounded-xs text-gray-200 mb-8">
                    <li>نام مشتری:</li>
                    <li className="border-b border-gray-300/50">{p.customerName}</li>

                    <li>شماره:</li>
                    <li className="border-b border-gray-300/50">{p.customerPhone}</li>

                    <li>تاریخ:</li>
                    <li className="border-b border-gray-300/50">{toPersianDate(p.date)}</li>

                    <li>تعداد اقلام:</li>
                    <li className='border-b border-gray-300/50'>{p.product.length}</li>

                    <li>بیل نمبر:</li>
                    <li className='border-b border-gray-300/50'>{p.bellNumber}</li>

                    <li>واحد پولی:</li>
                    <li className='border-b border-gray-300/50'>افغانی</li>

                    <li>مجموع:</li>
                    <li className="border-b border-gray-300/50">{Number(p.totalAmount || 0).toFixed(0)}</li>

                    <li>تخفیف:</li>
                    <li className="border-b border-gray-300/50">{Number(p.discount || 0).toFixed(0)}</li>

                    <li>رسید:</li>
                    <li className="border-b border-gray-300/50 flex justify-center gap-2"><img
                      onClick={() => {
                        setPayToggle(true)
                        setTransactionId(p._id)
                      }} src={assets.add_icon} className="w-6 cursor-pointer" />{Number(p.paidAmount || 0).toFixed(0)}</li>

                    <li>باقی:</li>
                    <li className="border-b border-gray-300/50">{Number(p.remainingAmount || 0).toFixed(0)}</li>

                  </ul>

                  {/* ---- products ---- */}
                  <ul className="grid grid-cols-7 font-bold px-2 mt-4">
                    <li>#</li>
                    <li>نام جنس</li>
                    <li>گرام</li>
                    <li>عیار</li>
                    <li>خرید</li>
                    <li>فروش</li>
                    <li>بارکود</li>
                  </ul>

                  {p.product.map((prod, i) => (
                    <ul
                      key={prod._id}
                      className="grid grid-cols-7 border-t px-2 py-2"
                    >
                      <li>{i + 1}</li>
                      <li>{prod.name}</li>
                      <li>{prod.gram}</li>
                      <li>{prod.karat}</li>
                      <li>{Number(prod.purchase).toFixed(0)}</li>
                      <li>{Number(prod.sold).toFixed(0)}</li>
                      <li className="flex justify-center">
                        <Barcode
                          value={prod.barcode}
                          format="CODE128"
                          width={0.8}
                          height={50}
                          displayValue={true}
                          fontSize={16}
                          margin={2}
                        />
                      </li>
                    </ul>
                  ))}
                </div>
              ))}
          </div>
        </div>
      )}
    </div>
  );
};

export default LoanManagement;
