import React, { useContext, useEffect, useState } from 'react'
import { AppContext } from '../context/AppContext'
import { toast } from "react-toastify";
import axios from 'axios'
import SearchCustomer from '../components/SearchCustomer';
import { assets } from '../assets/assets';
import { data, useNavigate } from 'react-router-dom';
import Barcode from "react-barcode";
import DateObject from "react-date-object";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import Cart from '../components/Cart';
import ProductCart from '../components/ProductCart';
import { useCart } from '../context/CartContext';
import SearchProduct from '../components/SearchProduct';

const SaleProducts = () => {

  const { backendUrl } = useContext(AppContext)
  const { cart, setCart, getTotalItems, getTotalPrice } = useCart()
  const [date, setDate] = useState(null)
  const [customerQuery, setCustomerQuery] = useState("");
  const [results, setResults] = useState([]);
  const [loading, setLoading] = useState(false);
  const [selectedCustomer, setSelectedCustomer] = useState(null)
  const [paidAmount, setPaidAmount] = useState("")
  const [bellNumber, setBellNumber] = useState("")
  const [note, setNote] = useState("")
  const [productQuery, setProductQuery] = useState("")
  const [showFinalPrice, setShowFinalPrice] = useState(false)


  const updateIsSold = async (productId, value) => {

    const product = productByBarcode

    if (product.isSold === value) {
      toast.warn(`این محصول قبلاً ${value ? "فروخته شده" : "موجود"} است`)
      return
    }

    try {

      const { data } = await axios.post(backendUrl + `/api/product/${productId}/sold`, { isSold: value })

      if (data.success) {
        console.log(data)
        toast.success(data.message)
      }
      setProductByBarcode(prev => ({ ...prev, isSold: value }))

    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  const handleKeyDownForProduct = (e) => {
    if (e.key === "Enter") {
      fetchDataProduct(productByBarcode)
      setProductByBarcode("")
    }
  }

  const fetchDataProduct = async (code) => {
    try {

      const { data } = await axios.get(backendUrl + `/api/product/search-barcode/${code}`)

      if (!data.success) {
        toast.error("جنس پیدا نشد")
      } else {
        console.log(data)
        setProductByBarcode(data.productByBarcode)
      }


    } catch (error) {
      console.log(error)
      toast.error(error.message)
    }
  }

  function toPersianDate(date) {
    if (!date) return "-";

    const d = new Date(date);
    if (isNaN(d.getTime())) return "-";

    return new Intl.DateTimeFormat('fa-IR', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
    }).format(d);
  }


  const handleClickOnCustomer = (customer) => {
    setSelectedCustomer(customer)
    setCustomerQuery(customer.name)
    setResults([])
  }

  const fetchDataCustomer = async () => {

    if (!customerQuery.trim()) {
      toast.warn("لطفا چیزی بنویسید")
    }

    try {
      setLoading(true)

      const { data } = await axios.get(backendUrl + `/api/customer/search-customer?q=${customerQuery}`)

      if (data.success) {
        setResults(data.result)
      } else {
        setResults([])
      }

      if (!data.result) {
        toast.warn("دیتا وجود ندارد")
      }

    } catch (error) {
      console.log(error)
      toast.error(error.message)
    } finally {
      setLoading(false)
    }
  }

  const handleKeyDownForCustomer = (e) => {
    if (e.key === "Enter") {
      fetchDataCustomer();
    }
  };


  const product = cart.map((item) => ({
    productId: item._id,
    productName: item.productName,
    purchasePriceToAfn: item.purchasePriceToAfn,
    salePrice: {
      price: item.salePrice,
      currency: item.saleCurrency
    },
    barcode: item.barcode,
    gram: item.gram,
    karat: item.karat,
  }))


  // create transaction
  const createTransaction = async () => {

    const totalAmount = getTotalPrice()
    const remainingAmount = getTotalPrice() - (paidAmount)
    const totalQuantity = getTotalItems()

    const receipt = {
      totalAmount,
      paidAmount,
      remainingAmount,
      totalQuantity,
      date
    }

    if (!selectedCustomer) {
      return toast.warn("لطفاً مشتری را انتخاب کنید");
    }
    if (!product || product.length === 0) {
      return toast.warn("سبد خرید خالی است");
    }

    if (!bellNumber) {
      return toast.warn("شماره بل وارد نشده است");
    }

    if (!paidAmount) {
      toast.warn("مقدار رسید وارد نشده است")
    }

    if (!receipt) {
      return toast.warn("رسید وارد نشده است");
    }

    try {

      const { data } = await axios.post(
        `${backendUrl}/api/transaction/create-transaction`,
        {
          customerId: selectedCustomer._id,
          product: product,
          receipt,
          bellNumber,
          note,
        }
      );

      if (data.success) {
        toast.success("ترانسکشن با موفقیت ثبت شد");
        setCart([]);
        setBellNumber("");
        setNote("");
        setPaidAmount(null)
        setSelectedCustomer(null)
        localStorage.removeItem('selectedCustomer')
        setShowFinalPrice(false)
      } else {
        toast.error(data.message || "خطا در ثبت تراکنش");
      }
    } catch (error) {
      console.error(error);
      toast.error(error.response?.data?.message || error.message);
    }
  };

  const showPriceTable = async () => {
    const getSelectedCustomer = JSON.parse(localStorage.getItem('selectedCustomer'))
    if (!getSelectedCustomer) {
      return toast.error("نام مشتری را انتخاب کنید")
    }

    setShowFinalPrice(true)
  }

  useEffect(() => {
    if (selectedCustomer) {
      localStorage.setItem('selectedCustomer', JSON.stringify(selectedCustomer))
    }

  }, [selectedCustomer])

  const convertToEnglish = (str) => {
    const map = {
      "۰": "0", "۱": "1", "۲": "2", "۳": "3", "۴": "4",
      "۵": "5", "۶": "6", "۷": "7", "۸": "8", "۹": "9",
      "٠": "0", "١": "1", "٢": "2", "٣": "3", "٤": "4",
      "٥": "5", "٦": "6", "٧": "7", "٨": "8", "٩": "9"
    };
    return str.replace(/[۰-۹٠-٩]/g, d => map[d]);
  };

  const navigate = useNavigate()

  return (
    <div className='w-full pr-1 pt-1'>

      {showFinalPrice && (<div className="fixed inset-0 bg-black/40 flex items-center justify-center z-50">
        <div className="bg-white p-4 rounded w-80">

          <div>
            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">نمبر بل</label>
              <input onChange={(e) => setBellNumber(Number(convertToEnglish(e.target.value)))} value={bellNumber} className='border border-gray-500 rounded-xs py-1' inputMode='numeric' type="text" />
            </div>

            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">تاریخ</label>
              <input onChange={(e) => setDate(e.target.value)} className='border border-gray-500 rounded-xs py-1' type="date" />
            </div>

            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">جزئیات</label>
              <textarea onChange={(e) => setNote(e.target.value)} className='border border-gray-500 rounded-xs py-1' type="text" rows={2} />
            </div>


            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">مقدار رسید</label>
              <input onChange={(e) => setPaidAmount(Number(convertToEnglish(e.target.value)))} value={paidAmount} className='border border-gray-500 rounded-xs py-1' type="text" inputMode='numeric' />
            </div>

            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">قیمت نهایی</label>
              <p>{getTotalPrice()}</p>
            </div>

            <div className='flex flex-col py-1 px-2 text-sm'>
              <label className='text-gray-700' htmlFor="">باقی</label>
              <p>{getTotalPrice() - (paidAmount)}</p>
            </div>
          </div>

          <div className="flex justify-end gap-2">
            <button
              onClick={() => (setShowFinalPrice(false))}
              className="px-3 py-1 border rounded"
            >
              لغو
            </button>

            <button
              className="px-3 py-1 bg-green-600 text-white rounded"
              onClick={createTransaction}
            >
              ثبت
            </button>
          </div>

        </div>

      </div>)}

      {/* ---- header ---- */}
      <div className=' w-full grid grid-cols-[1fr_4fr] mb-2 '>
        <div></div>
        <div className='flex justify-center items-center'><SearchProduct /></div>
      </div>


      {/* ---- content ---- */}
      <div className='grid grid-cols-[1fr_4fr]'>


        {/* ---- customer detail ---- */}
        <div className='relative w-full'>
          <div className=''>
            <SearchCustomer fetchData={fetchDataCustomer} query={customerQuery} setQuery={setCustomerQuery} handleKeyDown={handleKeyDownForCustomer} />
          </div>

          {/* ---- customers ---- */}
          <div className='absolute w-full'>
            {loading && <p className="text-sm mt-2">در حال جستجو...</p>}

            <ul className="mt-2 space-y-2 bg-gray-100 rounded w-full text-sm ">
              {results.map((item) => (
                <li
                  key={item._id}
                  onClick={() => {
                    handleClickOnCustomer(item)
                    setCustomerQuery(item.customerName)
                  }}
                  className="py-1 px-4 cursor-pointer w-full border-b border-gray-300 rounded-md hover:bg-white"
                >
                  <p className="font-bold text-gray-700">{item.customerName}</p>
                  <p className="text-sm text-gray-500">{item.phone}</p>
                </li>
              ))}
            </ul>
          </div>

          {/* ---- transaction detail -----  */}

          <div className='mt-3'>


            <div className='flex py-1 px-2 text-sm  '>
              <button onClick={() => navigate('/customer-registration')} className='flex items-center justify-center py-1 bg-teal-700 text-white w-full rounded-xs gap-2 h-8 cursor-pointer hover:-translate-y-1.5 transition-all duration-500'>افزودن مشتری</button>
            </div>

            <div className='flex py-1 px-2 text-sm  '>
              <button onClick={showPriceTable} className='flex items-center justify-center py-1 bg-teal-700 text-white w-full rounded-xs gap-2 h-8 cursor-pointer hover:-translate-y-1.5 transition-all duration-500'><span>ذخیره</span></button>
            </div>
          </div>

          <div className='py-1 px-2 mt-4'>
            <button onClick={() => navigate('/sold-product')} className='w-full bg-sky-600 text-white py-2 rounded cursor-pointer text-sm hover:-translate-y-2.5 transition-all duration-500'>اجناس فروخته شده</button>
          </div>


        </div>

        {/*----- product detail -----*/}
        <div>

          <div>
            <Cart paidAmount={paidAmount} setPaidAmount={setPaidAmount} />
          </div>


        </div>
      </div>

    </div>
  )
}

export default SaleProducts
