import React, { useContext, useEffect, useState } from 'react'
import { AppContext } from '../context/AppContext'
import DatePicker from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import { assets } from '../assets/assets';
import { toast } from 'react-toastify';
import axios from 'axios';

const Spent = () => {
  const { backendUrl, spents, setSpents, getSpents } = useContext(AppContext);

  const [editId, setEditId] = useState(null);
  const [search, setSearch] = useState("");
  const [products, setProducts] = useState([]);
  const [isToggle , setIsToggle] = useState(false)
  const [total , setIsTotal] = useState([])

  /* ---------------- utils ---------------- */
  const toPersianDate = (date) => {
    if (!date) return "-";
    const d = new Date(date);
    if (isNaN(d.getTime())) return "-";
    return new Intl.DateTimeFormat('fa-IR', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    }).format(d);
  };

  /* ---------------- change handler ---------------- */
  const handleChange = (id, field, value) => {
    setSpents(prev =>
      prev.map(item =>
        item._id === id ? { ...item, [field]: value } : item
      )
    );
  };

  /* ---------------- update ---------------- */
  const updateSpent = async (id) => {
    const spent = spents.find(s => s._id === id);
    if (!spent) return toast.error("دیتا وجود ندارد");

    try {
      const { data } = await axios.put(
        `${backendUrl}/api/expense/update-spent/${id}`,
        spent
      );

      if (data.success) {
        toast.success("مصرف آپدیت شد");
        setEditId(null);
        getSpents();
      }
    } catch (error) {
      console.log(error);
      toast.error(error.message);
    }
  };

  /* ---------------- delete ---------------- */
  const deleteSpent = async (id) => {
    try {
      const { data } = await axios.delete(
        `${backendUrl}/api/expense/delete-spent/${id}`
      );

      if (data.success) {
        toast.success(data.message);
        setSpents(prev => prev.filter(item => item._id !== id));
      }
    } catch (error) {
      console.log(error);
      toast.error(error.message);
    }
  };

  /* ---------------- search ---------------- */
  const handleSearch = async () => {
    const { data } = await axios.get(
      `${backendUrl}/api/expense/search`,
      { params: { search } }
    );
    return data;
  };

  useEffect(() => {
    if (!search.trim()) {
      setProducts([]);
      return;
    }

    const delay = setTimeout(async () => {
      try {
        const data = await handleSearch();
        if (data?.success) {
          setProducts(data.data || []);
        }
      } catch (err) {
        console.log(err);
      }
    }, 300);

    return () => clearTimeout(delay);
  }, [search]);

  const getTotal = async (req, res) => {
    try {
      const { data } = await axios.get(backendUrl + '/api/expense/total')
      if (data.success) {
        console.log("data is", data)
        setIsTotal(data.total)
        
      }
    } catch (error) {
      console.log(error);
      toast.error(error.message);
    }
  }

  /* ---------------- load data ---------------- */
  useEffect(() => {
    getSpents();
    getTotal()
  }, []);

  if (!spents) {
    return <p className="text-center mt-10">در حال بارگذاری...</p>;
  }


  const list = search.trim() ? products : spents;

  return (
    <div className='px-12 py-8 w-full flex flex-col h-[90vh] overflow-y-auto'>

      {/* search */}
      
      <div className='mb-3 flex justify-center'>
        <input
          type="text"
          placeholder="جستجو"
          value={search}
          onChange={(e) => setSearch(e.target.value)}
          className='border border-gray-500 py-1 px-3'
        />
      </div>

      <div className='flex justify-end mb-2 relative '>
        <img src={assets.drop_down_icon} onClick={()=>setIsToggle(!isToggle)} className='w-8 cursor-pointer' alt="" />
        <div className={`p-4 flex flex-col gap-4 rounded absolute left-4 top-8 bg-white ${isToggle ? 'block' : 'hidden'}`}>
          {total.map((item , index)=>(
            <p><span className='text-red-600 font-bold'>{item.price}</span> _ {item._id.currency} _ حساب {item._id.type}</p>
          ))}
        </div>
      </div>

      {/* header */}
      <div className='grid grid-cols-[1fr_3fr_3fr_2fr_2fr_2fr_2fr] w-full'>
        {["شماره", "نوعیت مصرف", "توضیحات", "مقدار", "واحد پول", "تاریخ", "ویرایش"].map(h => (
          <p key={h} className='bg-gray-300 border px-2 py-1 text-center'>{h}</p>
        ))}
      </div>

      {/* rows */}
      {list.map((spent, index) => (
        <div
          key={spent._id}
          className='grid grid-cols-[1fr_3fr_3fr_2fr_2fr_2fr_2fr] w-full'
        >
          <p className='border px-2 py-1 text-center'>{index + 1}</p>

          {editId === spent._id ? (
            <>
              <input
                value={spent.type}
                onChange={(e) => handleChange(spent._id, "type", e.target.value)}
                className='border px-2'
              />
              <input
                value={spent.detail}
                onChange={(e) => handleChange(spent._id, "detail", e.target.value)}
                className='border px-2'
              />
              <input
                type="number"
                value={spent.price}
                onChange={(e) => handleChange(spent._id, "price", e.target.value)}
                className='border px-2'
              />
              <input
                value={spent.currency}
                onChange={(e) => handleChange(spent._id, "currency", e.target.value)}
                className='border px-2'
              />
              <DatePicker
                value={spent.date}
                onChange={(v) => handleChange(spent._id, "date", v.toDate())}
                calendar={persian}
                locale={persian_fa}
                format="YYYY/MM/DD"
                inputClass="border px-2 w-full"
              />
            </>
          ) : (
            <>
              <p className='border px-2'>{spent.type}</p>
              <p className='border px-2'>{spent.detail}</p>
              <p className='border px-2'>{spent.price}</p>
              <p className='border px-2'>{spent.currency}</p>
              <p className='border px-2'>{toPersianDate(spent.date)}</p>
            </>
          )}

          <div className='border flex justify-center items-center gap-2'>
            {editId === spent._id ? (
              <>
                <img src={assets.sell_icon} className='w-6 cursor-pointer' onClick={() => updateSpent(spent._id)} />
                <img src={assets.back_icon} className='w-6 cursor-pointer' onClick={() => setEditId(null)} />
              </>
            ) : (
              <>
                <img src={assets.edit_icon} className='w-6 cursor-pointer' onClick={() => setEditId(spent._id)} />
                <img src={assets.delete_icon} className='w-6 cursor-pointer' onClick={() => deleteSpent(spent._id)} />
              </>
            )}
          </div>
        </div>
      ))}
    </div>
  );
};

export default Spent;
